# pyheif
Python 3.6+ interface to [libheif](https://github.com/strukturag/libheif) library using CFFI

*Note*: currently only reading is supported.

## Installation

### Simple installation - Linux (installs manylinux2014 wheel, doesn't work with Alpine)
```
pip install --upgrade pip
pip install pyheif
```

### Installing from source - MacOS
```
brew install libffi libheif
pip install git+https://github.com/carsales/pyheif.git
```

### Installing from source - Linux
```
apt install libffi libheif-dev libde265-dev
```
or
```
yum install libffi libheif-devel libde265-devel
```
then
```
pip install git+https://github.com/carsales/pyheif.git
```

### Installing from source - Windows
```
Sorry, not going to happen!
```

## Usage

### Read the primary image of a HEIF encoded file

The `pyheif.read(path_or_bytes)` function can be used to read the primary image of a HEIF encoded file. It can be passed any of the following:

* A string path to a file on disk
* A `pathlib.Path` path object
* A Python `bytes` or `bytearray` object containing HEIF content
* A file-like object with a `.read()` method that returns bytes

It returns a `HeifFile` object.

```python
import pyheif

# Using a file path:
heif_file = pyheif.read("IMG_7424.HEIC")
# Or using bytes directly:
heif_file = pyheif.read(open("IMG_7424.HEIC", "rb").read())
```

### Converting to a Pillow Image object

If your HEIF file contains an image that you would like to manipulate, you can do so using the [Pillow](https://pillow.readthedocs.io/) Python library. You can convert a `HeifFile` to a Pillow image like so:

```python
from PIL import Image
import pyheif

heif_file = pyheif.read("IMG_7424.HEIC")
image = Image.frombytes(
    heif_file.mode, 
    heif_file.size, 
    heif_file.data,
    "raw",
    heif_file.mode,
    heif_file.stride,
    )
```

*Note*: the `mode` property is passed twice - once to the `mode` argument of the `frombytes` method, and again to the `mode` argument of the `raw` decoder.

You can now use any Pillow method to manipulate the file. Here's how to convert it to JPEG:

```python
image.save("IMG_7424.jpg", "JPEG")
```

### Read the entire container within the HEIF file

The `pyheif.open_container(path_or_bytes)` function can be used to read the HEIF container from a HEIF encoded file. It takes the same parameter as `pyheif.read()`

It returns a `HeifContainer` object.

## Objects

### The HeifFile object

The `HeifFile` has the following properties:

* `mode` - the image mode, e.g. "RGB" or "RGBA"
* `size` - the size of the image as a `(width, height)` tuple of integers
* `data` - the raw decoded file data, as bytes
* `metadata` - a list of metadata dictionaries
* `color_profile` - a color profile dictionary
* `stride` - the number of bytes in a row of decoded file data
* `bit_depth` - the number of bits in each component of a pixel

### The UndecodedHeifFile object

This is a HEIF image that has not been decoded. Calling the `UndecodedHeifFile.load()` method will load the data and the object will become a `HeifFile`

### The HeifContainer object

The `HeifContainer` has the following properties:

* `primary_image` - the `HeifTopLevelImage` object of the primary image in the file.
* `top_level_images` - a list of all `HeifTopLevelImage` objects in the file.

### The HeifTopLevelImage object

The `HeifTopLevelImage` has the following properties:

* `id` - the id of the image
* `image` - the `UndecodedHeifFile` or `HeifFile` object of the image
* `is_primary` - is this the primary image in the container
* `depth_image` - the `HeifDepthImage` if there is one
* `auxiliary_images` - a list of `HeifAuxiliaryImage` objects

### The HeifDepthImage object

The `HeifDepthImage` has the following properties:

* `id` - the id of the image
* `image` - the `UndecodedHeifFile` or `HeifFile` object of the image

### The HeifAuxiliaryImage object

The `HeifAuxiliaryImage` has the following properties:

* `id` - the id of the image
* `image` - the `UndecodedHeifFile` or `HeifFile` object of the image
* `type` - a string indicating the type of auxiliary image



