# Licensed under a 3-clause BSD style license - see LICENSE.rst

# "core.py" is auto-generated by erfa_generator.py from the template
# "core.py.templ". Do *not* edit "core.py" directly, instead edit
# "core.py.templ" and run erfa_generator.py from the source directory to
# update it.

"""
Python wrappers for the ufunc wrappers of the ERFA library.

The key idea is that any function can be called with inputs that are arrays,
and the ufuncs will automatically vectorize and call the ERFA functions for
each item using broadcasting rules for numpy.  So the return values are always
numpy arrays of some sort.

For ERFA functions that take/return vectors or matrices, the vector/matrix
dimension(s) are always the *last* dimension(s).  For example, if you
want to give ten matrices (i.e., the ERFA input type is double[3][3]),
you would pass in a (10, 3, 3) numpy array.  If the output of the ERFA
function is scalar, you'll get back a length-10 1D array.
(Note that the ufuncs take this into account using structured dtypes.)

Note that the ufunc part of these functions are implemented in a separate
module (compiled as ``ufunc``), derived from the ``ufunc.c`` file.
"""

import warnings

import numpy

from . import ufunc

__all__ = [
    'ErfaError', 'ErfaWarning',
    'cal2jd', 'epb', 'epb2jd', 'epj', 'epj2jd', 'jd2cal', 'jdcalf', 'ab', 'apcg',
    'apcg13', 'apci', 'apci13', 'apco', 'apco13', 'apcs', 'apcs13', 'aper',
    'aper13', 'apio', 'apio13', 'atci13', 'atciq', 'atciqn', 'atciqz', 'atco13',
    'atic13', 'aticq', 'aticqn', 'atio13', 'atioq', 'atoc13', 'atoi13', 'atoiq',
    'ld', 'ldn', 'ldsun', 'pmpx', 'pmsafe', 'pvtob', 'refco', 'epv00', 'plan94',
    'fad03', 'fae03', 'faf03', 'faju03', 'fal03', 'falp03', 'fama03', 'fame03',
    'fane03', 'faom03', 'fapa03', 'fasa03', 'faur03', 'fave03', 'bi00', 'bp00',
    'bp06', 'bpn2xy', 'c2i00a', 'c2i00b', 'c2i06a', 'c2ibpn', 'c2ixy', 'c2ixys',
    'c2t00a', 'c2t00b', 'c2t06a', 'c2tcio', 'c2teqx', 'c2tpe', 'c2txy', 'eo06a',
    'eors', 'fw2m', 'fw2xy', 'ltp', 'ltpb', 'ltpecl', 'ltpequ', 'num00a', 'num00b',
    'num06a', 'numat', 'nut00a', 'nut00b', 'nut06a', 'nut80', 'nutm80', 'obl06',
    'obl80', 'p06e', 'pb06', 'pfw06', 'pmat00', 'pmat06', 'pmat76', 'pn00',
    'pn00a', 'pn00b', 'pn06', 'pn06a', 'pnm00a', 'pnm00b', 'pnm06a', 'pnm80',
    'pom00', 'pr00', 'prec76', 's00', 's00a', 's00b', 's06', 's06a', 'sp00',
    'xy06', 'xys00a', 'xys00b', 'xys06a', 'ee00', 'ee00a', 'ee00b', 'ee06a',
    'eect00', 'eqeq94', 'era00', 'gmst00', 'gmst06', 'gmst82', 'gst00a', 'gst00b',
    'gst06', 'gst06a', 'gst94', 'pvstar', 'starpv', 'fk425', 'fk45z', 'fk524',
    'fk52h', 'fk54z', 'fk5hip', 'fk5hz', 'h2fk5', 'hfk5z', 'starpm', 'eceq06',
    'ecm06', 'eqec06', 'lteceq', 'ltecm', 'lteqec', 'g2icrs', 'icrs2g', 'eform',
    'gc2gd', 'gc2gde', 'gd2gc', 'gd2gce', 'd2dtf', 'dat', 'dtdb', 'dtf2d', 'taitt',
    'taiut1', 'taiutc', 'tcbtdb', 'tcgtt', 'tdbtcb', 'tdbtt', 'tttai', 'tttcg',
    'tttdb', 'ttut1', 'ut1tai', 'ut1tt', 'ut1utc', 'utctai', 'utcut1', 'ae2hd',
    'hd2ae', 'hd2pa', 'tpors', 'tporv', 'tpsts', 'tpstv', 'tpxes', 'tpxev', 'a2af',
    'a2tf', 'af2a', 'anp', 'anpm', 'd2tf', 'tf2a', 'tf2d', 'rx', 'ry', 'rz', 'cp',
    'cpv', 'cr', 'p2pv', 'pv2p', 'ir', 'zp', 'zpv', 'zr', 'rxr', 'tr', 'rxp',
    'rxpv', 'trxp', 'trxpv', 'rm2v', 'rv2m', 'pap', 'pas', 'sepp', 'seps', 'c2s',
    'p2s', 'pv2s', 's2c', 's2p', 's2pv', 'pdp', 'pm', 'pmp', 'pn', 'ppp', 'ppsp',
    'pvdpv', 'pvm', 'pvmpv', 'pvppv', 'pvu', 'pvup', 'pvxpv', 'pxp', 's2xpv',
    'sxp', 'sxpv',
    'DPI', 'D2PI', 'DR2D', 'DD2R', 'DR2AS', 'DAS2R', 'DS2R', 'TURNAS', 'DMAS2R',
    'DTY', 'DAYSEC', 'DJY', 'DJC', 'DJM', 'DJ00', 'DJM0', 'DJM00', 'DJM77',
    'TTMTAI', 'DAU', 'CMPS', 'AULT', 'DC', 'ELG', 'ELB', 'TDB0', 'SRS', 'WGS84',
    'GRS80', 'WGS72']


class ErfaError(ValueError):
    """
    A class for errors triggered by ERFA functions (status codes < 0)

    Note: this class should *not* be referenced by fully-qualified name, because
    it may move to ERFA in a future version.  In a future such move it will
    still be imported here as an alias, but the true namespace of the class may
    change.
    """


class ErfaWarning(UserWarning):
    """
    A class for warnings triggered by ERFA functions (status codes > 0)

    Note: this class should *not* be referenced by fully-qualified name, because
    it may move to ERFA in a future version.  In a future such move it will
    still be imported here as an alias, but the true namespace of the class may
    change.
    """


# <---------------------------------Error-handling---------------------------->


STATUS_CODES = {}  # populated below before each function that returns an int

# This is a hard-coded list of status codes that need to be remapped,
# such as to turn errors into warnings.
STATUS_CODES_REMAP = {
    'cal2jd': {-3: 3}
}


def check_errwarn(statcodes, func_name):
    if not numpy.any(statcodes):
        return
    # Remap any errors into warnings in the STATUS_CODES_REMAP dict.
    if func_name in STATUS_CODES_REMAP:
        for before, after in STATUS_CODES_REMAP[func_name].items():
            statcodes[statcodes == before] = after
            STATUS_CODES[func_name][after] = STATUS_CODES[func_name][before]

    if numpy.any(statcodes < 0):
        # Errors present - only report the errors.
        if statcodes.shape:
            statcodes = statcodes[statcodes < 0]

        errcodes = numpy.unique(statcodes)

        errcounts = dict([(e, numpy.sum(statcodes == e)) for e in errcodes])

        elsemsg = STATUS_CODES[func_name].get('else', None)
        if elsemsg is None:
            errmsgs = dict([(e, STATUS_CODES[func_name].get(
                e, 'Return code ' + str(e))) for e in errcodes])
        else:
            errmsgs = dict([(e, STATUS_CODES[func_name].get(
                e, elsemsg)) for e in errcodes])

        emsg = ', '.join(['{0} of "{1}"'.format(errcounts[e], errmsgs[e])
                          for e in errcodes])
        raise ErfaError('ERFA function "{}" yielded {}'
                        .format(func_name, emsg))

    elif numpy.any(statcodes > 0):
        # Only warnings present.
        if statcodes.shape:
            statcodes = statcodes[statcodes > 0]

        warncodes = numpy.unique(statcodes)

        warncounts = dict([(w, numpy.sum(statcodes == w)) for w in warncodes])

        elsemsg = STATUS_CODES[func_name].get('else', None)
        if elsemsg is None:
            warnmsgs = dict([(w, STATUS_CODES[func_name].get(
                w, 'Return code ' + str(w))) for w in warncodes])
        else:
            warnmsgs = dict([(w, STATUS_CODES[func_name].get(
                w, elsemsg)) for w in warncodes])

        wmsg = ', '.join(['{0} of "{1}"'.format(warncounts[w], warnmsgs[w])
                          for w in warncodes])
        warnings.warn('ERFA function "{}" yielded {}'.format(func_name, wmsg),
                      ErfaWarning)


# <------------------------structured dtype conversion------------------------>

dt_bytes1 = numpy.dtype('S1')
dt_bytes12 = numpy.dtype('S12')

# <--------------------------Actual ERFA-wrapping code------------------------>


DPI = (3.141592653589793238462643)
"""Pi"""
D2PI = (6.283185307179586476925287)
"""2Pi"""
DR2D = (57.29577951308232087679815)
"""Radians to degrees"""
DD2R = (1.745329251994329576923691e-2)
"""Degrees to radians"""
DR2AS = (206264.8062470963551564734)
"""Radians to arcseconds"""
DAS2R = (4.848136811095359935899141e-6)
"""Arcseconds to radians"""
DS2R = (7.272205216643039903848712e-5)
"""Seconds of time to radians"""
TURNAS = (1296000.0)
"""Arcseconds in a full circle"""
DMAS2R = (DAS2R / 1e3)
"""Milliarcseconds to radians"""
DTY = (365.242198781)
"""Length of tropical year B1900 (days)"""
DAYSEC = (86400.0)
"""Seconds per day."""
DJY = (365.25)
"""Days per Julian year"""
DJC = (36525.0)
"""Days per Julian century"""
DJM = (365250.0)
"""Days per Julian millennium"""
DJ00 = (2451545.0)
"""Reference epoch (J2000.0), Julian Date"""
DJM0 = (2400000.5)
"""Julian Date of Modified Julian Date zero"""
DJM00 = (51544.5)
"""Reference epoch (J2000.0), Modified Julian Date"""
DJM77 = (43144.0)
"""1977 Jan 1.0 as MJD"""
TTMTAI = (32.184)
"""TT minus TAI (s)"""
DAU = (149597870.7e3)
"""Astronomical unit (m, IAU 2012)"""
CMPS = 299792458.0
"""Speed of light (m/s)"""
AULT = (DAU/CMPS)
"""Light time for 1 au (s)"""
DC = (DAYSEC/AULT)
"""Speed of light (au per day)"""
ELG = (6.969290134e-10)
"""L_G = 1 - d(TT)/d(TCG)"""
ELB = (1.550519768e-8)
"""L_B = 1 - d(TDB)/d(TCB), and TDB (s) at TAI 1977/1/1.0"""
TDB0 = (-6.55e-5)
"""L_B = 1 - d(TDB)/d(TCB), and TDB (s) at TAI 1977/1/1.0"""
SRS = 1.97412574336e-8
"""Schwarzschild radius of the Sun (au) = 2 * 1.32712440041e20 / (2.99792458e8)^2
/ 1.49597870700e11"""
WGS84 = 1
"""Reference ellipsoids"""
GRS80 = 2
"""Reference ellipsoids"""
WGS72 = 3
"""Reference ellipsoids"""


def cal2jd(iy, im, id):
    """
    Gregorian Calendar to Julian Date.

    Parameters
    ----------
    iy : int array
    im : int array
    id : int array

    Returns
    -------
    djm0 : double array
    djm : double array

    Notes
    -----
    Wraps ERFA function ``eraCal2jd``. The ERFA documentation is::

        - - - - - - - - - -
         e r a C a l 2 j d
        - - - - - - - - - -

        Gregorian Calendar to Julian Date.

        Given:
           iy,im,id  int     year, month, day in Gregorian calendar (Note 1)

        Returned:
           djm0      double  MJD zero-point: always 2400000.5
           djm       double  Modified Julian Date for 0 hrs

        Returned (function value):
                     int     status:
                                 0 = OK
                                -1 = bad year   (Note 3: JD not computed)
                                -2 = bad month  (JD not computed)
                                -3 = bad day    (JD computed)

        Notes:

        1) The algorithm used is valid from -4800 March 1, but this
           implementation rejects dates before -4799 January 1.

        2) The Julian Date is returned in two pieces, in the usual ERFA
           manner, which is designed to preserve time resolution.  The
           Julian Date is available as a single number by adding djm0 and
           djm.

        3) In early eras the conversion is from the "Proleptic Gregorian
           Calendar";  no account is taken of the date(s) of adoption of
           the Gregorian Calendar, nor is the AD/BC numbering convention
           observed.

        Reference:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992),
           Section 12.92 (p604).

        This revision:  2013 August 7

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    djm0, djm, c_retval = ufunc.cal2jd(iy, im, id)
    check_errwarn(c_retval, 'cal2jd')
    return djm0, djm


STATUS_CODES['cal2jd'] = {
    0: 'OK',
    -1: 'bad year   (Note 3: JD not computed)',
    -2: 'bad month  (JD not computed)',
    -3: 'bad day    (JD computed)',
}


def epb(dj1, dj2):
    """
    Julian Date to Besselian Epoch.

    Parameters
    ----------
    dj1 : double array
    dj2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraEpb``. The ERFA documentation is::

        - - - - - - -
         e r a E p b
        - - - - - - -

        Julian Date to Besselian Epoch.

        Given:
           dj1,dj2    double     Julian Date (see note)

        Returned (function value):
                      double     Besselian Epoch.

        Note:

           The Julian Date is supplied in two pieces, in the usual ERFA
           manner, which is designed to preserve time resolution.  The
           Julian Date is available as a single number by adding dj1 and
           dj2.  The maximum resolution is achieved if dj1 is 2451545.0
           (J2000.0).

        Reference:

           Lieske, J.H., 1979. Astron.Astrophys., 73, 282.

        This revision:  2013 August 21

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.epb(dj1, dj2)
    return c_retval


def epb2jd(epb):
    """
    Besselian Epoch to Julian Date.

    Parameters
    ----------
    epb : double array

    Returns
    -------
    djm0 : double array
    djm : double array

    Notes
    -----
    Wraps ERFA function ``eraEpb2jd``. The ERFA documentation is::

        - - - - - - - - - -
         e r a E p b 2 j d
        - - - - - - - - - -

        Besselian Epoch to Julian Date.

        Given:
           epb      double    Besselian Epoch (e.g. 1957.3)

        Returned:
           djm0     double    MJD zero-point: always 2400000.5
           djm      double    Modified Julian Date

        Note:

           The Julian Date is returned in two pieces, in the usual ERFA
           manner, which is designed to preserve time resolution.  The
           Julian Date is available as a single number by adding djm0 and
           djm.

        Reference:

           Lieske, J.H., 1979, Astron.Astrophys. 73, 282.

        This revision:  2013 August 13

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    djm0, djm = ufunc.epb2jd(epb)
    return djm0, djm


def epj(dj1, dj2):
    """
    Julian Date to Julian Epoch.

    Parameters
    ----------
    dj1 : double array
    dj2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraEpj``. The ERFA documentation is::

        - - - - - - -
         e r a E p j
        - - - - - - -

        Julian Date to Julian Epoch.

        Given:
           dj1,dj2    double     Julian Date (see note)

        Returned (function value):
                      double     Julian Epoch

        Note:

           The Julian Date is supplied in two pieces, in the usual ERFA
           manner, which is designed to preserve time resolution.  The
           Julian Date is available as a single number by adding dj1 and
           dj2.  The maximum resolution is achieved if dj1 is 2451545.0
           (J2000.0).

        Reference:

           Lieske, J.H., 1979, Astron.Astrophys. 73, 282.

        This revision:  2013 August 7

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.epj(dj1, dj2)
    return c_retval


def epj2jd(epj):
    """
    Julian Epoch to Julian Date.

    Parameters
    ----------
    epj : double array

    Returns
    -------
    djm0 : double array
    djm : double array

    Notes
    -----
    Wraps ERFA function ``eraEpj2jd``. The ERFA documentation is::

        - - - - - - - - - -
         e r a E p j 2 j d
        - - - - - - - - - -

        Julian Epoch to Julian Date.

        Given:
           epj      double    Julian Epoch (e.g. 1996.8)

        Returned:
           djm0     double    MJD zero-point: always 2400000.5
           djm      double    Modified Julian Date

        Note:

           The Julian Date is returned in two pieces, in the usual ERFA
           manner, which is designed to preserve time resolution.  The
           Julian Date is available as a single number by adding djm0 and
           djm.

        Reference:

           Lieske, J.H., 1979, Astron.Astrophys. 73, 282.

        This revision:  2013 August 7

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    djm0, djm = ufunc.epj2jd(epj)
    return djm0, djm


def jd2cal(dj1, dj2):
    """
    Julian Date to Gregorian year, month, day, and fraction of a day.

    Parameters
    ----------
    dj1 : double array
    dj2 : double array

    Returns
    -------
    iy : int array
    im : int array
    id : int array
    fd : double array

    Notes
    -----
    Wraps ERFA function ``eraJd2cal``. The ERFA documentation is::

        - - - - - - - - - -
         e r a J d 2 c a l
        - - - - - - - - - -

        Julian Date to Gregorian year, month, day, and fraction of a day.

        Given:
           dj1,dj2   double   Julian Date (Notes 1, 2)

        Returned (arguments):
           iy        int      year
           im        int      month
           id        int      day
           fd        double   fraction of day

        Returned (function value):
                     int      status:
                                 0 = OK
                                -1 = unacceptable date (Note 1)

        Notes:

        1) The earliest valid date is -68569.5 (-4900 March 1).  The
           largest value accepted is 1e9.

        2) The Julian Date is apportioned in any convenient way between
           the arguments dj1 and dj2.  For example, JD=2450123.7 could
           be expressed in any of these ways, among others:

                  dj1             dj2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           Separating integer and fraction uses the "compensated summation"
           algorithm of Kahan-Neumaier to preserve as much precision as
           possible irrespective of the jd1+jd2 apportionment.

        3) In early eras the conversion is from the "proleptic Gregorian
           calendar";  no account is taken of the date(s) of adoption of
           the Gregorian calendar, nor is the AD/BC numbering convention
           observed.

        References:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992),
           Section 12.92 (p604).

           Klein, A., A Generalized Kahan-Babuska-Summation-Algorithm.
           Computing, 76, 279-293 (2006), Section 3.

        This revision:  2020 October 21

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    iy, im, id, fd, c_retval = ufunc.jd2cal(dj1, dj2)
    check_errwarn(c_retval, 'jd2cal')
    return iy, im, id, fd


STATUS_CODES['jd2cal'] = {
    0: 'OK',
    -1: 'unacceptable date (Note 1)',
}


def jdcalf(ndp, dj1, dj2):
    """
    Julian Date to Gregorian Calendar, expressed in a form convenient
    for formatting messages:  rounded to a specified precision.

    Parameters
    ----------
    ndp : int array
    dj1 : double array
    dj2 : double array

    Returns
    -------
    iymdf : int array

    Notes
    -----
    Wraps ERFA function ``eraJdcalf``. The ERFA documentation is::

        - - - - - - - - - -
         e r a J d c a l f
        - - - - - - - - - -

        Julian Date to Gregorian Calendar, expressed in a form convenient
        for formatting messages:  rounded to a specified precision.

        Given:
           ndp       int      number of decimal places of days in fraction
           dj1,dj2   double   dj1+dj2 = Julian Date (Note 1)

        Returned:
           iymdf     int[4]   year, month, day, fraction in Gregorian
                              calendar

        Returned (function value):
                     int      status:
                                -1 = date out of range
                                 0 = OK
                                +1 = NDP not 0-9 (interpreted as 0)

        Notes:

        1) The Julian Date is apportioned in any convenient way between
           the arguments dj1 and dj2.  For example, JD=2450123.7 could
           be expressed in any of these ways, among others:

                   dj1            dj2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

        2) In early eras the conversion is from the "Proleptic Gregorian
           Calendar";  no account is taken of the date(s) of adoption of
           the Gregorian Calendar, nor is the AD/BC numbering convention
           observed.

        3) See also the function eraJd2cal.

        4) The number of decimal places ndp should be 4 or less if internal
           overflows are to be avoided on platforms which use 16-bit
           integers.

        Called:
           eraJd2cal    JD to Gregorian calendar

        Reference:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992),
           Section 12.92 (p604).

        This revision:  2020 October 22

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    iymdf, c_retval = ufunc.jdcalf(ndp, dj1, dj2)
    check_errwarn(c_retval, 'jdcalf')
    return iymdf


STATUS_CODES['jdcalf'] = {
    -1: 'date out of range',
    0: 'OK',
    1: 'NDP not 0-9 (interpreted as 0)',
}


def ab(pnat, v, s, bm1):
    """
    Apply aberration to transform natural direction into proper
    direction.

    Parameters
    ----------
    pnat : double array
    v : double array
    s : double array
    bm1 : double array

    Returns
    -------
    ppr : double array

    Notes
    -----
    Wraps ERFA function ``eraAb``. The ERFA documentation is::

        - - - - - -
         e r a A b
        - - - - - -

        Apply aberration to transform natural direction into proper
        direction.

        Given:
          pnat    double[3]   natural direction to the source (unit vector)
          v       double[3]   observer barycentric velocity in units of c
          s       double      distance between the Sun and the observer (au)
          bm1     double      sqrt(1-|v|^2): reciprocal of Lorenz factor

        Returned:
          ppr     double[3]   proper direction to source (unit vector)

        Notes:

        1) The algorithm is based on Expr. (7.40) in the Explanatory
           Supplement (Urban & Seidelmann 2013), but with the following
           changes:

           o  Rigorous rather than approximate normalization is applied.

           o  The gravitational potential term from Expr. (7) in
              Klioner (2003) is added, taking into account only the Sun's
              contribution.  This has a maximum effect of about
              0.4 microarcsecond.

        2) In almost all cases, the maximum accuracy will be limited by the
           supplied velocity.  For example, if the ERFA eraEpv00 function is
           used, errors of up to 5 microarcseconds could occur.

        References:

           Urban, S. & Seidelmann, P. K. (eds), Explanatory Supplement to
           the Astronomical Almanac, 3rd ed., University Science Books
           (2013).

           Klioner, Sergei A., "A practical relativistic model for micro-
           arcsecond astrometry in space", Astr. J. 125, 1580-1597 (2003).

        Called:
           eraPdp       scalar product of two p-vectors

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    ppr = ufunc.ab(pnat, v, s, bm1)
    return ppr


def apcg(date1, date2, ebpv, ehp):
    """
    For a geocentric observer, prepare star-independent astrometry
    parameters for transformations between ICRS and GCRS coordinates.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    ebpv : double array
    ehp : double array

    Returns
    -------
    astrom : eraASTROM array

    Notes
    -----
    Wraps ERFA function ``eraApcg``. The ERFA documentation is::

        - - - - - - - -
         e r a A p c g
        - - - - - - - -

        For a geocentric observer, prepare star-independent astrometry
        parameters for transformations between ICRS and GCRS coordinates.
        The Earth ephemeris is supplied by the caller.

        The parameters produced by this function are required in the
        parallax, light deflection and aberration parts of the astrometric
        transformation chain.

        Given:
           date1  double       TDB as a 2-part...
           date2  double       ...Julian Date (Note 1)
           ebpv   double[2][3] Earth barycentric pos/vel (au, au/day)
           ehp    double[3]    Earth heliocentric position (au)

        Returned:
           astrom eraASTROM         star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       unchanged
            xpl    double       unchanged
            ypl    double       unchanged
            sphi   double       unchanged
            cphi   double       unchanged
            diurab double       unchanged
            eral   double       unchanged
            refa   double       unchanged
            refb   double       unchanged

        Notes:

        1) The TDB date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TDB)=2450123.7 could be expressed in any of these ways, among
           others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 method is best matched to the way the
           argument is handled internally and will deliver the optimum
           resolution.  The MJD method and the date & time methods are both
           good compromises between resolution and convenience.  For most
           applications of this function the choice will not be at all
           critical.

           TT can be used instead of TDB without any significant impact on
           accuracy.

        2) All the vectors are with respect to BCRS axes.

        3) This is one of several functions that inserts into the astrom
           structure star-independent parameters needed for the chain of
           astrometric transformations ICRS <-> GCRS <-> CIRS <-> observed.

           The various functions support different classes of observer and
           portions of the transformation chain:

                functions         observer        transformation

             eraApcg eraApcg13    geocentric      ICRS <-> GCRS
             eraApci eraApci13    terrestrial     ICRS <-> CIRS
             eraApco eraApco13    terrestrial     ICRS <-> observed
             eraApcs eraApcs13    space           ICRS <-> GCRS
             eraAper eraAper13    terrestrial     update Earth rotation
             eraApio eraApio13    terrestrial     CIRS <-> observed

           Those with names ending in "13" use contemporary ERFA models to
           compute the various ephemerides.  The others accept ephemerides
           supplied by the caller.

           The transformation from ICRS to GCRS covers space motion,
           parallax, light deflection, and aberration.  From GCRS to CIRS
           comprises frame bias and precession-nutation.  From CIRS to
           observed takes account of Earth rotation, polar motion, diurnal
           aberration and parallax (unless subsumed into the ICRS <-> GCRS
           transformation), and atmospheric refraction.

        4) The context structure astrom produced by this function is used by
           eraAtciq* and eraAticq*.

        Called:
           eraApcs      astrometry parameters, ICRS-GCRS, space observer

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    astrom = ufunc.apcg(date1, date2, ebpv, ehp)
    return astrom


def apcg13(date1, date2):
    """
    For a geocentric observer, prepare star-independent astrometry
    parameters for transformations between ICRS and GCRS coordinates.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    astrom : eraASTROM array

    Notes
    -----
    Wraps ERFA function ``eraApcg13``. The ERFA documentation is::

        - - - - - - - - - -
         e r a A p c g 1 3
        - - - - - - - - - -

        For a geocentric observer, prepare star-independent astrometry
        parameters for transformations between ICRS and GCRS coordinates.
        The caller supplies the date, and ERFA models are used to predict
        the Earth ephemeris.

        The parameters produced by this function are required in the
        parallax, light deflection and aberration parts of the astrometric
        transformation chain.

        Given:
           date1  double     TDB as a 2-part...
           date2  double     ...Julian Date (Note 1)

        Returned:
           astrom eraASTROM* star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       unchanged
            xpl    double       unchanged
            ypl    double       unchanged
            sphi   double       unchanged
            cphi   double       unchanged
            diurab double       unchanged
            eral   double       unchanged
            refa   double       unchanged
            refb   double       unchanged

        Notes:

        1) The TDB date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TDB)=2450123.7 could be expressed in any of these ways, among
           others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 method is best matched to the way the
           argument is handled internally and will deliver the optimum
           resolution.  The MJD method and the date & time methods are both
           good compromises between resolution and convenience.  For most
           applications of this function the choice will not be at all
           critical.

           TT can be used instead of TDB without any significant impact on
           accuracy.

        2) All the vectors are with respect to BCRS axes.

        3) In cases where the caller wishes to supply his own Earth
           ephemeris, the function eraApcg can be used instead of the present
           function.

        4) This is one of several functions that inserts into the astrom
           structure star-independent parameters needed for the chain of
           astrometric transformations ICRS <-> GCRS <-> CIRS <-> observed.

           The various functions support different classes of observer and
           portions of the transformation chain:

                functions         observer        transformation

             eraApcg eraApcg13    geocentric      ICRS <-> GCRS
             eraApci eraApci13    terrestrial     ICRS <-> CIRS
             eraApco eraApco13    terrestrial     ICRS <-> observed
             eraApcs eraApcs13    space           ICRS <-> GCRS
             eraAper eraAper13    terrestrial     update Earth rotation
             eraApio eraApio13    terrestrial     CIRS <-> observed

           Those with names ending in "13" use contemporary ERFA models to
           compute the various ephemerides.  The others accept ephemerides
           supplied by the caller.

           The transformation from ICRS to GCRS covers space motion,
           parallax, light deflection, and aberration.  From GCRS to CIRS
           comprises frame bias and precession-nutation.  From CIRS to
           observed takes account of Earth rotation, polar motion, diurnal
           aberration and parallax (unless subsumed into the ICRS <-> GCRS
           transformation), and atmospheric refraction.

        5) The context structure astrom produced by this function is used by
           eraAtciq* and eraAticq*.

        Called:
           eraEpv00     Earth position and velocity
           eraApcg      astrometry parameters, ICRS-GCRS, geocenter

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    astrom = ufunc.apcg13(date1, date2)
    return astrom


def apci(date1, date2, ebpv, ehp, x, y, s):
    """
    For a terrestrial observer, prepare star-independent astrometry
    parameters for transformations between ICRS and geocentric CIRS
    coordinates.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    ebpv : double array
    ehp : double array
    x : double array
    y : double array
    s : double array

    Returns
    -------
    astrom : eraASTROM array

    Notes
    -----
    Wraps ERFA function ``eraApci``. The ERFA documentation is::

        - - - - - - - -
         e r a A p c i
        - - - - - - - -

        For a terrestrial observer, prepare star-independent astrometry
        parameters for transformations between ICRS and geocentric CIRS
        coordinates.  The Earth ephemeris and CIP/CIO are supplied by the
        caller.

        The parameters produced by this function are required in the
        parallax, light deflection, aberration, and bias-precession-nutation
        parts of the astrometric transformation chain.

        Given:
           date1  double       TDB as a 2-part...
           date2  double       ...Julian Date (Note 1)
           ebpv   double[2][3] Earth barycentric position/velocity (au, au/day)
           ehp    double[3]    Earth heliocentric position (au)
           x,y    double       CIP X,Y (components of unit vector)
           s      double       the CIO locator s (radians)

        Returned:
           astrom eraASTROM         star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       unchanged
            xpl    double       unchanged
            ypl    double       unchanged
            sphi   double       unchanged
            cphi   double       unchanged
            diurab double       unchanged
            eral   double       unchanged
            refa   double       unchanged
            refb   double       unchanged

        Notes:

        1) The TDB date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TDB)=2450123.7 could be expressed in any of these ways, among
           others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 method is best matched to the way the
           argument is handled internally and will deliver the optimum
           resolution.  The MJD method and the date & time methods are both
           good compromises between resolution and convenience.  For most
           applications of this function the choice will not be at all
           critical.

           TT can be used instead of TDB without any significant impact on
           accuracy.

        2) All the vectors are with respect to BCRS axes.

        3) In cases where the caller does not wish to provide the Earth
           ephemeris and CIP/CIO, the function eraApci13 can be used instead
           of the present function.  This computes the required quantities
           using other ERFA functions.

        4) This is one of several functions that inserts into the astrom
           structure star-independent parameters needed for the chain of
           astrometric transformations ICRS <-> GCRS <-> CIRS <-> observed.

           The various functions support different classes of observer and
           portions of the transformation chain:

                functions         observer        transformation

             eraApcg eraApcg13    geocentric      ICRS <-> GCRS
             eraApci eraApci13    terrestrial     ICRS <-> CIRS
             eraApco eraApco13    terrestrial     ICRS <-> observed
             eraApcs eraApcs13    space           ICRS <-> GCRS
             eraAper eraAper13    terrestrial     update Earth rotation
             eraApio eraApio13    terrestrial     CIRS <-> observed

           Those with names ending in "13" use contemporary ERFA models to
           compute the various ephemerides.  The others accept ephemerides
           supplied by the caller.

           The transformation from ICRS to GCRS covers space motion,
           parallax, light deflection, and aberration.  From GCRS to CIRS
           comprises frame bias and precession-nutation.  From CIRS to
           observed takes account of Earth rotation, polar motion, diurnal
           aberration and parallax (unless subsumed into the ICRS <-> GCRS
           transformation), and atmospheric refraction.

        5) The context structure astrom produced by this function is used by
           eraAtciq* and eraAticq*.

        Called:
           eraApcg      astrometry parameters, ICRS-GCRS, geocenter
           eraC2ixys    celestial-to-intermediate matrix, given X,Y and s

        This revision:   2013 September 25

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    astrom = ufunc.apci(date1, date2, ebpv, ehp, x, y, s)
    return astrom


def apci13(date1, date2):
    """
    For a terrestrial observer, prepare star-independent astrometry
    parameters for transformations between ICRS and geocentric CIRS
    coordinates.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    astrom : eraASTROM array
    eo : double array

    Notes
    -----
    Wraps ERFA function ``eraApci13``. The ERFA documentation is::

        - - - - - - - - - -
         e r a A p c i 1 3
        - - - - - - - - - -

        For a terrestrial observer, prepare star-independent astrometry
        parameters for transformations between ICRS and geocentric CIRS
        coordinates.  The caller supplies the date, and ERFA models are used
        to predict the Earth ephemeris and CIP/CIO.

        The parameters produced by this function are required in the
        parallax, light deflection, aberration, and bias-precession-nutation
        parts of the astrometric transformation chain.

        Given:
           date1  double      TDB as a 2-part...
           date2  double      ...Julian Date (Note 1)

        Returned:
           astrom eraASTROM        star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       unchanged
            xpl    double       unchanged
            ypl    double       unchanged
            sphi   double       unchanged
            cphi   double       unchanged
            diurab double       unchanged
            eral   double       unchanged
            refa   double       unchanged
            refb   double       unchanged
           eo     double           equation of the origins (ERA-GST)

        Notes:

        1) The TDB date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TDB)=2450123.7 could be expressed in any of these ways, among
           others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 method is best matched to the way the
           argument is handled internally and will deliver the optimum
           resolution.  The MJD method and the date & time methods are both
           good compromises between resolution and convenience.  For most
           applications of this function the choice will not be at all
           critical.

           TT can be used instead of TDB without any significant impact on
           accuracy.

        2) All the vectors are with respect to BCRS axes.

        3) In cases where the caller wishes to supply his own Earth
           ephemeris and CIP/CIO, the function eraApci can be used instead
           of the present function.

        4) This is one of several functions that inserts into the astrom
           structure star-independent parameters needed for the chain of
           astrometric transformations ICRS <-> GCRS <-> CIRS <-> observed.

           The various functions support different classes of observer and
           portions of the transformation chain:

                functions         observer        transformation

             eraApcg eraApcg13    geocentric      ICRS <-> GCRS
             eraApci eraApci13    terrestrial     ICRS <-> CIRS
             eraApco eraApco13    terrestrial     ICRS <-> observed
             eraApcs eraApcs13    space           ICRS <-> GCRS
             eraAper eraAper13    terrestrial     update Earth rotation
             eraApio eraApio13    terrestrial     CIRS <-> observed

           Those with names ending in "13" use contemporary ERFA models to
           compute the various ephemerides.  The others accept ephemerides
           supplied by the caller.

           The transformation from ICRS to GCRS covers space motion,
           parallax, light deflection, and aberration.  From GCRS to CIRS
           comprises frame bias and precession-nutation.  From CIRS to
           observed takes account of Earth rotation, polar motion, diurnal
           aberration and parallax (unless subsumed into the ICRS <-> GCRS
           transformation), and atmospheric refraction.

        5) The context structure astrom produced by this function is used by
           eraAtciq* and eraAticq*.

        Called:
           eraEpv00     Earth position and velocity
           eraPnm06a    classical NPB matrix, IAU 2006/2000A
           eraBpn2xy    extract CIP X,Y coordinates from NPB matrix
           eraS06       the CIO locator s, given X,Y, IAU 2006
           eraApci      astrometry parameters, ICRS-CIRS
           eraEors      equation of the origins, given NPB matrix and s

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    astrom, eo = ufunc.apci13(date1, date2)
    return astrom, eo


def apco(date1, date2, ebpv, ehp, x, y, s, theta, elong, phi, hm, xp, yp, sp, refa, refb):
    """
    For a terrestrial observer, prepare star-independent astrometry
    parameters for transformations between ICRS and observed
    coordinates.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    ebpv : double array
    ehp : double array
    x : double array
    y : double array
    s : double array
    theta : double array
    elong : double array
    phi : double array
    hm : double array
    xp : double array
    yp : double array
    sp : double array
    refa : double array
    refb : double array

    Returns
    -------
    astrom : eraASTROM array

    Notes
    -----
    Wraps ERFA function ``eraApco``. The ERFA documentation is::

        - - - - - - - -
         e r a A p c o
        - - - - - - - -

        For a terrestrial observer, prepare star-independent astrometry
        parameters for transformations between ICRS and observed
        coordinates.  The caller supplies the Earth ephemeris, the Earth
        rotation information and the refraction constants as well as the
        site coordinates.

        Given:
           date1  double       TDB as a 2-part...
           date2  double       ...Julian Date (Note 1)
           ebpv   double[2][3] Earth barycentric PV (au, au/day, Note 2)
           ehp    double[3]    Earth heliocentric P (au, Note 2)
           x,y    double       CIP X,Y (components of unit vector)
           s      double       the CIO locator s (radians)
           theta  double       Earth rotation angle (radians)
           elong  double       longitude (radians, east +ve, Note 3)
           phi    double       latitude (geodetic, radians, Note 3)
           hm     double       height above ellipsoid (m, geodetic, Note 3)
           xp,yp  double       polar motion coordinates (radians, Note 4)
           sp     double       the TIO locator s' (radians, Note 4)
           refa   double       refraction constant A (radians, Note 5)
           refb   double       refraction constant B (radians, Note 5)

        Returned:
           astrom eraASTROM         star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       adjusted longitude (radians)
            xpl    double       polar motion xp wrt local meridian (radians)
            ypl    double       polar motion yp wrt local meridian (radians)
            sphi   double       sine of geodetic latitude
            cphi   double       cosine of geodetic latitude
            diurab double       magnitude of diurnal aberration vector
            eral   double       "local" Earth rotation angle (radians)
            refa   double       refraction constant A (radians)
            refb   double       refraction constant B (radians)

        Notes:

        1) The TDB date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TDB)=2450123.7 could be expressed in any of these ways, among
           others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 method is best matched to the way the
           argument is handled internally and will deliver the optimum
           resolution.  The MJD method and the date & time methods are both
           good compromises between resolution and convenience.  For most
           applications of this function the choice will not be at all
           critical.

           TT can be used instead of TDB without any significant impact on
           accuracy.

        2) The vectors eb, eh, and all the astrom vectors, are with respect
           to BCRS axes.

        3) The geographical coordinates are with respect to the ERFA_WGS84
           reference ellipsoid.  TAKE CARE WITH THE LONGITUDE SIGN
           CONVENTION:  the longitude required by the present function is
           right-handed, i.e. east-positive, in accordance with geographical
           convention.

           The adjusted longitude stored in the astrom array takes into
           account the TIO locator and polar motion.

        4) xp and yp are the coordinates (in radians) of the Celestial
           Intermediate Pole with respect to the International Terrestrial
           Reference System (see IERS Conventions), measured along the
           meridians 0 and 90 deg west respectively.  sp is the TIO locator
           s', in radians, which positions the Terrestrial Intermediate
           Origin on the equator.  For many applications, xp, yp and
           (especially) sp can be set to zero.

           Internally, the polar motion is stored in a form rotated onto the
           local meridian.

        5) The refraction constants refa and refb are for use in a
           dZ = A*tan(Z)+B*tan^3(Z) model, where Z is the observed
           (i.e. refracted) zenith distance and dZ is the amount of
           refraction.

        6) It is advisable to take great care with units, as even unlikely
           values of the input parameters are accepted and processed in
           accordance with the models used.

        7) In cases where the caller does not wish to provide the Earth
           Ephemeris, the Earth rotation information and refraction
           constants, the function eraApco13 can be used instead of the
           present function.  This starts from UTC and weather readings etc.
           and computes suitable values using other ERFA functions.

        8) This is one of several functions that inserts into the astrom
           structure star-independent parameters needed for the chain of
           astrometric transformations ICRS <-> GCRS <-> CIRS <-> observed.

           The various functions support different classes of observer and
           portions of the transformation chain:

                functions         observer        transformation

             eraApcg eraApcg13    geocentric      ICRS <-> GCRS
             eraApci eraApci13    terrestrial     ICRS <-> CIRS
             eraApco eraApco13    terrestrial     ICRS <-> observed
             eraApcs eraApcs13    space           ICRS <-> GCRS
             eraAper eraAper13    terrestrial     update Earth rotation
             eraApio eraApio13    terrestrial     CIRS <-> observed

           Those with names ending in "13" use contemporary ERFA models to
           compute the various ephemerides.  The others accept ephemerides
           supplied by the caller.

           The transformation from ICRS to GCRS covers space motion,
           parallax, light deflection, and aberration.  From GCRS to CIRS
           comprises frame bias and precession-nutation.  From CIRS to
           observed takes account of Earth rotation, polar motion, diurnal
           aberration and parallax (unless subsumed into the ICRS <-> GCRS
           transformation), and atmospheric refraction.

        9) The context structure astrom produced by this function is used by
           eraAtioq, eraAtoiq, eraAtciq* and eraAticq*.

        Called:
           eraIr        initialize r-matrix to identity
           eraRz        rotate around Z-axis
           eraRy        rotate around Y-axis
           eraRx        rotate around X-axis
           eraAnpm      normalize angle into range +/- pi
           eraC2ixys    celestial-to-intermediate matrix, given X,Y and s
           eraPvtob     position/velocity of terrestrial station
           eraTrxpv     product of transpose of r-matrix and pv-vector
           eraApcs      astrometry parameters, ICRS-GCRS, space observer
           eraCr        copy r-matrix

        This revision:   2021 January 7

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    astrom = ufunc.apco(
        date1, date2, ebpv, ehp, x, y, s, theta, elong, phi, hm, xp, yp, sp, refa, refb)
    return astrom


def apco13(utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl):
    """
    For a terrestrial observer, prepare star-independent astrometry
    parameters for transformations between ICRS and observed
    coordinates.

    Parameters
    ----------
    utc1 : double array
    utc2 : double array
    dut1 : double array
    elong : double array
    phi : double array
    hm : double array
    xp : double array
    yp : double array
    phpa : double array
    tc : double array
    rh : double array
    wl : double array

    Returns
    -------
    astrom : eraASTROM array
    eo : double array

    Notes
    -----
    Wraps ERFA function ``eraApco13``. The ERFA documentation is::

        - - - - - - - - - -
         e r a A p c o 1 3
        - - - - - - - - - -

        For a terrestrial observer, prepare star-independent astrometry
        parameters for transformations between ICRS and observed
        coordinates.  The caller supplies UTC, site coordinates, ambient air
        conditions and observing wavelength, and ERFA models are used to
        obtain the Earth ephemeris, CIP/CIO and refraction constants.

        The parameters produced by this function are required in the
        parallax, light deflection, aberration, and bias-precession-nutation
        parts of the ICRS/CIRS transformations.

        Given:
           utc1   double     UTC as a 2-part...
           utc2   double     ...quasi Julian Date (Notes 1,2)
           dut1   double     UT1-UTC (seconds, Note 3)
           elong  double     longitude (radians, east +ve, Note 4)
           phi    double     latitude (geodetic, radians, Note 4)
           hm     double     height above ellipsoid (m, geodetic, Notes 4,6)
           xp,yp  double     polar motion coordinates (radians, Note 5)
           phpa   double     pressure at the observer (hPa = mB, Note 6)
           tc     double     ambient temperature at the observer (deg C)
           rh     double     relative humidity at the observer (range 0-1)
           wl     double     wavelength (micrometers, Note 7)

        Returned:
           astrom eraASTROM* star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       longitude + s' (radians)
            xpl    double       polar motion xp wrt local meridian (radians)
            ypl    double       polar motion yp wrt local meridian (radians)
            sphi   double       sine of geodetic latitude
            cphi   double       cosine of geodetic latitude
            diurab double       magnitude of diurnal aberration vector
            eral   double       "local" Earth rotation angle (radians)
            refa   double       refraction constant A (radians)
            refb   double       refraction constant B (radians)
           eo     double          equation of the origins (ERA-GST)

        Returned (function value):
                  int        status: +1 = dubious year (Note 2)
                                      0 = OK
                                     -1 = unacceptable date

        Notes:

        1)  utc1+utc2 is quasi Julian Date (see Note 2), apportioned in any
            convenient way between the two arguments, for example where utc1
            is the Julian Day Number and utc2 is the fraction of a day.

            However, JD cannot unambiguously represent UTC during a leap
            second unless special measures are taken.  The convention in the
            present function is that the JD day represents UTC days whether
            the length is 86399, 86400 or 86401 SI seconds.

            Applications should use the function eraDtf2d to convert from
            calendar date and time of day into 2-part quasi Julian Date, as
            it implements the leap-second-ambiguity convention just
            described.

        2)  The warning status "dubious year" flags UTCs that predate the
            introduction of the time scale or that are too far in the
            future to be trusted.  See eraDat for further details.

        3)  UT1-UTC is tabulated in IERS bulletins.  It increases by exactly
            one second at the end of each positive UTC leap second,
            introduced in order to keep UT1-UTC within +/- 0.9s.  n.b. This
            practice is under review, and in the future UT1-UTC may grow
            essentially without limit.

        4)  The geographical coordinates are with respect to the ERFA_WGS84
            reference ellipsoid.  TAKE CARE WITH THE LONGITUDE SIGN:  the
            longitude required by the present function is east-positive
            (i.e. right-handed), in accordance with geographical convention.

        5)  The polar motion xp,yp can be obtained from IERS bulletins.  The
            values are the coordinates (in radians) of the Celestial
            Intermediate Pole with respect to the International Terrestrial
            Reference System (see IERS Conventions 2003), measured along the
            meridians 0 and 90 deg west respectively.  For many
            applications, xp and yp can be set to zero.

            Internally, the polar motion is stored in a form rotated onto
            the local meridian.

        6)  If hm, the height above the ellipsoid of the observing station
            in meters, is not known but phpa, the pressure in hPa (=mB), is
            available, an adequate estimate of hm can be obtained from the
            expression

                  hm = -29.3 * tsl * log ( phpa / 1013.25 );

            where tsl is the approximate sea-level air temperature in K
            (See Astrophysical Quantities, C.W.Allen, 3rd edition, section
            52).  Similarly, if the pressure phpa is not known, it can be
            estimated from the height of the observing station, hm, as
            follows:

                  phpa = 1013.25 * exp ( -hm / ( 29.3 * tsl ) );

            Note, however, that the refraction is nearly proportional to
            the pressure and that an accurate phpa value is important for
            precise work.

        7)  The argument wl specifies the observing wavelength in
            micrometers.  The transition from optical to radio is assumed to
            occur at 100 micrometers (about 3000 GHz).

        8)  It is advisable to take great care with units, as even unlikely
            values of the input parameters are accepted and processed in
            accordance with the models used.

        9)  In cases where the caller wishes to supply his own Earth
            ephemeris, Earth rotation information and refraction constants,
            the function eraApco can be used instead of the present function.

        10) This is one of several functions that inserts into the astrom
            structure star-independent parameters needed for the chain of
            astrometric transformations ICRS <-> GCRS <-> CIRS <-> observed.

            The various functions support different classes of observer and
            portions of the transformation chain:

                functions         observer        transformation

             eraApcg eraApcg13    geocentric      ICRS <-> GCRS
             eraApci eraApci13    terrestrial     ICRS <-> CIRS
             eraApco eraApco13    terrestrial     ICRS <-> observed
             eraApcs eraApcs13    space           ICRS <-> GCRS
             eraAper eraAper13    terrestrial     update Earth rotation
             eraApio eraApio13    terrestrial     CIRS <-> observed

            Those with names ending in "13" use contemporary ERFA models to
            compute the various ephemerides.  The others accept ephemerides
            supplied by the caller.

            The transformation from ICRS to GCRS covers space motion,
            parallax, light deflection, and aberration.  From GCRS to CIRS
            comprises frame bias and precession-nutation.  From CIRS to
            observed takes account of Earth rotation, polar motion, diurnal
            aberration and parallax (unless subsumed into the ICRS <-> GCRS
            transformation), and atmospheric refraction.

        11) The context structure astrom produced by this function is used
            by eraAtioq, eraAtoiq, eraAtciq* and eraAticq*.

        Called:
           eraUtctai    UTC to TAI
           eraTaitt     TAI to TT
           eraUtcut1    UTC to UT1
           eraEpv00     Earth position and velocity
           eraPnm06a    classical NPB matrix, IAU 2006/2000A
           eraBpn2xy    extract CIP X,Y coordinates from NPB matrix
           eraS06       the CIO locator s, given X,Y, IAU 2006
           eraEra00     Earth rotation angle, IAU 2000
           eraSp00      the TIO locator s', IERS 2000
           eraRefco     refraction constants for given ambient conditions
           eraApco      astrometry parameters, ICRS-observed
           eraEors      equation of the origins, given NPB matrix and s

        This revision:   2013 December 5

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    astrom, eo, c_retval = ufunc.apco13(
        utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    check_errwarn(c_retval, 'apco13')
    return astrom, eo


STATUS_CODES['apco13'] = {
    1: 'dubious year (Note 2)',
    0: 'OK',
    -1: 'unacceptable date',
}


def apcs(date1, date2, pv, ebpv, ehp):
    """
    For an observer whose geocentric position and velocity are known,
    prepare star-independent astrometry parameters for transformations
    between ICRS and GCRS.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    pv : double array
    ebpv : double array
    ehp : double array

    Returns
    -------
    astrom : eraASTROM array

    Notes
    -----
    Wraps ERFA function ``eraApcs``. The ERFA documentation is::

        - - - - - - - -
         e r a A p c s
        - - - - - - - -

        For an observer whose geocentric position and velocity are known,
        prepare star-independent astrometry parameters for transformations
        between ICRS and GCRS.  The Earth ephemeris is supplied by the
        caller.

        The parameters produced by this function are required in the space
        motion, parallax, light deflection and aberration parts of the
        astrometric transformation chain.

        Given:
           date1  double       TDB as a 2-part...
           date2  double       ...Julian Date (Note 1)
           pv     double[2][3] observer's geocentric pos/vel (m, m/s)
           ebpv   double[2][3] Earth barycentric PV (au, au/day)
           ehp    double[3]    Earth heliocentric P (au)

        Returned:
           astrom eraASTROM         star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       unchanged
            xpl    double       unchanged
            ypl    double       unchanged
            sphi   double       unchanged
            cphi   double       unchanged
            diurab double       unchanged
            eral   double       unchanged
            refa   double       unchanged
            refb   double       unchanged

        Notes:

        1) The TDB date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TDB)=2450123.7 could be expressed in any of these ways, among
           others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 method is best matched to the way the
           argument is handled internally and will deliver the optimum
           resolution.  The MJD method and the date & time methods are both
           good compromises between resolution and convenience.  For most
           applications of this function the choice will not be at all
           critical.

           TT can be used instead of TDB without any significant impact on
           accuracy.

        2) All the vectors are with respect to BCRS axes.

        3) Providing separate arguments for (i) the observer's geocentric
           position and velocity and (ii) the Earth ephemeris is done for
           convenience in the geocentric, terrestrial and Earth orbit cases.
           For deep space applications it maybe more convenient to specify
           zero geocentric position and velocity and to supply the
           observer's position and velocity information directly instead of
           with respect to the Earth.  However, note the different units:
           m and m/s for the geocentric vectors, au and au/day for the
           heliocentric and barycentric vectors.

        4) In cases where the caller does not wish to provide the Earth
           ephemeris, the function eraApcs13 can be used instead of the
           present function.  This computes the Earth ephemeris using the
           ERFA function eraEpv00.

        5) This is one of several functions that inserts into the astrom
           structure star-independent parameters needed for the chain of
           astrometric transformations ICRS <-> GCRS <-> CIRS <-> observed.

           The various functions support different classes of observer and
           portions of the transformation chain:

                functions         observer        transformation

             eraApcg eraApcg13    geocentric      ICRS <-> GCRS
             eraApci eraApci13    terrestrial     ICRS <-> CIRS
             eraApco eraApco13    terrestrial     ICRS <-> observed
             eraApcs eraApcs13    space           ICRS <-> GCRS
             eraAper eraAper13    terrestrial     update Earth rotation
             eraApio eraApio13    terrestrial     CIRS <-> observed

           Those with names ending in "13" use contemporary ERFA models to
           compute the various ephemerides.  The others accept ephemerides
           supplied by the caller.

           The transformation from ICRS to GCRS covers space motion,
           parallax, light deflection, and aberration.  From GCRS to CIRS
           comprises frame bias and precession-nutation.  From CIRS to
           observed takes account of Earth rotation, polar motion, diurnal
           aberration and parallax (unless subsumed into the ICRS <-> GCRS
           transformation), and atmospheric refraction.

        6) The context structure astrom produced by this function is used by
           eraAtciq* and eraAticq*.

        Called:
           eraCp        copy p-vector
           eraPm        modulus of p-vector
           eraPn        decompose p-vector into modulus and direction
           eraIr        initialize r-matrix to identity

        This revision:   2017 March 16

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    astrom = ufunc.apcs(date1, date2, pv, ebpv, ehp)
    return astrom


def apcs13(date1, date2, pv):
    """
    For an observer whose geocentric position and velocity are known,
    prepare star-independent astrometry parameters for transformations
    between ICRS and GCRS.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    pv : double array

    Returns
    -------
    astrom : eraASTROM array

    Notes
    -----
    Wraps ERFA function ``eraApcs13``. The ERFA documentation is::

        - - - - - - - - - -
         e r a A p c s 1 3
        - - - - - - - - - -

        For an observer whose geocentric position and velocity are known,
        prepare star-independent astrometry parameters for transformations
        between ICRS and GCRS.  The Earth ephemeris is from ERFA models.

        The parameters produced by this function are required in the space
        motion, parallax, light deflection and aberration parts of the
        astrometric transformation chain.

        Given:
           date1  double       TDB as a 2-part...
           date2  double       ...Julian Date (Note 1)
           pv     double[2][3] observer's geocentric pos/vel (Note 3)

        Returned:
           astrom eraASTROM         star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       unchanged
            xpl    double       unchanged
            ypl    double       unchanged
            sphi   double       unchanged
            cphi   double       unchanged
            diurab double       unchanged
            eral   double       unchanged
            refa   double       unchanged
            refb   double       unchanged

        Notes:

        1) The TDB date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TDB)=2450123.7 could be expressed in any of these ways, among
           others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 method is best matched to the way the
           argument is handled internally and will deliver the optimum
           resolution.  The MJD method and the date & time methods are both
           good compromises between resolution and convenience.  For most
           applications of this function the choice will not be at all
           critical.

           TT can be used instead of TDB without any significant impact on
           accuracy.

        2) All the vectors are with respect to BCRS axes.

        3) The observer's position and velocity pv are geocentric but with
           respect to BCRS axes, and in units of m and m/s.  No assumptions
           are made about proximity to the Earth, and the function can be
           used for deep space applications as well as Earth orbit and
           terrestrial.

        4) In cases where the caller wishes to supply his own Earth
           ephemeris, the function eraApcs can be used instead of the present
           function.

        5) This is one of several functions that inserts into the astrom
           structure star-independent parameters needed for the chain of
           astrometric transformations ICRS <-> GCRS <-> CIRS <-> observed.

           The various functions support different classes of observer and
           portions of the transformation chain:

                functions         observer        transformation

             eraApcg eraApcg13    geocentric      ICRS <-> GCRS
             eraApci eraApci13    terrestrial     ICRS <-> CIRS
             eraApco eraApco13    terrestrial     ICRS <-> observed
             eraApcs eraApcs13    space           ICRS <-> GCRS
             eraAper eraAper13    terrestrial     update Earth rotation
             eraApio eraApio13    terrestrial     CIRS <-> observed

           Those with names ending in "13" use contemporary ERFA models to
           compute the various ephemerides.  The others accept ephemerides
           supplied by the caller.

           The transformation from ICRS to GCRS covers space motion,
           parallax, light deflection, and aberration.  From GCRS to CIRS
           comprises frame bias and precession-nutation.  From CIRS to
           observed takes account of Earth rotation, polar motion, diurnal
           aberration and parallax (unless subsumed into the ICRS <-> GCRS
           transformation), and atmospheric refraction.

        6) The context structure astrom produced by this function is used by
           eraAtciq* and eraAticq*.

        Called:
           eraEpv00     Earth position and velocity
           eraApcs      astrometry parameters, ICRS-GCRS, space observer

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    astrom = ufunc.apcs13(date1, date2, pv)
    return astrom


def aper(theta, astrom):
    """
    In the star-independent astrometry parameters, update only the
    Earth rotation angle, supplied by the caller explicitly.

    Parameters
    ----------
    theta : double array
    astrom : eraASTROM array

    Returns
    -------
    astrom : eraASTROM array

    Notes
    -----
    Wraps ERFA function ``eraAper``. Note that, unlike the erfa routine,
    the python wrapper does not change astrom in-place. The ERFA documentation is::

        - - - - - - - -
         e r a A p e r
        - - - - - - - -

        In the star-independent astrometry parameters, update only the
        Earth rotation angle, supplied by the caller explicitly.

        Given:
           theta   double      Earth rotation angle (radians, Note 2)
           astrom  eraASTROM        star-independent astrometry parameters:
            pmt    double       not used
            eb     double[3]    not used
            eh     double[3]    not used
            em     double       not used
            v      double[3]    not used
            bm1    double       not used
            bpn    double[3][3] not used
            along  double       longitude + s' (radians)
            xpl    double       not used
            ypl    double       not used
            sphi   double       not used
            cphi   double       not used
            diurab double       not used
            eral   double       not used
            refa   double       not used
            refb   double       not used

        Returned:
           astrom  eraASTROM        star-independent astrometry parameters:
            pmt    double       unchanged
            eb     double[3]    unchanged
            eh     double[3]    unchanged
            em     double       unchanged
            v      double[3]    unchanged
            bm1    double       unchanged
            bpn    double[3][3] unchanged
            along  double       unchanged
            xpl    double       unchanged
            ypl    double       unchanged
            sphi   double       unchanged
            cphi   double       unchanged
            diurab double       unchanged
            eral   double       "local" Earth rotation angle (radians)
            refa   double       unchanged
            refb   double       unchanged

        Notes:

        1) This function exists to enable sidereal-tracking applications to
           avoid wasteful recomputation of the bulk of the astrometry
           parameters:  only the Earth rotation is updated.

        2) For targets expressed as equinox based positions, such as
           classical geocentric apparent (RA,Dec), the supplied theta can be
           Greenwich apparent sidereal time rather than Earth rotation
           angle.

        3) The function eraAper13 can be used instead of the present
           function, and starts from UT1 rather than ERA itself.

        4) This is one of several functions that inserts into the astrom
           structure star-independent parameters needed for the chain of
           astrometric transformations ICRS <-> GCRS <-> CIRS <-> observed.

           The various functions support different classes of observer and
           portions of the transformation chain:

                functions         observer        transformation

             eraApcg eraApcg13    geocentric      ICRS <-> GCRS
             eraApci eraApci13    terrestrial     ICRS <-> CIRS
             eraApco eraApco13    terrestrial     ICRS <-> observed
             eraApcs eraApcs13    space           ICRS <-> GCRS
             eraAper eraAper13    terrestrial     update Earth rotation
             eraApio eraApio13    terrestrial     CIRS <-> observed

           Those with names ending in "13" use contemporary ERFA models to
           compute the various ephemerides.  The others accept ephemerides
           supplied by the caller.

           The transformation from ICRS to GCRS covers space motion,
           parallax, light deflection, and aberration.  From GCRS to CIRS
           comprises frame bias and precession-nutation.  From CIRS to
           observed takes account of Earth rotation, polar motion, diurnal
           aberration and parallax (unless subsumed into the ICRS <-> GCRS
           transformation), and atmospheric refraction.

        This revision:   2013 September 25

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    astrom = ufunc.aper(theta, astrom)
    return astrom


def aper13(ut11, ut12, astrom):
    """
    In the star-independent astrometry parameters, update only the
    Earth rotation angle.

    Parameters
    ----------
    ut11 : double array
    ut12 : double array
    astrom : eraASTROM array

    Returns
    -------
    astrom : eraASTROM array

    Notes
    -----
    Wraps ERFA function ``eraAper13``. Note that, unlike the erfa routine,
    the python wrapper does not change astrom in-place. The ERFA documentation is::

        - - - - - - - - - -
         e r a A p e r 1 3
        - - - - - - - - - -

        In the star-independent astrometry parameters, update only the
        Earth rotation angle.  The caller provides UT1, (n.b. not UTC).

        Given:
           ut11    double      UT1 as a 2-part...
           ut12    double      ...Julian Date (Note 1)
           astrom  eraASTROM        star-independent astrometry parameters:
            pmt    double       not used
            eb     double[3]    not used
            eh     double[3]    not used
            em     double       not used
            v      double[3]    not used
            bm1    double       not used
            bpn    double[3][3] not used
            along  double       longitude + s' (radians)
            xpl    double       not used
            ypl    double       not used
            sphi   double       not used
            cphi   double       not used
            diurab double       not used
            eral   double       not used
            refa   double       not used
            refb   double       not used

        Returned:
           astrom  eraASTROM        star-independent astrometry parameters:
            pmt    double       unchanged
            eb     double[3]    unchanged
            eh     double[3]    unchanged
            em     double       unchanged
            v      double[3]    unchanged
            bm1    double       unchanged
            bpn    double[3][3] unchanged
            along  double       unchanged
            xpl    double       unchanged
            ypl    double       unchanged
            sphi   double       unchanged
            cphi   double       unchanged
            diurab double       unchanged
            eral   double       "local" Earth rotation angle (radians)
            refa   double       unchanged
            refb   double       unchanged

        Notes:

        1) The UT1 date (n.b. not UTC) ut11+ut12 is a Julian Date,
           apportioned in any convenient way between the arguments ut11 and
           ut12.  For example, JD(UT1)=2450123.7 could be expressed in any
           of these ways, among others:

                  ut11           ut12

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 and MJD methods are good compromises
           between resolution and convenience.  The date & time method is
           best matched to the algorithm used:  maximum precision is
           delivered when the ut11 argument is for 0hrs UT1 on the day in
           question and the ut12 argument lies in the range 0 to 1, or vice
           versa.

        2) If the caller wishes to provide the Earth rotation angle itself,
           the function eraAper can be used instead.  One use of this
           technique is to substitute Greenwich apparent sidereal time and
           thereby to support equinox based transformations directly.

        3) This is one of several functions that inserts into the astrom
           structure star-independent parameters needed for the chain of
           astrometric transformations ICRS <-> GCRS <-> CIRS <-> observed.

           The various functions support different classes of observer and
           portions of the transformation chain:

                functions         observer        transformation

             eraApcg eraApcg13    geocentric      ICRS <-> GCRS
             eraApci eraApci13    terrestrial     ICRS <-> CIRS
             eraApco eraApco13    terrestrial     ICRS <-> observed
             eraApcs eraApcs13    space           ICRS <-> GCRS
             eraAper eraAper13    terrestrial     update Earth rotation
             eraApio eraApio13    terrestrial     CIRS <-> observed

           Those with names ending in "13" use contemporary ERFA models to
           compute the various ephemerides.  The others accept ephemerides
           supplied by the caller.

           The transformation from ICRS to GCRS covers space motion,
           parallax, light deflection, and aberration.  From GCRS to CIRS
           comprises frame bias and precession-nutation.  From CIRS to
           observed takes account of Earth rotation, polar motion, diurnal
           aberration and parallax (unless subsumed into the ICRS <-> GCRS
           transformation), and atmospheric refraction.

        Called:
           eraAper      astrometry parameters: update ERA
           eraEra00     Earth rotation angle, IAU 2000

        This revision:   2013 September 25

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    astrom = ufunc.aper13(ut11, ut12, astrom)
    return astrom


def apio(sp, theta, elong, phi, hm, xp, yp, refa, refb):
    """
    For a terrestrial observer, prepare star-independent astrometry
    parameters for transformations between CIRS and observed
    coordinates.

    Parameters
    ----------
    sp : double array
    theta : double array
    elong : double array
    phi : double array
    hm : double array
    xp : double array
    yp : double array
    refa : double array
    refb : double array

    Returns
    -------
    astrom : eraASTROM array

    Notes
    -----
    Wraps ERFA function ``eraApio``. The ERFA documentation is::

        - - - - - - - -
         e r a A p i o
        - - - - - - - -

        For a terrestrial observer, prepare star-independent astrometry
        parameters for transformations between CIRS and observed
        coordinates.  The caller supplies the Earth orientation information
        and the refraction constants as well as the site coordinates.

        Given:
           sp     double      the TIO locator s' (radians, Note 1)
           theta  double      Earth rotation angle (radians)
           elong  double      longitude (radians, east +ve, Note 2)
           phi    double      geodetic latitude (radians, Note 2)
           hm     double      height above ellipsoid (m, geodetic Note 2)
           xp,yp  double      polar motion coordinates (radians, Note 3)
           refa   double      refraction constant A (radians, Note 4)
           refb   double      refraction constant B (radians, Note 4)

        Returned:
           astrom eraASTROM        star-independent astrometry parameters:
            pmt    double       unchanged
            eb     double[3]    unchanged
            eh     double[3]    unchanged
            em     double       unchanged
            v      double[3]    unchanged
            bm1    double       unchanged
            bpn    double[3][3] unchanged
            along  double       adjusted longitude (radians)
            xpl    double       polar motion xp wrt local meridian (radians)
            ypl    double       polar motion yp wrt local meridian (radians)
            sphi   double       sine of geodetic latitude
            cphi   double       cosine of geodetic latitude
            diurab double       magnitude of diurnal aberration vector
            eral   double       "local" Earth rotation angle (radians)
            refa   double       refraction constant A (radians)
            refb   double       refraction constant B (radians)

        Notes:

        1) sp, the TIO locator s', is a tiny quantity needed only by the
           most precise applications.  It can either be set to zero or
           predicted using the ERFA function eraSp00.

        2) The geographical coordinates are with respect to the ERFA_WGS84
           reference ellipsoid.  TAKE CARE WITH THE LONGITUDE SIGN:  the
           longitude required by the present function is east-positive
           (i.e. right-handed), in accordance with geographical convention.

        3) The polar motion xp,yp can be obtained from IERS bulletins.  The
           values are the coordinates (in radians) of the Celestial
           Intermediate Pole with respect to the International Terrestrial
           Reference System (see IERS Conventions 2003), measured along the
           meridians 0 and 90 deg west respectively.  For many applications,
           xp and yp can be set to zero.

           Internally, the polar motion is stored in a form rotated onto the
           local meridian.

        4) The refraction constants refa and refb are for use in a
           dZ = A*tan(Z)+B*tan^3(Z) model, where Z is the observed
           (i.e. refracted) zenith distance and dZ is the amount of
           refraction.

        5) It is advisable to take great care with units, as even unlikely
           values of the input parameters are accepted and processed in
           accordance with the models used.

        6) In cases where the caller does not wish to provide the Earth
           rotation information and refraction constants, the function
           eraApio13 can be used instead of the present function.  This
           starts from UTC and weather readings etc. and computes suitable
           values using other ERFA functions.

        7) This is one of several functions that inserts into the astrom
           structure star-independent parameters needed for the chain of
           astrometric transformations ICRS <-> GCRS <-> CIRS <-> observed.

           The various functions support different classes of observer and
           portions of the transformation chain:

                functions         observer        transformation

             eraApcg eraApcg13    geocentric      ICRS <-> GCRS
             eraApci eraApci13    terrestrial     ICRS <-> CIRS
             eraApco eraApco13    terrestrial     ICRS <-> observed
             eraApcs eraApcs13    space           ICRS <-> GCRS
             eraAper eraAper13    terrestrial     update Earth rotation
             eraApio eraApio13    terrestrial     CIRS <-> observed

           Those with names ending in "13" use contemporary ERFA models to
           compute the various ephemerides.  The others accept ephemerides
           supplied by the caller.

           The transformation from ICRS to GCRS covers space motion,
           parallax, light deflection, and aberration.  From GCRS to CIRS
           comprises frame bias and precession-nutation.  From CIRS to
           observed takes account of Earth rotation, polar motion, diurnal
           aberration and parallax (unless subsumed into the ICRS <-> GCRS
           transformation), and atmospheric refraction.

        8) The context structure astrom produced by this function is used by
           eraAtioq and eraAtoiq.

        Called:
           eraIr        initialize r-matrix to identity
           eraRz        rotate around Z-axis
           eraRy        rotate around Y-axis
           eraRx        rotate around X-axis
           eraAnpm      normalize angle into range +/- pi
           eraPvtob     position/velocity of terrestrial station

        This revision:   2021 January 7

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    astrom = ufunc.apio(sp, theta, elong, phi, hm, xp, yp, refa, refb)
    return astrom


def apio13(utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl):
    """
    For a terrestrial observer, prepare star-independent astrometry
    parameters for transformations between CIRS and observed
    coordinates.

    Parameters
    ----------
    utc1 : double array
    utc2 : double array
    dut1 : double array
    elong : double array
    phi : double array
    hm : double array
    xp : double array
    yp : double array
    phpa : double array
    tc : double array
    rh : double array
    wl : double array

    Returns
    -------
    astrom : eraASTROM array

    Notes
    -----
    Wraps ERFA function ``eraApio13``. The ERFA documentation is::

        - - - - - - - - - -
         e r a A p i o 1 3
        - - - - - - - - - -

        For a terrestrial observer, prepare star-independent astrometry
        parameters for transformations between CIRS and observed
        coordinates.  The caller supplies UTC, site coordinates, ambient air
        conditions and observing wavelength.

        Given:
           utc1   double      UTC as a 2-part...
           utc2   double      ...quasi Julian Date (Notes 1,2)
           dut1   double      UT1-UTC (seconds)
           elong  double      longitude (radians, east +ve, Note 3)
           phi    double      geodetic latitude (radians, Note 3)
           hm     double      height above ellipsoid (m, geodetic Notes 4,6)
           xp,yp  double      polar motion coordinates (radians, Note 5)
           phpa   double      pressure at the observer (hPa = mB, Note 6)
           tc     double      ambient temperature at the observer (deg C)
           rh     double      relative humidity at the observer (range 0-1)
           wl     double      wavelength (micrometers, Note 7)

        Returned:
           astrom eraASTROM        star-independent astrometry parameters:
            pmt    double       unchanged
            eb     double[3]    unchanged
            eh     double[3]    unchanged
            em     double       unchanged
            v      double[3]    unchanged
            bm1    double       unchanged
            bpn    double[3][3] unchanged
            along  double       longitude + s' (radians)
            xpl    double       polar motion xp wrt local meridian (radians)
            ypl    double       polar motion yp wrt local meridian (radians)
            sphi   double       sine of geodetic latitude
            cphi   double       cosine of geodetic latitude
            diurab double       magnitude of diurnal aberration vector
            eral   double       "local" Earth rotation angle (radians)
            refa   double       refraction constant A (radians)
            refb   double       refraction constant B (radians)

        Returned (function value):
                  int         status: +1 = dubious year (Note 2)
                                       0 = OK
                                      -1 = unacceptable date

        Notes:

        1)  utc1+utc2 is quasi Julian Date (see Note 2), apportioned in any
            convenient way between the two arguments, for example where utc1
            is the Julian Day Number and utc2 is the fraction of a day.

            However, JD cannot unambiguously represent UTC during a leap
            second unless special measures are taken.  The convention in the
            present function is that the JD day represents UTC days whether
            the length is 86399, 86400 or 86401 SI seconds.

            Applications should use the function eraDtf2d to convert from
            calendar date and time of day into 2-part quasi Julian Date, as
            it implements the leap-second-ambiguity convention just
            described.

        2)  The warning status "dubious year" flags UTCs that predate the
            introduction of the time scale or that are too far in the future
            to be trusted.  See eraDat for further details.

        3)  UT1-UTC is tabulated in IERS bulletins.  It increases by exactly
            one second at the end of each positive UTC leap second,
            introduced in order to keep UT1-UTC within +/- 0.9s.  n.b. This
            practice is under review, and in the future UT1-UTC may grow
            essentially without limit.

        4)  The geographical coordinates are with respect to the ERFA_WGS84
            reference ellipsoid.  TAKE CARE WITH THE LONGITUDE SIGN:  the
            longitude required by the present function is east-positive
            (i.e. right-handed), in accordance with geographical convention.

        5)  The polar motion xp,yp can be obtained from IERS bulletins.  The
            values are the coordinates (in radians) of the Celestial
            Intermediate Pole with respect to the International Terrestrial
            Reference System (see IERS Conventions 2003), measured along the
            meridians 0 and 90 deg west respectively.  For many applications,
            xp and yp can be set to zero.

            Internally, the polar motion is stored in a form rotated onto
            the local meridian.

        6)  If hm, the height above the ellipsoid of the observing station
            in meters, is not known but phpa, the pressure in hPa (=mB), is
            available, an adequate estimate of hm can be obtained from the
            expression

                  hm = -29.3 * tsl * log ( phpa / 1013.25 );

            where tsl is the approximate sea-level air temperature in K
            (See Astrophysical Quantities, C.W.Allen, 3rd edition, section
            52).  Similarly, if the pressure phpa is not known, it can be
            estimated from the height of the observing station, hm, as
            follows:

                  phpa = 1013.25 * exp ( -hm / ( 29.3 * tsl ) );

            Note, however, that the refraction is nearly proportional to the
            pressure and that an accurate phpa value is important for
            precise work.

        7)  The argument wl specifies the observing wavelength in
            micrometers.  The transition from optical to radio is assumed to
            occur at 100 micrometers (about 3000 GHz).

        8)  It is advisable to take great care with units, as even unlikely
            values of the input parameters are accepted and processed in
            accordance with the models used.

        9)  In cases where the caller wishes to supply his own Earth
            rotation information and refraction constants, the function
            eraApc can be used instead of the present function.

        10) This is one of several functions that inserts into the astrom
            structure star-independent parameters needed for the chain of
            astrometric transformations ICRS <-> GCRS <-> CIRS <-> observed.

            The various functions support different classes of observer and
            portions of the transformation chain:

                functions         observer        transformation

             eraApcg eraApcg13    geocentric      ICRS <-> GCRS
             eraApci eraApci13    terrestrial     ICRS <-> CIRS
             eraApco eraApco13    terrestrial     ICRS <-> observed
             eraApcs eraApcs13    space           ICRS <-> GCRS
             eraAper eraAper13    terrestrial     update Earth rotation
             eraApio eraApio13    terrestrial     CIRS <-> observed

            Those with names ending in "13" use contemporary ERFA models to
            compute the various ephemerides.  The others accept ephemerides
            supplied by the caller.

            The transformation from ICRS to GCRS covers space motion,
            parallax, light deflection, and aberration.  From GCRS to CIRS
            comprises frame bias and precession-nutation.  From CIRS to
            observed takes account of Earth rotation, polar motion, diurnal
            aberration and parallax (unless subsumed into the ICRS <-> GCRS
            transformation), and atmospheric refraction.

        11) The context structure astrom produced by this function is used
            by eraAtioq and eraAtoiq.

        Called:
           eraUtctai    UTC to TAI
           eraTaitt     TAI to TT
           eraUtcut1    UTC to UT1
           eraSp00      the TIO locator s', IERS 2000
           eraEra00     Earth rotation angle, IAU 2000
           eraRefco     refraction constants for given ambient conditions
           eraApio      astrometry parameters, CIRS-observed

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    astrom, c_retval = ufunc.apio13(
        utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    check_errwarn(c_retval, 'apio13')
    return astrom


STATUS_CODES['apio13'] = {
    1: 'dubious year (Note 2)',
    0: 'OK',
    -1: 'unacceptable date',
}


def atci13(rc, dc, pr, pd, px, rv, date1, date2):
    """
    Transform ICRS star data, epoch J2000.0, to CIRS.

    Parameters
    ----------
    rc : double array
    dc : double array
    pr : double array
    pd : double array
    px : double array
    rv : double array
    date1 : double array
    date2 : double array

    Returns
    -------
    ri : double array
    di : double array
    eo : double array

    Notes
    -----
    Wraps ERFA function ``eraAtci13``. The ERFA documentation is::

        - - - - - - - - - -
         e r a A t c i 1 3
        - - - - - - - - - -

        Transform ICRS star data, epoch J2000.0, to CIRS.

        Given:
           rc     double   ICRS right ascension at J2000.0 (radians, Note 1)
           dc     double   ICRS declination at J2000.0 (radians, Note 1)
           pr     double   RA proper motion (radians/year; Note 2)
           pd     double   Dec proper motion (radians/year)
           px     double   parallax (arcsec)
           rv     double   radial velocity (km/s, +ve if receding)
           date1  double   TDB as a 2-part...
           date2  double   ...Julian Date (Note 3)

        Returned:
           ri,di  double        CIRS geocentric RA,Dec (radians)
           eo     double        equation of the origins (ERA-GST, Note 5)

        Notes:

        1) Star data for an epoch other than J2000.0 (for example from the
           Hipparcos catalog, which has an epoch of J1991.25) will require a
           preliminary call to eraPmsafe before use.

        2) The proper motion in RA is dRA/dt rather than cos(Dec)*dRA/dt.

        3) The TDB date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TDB)=2450123.7 could be expressed in any of these ways, among
           others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 method is best matched to the way the
           argument is handled internally and will deliver the optimum
           resolution.  The MJD method and the date & time methods are both
           good compromises between resolution and convenience.  For most
           applications of this function the choice will not be at all
           critical.

           TT can be used instead of TDB without any significant impact on
           accuracy.

        4) The available accuracy is better than 1 milliarcsecond, limited
           mainly by the precession-nutation model that is used, namely
           IAU 2000A/2006.  Very close to solar system bodies, additional
           errors of up to several milliarcseconds can occur because of
           unmodeled light deflection;  however, the Sun's contribution is
           taken into account, to first order.  The accuracy limitations of
           the ERFA function eraEpv00 (used to compute Earth position and
           velocity) can contribute aberration errors of up to
           5 microarcseconds.  Light deflection at the Sun's limb is
           uncertain at the 0.4 mas level.

        5) Should the transformation to (equinox based) apparent place be
           required rather than (CIO based) intermediate place, subtract the
           equation of the origins from the returned right ascension:
           RA = RI - EO. (The eraAnp function can then be applied, as
           required, to keep the result in the conventional 0-2pi range.)

        Called:
           eraApci13    astrometry parameters, ICRS-CIRS, 2013
           eraAtciq     quick ICRS to CIRS

        This revision:   2017 March 12

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    ri, di, eo = ufunc.atci13(rc, dc, pr, pd, px, rv, date1, date2)
    return ri, di, eo


def atciq(rc, dc, pr, pd, px, rv, astrom):
    """
    Quick ICRS, epoch J2000.0, to CIRS transformation, given precomputed
    star-independent astrometry parameters.

    Parameters
    ----------
    rc : double array
    dc : double array
    pr : double array
    pd : double array
    px : double array
    rv : double array
    astrom : eraASTROM array

    Returns
    -------
    ri : double array
    di : double array

    Notes
    -----
    Wraps ERFA function ``eraAtciq``. The ERFA documentation is::

        - - - - - - - - -
         e r a A t c i q
        - - - - - - - - -

        Quick ICRS, epoch J2000.0, to CIRS transformation, given precomputed
        star-independent astrometry parameters.

        Use of this function is appropriate when efficiency is important and
        where many star positions are to be transformed for one date.  The
        star-independent parameters can be obtained by calling one of the
        functions eraApci[13], eraApcg[13], eraApco[13] or eraApcs[13].

        If the parallax and proper motions are zero the eraAtciqz function
        can be used instead.

        Given:
           rc,dc  double     ICRS RA,Dec at J2000.0 (radians)
           pr     double     RA proper motion (radians/year; Note 3)
           pd     double     Dec proper motion (radians/year)
           px     double     parallax (arcsec)
           rv     double     radial velocity (km/s, +ve if receding)
           astrom eraASTROM* star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       longitude + s' (radians)
            xpl    double       polar motion xp wrt local meridian (radians)
            ypl    double       polar motion yp wrt local meridian (radians)
            sphi   double       sine of geodetic latitude
            cphi   double       cosine of geodetic latitude
            diurab double       magnitude of diurnal aberration vector
            eral   double       "local" Earth rotation angle (radians)
            refa   double       refraction constant A (radians)
            refb   double       refraction constant B (radians)

        Returned:
           ri,di   double    CIRS RA,Dec (radians)

        Notes:

        1) All the vectors are with respect to BCRS axes.

        2) Star data for an epoch other than J2000.0 (for example from the
           Hipparcos catalog, which has an epoch of J1991.25) will require a
           preliminary call to eraPmsafe before use.

        3) The proper motion in RA is dRA/dt rather than cos(Dec)*dRA/dt.

        Called:
           eraPmpx      proper motion and parallax
           eraLdsun     light deflection by the Sun
           eraAb        stellar aberration
           eraRxp       product of r-matrix and pv-vector
           eraC2s       p-vector to spherical
           eraAnp       normalize angle into range 0 to 2pi

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    ri, di = ufunc.atciq(rc, dc, pr, pd, px, rv, astrom)
    return ri, di


def atciqn(rc, dc, pr, pd, px, rv, astrom, b):
    """
    Quick ICRS, epoch J2000.0, to CIRS transformation, given precomputed
    star-independent astrometry parameters plus a list of light-
    deflecting bodies.

    Parameters
    ----------
    rc : double array
    dc : double array
    pr : double array
    pd : double array
    px : double array
    rv : double array
    astrom : eraASTROM array
    b : eraLDBODY array

    Returns
    -------
    ri : double array
    di : double array

    Notes
    -----
    Wraps ERFA function ``eraAtciqn``. The ERFA documentation is::

        - - - - - - - - - -
         e r a A t c i q n
        - - - - - - - - - -

        Quick ICRS, epoch J2000.0, to CIRS transformation, given precomputed
        star-independent astrometry parameters plus a list of light-
        deflecting bodies.

        Use of this function is appropriate when efficiency is important and
        where many star positions are to be transformed for one date.  The
        star-independent parameters can be obtained by calling one of the
        functions eraApci[13], eraApcg[13], eraApco[13] or eraApcs[13].


        If the only light-deflecting body to be taken into account is the
        Sun, the eraAtciq function can be used instead.  If in addition the
        parallax and proper motions are zero, the eraAtciqz function can be
        used.

        Given:
           rc,dc  double       ICRS RA,Dec at J2000.0 (radians)
           pr     double       RA proper motion (radians/year; Note 3)
           pd     double       Dec proper motion (radians/year)
           px     double       parallax (arcsec)
           rv     double       radial velocity (km/s, +ve if receding)
           astrom eraASTROM         star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       longitude + s' (radians)
            xpl    double       polar motion xp wrt local meridian (radians)
            ypl    double       polar motion yp wrt local meridian (radians)
            sphi   double       sine of geodetic latitude
            cphi   double       cosine of geodetic latitude
            diurab double       magnitude of diurnal aberration vector
            eral   double       "local" Earth rotation angle (radians)
            refa   double       refraction constant A (radians)
            refb   double       refraction constant B (radians)
           n      int          number of bodies (Note 3)
           b      eraLDBODY[n] data for each of the n bodies (Notes 3,4):
            bm     double       mass of the body (solar masses, Note 5)
            dl     double       deflection limiter (Note 6)
            pv     [2][3]       barycentric PV of the body (au, au/day)

        Returned:
           ri,di   double    CIRS RA,Dec (radians)

        Notes:

        1) Star data for an epoch other than J2000.0 (for example from the
           Hipparcos catalog, which has an epoch of J1991.25) will require a
           preliminary call to eraPmsafe before use.

        2) The proper motion in RA is dRA/dt rather than cos(Dec)*dRA/dt.

        3) The struct b contains n entries, one for each body to be
           considered.  If n = 0, no gravitational light deflection will be
           applied, not even for the Sun.

        4) The struct b should include an entry for the Sun as well as for
           any planet or other body to be taken into account.  The entries
           should be in the order in which the light passes the body.

        5) In the entry in the b struct for body i, the mass parameter
           b[i].bm can, as required, be adjusted in order to allow for such
           effects as quadrupole field.

        6) The deflection limiter parameter b[i].dl is phi^2/2, where phi is
           the angular separation (in radians) between star and body at
           which limiting is applied.  As phi shrinks below the chosen
           threshold, the deflection is artificially reduced, reaching zero
           for phi = 0.   Example values suitable for a terrestrial
           observer, together with masses, are as follows:

              body i     b[i].bm        b[i].dl

              Sun        1.0            6e-6
              Jupiter    0.00095435     3e-9
              Saturn     0.00028574     3e-10

        7) For efficiency, validation of the contents of the b array is
           omitted.  The supplied masses must be greater than zero, the
           position and velocity vectors must be right, and the deflection
           limiter greater than zero.

        Called:
           eraPmpx      proper motion and parallax
           eraLdn       light deflection by n bodies
           eraAb        stellar aberration
           eraRxp       product of r-matrix and pv-vector
           eraC2s       p-vector to spherical
           eraAnp       normalize angle into range 0 to 2pi

        This revision:   2021 January 6

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    ri, di = ufunc.atciqn(rc, dc, pr, pd, px, rv, astrom, b)
    return ri, di


def atciqz(rc, dc, astrom):
    """
    Quick ICRS to CIRS transformation, given precomputed star-
    independent astrometry parameters, and assuming zero parallax and
    proper motion.

    Parameters
    ----------
    rc : double array
    dc : double array
    astrom : eraASTROM array

    Returns
    -------
    ri : double array
    di : double array

    Notes
    -----
    Wraps ERFA function ``eraAtciqz``. The ERFA documentation is::

        - - - - - - - - - -
         e r a A t c i q z
        - - - - - - - - - -

        Quick ICRS to CIRS transformation, given precomputed star-
        independent astrometry parameters, and assuming zero parallax and
        proper motion.

        Use of this function is appropriate when efficiency is important and
        where many star positions are to be transformed for one date.  The
        star-independent parameters can be obtained by calling one of the
        functions eraApci[13], eraApcg[13], eraApco[13] or eraApcs[13].

        The corresponding function for the case of non-zero parallax and
        proper motion is eraAtciq.

        Given:
           rc,dc  double     ICRS astrometric RA,Dec (radians)
           astrom eraASTROM* star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       longitude + s' (radians)
            xpl    double       polar motion xp wrt local meridian (radians)
            ypl    double       polar motion yp wrt local meridian (radians)
            sphi   double       sine of geodetic latitude
            cphi   double       cosine of geodetic latitude
            diurab double       magnitude of diurnal aberration vector
            eral   double       "local" Earth rotation angle (radians)
            refa   double       refraction constant A (radians)
            refb   double       refraction constant B (radians)

        Returned:
           ri,di  double     CIRS RA,Dec (radians)

        Note:

           All the vectors are with respect to BCRS axes.

        References:

           Urban, S. & Seidelmann, P. K. (eds), Explanatory Supplement to
           the Astronomical Almanac, 3rd ed., University Science Books
           (2013).

           Klioner, Sergei A., "A practical relativistic model for micro-
           arcsecond astrometry in space", Astr. J. 125, 1580-1597 (2003).

        Called:
           eraS2c       spherical coordinates to unit vector
           eraLdsun     light deflection due to Sun
           eraAb        stellar aberration
           eraRxp       product of r-matrix and p-vector
           eraC2s       p-vector to spherical
           eraAnp       normalize angle into range +/- pi

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    ri, di = ufunc.atciqz(rc, dc, astrom)
    return ri, di


def atco13(rc, dc, pr, pd, px, rv, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl):
    """
    ICRS RA,Dec to observed place.  The caller supplies UTC, site
    coordinates, ambient air conditions and observing wavelength.

    Parameters
    ----------
    rc : double array
    dc : double array
    pr : double array
    pd : double array
    px : double array
    rv : double array
    utc1 : double array
    utc2 : double array
    dut1 : double array
    elong : double array
    phi : double array
    hm : double array
    xp : double array
    yp : double array
    phpa : double array
    tc : double array
    rh : double array
    wl : double array

    Returns
    -------
    aob : double array
    zob : double array
    hob : double array
    dob : double array
    rob : double array
    eo : double array

    Notes
    -----
    Wraps ERFA function ``eraAtco13``. The ERFA documentation is::

        - - - - - - - - - -
         e r a A t c o 1 3
        - - - - - - - - - -

        ICRS RA,Dec to observed place.  The caller supplies UTC, site
        coordinates, ambient air conditions and observing wavelength.

        ERFA models are used for the Earth ephemeris, bias-precession-
        nutation, Earth orientation and refraction.

        Given:
           rc,dc  double   ICRS right ascension at J2000.0 (radians, Note 1)
           pr     double   RA proper motion (radians/year; Note 2)
           pd     double   Dec proper motion (radians/year)
           px     double   parallax (arcsec)
           rv     double   radial velocity (km/s, +ve if receding)
           utc1   double   UTC as a 2-part...
           utc2   double   ...quasi Julian Date (Notes 3-4)
           dut1   double   UT1-UTC (seconds, Note 5)
           elong  double   longitude (radians, east +ve, Note 6)
           phi    double   latitude (geodetic, radians, Note 6)
           hm     double   height above ellipsoid (m, geodetic, Notes 6,8)
           xp,yp  double   polar motion coordinates (radians, Note 7)
           phpa   double   pressure at the observer (hPa = mB, Note 8)
           tc     double   ambient temperature at the observer (deg C)
           rh     double   relative humidity at the observer (range 0-1)
           wl     double   wavelength (micrometers, Note 9)

        Returned:
           aob    double        observed azimuth (radians: N=0,E=90)
           zob    double        observed zenith distance (radians)
           hob    double        observed hour angle (radians)
           dob    double        observed declination (radians)
           rob    double        observed right ascension (CIO-based, radians)
           eo     double        equation of the origins (ERA-GST)

        Returned (function value):
                  int      status: +1 = dubious year (Note 4)
                                    0 = OK
                                   -1 = unacceptable date

        Notes:

        1)  Star data for an epoch other than J2000.0 (for example from the
            Hipparcos catalog, which has an epoch of J1991.25) will require
            a preliminary call to eraPmsafe before use.

        2)  The proper motion in RA is dRA/dt rather than cos(Dec)*dRA/dt.

        3)  utc1+utc2 is quasi Julian Date (see Note 2), apportioned in any
            convenient way between the two arguments, for example where utc1
            is the Julian Day Number and utc2 is the fraction of a day.

            However, JD cannot unambiguously represent UTC during a leap
            second unless special measures are taken.  The convention in the
            present function is that the JD day represents UTC days whether
            the length is 86399, 86400 or 86401 SI seconds.

            Applications should use the function eraDtf2d to convert from
            calendar date and time of day into 2-part quasi Julian Date, as
            it implements the leap-second-ambiguity convention just
            described.

        4)  The warning status "dubious year" flags UTCs that predate the
            introduction of the time scale or that are too far in the
            future to be trusted.  See eraDat for further details.

        5)  UT1-UTC is tabulated in IERS bulletins.  It increases by exactly
            one second at the end of each positive UTC leap second,
            introduced in order to keep UT1-UTC within +/- 0.9s.  n.b. This
            practice is under review, and in the future UT1-UTC may grow
            essentially without limit.

        6)  The geographical coordinates are with respect to the ERFA_WGS84
            reference ellipsoid.  TAKE CARE WITH THE LONGITUDE SIGN:  the
            longitude required by the present function is east-positive
            (i.e. right-handed), in accordance with geographical convention.

        7)  The polar motion xp,yp can be obtained from IERS bulletins.  The
            values are the coordinates (in radians) of the Celestial
            Intermediate Pole with respect to the International Terrestrial
            Reference System (see IERS Conventions 2003), measured along the
            meridians 0 and 90 deg west respectively.  For many
            applications, xp and yp can be set to zero.

        8)  If hm, the height above the ellipsoid of the observing station
            in meters, is not known but phpa, the pressure in hPa (=mB),
            is available, an adequate estimate of hm can be obtained from
            the expression

                  hm = -29.3 * tsl * log ( phpa / 1013.25 );

            where tsl is the approximate sea-level air temperature in K
            (See Astrophysical Quantities, C.W.Allen, 3rd edition, section
            52).  Similarly, if the pressure phpa is not known, it can be
            estimated from the height of the observing station, hm, as
            follows:

                  phpa = 1013.25 * exp ( -hm / ( 29.3 * tsl ) );

            Note, however, that the refraction is nearly proportional to
            the pressure and that an accurate phpa value is important for
            precise work.

        9)  The argument wl specifies the observing wavelength in
            micrometers.  The transition from optical to radio is assumed to
            occur at 100 micrometers (about 3000 GHz).

        10) The accuracy of the result is limited by the corrections for
            refraction, which use a simple A*tan(z) + B*tan^3(z) model.
            Providing the meteorological parameters are known accurately and
            there are no gross local effects, the predicted observed
            coordinates should be within 0.05 arcsec (optical) or 1 arcsec
            (radio) for a zenith distance of less than 70 degrees, better
            than 30 arcsec (optical or radio) at 85 degrees and better
            than 20 arcmin (optical) or 30 arcmin (radio) at the horizon.

            Without refraction, the complementary functions eraAtco13 and
            eraAtoc13 are self-consistent to better than 1 microarcsecond
            all over the celestial sphere.  With refraction included,
            consistency falls off at high zenith distances, but is still
            better than 0.05 arcsec at 85 degrees.

        11) "Observed" Az,ZD means the position that would be seen by a
            perfect geodetically aligned theodolite.  (Zenith distance is
            used rather than altitude in order to reflect the fact that no
            allowance is made for depression of the horizon.)  This is
            related to the observed HA,Dec via the standard rotation, using
            the geodetic latitude (corrected for polar motion), while the
            observed HA and RA are related simply through the Earth rotation
            angle and the site longitude.  "Observed" RA,Dec or HA,Dec thus
            means the position that would be seen by a perfect equatorial
            with its polar axis aligned to the Earth's axis of rotation.

        12) It is advisable to take great care with units, as even unlikely
            values of the input parameters are accepted and processed in
            accordance with the models used.

        Called:
           eraApco13    astrometry parameters, ICRS-observed, 2013
           eraAtciq     quick ICRS to CIRS
           eraAtioq     quick CIRS to observed

        This revision:   2016 February 2

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    aob, zob, hob, dob, rob, eo, c_retval = ufunc.atco13(
        rc, dc, pr, pd, px, rv, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    check_errwarn(c_retval, 'atco13')
    return aob, zob, hob, dob, rob, eo


STATUS_CODES['atco13'] = {
    1: 'dubious year (Note 4)',
    0: 'OK',
    -1: 'unacceptable date',
}


def atic13(ri, di, date1, date2):
    """
    Transform star RA,Dec from geocentric CIRS to ICRS astrometric.

    Parameters
    ----------
    ri : double array
    di : double array
    date1 : double array
    date2 : double array

    Returns
    -------
    rc : double array
    dc : double array
    eo : double array

    Notes
    -----
    Wraps ERFA function ``eraAtic13``. The ERFA documentation is::

        - - - - - - - - - -
         e r a A t i c 1 3
        - - - - - - - - - -

        Transform star RA,Dec from geocentric CIRS to ICRS astrometric.

        Given:
           ri,di  double  CIRS geocentric RA,Dec (radians)
           date1  double  TDB as a 2-part...
           date2  double  ...Julian Date (Note 1)

        Returned:
           rc,dc  double  ICRS astrometric RA,Dec (radians)
           eo     double  equation of the origins (ERA-GST, Note 4)

        Notes:

        1) The TDB date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TDB)=2450123.7 could be expressed in any of these ways, among
           others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 method is best matched to the way the
           argument is handled internally and will deliver the optimum
           resolution.  The MJD method and the date & time methods are both
           good compromises between resolution and convenience.  For most
           applications of this function the choice will not be at all
           critical.

           TT can be used instead of TDB without any significant impact on
           accuracy.

        2) Iterative techniques are used for the aberration and light
           deflection corrections so that the functions eraAtic13 (or
           eraAticq) and eraAtci13 (or eraAtciq) are accurate inverses;
           even at the edge of the Sun's disk the discrepancy is only about
           1 nanoarcsecond.

        3) The available accuracy is better than 1 milliarcsecond, limited
           mainly by the precession-nutation model that is used, namely
           IAU 2000A/2006.  Very close to solar system bodies, additional
           errors of up to several milliarcseconds can occur because of
           unmodeled light deflection;  however, the Sun's contribution is
           taken into account, to first order.  The accuracy limitations of
           the ERFA function eraEpv00 (used to compute Earth position and
           velocity) can contribute aberration errors of up to
           5 microarcseconds.  Light deflection at the Sun's limb is
           uncertain at the 0.4 mas level.

        4) Should the transformation to (equinox based) J2000.0 mean place
           be required rather than (CIO based) ICRS coordinates, subtract the
           equation of the origins from the returned right ascension:
           RA = RI - EO.  (The eraAnp function can then be applied, as
           required, to keep the result in the conventional 0-2pi range.)

        Called:
           eraApci13    astrometry parameters, ICRS-CIRS, 2013
           eraAticq     quick CIRS to ICRS astrometric

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc, dc, eo = ufunc.atic13(ri, di, date1, date2)
    return rc, dc, eo


def aticq(ri, di, astrom):
    """
    Quick CIRS RA,Dec to ICRS astrometric place, given the star-
    independent astrometry parameters.

    Parameters
    ----------
    ri : double array
    di : double array
    astrom : eraASTROM array

    Returns
    -------
    rc : double array
    dc : double array

    Notes
    -----
    Wraps ERFA function ``eraAticq``. The ERFA documentation is::

        - - - - - - - - -
         e r a A t i c q
        - - - - - - - - -

        Quick CIRS RA,Dec to ICRS astrometric place, given the star-
        independent astrometry parameters.

        Use of this function is appropriate when efficiency is important and
        where many star positions are all to be transformed for one date.
        The star-independent astrometry parameters can be obtained by
        calling one of the functions eraApci[13], eraApcg[13], eraApco[13]
        or eraApcs[13].

        Given:
           ri,di  double     CIRS RA,Dec (radians)
           astrom eraASTROM* star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       longitude + s' (radians)
            xpl    double       polar motion xp wrt local meridian (radians)
            ypl    double       polar motion yp wrt local meridian (radians)
            sphi   double       sine of geodetic latitude
            cphi   double       cosine of geodetic latitude
            diurab double       magnitude of diurnal aberration vector
            eral   double       "local" Earth rotation angle (radians)
            refa   double       refraction constant A (radians)
            refb   double       refraction constant B (radians)

        Returned:
           rc,dc  double     ICRS astrometric RA,Dec (radians)

        Notes:

        1) Only the Sun is taken into account in the light deflection
           correction.

        2) Iterative techniques are used for the aberration and light
           deflection corrections so that the functions eraAtic13 (or
           eraAticq) and eraAtci13 (or eraAtciq) are accurate inverses;
           even at the edge of the Sun's disk the discrepancy is only about
           1 nanoarcsecond.

        Called:
           eraS2c       spherical coordinates to unit vector
           eraTrxp      product of transpose of r-matrix and p-vector
           eraZp        zero p-vector
           eraAb        stellar aberration
           eraLdsun     light deflection by the Sun
           eraC2s       p-vector to spherical
           eraAnp       normalize angle into range +/- pi

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc, dc = ufunc.aticq(ri, di, astrom)
    return rc, dc


def aticqn(ri, di, astrom, b):
    """
    Quick CIRS to ICRS astrometric place transformation, given the star-
    independent astrometry parameters plus a list of light-deflecting
    bodies.

    Parameters
    ----------
    ri : double array
    di : double array
    astrom : eraASTROM array
    b : eraLDBODY array

    Returns
    -------
    rc : double array
    dc : double array

    Notes
    -----
    Wraps ERFA function ``eraAticqn``. The ERFA documentation is::

        - - - - - - - - - -
         e r a A t i c q n
        - - - - - - - - - -

        Quick CIRS to ICRS astrometric place transformation, given the star-
        independent astrometry parameters plus a list of light-deflecting
        bodies.

        Use of this function is appropriate when efficiency is important and
        where many star positions are all to be transformed for one date.
        The star-independent astrometry parameters can be obtained by
        calling one of the functions eraApci[13], eraApcg[13], eraApco[13]
        or eraApcs[13].

        If the only light-deflecting body to be taken into account is the
        Sun, the eraAticq function can be used instead.

        Given:
           ri,di  double      CIRS RA,Dec (radians)
           astrom eraASTROM        star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       longitude + s' (radians)
            xpl    double       polar motion xp wrt local meridian (radians)
            ypl    double       polar motion yp wrt local meridian (radians)
            sphi   double       sine of geodetic latitude
            cphi   double       cosine of geodetic latitude
            diurab double       magnitude of diurnal aberration vector
            eral   double       "local" Earth rotation angle (radians)
            refa   double       refraction constant A (radians)
            refb   double       refraction constant B (radians)
           n      int          number of bodies (Note 3)
           b      eraLDBODY[n] data for each of the n bodies (Notes 3,4):
            bm     double       mass of the body (solar masses, Note 5)
            dl     double       deflection limiter (Note 6)
            pv     [2][3]       barycentric PV of the body (au, au/day)

        Returned:
           rc,dc  double     ICRS astrometric RA,Dec (radians)

        Notes:

        1) Iterative techniques are used for the aberration and light
           deflection corrections so that the functions eraAticqn and
           eraAtciqn are accurate inverses; even at the edge of the Sun's
           disk the discrepancy is only about 1 nanoarcsecond.

        2) If the only light-deflecting body to be taken into account is the
           Sun, the eraAticq function can be used instead.

        3) The struct b contains n entries, one for each body to be
           considered.  If n = 0, no gravitational light deflection will be
           applied, not even for the Sun.

        4) The struct b should include an entry for the Sun as well as for
           any planet or other body to be taken into account.  The entries
           should be in the order in which the light passes the body.

        5) In the entry in the b struct for body i, the mass parameter
           b[i].bm can, as required, be adjusted in order to allow for such
           effects as quadrupole field.

        6) The deflection limiter parameter b[i].dl is phi^2/2, where phi is
           the angular separation (in radians) between star and body at
           which limiting is applied.  As phi shrinks below the chosen
           threshold, the deflection is artificially reduced, reaching zero
           for phi = 0.   Example values suitable for a terrestrial
           observer, together with masses, are as follows:

              body i     b[i].bm        b[i].dl

              Sun        1.0            6e-6
              Jupiter    0.00095435     3e-9
              Saturn     0.00028574     3e-10

        7) For efficiency, validation of the contents of the b array is
           omitted.  The supplied masses must be greater than zero, the
           position and velocity vectors must be right, and the deflection
           limiter greater than zero.

        Called:
           eraS2c       spherical coordinates to unit vector
           eraTrxp      product of transpose of r-matrix and p-vector
           eraZp        zero p-vector
           eraAb        stellar aberration
           eraLdn       light deflection by n bodies
           eraC2s       p-vector to spherical
           eraAnp       normalize angle into range +/- pi

        This revision:   2021 January 6

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc, dc = ufunc.aticqn(ri, di, astrom, b)
    return rc, dc


def atio13(ri, di, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl):
    """
    CIRS RA,Dec to observed place.  The caller supplies UTC, site
    coordinates, ambient air conditions and observing wavelength.

    Parameters
    ----------
    ri : double array
    di : double array
    utc1 : double array
    utc2 : double array
    dut1 : double array
    elong : double array
    phi : double array
    hm : double array
    xp : double array
    yp : double array
    phpa : double array
    tc : double array
    rh : double array
    wl : double array

    Returns
    -------
    aob : double array
    zob : double array
    hob : double array
    dob : double array
    rob : double array

    Notes
    -----
    Wraps ERFA function ``eraAtio13``. The ERFA documentation is::

        - - - - - - - - - -
         e r a A t i o 1 3
        - - - - - - - - - -

        CIRS RA,Dec to observed place.  The caller supplies UTC, site
        coordinates, ambient air conditions and observing wavelength.

        Given:
           ri     double   CIRS right ascension (CIO-based, radians)
           di     double   CIRS declination (radians)
           utc1   double   UTC as a 2-part...
           utc2   double   ...quasi Julian Date (Notes 1,2)
           dut1   double   UT1-UTC (seconds, Note 3)
           elong  double   longitude (radians, east +ve, Note 4)
           phi    double   geodetic latitude (radians, Note 4)
           hm     double   height above ellipsoid (m, geodetic Notes 4,6)
           xp,yp  double   polar motion coordinates (radians, Note 5)
           phpa   double   pressure at the observer (hPa = mB, Note 6)
           tc     double   ambient temperature at the observer (deg C)
           rh     double   relative humidity at the observer (range 0-1)
           wl     double   wavelength (micrometers, Note 7)

        Returned:
           aob    double        observed azimuth (radians: N=0,E=90)
           zob    double        observed zenith distance (radians)
           hob    double        observed hour angle (radians)
           dob    double        observed declination (radians)
           rob    double        observed right ascension (CIO-based, radians)

        Returned (function value):
                  int      status: +1 = dubious year (Note 2)
                                    0 = OK
                                   -1 = unacceptable date

        Notes:

        1)  utc1+utc2 is quasi Julian Date (see Note 2), apportioned in any
            convenient way between the two arguments, for example where utc1
            is the Julian Day Number and utc2 is the fraction of a day.

            However, JD cannot unambiguously represent UTC during a leap
            second unless special measures are taken.  The convention in the
            present function is that the JD day represents UTC days whether
            the length is 86399, 86400 or 86401 SI seconds.

            Applications should use the function eraDtf2d to convert from
            calendar date and time of day into 2-part quasi Julian Date, as
            it implements the leap-second-ambiguity convention just
            described.

        2)  The warning status "dubious year" flags UTCs that predate the
            introduction of the time scale or that are too far in the
            future to be trusted.  See eraDat for further details.

        3)  UT1-UTC is tabulated in IERS bulletins.  It increases by exactly
            one second at the end of each positive UTC leap second,
            introduced in order to keep UT1-UTC within +/- 0.9s.  n.b. This
            practice is under review, and in the future UT1-UTC may grow
            essentially without limit.

        4)  The geographical coordinates are with respect to the ERFA_WGS84
            reference ellipsoid.  TAKE CARE WITH THE LONGITUDE SIGN:  the
            longitude required by the present function is east-positive
            (i.e. right-handed), in accordance with geographical convention.

        5)  The polar motion xp,yp can be obtained from IERS bulletins.  The
            values are the coordinates (in radians) of the Celestial
            Intermediate Pole with respect to the International Terrestrial
            Reference System (see IERS Conventions 2003), measured along the
            meridians 0 and 90 deg west respectively.  For many
            applications, xp and yp can be set to zero.

        6)  If hm, the height above the ellipsoid of the observing station
            in meters, is not known but phpa, the pressure in hPa (=mB), is
            available, an adequate estimate of hm can be obtained from the
            expression

                  hm = -29.3 * tsl * log ( phpa / 1013.25 );

            where tsl is the approximate sea-level air temperature in K
            (See Astrophysical Quantities, C.W.Allen, 3rd edition, section
            52).  Similarly, if the pressure phpa is not known, it can be
            estimated from the height of the observing station, hm, as
            follows:

                  phpa = 1013.25 * exp ( -hm / ( 29.3 * tsl ) );

            Note, however, that the refraction is nearly proportional to
            the pressure and that an accurate phpa value is important for
            precise work.

        7)  The argument wl specifies the observing wavelength in
            micrometers.  The transition from optical to radio is assumed to
            occur at 100 micrometers (about 3000 GHz).

        8)  "Observed" Az,ZD means the position that would be seen by a
            perfect geodetically aligned theodolite.  (Zenith distance is
            used rather than altitude in order to reflect the fact that no
            allowance is made for depression of the horizon.)  This is
            related to the observed HA,Dec via the standard rotation, using
            the geodetic latitude (corrected for polar motion), while the
            observed HA and RA are related simply through the Earth rotation
            angle and the site longitude.  "Observed" RA,Dec or HA,Dec thus
            means the position that would be seen by a perfect equatorial
            with its polar axis aligned to the Earth's axis of rotation.

        9)  The accuracy of the result is limited by the corrections for
            refraction, which use a simple A*tan(z) + B*tan^3(z) model.
            Providing the meteorological parameters are known accurately and
            there are no gross local effects, the predicted astrometric
            coordinates should be within 0.05 arcsec (optical) or 1 arcsec
            (radio) for a zenith distance of less than 70 degrees, better
            than 30 arcsec (optical or radio) at 85 degrees and better
            than 20 arcmin (optical) or 30 arcmin (radio) at the horizon.

        10) The complementary functions eraAtio13 and eraAtoi13 are self-
            consistent to better than 1 microarcsecond all over the
            celestial sphere.

        11) It is advisable to take great care with units, as even unlikely
            values of the input parameters are accepted and processed in
            accordance with the models used.

        Called:
           eraApio13    astrometry parameters, CIRS-observed, 2013
           eraAtioq     quick CIRS to observed

        This revision:   2016 February 2

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    aob, zob, hob, dob, rob, c_retval = ufunc.atio13(
        ri, di, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    check_errwarn(c_retval, 'atio13')
    return aob, zob, hob, dob, rob


STATUS_CODES['atio13'] = {
    1: 'dubious year (Note 2)',
    0: 'OK',
    -1: 'unacceptable date',
}


def atioq(ri, di, astrom):
    """
    Quick CIRS to observed place transformation.

    Parameters
    ----------
    ri : double array
    di : double array
    astrom : eraASTROM array

    Returns
    -------
    aob : double array
    zob : double array
    hob : double array
    dob : double array
    rob : double array

    Notes
    -----
    Wraps ERFA function ``eraAtioq``. The ERFA documentation is::

        - - - - - - - - -
         e r a A t i o q
        - - - - - - - - -

        Quick CIRS to observed place transformation.

        Use of this function is appropriate when efficiency is important and
        where many star positions are all to be transformed for one date.
        The star-independent astrometry parameters can be obtained by
        calling eraApio[13] or eraApco[13].

        Given:
           ri     double     CIRS right ascension
           di     double     CIRS declination
           astrom eraASTROM* star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       longitude + s' (radians)
            xpl    double       polar motion xp wrt local meridian (radians)
            ypl    double       polar motion yp wrt local meridian (radians)
            sphi   double       sine of geodetic latitude
            cphi   double       cosine of geodetic latitude
            diurab double       magnitude of diurnal aberration vector
            eral   double       "local" Earth rotation angle (radians)
            refa   double       refraction constant A (radians)
            refb   double       refraction constant B (radians)

        Returned:
           aob    double          observed azimuth (radians: N=0,E=90)
           zob    double          observed zenith distance (radians)
           hob    double          observed hour angle (radians)
           dob    double          observed declination (radians)
           rob    double          observed right ascension (CIO-based, radians)

        Notes:

        1) This function returns zenith distance rather than altitude in
           order to reflect the fact that no allowance is made for
           depression of the horizon.

        2) The accuracy of the result is limited by the corrections for
           refraction, which use a simple A*tan(z) + B*tan^3(z) model.
           Providing the meteorological parameters are known accurately and
           there are no gross local effects, the predicted observed
           coordinates should be within 0.05 arcsec (optical) or 1 arcsec
           (radio) for a zenith distance of less than 70 degrees, better
           than 30 arcsec (optical or radio) at 85 degrees and better
           than 20 arcmin (optical) or 30 arcmin (radio) at the horizon.

           Without refraction, the complementary functions eraAtioq and
           eraAtoiq are self-consistent to better than 1 microarcsecond all
           over the celestial sphere.  With refraction included, consistency
           falls off at high zenith distances, but is still better than
           0.05 arcsec at 85 degrees.

        3) It is advisable to take great care with units, as even unlikely
           values of the input parameters are accepted and processed in
           accordance with the models used.

        4) The CIRS RA,Dec is obtained from a star catalog mean place by
           allowing for space motion, parallax, the Sun's gravitational lens
           effect, annual aberration and precession-nutation.  For star
           positions in the ICRS, these effects can be applied by means of
           the eraAtci13 (etc.) functions.  Starting from classical "mean
           place" systems, additional transformations will be needed first.

        5) "Observed" Az,El means the position that would be seen by a
           perfect geodetically aligned theodolite.  This is obtained from
           the CIRS RA,Dec by allowing for Earth orientation and diurnal
           aberration, rotating from equator to horizon coordinates, and
           then adjusting for refraction.  The HA,Dec is obtained by
           rotating back into equatorial coordinates, and is the position
           that would be seen by a perfect equatorial with its polar axis
           aligned to the Earth's axis of rotation.  Finally, the RA is
           obtained by subtracting the HA from the local ERA.

        6) The star-independent CIRS-to-observed-place parameters in ASTROM
           may be computed with eraApio[13] or eraApco[13].  If nothing has
           changed significantly except the time, eraAper[13] may be used to
           perform the requisite adjustment to the astrom structure.

        Called:
           eraS2c       spherical coordinates to unit vector
           eraC2s       p-vector to spherical
           eraAnp       normalize angle into range 0 to 2pi

        This revision:   2020 December 7

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    aob, zob, hob, dob, rob = ufunc.atioq(ri, di, astrom)
    return aob, zob, hob, dob, rob


def atoc13(type, ob1, ob2, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl):
    """
    Observed place at a groundbased site to to ICRS astrometric RA,Dec.
    The caller supplies UTC, site coordinates, ambient air conditions
    and observing wavelength.

    Parameters
    ----------
    type : const char array
    ob1 : double array
    ob2 : double array
    utc1 : double array
    utc2 : double array
    dut1 : double array
    elong : double array
    phi : double array
    hm : double array
    xp : double array
    yp : double array
    phpa : double array
    tc : double array
    rh : double array
    wl : double array

    Returns
    -------
    rc : double array
    dc : double array

    Notes
    -----
    Wraps ERFA function ``eraAtoc13``. The ERFA documentation is::

        - - - - - - - - - -
         e r a A t o c 1 3
        - - - - - - - - - -

        Observed place at a groundbased site to to ICRS astrometric RA,Dec.
        The caller supplies UTC, site coordinates, ambient air conditions
        and observing wavelength.

        Given:
           type   char[]   type of coordinates - "R", "H" or "A" (Notes 1,2)
           ob1    double   observed Az, HA or RA (radians; Az is N=0,E=90)
           ob2    double   observed ZD or Dec (radians)
           utc1   double   UTC as a 2-part...
           utc2   double   ...quasi Julian Date (Notes 3,4)
           dut1   double   UT1-UTC (seconds, Note 5)
           elong  double   longitude (radians, east +ve, Note 6)
           phi    double   geodetic latitude (radians, Note 6)
           hm     double   height above ellipsoid (m, geodetic Notes 6,8)
           xp,yp  double   polar motion coordinates (radians, Note 7)
           phpa   double   pressure at the observer (hPa = mB, Note 8)
           tc     double   ambient temperature at the observer (deg C)
           rh     double   relative humidity at the observer (range 0-1)
           wl     double   wavelength (micrometers, Note 9)

        Returned:
           rc,dc  double   ICRS astrometric RA,Dec (radians)

        Returned (function value):
                  int      status: +1 = dubious year (Note 4)
                                    0 = OK
                                   -1 = unacceptable date

        Notes:

        1)  "Observed" Az,ZD means the position that would be seen by a
            perfect geodetically aligned theodolite.  (Zenith distance is
            used rather than altitude in order to reflect the fact that no
            allowance is made for depression of the horizon.)  This is
            related to the observed HA,Dec via the standard rotation, using
            the geodetic latitude (corrected for polar motion), while the
            observed HA and RA are related simply through the Earth rotation
            angle and the site longitude.  "Observed" RA,Dec or HA,Dec thus
            means the position that would be seen by a perfect equatorial
            with its polar axis aligned to the Earth's axis of rotation.

        2)  Only the first character of the type argument is significant.
            "R" or "r" indicates that ob1 and ob2 are the observed right
            ascension and declination;  "H" or "h" indicates that they are
            hour angle (west +ve) and declination;  anything else ("A" or
            "a" is recommended) indicates that ob1 and ob2 are azimuth
            (north zero, east 90 deg) and zenith distance.

        3)  utc1+utc2 is quasi Julian Date (see Note 2), apportioned in any
            convenient way between the two arguments, for example where utc1
            is the Julian Day Number and utc2 is the fraction of a day.

            However, JD cannot unambiguously represent UTC during a leap
            second unless special measures are taken.  The convention in the
            present function is that the JD day represents UTC days whether
            the length is 86399, 86400 or 86401 SI seconds.

            Applications should use the function eraDtf2d to convert from
            calendar date and time of day into 2-part quasi Julian Date, as
            it implements the leap-second-ambiguity convention just
            described.

        4)  The warning status "dubious year" flags UTCs that predate the
            introduction of the time scale or that are too far in the
            future to be trusted.  See eraDat for further details.

        5)  UT1-UTC is tabulated in IERS bulletins.  It increases by exactly
            one second at the end of each positive UTC leap second,
            introduced in order to keep UT1-UTC within +/- 0.9s.  n.b. This
            practice is under review, and in the future UT1-UTC may grow
            essentially without limit.

        6)  The geographical coordinates are with respect to the ERFA_WGS84
            reference ellipsoid.  TAKE CARE WITH THE LONGITUDE SIGN:  the
            longitude required by the present function is east-positive
            (i.e. right-handed), in accordance with geographical convention.

        7)  The polar motion xp,yp can be obtained from IERS bulletins.  The
            values are the coordinates (in radians) of the Celestial
            Intermediate Pole with respect to the International Terrestrial
            Reference System (see IERS Conventions 2003), measured along the
            meridians 0 and 90 deg west respectively.  For many
            applications, xp and yp can be set to zero.

        8)  If hm, the height above the ellipsoid of the observing station
            in meters, is not known but phpa, the pressure in hPa (=mB), is
            available, an adequate estimate of hm can be obtained from the
            expression

                  hm = -29.3 * tsl * log ( phpa / 1013.25 );

            where tsl is the approximate sea-level air temperature in K
            (See Astrophysical Quantities, C.W.Allen, 3rd edition, section
            52).  Similarly, if the pressure phpa is not known, it can be
            estimated from the height of the observing station, hm, as
            follows:

                  phpa = 1013.25 * exp ( -hm / ( 29.3 * tsl ) );

            Note, however, that the refraction is nearly proportional to
            the pressure and that an accurate phpa value is important for
            precise work.

        9)  The argument wl specifies the observing wavelength in
            micrometers.  The transition from optical to radio is assumed to
            occur at 100 micrometers (about 3000 GHz).

        10) The accuracy of the result is limited by the corrections for
            refraction, which use a simple A*tan(z) + B*tan^3(z) model.
            Providing the meteorological parameters are known accurately and
            there are no gross local effects, the predicted astrometric
            coordinates should be within 0.05 arcsec (optical) or 1 arcsec
            (radio) for a zenith distance of less than 70 degrees, better
            than 30 arcsec (optical or radio) at 85 degrees and better
            than 20 arcmin (optical) or 30 arcmin (radio) at the horizon.

            Without refraction, the complementary functions eraAtco13 and
            eraAtoc13 are self-consistent to better than 1 microarcsecond
            all over the celestial sphere.  With refraction included,
            consistency falls off at high zenith distances, but is still
            better than 0.05 arcsec at 85 degrees.

        11) It is advisable to take great care with units, as even unlikely
            values of the input parameters are accepted and processed in
            accordance with the models used.

        Called:
           eraApco13    astrometry parameters, ICRS-observed
           eraAtoiq     quick observed to CIRS
           eraAticq     quick CIRS to ICRS

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc, dc, c_retval = ufunc.atoc13(
        type, ob1, ob2, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    check_errwarn(c_retval, 'atoc13')
    return rc, dc


STATUS_CODES['atoc13'] = {
    1: 'dubious year (Note 4)',
    0: 'OK',
    -1: 'unacceptable date',
}


def atoi13(type, ob1, ob2, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl):
    """
    Observed place to CIRS.  The caller supplies UTC, site coordinates,
    ambient air conditions and observing wavelength.

    Parameters
    ----------
    type : const char array
    ob1 : double array
    ob2 : double array
    utc1 : double array
    utc2 : double array
    dut1 : double array
    elong : double array
    phi : double array
    hm : double array
    xp : double array
    yp : double array
    phpa : double array
    tc : double array
    rh : double array
    wl : double array

    Returns
    -------
    ri : double array
    di : double array

    Notes
    -----
    Wraps ERFA function ``eraAtoi13``. The ERFA documentation is::

        - - - - - - - - - -
         e r a A t o i 1 3
        - - - - - - - - - -

        Observed place to CIRS.  The caller supplies UTC, site coordinates,
        ambient air conditions and observing wavelength.

        Given:
           type   char[]   type of coordinates - "R", "H" or "A" (Notes 1,2)
           ob1    double   observed Az, HA or RA (radians; Az is N=0,E=90)
           ob2    double   observed ZD or Dec (radians)
           utc1   double   UTC as a 2-part...
           utc2   double   ...quasi Julian Date (Notes 3,4)
           dut1   double   UT1-UTC (seconds, Note 5)
           elong  double   longitude (radians, east +ve, Note 6)
           phi    double   geodetic latitude (radians, Note 6)
           hm     double   height above the ellipsoid (meters, Notes 6,8)
           xp,yp  double   polar motion coordinates (radians, Note 7)
           phpa   double   pressure at the observer (hPa = mB, Note 8)
           tc     double   ambient temperature at the observer (deg C)
           rh     double   relative humidity at the observer (range 0-1)
           wl     double   wavelength (micrometers, Note 9)

        Returned:
           ri     double        CIRS right ascension (CIO-based, radians)
           di     double        CIRS declination (radians)

        Returned (function value):
                  int      status: +1 = dubious year (Note 2)
                                    0 = OK
                                   -1 = unacceptable date

        Notes:

        1)  "Observed" Az,ZD means the position that would be seen by a
            perfect geodetically aligned theodolite.  (Zenith distance is
            used rather than altitude in order to reflect the fact that no
            allowance is made for depression of the horizon.)  This is
            related to the observed HA,Dec via the standard rotation, using
            the geodetic latitude (corrected for polar motion), while the
            observed HA and RA are related simply through the Earth rotation
            angle and the site longitude.  "Observed" RA,Dec or HA,Dec thus
            means the position that would be seen by a perfect equatorial
            with its polar axis aligned to the Earth's axis of rotation.

        2)  Only the first character of the type argument is significant.
            "R" or "r" indicates that ob1 and ob2 are the observed right
            ascension and declination;  "H" or "h" indicates that they are
            hour angle (west +ve) and declination;  anything else ("A" or
            "a" is recommended) indicates that ob1 and ob2 are azimuth
            (north zero, east 90 deg) and zenith distance.

        3)  utc1+utc2 is quasi Julian Date (see Note 2), apportioned in any
            convenient way between the two arguments, for example where utc1
            is the Julian Day Number and utc2 is the fraction of a day.

            However, JD cannot unambiguously represent UTC during a leap
            second unless special measures are taken.  The convention in the
            present function is that the JD day represents UTC days whether
            the length is 86399, 86400 or 86401 SI seconds.

            Applications should use the function eraDtf2d to convert from
            calendar date and time of day into 2-part quasi Julian Date, as
            it implements the leap-second-ambiguity convention just
            described.

        4)  The warning status "dubious year" flags UTCs that predate the
            introduction of the time scale or that are too far in the
            future to be trusted.  See eraDat for further details.

        5)  UT1-UTC is tabulated in IERS bulletins.  It increases by exactly
            one second at the end of each positive UTC leap second,
            introduced in order to keep UT1-UTC within +/- 0.9s.  n.b. This
            practice is under review, and in the future UT1-UTC may grow
            essentially without limit.

        6)  The geographical coordinates are with respect to the ERFA_WGS84
            reference ellipsoid.  TAKE CARE WITH THE LONGITUDE SIGN:  the
            longitude required by the present function is east-positive
            (i.e. right-handed), in accordance with geographical convention.

        7)  The polar motion xp,yp can be obtained from IERS bulletins.  The
            values are the coordinates (in radians) of the Celestial
            Intermediate Pole with respect to the International Terrestrial
            Reference System (see IERS Conventions 2003), measured along the
            meridians 0 and 90 deg west respectively.  For many
            applications, xp and yp can be set to zero.

        8)  If hm, the height above the ellipsoid of the observing station
            in meters, is not known but phpa, the pressure in hPa (=mB), is
            available, an adequate estimate of hm can be obtained from the
            expression

                  hm = -29.3 * tsl * log ( phpa / 1013.25 );

            where tsl is the approximate sea-level air temperature in K
            (See Astrophysical Quantities, C.W.Allen, 3rd edition, section
            52).  Similarly, if the pressure phpa is not known, it can be
            estimated from the height of the observing station, hm, as
            follows:

                  phpa = 1013.25 * exp ( -hm / ( 29.3 * tsl ) );

            Note, however, that the refraction is nearly proportional to
            the pressure and that an accurate phpa value is important for
            precise work.

        9)  The argument wl specifies the observing wavelength in
            micrometers.  The transition from optical to radio is assumed to
            occur at 100 micrometers (about 3000 GHz).

        10) The accuracy of the result is limited by the corrections for
            refraction, which use a simple A*tan(z) + B*tan^3(z) model.
            Providing the meteorological parameters are known accurately and
            there are no gross local effects, the predicted astrometric
            coordinates should be within 0.05 arcsec (optical) or 1 arcsec
            (radio) for a zenith distance of less than 70 degrees, better
            than 30 arcsec (optical or radio) at 85 degrees and better
            than 20 arcmin (optical) or 30 arcmin (radio) at the horizon.

            Without refraction, the complementary functions eraAtio13 and
            eraAtoi13 are self-consistent to better than 1 microarcsecond
            all over the celestial sphere.  With refraction included,
            consistency falls off at high zenith distances, but is still
            better than 0.05 arcsec at 85 degrees.

        12) It is advisable to take great care with units, as even unlikely
            values of the input parameters are accepted and processed in
            accordance with the models used.

        Called:
           eraApio13    astrometry parameters, CIRS-observed, 2013
           eraAtoiq     quick observed to CIRS

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    ri, di, c_retval = ufunc.atoi13(
        type, ob1, ob2, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    check_errwarn(c_retval, 'atoi13')
    return ri, di


STATUS_CODES['atoi13'] = {
    1: 'dubious year (Note 2)',
    0: 'OK',
    -1: 'unacceptable date',
}


def atoiq(type, ob1, ob2, astrom):
    """
    Quick observed place to CIRS, given the star-independent astrometry
    parameters.

    Parameters
    ----------
    type : const char array
    ob1 : double array
    ob2 : double array
    astrom : eraASTROM array

    Returns
    -------
    ri : double array
    di : double array

    Notes
    -----
    Wraps ERFA function ``eraAtoiq``. The ERFA documentation is::

        - - - - - - - - -
         e r a A t o i q
        - - - - - - - - -

        Quick observed place to CIRS, given the star-independent astrometry
        parameters.

        Use of this function is appropriate when efficiency is important and
        where many star positions are all to be transformed for one date.
        The star-independent astrometry parameters can be obtained by
        calling eraApio[13] or eraApco[13].

        Given:
           type   char[]     type of coordinates: "R", "H" or "A" (Note 1)
           ob1    double     observed Az, HA or RA (radians; Az is N=0,E=90)
           ob2    double     observed ZD or Dec (radians)
           astrom eraASTROM* star-independent astrometry parameters:
            pmt    double       PM time interval (SSB, Julian years)
            eb     double[3]    SSB to observer (vector, au)
            eh     double[3]    Sun to observer (unit vector)
            em     double       distance from Sun to observer (au)
            v      double[3]    barycentric observer velocity (vector, c)
            bm1    double       sqrt(1-|v|^2): reciprocal of Lorenz factor
            bpn    double[3][3] bias-precession-nutation matrix
            along  double       longitude + s' (radians)
            xpl    double       polar motion xp wrt local meridian (radians)
            ypl    double       polar motion yp wrt local meridian (radians)
            sphi   double       sine of geodetic latitude
            cphi   double       cosine of geodetic latitude
            diurab double       magnitude of diurnal aberration vector
            eral   double       "local" Earth rotation angle (radians)
            refa   double       refraction constant A (radians)
            refb   double       refraction constant B (radians)

        Returned:
           ri     double          CIRS right ascension (CIO-based, radians)
           di     double          CIRS declination (radians)

        Notes:

        1) "Observed" Az,El means the position that would be seen by a
           perfect geodetically aligned theodolite.  This is related to
           the observed HA,Dec via the standard rotation, using the geodetic
           latitude (corrected for polar motion), while the observed HA and
           RA are related simply through the Earth rotation angle and the
           site longitude.  "Observed" RA,Dec or HA,Dec thus means the
           position that would be seen by a perfect equatorial with its
           polar axis aligned to the Earth's axis of rotation.  By removing
           from the observed place the effects of atmospheric refraction and
           diurnal aberration, the CIRS RA,Dec is obtained.

        2) Only the first character of the type argument is significant.
           "R" or "r" indicates that ob1 and ob2 are the observed right
           ascension and declination;  "H" or "h" indicates that they are
           hour angle (west +ve) and declination;  anything else ("A" or
           "a" is recommended) indicates that ob1 and ob2 are azimuth (north
           zero, east 90 deg) and zenith distance.  (Zenith distance is used
           rather than altitude in order to reflect the fact that no
           allowance is made for depression of the horizon.)

        3) The accuracy of the result is limited by the corrections for
           refraction, which use a simple A*tan(z) + B*tan^3(z) model.
           Providing the meteorological parameters are known accurately and
           there are no gross local effects, the predicted intermediate
           coordinates should be within 0.05 arcsec (optical) or 1 arcsec
           (radio) for a zenith distance of less than 70 degrees, better
           than 30 arcsec (optical or radio) at 85 degrees and better than
           20 arcmin (optical) or 25 arcmin (radio) at the horizon.

           Without refraction, the complementary functions eraAtioq and
           eraAtoiq are self-consistent to better than 1 microarcsecond all
           over the celestial sphere.  With refraction included, consistency
           falls off at high zenith distances, but is still better than
           0.05 arcsec at 85 degrees.

        4) It is advisable to take great care with units, as even unlikely
           values of the input parameters are accepted and processed in
           accordance with the models used.

        Called:
           eraS2c       spherical coordinates to unit vector
           eraC2s       p-vector to spherical
           eraAnp       normalize angle into range 0 to 2pi

        This revision:   2020 December 7

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    ri, di = ufunc.atoiq(type, ob1, ob2, astrom)
    return ri, di


def ld(bm, p, q, e, em, dlim):
    """
    Apply light deflection by a solar-system body, as part of
    transforming coordinate direction into natural direction.

    Parameters
    ----------
    bm : double array
    p : double array
    q : double array
    e : double array
    em : double array
    dlim : double array

    Returns
    -------
    p1 : double array

    Notes
    -----
    Wraps ERFA function ``eraLd``. The ERFA documentation is::

        - - - - - -
         e r a L d
        - - - - - -

        Apply light deflection by a solar-system body, as part of
        transforming coordinate direction into natural direction.

        Given:
           bm     double     mass of the gravitating body (solar masses)
           p      double[3]  direction from observer to source (unit vector)
           q      double[3]  direction from body to source (unit vector)
           e      double[3]  direction from body to observer (unit vector)
           em     double     distance from body to observer (au)
           dlim   double     deflection limiter (Note 4)

        Returned:
           p1     double[3]  observer to deflected source (unit vector)

        Notes:

        1) The algorithm is based on Expr. (70) in Klioner (2003) and
           Expr. (7.63) in the Explanatory Supplement (Urban & Seidelmann
           2013), with some rearrangement to minimize the effects of machine
           precision.

        2) The mass parameter bm can, as required, be adjusted in order to
           allow for such effects as quadrupole field.

        3) The barycentric position of the deflecting body should ideally
           correspond to the time of closest approach of the light ray to
           the body.

        4) The deflection limiter parameter dlim is phi^2/2, where phi is
           the angular separation (in radians) between source and body at
           which limiting is applied.  As phi shrinks below the chosen
           threshold, the deflection is artificially reduced, reaching zero
           for phi = 0.

        5) The returned vector p1 is not normalized, but the consequential
           departure from unit magnitude is always negligible.

        6) The arguments p and p1 can be the same array.

        7) To accumulate total light deflection taking into account the
           contributions from several bodies, call the present function for
           each body in succession, in decreasing order of distance from the
           observer.

        8) For efficiency, validation is omitted.  The supplied vectors must
           be of unit magnitude, and the deflection limiter non-zero and
           positive.

        References:

           Urban, S. & Seidelmann, P. K. (eds), Explanatory Supplement to
           the Astronomical Almanac, 3rd ed., University Science Books
           (2013).

           Klioner, Sergei A., "A practical relativistic model for micro-
           arcsecond astrometry in space", Astr. J. 125, 1580-1597 (2003).

        Called:
           eraPdp       scalar product of two p-vectors
           eraPxp       vector product of two p-vectors

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    p1 = ufunc.ld(bm, p, q, e, em, dlim)
    return p1


def ldn(b, ob, sc):
    """
    For a star, apply light deflection by multiple solar-system bodies,
    as part of transforming coordinate direction into natural direction.

    Parameters
    ----------
    b : eraLDBODY array
    ob : double array
    sc : double array

    Returns
    -------
    sn : double array

    Notes
    -----
    Wraps ERFA function ``eraLdn``. The ERFA documentation is::

        - - - - - - -
         e r a L d n
        - - - - - - -

        For a star, apply light deflection by multiple solar-system bodies,
        as part of transforming coordinate direction into natural direction.

        Given:
           n    int           number of bodies (note 1)
           b    eraLDBODY[n]  data for each of the n bodies (Notes 1,2):
            bm   double         mass of the body (solar masses, Note 3)
            dl   double         deflection limiter (Note 4)
            pv   [2][3]         barycentric PV of the body (au, au/day)
           ob   double[3]     barycentric position of the observer (au)
           sc   double[3]     observer to star coord direction (unit vector)

        Returned:
           sn    double[3]      observer to deflected star (unit vector)

        1) The array b contains n entries, one for each body to be
           considered.  If n = 0, no gravitational light deflection will be
           applied, not even for the Sun.

        2) The array b should include an entry for the Sun as well as for
           any planet or other body to be taken into account.  The entries
           should be in the order in which the light passes the body.

        3) In the entry in the b array for body i, the mass parameter
           b[i].bm can, as required, be adjusted in order to allow for such
           effects as quadrupole field.

        4) The deflection limiter parameter b[i].dl is phi^2/2, where phi is
           the angular separation (in radians) between star and body at
           which limiting is applied.  As phi shrinks below the chosen
           threshold, the deflection is artificially reduced, reaching zero
           for phi = 0.   Example values suitable for a terrestrial
           observer, together with masses, are as follows:

              body i     b[i].bm        b[i].dl

              Sun        1.0            6e-6
              Jupiter    0.00095435     3e-9
              Saturn     0.00028574     3e-10

        5) For cases where the starlight passes the body before reaching the
           observer, the body is placed back along its barycentric track by
           the light time from that point to the observer.  For cases where
           the body is "behind" the observer no such shift is applied.  If
           a different treatment is preferred, the user has the option of
           instead using the eraLd function.  Similarly, eraLd can be used
           for cases where the source is nearby, not a star.

        6) The returned vector sn is not normalized, but the consequential
           departure from unit magnitude is always negligible.

        7) The arguments sc and sn can be the same array.

        8) For efficiency, validation is omitted.  The supplied masses must
           be greater than zero, the position and velocity vectors must be
           right, and the deflection limiter greater than zero.

        Reference:

           Urban, S. & Seidelmann, P. K. (eds), Explanatory Supplement to
           the Astronomical Almanac, 3rd ed., University Science Books
           (2013), Section 7.2.4.

        Called:
           eraCp        copy p-vector
           eraPdp       scalar product of two p-vectors
           eraPmp       p-vector minus p-vector
           eraPpsp      p-vector plus scaled p-vector
           eraPn        decompose p-vector into modulus and direction
           eraLd        light deflection by a solar-system body

        This revision:   2017 March 16

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    sn = ufunc.ldn(b, ob, sc)
    return sn


def ldsun(p, e, em):
    """
    Deflection of starlight by the Sun.

    Parameters
    ----------
    p : double array
    e : double array
    em : double array

    Returns
    -------
    p1 : double array

    Notes
    -----
    Wraps ERFA function ``eraLdsun``. The ERFA documentation is::

        - - - - - - - - -
         e r a L d s u n
        - - - - - - - - -

        Deflection of starlight by the Sun.

        Given:
           p      double[3]  direction from observer to star (unit vector)
           e      double[3]  direction from Sun to observer (unit vector)
           em     double     distance from Sun to observer (au)

        Returned:
           p1     double[3]  observer to deflected star (unit vector)

        Notes:

        1) The source is presumed to be sufficiently distant that its
           directions seen from the Sun and the observer are essentially
           the same.

        2) The deflection is restrained when the angle between the star and
           the center of the Sun is less than a threshold value, falling to
           zero deflection for zero separation.  The chosen threshold value
           is within the solar limb for all solar-system applications, and
           is about 5 arcminutes for the case of a terrestrial observer.

        3) The arguments p and p1 can be the same array.

        Called:
           eraLd        light deflection by a solar-system body

        This revision:   2016 June 16

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    p1 = ufunc.ldsun(p, e, em)
    return p1


def pmpx(rc, dc, pr, pd, px, rv, pmt, pob):
    """
    Proper motion and parallax.

    Parameters
    ----------
    rc : double array
    dc : double array
    pr : double array
    pd : double array
    px : double array
    rv : double array
    pmt : double array
    pob : double array

    Returns
    -------
    pco : double array

    Notes
    -----
    Wraps ERFA function ``eraPmpx``. The ERFA documentation is::

        - - - - - - - -
         e r a P m p x
        - - - - - - - -

        Proper motion and parallax.

        Given:
           rc,dc  double     ICRS RA,Dec at catalog epoch (radians)
           pr     double     RA proper motion (radians/year; Note 1)
           pd     double     Dec proper motion (radians/year)
           px     double     parallax (arcsec)
           rv     double     radial velocity (km/s, +ve if receding)
           pmt    double     proper motion time interval (SSB, Julian years)
           pob    double[3]  SSB to observer vector (au)

        Returned:
           pco    double[3]  coordinate direction (BCRS unit vector)

        Notes:

        1) The proper motion in RA is dRA/dt rather than cos(Dec)*dRA/dt.

        2) The proper motion time interval is for when the starlight
           reaches the solar system barycenter.

        3) To avoid the need for iteration, the Roemer effect (i.e. the
           small annual modulation of the proper motion coming from the
           changing light time) is applied approximately, using the
           direction of the star at the catalog epoch.

        References:

           1984 Astronomical Almanac, pp B39-B41.

           Urban, S. & Seidelmann, P. K. (eds), Explanatory Supplement to
           the Astronomical Almanac, 3rd ed., University Science Books
           (2013), Section 7.2.

        Called:
           eraPdp       scalar product of two p-vectors
           eraPn        decompose p-vector into modulus and direction

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    pco = ufunc.pmpx(rc, dc, pr, pd, px, rv, pmt, pob)
    return pco


def pmsafe(ra1, dec1, pmr1, pmd1, px1, rv1, ep1a, ep1b, ep2a, ep2b):
    """
    Star proper motion:  update star catalog data for space motion, with
    special handling to handle the zero parallax case.

    Parameters
    ----------
    ra1 : double array
    dec1 : double array
    pmr1 : double array
    pmd1 : double array
    px1 : double array
    rv1 : double array
    ep1a : double array
    ep1b : double array
    ep2a : double array
    ep2b : double array

    Returns
    -------
    ra2 : double array
    dec2 : double array
    pmr2 : double array
    pmd2 : double array
    px2 : double array
    rv2 : double array

    Notes
    -----
    Wraps ERFA function ``eraPmsafe``. The ERFA documentation is::

        - - - - - - - - - -
         e r a P m s a f e
        - - - - - - - - - -

        Star proper motion:  update star catalog data for space motion, with
        special handling to handle the zero parallax case.

        Given:
           ra1    double      right ascension (radians), before
           dec1   double      declination (radians), before
           pmr1   double      RA proper motion (radians/year), before
           pmd1   double      Dec proper motion (radians/year), before
           px1    double      parallax (arcseconds), before
           rv1    double      radial velocity (km/s, +ve = receding), before
           ep1a   double      "before" epoch, part A (Note 1)
           ep1b   double      "before" epoch, part B (Note 1)
           ep2a   double      "after" epoch, part A (Note 1)
           ep2b   double      "after" epoch, part B (Note 1)

        Returned:
           ra2    double      right ascension (radians), after
           dec2   double      declination (radians), after
           pmr2   double      RA proper motion (radians/year), after
           pmd2   double      Dec proper motion (radians/year), after
           px2    double      parallax (arcseconds), after
           rv2    double      radial velocity (km/s, +ve = receding), after

        Returned (function value):
                  int         status:
                               -1 = system error (should not occur)
                                0 = no warnings or errors
                                1 = distance overridden (Note 6)
                                2 = excessive velocity (Note 7)
                                4 = solution didn't converge (Note 8)
                             else = binary logical OR of the above warnings

        Notes:

        1) The starting and ending TDB epochs ep1a+ep1b and ep2a+ep2b are
           Julian Dates, apportioned in any convenient way between the two
           parts (A and B).  For example, JD(TDB)=2450123.7 could be
           expressed in any of these ways, among others:

                  epNa            epNb

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 method is best matched to the way the
           argument is handled internally and will deliver the optimum
           resolution.  The MJD method and the date & time methods are both
           good compromises between resolution and convenience.

        2) In accordance with normal star-catalog conventions, the object's
           right ascension and declination are freed from the effects of
           secular aberration.  The frame, which is aligned to the catalog
           equator and equinox, is Lorentzian and centered on the SSB.

           The proper motions are the rate of change of the right ascension
           and declination at the catalog epoch and are in radians per TDB
           Julian year.

           The parallax and radial velocity are in the same frame.

        3) Care is needed with units.  The star coordinates are in radians
           and the proper motions in radians per Julian year, but the
           parallax is in arcseconds.

        4) The RA proper motion is in terms of coordinate angle, not true
           angle.  If the catalog uses arcseconds for both RA and Dec proper
           motions, the RA proper motion will need to be divided by cos(Dec)
           before use.

        5) Straight-line motion at constant speed, in the inertial frame, is
           assumed.

        6) An extremely small (or zero or negative) parallax is overridden
           to ensure that the object is at a finite but very large distance,
           but not so large that the proper motion is equivalent to a large
           but safe speed (about 0.1c using the chosen constant).  A warning
           status of 1 is added to the status if this action has been taken.

        7) If the space velocity is a significant fraction of c (see the
           constant VMAX in the function eraStarpv), it is arbitrarily set
           to zero.  When this action occurs, 2 is added to the status.

        8) The relativistic adjustment carried out in the eraStarpv function
           involves an iterative calculation.  If the process fails to
           converge within a set number of iterations, 4 is added to the
           status.

        Called:
           eraSeps      angle between two points
           eraStarpm    update star catalog data for space motion

        This revision:   2014 July 1

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    ra2, dec2, pmr2, pmd2, px2, rv2, c_retval = ufunc.pmsafe(
        ra1, dec1, pmr1, pmd1, px1, rv1, ep1a, ep1b, ep2a, ep2b)
    check_errwarn(c_retval, 'pmsafe')
    return ra2, dec2, pmr2, pmd2, px2, rv2


STATUS_CODES['pmsafe'] = {
    -1: 'system error (should not occur)',
    0: 'no warnings or errors',
    1: 'distance overridden (Note 6)',
    2: 'excessive velocity (Note 7)',
    4: "solution didn't converge (Note 8)",
    'else': 'binary logical OR of the above warnings',
}


def pvtob(elong, phi, hm, xp, yp, sp, theta):
    """
    Position and velocity of a terrestrial observing station.

    Parameters
    ----------
    elong : double array
    phi : double array
    hm : double array
    xp : double array
    yp : double array
    sp : double array
    theta : double array

    Returns
    -------
    pv : double array

    Notes
    -----
    Wraps ERFA function ``eraPvtob``. The ERFA documentation is::

        - - - - - - - - -
         e r a P v t o b
        - - - - - - - - -

        Position and velocity of a terrestrial observing station.

        Given:
           elong   double       longitude (radians, east +ve, Note 1)
           phi     double       latitude (geodetic, radians, Note 1)
           hm      double       height above ref. ellipsoid (geodetic, m)
           xp,yp   double       coordinates of the pole (radians, Note 2)
           sp      double       the TIO locator s' (radians, Note 2)
           theta   double       Earth rotation angle (radians, Note 3)

        Returned:
           pv      double[2][3] position/velocity vector (m, m/s, CIRS)

        Notes:

        1) The terrestrial coordinates are with respect to the ERFA_WGS84
           reference ellipsoid.

        2) xp and yp are the coordinates (in radians) of the Celestial
           Intermediate Pole with respect to the International Terrestrial
           Reference System (see IERS Conventions), measured along the
           meridians 0 and 90 deg west respectively.  sp is the TIO locator
           s', in radians, which positions the Terrestrial Intermediate
           Origin on the equator.  For many applications, xp, yp and
           (especially) sp can be set to zero.

        3) If theta is Greenwich apparent sidereal time instead of Earth
           rotation angle, the result is with respect to the true equator
           and equinox of date, i.e. with the x-axis at the equinox rather
           than the celestial intermediate origin.

        4) The velocity units are meters per UT1 second, not per SI second.
           This is unlikely to have any practical consequences in the modern
           era.

        5) No validation is performed on the arguments.  Error cases that
           could lead to arithmetic exceptions are trapped by the eraGd2gc
           function, and the result set to zeros.

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Urban, S. & Seidelmann, P. K. (eds), Explanatory Supplement to
           the Astronomical Almanac, 3rd ed., University Science Books
           (2013), Section 7.4.3.3.

        Called:
           eraGd2gc     geodetic to geocentric transformation
           eraPom00     polar motion matrix
           eraTrxp      product of transpose of r-matrix and p-vector

        This revision:   2013 October 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    pv = ufunc.pvtob(elong, phi, hm, xp, yp, sp, theta)
    return pv


def refco(phpa, tc, rh, wl):
    """
    Determine the constants A and B in the atmospheric refraction model
    dZ = A tan Z + B tan^3 Z.

    Parameters
    ----------
    phpa : double array
    tc : double array
    rh : double array
    wl : double array

    Returns
    -------
    refa : double array
    refb : double array

    Notes
    -----
    Wraps ERFA function ``eraRefco``. The ERFA documentation is::

        - - - - - - - - -
         e r a R e f c o
        - - - - - - - - -

        Determine the constants A and B in the atmospheric refraction model
        dZ = A tan Z + B tan^3 Z.

        Z is the "observed" zenith distance (i.e. affected by refraction)
        and dZ is what to add to Z to give the "topocentric" (i.e. in vacuo)
        zenith distance.

        Given:
          phpa   double    pressure at the observer (hPa = millibar)
          tc     double    ambient temperature at the observer (deg C)
          rh     double    relative humidity at the observer (range 0-1)
          wl     double    wavelength (micrometers)

        Returned:
          refa   double         tan Z coefficient (radians)
          refb   double         tan^3 Z coefficient (radians)

        Notes:

        1) The model balances speed and accuracy to give good results in
           applications where performance at low altitudes is not paramount.
           Performance is maintained across a range of conditions, and
           applies to both optical/IR and radio.

        2) The model omits the effects of (i) height above sea level (apart
           from the reduced pressure itself), (ii) latitude (i.e. the
           flattening of the Earth), (iii) variations in tropospheric lapse
           rate and (iv) dispersive effects in the radio.

           The model was tested using the following range of conditions:

             lapse rates 0.0055, 0.0065, 0.0075 deg/meter
             latitudes 0, 25, 50, 75 degrees
             heights 0, 2500, 5000 meters ASL
             pressures mean for height -10% to +5% in steps of 5%
             temperatures -10 deg to +20 deg with respect to 280 deg at SL
             relative humidity 0, 0.5, 1
             wavelengths 0.4, 0.6, ... 2 micron, + radio
             zenith distances 15, 45, 75 degrees

           The accuracy with respect to raytracing through a model
           atmosphere was as follows:

                                  worst         RMS

             optical/IR           62 mas       8 mas
             radio               319 mas      49 mas

           For this particular set of conditions:

             lapse rate 0.0065 K/meter
             latitude 50 degrees
             sea level
             pressure 1005 mb
             temperature 280.15 K
             humidity 80%
             wavelength 5740 Angstroms

           the results were as follows:

             ZD       raytrace     eraRefco   Saastamoinen

             10         10.27        10.27        10.27
             20         21.19        21.20        21.19
             30         33.61        33.61        33.60
             40         48.82        48.83        48.81
             45         58.16        58.18        58.16
             50         69.28        69.30        69.27
             55         82.97        82.99        82.95
             60        100.51       100.54       100.50
             65        124.23       124.26       124.20
             70        158.63       158.68       158.61
             72        177.32       177.37       177.31
             74        200.35       200.38       200.32
             76        229.45       229.43       229.42
             78        267.44       267.29       267.41
             80        319.13       318.55       319.10

            deg        arcsec       arcsec       arcsec

           The values for Saastamoinen's formula (which includes terms
           up to tan^5) are taken from Hohenkerk and Sinclair (1985).

        3) A wl value in the range 0-100 selects the optical/IR case and is
           wavelength in micrometers.  Any value outside this range selects
           the radio case.

        4) Outlandish input parameters are silently limited to
           mathematically safe values.  Zero pressure is permissible, and
           causes zeroes to be returned.

        5) The algorithm draws on several sources, as follows:

           a) The formula for the saturation vapour pressure of water as
              a function of temperature and temperature is taken from
              Equations (A4.5-A4.7) of Gill (1982).

           b) The formula for the water vapour pressure, given the
              saturation pressure and the relative humidity, is from
              Crane (1976), Equation (2.5.5).

           c) The refractivity of air is a function of temperature,
              total pressure, water-vapour pressure and, in the case
              of optical/IR, wavelength.  The formulae for the two cases are
              developed from Hohenkerk & Sinclair (1985) and Rueger (2002).
              The IAG (1999) optical refractivity for dry air is used.

           d) The formula for beta, the ratio of the scale height of the
              atmosphere to the geocentric distance of the observer, is
              an adaption of Equation (9) from Stone (1996).  The
              adaptations, arrived at empirically, consist of (i) a small
              adjustment to the coefficient and (ii) a humidity term for the
              radio case only.

           e) The formulae for the refraction constants as a function of
              n-1 and beta are from Green (1987), Equation (4.31).

        References:

           Crane, R.K., Meeks, M.L. (ed), "Refraction Effects in the Neutral
           Atmosphere", Methods of Experimental Physics: Astrophysics 12B,
           Academic Press, 1976.

           Gill, Adrian E., "Atmosphere-Ocean Dynamics", Academic Press,
           1982.

           Green, R.M., "Spherical Astronomy", Cambridge University Press,
           1987.

           Hohenkerk, C.Y., & Sinclair, A.T., NAO Technical Note No. 63,
           1985.

           IAG Resolutions adopted at the XXIIth General Assembly in
           Birmingham, 1999, Resolution 3.

           Rueger, J.M., "Refractive Index Formulae for Electronic Distance
           Measurement with Radio and Millimetre Waves", in Unisurv Report
           S-68, School of Surveying and Spatial Information Systems,
           University of New South Wales, Sydney, Australia, 2002.

           Stone, Ronald C., P.A.S.P. 108, 1051-1058, 1996.

        This revision:   2020 August 17

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    refa, refb = ufunc.refco(phpa, tc, rh, wl)
    return refa, refb


def epv00(date1, date2):
    """
    Earth position and velocity, heliocentric and barycentric, with
    respect to the Barycentric Celestial Reference System.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    pvh : double array
    pvb : double array

    Notes
    -----
    Wraps ERFA function ``eraEpv00``. The ERFA documentation is::

        - - - - - - - - -
         e r a E p v 0 0
        - - - - - - - - -

        Earth position and velocity, heliocentric and barycentric, with
        respect to the Barycentric Celestial Reference System.

        Given:
           date1,date2  double        TDB date (Note 1)

        Returned:
           pvh          double[2][3]  heliocentric Earth position/velocity
           pvb          double[2][3]  barycentric Earth position/velocity

        Returned (function value):
                        int           status: 0 = OK
                                             +1 = warning: date outside
                                                  the range 1900-2100 AD

        Notes:

        1) The TDB date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TDB)=2450123.7 could be expressed in any of these ways, among
           others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 method is best matched to the way the
           argument is handled internally and will deliver the optimum
           resolution.  The MJD method and the date & time methods are both
           good compromises between resolution and convenience.  However,
           the accuracy of the result is more likely to be limited by the
           algorithm itself than the way the date has been expressed.

           n.b. TT can be used instead of TDB in most applications.

        2) On return, the arrays pvh and pvb contain the following:

              pvh[0][0]  x       }
              pvh[0][1]  y       } heliocentric position, au
              pvh[0][2]  z       }

              pvh[1][0]  xdot    }
              pvh[1][1]  ydot    } heliocentric velocity, au/d
              pvh[1][2]  zdot    }

              pvb[0][0]  x       }
              pvb[0][1]  y       } barycentric position, au
              pvb[0][2]  z       }

              pvb[1][0]  xdot    }
              pvb[1][1]  ydot    } barycentric velocity, au/d
              pvb[1][2]  zdot    }

           The vectors are with respect to the Barycentric Celestial
           Reference System.  The time unit is one day in TDB.

        3) The function is a SIMPLIFIED SOLUTION from the planetary theory
           VSOP2000 (X. Moisson, P. Bretagnon, 2001, Celes. Mechanics &
           Dyn. Astron., 80, 3/4, 205-213) and is an adaptation of original
           Fortran code supplied by P. Bretagnon (private comm., 2000).

        4) Comparisons over the time span 1900-2100 with this simplified
           solution and the JPL DE405 ephemeris give the following results:

                                      RMS    max
                 Heliocentric:
                    position error    3.7   11.2   km
                    velocity error    1.4    5.0   mm/s

                 Barycentric:
                    position error    4.6   13.4   km
                    velocity error    1.4    4.9   mm/s

           Comparisons with the JPL DE406 ephemeris show that by 1800 and
           2200 the position errors are approximately double their 1900-2100
           size.  By 1500 and 2500 the deterioration is a factor of 10 and
           by 1000 and 3000 a factor of 60.  The velocity accuracy falls off
           at about half that rate.

        5) It is permissible to use the same array for pvh and pvb, which
           will receive the barycentric values.

        This revision:  2019 June 23

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    pvh, pvb, c_retval = ufunc.epv00(date1, date2)
    check_errwarn(c_retval, 'epv00')
    return pvh, pvb


STATUS_CODES['epv00'] = {
    0: 'OK',
    1: 'warning: date outsidethe range 1900-2100 AD',
}


def plan94(date1, date2, np):
    """
    Approximate heliocentric position and velocity of a nominated major
    planet:  Mercury, Venus, EMB, Mars, Jupiter, Saturn, Uranus or
    Neptune (but not the Earth itself).

    Parameters
    ----------
    date1 : double array
    date2 : double array
    np : int array

    Returns
    -------
    pv : double array

    Notes
    -----
    Wraps ERFA function ``eraPlan94``. The ERFA documentation is::

        - - - - - - - - - -
         e r a P l a n 9 4
        - - - - - - - - - -

        Approximate heliocentric position and velocity of a nominated major
        planet:  Mercury, Venus, EMB, Mars, Jupiter, Saturn, Uranus or
        Neptune (but not the Earth itself).

        Given:
           date1  double       TDB date part A (Note 1)
           date2  double       TDB date part B (Note 1)
           np     int          planet (1=Mercury, 2=Venus, 3=EMB, 4=Mars,
                                   5=Jupiter, 6=Saturn, 7=Uranus, 8=Neptune)

        Returned (argument):
           pv     double[2][3] planet p,v (heliocentric, J2000.0, au,au/d)

        Returned (function value):
                  int          status: -1 = illegal NP (outside 1-8)
                                        0 = OK
                                       +1 = warning: year outside 1000-3000
                                       +2 = warning: failed to converge

        Notes:

        1) The date date1+date2 is in the TDB time scale (in practice TT can
           be used) and is a Julian Date, apportioned in any convenient way
           between the two arguments.  For example, JD(TDB)=2450123.7 could
           be expressed in any of these ways, among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 method is best matched to the way the
           argument is handled internally and will deliver the optimum
           resolution.  The MJD method and the date & time methods are both
           good compromises between resolution and convenience.  The limited
           accuracy of the present algorithm is such that any of the methods
           is satisfactory.

        2) If an np value outside the range 1-8 is supplied, an error status
           (function value -1) is returned and the pv vector set to zeroes.

        3) For np=3 the result is for the Earth-Moon Barycenter.  To obtain
           the heliocentric position and velocity of the Earth, use instead
           the ERFA function eraEpv00.

        4) On successful return, the array pv contains the following:

              pv[0][0]   x      }
              pv[0][1]   y      } heliocentric position, au
              pv[0][2]   z      }

              pv[1][0]   xdot   }
              pv[1][1]   ydot   } heliocentric velocity, au/d
              pv[1][2]   zdot   }

           The reference frame is equatorial and is with respect to the
           mean equator and equinox of epoch J2000.0.

        5) The algorithm is due to J.L. Simon, P. Bretagnon, J. Chapront,
           M. Chapront-Touze, G. Francou and J. Laskar (Bureau des
           Longitudes, Paris, France).  From comparisons with JPL
           ephemeris DE102, they quote the following maximum errors
           over the interval 1800-2050:

                           L (arcsec)    B (arcsec)      R (km)

              Mercury          4             1             300
              Venus            5             1             800
              EMB              6             1            1000
              Mars            17             1            7700
              Jupiter         71             5           76000
              Saturn          81            13          267000
              Uranus          86             7          712000
              Neptune         11             1          253000

           Over the interval 1000-3000, they report that the accuracy is no
           worse than 1.5 times that over 1800-2050.  Outside 1000-3000 the
           accuracy declines.

           Comparisons of the present function with the JPL DE200 ephemeris
           give the following RMS errors over the interval 1960-2025:

                            position (km)     velocity (m/s)

              Mercury            334               0.437
              Venus             1060               0.855
              EMB               2010               0.815
              Mars              7690               1.98
              Jupiter          71700               7.70
              Saturn          199000              19.4
              Uranus          564000              16.4
              Neptune         158000              14.4

           Comparisons against DE200 over the interval 1800-2100 gave the
           following maximum absolute differences.  (The results using
           DE406 were essentially the same.)

                         L (arcsec)   B (arcsec)     R (km)   Rdot (m/s)

              Mercury        7            1            500       0.7
              Venus          7            1           1100       0.9
              EMB            9            1           1300       1.0
              Mars          26            1           9000       2.5
              Jupiter       78            6          82000       8.2
              Saturn        87           14         263000      24.6
              Uranus        86            7         661000      27.4
              Neptune       11            2         248000      21.4

        6) The present ERFA re-implementation of the original Simon et al.
           Fortran code differs from the original in the following respects:

                     C instead of Fortran.

                     The date is supplied in two parts.

                     The result is returned only in equatorial Cartesian form;
                the ecliptic longitude, latitude and radius vector are not
                returned.

                     The result is in the J2000.0 equatorial frame, not ecliptic.

                     More is done in-line: there are fewer calls to subroutines.

                     Different error/warning status values are used.

                     A different Kepler's-equation-solver is used (avoiding
                use of double precision complex).

                     Polynomials in t are nested to minimize rounding errors.

                     Explicit double constants are used to avoid mixed-mode
                expressions.

           None of the above changes affects the result significantly.

        7) The returned status indicates the most serious condition
           encountered during execution of the function.  Illegal np is
           considered the most serious, overriding failure to converge,
           which in turn takes precedence over the remote date warning.

        Called:
           eraAnp       normalize angle into range 0 to 2pi

        Reference:  Simon, J.L, Bretagnon, P., Chapront, J.,
                    Chapront-Touze, M., Francou, G., and Laskar, J.,
                    Astron.Astrophys., 282, 663 (1994).

        This revision:  2019 June 23

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    pv, c_retval = ufunc.plan94(date1, date2, np)
    check_errwarn(c_retval, 'plan94')
    return pv


STATUS_CODES['plan94'] = {
    -1: 'illegal NP (outside 1-8)',
    0: 'OK',
    1: 'warning: year outside 1000-3000',
    2: 'warning: failed to converge',
}


def fad03(t):
    """
    Fundamental argument, IERS Conventions (2003):
    mean elongation of the Moon from the Sun.

    Parameters
    ----------
    t : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraFad03``. The ERFA documentation is::

        - - - - - - - - -
         e r a F a d 0 3
        - - - - - - - - -

        Fundamental argument, IERS Conventions (2003):
        mean elongation of the Moon from the Sun.

        Given:
           t     double    TDB, Julian centuries since J2000.0 (Note 1)

        Returned (function value):
                 double    D, radians (Note 2)

        Notes:

        1) Though t is strictly TDB, it is usually more convenient to use
           TT, which makes no significant difference.

        2) The expression used is as adopted in IERS Conventions (2003) and
           is from Simon et al. (1994).

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J. 1994, Astron.Astrophys. 282, 663-683

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.fad03(t)
    return c_retval


def fae03(t):
    """
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Earth.

    Parameters
    ----------
    t : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraFae03``. The ERFA documentation is::

        - - - - - - - - -
         e r a F a e 0 3
        - - - - - - - - -

        Fundamental argument, IERS Conventions (2003):
        mean longitude of Earth.

        Given:
           t     double    TDB, Julian centuries since J2000.0 (Note 1)

        Returned (function value):
                 double    mean longitude of Earth, radians (Note 2)

        Notes:

        1) Though t is strictly TDB, it is usually more convenient to use
           TT, which makes no significant difference.

        2) The expression used is as adopted in IERS Conventions (2003) and
           comes from Souchay et al. (1999) after Simon et al. (1994).

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J. 1994, Astron.Astrophys. 282, 663-683

           Souchay, J., Loysel, B., Kinoshita, H., Folgueira, M. 1999,
           Astron.Astrophys.Supp.Ser. 135, 111

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.fae03(t)
    return c_retval


def faf03(t):
    """
    Fundamental argument, IERS Conventions (2003):
    mean longitude of the Moon minus mean longitude of the ascending
    node.

    Parameters
    ----------
    t : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraFaf03``. The ERFA documentation is::

        - - - - - - - - -
         e r a F a f 0 3
        - - - - - - - - -

        Fundamental argument, IERS Conventions (2003):
        mean longitude of the Moon minus mean longitude of the ascending
        node.

        Given:
           t     double    TDB, Julian centuries since J2000.0 (Note 1)

        Returned (function value):
                 double    F, radians (Note 2)

        Notes:

        1) Though t is strictly TDB, it is usually more convenient to use
           TT, which makes no significant difference.

        2) The expression used is as adopted in IERS Conventions (2003) and
           is from Simon et al. (1994).

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J. 1994, Astron.Astrophys. 282, 663-683

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.faf03(t)
    return c_retval


def faju03(t):
    """
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Jupiter.

    Parameters
    ----------
    t : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraFaju03``. The ERFA documentation is::

        - - - - - - - - - -
         e r a F a j u 0 3
        - - - - - - - - - -

        Fundamental argument, IERS Conventions (2003):
        mean longitude of Jupiter.

        Given:
           t     double    TDB, Julian centuries since J2000.0 (Note 1)

        Returned (function value):
                 double    mean longitude of Jupiter, radians (Note 2)

        Notes:

        1) Though t is strictly TDB, it is usually more convenient to use
           TT, which makes no significant difference.

        2) The expression used is as adopted in IERS Conventions (2003) and
           comes from Souchay et al. (1999) after Simon et al. (1994).

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J. 1994, Astron.Astrophys. 282, 663-683

           Souchay, J., Loysel, B., Kinoshita, H., Folgueira, M. 1999,
           Astron.Astrophys.Supp.Ser. 135, 111

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.faju03(t)
    return c_retval


def fal03(t):
    """
    Fundamental argument, IERS Conventions (2003):
    mean anomaly of the Moon.

    Parameters
    ----------
    t : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraFal03``. The ERFA documentation is::

        - - - - - - - - -
         e r a F a l 0 3
        - - - - - - - - -

        Fundamental argument, IERS Conventions (2003):
        mean anomaly of the Moon.

        Given:
           t     double    TDB, Julian centuries since J2000.0 (Note 1)

        Returned (function value):
                 double    l, radians (Note 2)

        Notes:

        1) Though t is strictly TDB, it is usually more convenient to use
           TT, which makes no significant difference.

        2) The expression used is as adopted in IERS Conventions (2003) and
           is from Simon et al. (1994).

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J. 1994, Astron.Astrophys. 282, 663-683

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.fal03(t)
    return c_retval


def falp03(t):
    """
    Fundamental argument, IERS Conventions (2003):
    mean anomaly of the Sun.

    Parameters
    ----------
    t : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraFalp03``. The ERFA documentation is::

        - - - - - - - - - -
         e r a F a l p 0 3
        - - - - - - - - - -

        Fundamental argument, IERS Conventions (2003):
        mean anomaly of the Sun.

        Given:
           t     double    TDB, Julian centuries since J2000.0 (Note 1)

        Returned (function value):
                 double    l', radians (Note 2)

        Notes:

        1) Though t is strictly TDB, it is usually more convenient to use
           TT, which makes no significant difference.

        2) The expression used is as adopted in IERS Conventions (2003) and
           is from Simon et al. (1994).

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J. 1994, Astron.Astrophys. 282, 663-683

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.falp03(t)
    return c_retval


def fama03(t):
    """
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Mars.

    Parameters
    ----------
    t : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraFama03``. The ERFA documentation is::

        - - - - - - - - - -
         e r a F a m a 0 3
        - - - - - - - - - -

        Fundamental argument, IERS Conventions (2003):
        mean longitude of Mars.

        Given:
           t     double    TDB, Julian centuries since J2000.0 (Note 1)

        Returned (function value):
                 double    mean longitude of Mars, radians (Note 2)

        Notes:

        1) Though t is strictly TDB, it is usually more convenient to use
           TT, which makes no significant difference.

        2) The expression used is as adopted in IERS Conventions (2003) and
           comes from Souchay et al. (1999) after Simon et al. (1994).

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J. 1994, Astron.Astrophys. 282, 663-683

           Souchay, J., Loysel, B., Kinoshita, H., Folgueira, M. 1999,
           Astron.Astrophys.Supp.Ser. 135, 111

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.fama03(t)
    return c_retval


def fame03(t):
    """
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Mercury.

    Parameters
    ----------
    t : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraFame03``. The ERFA documentation is::

        - - - - - - - - - -
         e r a F a m e 0 3
        - - - - - - - - - -

        Fundamental argument, IERS Conventions (2003):
        mean longitude of Mercury.

        Given:
           t     double    TDB, Julian centuries since J2000.0 (Note 1)

        Returned (function value):
                 double    mean longitude of Mercury, radians (Note 2)

        Notes:

        1) Though t is strictly TDB, it is usually more convenient to use
           TT, which makes no significant difference.

        2) The expression used is as adopted in IERS Conventions (2003) and
           comes from Souchay et al. (1999) after Simon et al. (1994).

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J. 1994, Astron.Astrophys. 282, 663-683

           Souchay, J., Loysel, B., Kinoshita, H., Folgueira, M. 1999,
           Astron.Astrophys.Supp.Ser. 135, 111

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.fame03(t)
    return c_retval


def fane03(t):
    """
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Neptune.

    Parameters
    ----------
    t : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraFane03``. The ERFA documentation is::

        - - - - - - - - - -
         e r a F a n e 0 3
        - - - - - - - - - -

        Fundamental argument, IERS Conventions (2003):
        mean longitude of Neptune.

        Given:
           t     double    TDB, Julian centuries since J2000.0 (Note 1)

        Returned (function value):
                 double    mean longitude of Neptune, radians (Note 2)

        Notes:

        1) Though t is strictly TDB, it is usually more convenient to use
           TT, which makes no significant difference.

        2) The expression used is as adopted in IERS Conventions (2003) and
           is adapted from Simon et al. (1994).

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J. 1994, Astron.Astrophys. 282, 663-683

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.fane03(t)
    return c_retval


def faom03(t):
    """
    Fundamental argument, IERS Conventions (2003):
    mean longitude of the Moon's ascending node.

    Parameters
    ----------
    t : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraFaom03``. The ERFA documentation is::

        - - - - - - - - - -
         e r a F a o m 0 3
        - - - - - - - - - -

        Fundamental argument, IERS Conventions (2003):
        mean longitude of the Moon's ascending node.

        Given:
           t     double    TDB, Julian centuries since J2000.0 (Note 1)

        Returned (function value):
                 double    Omega, radians (Note 2)

        Notes:

        1) Though t is strictly TDB, it is usually more convenient to use
           TT, which makes no significant difference.

        2) The expression used is as adopted in IERS Conventions (2003) and
           is from Simon et al. (1994).

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J., 1994, Astron.Astrophys. 282, 663-683.

        This revision:  2020 November 16

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.faom03(t)
    return c_retval


def fapa03(t):
    """
    Fundamental argument, IERS Conventions (2003):
    general accumulated precession in longitude.

    Parameters
    ----------
    t : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraFapa03``. The ERFA documentation is::

        - - - - - - - - - -
         e r a F a p a 0 3
        - - - - - - - - - -

        Fundamental argument, IERS Conventions (2003):
        general accumulated precession in longitude.

        Given:
           t     double    TDB, Julian centuries since J2000.0 (Note 1)

        Returned (function value):
                 double    general precession in longitude, radians (Note 2)

        Notes:

        1) Though t is strictly TDB, it is usually more convenient to use
           TT, which makes no significant difference.

        2) The expression used is as adopted in IERS Conventions (2003).  It
           is taken from Kinoshita & Souchay (1990) and comes originally
           from Lieske et al. (1977).

        References:

           Kinoshita, H. and Souchay J. 1990, Celest.Mech. and Dyn.Astron.
           48, 187

           Lieske, J.H., Lederle, T., Fricke, W. & Morando, B. 1977,
           Astron.Astrophys. 58, 1-16

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.fapa03(t)
    return c_retval


def fasa03(t):
    """
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Saturn.

    Parameters
    ----------
    t : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraFasa03``. The ERFA documentation is::

        - - - - - - - - - -
         e r a F a s a 0 3
        - - - - - - - - - -

        Fundamental argument, IERS Conventions (2003):
        mean longitude of Saturn.

        Given:
           t     double    TDB, Julian centuries since J2000.0 (Note 1)

        Returned (function value):
                 double    mean longitude of Saturn, radians (Note 2)

        Notes:

        1) Though t is strictly TDB, it is usually more convenient to use
           TT, which makes no significant difference.

        2) The expression used is as adopted in IERS Conventions (2003) and
           comes from Souchay et al. (1999) after Simon et al. (1994).

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J. 1994, Astron.Astrophys. 282, 663-683

           Souchay, J., Loysel, B., Kinoshita, H., Folgueira, M. 1999,
           Astron.Astrophys.Supp.Ser. 135, 111

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.fasa03(t)
    return c_retval


def faur03(t):
    """
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Uranus.

    Parameters
    ----------
    t : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraFaur03``. The ERFA documentation is::

        - - - - - - - - - -
         e r a F a u r 0 3
        - - - - - - - - - -

        Fundamental argument, IERS Conventions (2003):
        mean longitude of Uranus.

        Given:
           t     double    TDB, Julian centuries since J2000.0 (Note 1)

        Returned  (function value):
                 double    mean longitude of Uranus, radians (Note 2)

        Notes:

        1) Though t is strictly TDB, it is usually more convenient to use
           TT, which makes no significant difference.

        2) The expression used is as adopted in IERS Conventions (2003) and
           is adapted from Simon et al. (1994).

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J. 1994, Astron.Astrophys. 282, 663-683

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.faur03(t)
    return c_retval


def fave03(t):
    """
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Venus.

    Parameters
    ----------
    t : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraFave03``. The ERFA documentation is::

        - - - - - - - - - -
         e r a F a v e 0 3
        - - - - - - - - - -

        Fundamental argument, IERS Conventions (2003):
        mean longitude of Venus.

        Given:
           t     double    TDB, Julian centuries since J2000.0 (Note 1)

        Returned (function value):
                 double    mean longitude of Venus, radians (Note 2)

        Notes:

        1) Though t is strictly TDB, it is usually more convenient to use
           TT, which makes no significant difference.

        2) The expression used is as adopted in IERS Conventions (2003) and
           comes from Souchay et al. (1999) after Simon et al. (1994).

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J. 1994, Astron.Astrophys. 282, 663-683

           Souchay, J., Loysel, B., Kinoshita, H., Folgueira, M. 1999,
           Astron.Astrophys.Supp.Ser. 135, 111

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.fave03(t)
    return c_retval


def bi00():
    """
    Frame bias components of IAU 2000 precession-nutation models;  part
    of the Mathews-Herring-Buffett (MHB2000) nutation series, with
    additions.

    Parameters
    ----------

    Returns
    -------
    dpsibi : double array
    depsbi : double array
    dra : double array

    Notes
    -----
    Wraps ERFA function ``eraBi00``. The ERFA documentation is::

        - - - - - - - -
         e r a B i 0 0
        - - - - - - - -

        Frame bias components of IAU 2000 precession-nutation models;  part
        of the Mathews-Herring-Buffett (MHB2000) nutation series, with
        additions.

        Returned:
           dpsibi,depsbi  double  longitude and obliquity corrections
           dra            double  the ICRS RA of the J2000.0 mean equinox

        Notes:

        1) The frame bias corrections in longitude and obliquity (radians)
           are required in order to correct for the offset between the GCRS
           pole and the mean J2000.0 pole.  They define, with respect to the
           GCRS frame, a J2000.0 mean pole that is consistent with the rest
           of the IAU 2000A precession-nutation model.

        2) In addition to the displacement of the pole, the complete
           description of the frame bias requires also an offset in right
           ascension.  This is not part of the IAU 2000A model, and is from
           Chapront et al. (2002).  It is returned in radians.

        3) This is a supplemented implementation of one aspect of the IAU
           2000A nutation model, formally adopted by the IAU General
           Assembly in 2000, namely MHB2000 (Mathews et al. 2002).

        References:

           Chapront, J., Chapront-Touze, M. & Francou, G., Astron.
           Astrophys., 387, 700, 2002.

           Mathews, P.M., Herring, T.A., Buffet, B.A., "Modeling of nutation
           and precession:  New nutation series for nonrigid Earth and
           insights into the Earth's interior", J.Geophys.Res., 107, B4,
           2002.  The MHB2000 code itself was obtained on 2002 September 9
           from ftp://maia.usno.navy.mil/conv2000/chapter5/IAU2000A.

        This revision:  2020 October 29

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dpsibi, depsbi, dra = ufunc.bi00()
    return dpsibi, depsbi, dra


def bp00(date1, date2):
    """
    Frame bias and precession, IAU 2000.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rb : double array
    rp : double array
    rbp : double array

    Notes
    -----
    Wraps ERFA function ``eraBp00``. The ERFA documentation is::

        - - - - - - - -
         e r a B p 0 0
        - - - - - - - -

        Frame bias and precession, IAU 2000.

        Given:
           date1,date2  double         TT as a 2-part Julian Date (Note 1)

        Returned:
           rb           double[3][3]   frame bias matrix (Note 2)
           rp           double[3][3]   precession matrix (Note 3)
           rbp          double[3][3]   bias-precession matrix (Note 4)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                   date1         date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix rb transforms vectors from GCRS to mean J2000.0 by
           applying frame bias.

        3) The matrix rp transforms vectors from J2000.0 mean equator and
           equinox to mean equator and equinox of date by applying
           precession.

        4) The matrix rbp transforms vectors from GCRS to mean equator and
           equinox of date by applying frame bias then precession.  It is
           the product rp x rb.

        5) It is permissible to re-use the same array in the returned
           arguments.  The arrays are filled in the order given.

        Called:
           eraBi00      frame bias components, IAU 2000
           eraPr00      IAU 2000 precession adjustments
           eraIr        initialize r-matrix to identity
           eraRx        rotate around X-axis
           eraRy        rotate around Y-axis
           eraRz        rotate around Z-axis
           eraCr        copy r-matrix
           eraRxr       product of two r-matrices

        Reference:
           "Expressions for the Celestial Intermediate Pole and Celestial
           Ephemeris Origin consistent with the IAU 2000A precession-
           nutation model", Astron.Astrophys. 400, 1145-1154 (2003)

           n.b. The celestial ephemeris origin (CEO) was renamed "celestial
                intermediate origin" (CIO) by IAU 2006 Resolution 2.

        This revision:  2013 August 21

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rb, rp, rbp = ufunc.bp00(date1, date2)
    return rb, rp, rbp


def bp06(date1, date2):
    """
    Frame bias and precession, IAU 2006.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rb : double array
    rp : double array
    rbp : double array

    Notes
    -----
    Wraps ERFA function ``eraBp06``. The ERFA documentation is::

        - - - - - - - -
         e r a B p 0 6
        - - - - - - - -

        Frame bias and precession, IAU 2006.

        Given:
           date1,date2  double         TT as a 2-part Julian Date (Note 1)

        Returned:
           rb           double[3][3]   frame bias matrix (Note 2)
           rp           double[3][3]   precession matrix (Note 3)
           rbp          double[3][3]   bias-precession matrix (Note 4)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                   date1         date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix rb transforms vectors from GCRS to mean J2000.0 by
           applying frame bias.

        3) The matrix rp transforms vectors from mean J2000.0 to mean of
           date by applying precession.

        4) The matrix rbp transforms vectors from GCRS to mean of date by
           applying frame bias then precession.  It is the product rp x rb.

        5) It is permissible to re-use the same array in the returned
           arguments.  The arrays are filled in the order given.

        Called:
           eraPfw06     bias-precession F-W angles, IAU 2006
           eraFw2m      F-W angles to r-matrix
           eraPmat06    PB matrix, IAU 2006
           eraTr        transpose r-matrix
           eraRxr       product of two r-matrices
           eraCr        copy r-matrix

        References:

           Capitaine, N. & Wallace, P.T., 2006, Astron.Astrophys. 450, 855

           Wallace, P.T. & Capitaine, N., 2006, Astron.Astrophys. 459, 981

        This revision:  2013 August 21

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rb, rp, rbp = ufunc.bp06(date1, date2)
    return rb, rp, rbp


def bpn2xy(rbpn):
    """
    Extract from the bias-precession-nutation matrix the X,Y coordinates
    of the Celestial Intermediate Pole.

    Parameters
    ----------
    rbpn : double array

    Returns
    -------
    x : double array
    y : double array

    Notes
    -----
    Wraps ERFA function ``eraBpn2xy``. The ERFA documentation is::

        - - - - - - - - - -
         e r a B p n 2 x y
        - - - - - - - - - -

        Extract from the bias-precession-nutation matrix the X,Y coordinates
        of the Celestial Intermediate Pole.

        Given:
           rbpn      double[3][3]  celestial-to-true matrix (Note 1)

        Returned:
           x,y       double        Celestial Intermediate Pole (Note 2)

        Notes:

        1) The matrix rbpn transforms vectors from GCRS to true equator (and
           CIO or equinox) of date, and therefore the Celestial Intermediate
           Pole unit vector is the bottom row of the matrix.

        2) The arguments x,y are components of the Celestial Intermediate
           Pole unit vector in the Geocentric Celestial Reference System.

        Reference:

           "Expressions for the Celestial Intermediate Pole and Celestial
           Ephemeris Origin consistent with the IAU 2000A precession-
           nutation model", Astron.Astrophys. 400, 1145-1154
           (2003)

           n.b. The celestial ephemeris origin (CEO) was renamed "celestial
                intermediate origin" (CIO) by IAU 2006 Resolution 2.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    x, y = ufunc.bpn2xy(rbpn)
    return x, y


def c2i00a(date1, date2):
    """
    Form the celestial-to-intermediate matrix for a given date using the
    IAU 2000A precession-nutation model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rc2i : double array

    Notes
    -----
    Wraps ERFA function ``eraC2i00a``. The ERFA documentation is::

        - - - - - - - - - -
         e r a C 2 i 0 0 a
        - - - - - - - - - -

        Form the celestial-to-intermediate matrix for a given date using the
        IAU 2000A precession-nutation model.

        Given:
           date1,date2 double       TT as a 2-part Julian Date (Note 1)

        Returned:
           rc2i        double[3][3] celestial-to-intermediate matrix (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix rc2i is the first stage in the transformation from
           celestial to terrestrial coordinates:

              [TRS]  =  RPOM * R_3(ERA) * rc2i * [CRS]

                     =  rc2t * [CRS]

           where [CRS] is a vector in the Geocentric Celestial Reference
           System and [TRS] is a vector in the International Terrestrial
           Reference System (see IERS Conventions 2003), ERA is the Earth
           Rotation Angle and RPOM is the polar motion matrix.

        3) A faster, but slightly less accurate, result (about 1 mas) can be
           obtained by using instead the eraC2i00b function.

        Called:
           eraPnm00a    classical NPB matrix, IAU 2000A
           eraC2ibpn    celestial-to-intermediate matrix, given NPB matrix

        References:

           "Expressions for the Celestial Intermediate Pole and Celestial
           Ephemeris Origin consistent with the IAU 2000A precession-
           nutation model", Astron.Astrophys. 400, 1145-1154
           (2003)

           n.b. The celestial ephemeris origin (CEO) was renamed "celestial
                intermediate origin" (CIO) by IAU 2006 Resolution 2.

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2020 November 27

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc2i = ufunc.c2i00a(date1, date2)
    return rc2i


def c2i00b(date1, date2):
    """
    Form the celestial-to-intermediate matrix for a given date using the
    IAU 2000B precession-nutation model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rc2i : double array

    Notes
    -----
    Wraps ERFA function ``eraC2i00b``. The ERFA documentation is::

        - - - - - - - - - -
         e r a C 2 i 0 0 b
        - - - - - - - - - -

        Form the celestial-to-intermediate matrix for a given date using the
        IAU 2000B precession-nutation model.

        Given:
           date1,date2 double       TT as a 2-part Julian Date (Note 1)

        Returned:
           rc2i        double[3][3] celestial-to-intermediate matrix (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix rc2i is the first stage in the transformation from
           celestial to terrestrial coordinates:

              [TRS]  =  RPOM * R_3(ERA) * rc2i * [CRS]

                     =  rc2t * [CRS]

           where [CRS] is a vector in the Geocentric Celestial Reference
           System and [TRS] is a vector in the International Terrestrial
           Reference System (see IERS Conventions 2003), ERA is the Earth
           Rotation Angle and RPOM is the polar motion matrix.

        3) The present function is faster, but slightly less accurate (about
           1 mas), than the eraC2i00a function.

        Called:
           eraPnm00b    classical NPB matrix, IAU 2000B
           eraC2ibpn    celestial-to-intermediate matrix, given NPB matrix

        References:

           "Expressions for the Celestial Intermediate Pole and Celestial
           Ephemeris Origin consistent with the IAU 2000A precession-
           nutation model", Astron.Astrophys. 400, 1145-1154
           (2003)

           n.b. The celestial ephemeris origin (CEO) was renamed "celestial
                intermediate origin" (CIO) by IAU 2006 Resolution 2.

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc2i = ufunc.c2i00b(date1, date2)
    return rc2i


def c2i06a(date1, date2):
    """
    Form the celestial-to-intermediate matrix for a given date using the
    IAU 2006 precession and IAU 2000A nutation models.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rc2i : double array

    Notes
    -----
    Wraps ERFA function ``eraC2i06a``. The ERFA documentation is::

        - - - - - - - - - -
         e r a C 2 i 0 6 a
        - - - - - - - - - -

        Form the celestial-to-intermediate matrix for a given date using the
        IAU 2006 precession and IAU 2000A nutation models.

        Given:
           date1,date2 double       TT as a 2-part Julian Date (Note 1)

        Returned:
           rc2i        double[3][3] celestial-to-intermediate matrix (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix rc2i is the first stage in the transformation from
           celestial to terrestrial coordinates:

              [TRS]  =  RPOM * R_3(ERA) * rc2i * [CRS]

                     =  RC2T * [CRS]

           where [CRS] is a vector in the Geocentric Celestial Reference
           System and [TRS] is a vector in the International Terrestrial
           Reference System (see IERS Conventions 2003), ERA is the Earth
           Rotation Angle and RPOM is the polar motion matrix.

        Called:
           eraPnm06a    classical NPB matrix, IAU 2006/2000A
           eraBpn2xy    extract CIP X,Y coordinates from NPB matrix
           eraS06       the CIO locator s, given X,Y, IAU 2006
           eraC2ixys    celestial-to-intermediate matrix, given X,Y and s

        References:

           McCarthy, D. D., Petit, G. (eds.), 2004, IERS Conventions (2003),
           IERS Technical Note No. 32, BKG

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc2i = ufunc.c2i06a(date1, date2)
    return rc2i


def c2ibpn(date1, date2, rbpn):
    """
    Form the celestial-to-intermediate matrix for a given date given
    the bias-precession-nutation matrix.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    rbpn : double array

    Returns
    -------
    rc2i : double array

    Notes
    -----
    Wraps ERFA function ``eraC2ibpn``. The ERFA documentation is::

        - - - - - - - - - -
         e r a C 2 i b p n
        - - - - - - - - - -

        Form the celestial-to-intermediate matrix for a given date given
        the bias-precession-nutation matrix.  IAU 2000.

        Given:
           date1,date2 double       TT as a 2-part Julian Date (Note 1)
           rbpn        double[3][3] celestial-to-true matrix (Note 2)

        Returned:
           rc2i        double[3][3] celestial-to-intermediate matrix (Note 3)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix rbpn transforms vectors from GCRS to true equator (and
           CIO or equinox) of date.  Only the CIP (bottom row) is used.

        3) The matrix rc2i is the first stage in the transformation from
           celestial to terrestrial coordinates:

              [TRS] = RPOM * R_3(ERA) * rc2i * [CRS]

                    = RC2T * [CRS]

           where [CRS] is a vector in the Geocentric Celestial Reference
           System and [TRS] is a vector in the International Terrestrial
           Reference System (see IERS Conventions 2003), ERA is the Earth
           Rotation Angle and RPOM is the polar motion matrix.

        4) Although its name does not include "00", This function is in fact
           specific to the IAU 2000 models.

        Called:
           eraBpn2xy    extract CIP X,Y coordinates from NPB matrix
           eraC2ixy     celestial-to-intermediate matrix, given X,Y

        References:
           "Expressions for the Celestial Intermediate Pole and Celestial
           Ephemeris Origin consistent with the IAU 2000A precession-
           nutation model", Astron.Astrophys. 400, 1145-1154 (2003)

           n.b. The celestial ephemeris origin (CEO) was renamed "celestial
                intermediate origin" (CIO) by IAU 2006 Resolution 2.

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc2i = ufunc.c2ibpn(date1, date2, rbpn)
    return rc2i


def c2ixy(date1, date2, x, y):
    """
    Form the celestial to intermediate-frame-of-date matrix for a given
    date when the CIP X,Y coordinates are known.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    x : double array
    y : double array

    Returns
    -------
    rc2i : double array

    Notes
    -----
    Wraps ERFA function ``eraC2ixy``. The ERFA documentation is::

        - - - - - - - - -
         e r a C 2 i x y
        - - - - - - - - -

        Form the celestial to intermediate-frame-of-date matrix for a given
        date when the CIP X,Y coordinates are known.  IAU 2000.

        Given:
           date1,date2 double       TT as a 2-part Julian Date (Note 1)
           x,y         double       Celestial Intermediate Pole (Note 2)

        Returned:
           rc2i        double[3][3] celestial-to-intermediate matrix (Note 3)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The Celestial Intermediate Pole coordinates are the x,y components
           of the unit vector in the Geocentric Celestial Reference System.

        3) The matrix rc2i is the first stage in the transformation from
           celestial to terrestrial coordinates:

              [TRS] = RPOM * R_3(ERA) * rc2i * [CRS]

                    = RC2T * [CRS]

           where [CRS] is a vector in the Geocentric Celestial Reference
           System and [TRS] is a vector in the International Terrestrial
           Reference System (see IERS Conventions 2003), ERA is the Earth
           Rotation Angle and RPOM is the polar motion matrix.

        4) Although its name does not include "00", This function is in fact
           specific to the IAU 2000 models.

        Called:
           eraC2ixys    celestial-to-intermediate matrix, given X,Y and s
           eraS00       the CIO locator s, given X,Y, IAU 2000A

        Reference:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc2i = ufunc.c2ixy(date1, date2, x, y)
    return rc2i


def c2ixys(x, y, s):
    """
    Form the celestial to intermediate-frame-of-date matrix given the CIP
    X,Y and the CIO locator s.

    Parameters
    ----------
    x : double array
    y : double array
    s : double array

    Returns
    -------
    rc2i : double array

    Notes
    -----
    Wraps ERFA function ``eraC2ixys``. The ERFA documentation is::

        - - - - - - - - - -
         e r a C 2 i x y s
        - - - - - - - - - -

        Form the celestial to intermediate-frame-of-date matrix given the CIP
        X,Y and the CIO locator s.

        Given:
           x,y      double         Celestial Intermediate Pole (Note 1)
           s        double         the CIO locator s (Note 2)

        Returned:
           rc2i     double[3][3]   celestial-to-intermediate matrix (Note 3)

        Notes:

        1) The Celestial Intermediate Pole coordinates are the x,y
           components of the unit vector in the Geocentric Celestial
           Reference System.

        2) The CIO locator s (in radians) positions the Celestial
           Intermediate Origin on the equator of the CIP.

        3) The matrix rc2i is the first stage in the transformation from
           celestial to terrestrial coordinates:

              [TRS] = RPOM * R_3(ERA) * rc2i * [CRS]

                    = RC2T * [CRS]

           where [CRS] is a vector in the Geocentric Celestial Reference
           System and [TRS] is a vector in the International Terrestrial
           Reference System (see IERS Conventions 2003), ERA is the Earth
           Rotation Angle and RPOM is the polar motion matrix.

        Called:
           eraIr        initialize r-matrix to identity
           eraRz        rotate around Z-axis
           eraRy        rotate around Y-axis

        Reference:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2014 November 7

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc2i = ufunc.c2ixys(x, y, s)
    return rc2i


def c2t00a(tta, ttb, uta, utb, xp, yp):
    """
    Form the celestial to terrestrial matrix given the date, the UT1 and
    the polar motion, using the IAU 2000A precession-nutation model.

    Parameters
    ----------
    tta : double array
    ttb : double array
    uta : double array
    utb : double array
    xp : double array
    yp : double array

    Returns
    -------
    rc2t : double array

    Notes
    -----
    Wraps ERFA function ``eraC2t00a``. The ERFA documentation is::

        - - - - - - - - - -
         e r a C 2 t 0 0 a
        - - - - - - - - - -

        Form the celestial to terrestrial matrix given the date, the UT1 and
        the polar motion, using the IAU 2000A precession-nutation model.

        Given:
           tta,ttb  double         TT as a 2-part Julian Date (Note 1)
           uta,utb  double         UT1 as a 2-part Julian Date (Note 1)
           xp,yp    double         CIP coordinates (radians, Note 2)

        Returned:
           rc2t     double[3][3]   celestial-to-terrestrial matrix (Note 3)

        Notes:

        1) The TT and UT1 dates tta+ttb and uta+utb are Julian Dates,
           apportioned in any convenient way between the arguments uta and
           utb.  For example, JD(UT1)=2450123.7 could be expressed in any of
           these ways, among others:

                   uta            utb

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution is
           acceptable.  The J2000 and MJD methods are good compromises
           between resolution and convenience.  In the case of uta,utb, the
           date & time method is best matched to the Earth rotation angle
           algorithm used:  maximum precision is delivered when the uta
           argument is for 0hrs UT1 on the day in question and the utb
           argument lies in the range 0 to 1, or vice versa.

        2) The arguments xp and yp are the coordinates (in radians) of the
           Celestial Intermediate Pole with respect to the International
           Terrestrial Reference System (see IERS Conventions 2003),
           measured along the meridians 0 and 90 deg west respectively.

        3) The matrix rc2t transforms from celestial to terrestrial
           coordinates:

              [TRS] = RPOM * R_3(ERA) * RC2I * [CRS]

                    = rc2t * [CRS]

           where [CRS] is a vector in the Geocentric Celestial Reference
           System and [TRS] is a vector in the International Terrestrial
           Reference System (see IERS Conventions 2003), RC2I is the
           celestial-to-intermediate matrix, ERA is the Earth rotation
           angle and RPOM is the polar motion matrix.

        4) A faster, but slightly less accurate, result (about 1 mas) can
           be obtained by using instead the eraC2t00b function.

        Called:
           eraC2i00a    celestial-to-intermediate matrix, IAU 2000A
           eraEra00     Earth rotation angle, IAU 2000
           eraSp00      the TIO locator s', IERS 2000
           eraPom00     polar motion matrix
           eraC2tcio    form CIO-based celestial-to-terrestrial matrix

        Reference:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2020 November 27

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc2t = ufunc.c2t00a(tta, ttb, uta, utb, xp, yp)
    return rc2t


def c2t00b(tta, ttb, uta, utb, xp, yp):
    """
    Form the celestial to terrestrial matrix given the date, the UT1 and
    the polar motion, using the IAU 2000B precession-nutation model.

    Parameters
    ----------
    tta : double array
    ttb : double array
    uta : double array
    utb : double array
    xp : double array
    yp : double array

    Returns
    -------
    rc2t : double array

    Notes
    -----
    Wraps ERFA function ``eraC2t00b``. The ERFA documentation is::

        - - - - - - - - - -
         e r a C 2 t 0 0 b
        - - - - - - - - - -

        Form the celestial to terrestrial matrix given the date, the UT1 and
        the polar motion, using the IAU 2000B precession-nutation model.

        Given:
           tta,ttb  double         TT as a 2-part Julian Date (Note 1)
           uta,utb  double         UT1 as a 2-part Julian Date (Note 1)
           xp,yp    double         coordinates of the pole (radians, Note 2)

        Returned:
           rc2t     double[3][3]   celestial-to-terrestrial matrix (Note 3)

        Notes:

        1) The TT and UT1 dates tta+ttb and uta+utb are Julian Dates,
           apportioned in any convenient way between the arguments uta and
           utb.  For example, JD(UT1)=2450123.7 could be expressed in any of
           these ways, among others:

                   uta            utb

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution is
           acceptable.  The J2000 and MJD methods are good compromises
           between resolution and convenience.  In the case of uta,utb, the
           date & time method is best matched to the Earth rotation angle
           algorithm used:  maximum precision is delivered when the uta
           argument is for 0hrs UT1 on the day in question and the utb
           argument lies in the range 0 to 1, or vice versa.

        2) The arguments xp and yp are the coordinates (in radians) of the
           Celestial Intermediate Pole with respect to the International
           Terrestrial Reference System (see IERS Conventions 2003),
           measured along the meridians 0 and 90 deg west respectively.

        3) The matrix rc2t transforms from celestial to terrestrial
           coordinates:

              [TRS] = RPOM * R_3(ERA) * RC2I * [CRS]

                    = rc2t * [CRS]

           where [CRS] is a vector in the Geocentric Celestial Reference
           System and [TRS] is a vector in the International Terrestrial
           Reference System (see IERS Conventions 2003), RC2I is the
           celestial-to-intermediate matrix, ERA is the Earth rotation
           angle and RPOM is the polar motion matrix.

        4) The present function is faster, but slightly less accurate (about
           1 mas), than the eraC2t00a function.

        Called:
           eraC2i00b    celestial-to-intermediate matrix, IAU 2000B
           eraEra00     Earth rotation angle, IAU 2000
           eraPom00     polar motion matrix
           eraC2tcio    form CIO-based celestial-to-terrestrial matrix

        Reference:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2020 October 31

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc2t = ufunc.c2t00b(tta, ttb, uta, utb, xp, yp)
    return rc2t


def c2t06a(tta, ttb, uta, utb, xp, yp):
    """
    Form the celestial to terrestrial matrix given the date, the UT1 and
    the polar motion, using the IAU 2006/2000A precession-nutation
    model.

    Parameters
    ----------
    tta : double array
    ttb : double array
    uta : double array
    utb : double array
    xp : double array
    yp : double array

    Returns
    -------
    rc2t : double array

    Notes
    -----
    Wraps ERFA function ``eraC2t06a``. The ERFA documentation is::

        - - - - - - - - - -
         e r a C 2 t 0 6 a
        - - - - - - - - - -

        Form the celestial to terrestrial matrix given the date, the UT1 and
        the polar motion, using the IAU 2006/2000A precession-nutation
        model.

        Given:
           tta,ttb  double         TT as a 2-part Julian Date (Note 1)
           uta,utb  double         UT1 as a 2-part Julian Date (Note 1)
           xp,yp    double         coordinates of the pole (radians, Note 2)

        Returned:
           rc2t     double[3][3]   celestial-to-terrestrial matrix (Note 3)

        Notes:

        1) The TT and UT1 dates tta+ttb and uta+utb are Julian Dates,
           apportioned in any convenient way between the arguments uta and
           utb.  For example, JD(UT1)=2450123.7 could be expressed in any of
           these ways, among others:

                   uta            utb

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution is
           acceptable.  The J2000 and MJD methods are good compromises
           between resolution and convenience.  In the case of uta,utb, the
           date & time method is best matched to the Earth rotation angle
           algorithm used:  maximum precision is delivered when the uta
           argument is for 0hrs UT1 on the day in question and the utb
           argument lies in the range 0 to 1, or vice versa.

        2) The arguments xp and yp are the coordinates (in radians) of the
           Celestial Intermediate Pole with respect to the International
           Terrestrial Reference System (see IERS Conventions 2003),
           measured along the meridians 0 and 90 deg west respectively.

        3) The matrix rc2t transforms from celestial to terrestrial
           coordinates:

              [TRS] = RPOM * R_3(ERA) * RC2I * [CRS]

                    = rc2t * [CRS]

           where [CRS] is a vector in the Geocentric Celestial Reference
           System and [TRS] is a vector in the International Terrestrial
           Reference System (see IERS Conventions 2003), RC2I is the
           celestial-to-intermediate matrix, ERA is the Earth rotation
           angle and RPOM is the polar motion matrix.

        Called:
           eraC2i06a    celestial-to-intermediate matrix, IAU 2006/2000A
           eraEra00     Earth rotation angle, IAU 2000
           eraSp00      the TIO locator s', IERS 2000
           eraPom00     polar motion matrix
           eraC2tcio    form CIO-based celestial-to-terrestrial matrix

        Reference:

           McCarthy, D. D., Petit, G. (eds.), 2004, IERS Conventions (2003),
           IERS Technical Note No. 32, BKG

        This revision:  2020 October 31

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc2t = ufunc.c2t06a(tta, ttb, uta, utb, xp, yp)
    return rc2t


def c2tcio(rc2i, era, rpom):
    """
    Assemble the celestial to terrestrial matrix from CIO-based
    components (the celestial-to-intermediate matrix, the Earth Rotation
    Angle and the polar motion matrix).

    Parameters
    ----------
    rc2i : double array
    era : double array
    rpom : double array

    Returns
    -------
    rc2t : double array

    Notes
    -----
    Wraps ERFA function ``eraC2tcio``. The ERFA documentation is::

        - - - - - - - - - -
         e r a C 2 t c i o
        - - - - - - - - - -

        Assemble the celestial to terrestrial matrix from CIO-based
        components (the celestial-to-intermediate matrix, the Earth Rotation
        Angle and the polar motion matrix).

        Given:
           rc2i     double[3][3]    celestial-to-intermediate matrix
           era      double          Earth rotation angle (radians)
           rpom     double[3][3]    polar-motion matrix

        Returned:
           rc2t     double[3][3]    celestial-to-terrestrial matrix

        Notes:

        1) This function constructs the rotation matrix that transforms
           vectors in the celestial system into vectors in the terrestrial
           system.  It does so starting from precomputed components, namely
           the matrix which rotates from celestial coordinates to the
           intermediate frame, the Earth rotation angle and the polar motion
           matrix.  One use of the present function is when generating a
           series of celestial-to-terrestrial matrices where only the Earth
           Rotation Angle changes, avoiding the considerable overhead of
           recomputing the precession-nutation more often than necessary to
           achieve given accuracy objectives.

        2) The relationship between the arguments is as follows:

              [TRS] = RPOM * R_3(ERA) * rc2i * [CRS]

                    = rc2t * [CRS]

           where [CRS] is a vector in the Geocentric Celestial Reference
           System and [TRS] is a vector in the International Terrestrial
           Reference System (see IERS Conventions 2003).

        Called:
           eraCr        copy r-matrix
           eraRz        rotate around Z-axis
           eraRxr       product of two r-matrices

        Reference:

           McCarthy, D. D., Petit, G. (eds.), 2004, IERS Conventions (2003),
           IERS Technical Note No. 32, BKG

        This revision:  2013 August 24

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc2t = ufunc.c2tcio(rc2i, era, rpom)
    return rc2t


def c2teqx(rbpn, gst, rpom):
    """
    Assemble the celestial to terrestrial matrix from equinox-based
    components (the celestial-to-true matrix, the Greenwich Apparent
    Sidereal Time and the polar motion matrix).

    Parameters
    ----------
    rbpn : double array
    gst : double array
    rpom : double array

    Returns
    -------
    rc2t : double array

    Notes
    -----
    Wraps ERFA function ``eraC2teqx``. The ERFA documentation is::

        - - - - - - - - - -
         e r a C 2 t e q x
        - - - - - - - - - -

        Assemble the celestial to terrestrial matrix from equinox-based
        components (the celestial-to-true matrix, the Greenwich Apparent
        Sidereal Time and the polar motion matrix).

        Given:
           rbpn   double[3][3]  celestial-to-true matrix
           gst    double        Greenwich (apparent) Sidereal Time (radians)
           rpom   double[3][3]  polar-motion matrix

        Returned:
           rc2t   double[3][3]  celestial-to-terrestrial matrix (Note 2)

        Notes:

        1) This function constructs the rotation matrix that transforms
           vectors in the celestial system into vectors in the terrestrial
           system.  It does so starting from precomputed components, namely
           the matrix which rotates from celestial coordinates to the
           true equator and equinox of date, the Greenwich Apparent Sidereal
           Time and the polar motion matrix.  One use of the present function
           is when generating a series of celestial-to-terrestrial matrices
           where only the Sidereal Time changes, avoiding the considerable
           overhead of recomputing the precession-nutation more often than
           necessary to achieve given accuracy objectives.

        2) The relationship between the arguments is as follows:

              [TRS] = rpom * R_3(gst) * rbpn * [CRS]

                    = rc2t * [CRS]

           where [CRS] is a vector in the Geocentric Celestial Reference
           System and [TRS] is a vector in the International Terrestrial
           Reference System (see IERS Conventions 2003).

        Called:
           eraCr        copy r-matrix
           eraRz        rotate around Z-axis
           eraRxr       product of two r-matrices

        Reference:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2013 August 24

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc2t = ufunc.c2teqx(rbpn, gst, rpom)
    return rc2t


def c2tpe(tta, ttb, uta, utb, dpsi, deps, xp, yp):
    """
    Form the celestial to terrestrial matrix given the date, the UT1,
    the nutation and the polar motion.

    Parameters
    ----------
    tta : double array
    ttb : double array
    uta : double array
    utb : double array
    dpsi : double array
    deps : double array
    xp : double array
    yp : double array

    Returns
    -------
    rc2t : double array

    Notes
    -----
    Wraps ERFA function ``eraC2tpe``. The ERFA documentation is::

        - - - - - - - - -
         e r a C 2 t p e
        - - - - - - - - -

        Form the celestial to terrestrial matrix given the date, the UT1,
        the nutation and the polar motion.  IAU 2000.

        Given:
           tta,ttb    double        TT as a 2-part Julian Date (Note 1)
           uta,utb    double        UT1 as a 2-part Julian Date (Note 1)
           dpsi,deps  double        nutation (Note 2)
           xp,yp      double        coordinates of the pole (radians, Note 3)

        Returned:
           rc2t       double[3][3]  celestial-to-terrestrial matrix (Note 4)

        Notes:

        1) The TT and UT1 dates tta+ttb and uta+utb are Julian Dates,
           apportioned in any convenient way between the arguments uta and
           utb.  For example, JD(UT1)=2450123.7 could be expressed in any of
           these ways, among others:

                   uta            utb

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution is
           acceptable.  The J2000 and MJD methods are good compromises
           between resolution and convenience.  In the case of uta,utb, the
           date & time method is best matched to the Earth rotation angle
           algorithm used:  maximum precision is delivered when the uta
           argument is for 0hrs UT1 on the day in question and the utb
           argument lies in the range 0 to 1, or vice versa.

        2) The caller is responsible for providing the nutation components;
           they are in longitude and obliquity, in radians and are with
           respect to the equinox and ecliptic of date.  For high-accuracy
           applications, free core nutation should be included as well as
           any other relevant corrections to the position of the CIP.

        3) The arguments xp and yp are the coordinates (in radians) of the
           Celestial Intermediate Pole with respect to the International
           Terrestrial Reference System (see IERS Conventions 2003),
           measured along the meridians 0 and 90 deg west respectively.

        4) The matrix rc2t transforms from celestial to terrestrial
           coordinates:

              [TRS] = RPOM * R_3(GST) * RBPN * [CRS]

                    = rc2t * [CRS]

           where [CRS] is a vector in the Geocentric Celestial Reference
           System and [TRS] is a vector in the International Terrestrial
           Reference System (see IERS Conventions 2003), RBPN is the
           bias-precession-nutation matrix, GST is the Greenwich (apparent)
           Sidereal Time and RPOM is the polar motion matrix.

        5) Although its name does not include "00", This function is in fact
           specific to the IAU 2000 models.

        Called:
           eraPn00      bias/precession/nutation results, IAU 2000
           eraGmst00    Greenwich mean sidereal time, IAU 2000
           eraSp00      the TIO locator s', IERS 2000
           eraEe00      equation of the equinoxes, IAU 2000
           eraPom00     polar motion matrix
           eraC2teqx    form equinox-based celestial-to-terrestrial matrix

        Reference:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2020 October 31

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc2t = ufunc.c2tpe(tta, ttb, uta, utb, dpsi, deps, xp, yp)
    return rc2t


def c2txy(tta, ttb, uta, utb, x, y, xp, yp):
    """
    Form the celestial to terrestrial matrix given the date, the UT1,
    the CIP coordinates and the polar motion.

    Parameters
    ----------
    tta : double array
    ttb : double array
    uta : double array
    utb : double array
    x : double array
    y : double array
    xp : double array
    yp : double array

    Returns
    -------
    rc2t : double array

    Notes
    -----
    Wraps ERFA function ``eraC2txy``. The ERFA documentation is::

        - - - - - - - - -
         e r a C 2 t x y
        - - - - - - - - -

        Form the celestial to terrestrial matrix given the date, the UT1,
        the CIP coordinates and the polar motion.  IAU 2000.

        Given:
           tta,ttb  double         TT as a 2-part Julian Date (Note 1)
           uta,utb  double         UT1 as a 2-part Julian Date (Note 1)
           x,y      double         Celestial Intermediate Pole (Note 2)
           xp,yp    double         coordinates of the pole (radians, Note 3)

        Returned:
           rc2t     double[3][3]   celestial-to-terrestrial matrix (Note 4)

        Notes:

        1) The TT and UT1 dates tta+ttb and uta+utb are Julian Dates,
           apportioned in any convenient way between the arguments uta and
           utb.  For example, JD(UT1)=2450123.7 could be expressed in any o
           these ways, among others:

                   uta            utb

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution is
           acceptable.  The J2000 and MJD methods are good compromises
           between resolution and convenience.  In the case of uta,utb, the
           date & time method is best matched to the Earth rotation angle
           algorithm used:  maximum precision is delivered when the uta
           argument is for 0hrs UT1 on the day in question and the utb
           argument lies in the range 0 to 1, or vice versa.

        2) The Celestial Intermediate Pole coordinates are the x,y
           components of the unit vector in the Geocentric Celestial
           Reference System.

        3) The arguments xp and yp are the coordinates (in radians) of the
           Celestial Intermediate Pole with respect to the International
           Terrestrial Reference System (see IERS Conventions 2003),
           measured along the meridians 0 and 90 deg west respectively.

        4) The matrix rc2t transforms from celestial to terrestrial
           coordinates:

              [TRS] = RPOM * R_3(ERA) * RC2I * [CRS]

                    = rc2t * [CRS]

           where [CRS] is a vector in the Geocentric Celestial Reference
           System and [TRS] is a vector in the International Terrestrial
           Reference System (see IERS Conventions 2003), ERA is the Earth
           Rotation Angle and RPOM is the polar motion matrix.

        5) Although its name does not include "00", This function is in fact
           specific to the IAU 2000 models.

        Called:
           eraC2ixy     celestial-to-intermediate matrix, given X,Y
           eraEra00     Earth rotation angle, IAU 2000
           eraSp00      the TIO locator s', IERS 2000
           eraPom00     polar motion matrix
           eraC2tcio    form CIO-based celestial-to-terrestrial matrix

       Reference:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2020 October 31

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rc2t = ufunc.c2txy(tta, ttb, uta, utb, x, y, xp, yp)
    return rc2t


def eo06a(date1, date2):
    """
    Equation of the origins, IAU 2006 precession and IAU 2000A nutation.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraEo06a``. The ERFA documentation is::

        - - - - - - - - -
         e r a E o 0 6 a
        - - - - - - - - -

        Equation of the origins, IAU 2006 precession and IAU 2000A nutation.

        Given:
           date1,date2  double    TT as a 2-part Julian Date (Note 1)

        Returned (function value):
                        double    the equation of the origins in radians

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The equation of the origins is the distance between the true
           equinox and the celestial intermediate origin and, equivalently,
           the difference between Earth rotation angle and Greenwich
           apparent sidereal time (ERA-GST).  It comprises the precession
           (since J2000.0) in right ascension plus the equation of the
           equinoxes (including the small correction terms).

        Called:
           eraPnm06a    classical NPB matrix, IAU 2006/2000A
           eraBpn2xy    extract CIP X,Y coordinates from NPB matrix
           eraS06       the CIO locator s, given X,Y, IAU 2006
           eraEors      equation of the origins, given NPB matrix and s

        References:

           Capitaine, N. & Wallace, P.T., 2006, Astron.Astrophys. 450, 855

           Wallace, P.T. & Capitaine, N., 2006, Astron.Astrophys. 459, 981

        This revision:  2020 November 14

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.eo06a(date1, date2)
    return c_retval


def eors(rnpb, s):
    """
    Equation of the origins, given the classical NPB matrix and the
    quantity s.

    Parameters
    ----------
    rnpb : double array
    s : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraEors``. The ERFA documentation is::

        - - - - - - - -
         e r a E o r s
        - - - - - - - -

        Equation of the origins, given the classical NPB matrix and the
        quantity s.

        Given:
           rnpb  double[3][3]  classical nutation x precession x bias matrix
           s     double        the quantity s (the CIO locator) in radians

        Returned (function value):
                 double        the equation of the origins in radians

        Notes:

        1)  The equation of the origins is the distance between the true
            equinox and the celestial intermediate origin and, equivalently,
            the difference between Earth rotation angle and Greenwich
            apparent sidereal time (ERA-GST).  It comprises the precession
            (since J2000.0) in right ascension plus the equation of the
            equinoxes (including the small correction terms).

        2)  The algorithm is from Wallace & Capitaine (2006).

       References:

           Capitaine, N. & Wallace, P.T., 2006, Astron.Astrophys. 450, 855

           Wallace, P. & Capitaine, N., 2006, Astron.Astrophys. 459, 981

        This revision:  2020 November 14

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.eors(rnpb, s)
    return c_retval


def fw2m(gamb, phib, psi, eps):
    """
    Form rotation matrix given the Fukushima-Williams angles.

    Parameters
    ----------
    gamb : double array
    phib : double array
    psi : double array
    eps : double array

    Returns
    -------
    r : double array

    Notes
    -----
    Wraps ERFA function ``eraFw2m``. The ERFA documentation is::

        - - - - - - - -
         e r a F w 2 m
        - - - - - - - -

        Form rotation matrix given the Fukushima-Williams angles.

        Given:
           gamb     double         F-W angle gamma_bar (radians)
           phib     double         F-W angle phi_bar (radians)
           psi      double         F-W angle psi (radians)
           eps      double         F-W angle epsilon (radians)

        Returned:
           r        double[3][3]   rotation matrix

        Notes:

        1) Naming the following points:

                 e = J2000.0 ecliptic pole,
                 p = GCRS pole,
                 E = ecliptic pole of date,
           and   P = CIP,

           the four Fukushima-Williams angles are as follows:

              gamb = gamma = epE
              phib = phi = pE
              psi = psi = pEP
              eps = epsilon = EP

        2) The matrix representing the combined effects of frame bias,
           precession and nutation is:

              NxPxB = R_1(-eps).R_3(-psi).R_1(phib).R_3(gamb)

        3) The present function can construct three different matrices,
           depending on which angles are supplied as the arguments gamb,
           phib, psi and eps:

           o  To obtain the nutation x precession x frame bias matrix,
              first generate the four precession angles known conventionally
              as gamma_bar, phi_bar, psi_bar and epsilon_A, then generate
              the nutation components Dpsi and Depsilon and add them to
              psi_bar and epsilon_A, and finally call the present function
              using those four angles as arguments.

           o  To obtain the precession x frame bias matrix, generate the
              four precession angles and call the present function.

           o  To obtain the frame bias matrix, generate the four precession
              angles for date J2000.0 and call the present function.

           The nutation-only and precession-only matrices can if necessary
           be obtained by combining these three appropriately.

        Called:
           eraIr        initialize r-matrix to identity
           eraRz        rotate around Z-axis
           eraRx        rotate around X-axis

        References:

           Capitaine, N. & Wallace, P.T., 2006, Astron.Astrophys. 450, 855

           Hilton, J. et al., 2006, Celest.Mech.Dyn.Astron. 94, 351

        This revision:  2020 November 17

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r = ufunc.fw2m(gamb, phib, psi, eps)
    return r


def fw2xy(gamb, phib, psi, eps):
    """
    CIP X,Y given Fukushima-Williams bias-precession-nutation angles.

    Parameters
    ----------
    gamb : double array
    phib : double array
    psi : double array
    eps : double array

    Returns
    -------
    x : double array
    y : double array

    Notes
    -----
    Wraps ERFA function ``eraFw2xy``. The ERFA documentation is::

        - - - - - - - - -
         e r a F w 2 x y
        - - - - - - - - -

        CIP X,Y given Fukushima-Williams bias-precession-nutation angles.

        Given:
           gamb     double    F-W angle gamma_bar (radians)
           phib     double    F-W angle phi_bar (radians)
           psi      double    F-W angle psi (radians)
           eps      double    F-W angle epsilon (radians)

        Returned:
           x,y      double    CIP unit vector X,Y

        Notes:

        1) Naming the following points:

                 e = J2000.0 ecliptic pole,
                 p = GCRS pole
                 E = ecliptic pole of date,
           and   P = CIP,

           the four Fukushima-Williams angles are as follows:

              gamb = gamma = epE
              phib = phi = pE
              psi = psi = pEP
              eps = epsilon = EP

        2) The matrix representing the combined effects of frame bias,
           precession and nutation is:

              NxPxB = R_1(-epsA).R_3(-psi).R_1(phib).R_3(gamb)

           The returned values x,y are elements [2][0] and [2][1] of the
           matrix.  Near J2000.0, they are essentially angles in radians.

        Called:
           eraFw2m      F-W angles to r-matrix
           eraBpn2xy    extract CIP X,Y coordinates from NPB matrix

        Reference:

           Hilton, J. et al., 2006, Celest.Mech.Dyn.Astron. 94, 351

        This revision:  2013 September 2

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    x, y = ufunc.fw2xy(gamb, phib, psi, eps)
    return x, y


def ltp(epj):
    """
    Long-term precession matrix.

    Parameters
    ----------
    epj : double array

    Returns
    -------
    rp : double array

    Notes
    -----
    Wraps ERFA function ``eraLtp``. The ERFA documentation is::

        - - - - - - -
         e r a L t p
        - - - - - - -

        Long-term precession matrix.

        Given:
           epj     double         Julian epoch (TT)

        Returned:
           rp      double[3][3]   precession matrix, J2000.0 to date

        Notes:

        1) The matrix is in the sense

              P_date = rp x P_J2000,

           where P_J2000 is a vector with respect to the J2000.0 mean
           equator and equinox and P_date is the same vector with respect to
           the equator and equinox of epoch epj.

        2) The Vondrak et al. (2011, 2012) 400 millennia precession model
           agrees with the IAU 2006 precession at J2000.0 and stays within
           100 microarcseconds during the 20th and 21st centuries.  It is
           accurate to a few arcseconds throughout the historical period,
           worsening to a few tenths of a degree at the end of the
           +/- 200,000 year time span.

        Called:
           eraLtpequ    equator pole, long term
           eraLtpecl    ecliptic pole, long term
           eraPxp       vector product
           eraPn        normalize vector

        References:

          Vondrak, J., Capitaine, N. and Wallace, P., 2011, New precession
          expressions, valid for long time intervals, Astron.Astrophys. 534,
          A22

          Vondrak, J., Capitaine, N. and Wallace, P., 2012, New precession
          expressions, valid for long time intervals (Corrigendum),
          Astron.Astrophys. 541, C1

        This revision:  2015 December 6

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rp = ufunc.ltp(epj)
    return rp


def ltpb(epj):
    """
    Long-term precession matrix, including ICRS frame bias.

    Parameters
    ----------
    epj : double array

    Returns
    -------
    rpb : double array

    Notes
    -----
    Wraps ERFA function ``eraLtpb``. The ERFA documentation is::

        - - - - - - - -
         e r a L t p b
        - - - - - - - -

        Long-term precession matrix, including ICRS frame bias.

        Given:
           epj     double         Julian epoch (TT)

        Returned:
           rpb     double[3][3]   precession-bias matrix, J2000.0 to date

        Notes:

        1) The matrix is in the sense

              P_date = rpb x P_ICRS,

           where P_ICRS is a vector in the Geocentric Celestial Reference
           System, and P_date is the vector with respect to the Celestial
           Intermediate Reference System at that date but with nutation
           neglected.

        2) A first order frame bias formulation is used, of sub-
           microarcsecond accuracy compared with a full 3D rotation.

        3) The Vondrak et al. (2011, 2012) 400 millennia precession model
           agrees with the IAU 2006 precession at J2000.0 and stays within
           100 microarcseconds during the 20th and 21st centuries.  It is
           accurate to a few arcseconds throughout the historical period,
           worsening to a few tenths of a degree at the end of the
           +/- 200,000 year time span.

        References:

          Vondrak, J., Capitaine, N. and Wallace, P., 2011, New precession
          expressions, valid for long time intervals, Astron.Astrophys. 534,
          A22

          Vondrak, J., Capitaine, N. and Wallace, P., 2012, New precession
          expressions, valid for long time intervals (Corrigendum),
          Astron.Astrophys. 541, C1

        This revision:  2015 December 6

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rpb = ufunc.ltpb(epj)
    return rpb


def ltpecl(epj):
    """
    Long-term precession of the ecliptic.

    Parameters
    ----------
    epj : double array

    Returns
    -------
    vec : double array

    Notes
    -----
    Wraps ERFA function ``eraLtpecl``. The ERFA documentation is::

        - - - - - - - - - -
         e r a L t p e c l
        - - - - - - - - - -

        Long-term precession of the ecliptic.

        Given:
           epj     double         Julian epoch (TT)

        Returned:
           vec     double[3]      ecliptic pole unit vector

        Notes:

        1) The returned vector is with respect to the J2000.0 mean equator
           and equinox.

        2) The Vondrak et al. (2011, 2012) 400 millennia precession model
           agrees with the IAU 2006 precession at J2000.0 and stays within
           100 microarcseconds during the 20th and 21st centuries.  It is
           accurate to a few arcseconds throughout the historical period,
           worsening to a few tenths of a degree at the end of the
           +/- 200,000 year time span.

        References:

          Vondrak, J., Capitaine, N. and Wallace, P., 2011, New precession
          expressions, valid for long time intervals, Astron.Astrophys. 534,
          A22

          Vondrak, J., Capitaine, N. and Wallace, P., 2012, New precession
          expressions, valid for long time intervals (Corrigendum),
          Astron.Astrophys. 541, C1

        This revision:  2016 February 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    vec = ufunc.ltpecl(epj)
    return vec


def ltpequ(epj):
    """
    Long-term precession of the equator.

    Parameters
    ----------
    epj : double array

    Returns
    -------
    veq : double array

    Notes
    -----
    Wraps ERFA function ``eraLtpequ``. The ERFA documentation is::

        - - - - - - - - - -
         e r a L t p e q u
        - - - - - - - - - -

        Long-term precession of the equator.

        Given:
           epj     double         Julian epoch (TT)

        Returned:
           veq     double[3]      equator pole unit vector

        Notes:

        1) The returned vector is with respect to the J2000.0 mean equator
           and equinox.

        2) The Vondrak et al. (2011, 2012) 400 millennia precession model
           agrees with the IAU 2006 precession at J2000.0 and stays within
           100 microarcseconds during the 20th and 21st centuries.  It is
           accurate to a few arcseconds throughout the historical period,
           worsening to a few tenths of a degree at the end of the
           +/- 200,000 year time span.

        References:

          Vondrak, J., Capitaine, N. and Wallace, P., 2011, New precession
          expressions, valid for long time intervals, Astron.Astrophys. 534,
          A22

          Vondrak, J., Capitaine, N. and Wallace, P., 2012, New precession
          expressions, valid for long time intervals (Corrigendum),
          Astron.Astrophys. 541, C1

        This revision:  2016 February 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    veq = ufunc.ltpequ(epj)
    return veq


def num00a(date1, date2):
    """
    Form the matrix of nutation for a given date, IAU 2000A model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rmatn : double array

    Notes
    -----
    Wraps ERFA function ``eraNum00a``. The ERFA documentation is::

        - - - - - - - - - -
         e r a N u m 0 0 a
        - - - - - - - - - -

        Form the matrix of nutation for a given date, IAU 2000A model.

        Given:
           date1,date2  double          TT as a 2-part Julian Date (Note 1)

        Returned:
           rmatn        double[3][3]    nutation matrix

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix operates in the sense V(true) = rmatn * V(mean), where
           the p-vector V(true) is with respect to the true equatorial triad
           of date and the p-vector V(mean) is with respect to the mean
           equatorial triad of date.

        3) A faster, but slightly less accurate, result (about 1 mas) can be
           obtained by using instead the eraNum00b function.

        Called:
           eraPn00a     bias/precession/nutation, IAU 2000A

        Reference:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992),
           Section 3.222-3 (p114).

        This revision:  2020 November 27

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rmatn = ufunc.num00a(date1, date2)
    return rmatn


def num00b(date1, date2):
    """
    Form the matrix of nutation for a given date, IAU 2000B model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rmatn : double array

    Notes
    -----
    Wraps ERFA function ``eraNum00b``. The ERFA documentation is::

        - - - - - - - - - -
         e r a N u m 0 0 b
        - - - - - - - - - -

        Form the matrix of nutation for a given date, IAU 2000B model.

        Given:
           date1,date2  double         TT as a 2-part Julian Date (Note 1)

        Returned:
           rmatn        double[3][3]   nutation matrix

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix operates in the sense V(true) = rmatn * V(mean), where
           the p-vector V(true) is with respect to the true equatorial triad
           of date and the p-vector V(mean) is with respect to the mean
           equatorial triad of date.

        3) The present function is faster, but slightly less accurate (about
           1 mas), than the eraNum00a function.

        Called:
           eraPn00b     bias/precession/nutation, IAU 2000B

        Reference:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992),
           Section 3.222-3 (p114).

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rmatn = ufunc.num00b(date1, date2)
    return rmatn


def num06a(date1, date2):
    """
    Form the matrix of nutation for a given date, IAU 2006/2000A model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rmatn : double array

    Notes
    -----
    Wraps ERFA function ``eraNum06a``. The ERFA documentation is::

        - - - - - - - - - -
         e r a N u m 0 6 a
        - - - - - - - - - -

        Form the matrix of nutation for a given date, IAU 2006/2000A model.

        Given:
           date1,date2   double          TT as a 2-part Julian Date (Note 1)

        Returned:
           rmatn         double[3][3]    nutation matrix

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix operates in the sense V(true) = rmatn * V(mean), where
           the p-vector V(true) is with respect to the true equatorial triad
           of date and the p-vector V(mean) is with respect to the mean
           equatorial triad of date.

        Called:
           eraObl06     mean obliquity, IAU 2006
           eraNut06a    nutation, IAU 2006/2000A
           eraNumat     form nutation matrix

        Reference:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992),
           Section 3.222-3 (p114).

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rmatn = ufunc.num06a(date1, date2)
    return rmatn


def numat(epsa, dpsi, deps):
    """
    Form the matrix of nutation.

    Parameters
    ----------
    epsa : double array
    dpsi : double array
    deps : double array

    Returns
    -------
    rmatn : double array

    Notes
    -----
    Wraps ERFA function ``eraNumat``. The ERFA documentation is::

        - - - - - - - - -
         e r a N u m a t
        - - - - - - - - -

        Form the matrix of nutation.

        Given:
           epsa        double         mean obliquity of date (Note 1)
           dpsi,deps   double         nutation (Note 2)

        Returned:
           rmatn       double[3][3]   nutation matrix (Note 3)

        Notes:


        1) The supplied mean obliquity epsa, must be consistent with the
           precession-nutation models from which dpsi and deps were obtained.

        2) The caller is responsible for providing the nutation components;
           they are in longitude and obliquity, in radians and are with
           respect to the equinox and ecliptic of date.

        3) The matrix operates in the sense V(true) = rmatn * V(mean),
           where the p-vector V(true) is with respect to the true
           equatorial triad of date and the p-vector V(mean) is with
           respect to the mean equatorial triad of date.

        Called:
           eraIr        initialize r-matrix to identity
           eraRx        rotate around X-axis
           eraRz        rotate around Z-axis

        Reference:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992),
           Section 3.222-3 (p114).

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rmatn = ufunc.numat(epsa, dpsi, deps)
    return rmatn


def nut00a(date1, date2):
    """
    Nutation, IAU 2000A model (MHB2000 luni-solar and planetary nutation
    with free core nutation omitted).

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    dpsi : double array
    deps : double array

    Notes
    -----
    Wraps ERFA function ``eraNut00a``. The ERFA documentation is::

        - - - - - - - - - -
         e r a N u t 0 0 a
        - - - - - - - - - -

        Nutation, IAU 2000A model (MHB2000 luni-solar and planetary nutation
        with free core nutation omitted).

        Given:
           date1,date2   double   TT as a 2-part Julian Date (Note 1)

        Returned:
           dpsi,deps     double   nutation, luni-solar + planetary (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The nutation components in longitude and obliquity are in radians
           and with respect to the equinox and ecliptic of date.  The
           obliquity at J2000.0 is assumed to be the Lieske et al. (1977)
           value of 84381.448 arcsec.

           Both the luni-solar and planetary nutations are included.  The
           latter are due to direct planetary nutations and the
           perturbations of the lunar and terrestrial orbits.

        3) The function computes the MHB2000 nutation series with the
           associated corrections for planetary nutations.  It is an
           implementation of the nutation part of the IAU 2000A precession-
           nutation model, formally adopted by the IAU General Assembly in
           2000, namely MHB2000 (Mathews et al. 2002), but with the free
           core nutation (FCN - see Note 4) omitted.

        4) The full MHB2000 model also contains contributions to the
           nutations in longitude and obliquity due to the free-excitation
           of the free-core-nutation during the period 1979-2000.  These FCN
           terms, which are time-dependent and unpredictable, are NOT
           included in the present function and, if required, must be
           independently computed.  With the FCN corrections included, the
           present function delivers a pole which is at current epochs
           accurate to a few hundred microarcseconds.  The omission of FCN
           introduces further errors of about that size.

        5) The present function provides classical nutation.  The MHB2000
           algorithm, from which it is adapted, deals also with (i) the
           offsets between the GCRS and mean poles and (ii) the adjustments
           in longitude and obliquity due to the changed precession rates.
           These additional functions, namely frame bias and precession
           adjustments, are supported by the ERFA functions eraBi00  and
           eraPr00.

        6) The MHB2000 algorithm also provides "total" nutations, comprising
           the arithmetic sum of the frame bias, precession adjustments,
           luni-solar nutation and planetary nutation.  These total
           nutations can be used in combination with an existing IAU 1976
           precession implementation, such as eraPmat76,  to deliver GCRS-
           to-true predictions of sub-mas accuracy at current dates.
           However, there are three shortcomings in the MHB2000 model that
           must be taken into account if more accurate or definitive results
           are required (see Wallace 2002):

             (i) The MHB2000 total nutations are simply arithmetic sums,
                 yet in reality the various components are successive Euler
                 rotations.  This slight lack of rigor leads to cross terms
                 that exceed 1 mas after a century.  The rigorous procedure
                 is to form the GCRS-to-true rotation matrix by applying the
                 bias, precession and nutation in that order.

            (ii) Although the precession adjustments are stated to be with
                 respect to Lieske et al. (1977), the MHB2000 model does
                 not specify which set of Euler angles are to be used and
                 how the adjustments are to be applied.  The most literal
                 and straightforward procedure is to adopt the 4-rotation
                 epsilon_0, psi_A, omega_A, xi_A option, and to add DPSIPR
                 to psi_A and DEPSPR to both omega_A and eps_A.

           (iii) The MHB2000 model predates the determination by Chapront
                 et al. (2002) of a 14.6 mas displacement between the
                 J2000.0 mean equinox and the origin of the ICRS frame.  It
                 should, however, be noted that neglecting this displacement
                 when calculating star coordinates does not lead to a
                 14.6 mas change in right ascension, only a small second-
                 order distortion in the pattern of the precession-nutation
                 effect.

           For these reasons, the ERFA functions do not generate the "total
           nutations" directly, though they can of course easily be
           generated by calling eraBi00, eraPr00 and the present function
           and adding the results.

        7) The MHB2000 model contains 41 instances where the same frequency
           appears multiple times, of which 38 are duplicates and three are
           triplicates.  To keep the present code close to the original MHB
           algorithm, this small inefficiency has not been corrected.

        Called:
           eraFal03     mean anomaly of the Moon
           eraFaf03     mean argument of the latitude of the Moon
           eraFaom03    mean longitude of the Moon's ascending node
           eraFame03    mean longitude of Mercury
           eraFave03    mean longitude of Venus
           eraFae03     mean longitude of Earth
           eraFama03    mean longitude of Mars
           eraFaju03    mean longitude of Jupiter
           eraFasa03    mean longitude of Saturn
           eraFaur03    mean longitude of Uranus
           eraFapa03    general accumulated precession in longitude

        References:

           Chapront, J., Chapront-Touze, M. & Francou, G. 2002,
           Astron.Astrophys. 387, 700

           Lieske, J.H., Lederle, T., Fricke, W. & Morando, B. 1977,
           Astron.Astrophys. 58, 1-16

           Mathews, P.M., Herring, T.A., Buffet, B.A. 2002, J.Geophys.Res.
           107, B4.  The MHB_2000 code itself was obtained on 9th September
           2002 from ftp//maia.usno.navy.mil/conv2000/chapter5/IAU2000A.

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J. 1994, Astron.Astrophys. 282, 663-683

           Souchay, J., Loysel, B., Kinoshita, H., Folgueira, M. 1999,
           Astron.Astrophys.Supp.Ser. 135, 111

           Wallace, P.T., "Software for Implementing the IAU 2000
           Resolutions", in IERS Workshop 5.1 (2002)

        This revision:  2019 June 23

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dpsi, deps = ufunc.nut00a(date1, date2)
    return dpsi, deps


def nut00b(date1, date2):
    """
    Nutation, IAU 2000B model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    dpsi : double array
    deps : double array

    Notes
    -----
    Wraps ERFA function ``eraNut00b``. The ERFA documentation is::

        - - - - - - - - - -
         e r a N u t 0 0 b
        - - - - - - - - - -

        Nutation, IAU 2000B model.

        Given:
           date1,date2   double    TT as a 2-part Julian Date (Note 1)

        Returned:
           dpsi,deps     double    nutation, luni-solar + planetary (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The nutation components in longitude and obliquity are in radians
           and with respect to the equinox and ecliptic of date.  The
           obliquity at J2000.0 is assumed to be the Lieske et al. (1977)
           value of 84381.448 arcsec.  (The errors that result from using
           this function with the IAU 2006 value of 84381.406 arcsec can be
           neglected.)

           The nutation model consists only of luni-solar terms, but
           includes also a fixed offset which compensates for certain long-
           period planetary terms (Note 7).

        3) This function is an implementation of the IAU 2000B abridged
           nutation model formally adopted by the IAU General Assembly in
           2000.  The function computes the MHB_2000_SHORT luni-solar
           nutation series (Luzum 2001), but without the associated
           corrections for the precession rate adjustments and the offset
           between the GCRS and J2000.0 mean poles.

        4) The full IAU 2000A (MHB2000) nutation model contains nearly 1400
           terms.  The IAU 2000B model (McCarthy & Luzum 2003) contains only
           77 terms, plus additional simplifications, yet still delivers
           results of 1 mas accuracy at present epochs.  This combination of
           accuracy and size makes the IAU 2000B abridged nutation model
           suitable for most practical applications.

           The function delivers a pole accurate to 1 mas from 1900 to 2100
           (usually better than 1 mas, very occasionally just outside
           1 mas).  The full IAU 2000A model, which is implemented in the
           function eraNut00a (q.v.), delivers considerably greater accuracy
           at current dates;  however, to realize this improved accuracy,
           corrections for the essentially unpredictable free-core-nutation
           (FCN) must also be included.

        5) The present function provides classical nutation.  The
           MHB_2000_SHORT algorithm, from which it is adapted, deals also
           with (i) the offsets between the GCRS and mean poles and (ii) the
           adjustments in longitude and obliquity due to the changed
           precession rates.  These additional functions, namely frame bias
           and precession adjustments, are supported by the ERFA functions
           eraBi00  and eraPr00.

        6) The MHB_2000_SHORT algorithm also provides "total" nutations,
           comprising the arithmetic sum of the frame bias, precession
           adjustments, and nutation (luni-solar + planetary).  These total
           nutations can be used in combination with an existing IAU 1976
           precession implementation, such as eraPmat76,  to deliver GCRS-
           to-true predictions of mas accuracy at current epochs.  However,
           for symmetry with the eraNut00a  function (q.v. for the reasons),
           the ERFA functions do not generate the "total nutations"
           directly.  Should they be required, they could of course easily
           be generated by calling eraBi00, eraPr00 and the present function
           and adding the results.

        7) The IAU 2000B model includes "planetary bias" terms that are
           fixed in size but compensate for long-period nutations.  The
           amplitudes quoted in McCarthy & Luzum (2003), namely
           Dpsi = -1.5835 mas and Depsilon = +1.6339 mas, are optimized for
           the "total nutations" method described in Note 6.  The Luzum
           (2001) values used in this ERFA implementation, namely -0.135 mas
           and +0.388 mas, are optimized for the "rigorous" method, where
           frame bias, precession and nutation are applied separately and in
           that order.  During the interval 1995-2050, the ERFA
           implementation delivers a maximum error of 1.001 mas (not
           including FCN).

        References:

           Lieske, J.H., Lederle, T., Fricke, W., Morando, B., "Expressions
           for the precession quantities based upon the IAU /1976/ system of
           astronomical constants", Astron.Astrophys. 58, 1-2, 1-16. (1977)

           Luzum, B., private communication, 2001 (Fortran code
           MHB_2000_SHORT)

           McCarthy, D.D. & Luzum, B.J., "An abridged model of the
           precession-nutation of the celestial pole", Cel.Mech.Dyn.Astron.
           85, 37-49 (2003)

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J., Astron.Astrophys. 282, 663-683 (1994)

        This revision:  2019 June 23

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dpsi, deps = ufunc.nut00b(date1, date2)
    return dpsi, deps


def nut06a(date1, date2):
    """
    IAU 2000A nutation with adjustments to match the IAU 2006
    precession.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    dpsi : double array
    deps : double array

    Notes
    -----
    Wraps ERFA function ``eraNut06a``. The ERFA documentation is::

        - - - - - - - - - -
         e r a N u t 0 6 a
        - - - - - - - - - -

        IAU 2000A nutation with adjustments to match the IAU 2006
        precession.

        Given:
           date1,date2   double   TT as a 2-part Julian Date (Note 1)

        Returned:
           dpsi,deps     double   nutation, luni-solar + planetary (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The nutation components in longitude and obliquity are in radians
           and with respect to the mean equinox and ecliptic of date,
           IAU 2006 precession model (Hilton et al. 2006, Capitaine et al.
           2005).

        3) The function first computes the IAU 2000A nutation, then applies
           adjustments for (i) the consequences of the change in obliquity
           from the IAU 1980 ecliptic to the IAU 2006 ecliptic and (ii) the
           secular variation in the Earth's dynamical form factor J2.

        4) The present function provides classical nutation, complementing
           the IAU 2000 frame bias and IAU 2006 precession.  It delivers a
           pole which is at current epochs accurate to a few tens of
           microarcseconds, apart from the free core nutation.

        Called:
           eraNut00a    nutation, IAU 2000A

        References:

           Chapront, J., Chapront-Touze, M. & Francou, G. 2002,
           Astron.Astrophys. 387, 700

           Lieske, J.H., Lederle, T., Fricke, W. & Morando, B. 1977,
           Astron.Astrophys. 58, 1-16

           Mathews, P.M., Herring, T.A., Buffet, B.A. 2002, J.Geophys.Res.
           107, B4.  The MHB_2000 code itself was obtained on 9th September
           2002 from ftp//maia.usno.navy.mil/conv2000/chapter5/IAU2000A.

           Simon, J.-L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G., Laskar, J. 1994, Astron.Astrophys. 282, 663-683

           Souchay, J., Loysel, B., Kinoshita, H., Folgueira, M. 1999,
           Astron.Astrophys.Supp.Ser. 135, 111

           Wallace, P.T., "Software for Implementing the IAU 2000
           Resolutions", in IERS Workshop 5.1 (2002)

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dpsi, deps = ufunc.nut06a(date1, date2)
    return dpsi, deps


def nut80(date1, date2):
    """
    Nutation, IAU 1980 model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    dpsi : double array
    deps : double array

    Notes
    -----
    Wraps ERFA function ``eraNut80``. The ERFA documentation is::

        - - - - - - - - -
         e r a N u t 8 0
        - - - - - - - - -

        Nutation, IAU 1980 model.

        Given:
           date1,date2   double    TT as a 2-part Julian Date (Note 1)

        Returned:
           dpsi          double    nutation in longitude (radians)
           deps          double    nutation in obliquity (radians)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The nutation components are with respect to the ecliptic of
           date.

        Called:
           eraAnpm      normalize angle into range +/- pi

        Reference:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992),
           Section 3.222 (p111).

        This revision:  2019 June 23

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dpsi, deps = ufunc.nut80(date1, date2)
    return dpsi, deps


def nutm80(date1, date2):
    """
    Form the matrix of nutation for a given date, IAU 1980 model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rmatn : double array

    Notes
    -----
    Wraps ERFA function ``eraNutm80``. The ERFA documentation is::

        - - - - - - - - - -
         e r a N u t m 8 0
        - - - - - - - - - -

        Form the matrix of nutation for a given date, IAU 1980 model.

        Given:
           date1,date2    double          TDB date (Note 1)

        Returned:
           rmatn          double[3][3]    nutation matrix

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix operates in the sense V(true) = rmatn * V(mean),
           where the p-vector V(true) is with respect to the true
           equatorial triad of date and the p-vector V(mean) is with
           respect to the mean equatorial triad of date.

        Called:
           eraNut80     nutation, IAU 1980
           eraObl80     mean obliquity, IAU 1980
           eraNumat     form nutation matrix

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rmatn = ufunc.nutm80(date1, date2)
    return rmatn


def obl06(date1, date2):
    """
    Mean obliquity of the ecliptic, IAU 2006 precession model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraObl06``. The ERFA documentation is::

        - - - - - - - - -
         e r a O b l 0 6
        - - - - - - - - -

        Mean obliquity of the ecliptic, IAU 2006 precession model.

        Given:
           date1,date2  double   TT as a 2-part Julian Date (Note 1)

        Returned (function value):
                        double   obliquity of the ecliptic (radians, Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The result is the angle between the ecliptic and mean equator of
           date date1+date2.

        Reference:

           Hilton, J. et al., 2006, Celest.Mech.Dyn.Astron. 94, 351

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.obl06(date1, date2)
    return c_retval


def obl80(date1, date2):
    """
    Mean obliquity of the ecliptic, IAU 1980 model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraObl80``. The ERFA documentation is::

        - - - - - - - - -
         e r a O b l 8 0
        - - - - - - - - -

        Mean obliquity of the ecliptic, IAU 1980 model.

        Given:
           date1,date2   double    TT as a 2-part Julian Date (Note 1)

        Returned (function value):
                         double    obliquity of the ecliptic (radians, Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The result is the angle between the ecliptic and mean equator of
           date date1+date2.

        Reference:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992),
           Expression 3.222-1 (p114).

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.obl80(date1, date2)
    return c_retval


def p06e(date1, date2):
    """
    Precession angles, IAU 2006, equinox based.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    eps0 : double array
    psia : double array
    oma : double array
    bpa : double array
    bqa : double array
    pia : double array
    bpia : double array
    epsa : double array
    chia : double array
    za : double array
    zetaa : double array
    thetaa : double array
    pa : double array
    gam : double array
    phi : double array
    psi : double array

    Notes
    -----
    Wraps ERFA function ``eraP06e``. The ERFA documentation is::

        - - - - - - - -
         e r a P 0 6 e
        - - - - - - - -

        Precession angles, IAU 2006, equinox based.

        Given:
           date1,date2   double   TT as a 2-part Julian Date (Note 1)

        Returned (see Note 2):
           eps0          double   epsilon_0
           psia          double   psi_A
           oma           double   omega_A
           bpa           double   P_A
           bqa           double   Q_A
           pia           double   pi_A
           bpia          double   Pi_A
           epsa          double   obliquity epsilon_A
           chia          double   chi_A
           za            double   z_A
           zetaa         double   zeta_A
           thetaa        double   theta_A
           pa            double   p_A
           gam           double   F-W angle gamma_J2000
           phi           double   F-W angle phi_J2000
           psi           double   F-W angle psi_J2000

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) This function returns the set of equinox based angles for the
           Capitaine et al. "P03" precession theory, adopted by the IAU in
           2006.  The angles are set out in Table 1 of Hilton et al. (2006):

           eps0   epsilon_0   obliquity at J2000.0
           psia   psi_A       luni-solar precession
           oma    omega_A     inclination of equator wrt J2000.0 ecliptic
           bpa    P_A         ecliptic pole x, J2000.0 ecliptic triad
           bqa    Q_A         ecliptic pole -y, J2000.0 ecliptic triad
           pia    pi_A        angle between moving and J2000.0 ecliptics
           bpia   Pi_A        longitude of ascending node of the ecliptic
           epsa   epsilon_A   obliquity of the ecliptic
           chia   chi_A       planetary precession
           za     z_A         equatorial precession: -3rd 323 Euler angle
           zetaa  zeta_A      equatorial precession: -1st 323 Euler angle
           thetaa theta_A     equatorial precession: 2nd 323 Euler angle
           pa     p_A         general precession (n.b. see below)
           gam    gamma_J2000 J2000.0 RA difference of ecliptic poles
           phi    phi_J2000   J2000.0 codeclination of ecliptic pole
           psi    psi_J2000   longitude difference of equator poles, J2000.0

           The returned values are all radians.

           Note that the t^5 coefficient in the series for p_A from
           Capitaine et al. (2003) is incorrectly signed in Hilton et al.
           (2006).

        3) Hilton et al. (2006) Table 1 also contains angles that depend on
           models distinct from the P03 precession theory itself, namely the
           IAU 2000A frame bias and nutation.  The quoted polynomials are
           used in other ERFA functions:

           . eraXy06  contains the polynomial parts of the X and Y series.

           . eraS06  contains the polynomial part of the s+XY/2 series.

           . eraPfw06  implements the series for the Fukushima-Williams
             angles that are with respect to the GCRS pole (i.e. the variants
             that include frame bias).

        4) The IAU resolution stipulated that the choice of parameterization
           was left to the user, and so an IAU compliant precession
           implementation can be constructed using various combinations of
           the angles returned by the present function.

        5) The parameterization used by ERFA is the version of the Fukushima-
           Williams angles that refers directly to the GCRS pole.  These
           angles may be calculated by calling the function eraPfw06.  ERFA
           also supports the direct computation of the CIP GCRS X,Y by
           series, available by calling eraXy06.

        6) The agreement between the different parameterizations is at the
           1 microarcsecond level in the present era.

        7) When constructing a precession formulation that refers to the GCRS
           pole rather than the dynamical pole, it may (depending on the
           choice of angles) be necessary to introduce the frame bias
           explicitly.

        8) It is permissible to re-use the same variable in the returned
           arguments.  The quantities are stored in the stated order.

        References:

           Capitaine, N., Wallace, P.T. & Chapront, J., 2003,
           Astron.Astrophys., 412, 567

           Hilton, J. et al., 2006, Celest.Mech.Dyn.Astron. 94, 351

        Called:
           eraObl06     mean obliquity, IAU 2006

        This revision:  2020 June 2

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    (eps0, psia, oma, bpa, bqa, pia, bpia, epsa, chia, za, zetaa, thetaa, pa,
     gam, phi, psi) = ufunc.p06e(date1, date2)
    return eps0, psia, oma, bpa, bqa, pia, bpia, epsa, chia, za, zetaa, thetaa, pa, gam, phi, psi


def pb06(date1, date2):
    """
    This function forms three Euler angles which implement general
    precession from epoch J2000.0, using the IAU 2006 model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    bzeta : double array
    bz : double array
    btheta : double array

    Notes
    -----
    Wraps ERFA function ``eraPb06``. The ERFA documentation is::

        - - - - - - - -
         e r a P b 0 6
        - - - - - - - -

        This function forms three Euler angles which implement general
        precession from epoch J2000.0, using the IAU 2006 model.  Frame
        bias (the offset between ICRS and mean J2000.0) is included.

        Given:
           date1,date2  double   TT as a 2-part Julian Date (Note 1)

        Returned:
           bzeta        double   1st rotation: radians cw around z
           bz           double   3rd rotation: radians cw around z
           btheta       double   2nd rotation: radians ccw around y

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The traditional accumulated precession angles zeta_A, z_A,
           theta_A cannot be obtained in the usual way, namely through
           polynomial expressions, because of the frame bias.  The latter
           means that two of the angles undergo rapid changes near this
           date.  They are instead the results of decomposing the
           precession-bias matrix obtained by using the Fukushima-Williams
           method, which does not suffer from the problem.  The
           decomposition returns values which can be used in the
           conventional formulation and which include frame bias.

        3) The three angles are returned in the conventional order, which
           is not the same as the order of the corresponding Euler
           rotations.  The precession-bias matrix is
           R_3(-z) x R_2(+theta) x R_3(-zeta).

        4) Should zeta_A, z_A, theta_A angles be required that do not
           contain frame bias, they are available by calling the ERFA
           function eraP06e.

        Called:
           eraPmat06    PB matrix, IAU 2006
           eraRz        rotate around Z-axis

        This revision:  2020 May 27

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    bzeta, bz, btheta = ufunc.pb06(date1, date2)
    return bzeta, bz, btheta


def pfw06(date1, date2):
    """
    Precession angles, IAU 2006 (Fukushima-Williams 4-angle formulation).

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    gamb : double array
    phib : double array
    psib : double array
    epsa : double array

    Notes
    -----
    Wraps ERFA function ``eraPfw06``. The ERFA documentation is::

        - - - - - - - - -
         e r a P f w 0 6
        - - - - - - - - -

        Precession angles, IAU 2006 (Fukushima-Williams 4-angle formulation).

        Given:
           date1,date2  double   TT as a 2-part Julian Date (Note 1)

        Returned:
           gamb         double   F-W angle gamma_bar (radians)
           phib         double   F-W angle phi_bar (radians)
           psib         double   F-W angle psi_bar (radians)
           epsa         double   F-W angle epsilon_A (radians)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) Naming the following points:

                 e = J2000.0 ecliptic pole,
                 p = GCRS pole,
                 E = mean ecliptic pole of date,
           and   P = mean pole of date,

           the four Fukushima-Williams angles are as follows:

              gamb = gamma_bar = epE
              phib = phi_bar = pE
              psib = psi_bar = pEP
              epsa = epsilon_A = EP

        3) The matrix representing the combined effects of frame bias and
           precession is:

              PxB = R_1(-epsa).R_3(-psib).R_1(phib).R_3(gamb)

        4) The matrix representing the combined effects of frame bias,
           precession and nutation is simply:

              NxPxB = R_1(-epsa-dE).R_3(-psib-dP).R_1(phib).R_3(gamb)

           where dP and dE are the nutation components with respect to the
           ecliptic of date.

        Reference:

           Hilton, J. et al., 2006, Celest.Mech.Dyn.Astron. 94, 351

        Called:
           eraObl06     mean obliquity, IAU 2006

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    gamb, phib, psib, epsa = ufunc.pfw06(date1, date2)
    return gamb, phib, psib, epsa


def pmat00(date1, date2):
    """
    Precession matrix (including frame bias) from GCRS to a specified
    date, IAU 2000 model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rbp : double array

    Notes
    -----
    Wraps ERFA function ``eraPmat00``. The ERFA documentation is::

        - - - - - - - - - -
         e r a P m a t 0 0
        - - - - - - - - - -

        Precession matrix (including frame bias) from GCRS to a specified
        date, IAU 2000 model.

        Given:
           date1,date2  double          TT as a 2-part Julian Date (Note 1)

        Returned:
           rbp          double[3][3]    bias-precession matrix (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix operates in the sense V(date) = rbp * V(GCRS), where
           the p-vector V(GCRS) is with respect to the Geocentric Celestial
           Reference System (IAU, 2000) and the p-vector V(date) is with
           respect to the mean equatorial triad of the given date.

        Called:
           eraBp00      frame bias and precession matrices, IAU 2000

        Reference:

           IAU: Trans. International Astronomical Union, Vol. XXIVB;  Proc.
           24th General Assembly, Manchester, UK.  Resolutions B1.3, B1.6.
           (2000)

        This revision:  2020 November 24

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rbp = ufunc.pmat00(date1, date2)
    return rbp


def pmat06(date1, date2):
    """
    Precession matrix (including frame bias) from GCRS to a specified
    date, IAU 2006 model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rbp : double array

    Notes
    -----
    Wraps ERFA function ``eraPmat06``. The ERFA documentation is::

        - - - - - - - - - -
         e r a P m a t 0 6
        - - - - - - - - - -

        Precession matrix (including frame bias) from GCRS to a specified
        date, IAU 2006 model.

        Given:
           date1,date2  double          TT as a 2-part Julian Date (Note 1)

        Returned:
           rbp          double[3][3]    bias-precession matrix (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix operates in the sense V(date) = rbp * V(GCRS), where
           the p-vector V(GCRS) is with respect to the Geocentric Celestial
           Reference System (IAU, 2000) and the p-vector V(date) is with
           respect to the mean equatorial triad of the given date.

        Called:
           eraPfw06     bias-precession F-W angles, IAU 2006
           eraFw2m      F-W angles to r-matrix

        References:

           Capitaine, N. & Wallace, P.T., 2006, Astron.Astrophys. 450, 855

           IAU: Trans. International Astronomical Union, Vol. XXIVB;  Proc.
           24th General Assembly, Manchester, UK.  Resolutions B1.3, B1.6.
           (2000)

           Wallace, P.T. & Capitaine, N., 2006, Astron.Astrophys. 459, 981

        This revision:  2020 November 24

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rbp = ufunc.pmat06(date1, date2)
    return rbp


def pmat76(date1, date2):
    """
    Precession matrix from J2000.0 to a specified date, IAU 1976 model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rmatp : double array

    Notes
    -----
    Wraps ERFA function ``eraPmat76``. The ERFA documentation is::

        - - - - - - - - - -
         e r a P m a t 7 6
        - - - - - - - - - -

        Precession matrix from J2000.0 to a specified date, IAU 1976 model.

        Given:
           date1,date2 double       ending date, TT (Note 1)

        Returned:
           rmatp       double[3][3] precession matrix, J2000.0 -> date1+date2

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix operates in the sense V(date) = RMATP * V(J2000),
           where the p-vector V(J2000) is with respect to the mean
           equatorial triad of epoch J2000.0 and the p-vector V(date)
           is with respect to the mean equatorial triad of the given
           date.

        3) Though the matrix method itself is rigorous, the precession
           angles are expressed through canonical polynomials which are
           valid only for a limited time span.  In addition, the IAU 1976
           precession rate is known to be imperfect.  The absolute accuracy
           of the present formulation is better than 0.1 arcsec from
           1960AD to 2040AD, better than 1 arcsec from 1640AD to 2360AD,
           and remains below 3 arcsec for the whole of the period
           500BC to 3000AD.  The errors exceed 10 arcsec outside the
           range 1200BC to 3900AD, exceed 100 arcsec outside 4200BC to
           5600AD and exceed 1000 arcsec outside 6800BC to 8200AD.

        Called:
           eraPrec76    accumulated precession angles, IAU 1976
           eraIr        initialize r-matrix to identity
           eraRz        rotate around Z-axis
           eraRy        rotate around Y-axis
           eraCr        copy r-matrix

        References:

           Lieske, J.H., 1979, Astron.Astrophys. 73, 282.
            equations (6) & (7), p283.

           Kaplan,G.H., 1981. USNO circular no. 163, pA2.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rmatp = ufunc.pmat76(date1, date2)
    return rmatp


def pn00(date1, date2, dpsi, deps):
    """
    Precession-nutation, IAU 2000 model:  a multi-purpose function,
    supporting classical (equinox-based) use directly and CIO-based
    use indirectly.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    dpsi : double array
    deps : double array

    Returns
    -------
    epsa : double array
    rb : double array
    rp : double array
    rbp : double array
    rn : double array
    rbpn : double array

    Notes
    -----
    Wraps ERFA function ``eraPn00``. The ERFA documentation is::

        - - - - - - - -
         e r a P n 0 0
        - - - - - - - -

        Precession-nutation, IAU 2000 model:  a multi-purpose function,
        supporting classical (equinox-based) use directly and CIO-based
        use indirectly.

        Given:
           date1,date2  double          TT as a 2-part Julian Date (Note 1)
           dpsi,deps    double          nutation (Note 2)

        Returned:
           epsa         double          mean obliquity (Note 3)
           rb           double[3][3]    frame bias matrix (Note 4)
           rp           double[3][3]    precession matrix (Note 5)
           rbp          double[3][3]    bias-precession matrix (Note 6)
           rn           double[3][3]    nutation matrix (Note 7)
           rbpn         double[3][3]    GCRS-to-true matrix (Note 8)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The caller is responsible for providing the nutation components;
           they are in longitude and obliquity, in radians and are with
           respect to the equinox and ecliptic of date.  For high-accuracy
           applications, free core nutation should be included as well as
           any other relevant corrections to the position of the CIP.

        3) The returned mean obliquity is consistent with the IAU 2000
           precession-nutation models.

        4) The matrix rb transforms vectors from GCRS to J2000.0 mean
           equator and equinox by applying frame bias.

        5) The matrix rp transforms vectors from J2000.0 mean equator and
           equinox to mean equator and equinox of date by applying
           precession.

        6) The matrix rbp transforms vectors from GCRS to mean equator and
           equinox of date by applying frame bias then precession.  It is
           the product rp x rb.

        7) The matrix rn transforms vectors from mean equator and equinox of
           date to true equator and equinox of date by applying the nutation
           (luni-solar + planetary).

        8) The matrix rbpn transforms vectors from GCRS to true equator and
           equinox of date.  It is the product rn x rbp, applying frame
           bias, precession and nutation in that order.

        9) It is permissible to re-use the same array in the returned
           arguments.  The arrays are filled in the order given.

        Called:
           eraPr00      IAU 2000 precession adjustments
           eraObl80     mean obliquity, IAU 1980
           eraBp00      frame bias and precession matrices, IAU 2000
           eraCr        copy r-matrix
           eraNumat     form nutation matrix
           eraRxr       product of two r-matrices

        Reference:

           Capitaine, N., Chapront, J., Lambert, S. and Wallace, P.,
           "Expressions for the Celestial Intermediate Pole and Celestial
           Ephemeris Origin consistent with the IAU 2000A precession-
           nutation model", Astron.Astrophys. 400, 1145-1154 (2003)

           n.b. The celestial ephemeris origin (CEO) was renamed "celestial
                intermediate origin" (CIO) by IAU 2006 Resolution 2.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    epsa, rb, rp, rbp, rn, rbpn = ufunc.pn00(date1, date2, dpsi, deps)
    return epsa, rb, rp, rbp, rn, rbpn


def pn00a(date1, date2):
    """
    Precession-nutation, IAU 2000A model:  a multi-purpose function,
    supporting classical (equinox-based) use directly and CIO-based
    use indirectly.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    dpsi : double array
    deps : double array
    epsa : double array
    rb : double array
    rp : double array
    rbp : double array
    rn : double array
    rbpn : double array

    Notes
    -----
    Wraps ERFA function ``eraPn00a``. The ERFA documentation is::

        - - - - - - - - -
         e r a P n 0 0 a
        - - - - - - - - -

        Precession-nutation, IAU 2000A model:  a multi-purpose function,
        supporting classical (equinox-based) use directly and CIO-based
        use indirectly.

        Given:
           date1,date2  double          TT as a 2-part Julian Date (Note 1)

        Returned:
           dpsi,deps    double          nutation (Note 2)
           epsa         double          mean obliquity (Note 3)
           rb           double[3][3]    frame bias matrix (Note 4)
           rp           double[3][3]    precession matrix (Note 5)
           rbp          double[3][3]    bias-precession matrix (Note 6)
           rn           double[3][3]    nutation matrix (Note 7)
           rbpn         double[3][3]    GCRS-to-true matrix (Notes 8,9)

        Notes:

        1)  The TT date date1+date2 is a Julian Date, apportioned in any
            convenient way between the two arguments.  For example,
            JD(TT)=2450123.7 could be expressed in any of these ways,
            among others:

                   date1          date2

                2450123.7           0.0       (JD method)
                2451545.0       -1421.3       (J2000 method)
                2400000.5       50123.2       (MJD method)
                2450123.5           0.2       (date & time method)

            The JD method is the most natural and convenient to use in
            cases where the loss of several decimal digits of resolution
            is acceptable.  The J2000 method is best matched to the way
            the argument is handled internally and will deliver the
            optimum resolution.  The MJD method and the date & time methods
            are both good compromises between resolution and convenience.

        2)  The nutation components (luni-solar + planetary, IAU 2000A) in
            longitude and obliquity are in radians and with respect to the
            equinox and ecliptic of date.  Free core nutation is omitted;
            for the utmost accuracy, use the eraPn00 function, where the
            nutation components are caller-specified.  For faster but
            slightly less accurate results, use the eraPn00b function.

        3)  The mean obliquity is consistent with the IAU 2000 precession.

        4)  The matrix rb transforms vectors from GCRS to J2000.0 mean
            equator and equinox by applying frame bias.

        5)  The matrix rp transforms vectors from J2000.0 mean equator and
            equinox to mean equator and equinox of date by applying
            precession.

        6)  The matrix rbp transforms vectors from GCRS to mean equator and
            equinox of date by applying frame bias then precession.  It is
            the product rp x rb.

        7)  The matrix rn transforms vectors from mean equator and equinox
            of date to true equator and equinox of date by applying the
            nutation (luni-solar + planetary).

        8)  The matrix rbpn transforms vectors from GCRS to true equator and
            equinox of date.  It is the product rn x rbp, applying frame
            bias, precession and nutation in that order.

        9)  The X,Y,Z coordinates of the IAU 2000A Celestial Intermediate
            Pole are elements (3,1-3) of the GCRS-to-true matrix,
            i.e. rbpn[2][0-2].

        10) It is permissible to re-use the same array in the returned
            arguments.  The arrays are filled in the stated order.

        Called:
           eraNut00a    nutation, IAU 2000A
           eraPn00      bias/precession/nutation results, IAU 2000

        Reference:

           Capitaine, N., Chapront, J., Lambert, S. and Wallace, P.,
           "Expressions for the Celestial Intermediate Pole and Celestial
           Ephemeris Origin consistent with the IAU 2000A precession-
           nutation model", Astron.Astrophys. 400, 1145-1154 (2003)

           n.b. The celestial ephemeris origin (CEO) was renamed "celestial
                intermediate origin" (CIO) by IAU 2006 Resolution 2.

        This revision:  2020 November 25

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dpsi, deps, epsa, rb, rp, rbp, rn, rbpn = ufunc.pn00a(date1, date2)
    return dpsi, deps, epsa, rb, rp, rbp, rn, rbpn


def pn00b(date1, date2):
    """
    Precession-nutation, IAU 2000B model:  a multi-purpose function,
    supporting classical (equinox-based) use directly and CIO-based
    use indirectly.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    dpsi : double array
    deps : double array
    epsa : double array
    rb : double array
    rp : double array
    rbp : double array
    rn : double array
    rbpn : double array

    Notes
    -----
    Wraps ERFA function ``eraPn00b``. The ERFA documentation is::

        - - - - - - - - -
         e r a P n 0 0 b
        - - - - - - - - -

        Precession-nutation, IAU 2000B model:  a multi-purpose function,
        supporting classical (equinox-based) use directly and CIO-based
        use indirectly.

        Given:
           date1,date2  double          TT as a 2-part Julian Date (Note 1)

        Returned:
           dpsi,deps    double          nutation (Note 2)
           epsa         double          mean obliquity (Note 3)
           rb           double[3][3]    frame bias matrix (Note 4)
           rp           double[3][3]    precession matrix (Note 5)
           rbp          double[3][3]    bias-precession matrix (Note 6)
           rn           double[3][3]    nutation matrix (Note 7)
           rbpn         double[3][3]    GCRS-to-true matrix (Notes 8,9)

        Notes:

        1)  The TT date date1+date2 is a Julian Date, apportioned in any
            convenient way between the two arguments.  For example,
            JD(TT)=2450123.7 could be expressed in any of these ways,
            among others:

                   date1          date2

                2450123.7           0.0       (JD method)
                2451545.0       -1421.3       (J2000 method)
                2400000.5       50123.2       (MJD method)
                2450123.5           0.2       (date & time method)

            The JD method is the most natural and convenient to use in
            cases where the loss of several decimal digits of resolution
            is acceptable.  The J2000 method is best matched to the way
            the argument is handled internally and will deliver the
            optimum resolution.  The MJD method and the date & time methods
            are both good compromises between resolution and convenience.

        2)  The nutation components (luni-solar + planetary, IAU 2000B) in
            longitude and obliquity are in radians and with respect to the
            equinox and ecliptic of date.  For more accurate results, but
            at the cost of increased computation, use the eraPn00a function.
            For the utmost accuracy, use the eraPn00 function, where the
            nutation components are caller-specified.

        3)  The mean obliquity is consistent with the IAU 2000 precession.

        4)  The matrix rb transforms vectors from GCRS to J2000.0 mean
            equator and equinox by applying frame bias.

        5)  The matrix rp transforms vectors from J2000.0 mean equator and
            equinox to mean equator and equinox of date by applying
            precession.

        6)  The matrix rbp transforms vectors from GCRS to mean equator and
            equinox of date by applying frame bias then precession.  It is
            the product rp x rb.

        7)  The matrix rn transforms vectors from mean equator and equinox
            of date to true equator and equinox of date by applying the
            nutation (luni-solar + planetary).

        8)  The matrix rbpn transforms vectors from GCRS to true equator and
            equinox of date.  It is the product rn x rbp, applying frame
            bias, precession and nutation in that order.

        9)  The X,Y,Z coordinates of the IAU 2000B Celestial Intermediate
            Pole are elements (3,1-3) of the GCRS-to-true matrix,
            i.e. rbpn[2][0-2].

        10) It is permissible to re-use the same array in the returned
            arguments.  The arrays are filled in the stated order.

        Called:
           eraNut00b    nutation, IAU 2000B
           eraPn00      bias/precession/nutation results, IAU 2000

        Reference:

           Capitaine, N., Chapront, J., Lambert, S. and Wallace, P.,
           "Expressions for the Celestial Intermediate Pole and Celestial
           Ephemeris Origin consistent with the IAU 2000A precession-
           nutation model", Astron.Astrophys. 400, 1145-1154 (2003).

           n.b. The celestial ephemeris origin (CEO) was renamed "celestial
                intermediate origin" (CIO) by IAU 2006 Resolution 2.

        This revision:  2020 November 25

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dpsi, deps, epsa, rb, rp, rbp, rn, rbpn = ufunc.pn00b(date1, date2)
    return dpsi, deps, epsa, rb, rp, rbp, rn, rbpn


def pn06(date1, date2, dpsi, deps):
    """
    Precession-nutation, IAU 2006 model:  a multi-purpose function,
    supporting classical (equinox-based) use directly and CIO-based use
    indirectly.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    dpsi : double array
    deps : double array

    Returns
    -------
    epsa : double array
    rb : double array
    rp : double array
    rbp : double array
    rn : double array
    rbpn : double array

    Notes
    -----
    Wraps ERFA function ``eraPn06``. The ERFA documentation is::

        - - - - - - - -
         e r a P n 0 6
        - - - - - - - -

        Precession-nutation, IAU 2006 model:  a multi-purpose function,
        supporting classical (equinox-based) use directly and CIO-based use
        indirectly.

        Given:
           date1,date2  double          TT as a 2-part Julian Date (Note 1)
           dpsi,deps    double          nutation (Note 2)

        Returned:
           epsa         double          mean obliquity (Note 3)
           rb           double[3][3]    frame bias matrix (Note 4)
           rp           double[3][3]    precession matrix (Note 5)
           rbp          double[3][3]    bias-precession matrix (Note 6)
           rn           double[3][3]    nutation matrix (Note 7)
           rbpn         double[3][3]    GCRS-to-true matrix (Notes 8,9)

        Notes:

        1)  The TT date date1+date2 is a Julian Date, apportioned in any
            convenient way between the two arguments.  For example,
            JD(TT)=2450123.7 could be expressed in any of these ways,
            among others:

                   date1          date2

                2450123.7           0.0       (JD method)
                2451545.0       -1421.3       (J2000 method)
                2400000.5       50123.2       (MJD method)
                2450123.5           0.2       (date & time method)

            The JD method is the most natural and convenient to use in
            cases where the loss of several decimal digits of resolution
            is acceptable.  The J2000 method is best matched to the way
            the argument is handled internally and will deliver the
            optimum resolution.  The MJD method and the date & time methods
            are both good compromises between resolution and convenience.

        2)  The caller is responsible for providing the nutation components;
            they are in longitude and obliquity, in radians and are with
            respect to the equinox and ecliptic of date.  For high-accuracy
            applications, free core nutation should be included as well as
            any other relevant corrections to the position of the CIP.

        3)  The returned mean obliquity is consistent with the IAU 2006
            precession.

        4)  The matrix rb transforms vectors from GCRS to J2000.0 mean
            equator and equinox by applying frame bias.

        5)  The matrix rp transforms vectors from J2000.0 mean equator and
            equinox to mean equator and equinox of date by applying
            precession.

        6)  The matrix rbp transforms vectors from GCRS to mean equator and
            equinox of date by applying frame bias then precession.  It is
            the product rp x rb.

        7)  The matrix rn transforms vectors from mean equator and equinox
            of date to true equator and equinox of date by applying the
            nutation (luni-solar + planetary).

        8)  The matrix rbpn transforms vectors from GCRS to true equator and
            equinox of date.  It is the product rn x rbp, applying frame
            bias, precession and nutation in that order.

        9)  The X,Y,Z coordinates of the Celestial Intermediate Pole are
            elements (3,1-3) of the GCRS-to-true matrix, i.e. rbpn[2][0-2].

        10) It is permissible to re-use the same array in the returned
            arguments.  The arrays are filled in the stated order.

        Called:
           eraPfw06     bias-precession F-W angles, IAU 2006
           eraFw2m      F-W angles to r-matrix
           eraCr        copy r-matrix
           eraTr        transpose r-matrix
           eraRxr       product of two r-matrices

        References:

           Capitaine, N. & Wallace, P.T., 2006, Astron.Astrophys. 450, 855

           Wallace, P.T. & Capitaine, N., 2006, Astron.Astrophys. 459, 981

        This revision:  2020 November 25

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    epsa, rb, rp, rbp, rn, rbpn = ufunc.pn06(date1, date2, dpsi, deps)
    return epsa, rb, rp, rbp, rn, rbpn


def pn06a(date1, date2):
    """
    Precession-nutation, IAU 2006/2000A models:  a multi-purpose function,
    supporting classical (equinox-based) use directly and CIO-based use
    indirectly.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    dpsi : double array
    deps : double array
    epsa : double array
    rb : double array
    rp : double array
    rbp : double array
    rn : double array
    rbpn : double array

    Notes
    -----
    Wraps ERFA function ``eraPn06a``. The ERFA documentation is::

        - - - - - - - - -
         e r a P n 0 6 a
        - - - - - - - - -

        Precession-nutation, IAU 2006/2000A models:  a multi-purpose function,
        supporting classical (equinox-based) use directly and CIO-based use
        indirectly.

        Given:
           date1,date2  double          TT as a 2-part Julian Date (Note 1)

        Returned:
           dpsi,deps    double          nutation (Note 2)
           epsa         double          mean obliquity (Note 3)
           rb           double[3][3]    frame bias matrix (Note 4)
           rp           double[3][3]    precession matrix (Note 5)
           rbp          double[3][3]    bias-precession matrix (Note 6)
           rn           double[3][3]    nutation matrix (Note 7)
           rbpn         double[3][3]    GCRS-to-true matrix (Notes 8,9)

        Notes:

        1)  The TT date date1+date2 is a Julian Date, apportioned in any
            convenient way between the two arguments.  For example,
            JD(TT)=2450123.7 could be expressed in any of these ways,
            among others:

                   date1          date2

                2450123.7           0.0       (JD method)
                2451545.0       -1421.3       (J2000 method)
                2400000.5       50123.2       (MJD method)
                2450123.5           0.2       (date & time method)

            The JD method is the most natural and convenient to use in
            cases where the loss of several decimal digits of resolution
            is acceptable.  The J2000 method is best matched to the way
            the argument is handled internally and will deliver the
            optimum resolution.  The MJD method and the date & time methods
            are both good compromises between resolution and convenience.

        2)  The nutation components (luni-solar + planetary, IAU 2000A) in
            longitude and obliquity are in radians and with respect to the
            equinox and ecliptic of date.  Free core nutation is omitted;
            for the utmost accuracy, use the eraPn06 function, where the
            nutation components are caller-specified.

        3)  The mean obliquity is consistent with the IAU 2006 precession.

        4)  The matrix rb transforms vectors from GCRS to mean J2000.0 by
            applying frame bias.

        5)  The matrix rp transforms vectors from mean J2000.0 to mean of
            date by applying precession.

        6)  The matrix rbp transforms vectors from GCRS to mean of date by
            applying frame bias then precession.  It is the product rp x rb.

        7)  The matrix rn transforms vectors from mean of date to true of
            date by applying the nutation (luni-solar + planetary).

        8)  The matrix rbpn transforms vectors from GCRS to true of date
            (CIP/equinox).  It is the product rn x rbp, applying frame bias,
            precession and nutation in that order.

        9)  The X,Y,Z coordinates of the IAU 2006/2000A Celestial
            Intermediate Pole are elements (3,1-3) of the GCRS-to-true
            matrix, i.e. rbpn[2][0-2].

        10) It is permissible to re-use the same array in the returned
            arguments.  The arrays are filled in the stated order.

        Called:
           eraNut06a    nutation, IAU 2006/2000A
           eraPn06      bias/precession/nutation results, IAU 2006

        Reference:

           Capitaine, N. & Wallace, P.T., 2006, Astron.Astrophys. 450, 855

        This revision:  2013 November 13

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dpsi, deps, epsa, rb, rp, rbp, rn, rbpn = ufunc.pn06a(date1, date2)
    return dpsi, deps, epsa, rb, rp, rbp, rn, rbpn


def pnm00a(date1, date2):
    """
    Form the matrix of precession-nutation for a given date (including
    frame bias), equinox based, IAU 2000A model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rbpn : double array

    Notes
    -----
    Wraps ERFA function ``eraPnm00a``. The ERFA documentation is::

        - - - - - - - - - -
         e r a P n m 0 0 a
        - - - - - - - - - -

        Form the matrix of precession-nutation for a given date (including
        frame bias), equinox based, IAU 2000A model.

        Given:
           date1,date2 double       TT as a 2-part Julian Date (Note 1)

        Returned:
           rbpn        double[3][3] bias-precession-nutation matrix (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways, among
           others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix operates in the sense V(date) = rbpn * V(GCRS), where
           the p-vector V(date) is with respect to the true equatorial triad
           of date date1+date2 and the p-vector V(GCRS) is with respect to
           the Geocentric Celestial Reference System (IAU, 2000).

        3) A faster, but slightly less accurate, result (about 1 mas) can be
           obtained by using instead the eraPnm00b function.

        Called:
           eraPn00a     bias/precession/nutation, IAU 2000A

        Reference:

           IAU: Trans. International Astronomical Union, Vol. XXIVB;  Proc.
           24th General Assembly, Manchester, UK.  Resolutions B1.3, B1.6.
           (2000)

        This revision:  2020 November 27

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rbpn = ufunc.pnm00a(date1, date2)
    return rbpn


def pnm00b(date1, date2):
    """
    Form the matrix of precession-nutation for a given date (including
    frame bias), equinox-based, IAU 2000B model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rbpn : double array

    Notes
    -----
    Wraps ERFA function ``eraPnm00b``. The ERFA documentation is::

        - - - - - - - - - -
         e r a P n m 0 0 b
        - - - - - - - - - -

        Form the matrix of precession-nutation for a given date (including
        frame bias), equinox-based, IAU 2000B model.

        Given:
           date1,date2 double       TT as a 2-part Julian Date (Note 1)

        Returned:
           rbpn        double[3][3] bias-precession-nutation matrix (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways, among
           others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix operates in the sense V(date) = rbpn * V(GCRS), where
           the p-vector V(date) is with respect to the true equatorial triad
           of date date1+date2 and the p-vector V(GCRS) is with respect to
           the Geocentric Celestial Reference System (IAU, 2000).

        3) The present function is faster, but slightly less accurate (about
           1 mas), than the eraPnm00a function.

        Called:
           eraPn00b     bias/precession/nutation, IAU 2000B

        Reference:

           IAU: Trans. International Astronomical Union, Vol. XXIVB;  Proc.
           24th General Assembly, Manchester, UK.  Resolutions B1.3, B1.6.
           (2000)

        This revision:  2020 November 27

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rbpn = ufunc.pnm00b(date1, date2)
    return rbpn


def pnm06a(date1, date2):
    """
    Form the matrix of precession-nutation for a given date (including
    frame bias), equinox based, IAU 2006 precession and IAU 2000A
    nutation models.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rbpn : double array

    Notes
    -----
    Wraps ERFA function ``eraPnm06a``. The ERFA documentation is::

        - - - - - - - - - -
         e r a P n m 0 6 a
        - - - - - - - - - -

        Form the matrix of precession-nutation for a given date (including
        frame bias), equinox based, IAU 2006 precession and IAU 2000A
        nutation models.

        Given:
           date1,date2 double       TT as a 2-part Julian Date (Note 1)

        Returned:
           rbpn        double[3][3] bias-precession-nutation matrix (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways, among
           others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix operates in the sense V(date) = rbpn * V(GCRS), where
           the p-vector V(date) is with respect to the true equatorial triad
           of date date1+date2 and the p-vector V(GCRS) is with respect to
           the Geocentric Celestial Reference System (IAU, 2000).

        Called:
           eraPfw06     bias-precession F-W angles, IAU 2006
           eraNut06a    nutation, IAU 2006/2000A
           eraFw2m      F-W angles to r-matrix

        Reference:

           Capitaine, N. & Wallace, P.T., 2006, Astron.Astrophys. 450, 855.

        This revision:  2020 November 27

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rbpn = ufunc.pnm06a(date1, date2)
    return rbpn


def pnm80(date1, date2):
    """
    Form the matrix of precession/nutation for a given date, IAU 1976
    precession model, IAU 1980 nutation model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rmatpn : double array

    Notes
    -----
    Wraps ERFA function ``eraPnm80``. The ERFA documentation is::

        - - - - - - - - -
         e r a P n m 8 0
        - - - - - - - - -

        Form the matrix of precession/nutation for a given date, IAU 1976
        precession model, IAU 1980 nutation model.

        Given:
           date1,date2 double       TT as a 2-part Julian Date (Note 1)

        Returned:
           rmatpn         double[3][3]   combined precession/nutation matrix

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The matrix operates in the sense V(date) = rmatpn * V(J2000),
           where the p-vector V(date) is with respect to the true equatorial
           triad of date date1+date2 and the p-vector V(J2000) is with
           respect to the mean equatorial triad of epoch J2000.0.

        Called:
           eraPmat76    precession matrix, IAU 1976
           eraNutm80    nutation matrix, IAU 1980
           eraRxr       product of two r-matrices

        Reference:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992),
           Section 3.3 (p145).

        This revision:  2020 November 27

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rmatpn = ufunc.pnm80(date1, date2)
    return rmatpn


def pom00(xp, yp, sp):
    """
    Form the matrix of polar motion for a given date, IAU 2000.

    Parameters
    ----------
    xp : double array
    yp : double array
    sp : double array

    Returns
    -------
    rpom : double array

    Notes
    -----
    Wraps ERFA function ``eraPom00``. The ERFA documentation is::

        - - - - - - - - - -
         e r a P o m 0 0
        - - - - - - - - - -

        Form the matrix of polar motion for a given date, IAU 2000.

        Given:
           xp,yp    double    coordinates of the pole (radians, Note 1)
           sp       double    the TIO locator s' (radians, Note 2)

        Returned:
           rpom     double[3][3]   polar-motion matrix (Note 3)

        Notes:

        1) The arguments xp and yp are the coordinates (in radians) of the
           Celestial Intermediate Pole with respect to the International
           Terrestrial Reference System (see IERS Conventions 2003),
           measured along the meridians 0 and 90 deg west respectively.

        2) The argument sp is the TIO locator s', in radians, which
           positions the Terrestrial Intermediate Origin on the equator.  It
           is obtained from polar motion observations by numerical
           integration, and so is in essence unpredictable.  However, it is
           dominated by a secular drift of about 47 microarcseconds per
           century, and so can be taken into account by using s' = -47*t,
           where t is centuries since J2000.0.  The function eraSp00
           implements this approximation.

        3) The matrix operates in the sense V(TRS) = rpom * V(CIP), meaning
           that it is the final rotation when computing the pointing
           direction to a celestial source.

        Called:
           eraIr        initialize r-matrix to identity
           eraRz        rotate around Z-axis
           eraRy        rotate around Y-axis
           eraRx        rotate around X-axis

        Reference:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2020 October 31

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rpom = ufunc.pom00(xp, yp, sp)
    return rpom


def pr00(date1, date2):
    """
    Precession-rate part of the IAU 2000 precession-nutation models
    (part of MHB2000).

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    dpsipr : double array
    depspr : double array

    Notes
    -----
    Wraps ERFA function ``eraPr00``. The ERFA documentation is::

        - - - - - - - -
         e r a P r 0 0
        - - - - - - - -

        Precession-rate part of the IAU 2000 precession-nutation models
        (part of MHB2000).

        Given:
           date1,date2    double  TT as a 2-part Julian Date (Note 1)

        Returned:
           dpsipr,depspr  double  precession corrections (Notes 2,3)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The precession adjustments are expressed as "nutation
           components", corrections in longitude and obliquity with respect
           to the J2000.0 equinox and ecliptic.

        3) Although the precession adjustments are stated to be with respect
           to Lieske et al. (1977), the MHB2000 model does not specify which
           set of Euler angles are to be used and how the adjustments are to
           be applied.  The most literal and straightforward procedure is to
           adopt the 4-rotation epsilon_0, psi_A, omega_A, xi_A option, and
           to add dpsipr to psi_A and depspr to both omega_A and eps_A.

        4) This is an implementation of one aspect of the IAU 2000A nutation
           model, formally adopted by the IAU General Assembly in 2000,
           namely MHB2000 (Mathews et al. 2002).

        References:

           Lieske, J.H., Lederle, T., Fricke, W. & Morando, B., "Expressions
           for the precession quantities based upon the IAU (1976) System of
           Astronomical Constants", Astron.Astrophys., 58, 1-16 (1977)

           Mathews, P.M., Herring, T.A., Buffet, B.A., "Modeling of nutation
           and precession   New nutation series for nonrigid Earth and
           insights into the Earth's interior", J.Geophys.Res., 107, B4,
           2002.  The MHB2000 code itself was obtained on 9th September 2002
           from ftp://maia.usno.navy.mil/conv2000/chapter5/IAU2000A.

           Wallace, P.T., "Software for Implementing the IAU 2000
           Resolutions", in IERS Workshop 5.1 (2002).

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dpsipr, depspr = ufunc.pr00(date1, date2)
    return dpsipr, depspr


def prec76(date01, date02, date11, date12):
    """
    IAU 1976 precession model.

    Parameters
    ----------
    date01 : double array
    date02 : double array
    date11 : double array
    date12 : double array

    Returns
    -------
    zeta : double array
    z : double array
    theta : double array

    Notes
    -----
    Wraps ERFA function ``eraPrec76``. The ERFA documentation is::

        - - - - - - - - - -
         e r a P r e c 7 6
        - - - - - - - - - -

        IAU 1976 precession model.

        This function forms the three Euler angles which implement general
        precession between two dates, using the IAU 1976 model (as for the
        FK5 catalog).

        Given:
           date01,date02   double    TDB starting date (Note 1)
           date11,date12   double    TDB ending date (Note 1)

        Returned:
           zeta            double    1st rotation: radians cw around z
           z               double    3rd rotation: radians cw around z
           theta           double    2nd rotation: radians ccw around y

        Notes:

        1) The dates date01+date02 and date11+date12 are Julian Dates,
           apportioned in any convenient way between the arguments daten1
           and daten2.  For example, JD(TDB)=2450123.7 could be expressed in
           any of these ways, among others:

                 daten1        daten2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 method is best matched to the way the
           argument is handled internally and will deliver the optimum
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.
           The two dates may be expressed using different methods, but at
           the risk of losing some resolution.

        2) The accumulated precession angles zeta, z, theta are expressed
           through canonical polynomials which are valid only for a limited
           time span.  In addition, the IAU 1976 precession rate is known to
           be imperfect.  The absolute accuracy of the present formulation
           is better than 0.1 arcsec from 1960AD to 2040AD, better than
           1 arcsec from 1640AD to 2360AD, and remains below 3 arcsec for
           the whole of the period 500BC to 3000AD.  The errors exceed
           10 arcsec outside the range 1200BC to 3900AD, exceed 100 arcsec
           outside 4200BC to 5600AD and exceed 1000 arcsec outside 6800BC to
           8200AD.

        3) The three angles are returned in the conventional order, which
           is not the same as the order of the corresponding Euler
           rotations.  The precession matrix is
           R_3(-z) x R_2(+theta) x R_3(-zeta).

        Reference:

           Lieske, J.H., 1979, Astron.Astrophys. 73, 282, equations
           (6) & (7), p283.

        This revision:  2013 November 19

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    zeta, z, theta = ufunc.prec76(date01, date02, date11, date12)
    return zeta, z, theta


def s00(date1, date2, x, y):
    """
    The CIO locator s, positioning the Celestial Intermediate Origin on
    the equator of the Celestial Intermediate Pole, given the CIP's X,Y
    coordinates.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    x : double array
    y : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraS00``. The ERFA documentation is::

        - - - - - - -
         e r a S 0 0
        - - - - - - -

        The CIO locator s, positioning the Celestial Intermediate Origin on
        the equator of the Celestial Intermediate Pole, given the CIP's X,Y
        coordinates.  Compatible with IAU 2000A precession-nutation.

        Given:
           date1,date2   double    TT as a 2-part Julian Date (Note 1)
           x,y           double    CIP coordinates (Note 3)

        Returned (function value):
                         double    the CIO locator s in radians (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The CIO locator s is the difference between the right ascensions
           of the same point in two systems:  the two systems are the GCRS
           and the CIP,CIO, and the point is the ascending node of the
           CIP equator.  The quantity s remains below 0.1 arcsecond
           throughout 1900-2100.

        3) The series used to compute s is in fact for s+XY/2, where X and Y
           are the x and y components of the CIP unit vector;  this series
           is more compact than a direct series for s would be.  This
           function requires X,Y to be supplied by the caller, who is
           responsible for providing values that are consistent with the
           supplied date.

        4) The model is consistent with the IAU 2000A precession-nutation.

        Called:
           eraFal03     mean anomaly of the Moon
           eraFalp03    mean anomaly of the Sun
           eraFaf03     mean argument of the latitude of the Moon
           eraFad03     mean elongation of the Moon from the Sun
           eraFaom03    mean longitude of the Moon's ascending node
           eraFave03    mean longitude of Venus
           eraFae03     mean longitude of Earth
           eraFapa03    general accumulated precession in longitude

        References:

           Capitaine, N., Chapront, J., Lambert, S. and Wallace, P.,
           "Expressions for the Celestial Intermediate Pole and Celestial
           Ephemeris Origin consistent with the IAU 2000A precession-
           nutation model", Astron.Astrophys. 400, 1145-1154 (2003)

           n.b. The celestial ephemeris origin (CEO) was renamed "celestial
                intermediate origin" (CIO) by IAU 2006 Resolution 2.

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2020 November 4

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.s00(date1, date2, x, y)
    return c_retval


def s00a(date1, date2):
    """
    The CIO locator s, positioning the Celestial Intermediate Origin on
    the equator of the Celestial Intermediate Pole, using the IAU 2000A
    precession-nutation model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraS00a``. The ERFA documentation is::

        - - - - - - - -
         e r a S 0 0 a
        - - - - - - - -

        The CIO locator s, positioning the Celestial Intermediate Origin on
        the equator of the Celestial Intermediate Pole, using the IAU 2000A
        precession-nutation model.

        Given:
           date1,date2  double    TT as a 2-part Julian Date (Note 1)

        Returned (function value):
                        double    the CIO locator s in radians (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The CIO locator s is the difference between the right ascensions
           of the same point in two systems.  The two systems are the GCRS
           and the CIP,CIO, and the point is the ascending node of the
           CIP equator.  The CIO locator s remains a small fraction of
           1 arcsecond throughout 1900-2100.

        3) The series used to compute s is in fact for s+XY/2, where X and Y
           are the x and y components of the CIP unit vector;  this series
           is more compact than a direct series for s would be.  The present
           function uses the full IAU 2000A nutation model when predicting
           the CIP position.  Faster results, with no significant loss of
           accuracy, can be obtained via the function eraS00b, which uses
           instead the IAU 2000B truncated model.

        Called:
           eraPnm00a    classical NPB matrix, IAU 2000A
           eraBnp2xy    extract CIP X,Y from the BPN matrix
           eraS00       the CIO locator s, given X,Y, IAU 2000A

        References:

           Capitaine, N., Chapront, J., Lambert, S. and Wallace, P.,
           "Expressions for the Celestial Intermediate Pole and Celestial
           Ephemeris Origin consistent with the IAU 2000A precession-
           nutation model", Astron.Astrophys. 400, 1145-1154 (2003)

           n.b. The celestial ephemeris origin (CEO) was renamed "celestial
                intermediate origin" (CIO) by IAU 2006 Resolution 2.

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.s00a(date1, date2)
    return c_retval


def s00b(date1, date2):
    """
    The CIO locator s, positioning the Celestial Intermediate Origin on
    the equator of the Celestial Intermediate Pole, using the IAU 2000B
    precession-nutation model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraS00b``. The ERFA documentation is::

        - - - - - - - -
         e r a S 0 0 b
        - - - - - - - -

        The CIO locator s, positioning the Celestial Intermediate Origin on
        the equator of the Celestial Intermediate Pole, using the IAU 2000B
        precession-nutation model.

        Given:
           date1,date2  double    TT as a 2-part Julian Date (Note 1)

        Returned (function value):
                        double    the CIO locator s in radians (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The CIO locator s is the difference between the right ascensions
           of the same point in two systems.  The two systems are the GCRS
           and the CIP,CIO, and the point is the ascending node of the
           CIP equator.  The CIO locator s remains a small fraction of
           1 arcsecond throughout 1900-2100.

        3) The series used to compute s is in fact for s+XY/2, where X and Y
           are the x and y components of the CIP unit vector;  this series
           is more compact than a direct series for s would be.  The present
           function uses the IAU 2000B truncated nutation model when
           predicting the CIP position.  The function eraS00a uses instead
           the full IAU 2000A model, but with no significant increase in
           accuracy and at some cost in speed.

        Called:
           eraPnm00b    classical NPB matrix, IAU 2000B
           eraBnp2xy    extract CIP X,Y from the BPN matrix
           eraS00       the CIO locator s, given X,Y, IAU 2000A

        References:

           Capitaine, N., Chapront, J., Lambert, S. and Wallace, P.,
           "Expressions for the Celestial Intermediate Pole and Celestial
           Ephemeris Origin consistent with the IAU 2000A precession-
           nutation model", Astron.Astrophys. 400, 1145-1154 (2003)

           n.b. The celestial ephemeris origin (CEO) was renamed "celestial
                intermediate origin" (CIO) by IAU 2006 Resolution 2.

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.s00b(date1, date2)
    return c_retval


def s06(date1, date2, x, y):
    """
    The CIO locator s, positioning the Celestial Intermediate Origin on
    the equator of the Celestial Intermediate Pole, given the CIP's X,Y
    coordinates.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    x : double array
    y : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraS06``. The ERFA documentation is::

        - - - - - - -
         e r a S 0 6
        - - - - - - -

        The CIO locator s, positioning the Celestial Intermediate Origin on
        the equator of the Celestial Intermediate Pole, given the CIP's X,Y
        coordinates.  Compatible with IAU 2006/2000A precession-nutation.

        Given:
           date1,date2   double    TT as a 2-part Julian Date (Note 1)
           x,y           double    CIP coordinates (Note 3)

        Returned (function value):
                         double    the CIO locator s in radians (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The CIO locator s is the difference between the right ascensions
           of the same point in two systems:  the two systems are the GCRS
           and the CIP,CIO, and the point is the ascending node of the
           CIP equator.  The quantity s remains below 0.1 arcsecond
           throughout 1900-2100.

        3) The series used to compute s is in fact for s+XY/2, where X and Y
           are the x and y components of the CIP unit vector;  this series
           is more compact than a direct series for s would be.  This
           function requires X,Y to be supplied by the caller, who is
           responsible for providing values that are consistent with the
           supplied date.

        4) The model is consistent with the "P03" precession (Capitaine et
           al. 2003), adopted by IAU 2006 Resolution 1, 2006, and the
           IAU 2000A nutation (with P03 adjustments).

        Called:
           eraFal03     mean anomaly of the Moon
           eraFalp03    mean anomaly of the Sun
           eraFaf03     mean argument of the latitude of the Moon
           eraFad03     mean elongation of the Moon from the Sun
           eraFaom03    mean longitude of the Moon's ascending node
           eraFave03    mean longitude of Venus
           eraFae03     mean longitude of Earth
           eraFapa03    general accumulated precession in longitude

        References:

           Capitaine, N., Wallace, P.T. & Chapront, J., 2003, Astron.
           Astrophys. 432, 355

           McCarthy, D.D., Petit, G. (eds.) 2004, IERS Conventions (2003),
           IERS Technical Note No. 32, BKG

        This revision:  2019 June 23

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.s06(date1, date2, x, y)
    return c_retval


def s06a(date1, date2):
    """
    The CIO locator s, positioning the Celestial Intermediate Origin on
    the equator of the Celestial Intermediate Pole, using the IAU 2006
    precession and IAU 2000A nutation models.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraS06a``. The ERFA documentation is::

        - - - - - - - -
         e r a S 0 6 a
        - - - - - - - -

        The CIO locator s, positioning the Celestial Intermediate Origin on
        the equator of the Celestial Intermediate Pole, using the IAU 2006
        precession and IAU 2000A nutation models.

        Given:
           date1,date2  double    TT as a 2-part Julian Date (Note 1)

        Returned (function value):
                        double    the CIO locator s in radians (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The CIO locator s is the difference between the right ascensions
           of the same point in two systems.  The two systems are the GCRS
           and the CIP,CIO, and the point is the ascending node of the
           CIP equator.  The CIO locator s remains a small fraction of
           1 arcsecond throughout 1900-2100.

        3) The series used to compute s is in fact for s+XY/2, where X and Y
           are the x and y components of the CIP unit vector;  this series is
           more compact than a direct series for s would be.  The present
           function uses the full IAU 2000A nutation model when predicting
           the CIP position.

        Called:
           eraPnm06a    classical NPB matrix, IAU 2006/2000A
           eraBpn2xy    extract CIP X,Y coordinates from NPB matrix
           eraS06       the CIO locator s, given X,Y, IAU 2006

        References:

           Capitaine, N., Chapront, J., Lambert, S. and Wallace, P.,
           "Expressions for the Celestial Intermediate Pole and Celestial
           Ephemeris Origin consistent with the IAU 2000A precession-
           nutation model", Astron.Astrophys. 400, 1145-1154 (2003)

           n.b. The celestial ephemeris origin (CEO) was renamed "celestial
                intermediate origin" (CIO) by IAU 2006 Resolution 2.

           Capitaine, N. & Wallace, P.T., 2006, Astron.Astrophys. 450, 855

           McCarthy, D. D., Petit, G. (eds.), 2004, IERS Conventions (2003),
           IERS Technical Note No. 32, BKG

           Wallace, P.T. & Capitaine, N., 2006, Astron.Astrophys. 459, 981

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.s06a(date1, date2)
    return c_retval


def sp00(date1, date2):
    """
    The TIO locator s', positioning the Terrestrial Intermediate Origin
    on the equator of the Celestial Intermediate Pole.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraSp00``. The ERFA documentation is::

        - - - - - - - -
         e r a S p 0 0
        - - - - - - - -

        The TIO locator s', positioning the Terrestrial Intermediate Origin
        on the equator of the Celestial Intermediate Pole.

        Given:
           date1,date2  double    TT as a 2-part Julian Date (Note 1)

        Returned (function value):
                        double    the TIO locator s' in radians (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The TIO locator s' is obtained from polar motion observations by
           numerical integration, and so is in essence unpredictable.
           However, it is dominated by a secular drift of about
           47 microarcseconds per century, which is the approximation
           evaluated by the present function.

        Reference:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.sp00(date1, date2)
    return c_retval


def xy06(date1, date2):
    """
    X,Y coordinates of celestial intermediate pole from series based
    on IAU 2006 precession and IAU 2000A nutation.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    x : double array
    y : double array

    Notes
    -----
    Wraps ERFA function ``eraXy06``. The ERFA documentation is::

        - - - - - - - -
         e r a X y 0 6
        - - - - - - - -

        X,Y coordinates of celestial intermediate pole from series based
        on IAU 2006 precession and IAU 2000A nutation.

        Given:
           date1,date2  double     TT as a 2-part Julian Date (Note 1)

        Returned:
           x,y          double     CIP X,Y coordinates (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The X,Y coordinates are those of the unit vector towards the
           celestial intermediate pole.  They represent the combined effects
           of frame bias, precession and nutation.

        3) The fundamental arguments used are as adopted in IERS Conventions
           (2003) and are from Simon et al. (1994) and Souchay et al.
           (1999).

        4) This is an alternative to the angles-based method, via the ERFA
           function eraFw2xy and as used in eraXys06a for example.  The two
           methods agree at the 1 microarcsecond level (at present), a
           negligible amount compared with the intrinsic accuracy of the
           models.  However, it would be unwise to mix the two methods
           (angles-based and series-based) in a single application.

        Called:
           eraFal03     mean anomaly of the Moon
           eraFalp03    mean anomaly of the Sun
           eraFaf03     mean argument of the latitude of the Moon
           eraFad03     mean elongation of the Moon from the Sun
           eraFaom03    mean longitude of the Moon's ascending node
           eraFame03    mean longitude of Mercury
           eraFave03    mean longitude of Venus
           eraFae03     mean longitude of Earth
           eraFama03    mean longitude of Mars
           eraFaju03    mean longitude of Jupiter
           eraFasa03    mean longitude of Saturn
           eraFaur03    mean longitude of Uranus
           eraFane03    mean longitude of Neptune
           eraFapa03    general accumulated precession in longitude

        References:

           Capitaine, N., Wallace, P.T. & Chapront, J., 2003,
           Astron.Astrophys., 412, 567

           Capitaine, N. & Wallace, P.T., 2006, Astron.Astrophys. 450, 855

           McCarthy, D. D., Petit, G. (eds.), 2004, IERS Conventions (2003),
           IERS Technical Note No. 32, BKG

           Simon, J.L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G. & Laskar, J., Astron.Astrophys., 1994, 282, 663

           Souchay, J., Loysel, B., Kinoshita, H., Folgueira, M., 1999,
           Astron.Astrophys.Supp.Ser. 135, 111

           Wallace, P.T. & Capitaine, N., 2006, Astron.Astrophys. 459, 981

        This revision:  2019 June 23

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    x, y = ufunc.xy06(date1, date2)
    return x, y


def xys00a(date1, date2):
    """
    For a given TT date, compute the X,Y coordinates of the Celestial
    Intermediate Pole and the CIO locator s, using the IAU 2000A
    precession-nutation model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    x : double array
    y : double array
    s : double array

    Notes
    -----
    Wraps ERFA function ``eraXys00a``. The ERFA documentation is::

        - - - - - - - - - -
         e r a X y s 0 0 a
        - - - - - - - - - -

        For a given TT date, compute the X,Y coordinates of the Celestial
        Intermediate Pole and the CIO locator s, using the IAU 2000A
        precession-nutation model.

        Given:
           date1,date2  double   TT as a 2-part Julian Date (Note 1)

        Returned:
           x,y          double   Celestial Intermediate Pole (Note 2)
           s            double   the CIO locator s (Note 3)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The Celestial Intermediate Pole coordinates are the x,y
           components of the unit vector in the Geocentric Celestial
           Reference System.

        3) The CIO locator s (in radians) positions the Celestial
           Intermediate Origin on the equator of the CIP.

        4) A faster, but slightly less accurate result (about 1 mas for
           X,Y), can be obtained by using instead the eraXys00b function.

        Called:
           eraPnm00a    classical NPB matrix, IAU 2000A
           eraBpn2xy    extract CIP X,Y coordinates from NPB matrix
           eraS00       the CIO locator s, given X,Y, IAU 2000A

        Reference:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2019 November 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    x, y, s = ufunc.xys00a(date1, date2)
    return x, y, s


def xys00b(date1, date2):
    """
    For a given TT date, compute the X,Y coordinates of the Celestial
    Intermediate Pole and the CIO locator s, using the IAU 2000B
    precession-nutation model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    x : double array
    y : double array
    s : double array

    Notes
    -----
    Wraps ERFA function ``eraXys00b``. The ERFA documentation is::

        - - - - - - - - - -
         e r a X y s 0 0 b
        - - - - - - - - - -

        For a given TT date, compute the X,Y coordinates of the Celestial
        Intermediate Pole and the CIO locator s, using the IAU 2000B
        precession-nutation model.

        Given:
           date1,date2  double   TT as a 2-part Julian Date (Note 1)

        Returned:
           x,y          double   Celestial Intermediate Pole (Note 2)
           s            double   the CIO locator s (Note 3)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The Celestial Intermediate Pole coordinates are the x,y
           components of the unit vector in the Geocentric Celestial
           Reference System.

        3) The CIO locator s (in radians) positions the Celestial
           Intermediate Origin on the equator of the CIP.

        4) The present function is faster, but slightly less accurate (about
           1 mas in X,Y), than the eraXys00a function.

        Called:
           eraPnm00b    classical NPB matrix, IAU 2000B
           eraBpn2xy    extract CIP X,Y coordinates from NPB matrix
           eraS00       the CIO locator s, given X,Y, IAU 2000A

        Reference:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2020 December 4

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    x, y, s = ufunc.xys00b(date1, date2)
    return x, y, s


def xys06a(date1, date2):
    """
    For a given TT date, compute the X,Y coordinates of the Celestial
    Intermediate Pole and the CIO locator s, using the IAU 2006
    precession and IAU 2000A nutation models.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    x : double array
    y : double array
    s : double array

    Notes
    -----
    Wraps ERFA function ``eraXys06a``. The ERFA documentation is::

        - - - - - - - - - -
         e r a X y s 0 6 a
        - - - - - - - - - -

        For a given TT date, compute the X,Y coordinates of the Celestial
        Intermediate Pole and the CIO locator s, using the IAU 2006
        precession and IAU 2000A nutation models.

        Given:
           date1,date2  double  TT as a 2-part Julian Date (Note 1)

        Returned:
           x,y          double  Celestial Intermediate Pole (Note 2)
           s            double  the CIO locator s (Note 3)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The Celestial Intermediate Pole coordinates are the x,y components
           of the unit vector in the Geocentric Celestial Reference System.

        3) The CIO locator s (in radians) positions the Celestial
           Intermediate Origin on the equator of the CIP.

        4) Series-based solutions for generating X and Y are also available:
           see Capitaine & Wallace (2006) and eraXy06.

        Called:
           eraPnm06a    classical NPB matrix, IAU 2006/2000A
           eraBpn2xy    extract CIP X,Y coordinates from NPB matrix
           eraS06       the CIO locator s, given X,Y, IAU 2006

        References:

           Capitaine, N. & Wallace, P.T., 2006, Astron.Astrophys. 450, 855

           Wallace, P.T. & Capitaine, N., 2006, Astron.Astrophys. 459, 981

        This revision:  2020 December 4

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    x, y, s = ufunc.xys06a(date1, date2)
    return x, y, s


def ee00(date1, date2, epsa, dpsi):
    """
    The equation of the equinoxes, compatible with IAU 2000 resolutions,
    given the nutation in longitude and the mean obliquity.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    epsa : double array
    dpsi : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraEe00``. The ERFA documentation is::

        - - - - - - - -
         e r a E e 0 0
        - - - - - - - -

        The equation of the equinoxes, compatible with IAU 2000 resolutions,
        given the nutation in longitude and the mean obliquity.

        Given:
           date1,date2  double    TT as a 2-part Julian Date (Note 1)
           epsa         double    mean obliquity (Note 2)
           dpsi         double    nutation in longitude (Note 3)

        Returned (function value):
                        double    equation of the equinoxes (Note 4)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The obliquity, in radians, is mean of date.

        3) The result, which is in radians, operates in the following sense:

              Greenwich apparent ST = GMST + equation of the equinoxes

        4) The result is compatible with the IAU 2000 resolutions.  For
           further details, see IERS Conventions 2003 and Capitaine et al.
           (2002).

        Called:
           eraEect00    equation of the equinoxes complementary terms

        References:

           Capitaine, N., Wallace, P.T. and McCarthy, D.D., "Expressions to
           implement the IAU 2000 definition of UT1", Astronomy &
           Astrophysics, 406, 1135-1149 (2003)

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2008 May 16

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.ee00(date1, date2, epsa, dpsi)
    return c_retval


def ee00a(date1, date2):
    """
    Equation of the equinoxes, compatible with IAU 2000 resolutions.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraEe00a``. The ERFA documentation is::

        - - - - - - - - -
         e r a E e 0 0 a
        - - - - - - - - -

        Equation of the equinoxes, compatible with IAU 2000 resolutions.

        Given:
           date1,date2  double    TT as a 2-part Julian Date (Note 1)

        Returned (function value):
                        double    equation of the equinoxes (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The result, which is in radians, operates in the following sense:

              Greenwich apparent ST = GMST + equation of the equinoxes

        3) The result is compatible with the IAU 2000 resolutions.  For
           further details, see IERS Conventions 2003 and Capitaine et al.
           (2002).

        Called:
           eraPr00      IAU 2000 precession adjustments
           eraObl80     mean obliquity, IAU 1980
           eraNut00a    nutation, IAU 2000A
           eraEe00      equation of the equinoxes, IAU 2000

        References:

           Capitaine, N., Wallace, P.T. and McCarthy, D.D., "Expressions to
           implement the IAU 2000 definition of UT1", Astronomy &
           Astrophysics, 406, 1135-1149 (2003).

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004).

        This revision:  2008 May 16

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.ee00a(date1, date2)
    return c_retval


def ee00b(date1, date2):
    """
    Equation of the equinoxes, compatible with IAU 2000 resolutions but
    using the truncated nutation model IAU 2000B.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraEe00b``. The ERFA documentation is::

        - - - - - - - - -
         e r a E e 0 0 b
        - - - - - - - - -

        Equation of the equinoxes, compatible with IAU 2000 resolutions but
        using the truncated nutation model IAU 2000B.

        Given:
           date1,date2  double    TT as a 2-part Julian Date (Note 1)

        Returned (function value):
                        double    equation of the equinoxes (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The result, which is in radians, operates in the following sense:

              Greenwich apparent ST = GMST + equation of the equinoxes

        3) The result is compatible with the IAU 2000 resolutions except
           that accuracy has been compromised (1 mas) for the sake of speed.
           For further details, see McCarthy & Luzum (2003), IERS
           Conventions 2003 and Capitaine et al. (2003).

        Called:
           eraPr00      IAU 2000 precession adjustments
           eraObl80     mean obliquity, IAU 1980
           eraNut00b    nutation, IAU 2000B
           eraEe00      equation of the equinoxes, IAU 2000

        References:

           Capitaine, N., Wallace, P.T. and McCarthy, D.D., "Expressions to
           implement the IAU 2000 definition of UT1", Astronomy &
           Astrophysics, 406, 1135-1149 (2003)

           McCarthy, D.D. & Luzum, B.J., "An abridged model of the
           precession-nutation of the celestial pole", Celestial Mechanics &
           Dynamical Astronomy, 85, 37-49 (2003)

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2020 November 3

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.ee00b(date1, date2)
    return c_retval


def ee06a(date1, date2):
    """
    Equation of the equinoxes, compatible with IAU 2000 resolutions and
    IAU 2006/2000A precession-nutation.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraEe06a``. The ERFA documentation is::

        - - - - - - - - -
         e r a E e 0 6 a
        - - - - - - - - -

        Equation of the equinoxes, compatible with IAU 2000 resolutions and
        IAU 2006/2000A precession-nutation.

        Given:
           date1,date2  double    TT as a 2-part Julian Date (Note 1)

        Returned (function value):
                        double    equation of the equinoxes (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The result, which is in radians, operates in the following sense:

              Greenwich apparent ST = GMST + equation of the equinoxes

        Called:
           eraAnpm      normalize angle into range +/- pi
           eraGst06a    Greenwich apparent sidereal time, IAU 2006/2000A
           eraGmst06    Greenwich mean sidereal time, IAU 2006

        Reference:

           McCarthy, D. D., Petit, G. (eds.), 2004, IERS Conventions (2003),
           IERS Technical Note No. 32, BKG

        This revision:  2008 May 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.ee06a(date1, date2)
    return c_retval


def eect00(date1, date2):
    """
    Equation of the equinoxes complementary terms, consistent with
    IAU 2000 resolutions.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraEect00``. The ERFA documentation is::

        - - - - - - - - - -
         e r a E e c t 0 0
        - - - - - - - - - -

        Equation of the equinoxes complementary terms, consistent with
        IAU 2000 resolutions.

        Given:
           date1,date2  double   TT as a 2-part Julian Date (Note 1)

        Returned (function value):
                        double   complementary terms (Note 2)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The "complementary terms" are part of the equation of the
           equinoxes (EE), classically the difference between apparent and
           mean Sidereal Time:

              GAST = GMST + EE

           with:

              EE = dpsi * cos(eps)

           where dpsi is the nutation in longitude and eps is the obliquity
           of date.  However, if the rotation of the Earth were constant in
           an inertial frame the classical formulation would lead to
           apparent irregularities in the UT1 timescale traceable to side-
           effects of precession-nutation.  In order to eliminate these
           effects from UT1, "complementary terms" were introduced in 1994
           (IAU, 1994) and took effect from 1997 (Capitaine and Gontier,
           1993):

              GAST = GMST + CT + EE

           By convention, the complementary terms are included as part of
           the equation of the equinoxes rather than as part of the mean
           Sidereal Time.  This slightly compromises the "geometrical"
           interpretation of mean sidereal time but is otherwise
           inconsequential.

           The present function computes CT in the above expression,
           compatible with IAU 2000 resolutions (Capitaine et al., 2002, and
           IERS Conventions 2003).

        Called:
           eraFal03     mean anomaly of the Moon
           eraFalp03    mean anomaly of the Sun
           eraFaf03     mean argument of the latitude of the Moon
           eraFad03     mean elongation of the Moon from the Sun
           eraFaom03    mean longitude of the Moon's ascending node
           eraFave03    mean longitude of Venus
           eraFae03     mean longitude of Earth
           eraFapa03    general accumulated precession in longitude

        References:

           Capitaine, N. & Gontier, A.-M., Astron.Astrophys., 275,
           645-650 (1993)

           Capitaine, N., Wallace, P.T. and McCarthy, D.D., "Expressions to
           implement the IAU 2000 definition of UT1", Astron.Astrophys., 406,
           1135-1149 (2003)

           IAU Resolution C7, Recommendation 3 (1994)

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2019 June 23

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.eect00(date1, date2)
    return c_retval


def eqeq94(date1, date2):
    """
    Equation of the equinoxes, IAU 1994 model.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraEqeq94``. The ERFA documentation is::

        - - - - - - - - - -
         e r a E q e q 9 4
        - - - - - - - - - -

        Equation of the equinoxes, IAU 1994 model.

        Given:
           date1,date2   double     TDB date (Note 1)

        Returned (function value):
                         double     equation of the equinoxes (Note 2)

        Notes:

        1) The date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The result, which is in radians, operates in the following sense:

              Greenwich apparent ST = GMST + equation of the equinoxes

        Called:
           eraAnpm      normalize angle into range +/- pi
           eraNut80     nutation, IAU 1980
           eraObl80     mean obliquity, IAU 1980

        References:

           IAU Resolution C7, Recommendation 3 (1994).

           Capitaine, N. & Gontier, A.-M., 1993, Astron.Astrophys., 275,
           645-650.

        This revision:  2017 October 12

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.eqeq94(date1, date2)
    return c_retval


def era00(dj1, dj2):
    """
    Earth rotation angle (IAU 2000 model).

    Parameters
    ----------
    dj1 : double array
    dj2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraEra00``. The ERFA documentation is::

        - - - - - - - - -
         e r a E r a 0 0
        - - - - - - - - -

        Earth rotation angle (IAU 2000 model).

        Given:
           dj1,dj2   double    UT1 as a 2-part Julian Date (see note)

        Returned (function value):
                     double    Earth rotation angle (radians), range 0-2pi

        Notes:

        1) The UT1 date dj1+dj2 is a Julian Date, apportioned in any
           convenient way between the arguments dj1 and dj2.  For example,
           JD(UT1)=2450123.7 could be expressed in any of these ways,
           among others:

                   dj1            dj2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 and MJD methods are good compromises
           between resolution and convenience.  The date & time method is
           best matched to the algorithm used:  maximum precision is
           delivered when the dj1 argument is for 0hrs UT1 on the day in
           question and the dj2 argument lies in the range 0 to 1, or vice
           versa.

        2) The algorithm is adapted from Expression 22 of Capitaine et al.
           2000.  The time argument has been expressed in days directly,
           and, to retain precision, integer contributions have been
           eliminated.  The same formulation is given in IERS Conventions
           (2003), Chap. 5, Eq. 14.

        Called:
           eraAnp       normalize angle into range 0 to 2pi

        References:

           Capitaine N., Guinot B. and McCarthy D.D, 2000, Astron.
           Astrophys., 355, 398-405.

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.era00(dj1, dj2)
    return c_retval


def gmst00(uta, utb, tta, ttb):
    """
    Greenwich mean sidereal time (model consistent with IAU 2000
    resolutions).

    Parameters
    ----------
    uta : double array
    utb : double array
    tta : double array
    ttb : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraGmst00``. The ERFA documentation is::

        - - - - - - - - - -
         e r a G m s t 0 0
        - - - - - - - - - -

        Greenwich mean sidereal time (model consistent with IAU 2000
        resolutions).

        Given:
           uta,utb    double    UT1 as a 2-part Julian Date (Notes 1,2)
           tta,ttb    double    TT as a 2-part Julian Date (Notes 1,2)

        Returned (function value):
                      double    Greenwich mean sidereal time (radians)

        Notes:

        1) The UT1 and TT dates uta+utb and tta+ttb respectively, are both
           Julian Dates, apportioned in any convenient way between the
           argument pairs.  For example, JD(UT1)=2450123.7 could be
           expressed in any of these ways, among others:

                  Part A         Part B

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable (in the case of UT;  the TT is not at all critical
           in this respect).  The J2000 and MJD methods are good compromises
           between resolution and convenience.  For UT, the date & time
           method is best matched to the algorithm that is used by the Earth
           Rotation Angle function, called internally:  maximum precision is
           delivered when the uta argument is for 0hrs UT1 on the day in
           question and the utb argument lies in the range 0 to 1, or vice
           versa.

        2) Both UT1 and TT are required, UT1 to predict the Earth rotation
           and TT to predict the effects of precession.  If UT1 is used for
           both purposes, errors of order 100 microarcseconds result.

        3) This GMST is compatible with the IAU 2000 resolutions and must be
           used only in conjunction with other IAU 2000 compatible
           components such as precession-nutation and equation of the
           equinoxes.

        4) The result is returned in the range 0 to 2pi.

        5) The algorithm is from Capitaine et al. (2003) and IERS
           Conventions 2003.

        Called:
           eraEra00     Earth rotation angle, IAU 2000
           eraAnp       normalize angle into range 0 to 2pi

        References:

           Capitaine, N., Wallace, P.T. and McCarthy, D.D., "Expressions to
           implement the IAU 2000 definition of UT1", Astronomy &
           Astrophysics, 406, 1135-1149 (2003)

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2020 November 17

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.gmst00(uta, utb, tta, ttb)
    return c_retval


def gmst06(uta, utb, tta, ttb):
    """
    Greenwich mean sidereal time (consistent with IAU 2006 precession).

    Parameters
    ----------
    uta : double array
    utb : double array
    tta : double array
    ttb : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraGmst06``. The ERFA documentation is::

        - - - - - - - - - -
         e r a G m s t 0 6
        - - - - - - - - - -

        Greenwich mean sidereal time (consistent with IAU 2006 precession).

        Given:
           uta,utb    double    UT1 as a 2-part Julian Date (Notes 1,2)
           tta,ttb    double    TT as a 2-part Julian Date (Notes 1,2)

        Returned (function value):
                      double    Greenwich mean sidereal time (radians)

        Notes:

        1) The UT1 and TT dates uta+utb and tta+ttb respectively, are both
           Julian Dates, apportioned in any convenient way between the
           argument pairs.  For example, JD=2450123.7 could be expressed in
           any of these ways, among others:

                  Part A        Part B

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable (in the case of UT;  the TT is not at all critical
           in this respect).  The J2000 and MJD methods are good compromises
           between resolution and convenience.  For UT, the date & time
           method is best matched to the algorithm that is used by the Earth
           rotation angle function, called internally:  maximum precision is
           delivered when the uta argument is for 0hrs UT1 on the day in
           question and the utb argument lies in the range 0 to 1, or vice
           versa.

        2) Both UT1 and TT are required, UT1 to predict the Earth rotation
           and TT to predict the effects of precession.  If UT1 is used for
           both purposes, errors of order 100 microarcseconds result.

        3) This GMST is compatible with the IAU 2006 precession and must not
           be used with other precession models.

        4) The result is returned in the range 0 to 2pi.

        Called:
           eraEra00     Earth rotation angle, IAU 2000
           eraAnp       normalize angle into range 0 to 2pi

        Reference:

           Capitaine, N., Wallace, P.T. & Chapront, J., 2005,
           Astron.Astrophys. 432, 355

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.gmst06(uta, utb, tta, ttb)
    return c_retval


def gmst82(dj1, dj2):
    """
    Universal Time to Greenwich mean sidereal time (IAU 1982 model).

    Parameters
    ----------
    dj1 : double array
    dj2 : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraGmst82``. The ERFA documentation is::

        - - - - - - - - - -
         e r a G m s t 8 2
        - - - - - - - - - -

        Universal Time to Greenwich mean sidereal time (IAU 1982 model).

        Given:
           dj1,dj2    double    UT1 Julian Date (see note)

        Returned (function value):
                      double    Greenwich mean sidereal time (radians)

        Notes:

        1) The UT1 date dj1+dj2 is a Julian Date, apportioned in any
           convenient way between the arguments dj1 and dj2.  For example,
           JD(UT1)=2450123.7 could be expressed in any of these ways,
           among others:

                   dj1            dj2

               2450123.7          0          (JD method)
                2451545        -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5         0.2         (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 and MJD methods are good compromises
           between resolution and convenience.  The date & time method is
           best matched to the algorithm used:  maximum accuracy (or, at
           least, minimum noise) is delivered when the dj1 argument is for
           0hrs UT1 on the day in question and the dj2 argument lies in the
           range 0 to 1, or vice versa.

        2) The algorithm is based on the IAU 1982 expression.  This is
           always described as giving the GMST at 0 hours UT1.  In fact, it
           gives the difference between the GMST and the UT, the steady
           4-minutes-per-day drawing-ahead of ST with respect to UT.  When
           whole days are ignored, the expression happens to equal the GMST
           at 0 hours UT1 each day.

        3) In this function, the entire UT1 (the sum of the two arguments
           dj1 and dj2) is used directly as the argument for the standard
           formula, the constant term of which is adjusted by 12 hours to
           take account of the noon phasing of Julian Date.  The UT1 is then
           added, but omitting whole days to conserve accuracy.

        Called:
           eraAnp       normalize angle into range 0 to 2pi

        References:

           Transactions of the International Astronomical Union,
           XVIII B, 67 (1983).

           Aoki et al., Astron.Astrophys., 105, 359-361 (1982).

        This revision:  2020 January 12

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.gmst82(dj1, dj2)
    return c_retval


def gst00a(uta, utb, tta, ttb):
    """
    Greenwich apparent sidereal time (consistent with IAU 2000
    resolutions).

    Parameters
    ----------
    uta : double array
    utb : double array
    tta : double array
    ttb : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraGst00a``. The ERFA documentation is::

        - - - - - - - - - -
         e r a G s t 0 0 a
        - - - - - - - - - -

        Greenwich apparent sidereal time (consistent with IAU 2000
        resolutions).

        Given:
           uta,utb    double    UT1 as a 2-part Julian Date (Notes 1,2)
           tta,ttb    double    TT as a 2-part Julian Date (Notes 1,2)

        Returned (function value):
                      double    Greenwich apparent sidereal time (radians)

        Notes:

        1) The UT1 and TT dates uta+utb and tta+ttb respectively, are both
           Julian Dates, apportioned in any convenient way between the
           argument pairs.  For example, JD(UT1)=2450123.7 could be
           expressed in any of these ways, among others:

                   uta            utb

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable (in the case of UT;  the TT is not at all critical
           in this respect).  The J2000 and MJD methods are good compromises
           between resolution and convenience.  For UT, the date & time
           method is best matched to the algorithm that is used by the Earth
           Rotation Angle function, called internally:  maximum precision is
           delivered when the uta argument is for 0hrs UT1 on the day in
           question and the utb argument lies in the range 0 to 1, or vice
           versa.

        2) Both UT1 and TT are required, UT1 to predict the Earth rotation
           and TT to predict the effects of precession-nutation.  If UT1 is
           used for both purposes, errors of order 100 microarcseconds
           result.

        3) This GAST is compatible with the IAU 2000 resolutions and must be
           used only in conjunction with other IAU 2000 compatible
           components such as precession-nutation.

        4) The result is returned in the range 0 to 2pi.

        5) The algorithm is from Capitaine et al. (2003) and IERS
           Conventions 2003.

        Called:
           eraGmst00    Greenwich mean sidereal time, IAU 2000
           eraEe00a     equation of the equinoxes, IAU 2000A
           eraAnp       normalize angle into range 0 to 2pi

        References:

           Capitaine, N., Wallace, P.T. and McCarthy, D.D., "Expressions to
           implement the IAU 2000 definition of UT1", Astronomy &
           Astrophysics, 406, 1135-1149 (2003)

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2020 November 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.gst00a(uta, utb, tta, ttb)
    return c_retval


def gst00b(uta, utb):
    """
    Greenwich apparent sidereal time (consistent with IAU 2000
    resolutions but using the truncated nutation model IAU 2000B).

    Parameters
    ----------
    uta : double array
    utb : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraGst00b``. The ERFA documentation is::

        - - - - - - - - - -
         e r a G s t 0 0 b
        - - - - - - - - - -

        Greenwich apparent sidereal time (consistent with IAU 2000
        resolutions but using the truncated nutation model IAU 2000B).

        Given:
           uta,utb    double    UT1 as a 2-part Julian Date (Notes 1,2)

        Returned (function value):
                      double    Greenwich apparent sidereal time (radians)

        Notes:

        1) The UT1 date uta+utb is a Julian Date, apportioned in any
           convenient way between the argument pair.  For example,
           JD(UT1)=2450123.7 could be expressed in any of these ways,
           among others:

                   uta            utb

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 and MJD methods are good compromises
           between resolution and convenience.  For UT, the date & time
           method is best matched to the algorithm that is used by the Earth
           Rotation Angle function, called internally:  maximum precision is
           delivered when the uta argument is for 0hrs UT1 on the day in
           question and the utb argument lies in the range 0 to 1, or vice
           versa.

        2) The result is compatible with the IAU 2000 resolutions, except
           that accuracy has been compromised for the sake of speed and
           convenience in two respects:

           . UT is used instead of TDB (or TT) to compute the precession
             component of GMST and the equation of the equinoxes.  This
             results in errors of order 0.1 mas at present.

           . The IAU 2000B abridged nutation model (McCarthy & Luzum, 2003)
             is used, introducing errors of up to 1 mas.

        3) This GAST is compatible with the IAU 2000 resolutions and must be
           used only in conjunction with other IAU 2000 compatible
           components such as precession-nutation.

        4) The result is returned in the range 0 to 2pi.

        5) The algorithm is from Capitaine et al. (2003) and IERS
           Conventions 2003.

        Called:
           eraGmst00    Greenwich mean sidereal time, IAU 2000
           eraEe00b     equation of the equinoxes, IAU 2000B
           eraAnp       normalize angle into range 0 to 2pi

        References:

           Capitaine, N., Wallace, P.T. and McCarthy, D.D., "Expressions to
           implement the IAU 2000 definition of UT1", Astronomy &
           Astrophysics, 406, 1135-1149 (2003)

           McCarthy, D.D. & Luzum, B.J., "An abridged model of the
           precession-nutation of the celestial pole", Celestial Mechanics &
           Dynamical Astronomy, 85, 37-49 (2003)

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

        This revision:  2020 November 19

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.gst00b(uta, utb)
    return c_retval


def gst06(uta, utb, tta, ttb, rnpb):
    """
    Greenwich apparent sidereal time, IAU 2006, given the NPB matrix.

    Parameters
    ----------
    uta : double array
    utb : double array
    tta : double array
    ttb : double array
    rnpb : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraGst06``. The ERFA documentation is::

        - - - - - - - - -
         e r a G s t 0 6
        - - - - - - - - -

        Greenwich apparent sidereal time, IAU 2006, given the NPB matrix.

        Given:
           uta,utb  double        UT1 as a 2-part Julian Date (Notes 1,2)
           tta,ttb  double        TT as a 2-part Julian Date (Notes 1,2)
           rnpb     double[3][3]  nutation x precession x bias matrix

        Returned (function value):
                    double        Greenwich apparent sidereal time (radians)

        Notes:

        1) The UT1 and TT dates uta+utb and tta+ttb respectively, are both
           Julian Dates, apportioned in any convenient way between the
           argument pairs.  For example, JD(UT1)=2450123.7 could be
           expressed in any of these ways, among others:

                   uta            utb

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable (in the case of UT;  the TT is not at all critical
           in this respect).  The J2000 and MJD methods are good compromises
           between resolution and convenience.  For UT, the date & time
           method is best matched to the algorithm that is used by the Earth
           rotation angle function, called internally:  maximum precision is
           delivered when the uta argument is for 0hrs UT1 on the day in
           question and the utb argument lies in the range 0 to 1, or vice
           versa.

        2) Both UT1 and TT are required, UT1 to predict the Earth rotation
           and TT to predict the effects of precession-nutation.  If UT1 is
           used for both purposes, errors of order 100 microarcseconds
           result.

        3) Although the function uses the IAU 2006 series for s+XY/2, it is
           otherwise independent of the precession-nutation model and can in
           practice be used with any equinox-based NPB matrix.

        4) The result is returned in the range 0 to 2pi.

        Called:
           eraBpn2xy    extract CIP X,Y coordinates from NPB matrix
           eraS06       the CIO locator s, given X,Y, IAU 2006
           eraAnp       normalize angle into range 0 to 2pi
           eraEra00     Earth rotation angle, IAU 2000
           eraEors      equation of the origins, given NPB matrix and s

        Reference:

           Wallace, P.T. & Capitaine, N., 2006, Astron.Astrophys. 459, 981

        This revision:  2020 November 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.gst06(uta, utb, tta, ttb, rnpb)
    return c_retval


def gst06a(uta, utb, tta, ttb):
    """
    Greenwich apparent sidereal time (consistent with IAU 2000 and 2006
    resolutions).

    Parameters
    ----------
    uta : double array
    utb : double array
    tta : double array
    ttb : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraGst06a``. The ERFA documentation is::

        - - - - - - - - - -
         e r a G s t 0 6 a
        - - - - - - - - - -

        Greenwich apparent sidereal time (consistent with IAU 2000 and 2006
        resolutions).

        Given:
           uta,utb    double    UT1 as a 2-part Julian Date (Notes 1,2)
           tta,ttb    double    TT as a 2-part Julian Date (Notes 1,2)

        Returned (function value):
                      double    Greenwich apparent sidereal time (radians)

        Notes:

        1) The UT1 and TT dates uta+utb and tta+ttb respectively, are both
           Julian Dates, apportioned in any convenient way between the
           argument pairs.  For example, JD(UT1)=2450123.7 could be
           expressed in any of these ways, among others:

                   uta            utb

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable (in the case of UT;  the TT is not at all critical
           in this respect).  The J2000 and MJD methods are good compromises
           between resolution and convenience.  For UT, the date & time
           method is best matched to the algorithm that is used by the Earth
           rotation angle function, called internally:  maximum precision is
           delivered when the uta argument is for 0hrs UT1 on the day in
           question and the utb argument lies in the range 0 to 1, or vice
           versa.

        2) Both UT1 and TT are required, UT1 to predict the Earth rotation
           and TT to predict the effects of precession-nutation.  If UT1 is
           used for both purposes, errors of order 100 microarcseconds
           result.

        3) This GAST is compatible with the IAU 2000/2006 resolutions and
           must be used only in conjunction with IAU 2006 precession and
           IAU 2000A nutation.

        4) The result is returned in the range 0 to 2pi.

        Called:
           eraPnm06a    classical NPB matrix, IAU 2006/2000A
           eraGst06     Greenwich apparent ST, IAU 2006, given NPB matrix

        Reference:

           Wallace, P.T. & Capitaine, N., 2006, Astron.Astrophys. 459, 981

        This revision:  2020 November 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.gst06a(uta, utb, tta, ttb)
    return c_retval


def gst94(uta, utb):
    """
    Greenwich apparent sidereal time (consistent with IAU 1982/94
    resolutions).

    Parameters
    ----------
    uta : double array
    utb : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraGst94``. The ERFA documentation is::

        - - - - - - - - -
         e r a G s t 9 4
        - - - - - - - - -

        Greenwich apparent sidereal time (consistent with IAU 1982/94
        resolutions).

        Given:
           uta,utb    double    UT1 as a 2-part Julian Date (Notes 1,2)

        Returned (function value):
                      double    Greenwich apparent sidereal time (radians)

        Notes:

        1) The UT1 date uta+utb is a Julian Date, apportioned in any
           convenient way between the argument pair.  For example,
           JD(UT1)=2450123.7 could be expressed in any of these ways, among
           others:

                   uta            utb

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in cases
           where the loss of several decimal digits of resolution is
           acceptable.  The J2000 and MJD methods are good compromises
           between resolution and convenience.  For UT, the date & time
           method is best matched to the algorithm that is used by the Earth
           Rotation Angle function, called internally:  maximum precision is
           delivered when the uta argument is for 0hrs UT1 on the day in
           question and the utb argument lies in the range 0 to 1, or vice
           versa.

        2) The result is compatible with the IAU 1982 and 1994 resolutions,
           except that accuracy has been compromised for the sake of
           convenience in that UT is used instead of TDB (or TT) to compute
           the equation of the equinoxes.

        3) This GAST must be used only in conjunction with contemporaneous
           IAU standards such as 1976 precession, 1980 obliquity and 1982
           nutation.  It is not compatible with the IAU 2000 resolutions.

        4) The result is returned in the range 0 to 2pi.

        Called:
           eraGmst82    Greenwich mean sidereal time, IAU 1982
           eraEqeq94    equation of the equinoxes, IAU 1994
           eraAnp       normalize angle into range 0 to 2pi

        References:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992)

           IAU Resolution C7, Recommendation 3 (1994)

        This revision:  2020 November 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.gst94(uta, utb)
    return c_retval


def pvstar(pv):
    """
    Convert star position+velocity vector to catalog coordinates.

    Parameters
    ----------
    pv : double array

    Returns
    -------
    ra : double array
    dec : double array
    pmr : double array
    pmd : double array
    px : double array
    rv : double array

    Notes
    -----
    Wraps ERFA function ``eraPvstar``. The ERFA documentation is::

        - - - - - - - - - -
         e r a P v s t a r
        - - - - - - - - - -

        Convert star position+velocity vector to catalog coordinates.

        Given (Note 1):
           pv     double[2][3]   pv-vector (au, au/day)

        Returned (Note 2):
           ra     double         right ascension (radians)
           dec    double         declination (radians)
           pmr    double         RA proper motion (radians/year)
           pmd    double         Dec proper motion (radians/year)
           px     double         parallax (arcsec)
           rv     double         radial velocity (km/s, positive = receding)

        Returned (function value):
                  int            status:
                                    0 = OK
                                   -1 = superluminal speed (Note 5)
                                   -2 = null position vector

        Notes:

        1) The specified pv-vector is the coordinate direction (and its rate
           of change) for the date at which the light leaving the star
           reached the solar-system barycenter.

        2) The star data returned by this function are "observables" for an
           imaginary observer at the solar-system barycenter.  Proper motion
           and radial velocity are, strictly, in terms of barycentric
           coordinate time, TCB.  For most practical applications, it is
           permissible to neglect the distinction between TCB and ordinary
           "proper" time on Earth (TT/TAI).  The result will, as a rule, be
           limited by the intrinsic accuracy of the proper-motion and
           radial-velocity data;  moreover, the supplied pv-vector is likely
           to be merely an intermediate result (for example generated by the
           function eraStarpv), so that a change of time unit will cancel
           out overall.

           In accordance with normal star-catalog conventions, the object's
           right ascension and declination are freed from the effects of
           secular aberration.  The frame, which is aligned to the catalog
           equator and equinox, is Lorentzian and centered on the SSB.

           Summarizing, the specified pv-vector is for most stars almost
           identical to the result of applying the standard geometrical
           "space motion" transformation to the catalog data.  The
           differences, which are the subject of the Stumpff paper cited
           below, are:

           (i) In stars with significant radial velocity and proper motion,
           the constantly changing light-time distorts the apparent proper
           motion.  Note that this is a classical, not a relativistic,
           effect.

           (ii) The transformation complies with special relativity.

        3) Care is needed with units.  The star coordinates are in radians
           and the proper motions in radians per Julian year, but the
           parallax is in arcseconds; the radial velocity is in km/s, but
           the pv-vector result is in au and au/day.

        4) The proper motions are the rate of change of the right ascension
           and declination at the catalog epoch and are in radians per Julian
           year.  The RA proper motion is in terms of coordinate angle, not
           true angle, and will thus be numerically larger at high
           declinations.

        5) Straight-line motion at constant speed in the inertial frame is
           assumed.  If the speed is greater than or equal to the speed of
           light, the function aborts with an error status.

        6) The inverse transformation is performed by the function eraStarpv.

        Called:
           eraPn        decompose p-vector into modulus and direction
           eraPdp       scalar product of two p-vectors
           eraSxp       multiply p-vector by scalar
           eraPmp       p-vector minus p-vector
           eraPm        modulus of p-vector
           eraPpp       p-vector plus p-vector
           eraPv2s      pv-vector to spherical
           eraAnp       normalize angle into range 0 to 2pi

        Reference:

           Stumpff, P., 1985, Astron.Astrophys. 144, 232-240.

        This revision:  2017 March 16

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    ra, dec, pmr, pmd, px, rv, c_retval = ufunc.pvstar(pv)
    check_errwarn(c_retval, 'pvstar')
    return ra, dec, pmr, pmd, px, rv


STATUS_CODES['pvstar'] = {
    0: 'OK',
    -1: 'superluminal speed (Note 5)',
    -2: 'null position vector',
}


def starpv(ra, dec, pmr, pmd, px, rv):
    """
    Convert star catalog coordinates to position+velocity vector.

    Parameters
    ----------
    ra : double array
    dec : double array
    pmr : double array
    pmd : double array
    px : double array
    rv : double array

    Returns
    -------
    pv : double array

    Notes
    -----
    Wraps ERFA function ``eraStarpv``. The ERFA documentation is::

        - - - - - - - - - -
         e r a S t a r p v
        - - - - - - - - - -

        Convert star catalog coordinates to position+velocity vector.

        Given (Note 1):
           ra     double        right ascension (radians)
           dec    double        declination (radians)
           pmr    double        RA proper motion (radians/year)
           pmd    double        Dec proper motion (radians/year)
           px     double        parallax (arcseconds)
           rv     double        radial velocity (km/s, positive = receding)

        Returned (Note 2):
           pv     double[2][3]  pv-vector (au, au/day)

        Returned (function value):
                  int           status:
                                    0 = no warnings
                                    1 = distance overridden (Note 6)
                                    2 = excessive speed (Note 7)
                                    4 = solution didn't converge (Note 8)
                                 else = binary logical OR of the above

        Notes:

        1) The star data accepted by this function are "observables" for an
           imaginary observer at the solar-system barycenter.  Proper motion
           and radial velocity are, strictly, in terms of barycentric
           coordinate time, TCB.  For most practical applications, it is
           permissible to neglect the distinction between TCB and ordinary
           "proper" time on Earth (TT/TAI).  The result will, as a rule, be
           limited by the intrinsic accuracy of the proper-motion and
           radial-velocity data;  moreover, the pv-vector is likely to be
           merely an intermediate result, so that a change of time unit
           would cancel out overall.

           In accordance with normal star-catalog conventions, the object's
           right ascension and declination are freed from the effects of
           secular aberration.  The frame, which is aligned to the catalog
           equator and equinox, is Lorentzian and centered on the SSB.

        2) The resulting position and velocity pv-vector is with respect to
           the same frame and, like the catalog coordinates, is freed from
           the effects of secular aberration.  Should the "coordinate
           direction", where the object was located at the catalog epoch, be
           required, it may be obtained by calculating the magnitude of the
           position vector pv[0][0-2] dividing by the speed of light in
           au/day to give the light-time, and then multiplying the space
           velocity pv[1][0-2] by this light-time and adding the result to
           pv[0][0-2].

           Summarizing, the pv-vector returned is for most stars almost
           identical to the result of applying the standard geometrical
           "space motion" transformation.  The differences, which are the
           subject of the Stumpff paper referenced below, are:

           (i) In stars with significant radial velocity and proper motion,
           the constantly changing light-time distorts the apparent proper
           motion.  Note that this is a classical, not a relativistic,
           effect.

           (ii) The transformation complies with special relativity.

        3) Care is needed with units.  The star coordinates are in radians
           and the proper motions in radians per Julian year, but the
           parallax is in arcseconds; the radial velocity is in km/s, but
           the pv-vector result is in au and au/day.

        4) The RA proper motion is in terms of coordinate angle, not true
           angle.  If the catalog uses arcseconds for both RA and Dec proper
           motions, the RA proper motion will need to be divided by cos(Dec)
           before use.

        5) Straight-line motion at constant speed, in the inertial frame,
           is assumed.

        6) An extremely small (or zero or negative) parallax is interpreted
           to mean that the object is on the "celestial sphere", the radius
           of which is an arbitrary (large) value (see the constant PXMIN).
           When the distance is overridden in this way, the status,
           initially zero, has 1 added to it.

        7) If the space velocity is a significant fraction of c (see the
           constant VMAX), it is arbitrarily set to zero.  When this action
           occurs, 2 is added to the status.

        8) The relativistic adjustment involves an iterative calculation.
           If the process fails to converge within a set number (IMAX) of
           iterations, 4 is added to the status.

        9) The inverse transformation is performed by the function
           eraPvstar.

        Called:
           eraS2pv      spherical coordinates to pv-vector
           eraPm        modulus of p-vector
           eraZp        zero p-vector
           eraPn        decompose p-vector into modulus and direction
           eraPdp       scalar product of two p-vectors
           eraSxp       multiply p-vector by scalar
           eraPmp       p-vector minus p-vector
           eraPpp       p-vector plus p-vector

        Reference:

           Stumpff, P., 1985, Astron.Astrophys. 144, 232-240.

        This revision:  2017 March 16

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    pv, c_retval = ufunc.starpv(ra, dec, pmr, pmd, px, rv)
    check_errwarn(c_retval, 'starpv')
    return pv


STATUS_CODES['starpv'] = {
    0: 'no warnings',
    1: 'distance overridden (Note 6)',
    2: 'excessive speed (Note 7)',
    4: "solution didn't converge (Note 8)",
    'else': 'binary logical OR of the above',
}


def fk425(r1950, d1950, dr1950, dd1950, p1950, v1950):
    """
    Convert B1950.0 FK4 star catalog data to J2000.0 FK5.

    Parameters
    ----------
    r1950 : double array
    d1950 : double array
    dr1950 : double array
    dd1950 : double array
    p1950 : double array
    v1950 : double array

    Returns
    -------
    r2000 : double array
    d2000 : double array
    dr2000 : double array
    dd2000 : double array
    p2000 : double array
    v2000 : double array

    Notes
    -----
    Wraps ERFA function ``eraFk425``. The ERFA documentation is::

        - - - - - - - - -
         e r a F k 4 2 5
        - - - - - - - - -

        Convert B1950.0 FK4 star catalog data to J2000.0 FK5.

        This function converts a star's catalog data from the old FK4
       (Bessel-Newcomb) system to the later IAU 1976 FK5 (Fricke) system.

        Given: (all B1950.0, FK4)
           r1950,d1950    double   B1950.0 RA,Dec (rad)
           dr1950,dd1950  double   B1950.0 proper motions (rad/trop.yr)
           p1950          double   parallax (arcsec)
           v1950          double   radial velocity (km/s, +ve = moving away)

        Returned: (all J2000.0, FK5)
           r2000,d2000    double   J2000.0 RA,Dec (rad)
           dr2000,dd2000  double   J2000.0 proper motions (rad/Jul.yr)
           p2000          double   parallax (arcsec)
           v2000          double   radial velocity (km/s, +ve = moving away)

        Notes:

        1) The proper motions in RA are dRA/dt rather than cos(Dec)*dRA/dt,
           and are per year rather than per century.

        2) The conversion is somewhat complicated, for several reasons:

           . Change of standard epoch from B1950.0 to J2000.0.

           . An intermediate transition date of 1984 January 1.0 TT.

           . A change of precession model.

           . Change of time unit for proper motion (tropical to Julian).

           . FK4 positions include the E-terms of aberration, to simplify
             the hand computation of annual aberration.  FK5 positions
             assume a rigorous aberration computation based on the Earth's
             barycentric velocity.

           . The E-terms also affect proper motions, and in particular cause
             objects at large distances to exhibit fictitious proper
             motions.

           The algorithm is based on Smith et al. (1989) and Yallop et al.
           (1989), which presented a matrix method due to Standish (1982) as
           developed by Aoki et al. (1983), using Kinoshita's development of
           Andoyer's post-Newcomb precession.  The numerical constants from
           Seidelmann (1992) are used canonically.

        3) Conversion from B1950.0 FK4 to J2000.0 FK5 only is provided for.
           Conversions for different epochs and equinoxes would require
           additional treatment for precession, proper motion and E-terms.

        4) In the FK4 catalog the proper motions of stars within 10 degrees
           of the poles do not embody differential E-terms effects and
           should, strictly speaking, be handled in a different manner from
           stars outside these regions.  However, given the general lack of
           homogeneity of the star data available for routine astrometry,
           the difficulties of handling positions that may have been
           determined from astrometric fields spanning the polar and non-
           polar regions, the likelihood that the differential E-terms
           effect was not taken into account when allowing for proper motion
           in past astrometry, and the undesirability of a discontinuity in
           the algorithm, the decision has been made in this ERFA algorithm
           to include the effects of differential E-terms on the proper
           motions for all stars, whether polar or not.  At epoch J2000.0,
           and measuring "on the sky" rather than in terms of RA change, the
           errors resulting from this simplification are less than
           1 milliarcsecond in position and 1 milliarcsecond per century in
           proper motion.

        Called:
           eraAnp       normalize angle into range 0 to 2pi
           eraPv2s      pv-vector to spherical coordinates
           eraPdp       scalar product of two p-vectors
           eraPvmpv     pv-vector minus pv_vector
           eraPvppv     pv-vector plus pv_vector
           eraS2pv      spherical coordinates to pv-vector
           eraSxp       multiply p-vector by scalar

        References:

           Aoki, S. et al., 1983, "Conversion matrix of epoch B1950.0
           FK4-based positions of stars to epoch J2000.0 positions in
           accordance with the new IAU resolutions".  Astron.Astrophys.
           128, 263-267.

           Seidelmann, P.K. (ed), 1992, "Explanatory Supplement to the
           Astronomical Almanac", ISBN 0-935702-68-7.

           Smith, C.A. et al., 1989, "The transformation of astrometric
           catalog systems to the equinox J2000.0".  Astron.J. 97, 265.

           Standish, E.M., 1982, "Conversion of positions and proper motions
           from B1950.0 to the IAU system at J2000.0".  Astron.Astrophys.,
           115, 1, 20-22.

           Yallop, B.D. et al., 1989, "Transformation of mean star places
           from FK4 B1950.0 to FK5 J2000.0 using matrices in 6-space".
           Astron.J. 97, 274.

        This revision:   2018 December 5

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r2000, d2000, dr2000, dd2000, p2000, v2000 = ufunc.fk425(
        r1950, d1950, dr1950, dd1950, p1950, v1950)
    return r2000, d2000, dr2000, dd2000, p2000, v2000


def fk45z(r1950, d1950, bepoch):
    """
    Convert a B1950.0 FK4 star position to J2000.0 FK5, assuming zero
    proper motion in the FK5 system.

    Parameters
    ----------
    r1950 : double array
    d1950 : double array
    bepoch : double array

    Returns
    -------
    r2000 : double array
    d2000 : double array

    Notes
    -----
    Wraps ERFA function ``eraFk45z``. The ERFA documentation is::

        - - - - - - - - -
         e r a F k 4 5 z
        - - - - - - - - -

        Convert a B1950.0 FK4 star position to J2000.0 FK5, assuming zero
        proper motion in the FK5 system.

        This function converts a star's catalog data from the old FK4
        (Bessel-Newcomb) system to the later IAU 1976 FK5 (Fricke) system,
        in such a way that the FK5 proper motion is zero.  Because such a
        star has, in general, a non-zero proper motion in the FK4 system,
        the function requires the epoch at which the position in the FK4
        system was determined.

        Given:
           r1950,d1950    double   B1950.0 FK4 RA,Dec at epoch (rad)
           bepoch         double   Besselian epoch (e.g. 1979.3)

        Returned:
           r2000,d2000    double   J2000.0 FK5 RA,Dec (rad)

        Notes:

        1) The epoch bepoch is strictly speaking Besselian, but if a
           Julian epoch is supplied the result will be affected only to a
           negligible extent.

        2) The method is from Appendix 2 of Aoki et al. (1983), but using
           the constants of Seidelmann (1992).  See the function eraFk425
           for a general introduction to the FK4 to FK5 conversion.

        3) Conversion from equinox B1950.0 FK4 to equinox J2000.0 FK5 only
           is provided for.  Conversions for different starting and/or
           ending epochs would require additional treatment for precession,
           proper motion and E-terms.

        4) In the FK4 catalog the proper motions of stars within 10 degrees
           of the poles do not embody differential E-terms effects and
           should, strictly speaking, be handled in a different manner from
           stars outside these regions.  However, given the general lack of
           homogeneity of the star data available for routine astrometry,
           the difficulties of handling positions that may have been
           determined from astrometric fields spanning the polar and non-
           polar regions, the likelihood that the differential E-terms
           effect was not taken into account when allowing for proper motion
           in past astrometry, and the undesirability of a discontinuity in
           the algorithm, the decision has been made in this ERFA algorithm
           to include the effects of differential E-terms on the proper
           motions for all stars, whether polar or not.  At epoch 2000.0,
           and measuring "on the sky" rather than in terms of RA change, the
           errors resulting from this simplification are less than
           1 milliarcsecond in position and 1 milliarcsecond per century in
           proper motion.

        References:

           Aoki, S. et al., 1983, "Conversion matrix of epoch B1950.0
           FK4-based positions of stars to epoch J2000.0 positions in
           accordance with the new IAU resolutions".  Astron.Astrophys.
           128, 263-267.

           Seidelmann, P.K. (ed), 1992, "Explanatory Supplement to the
           Astronomical Almanac", ISBN 0-935702-68-7.

        Called:
           eraAnp       normalize angle into range 0 to 2pi
           eraC2s       p-vector to spherical
           eraEpb2jd    Besselian epoch to Julian date
           eraEpj       Julian date to Julian epoch
           eraPdp       scalar product of two p-vectors
           eraPmp       p-vector minus p-vector
           eraPpsp      p-vector plus scaled p-vector
           eraPvu       update a pv-vector
           eraS2c       spherical to p-vector

        This revision:   2020 November 19

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r2000, d2000 = ufunc.fk45z(r1950, d1950, bepoch)
    return r2000, d2000


def fk524(r2000, d2000, dr2000, dd2000, p2000, v2000):
    """
    Convert J2000.0 FK5 star catalog data to B1950.0 FK4.

    Parameters
    ----------
    r2000 : double array
    d2000 : double array
    dr2000 : double array
    dd2000 : double array
    p2000 : double array
    v2000 : double array

    Returns
    -------
    r1950 : double array
    d1950 : double array
    dr1950 : double array
    dd1950 : double array
    p1950 : double array
    v1950 : double array

    Notes
    -----
    Wraps ERFA function ``eraFk524``. The ERFA documentation is::

        - - - - - - - - -
         e r a F k 5 2 4
        - - - - - - - - -

        Convert J2000.0 FK5 star catalog data to B1950.0 FK4.

        Given: (all J2000.0, FK5)
           r2000,d2000    double   J2000.0 RA,Dec (rad)
           dr2000,dd2000  double   J2000.0 proper motions (rad/Jul.yr)
           p2000          double   parallax (arcsec)
           v2000          double   radial velocity (km/s, +ve = moving away)

        Returned: (all B1950.0, FK4)
           r1950,d1950    double   B1950.0 RA,Dec (rad)
           dr1950,dd1950  double   B1950.0 proper motions (rad/trop.yr)
           p1950          double   parallax (arcsec)
           v1950          double   radial velocity (km/s, +ve = moving away)

        Notes:

        1) The proper motions in RA are dRA/dt rather than cos(Dec)*dRA/dt,
           and are per year rather than per century.

        2) The conversion is somewhat complicated, for several reasons:

           . Change of standard epoch from J2000.0 to B1950.0.

           . An intermediate transition date of 1984 January 1.0 TT.

           . A change of precession model.

           . Change of time unit for proper motion (Julian to tropical).

           . FK4 positions include the E-terms of aberration, to simplify
             the hand computation of annual aberration.  FK5 positions
             assume a rigorous aberration computation based on the Earth's
             barycentric velocity.

           . The E-terms also affect proper motions, and in particular cause
             objects at large distances to exhibit fictitious proper
             motions.

           The algorithm is based on Smith et al. (1989) and Yallop et al.
           (1989), which presented a matrix method due to Standish (1982) as
           developed by Aoki et al. (1983), using Kinoshita's development of
           Andoyer's post-Newcomb precession.  The numerical constants from
           Seidelmann (1992) are used canonically.

        4) In the FK4 catalog the proper motions of stars within 10 degrees
           of the poles do not embody differential E-terms effects and
           should, strictly speaking, be handled in a different manner from
           stars outside these regions.  However, given the general lack of
           homogeneity of the star data available for routine astrometry,
           the difficulties of handling positions that may have been
           determined from astrometric fields spanning the polar and non-
           polar regions, the likelihood that the differential E-terms
           effect was not taken into account when allowing for proper motion
           in past astrometry, and the undesirability of a discontinuity in
           the algorithm, the decision has been made in this ERFA algorithm
           to include the effects of differential E-terms on the proper
           motions for all stars, whether polar or not.  At epoch J2000.0,
           and measuring "on the sky" rather than in terms of RA change, the
           errors resulting from this simplification are less than
           1 milliarcsecond in position and 1 milliarcsecond per century in
           proper motion.

        Called:
           eraAnp       normalize angle into range 0 to 2pi
           eraPdp       scalar product of two p-vectors
           eraPm        modulus of p-vector
           eraPmp       p-vector minus p-vector
           eraPpp       p-vector pluus p-vector
           eraPv2s      pv-vector to spherical coordinates
           eraS2pv      spherical coordinates to pv-vector
           eraSxp       multiply p-vector by scalar

        References:

           Aoki, S. et al., 1983, "Conversion matrix of epoch B1950.0
           FK4-based positions of stars to epoch J2000.0 positions in
           accordance with the new IAU resolutions".  Astron.Astrophys.
           128, 263-267.

           Seidelmann, P.K. (ed), 1992, "Explanatory Supplement to the
           Astronomical Almanac", ISBN 0-935702-68-7.

           Smith, C.A. et al., 1989, "The transformation of astrometric
           catalog systems to the equinox J2000.0".  Astron.J. 97, 265.

           Standish, E.M., 1982, "Conversion of positions and proper motions
           from B1950.0 to the IAU system at J2000.0".  Astron.Astrophys.,
           115, 1, 20-22.

           Yallop, B.D. et al., 1989, "Transformation of mean star places
           from FK4 B1950.0 to FK5 J2000.0 using matrices in 6-space".
           Astron.J. 97, 274.

        This revision:   2019 October 3

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r1950, d1950, dr1950, dd1950, p1950, v1950 = ufunc.fk524(
        r2000, d2000, dr2000, dd2000, p2000, v2000)
    return r1950, d1950, dr1950, dd1950, p1950, v1950


def fk52h(r5, d5, dr5, dd5, px5, rv5):
    """
    Transform FK5 (J2000.0) star data into the Hipparcos system.

    Parameters
    ----------
    r5 : double array
    d5 : double array
    dr5 : double array
    dd5 : double array
    px5 : double array
    rv5 : double array

    Returns
    -------
    rh : double array
    dh : double array
    drh : double array
    ddh : double array
    pxh : double array
    rvh : double array

    Notes
    -----
    Wraps ERFA function ``eraFk52h``. The ERFA documentation is::

        - - - - - - - - -
         e r a F k 5 2 h
        - - - - - - - - -

        Transform FK5 (J2000.0) star data into the Hipparcos system.

        Given (all FK5, equinox J2000.0, epoch J2000.0):
           r5      double    RA (radians)
           d5      double    Dec (radians)
           dr5     double    proper motion in RA (dRA/dt, rad/Jyear)
           dd5     double    proper motion in Dec (dDec/dt, rad/Jyear)
           px5     double    parallax (arcsec)
           rv5     double    radial velocity (km/s, positive = receding)

        Returned (all Hipparcos, epoch J2000.0):
           rh      double    RA (radians)
           dh      double    Dec (radians)
           drh     double    proper motion in RA (dRA/dt, rad/Jyear)
           ddh     double    proper motion in Dec (dDec/dt, rad/Jyear)
           pxh     double    parallax (arcsec)
           rvh     double    radial velocity (km/s, positive = receding)

        Notes:

        1) This function transforms FK5 star positions and proper motions
           into the system of the Hipparcos catalog.

        2) The proper motions in RA are dRA/dt rather than
           cos(Dec)*dRA/dt, and are per year rather than per century.

        3) The FK5 to Hipparcos transformation is modeled as a pure
           rotation and spin;  zonal errors in the FK5 catalog are not
           taken into account.

        4) See also eraH2fk5, eraFk5hz, eraHfk5z.

        Called:
           eraStarpv    star catalog data to space motion pv-vector
           eraFk5hip    FK5 to Hipparcos rotation and spin
           eraRxp       product of r-matrix and p-vector
           eraPxp       vector product of two p-vectors
           eraPpp       p-vector plus p-vector
           eraPvstar    space motion pv-vector to star catalog data

        Reference:

           F.Mignard & M.Froeschle, Astron.Astrophys., 354, 732-739 (2000).

        This revision:  2017 October 12

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rh, dh, drh, ddh, pxh, rvh = ufunc.fk52h(r5, d5, dr5, dd5, px5, rv5)
    return rh, dh, drh, ddh, pxh, rvh


def fk54z(r2000, d2000, bepoch):
    """
    Convert a J2000.0 FK5 star position to B1950.0 FK4, assuming zero
    proper motion in FK5 and parallax.

    Parameters
    ----------
    r2000 : double array
    d2000 : double array
    bepoch : double array

    Returns
    -------
    r1950 : double array
    d1950 : double array
    dr1950 : double array
    dd1950 : double array

    Notes
    -----
    Wraps ERFA function ``eraFk54z``. The ERFA documentation is::

        - - - - - - - - -
         e r a F k 5 4 z
        - - - - - - - - -

        Convert a J2000.0 FK5 star position to B1950.0 FK4, assuming zero
        proper motion in FK5 and parallax.

        Given:
           r2000,d2000    double   J2000.0 FK5 RA,Dec (rad)
           bepoch         double   Besselian epoch (e.g. 1950.0)

        Returned:
           r1950,d1950    double   B1950.0 FK4 RA,Dec (rad) at epoch BEPOCH
           dr1950,dd1950  double   B1950.0 FK4 proper motions (rad/trop.yr)

        Notes:

        1) In contrast to the eraFk524 function, here the FK5 proper
           motions, the parallax and the radial velocity are presumed zero.

        2) This function converts a star position from the IAU 1976 FK5
          (Fricke) system to the former FK4 (Bessel-Newcomb) system, for
           cases such as distant radio sources where it is presumed there is
           zero parallax and no proper motion.  Because of the E-terms of
           aberration, such objects have (in general) non-zero proper motion
           in FK4, and the present function returns those fictitious proper
           motions.

        3) Conversion from B1950.0 FK4 to J2000.0 FK5 only is provided for.
           Conversions involving other equinoxes would require additional
           treatment for precession.

        4) The position returned by this function is in the B1950.0 FK4
           reference system but at Besselian epoch BEPOCH.  For comparison
           with catalogs the BEPOCH argument will frequently be 1950.0. (In
           this context the distinction between Besselian and Julian epoch
           is insignificant.)

        5) The RA component of the returned (fictitious) proper motion is
           dRA/dt rather than cos(Dec)*dRA/dt.

        Called:
           eraAnp       normalize angle into range 0 to 2pi
           eraC2s       p-vector to spherical
           eraFk524     FK4 to FK5
           eraS2c       spherical to p-vector

        This revision:   2020 November 19

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r1950, d1950, dr1950, dd1950 = ufunc.fk54z(r2000, d2000, bepoch)
    return r1950, d1950, dr1950, dd1950


def fk5hip():
    """
    FK5 to Hipparcos rotation and spin.

    Parameters
    ----------

    Returns
    -------
    r5h : double array
    s5h : double array

    Notes
    -----
    Wraps ERFA function ``eraFk5hip``. The ERFA documentation is::

        - - - - - - - - - -
         e r a F k 5 h i p
        - - - - - - - - - -

        FK5 to Hipparcos rotation and spin.

        Returned:
           r5h   double[3][3]  r-matrix: FK5 rotation wrt Hipparcos (Note 2)
           s5h   double[3]     r-vector: FK5 spin wrt Hipparcos (Note 3)

        Notes:

        1) This function models the FK5 to Hipparcos transformation as a
           pure rotation and spin;  zonal errors in the FK5 catalogue are
           not taken into account.

        2) The r-matrix r5h operates in the sense:

                 P_Hipparcos = r5h x P_FK5

           where P_FK5 is a p-vector in the FK5 frame, and P_Hipparcos is
           the equivalent Hipparcos p-vector.

        3) The r-vector s5h represents the time derivative of the FK5 to
           Hipparcos rotation.  The units are radians per year (Julian,
           TDB).

        Called:
           eraRv2m      r-vector to r-matrix

        Reference:

           F.Mignard & M.Froeschle, Astron.Astrophys., 354, 732-739 (2000).

        This revision:  2017 October 12

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r5h, s5h = ufunc.fk5hip()
    return r5h, s5h


def fk5hz(r5, d5, date1, date2):
    """
    Transform an FK5 (J2000.0) star position into the system of the
    Hipparcos catalogue, assuming zero Hipparcos proper motion.

    Parameters
    ----------
    r5 : double array
    d5 : double array
    date1 : double array
    date2 : double array

    Returns
    -------
    rh : double array
    dh : double array

    Notes
    -----
    Wraps ERFA function ``eraFk5hz``. The ERFA documentation is::

        - - - - - - - - -
         e r a F k 5 h z
        - - - - - - - - -

        Transform an FK5 (J2000.0) star position into the system of the
        Hipparcos catalogue, assuming zero Hipparcos proper motion.

        Given:
           r5           double   FK5 RA (radians), equinox J2000.0, at date
           d5           double   FK5 Dec (radians), equinox J2000.0, at date
           date1,date2  double   TDB date (Notes 1,2)

        Returned:
           rh           double   Hipparcos RA (radians)
           dh           double   Hipparcos Dec (radians)

        Notes:

        1) This function converts a star position from the FK5 system to
           the Hipparcos system, in such a way that the Hipparcos proper
           motion is zero.  Because such a star has, in general, a non-zero
           proper motion in the FK5 system, the function requires the date
           at which the position in the FK5 system was determined.

        2) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        3) The FK5 to Hipparcos transformation is modeled as a pure
           rotation and spin;  zonal errors in the FK5 catalogue are not
           taken into account.

        4) The position returned by this function is in the Hipparcos
           reference system but at date date1+date2.

        5) See also eraFk52h, eraH2fk5, eraHfk5z.

        Called:
           eraS2c       spherical coordinates to unit vector
           eraFk5hip    FK5 to Hipparcos rotation and spin
           eraSxp       multiply p-vector by scalar
           eraRv2m      r-vector to r-matrix
           eraTrxp      product of transpose of r-matrix and p-vector
           eraPxp       vector product of two p-vectors
           eraC2s       p-vector to spherical
           eraAnp       normalize angle into range 0 to 2pi

        Reference:

           F.Mignard & M.Froeschle, 2000, Astron.Astrophys. 354, 732-739.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rh, dh = ufunc.fk5hz(r5, d5, date1, date2)
    return rh, dh


def h2fk5(rh, dh, drh, ddh, pxh, rvh):
    """
    Transform Hipparcos star data into the FK5 (J2000.0) system.

    Parameters
    ----------
    rh : double array
    dh : double array
    drh : double array
    ddh : double array
    pxh : double array
    rvh : double array

    Returns
    -------
    r5 : double array
    d5 : double array
    dr5 : double array
    dd5 : double array
    px5 : double array
    rv5 : double array

    Notes
    -----
    Wraps ERFA function ``eraH2fk5``. The ERFA documentation is::

        - - - - - - - - -
         e r a H 2 f k 5
        - - - - - - - - -

        Transform Hipparcos star data into the FK5 (J2000.0) system.

        Given (all Hipparcos, epoch J2000.0):
           rh      double    RA (radians)
           dh      double    Dec (radians)
           drh     double    proper motion in RA (dRA/dt, rad/Jyear)
           ddh     double    proper motion in Dec (dDec/dt, rad/Jyear)
           pxh     double    parallax (arcsec)
           rvh     double    radial velocity (km/s, positive = receding)

        Returned (all FK5, equinox J2000.0, epoch J2000.0):
           r5      double    RA (radians)
           d5      double    Dec (radians)
           dr5     double    proper motion in RA (dRA/dt, rad/Jyear)
           dd5     double    proper motion in Dec (dDec/dt, rad/Jyear)
           px5     double    parallax (arcsec)
           rv5     double    radial velocity (km/s, positive = receding)

        Notes:

        1) This function transforms Hipparcos star positions and proper
           motions into FK5 J2000.0.

        2) The proper motions in RA are dRA/dt rather than
           cos(Dec)*dRA/dt, and are per year rather than per century.

        3) The FK5 to Hipparcos transformation is modeled as a pure
           rotation and spin;  zonal errors in the FK5 catalog are not
           taken into account.

        4) See also eraFk52h, eraFk5hz, eraHfk5z.

        Called:
           eraStarpv    star catalog data to space motion pv-vector
           eraFk5hip    FK5 to Hipparcos rotation and spin
           eraRv2m      r-vector to r-matrix
           eraRxp       product of r-matrix and p-vector
           eraTrxp      product of transpose of r-matrix and p-vector
           eraPxp       vector product of two p-vectors
           eraPmp       p-vector minus p-vector
           eraPvstar    space motion pv-vector to star catalog data

        Reference:

           F.Mignard & M.Froeschle, Astron.Astrophys., 354, 732-739 (2000).

        This revision:  2017 October 12

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r5, d5, dr5, dd5, px5, rv5 = ufunc.h2fk5(rh, dh, drh, ddh, pxh, rvh)
    return r5, d5, dr5, dd5, px5, rv5


def hfk5z(rh, dh, date1, date2):
    """
    Transform a Hipparcos star position into FK5 J2000.0, assuming
    zero Hipparcos proper motion.

    Parameters
    ----------
    rh : double array
    dh : double array
    date1 : double array
    date2 : double array

    Returns
    -------
    r5 : double array
    d5 : double array
    dr5 : double array
    dd5 : double array

    Notes
    -----
    Wraps ERFA function ``eraHfk5z``. The ERFA documentation is::

        - - - - - - - - -
         e r a H f k 5 z
        - - - - - - - - -

        Transform a Hipparcos star position into FK5 J2000.0, assuming
        zero Hipparcos proper motion.

        Given:
           rh            double    Hipparcos RA (radians)
           dh            double    Hipparcos Dec (radians)
           date1,date2   double    TDB date (Note 1)

        Returned (all FK5, equinox J2000.0, date date1+date2):
           r5            double    RA (radians)
           d5            double    Dec (radians)
           dr5           double    FK5 RA proper motion (rad/year, Note 4)
           dd5           double    Dec proper motion (rad/year, Note 4)

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) The proper motion in RA is dRA/dt rather than cos(Dec)*dRA/dt.

        3) The FK5 to Hipparcos transformation is modeled as a pure rotation
           and spin;  zonal errors in the FK5 catalogue are not taken into
           account.

        4) It was the intention that Hipparcos should be a close
           approximation to an inertial frame, so that distant objects have
           zero proper motion;  such objects have (in general) non-zero
           proper motion in FK5, and this function returns those fictitious
           proper motions.

        5) The position returned by this function is in the FK5 J2000.0
           reference system but at date date1+date2.

        6) See also eraFk52h, eraH2fk5, eraFk5zhz.

        Called:
           eraS2c       spherical coordinates to unit vector
           eraFk5hip    FK5 to Hipparcos rotation and spin
           eraRxp       product of r-matrix and p-vector
           eraSxp       multiply p-vector by scalar
           eraRxr       product of two r-matrices
           eraTrxp      product of transpose of r-matrix and p-vector
           eraPxp       vector product of two p-vectors
           eraPv2s      pv-vector to spherical
           eraAnp       normalize angle into range 0 to 2pi

        Reference:

           F.Mignard & M.Froeschle, 2000, Astron.Astrophys. 354, 732-739.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r5, d5, dr5, dd5 = ufunc.hfk5z(rh, dh, date1, date2)
    return r5, d5, dr5, dd5


def starpm(ra1, dec1, pmr1, pmd1, px1, rv1, ep1a, ep1b, ep2a, ep2b):
    """
    Star proper motion:  update star catalog data for space motion.

    Parameters
    ----------
    ra1 : double array
    dec1 : double array
    pmr1 : double array
    pmd1 : double array
    px1 : double array
    rv1 : double array
    ep1a : double array
    ep1b : double array
    ep2a : double array
    ep2b : double array

    Returns
    -------
    ra2 : double array
    dec2 : double array
    pmr2 : double array
    pmd2 : double array
    px2 : double array
    rv2 : double array

    Notes
    -----
    Wraps ERFA function ``eraStarpm``. The ERFA documentation is::

        - - - - - - - - - -
         e r a S t a r p m
        - - - - - - - - - -

        Star proper motion:  update star catalog data for space motion.

        Given:
           ra1    double     right ascension (radians), before
           dec1   double     declination (radians), before
           pmr1   double     RA proper motion (radians/year), before
           pmd1   double     Dec proper motion (radians/year), before
           px1    double     parallax (arcseconds), before
           rv1    double     radial velocity (km/s, +ve = receding), before
           ep1a   double     "before" epoch, part A (Note 1)
           ep1b   double     "before" epoch, part B (Note 1)
           ep2a   double     "after" epoch, part A (Note 1)
           ep2b   double     "after" epoch, part B (Note 1)

        Returned:
           ra2    double     right ascension (radians), after
           dec2   double     declination (radians), after
           pmr2   double     RA proper motion (radians/year), after
           pmd2   double     Dec proper motion (radians/year), after
           px2    double     parallax (arcseconds), after
           rv2    double     radial velocity (km/s, +ve = receding), after

        Returned (function value):
                  int        status:
                                -1 = system error (should not occur)
                                 0 = no warnings or errors
                                 1 = distance overridden (Note 6)
                                 2 = excessive velocity (Note 7)
                                 4 = solution didn't converge (Note 8)
                              else = binary logical OR of the above warnings

        Notes:

        1) The starting and ending TDB dates ep1a+ep1b and ep2a+ep2b are
           Julian Dates, apportioned in any convenient way between the two
           parts (A and B).  For example, JD(TDB)=2450123.7 could be
           expressed in any of these ways, among others:

                   epna          epnb

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) In accordance with normal star-catalog conventions, the object's
           right ascension and declination are freed from the effects of
           secular aberration.  The frame, which is aligned to the catalog
           equator and equinox, is Lorentzian and centered on the SSB.

           The proper motions are the rate of change of the right ascension
           and declination at the catalog epoch and are in radians per TDB
           Julian year.

           The parallax and radial velocity are in the same frame.

        3) Care is needed with units.  The star coordinates are in radians
           and the proper motions in radians per Julian year, but the
           parallax is in arcseconds.

        4) The RA proper motion is in terms of coordinate angle, not true
           angle.  If the catalog uses arcseconds for both RA and Dec proper
           motions, the RA proper motion will need to be divided by cos(Dec)
           before use.

        5) Straight-line motion at constant speed, in the inertial frame,
           is assumed.

        6) An extremely small (or zero or negative) parallax is interpreted
           to mean that the object is on the "celestial sphere", the radius
           of which is an arbitrary (large) value (see the eraStarpv
           function for the value used).  When the distance is overridden in
           this way, the status, initially zero, has 1 added to it.

        7) If the space velocity is a significant fraction of c (see the
           constant VMAX in the function eraStarpv), it is arbitrarily set
           to zero.  When this action occurs, 2 is added to the status.

        8) The relativistic adjustment carried out in the eraStarpv function
           involves an iterative calculation.  If the process fails to
           converge within a set number of iterations, 4 is added to the
           status.

        Called:
           eraStarpv    star catalog data to space motion pv-vector
           eraPvu       update a pv-vector
           eraPdp       scalar product of two p-vectors
           eraPvstar    space motion pv-vector to star catalog data

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    ra2, dec2, pmr2, pmd2, px2, rv2, c_retval = ufunc.starpm(
        ra1, dec1, pmr1, pmd1, px1, rv1, ep1a, ep1b, ep2a, ep2b)
    check_errwarn(c_retval, 'starpm')
    return ra2, dec2, pmr2, pmd2, px2, rv2


STATUS_CODES['starpm'] = {
    -1: 'system error (should not occur)',
    0: 'no warnings or errors',
    1: 'distance overridden (Note 6)',
    2: 'excessive velocity (Note 7)',
    4: "solution didn't converge (Note 8)",
    'else': 'binary logical OR of the above warnings',
}


def eceq06(date1, date2, dl, db):
    """
    Transformation from ecliptic coordinates (mean equinox and ecliptic
    of date) to ICRS RA,Dec, using the IAU 2006 precession model.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    dl : double array
    db : double array

    Returns
    -------
    dr : double array
    dd : double array

    Notes
    -----
    Wraps ERFA function ``eraEceq06``. The ERFA documentation is::

        - - - - - - - - - -
         e r a E c e q 0 6
        - - - - - - - - - -

        Transformation from ecliptic coordinates (mean equinox and ecliptic
        of date) to ICRS RA,Dec, using the IAU 2006 precession model.

        Given:
           date1,date2 double TT as a 2-part Julian date (Note 1)
           dl,db       double ecliptic longitude and latitude (radians)

        Returned:
           dr,dd       double ICRS right ascension and declination (radians)

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) No assumptions are made about whether the coordinates represent
           starlight and embody astrometric effects such as parallax or
           aberration.

        3) The transformation is approximately that from ecliptic longitude
           and latitude (mean equinox and ecliptic of date) to mean J2000.0
           right ascension and declination, with only frame bias (always
           less than 25 mas) to disturb this classical picture.

        Called:
           eraS2c       spherical coordinates to unit vector
           eraEcm06     J2000.0 to ecliptic rotation matrix, IAU 2006
           eraTrxp      product of transpose of r-matrix and p-vector
           eraC2s       unit vector to spherical coordinates
           eraAnp       normalize angle into range 0 to 2pi
           eraAnpm      normalize angle into range +/- pi

        This revision:  2016 February 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dr, dd = ufunc.eceq06(date1, date2, dl, db)
    return dr, dd


def ecm06(date1, date2):
    """
    ICRS equatorial to ecliptic rotation matrix, IAU 2006.

    Parameters
    ----------
    date1 : double array
    date2 : double array

    Returns
    -------
    rm : double array

    Notes
    -----
    Wraps ERFA function ``eraEcm06``. The ERFA documentation is::

        - - - - - - - - -
         e r a E c m 0 6
        - - - - - - - - -

        ICRS equatorial to ecliptic rotation matrix, IAU 2006.

        Given:
           date1,date2  double         TT as a 2-part Julian date (Note 1)

        Returned:
           rm           double[3][3]   ICRS to ecliptic rotation matrix

        Notes:

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        1) The matrix is in the sense

              E_ep = rm x P_ICRS,

           where P_ICRS is a vector with respect to ICRS right ascension
           and declination axes and E_ep is the same vector with respect to
           the (inertial) ecliptic and equinox of date.

        2) P_ICRS is a free vector, merely a direction, typically of unit
           magnitude, and not bound to any particular spatial origin, such
           as the Earth, Sun or SSB.  No assumptions are made about whether
           it represents starlight and embodies astrometric effects such as
           parallax or aberration.  The transformation is approximately that
           between mean J2000.0 right ascension and declination and ecliptic
           longitude and latitude, with only frame bias (always less than
           25 mas) to disturb this classical picture.

        Called:
           eraObl06     mean obliquity, IAU 2006
           eraPmat06    PB matrix, IAU 2006
           eraIr        initialize r-matrix to identity
           eraRx        rotate around X-axis
           eraRxr       product of two r-matrices

        This revision:  2015 December 11

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rm = ufunc.ecm06(date1, date2)
    return rm


def eqec06(date1, date2, dr, dd):
    """
    Transformation from ICRS equatorial coordinates to ecliptic
    coordinates (mean equinox and ecliptic of date) using IAU 2006
    precession model.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    dr : double array
    dd : double array

    Returns
    -------
    dl : double array
    db : double array

    Notes
    -----
    Wraps ERFA function ``eraEqec06``. The ERFA documentation is::

        - - - - - - - - - -
         e r a E q e c 0 6
        - - - - - - - - - -

        Transformation from ICRS equatorial coordinates to ecliptic
        coordinates (mean equinox and ecliptic of date) using IAU 2006
        precession model.

        Given:
           date1,date2 double TT as a 2-part Julian date (Note 1)
           dr,dd       double ICRS right ascension and declination (radians)

        Returned:
           dl,db       double ecliptic longitude and latitude (radians)

        1) The TT date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

        2) No assumptions are made about whether the coordinates represent
           starlight and embody astrometric effects such as parallax or
           aberration.

        3) The transformation is approximately that from mean J2000.0 right
           ascension and declination to ecliptic longitude and latitude
           (mean equinox and ecliptic of date), with only frame bias (always
           less than 25 mas) to disturb this classical picture.

        Called:
           eraS2c       spherical coordinates to unit vector
           eraEcm06     J2000.0 to ecliptic rotation matrix, IAU 2006
           eraRxp       product of r-matrix and p-vector
           eraC2s       unit vector to spherical coordinates
           eraAnp       normalize angle into range 0 to 2pi
           eraAnpm      normalize angle into range +/- pi

        This revision:  2016 February 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dl, db = ufunc.eqec06(date1, date2, dr, dd)
    return dl, db


def lteceq(epj, dl, db):
    """
    Transformation from ecliptic coordinates (mean equinox and ecliptic
    of date) to ICRS RA,Dec, using a long-term precession model.

    Parameters
    ----------
    epj : double array
    dl : double array
    db : double array

    Returns
    -------
    dr : double array
    dd : double array

    Notes
    -----
    Wraps ERFA function ``eraLteceq``. The ERFA documentation is::

        - - - - - - - - - -
         e r a L t e c e q
        - - - - - - - - - -

        Transformation from ecliptic coordinates (mean equinox and ecliptic
        of date) to ICRS RA,Dec, using a long-term precession model.

        Given:
           epj     double     Julian epoch (TT)
           dl,db   double     ecliptic longitude and latitude (radians)

        Returned:
           dr,dd   double     ICRS right ascension and declination (radians)

        1) No assumptions are made about whether the coordinates represent
           starlight and embody astrometric effects such as parallax or
           aberration.

        2) The transformation is approximately that from ecliptic longitude
           and latitude (mean equinox and ecliptic of date) to mean J2000.0
           right ascension and declination, with only frame bias (always
           less than 25 mas) to disturb this classical picture.

        3) The Vondrak et al. (2011, 2012) 400 millennia precession model
           agrees with the IAU 2006 precession at J2000.0 and stays within
           100 microarcseconds during the 20th and 21st centuries.  It is
           accurate to a few arcseconds throughout the historical period,
           worsening to a few tenths of a degree at the end of the
           +/- 200,000 year time span.

        Called:
           eraS2c       spherical coordinates to unit vector
           eraLtecm     J2000.0 to ecliptic rotation matrix, long term
           eraTrxp      product of transpose of r-matrix and p-vector
           eraC2s       unit vector to spherical coordinates
           eraAnp       normalize angle into range 0 to 2pi
           eraAnpm      normalize angle into range +/- pi

        References:

          Vondrak, J., Capitaine, N. and Wallace, P., 2011, New precession
          expressions, valid for long time intervals, Astron.Astrophys. 534,
          A22

          Vondrak, J., Capitaine, N. and Wallace, P., 2012, New precession
          expressions, valid for long time intervals (Corrigendum),
          Astron.Astrophys. 541, C1

        This revision:  2016 February 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dr, dd = ufunc.lteceq(epj, dl, db)
    return dr, dd


def ltecm(epj):
    """
    ICRS equatorial to ecliptic rotation matrix, long-term.

    Parameters
    ----------
    epj : double array

    Returns
    -------
    rm : double array

    Notes
    -----
    Wraps ERFA function ``eraLtecm``. The ERFA documentation is::

        - - - - - - - - -
         e r a L t e c m
        - - - - - - - - -

        ICRS equatorial to ecliptic rotation matrix, long-term.

        Given:
           epj     double         Julian epoch (TT)

        Returned:
           rm      double[3][3]   ICRS to ecliptic rotation matrix

        Notes:

        1) The matrix is in the sense

              E_ep = rm x P_ICRS,

           where P_ICRS is a vector with respect to ICRS right ascension
           and declination axes and E_ep is the same vector with respect to
           the (inertial) ecliptic and equinox of epoch epj.

        2) P_ICRS is a free vector, merely a direction, typically of unit
           magnitude, and not bound to any particular spatial origin, such
           as the Earth, Sun or SSB.  No assumptions are made about whether
           it represents starlight and embodies astrometric effects such as
           parallax or aberration.  The transformation is approximately that
           between mean J2000.0 right ascension and declination and ecliptic
           longitude and latitude, with only frame bias (always less than
           25 mas) to disturb this classical picture.

        3) The Vondrak et al. (2011, 2012) 400 millennia precession model
           agrees with the IAU 2006 precession at J2000.0 and stays within
           100 microarcseconds during the 20th and 21st centuries.  It is
           accurate to a few arcseconds throughout the historical period,
           worsening to a few tenths of a degree at the end of the
           +/- 200,000 year time span.

        Called:
           eraLtpequ    equator pole, long term
           eraLtpecl    ecliptic pole, long term
           eraPxp       vector product
           eraPn        normalize vector

        References:

          Vondrak, J., Capitaine, N. and Wallace, P., 2011, New precession
          expressions, valid for long time intervals, Astron.Astrophys. 534,
          A22

          Vondrak, J., Capitaine, N. and Wallace, P., 2012, New precession
          expressions, valid for long time intervals (Corrigendum),
          Astron.Astrophys. 541, C1

        This revision:  2015 December 6

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rm = ufunc.ltecm(epj)
    return rm


def lteqec(epj, dr, dd):
    """
    Transformation from ICRS equatorial coordinates to ecliptic
    coordinates (mean equinox and ecliptic of date) using a long-term
    precession model.

    Parameters
    ----------
    epj : double array
    dr : double array
    dd : double array

    Returns
    -------
    dl : double array
    db : double array

    Notes
    -----
    Wraps ERFA function ``eraLteqec``. The ERFA documentation is::

        - - - - - - - - - -
         e r a L t e q e c
        - - - - - - - - - -

        Transformation from ICRS equatorial coordinates to ecliptic
        coordinates (mean equinox and ecliptic of date) using a long-term
        precession model.

        Given:
           epj     double     Julian epoch (TT)
           dr,dd   double     ICRS right ascension and declination (radians)

        Returned:
           dl,db   double     ecliptic longitude and latitude (radians)

        1) No assumptions are made about whether the coordinates represent
           starlight and embody astrometric effects such as parallax or
           aberration.

        2) The transformation is approximately that from mean J2000.0 right
           ascension and declination to ecliptic longitude and latitude
           (mean equinox and ecliptic of date), with only frame bias (always
           less than 25 mas) to disturb this classical picture.

        3) The Vondrak et al. (2011, 2012) 400 millennia precession model
           agrees with the IAU 2006 precession at J2000.0 and stays within
           100 microarcseconds during the 20th and 21st centuries.  It is
           accurate to a few arcseconds throughout the historical period,
           worsening to a few tenths of a degree at the end of the
           +/- 200,000 year time span.

        Called:
           eraS2c       spherical coordinates to unit vector
           eraLtecm     J2000.0 to ecliptic rotation matrix, long term
           eraRxp       product of r-matrix and p-vector
           eraC2s       unit vector to spherical coordinates
           eraAnp       normalize angle into range 0 to 2pi
           eraAnpm      normalize angle into range +/- pi

        References:

          Vondrak, J., Capitaine, N. and Wallace, P., 2011, New precession
          expressions, valid for long time intervals, Astron.Astrophys. 534,
          A22

          Vondrak, J., Capitaine, N. and Wallace, P., 2012, New precession
          expressions, valid for long time intervals (Corrigendum),
          Astron.Astrophys. 541, C1

        This revision:  2016 February 9

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dl, db = ufunc.lteqec(epj, dr, dd)
    return dl, db


def g2icrs(dl, db):
    """
    Transformation from Galactic Coordinates to ICRS.

    Parameters
    ----------
    dl : double array
    db : double array

    Returns
    -------
    dr : double array
    dd : double array

    Notes
    -----
    Wraps ERFA function ``eraG2icrs``. The ERFA documentation is::

        - - - - - - - - - -
         e r a G 2 i c r s
        - - - - - - - - - -

        Transformation from Galactic Coordinates to ICRS.

        Given:
           dl     double      galactic longitude (radians)
           db     double      galactic latitude (radians)

        Returned:
           dr     double      ICRS right ascension (radians)
           dd     double      ICRS declination (radians)

        Notes:

        1) The IAU 1958 system of Galactic coordinates was defined with
           respect to the now obsolete reference system FK4 B1950.0.  When
           interpreting the system in a modern context, several factors have
           to be taken into account:

           . The inclusion in FK4 positions of the E-terms of aberration.

           . The distortion of the FK4 proper motion system by differential
             Galactic rotation.

           . The use of the B1950.0 equinox rather than the now-standard
             J2000.0.

           . The frame bias between ICRS and the J2000.0 mean place system.

           The Hipparcos Catalogue (Perryman & ESA 1997) provides a rotation
           matrix that transforms directly between ICRS and Galactic
           coordinates with the above factors taken into account.  The
           matrix is derived from three angles, namely the ICRS coordinates
           of the Galactic pole and the longitude of the ascending node of
           the galactic equator on the ICRS equator.  They are given in
           degrees to five decimal places and for canonical purposes are
           regarded as exact.  In the Hipparcos Catalogue the matrix
           elements are given to 10 decimal places (about 20 microarcsec).
           In the present ERFA function the matrix elements have been
           recomputed from the canonical three angles and are given to 30
           decimal places.

        2) The inverse transformation is performed by the function eraIcrs2g.

        Called:
           eraAnp       normalize angle into range 0 to 2pi
           eraAnpm      normalize angle into range +/- pi
           eraS2c       spherical coordinates to unit vector
           eraTrxp      product of transpose of r-matrix and p-vector
           eraC2s       p-vector to spherical

        Reference:
           Perryman M.A.C. & ESA, 1997, ESA SP-1200, The Hipparcos and Tycho
           catalogues.  Astrometric and photometric star catalogues
           derived from the ESA Hipparcos Space Astrometry Mission.  ESA
           Publications Division, Noordwijk, Netherlands.

        This revision:   2018 January 2

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dr, dd = ufunc.g2icrs(dl, db)
    return dr, dd


def icrs2g(dr, dd):
    """
    Transformation from ICRS to Galactic Coordinates.

    Parameters
    ----------
    dr : double array
    dd : double array

    Returns
    -------
    dl : double array
    db : double array

    Notes
    -----
    Wraps ERFA function ``eraIcrs2g``. The ERFA documentation is::

        - - - - - - - - - -
         e r a I c r s 2 g
        - - - - - - - - - -

        Transformation from ICRS to Galactic Coordinates.

        Given:
           dr     double      ICRS right ascension (radians)
           dd     double      ICRS declination (radians)

        Returned:
           dl     double      galactic longitude (radians)
           db     double      galactic latitude (radians)

        Notes:

        1) The IAU 1958 system of Galactic coordinates was defined with
           respect to the now obsolete reference system FK4 B1950.0.  When
           interpreting the system in a modern context, several factors have
           to be taken into account:

           . The inclusion in FK4 positions of the E-terms of aberration.

           . The distortion of the FK4 proper motion system by differential
             Galactic rotation.

           . The use of the B1950.0 equinox rather than the now-standard
             J2000.0.

           . The frame bias between ICRS and the J2000.0 mean place system.

           The Hipparcos Catalogue (Perryman & ESA 1997) provides a rotation
           matrix that transforms directly between ICRS and Galactic
           coordinates with the above factors taken into account.  The
           matrix is derived from three angles, namely the ICRS coordinates
           of the Galactic pole and the longitude of the ascending node of
           the galactic equator on the ICRS equator.  They are given in
           degrees to five decimal places and for canonical purposes are
           regarded as exact.  In the Hipparcos Catalogue the matrix
           elements are given to 10 decimal places (about 20 microarcsec).
           In the present ERFA function the matrix elements have been
           recomputed from the canonical three angles and are given to 30
           decimal places.

        2) The inverse transformation is performed by the function eraG2icrs.

        Called:
           eraAnp       normalize angle into range 0 to 2pi
           eraAnpm      normalize angle into range +/- pi
           eraS2c       spherical coordinates to unit vector
           eraRxp       product of r-matrix and p-vector
           eraC2s       p-vector to spherical

        Reference:
           Perryman M.A.C. & ESA, 1997, ESA SP-1200, The Hipparcos and Tycho
           catalogues.  Astrometric and photometric star catalogues
           derived from the ESA Hipparcos Space Astrometry Mission.  ESA
           Publications Division, Noordwijk, Netherlands.

        This revision:   2018 January 2

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    dl, db = ufunc.icrs2g(dr, dd)
    return dl, db


def eform(n):
    """
    Earth reference ellipsoids.

    Parameters
    ----------
    n : int array

    Returns
    -------
    a : double array
    f : double array

    Notes
    -----
    Wraps ERFA function ``eraEform``. The ERFA documentation is::

        - - - - - - - - -
         e r a E f o r m
        - - - - - - - - -

        Earth reference ellipsoids.

        Given:
           n    int         ellipsoid identifier (Note 1)

        Returned:
           a    double      equatorial radius (meters, Note 2)
           f    double      flattening (Note 2)

        Returned (function value):
                int         status:  0 = OK
                                    -1 = illegal identifier (Note 3)

        Notes:

        1) The identifier n is a number that specifies the choice of
           reference ellipsoid.  The following are supported:

              n    ellipsoid

              1     ERFA_WGS84
              2     ERFA_GRS80
              3     ERFA_WGS72

           The n value has no significance outside the ERFA software.  For
           convenience, symbols ERFA_WGS84 etc. are defined in erfam.h.

        2) The ellipsoid parameters are returned in the form of equatorial
           radius in meters (a) and flattening (f).  The latter is a number
           around 0.00335, i.e. around 1/298.

        3) For the case where an unsupported n value is supplied, zero a and
           f are returned, as well as error status.

        References:

           Department of Defense World Geodetic System 1984, National
           Imagery and Mapping Agency Technical Report 8350.2, Third
           Edition, p3-2.

           Moritz, H., Bull. Geodesique 66-2, 187 (1992).

           The Department of Defense World Geodetic System 1972, World
           Geodetic System Committee, May 1974.

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992),
           p220.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    a, f, c_retval = ufunc.eform(n)
    check_errwarn(c_retval, 'eform')
    return a, f


STATUS_CODES['eform'] = {
    0: 'OK',
    -1: 'illegal identifier (Note 3)',
}


def gc2gd(n, xyz):
    """
    Transform geocentric coordinates to geodetic using the specified
    reference ellipsoid.

    Parameters
    ----------
    n : int array
    xyz : double array

    Returns
    -------
    elong : double array
    phi : double array
    height : double array

    Notes
    -----
    Wraps ERFA function ``eraGc2gd``. The ERFA documentation is::

        - - - - - - - - -
         e r a G c 2 g d
        - - - - - - - - -

        Transform geocentric coordinates to geodetic using the specified
        reference ellipsoid.

        Given:
           n       int        ellipsoid identifier (Note 1)
           xyz     double[3]  geocentric vector (Note 2)

        Returned:
           elong   double     longitude (radians, east +ve, Note 3)
           phi     double     latitude (geodetic, radians, Note 3)
           height  double     height above ellipsoid (geodetic, Notes 2,3)

        Returned (function value):
                  int         status:  0 = OK
                                      -1 = illegal identifier (Note 3)
                                      -2 = internal error (Note 3)

        Notes:

        1) The identifier n is a number that specifies the choice of
           reference ellipsoid.  The following are supported:

              n    ellipsoid

              1     ERFA_WGS84
              2     ERFA_GRS80
              3     ERFA_WGS72

           The n value has no significance outside the ERFA software.  For
           convenience, symbols ERFA_WGS84 etc. are defined in erfam.h.

        2) The geocentric vector (xyz, given) and height (height, returned)
           are in meters.

        3) An error status -1 means that the identifier n is illegal.  An
           error status -2 is theoretically impossible.  In all error cases,
           all three results are set to -1e9.

        4) The inverse transformation is performed in the function eraGd2gc.

        Called:
           eraEform     Earth reference ellipsoids
           eraGc2gde    geocentric to geodetic transformation, general

        This revision:  2013 September 1

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    elong, phi, height, c_retval = ufunc.gc2gd(n, xyz)
    check_errwarn(c_retval, 'gc2gd')
    return elong, phi, height


STATUS_CODES['gc2gd'] = {
    0: 'OK',
    -1: 'illegal identifier (Note 3)',
    -2: 'internal error (Note 3)',
}


def gc2gde(a, f, xyz):
    """
    Transform geocentric coordinates to geodetic for a reference
    ellipsoid of specified form.

    Parameters
    ----------
    a : double array
    f : double array
    xyz : double array

    Returns
    -------
    elong : double array
    phi : double array
    height : double array

    Notes
    -----
    Wraps ERFA function ``eraGc2gde``. The ERFA documentation is::

        - - - - - - - - - -
         e r a G c 2 g d e
        - - - - - - - - - -

        Transform geocentric coordinates to geodetic for a reference
        ellipsoid of specified form.

        Given:
           a       double     equatorial radius (Notes 2,4)
           f       double     flattening (Note 3)
           xyz     double[3]  geocentric vector (Note 4)

        Returned:
           elong   double     longitude (radians, east +ve)
           phi     double     latitude (geodetic, radians)
           height  double     height above ellipsoid (geodetic, Note 4)

        Returned (function value):
                   int        status:  0 = OK
                                      -1 = illegal f
                                      -2 = illegal a

        Notes:

        1) This function is based on the GCONV2H Fortran subroutine by
           Toshio Fukushima (see reference).

        2) The equatorial radius, a, can be in any units, but meters is
           the conventional choice.

        3) The flattening, f, is (for the Earth) a value around 0.00335,
           i.e. around 1/298.

        4) The equatorial radius, a, and the geocentric vector, xyz,
           must be given in the same units, and determine the units of
           the returned height, height.

        5) If an error occurs (status < 0), elong, phi and height are
           unchanged.

        6) The inverse transformation is performed in the function
           eraGd2gce.

        7) The transformation for a standard ellipsoid (such as ERFA_WGS84) can
           more conveniently be performed by calling eraGc2gd, which uses a
           numerical code to identify the required A and F values.

        Reference:

           Fukushima, T., "Transformation from Cartesian to geodetic
           coordinates accelerated by Halley's method", J.Geodesy (2006)
           79: 689-693

        This revision:  2014 November 7

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    elong, phi, height, c_retval = ufunc.gc2gde(a, f, xyz)
    check_errwarn(c_retval, 'gc2gde')
    return elong, phi, height


STATUS_CODES['gc2gde'] = {
    0: 'OK',
    -1: 'illegal f',
    -2: 'illegal a',
}


def gd2gc(n, elong, phi, height):
    """
    Transform geodetic coordinates to geocentric using the specified
    reference ellipsoid.

    Parameters
    ----------
    n : int array
    elong : double array
    phi : double array
    height : double array

    Returns
    -------
    xyz : double array

    Notes
    -----
    Wraps ERFA function ``eraGd2gc``. The ERFA documentation is::

        - - - - - - - - -
         e r a G d 2 g c
        - - - - - - - - -

        Transform geodetic coordinates to geocentric using the specified
        reference ellipsoid.

        Given:
           n       int        ellipsoid identifier (Note 1)
           elong   double     longitude (radians, east +ve)
           phi     double     latitude (geodetic, radians, Note 3)
           height  double     height above ellipsoid (geodetic, Notes 2,3)

        Returned:
           xyz     double[3]  geocentric vector (Note 2)

        Returned (function value):
                   int        status:  0 = OK
                                      -1 = illegal identifier (Note 3)
                                      -2 = illegal case (Note 3)

        Notes:

        1) The identifier n is a number that specifies the choice of
           reference ellipsoid.  The following are supported:

              n    ellipsoid

              1     ERFA_WGS84
              2     ERFA_GRS80
              3     ERFA_WGS72

           The n value has no significance outside the ERFA software.  For
           convenience, symbols ERFA_WGS84 etc. are defined in erfam.h.

        2) The height (height, given) and the geocentric vector (xyz,
           returned) are in meters.

        3) No validation is performed on the arguments elong, phi and
           height.  An error status -1 means that the identifier n is
           illegal.  An error status -2 protects against cases that would
           lead to arithmetic exceptions.  In all error cases, xyz is set
           to zeros.

        4) The inverse transformation is performed in the function eraGc2gd.

        Called:
           eraEform     Earth reference ellipsoids
           eraGd2gce    geodetic to geocentric transformation, general
           eraZp        zero p-vector

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    xyz, c_retval = ufunc.gd2gc(n, elong, phi, height)
    check_errwarn(c_retval, 'gd2gc')
    return xyz


STATUS_CODES['gd2gc'] = {
    0: 'OK',
    -1: 'illegal identifier (Note 3)',
    -2: 'illegal case (Note 3)',
}


def gd2gce(a, f, elong, phi, height):
    """
    Transform geodetic coordinates to geocentric for a reference
    ellipsoid of specified form.

    Parameters
    ----------
    a : double array
    f : double array
    elong : double array
    phi : double array
    height : double array

    Returns
    -------
    xyz : double array

    Notes
    -----
    Wraps ERFA function ``eraGd2gce``. The ERFA documentation is::

        - - - - - - - - - -
         e r a G d 2 g c e
        - - - - - - - - - -

        Transform geodetic coordinates to geocentric for a reference
        ellipsoid of specified form.

        Given:
           a       double     equatorial radius (Notes 1,4)
           f       double     flattening (Notes 2,4)
           elong   double     longitude (radians, east +ve)
           phi     double     latitude (geodetic, radians, Note 4)
           height  double     height above ellipsoid (geodetic, Notes 3,4)

        Returned:
           xyz     double[3]  geocentric vector (Note 3)

        Returned (function value):
                   int        status:  0 = OK
                                      -1 = illegal case (Note 4)
        Notes:

        1) The equatorial radius, a, can be in any units, but meters is
           the conventional choice.

        2) The flattening, f, is (for the Earth) a value around 0.00335,
           i.e. around 1/298.

        3) The equatorial radius, a, and the height, height, must be
           given in the same units, and determine the units of the
           returned geocentric vector, xyz.

        4) No validation is performed on individual arguments.  The error
           status -1 protects against (unrealistic) cases that would lead
           to arithmetic exceptions.  If an error occurs, xyz is unchanged.

        5) The inverse transformation is performed in the function
           eraGc2gde.

        6) The transformation for a standard ellipsoid (such as ERFA_WGS84) can
           more conveniently be performed by calling eraGd2gc,  which uses a
           numerical code to identify the required a and f values.

        References:

           Green, R.M., Spherical Astronomy, Cambridge University Press,
           (1985) Section 4.5, p96.

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992),
           Section 4.22, p202.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    xyz, c_retval = ufunc.gd2gce(a, f, elong, phi, height)
    check_errwarn(c_retval, 'gd2gce')
    return xyz


STATUS_CODES['gd2gce'] = {
    0: 'OK',
    -1: 'illegal case (Note 4)Notes:',
}


def d2dtf(scale, ndp, d1, d2):
    """
    Format for output a 2-part Julian Date (or in the case of UTC a
    quasi-JD form that includes special provision for leap seconds).

    Parameters
    ----------
    scale : const char array
    ndp : int array
    d1 : double array
    d2 : double array

    Returns
    -------
    iy : int array
    im : int array
    id : int array
    ihmsf : int array

    Notes
    -----
    Wraps ERFA function ``eraD2dtf``. The ERFA documentation is::

        - - - - - - - - -
         e r a D 2 d t f
        - - - - - - - - -

        Format for output a 2-part Julian Date (or in the case of UTC a
        quasi-JD form that includes special provision for leap seconds).

        Given:
           scale     char[]  time scale ID (Note 1)
           ndp       int     resolution (Note 2)
           d1,d2     double  time as a 2-part Julian Date (Notes 3,4)

        Returned:
           iy,im,id  int     year, month, day in Gregorian calendar (Note 5)
           ihmsf     int[4]  hours, minutes, seconds, fraction (Note 1)

        Returned (function value):
                     int     status: +1 = dubious year (Note 5)
                                      0 = OK
                                     -1 = unacceptable date (Note 6)

        Notes:

        1) scale identifies the time scale.  Only the value "UTC" (in upper
           case) is significant, and enables handling of leap seconds (see
           Note 4).

        2) ndp is the number of decimal places in the seconds field, and can
           have negative as well as positive values, such as:

           ndp         resolution
           -4            1 00 00
           -3            0 10 00
           -2            0 01 00
           -1            0 00 10
            0            0 00 01
            1            0 00 00.1
            2            0 00 00.01
            3            0 00 00.001

           The limits are platform dependent, but a safe range is -5 to +9.

        3) d1+d2 is Julian Date, apportioned in any convenient way between
           the two arguments, for example where d1 is the Julian Day Number
           and d2 is the fraction of a day.  In the case of UTC, where the
           use of JD is problematical, special conventions apply:  see the
           next note.

        4) JD cannot unambiguously represent UTC during a leap second unless
           special measures are taken.  The ERFA internal convention is that
           the quasi-JD day represents UTC days whether the length is 86399,
           86400 or 86401 SI seconds.  In the 1960-1972 era there were
           smaller jumps (in either direction) each time the linear UTC(TAI)
           expression was changed, and these "mini-leaps" are also included
           in the ERFA convention.

        5) The warning status "dubious year" flags UTCs that predate the
           introduction of the time scale or that are too far in the future
           to be trusted.  See eraDat for further details.

        6) For calendar conventions and limitations, see eraCal2jd.

        Called:
           eraJd2cal    JD to Gregorian calendar
           eraD2tf      decompose days to hms
           eraDat       delta(AT) = TAI-UTC

        This revision:  2021 February 12

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    iy, im, id, ihmsf, c_retval = ufunc.d2dtf(scale, ndp, d1, d2)
    check_errwarn(c_retval, 'd2dtf')
    return iy, im, id, ihmsf


STATUS_CODES['d2dtf'] = {
    1: 'dubious year (Note 5)',
    0: 'OK',
    -1: 'unacceptable date (Note 6)',
}


def dat(iy, im, id, fd):
    """
    For a given UTC date, calculate Delta(AT) = TAI-UTC.

    Parameters
    ----------
    iy : int array
    im : int array
    id : int array
    fd : double array

    Returns
    -------
    deltat : double array

    Notes
    -----
    Wraps ERFA function ``eraDat``. The ERFA documentation is::

        - - - - - - -
         e r a D a t
        - - - - - - -

        For a given UTC date, calculate Delta(AT) = TAI-UTC.

           :------------------------------------------:
           :                                          :
           :                 IMPORTANT                :
           :                                          :
           :  A new version of this function must be  :
           :  produced whenever a new leap second is  :
           :  announced.  There are four items to     :
           :  change on each such occasion:           :
           :                                          :
           :  1) A new line must be added to the set  :
           :     of statements that initialize the    :
           :     array "changes".                     :
           :                                          :
           :  2) The constant IYV must be set to the  :
           :     current year.                        :
           :                                          :
           :  3) The "Latest leap second" comment     :
           :     below must be set to the new leap    :
           :     second date.                         :
           :                                          :
           :  4) The "This revision" comment, later,  :
           :     must be set to the current date.     :
           :                                          :
           :  Change (2) must also be carried out     :
           :  whenever the function is re-issued,     :
           :  even if no leap seconds have been       :
           :  added.                                  :
           :                                          :
           :  Latest leap second:  2016 December 31   :
           :                                          :
           :__________________________________________:

        Given:
           iy     int      UTC:  year (Notes 1 and 2)
           im     int            month (Note 2)
           id     int            day (Notes 2 and 3)
           fd     double         fraction of day (Note 4)

        Returned:
           deltat double   TAI minus UTC, seconds

        Returned (function value):
                  int      status (Note 5):
                             1 = dubious year (Note 1)
                             0 = OK
                            -1 = bad year
                            -2 = bad month
                            -3 = bad day (Note 3)
                            -4 = bad fraction (Note 4)
                            -5 = internal error (Note 5)

        Notes:

        1) UTC began at 1960 January 1.0 (JD 2436934.5) and it is improper
           to call the function with an earlier date.  If this is attempted,
           zero is returned together with a warning status.

           Because leap seconds cannot, in principle, be predicted in
           advance, a reliable check for dates beyond the valid range is
           impossible.  To guard against gross errors, a year five or more
           after the release year of the present function (see the constant
           IYV) is considered dubious.  In this case a warning status is
           returned but the result is computed in the normal way.

           For both too-early and too-late years, the warning status is +1.
           This is distinct from the error status -1, which signifies a year
           so early that JD could not be computed.

        2) If the specified date is for a day which ends with a leap second,
           the TAI-UTC value returned is for the period leading up to the
           leap second.  If the date is for a day which begins as a leap
           second ends, the TAI-UTC returned is for the period following the
           leap second.

        3) The day number must be in the normal calendar range, for example
           1 through 30 for April.  The "almanac" convention of allowing
           such dates as January 0 and December 32 is not supported in this
           function, in order to avoid confusion near leap seconds.

        4) The fraction of day is used only for dates before the
           introduction of leap seconds, the first of which occurred at the
           end of 1971.  It is tested for validity (0 to 1 is the valid
           range) even if not used;  if invalid, zero is used and status -4
           is returned.  For many applications, setting fd to zero is
           acceptable;  the resulting error is always less than 3 ms (and
           occurs only pre-1972).

        5) The status value returned in the case where there are multiple
           errors refers to the first error detected.  For example, if the
           month and day are 13 and 32 respectively, status -2 (bad month)
           will be returned.  The "internal error" status refers to a
           case that is impossible but causes some compilers to issue a
           warning.

        6) In cases where a valid result is not available, zero is returned.

        References:

        1) For dates from 1961 January 1 onwards, the expressions from the
           file ftp://maia.usno.navy.mil/ser7/tai-utc.dat are used.

        2) The 5ms timestep at 1961 January 1 is taken from 2.58.1 (p87) of
           the 1992 Explanatory Supplement.

        Called:
           eraCal2jd    Gregorian calendar to JD

        This revision:  2020 May 31

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    deltat, c_retval = ufunc.dat(iy, im, id, fd)
    check_errwarn(c_retval, 'dat')
    return deltat


STATUS_CODES['dat'] = {
    1: 'dubious year (Note 1)',
    0: 'OK',
    -1: 'bad year',
    -2: 'bad month',
    -3: 'bad day (Note 3)',
    -4: 'bad fraction (Note 4)',
    -5: 'internal error (Note 5)',
}


def dtdb(date1, date2, ut, elong, u, v):
    """
    An approximation to TDB-TT, the difference between barycentric
    dynamical time and terrestrial time, for an observer on the Earth.

    Parameters
    ----------
    date1 : double array
    date2 : double array
    ut : double array
    elong : double array
    u : double array
    v : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraDtdb``. The ERFA documentation is::

        - - - - - - - -
         e r a D t d b
        - - - - - - - -

        An approximation to TDB-TT, the difference between barycentric
        dynamical time and terrestrial time, for an observer on the Earth.

        The different time scales - proper, coordinate and realized - are
        related to each other:

                  TAI             <-  physically realized
                   :
                offset            <-  observed (nominally +32.184s)
                   :
                  TT              <-  terrestrial time
                   :
          rate adjustment (L_G)   <-  definition of TT
                   :
                  TCG             <-  time scale for GCRS
                   :
            "periodic" terms      <-  eraDtdb  is an implementation
                   :
          rate adjustment (L_C)   <-  function of solar-system ephemeris
                   :
                  TCB             <-  time scale for BCRS
                   :
          rate adjustment (-L_B)  <-  definition of TDB
                   :
                  TDB             <-  TCB scaled to track TT
                   :
            "periodic" terms      <-  -eraDtdb is an approximation
                   :
                  TT              <-  terrestrial time

        Adopted values for the various constants can be found in the IERS
        Conventions (McCarthy & Petit 2003).

        Given:
           date1,date2   double  date, TDB (Notes 1-3)
           ut            double  universal time (UT1, fraction of one day)
           elong         double  longitude (east positive, radians)
           u             double  distance from Earth spin axis (km)
           v             double  distance north of equatorial plane (km)

        Returned (function value):
                         double  TDB-TT (seconds)

        Notes:

        1) The date date1+date2 is a Julian Date, apportioned in any
           convenient way between the two arguments.  For example,
           JD(TT)=2450123.7 could be expressed in any of these ways,
           among others:

                  date1          date2

               2450123.7           0.0       (JD method)
               2451545.0       -1421.3       (J2000 method)
               2400000.5       50123.2       (MJD method)
               2450123.5           0.2       (date & time method)

           The JD method is the most natural and convenient to use in
           cases where the loss of several decimal digits of resolution
           is acceptable.  The J2000 method is best matched to the way
           the argument is handled internally and will deliver the
           optimum resolution.  The MJD method and the date & time methods
           are both good compromises between resolution and convenience.

           Although the date is, formally, barycentric dynamical time (TDB),
           the terrestrial dynamical time (TT) can be used with no practical
           effect on the accuracy of the prediction.

        2) TT can be regarded as a coordinate time that is realized as an
           offset of 32.184s from International Atomic Time, TAI.  TT is a
           specific linear transformation of geocentric coordinate time TCG,
           which is the time scale for the Geocentric Celestial Reference
           System, GCRS.

        3) TDB is a coordinate time, and is a specific linear transformation
           of barycentric coordinate time TCB, which is the time scale for
           the Barycentric Celestial Reference System, BCRS.

        4) The difference TCG-TCB depends on the masses and positions of the
           bodies of the solar system and the velocity of the Earth.  It is
           dominated by a rate difference, the residual being of a periodic
           character.  The latter, which is modeled by the present function,
           comprises a main (annual) sinusoidal term of amplitude
           approximately 0.00166 seconds, plus planetary terms up to about
           20 microseconds, and lunar and diurnal terms up to 2 microseconds.
           These effects come from the changing transverse Doppler effect
           and gravitational red-shift as the observer (on the Earth's
           surface) experiences variations in speed (with respect to the
           BCRS) and gravitational potential.

        5) TDB can be regarded as the same as TCB but with a rate adjustment
           to keep it close to TT, which is convenient for many applications.
           The history of successive attempts to define TDB is set out in
           Resolution 3 adopted by the IAU General Assembly in 2006, which
           defines a fixed TDB(TCB) transformation that is consistent with
           contemporary solar-system ephemerides.  Future ephemerides will
           imply slightly changed transformations between TCG and TCB, which
           could introduce a linear drift between TDB and TT;  however, any
           such drift is unlikely to exceed 1 nanosecond per century.

        6) The geocentric TDB-TT model used in the present function is that of
           Fairhead & Bretagnon (1990), in its full form.  It was originally
           supplied by Fairhead (private communications with P.T.Wallace,
           1990) as a Fortran subroutine.  The present C function contains an
           adaptation of the Fairhead code.  The numerical results are
           essentially unaffected by the changes, the differences with
           respect to the Fairhead & Bretagnon original being at the 1e-20 s
           level.

           The topocentric part of the model is from Moyer (1981) and
           Murray (1983), with fundamental arguments adapted from
           Simon et al. 1994.  It is an approximation to the expression
           ( v / c ) . ( r / c ), where v is the barycentric velocity of
           the Earth, r is the geocentric position of the observer and
           c is the speed of light.

           By supplying zeroes for u and v, the topocentric part of the
           model can be nullified, and the function will return the Fairhead
           & Bretagnon result alone.

        7) During the interval 1950-2050, the absolute accuracy is better
           than +/- 3 nanoseconds relative to time ephemerides obtained by
           direct numerical integrations based on the JPL DE405 solar system
           ephemeris.

        8) It must be stressed that the present function is merely a model,
           and that numerical integration of solar-system ephemerides is the
           definitive method for predicting the relationship between TCG and
           TCB and hence between TT and TDB.

        References:

           Fairhead, L., & Bretagnon, P., Astron.Astrophys., 229, 240-247
           (1990).

           IAU 2006 Resolution 3.

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Moyer, T.D., Cel.Mech., 23, 33 (1981).

           Murray, C.A., Vectorial Astrometry, Adam Hilger (1983).

           Seidelmann, P.K. et al., Explanatory Supplement to the
           Astronomical Almanac, Chapter 2, University Science Books (1992).

           Simon, J.L., Bretagnon, P., Chapront, J., Chapront-Touze, M.,
           Francou, G. & Laskar, J., Astron.Astrophys., 282, 663-683 (1994).

        This revision:  2018 January 2

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.dtdb(date1, date2, ut, elong, u, v)
    return c_retval


def dtf2d(scale, iy, im, id, ihr, imn, sec):
    """
    Encode date and time fields into 2-part Julian Date (or in the case
    of UTC a quasi-JD form that includes special provision for leap
    seconds).

    Parameters
    ----------
    scale : const char array
    iy : int array
    im : int array
    id : int array
    ihr : int array
    imn : int array
    sec : double array

    Returns
    -------
    d1 : double array
    d2 : double array

    Notes
    -----
    Wraps ERFA function ``eraDtf2d``. The ERFA documentation is::

        - - - - - - - - -
         e r a D t f 2 d
        - - - - - - - - -

        Encode date and time fields into 2-part Julian Date (or in the case
        of UTC a quasi-JD form that includes special provision for leap
        seconds).

        Given:
           scale     char[]  time scale ID (Note 1)
           iy,im,id  int     year, month, day in Gregorian calendar (Note 2)
           ihr,imn   int     hour, minute
           sec       double  seconds

        Returned:
           d1,d2     double  2-part Julian Date (Notes 3,4)

        Returned (function value):
                     int     status: +3 = both of next two
                                     +2 = time is after end of day (Note 5)
                                     +1 = dubious year (Note 6)
                                      0 = OK
                                     -1 = bad year
                                     -2 = bad month
                                     -3 = bad day
                                     -4 = bad hour
                                     -5 = bad minute
                                     -6 = bad second (<0)

        Notes:

        1) scale identifies the time scale.  Only the value "UTC" (in upper
           case) is significant, and enables handling of leap seconds (see
           Note 4).

        2) For calendar conventions and limitations, see eraCal2jd.

        3) The sum of the results, d1+d2, is Julian Date, where normally d1
           is the Julian Day Number and d2 is the fraction of a day.  In the
           case of UTC, where the use of JD is problematical, special
           conventions apply:  see the next note.

        4) JD cannot unambiguously represent UTC during a leap second unless
           special measures are taken.  The ERFA internal convention is that
           the quasi-JD day represents UTC days whether the length is 86399,
           86400 or 86401 SI seconds.  In the 1960-1972 era there were
           smaller jumps (in either direction) each time the linear UTC(TAI)
           expression was changed, and these "mini-leaps" are also included
           in the ERFA convention.

        5) The warning status "time is after end of day" usually means that
           the sec argument is greater than 60.0.  However, in a day ending
           in a leap second the limit changes to 61.0 (or 59.0 in the case
           of a negative leap second).

        6) The warning status "dubious year" flags UTCs that predate the
           introduction of the time scale or that are too far in the future
           to be trusted.  See eraDat for further details.

        7) Only in the case of continuous and regular time scales (TAI, TT,
           TCG, TCB and TDB) is the result d1+d2 a Julian Date, strictly
           speaking.  In the other cases (UT1 and UTC) the result must be
           used with circumspection;  in particular the difference between
           two such results cannot be interpreted as a precise time
           interval.

        Called:
           eraCal2jd    Gregorian calendar to JD
           eraDat       delta(AT) = TAI-UTC
           eraJd2cal    JD to Gregorian calendar

        This revision:  2013 July 26

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    d1, d2, c_retval = ufunc.dtf2d(scale, iy, im, id, ihr, imn, sec)
    check_errwarn(c_retval, 'dtf2d')
    return d1, d2


STATUS_CODES['dtf2d'] = {
    3: 'both of next two',
    2: 'time is after end of day (Note 5)',
    1: 'dubious year (Note 6)',
    0: 'OK',
    -1: 'bad year',
    -2: 'bad month',
    -3: 'bad day',
    -4: 'bad hour',
    -5: 'bad minute',
    -6: 'bad second (<0)',
}


def taitt(tai1, tai2):
    """
    Time scale transformation:  International Atomic Time, TAI, to
    Terrestrial Time, TT.

    Parameters
    ----------
    tai1 : double array
    tai2 : double array

    Returns
    -------
    tt1 : double array
    tt2 : double array

    Notes
    -----
    Wraps ERFA function ``eraTaitt``. The ERFA documentation is::

        - - - - - - - - -
         e r a T a i t t
        - - - - - - - - -

        Time scale transformation:  International Atomic Time, TAI, to
        Terrestrial Time, TT.

        Given:
           tai1,tai2  double    TAI as a 2-part Julian Date

        Returned:
           tt1,tt2    double    TT as a 2-part Julian Date

        Returned (function value):
                      int       status:  0 = OK

        Note:

           tai1+tai2 is Julian Date, apportioned in any convenient way
           between the two arguments, for example where tai1 is the Julian
           Day Number and tai2 is the fraction of a day.  The returned
           tt1,tt2 follow suit.

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992)

        This revision:  2019 June 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    tt1, tt2, c_retval = ufunc.taitt(tai1, tai2)
    check_errwarn(c_retval, 'taitt')
    return tt1, tt2


STATUS_CODES['taitt'] = {
    0: 'OK',
}


def taiut1(tai1, tai2, dta):
    """
    Time scale transformation:  International Atomic Time, TAI, to
    Universal Time, UT1.

    Parameters
    ----------
    tai1 : double array
    tai2 : double array
    dta : double array

    Returns
    -------
    ut11 : double array
    ut12 : double array

    Notes
    -----
    Wraps ERFA function ``eraTaiut1``. The ERFA documentation is::

        - - - - - - - - - -
         e r a T a i u t 1
        - - - - - - - - - -

        Time scale transformation:  International Atomic Time, TAI, to
        Universal Time, UT1.

        Given:
           tai1,tai2  double    TAI as a 2-part Julian Date
           dta        double    UT1-TAI in seconds

        Returned:
           ut11,ut12  double    UT1 as a 2-part Julian Date

        Returned (function value):
                      int       status:  0 = OK

        Notes:

        1) tai1+tai2 is Julian Date, apportioned in any convenient way
           between the two arguments, for example where tai1 is the Julian
           Day Number and tai2 is the fraction of a day.  The returned
           UT11,UT12 follow suit.

        2) The argument dta, i.e. UT1-TAI, is an observed quantity, and is
           available from IERS tabulations.

        Reference:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992)

        This revision:  2019 June 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    ut11, ut12, c_retval = ufunc.taiut1(tai1, tai2, dta)
    check_errwarn(c_retval, 'taiut1')
    return ut11, ut12


STATUS_CODES['taiut1'] = {
    0: 'OK',
}


def taiutc(tai1, tai2):
    """
    Time scale transformation:  International Atomic Time, TAI, to
    Coordinated Universal Time, UTC.

    Parameters
    ----------
    tai1 : double array
    tai2 : double array

    Returns
    -------
    utc1 : double array
    utc2 : double array

    Notes
    -----
    Wraps ERFA function ``eraTaiutc``. The ERFA documentation is::

        - - - - - - - - - -
         e r a T a i u t c
        - - - - - - - - - -

        Time scale transformation:  International Atomic Time, TAI, to
        Coordinated Universal Time, UTC.

        Given:
           tai1,tai2  double   TAI as a 2-part Julian Date (Note 1)

        Returned:
           utc1,utc2  double   UTC as a 2-part quasi Julian Date (Notes 1-3)

        Returned (function value):
                      int      status: +1 = dubious year (Note 4)
                                        0 = OK
                                       -1 = unacceptable date

        Notes:

        1) tai1+tai2 is Julian Date, apportioned in any convenient way
           between the two arguments, for example where tai1 is the Julian
           Day Number and tai2 is the fraction of a day.  The returned utc1
           and utc2 form an analogous pair, except that a special convention
           is used, to deal with the problem of leap seconds - see the next
           note.

        2) JD cannot unambiguously represent UTC during a leap second unless
           special measures are taken.  The convention in the present
           function is that the JD day represents UTC days whether the
           length is 86399, 86400 or 86401 SI seconds.  In the 1960-1972 era
           there were smaller jumps (in either direction) each time the
           linear UTC(TAI) expression was changed, and these "mini-leaps"
           are also included in the ERFA convention.

        3) The function eraD2dtf can be used to transform the UTC quasi-JD
           into calendar date and clock time, including UTC leap second
           handling.

        4) The warning status "dubious year" flags UTCs that predate the
           introduction of the time scale or that are too far in the future
           to be trusted.  See eraDat for further details.

        Called:
           eraUtctai    UTC to TAI

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992)

        This revision:  2019 June 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    utc1, utc2, c_retval = ufunc.taiutc(tai1, tai2)
    check_errwarn(c_retval, 'taiutc')
    return utc1, utc2


STATUS_CODES['taiutc'] = {
    1: 'dubious year (Note 4)',
    0: 'OK',
    -1: 'unacceptable date',
}


def tcbtdb(tcb1, tcb2):
    """
    Time scale transformation:  Barycentric Coordinate Time, TCB, to
    Barycentric Dynamical Time, TDB.

    Parameters
    ----------
    tcb1 : double array
    tcb2 : double array

    Returns
    -------
    tdb1 : double array
    tdb2 : double array

    Notes
    -----
    Wraps ERFA function ``eraTcbtdb``. The ERFA documentation is::

        - - - - - - - - - -
         e r a T c b t d b
        - - - - - - - - - -

        Time scale transformation:  Barycentric Coordinate Time, TCB, to
        Barycentric Dynamical Time, TDB.

        Given:
           tcb1,tcb2  double    TCB as a 2-part Julian Date

        Returned:
           tdb1,tdb2  double    TDB as a 2-part Julian Date

        Returned (function value):
                      int       status:  0 = OK

        Notes:

        1) tcb1+tcb2 is Julian Date, apportioned in any convenient way
           between the two arguments, for example where tcb1 is the Julian
           Day Number and tcb2 is the fraction of a day.  The returned
           tdb1,tdb2 follow suit.

        2) The 2006 IAU General Assembly introduced a conventional linear
           transformation between TDB and TCB.  This transformation
           compensates for the drift between TCB and terrestrial time TT,
           and keeps TDB approximately centered on TT.  Because the
           relationship between TT and TCB depends on the adopted solar
           system ephemeris, the degree of alignment between TDB and TT over
           long intervals will vary according to which ephemeris is used.
           Former definitions of TDB attempted to avoid this problem by
           stipulating that TDB and TT should differ only by periodic
           effects.  This is a good description of the nature of the
           relationship but eluded precise mathematical formulation.  The
           conventional linear relationship adopted in 2006 sidestepped
           these difficulties whilst delivering a TDB that in practice was
           consistent with values before that date.

        3) TDB is essentially the same as Teph, the time argument for the
           JPL solar system ephemerides.

        Reference:

           IAU 2006 Resolution B3

        This revision:  2019 June 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    tdb1, tdb2, c_retval = ufunc.tcbtdb(tcb1, tcb2)
    check_errwarn(c_retval, 'tcbtdb')
    return tdb1, tdb2


STATUS_CODES['tcbtdb'] = {
    0: 'OK',
}


def tcgtt(tcg1, tcg2):
    """
    Time scale transformation:  Geocentric Coordinate Time, TCG, to
    Terrestrial Time, TT.

    Parameters
    ----------
    tcg1 : double array
    tcg2 : double array

    Returns
    -------
    tt1 : double array
    tt2 : double array

    Notes
    -----
    Wraps ERFA function ``eraTcgtt``. The ERFA documentation is::

        - - - - - - - - -
         e r a T c g t t
        - - - - - - - - -

        Time scale transformation:  Geocentric Coordinate Time, TCG, to
        Terrestrial Time, TT.

        Given:
           tcg1,tcg2  double    TCG as a 2-part Julian Date

        Returned:
           tt1,tt2    double    TT as a 2-part Julian Date

        Returned (function value):
                      int       status:  0 = OK

        Note:

           tcg1+tcg2 is Julian Date, apportioned in any convenient way
           between the two arguments, for example where tcg1 is the Julian
           Day Number and tcg22 is the fraction of a day.  The returned
           tt1,tt2 follow suit.

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           IAU 2000 Resolution B1.9

        This revision:  2020 October 23

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    tt1, tt2, c_retval = ufunc.tcgtt(tcg1, tcg2)
    check_errwarn(c_retval, 'tcgtt')
    return tt1, tt2


STATUS_CODES['tcgtt'] = {
    0: 'OK',
}


def tdbtcb(tdb1, tdb2):
    """
    Time scale transformation:  Barycentric Dynamical Time, TDB, to
    Barycentric Coordinate Time, TCB.

    Parameters
    ----------
    tdb1 : double array
    tdb2 : double array

    Returns
    -------
    tcb1 : double array
    tcb2 : double array

    Notes
    -----
    Wraps ERFA function ``eraTdbtcb``. The ERFA documentation is::

        - - - - - - - - - -
         e r a T d b t c b
        - - - - - - - - - -

        Time scale transformation:  Barycentric Dynamical Time, TDB, to
        Barycentric Coordinate Time, TCB.

        Given:
           tdb1,tdb2  double    TDB as a 2-part Julian Date

        Returned:
           tcb1,tcb2  double    TCB as a 2-part Julian Date

        Returned (function value):
                      int       status:  0 = OK

        Notes:

        1) tdb1+tdb2 is Julian Date, apportioned in any convenient way
           between the two arguments, for example where tdb1 is the Julian
           Day Number and tdb2 is the fraction of a day.  The returned
           tcb1,tcb2 follow suit.

        2) The 2006 IAU General Assembly introduced a conventional linear
           transformation between TDB and TCB.  This transformation
           compensates for the drift between TCB and terrestrial time TT,
           and keeps TDB approximately centered on TT.  Because the
           relationship between TT and TCB depends on the adopted solar
           system ephemeris, the degree of alignment between TDB and TT over
           long intervals will vary according to which ephemeris is used.
           Former definitions of TDB attempted to avoid this problem by
           stipulating that TDB and TT should differ only by periodic
           effects.  This is a good description of the nature of the
           relationship but eluded precise mathematical formulation.  The
           conventional linear relationship adopted in 2006 sidestepped
           these difficulties whilst delivering a TDB that in practice was
           consistent with values before that date.

        3) TDB is essentially the same as Teph, the time argument for the
           JPL solar system ephemerides.

        Reference:

           IAU 2006 Resolution B3

        This revision:  2019 June 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    tcb1, tcb2, c_retval = ufunc.tdbtcb(tdb1, tdb2)
    check_errwarn(c_retval, 'tdbtcb')
    return tcb1, tcb2


STATUS_CODES['tdbtcb'] = {
    0: 'OK',
}


def tdbtt(tdb1, tdb2, dtr):
    """
    Time scale transformation:  Barycentric Dynamical Time, TDB, to
    Terrestrial Time, TT.

    Parameters
    ----------
    tdb1 : double array
    tdb2 : double array
    dtr : double array

    Returns
    -------
    tt1 : double array
    tt2 : double array

    Notes
    -----
    Wraps ERFA function ``eraTdbtt``. The ERFA documentation is::

        - - - - - - - - -
         e r a T d b t t
        - - - - - - - - -

        Time scale transformation:  Barycentric Dynamical Time, TDB, to
        Terrestrial Time, TT.

        Given:
           tdb1,tdb2  double    TDB as a 2-part Julian Date
           dtr        double    TDB-TT in seconds

        Returned:
           tt1,tt2    double    TT as a 2-part Julian Date

        Returned (function value):
                      int       status:  0 = OK

        Notes:

        1) tdb1+tdb2 is Julian Date, apportioned in any convenient way
           between the two arguments, for example where tdb1 is the Julian
           Day Number and tdb2 is the fraction of a day.  The returned
           tt1,tt2 follow suit.

        2) The argument dtr represents the quasi-periodic component of the
           GR transformation between TT and TCB.  It is dependent upon the
           adopted solar-system ephemeris, and can be obtained by numerical
           integration, by interrogating a precomputed time ephemeris or by
           evaluating a model such as that implemented in the ERFA function
           eraDtdb.   The quantity is dominated by an annual term of 1.7 ms
           amplitude.

        3) TDB is essentially the same as Teph, the time argument for the
           JPL solar system ephemerides.

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           IAU 2006 Resolution 3

        This revision:  2019 June 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    tt1, tt2, c_retval = ufunc.tdbtt(tdb1, tdb2, dtr)
    check_errwarn(c_retval, 'tdbtt')
    return tt1, tt2


STATUS_CODES['tdbtt'] = {
    0: 'OK',
}


def tttai(tt1, tt2):
    """
    Time scale transformation:  Terrestrial Time, TT, to International
    Atomic Time, TAI.

    Parameters
    ----------
    tt1 : double array
    tt2 : double array

    Returns
    -------
    tai1 : double array
    tai2 : double array

    Notes
    -----
    Wraps ERFA function ``eraTttai``. The ERFA documentation is::

        - - - - - - - - -
         e r a T t t a i
        - - - - - - - - -

        Time scale transformation:  Terrestrial Time, TT, to International
        Atomic Time, TAI.

        Given:
           tt1,tt2    double    TT as a 2-part Julian Date

        Returned:
           tai1,tai2  double    TAI as a 2-part Julian Date

        Returned (function value):
                      int       status:  0 = OK

        Note:

           tt1+tt2 is Julian Date, apportioned in any convenient way between
           the two arguments, for example where tt1 is the Julian Day Number
           and tt2 is the fraction of a day.  The returned tai1,tai2 follow
           suit.

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992)

        This revision:  2019 June 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    tai1, tai2, c_retval = ufunc.tttai(tt1, tt2)
    check_errwarn(c_retval, 'tttai')
    return tai1, tai2


STATUS_CODES['tttai'] = {
    0: 'OK',
}


def tttcg(tt1, tt2):
    """
    Time scale transformation:  Terrestrial Time, TT, to Geocentric
    Coordinate Time, TCG.

    Parameters
    ----------
    tt1 : double array
    tt2 : double array

    Returns
    -------
    tcg1 : double array
    tcg2 : double array

    Notes
    -----
    Wraps ERFA function ``eraTttcg``. The ERFA documentation is::

        - - - - - - - - -
         e r a T t t c g
        - - - - - - - - -

        Time scale transformation:  Terrestrial Time, TT, to Geocentric
        Coordinate Time, TCG.

        Given:
           tt1,tt2    double    TT as a 2-part Julian Date

        Returned:
           tcg1,tcg2  double    TCG as a 2-part Julian Date

        Returned (function value):
                      int       status:  0 = OK

        Note:

           tt1+tt2 is Julian Date, apportioned in any convenient way between
           the two arguments, for example where tt1 is the Julian Day Number
           and tt2 is the fraction of a day.  The returned tcg1,tcg2 follow
           suit.

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           IAU 2000 Resolution B1.9

        This revision:  2019 June 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    tcg1, tcg2, c_retval = ufunc.tttcg(tt1, tt2)
    check_errwarn(c_retval, 'tttcg')
    return tcg1, tcg2


STATUS_CODES['tttcg'] = {
    0: 'OK',
}


def tttdb(tt1, tt2, dtr):
    """
    Time scale transformation:  Terrestrial Time, TT, to Barycentric
    Dynamical Time, TDB.

    Parameters
    ----------
    tt1 : double array
    tt2 : double array
    dtr : double array

    Returns
    -------
    tdb1 : double array
    tdb2 : double array

    Notes
    -----
    Wraps ERFA function ``eraTttdb``. The ERFA documentation is::

        - - - - - - - - -
         e r a T t t d b
        - - - - - - - - -

        Time scale transformation:  Terrestrial Time, TT, to Barycentric
        Dynamical Time, TDB.

        Given:
           tt1,tt2    double    TT as a 2-part Julian Date
           dtr        double    TDB-TT in seconds

        Returned:
           tdb1,tdb2  double    TDB as a 2-part Julian Date

        Returned (function value):
                      int       status:  0 = OK

        Notes:

        1) tt1+tt2 is Julian Date, apportioned in any convenient way between
           the two arguments, for example where tt1 is the Julian Day Number
           and tt2 is the fraction of a day.  The returned tdb1,tdb2 follow
           suit.

        2) The argument dtr represents the quasi-periodic component of the
           GR transformation between TT and TCB.  It is dependent upon the
           adopted solar-system ephemeris, and can be obtained by numerical
           integration, by interrogating a precomputed time ephemeris or by
           evaluating a model such as that implemented in the ERFA function
           eraDtdb.   The quantity is dominated by an annual term of 1.7 ms
           amplitude.

        3) TDB is essentially the same as Teph, the time argument for the JPL
           solar system ephemerides.

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           IAU 2006 Resolution 3

        This revision:  2019 June 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    tdb1, tdb2, c_retval = ufunc.tttdb(tt1, tt2, dtr)
    check_errwarn(c_retval, 'tttdb')
    return tdb1, tdb2


STATUS_CODES['tttdb'] = {
    0: 'OK',
}


def ttut1(tt1, tt2, dt):
    """
    Time scale transformation:  Terrestrial Time, TT, to Universal Time,
    UT1.

    Parameters
    ----------
    tt1 : double array
    tt2 : double array
    dt : double array

    Returns
    -------
    ut11 : double array
    ut12 : double array

    Notes
    -----
    Wraps ERFA function ``eraTtut1``. The ERFA documentation is::

        - - - - - - - - -
         e r a T t u t 1
        - - - - - - - - -

        Time scale transformation:  Terrestrial Time, TT, to Universal Time,
        UT1.

        Given:
           tt1,tt2    double    TT as a 2-part Julian Date
           dt         double    TT-UT1 in seconds

        Returned:
           ut11,ut12  double    UT1 as a 2-part Julian Date

        Returned (function value):
                      int       status:  0 = OK

        Notes:

        1) tt1+tt2 is Julian Date, apportioned in any convenient way between
           the two arguments, for example where tt1 is the Julian Day Number
           and tt2 is the fraction of a day.  The returned ut11,ut12 follow
           suit.

        2) The argument dt is classical Delta T.

        Reference:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992)

        This revision:  2019 June 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    ut11, ut12, c_retval = ufunc.ttut1(tt1, tt2, dt)
    check_errwarn(c_retval, 'ttut1')
    return ut11, ut12


STATUS_CODES['ttut1'] = {
    0: 'OK',
}


def ut1tai(ut11, ut12, dta):
    """
    Time scale transformation:  Universal Time, UT1, to International
    Atomic Time, TAI.

    Parameters
    ----------
    ut11 : double array
    ut12 : double array
    dta : double array

    Returns
    -------
    tai1 : double array
    tai2 : double array

    Notes
    -----
    Wraps ERFA function ``eraUt1tai``. The ERFA documentation is::

        - - - - - - - - - -
         e r a U t 1 t a i
        - - - - - - - - - -

        Time scale transformation:  Universal Time, UT1, to International
        Atomic Time, TAI.

        Given:
           ut11,ut12  double    UT1 as a 2-part Julian Date
           dta        double    UT1-TAI in seconds

        Returned:
           tai1,tai2  double    TAI as a 2-part Julian Date

        Returned (function value):
                      int       status:  0 = OK

        Notes:

        1) ut11+ut12 is Julian Date, apportioned in any convenient way
           between the two arguments, for example where ut11 is the Julian
           Day Number and ut12 is the fraction of a day.  The returned
           tai1,tai2 follow suit.

        2) The argument dta, i.e. UT1-TAI, is an observed quantity, and is
           available from IERS tabulations.

        Reference:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992)

        This revision:  2019 June 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    tai1, tai2, c_retval = ufunc.ut1tai(ut11, ut12, dta)
    check_errwarn(c_retval, 'ut1tai')
    return tai1, tai2


STATUS_CODES['ut1tai'] = {
    0: 'OK',
}


def ut1tt(ut11, ut12, dt):
    """
    Time scale transformation:  Universal Time, UT1, to Terrestrial
    Time, TT.

    Parameters
    ----------
    ut11 : double array
    ut12 : double array
    dt : double array

    Returns
    -------
    tt1 : double array
    tt2 : double array

    Notes
    -----
    Wraps ERFA function ``eraUt1tt``. The ERFA documentation is::

        - - - - - - - - -
         e r a U t 1 t t
        - - - - - - - - -

        Time scale transformation:  Universal Time, UT1, to Terrestrial
        Time, TT.

        Given:
           ut11,ut12  double    UT1 as a 2-part Julian Date
           dt         double    TT-UT1 in seconds

        Returned:
           tt1,tt2    double    TT as a 2-part Julian Date

        Returned (function value):
                      int       status:  0 = OK

        Notes:

        1) ut11+ut12 is Julian Date, apportioned in any convenient way
           between the two arguments, for example where ut11 is the Julian
           Day Number and ut12 is the fraction of a day.  The returned
           tt1,tt2 follow suit.

        2) The argument dt is classical Delta T.

        Reference:

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992)

        This revision:  2019 June 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    tt1, tt2, c_retval = ufunc.ut1tt(ut11, ut12, dt)
    check_errwarn(c_retval, 'ut1tt')
    return tt1, tt2


STATUS_CODES['ut1tt'] = {
    0: 'OK',
}


def ut1utc(ut11, ut12, dut1):
    """
    Time scale transformation:  Universal Time, UT1, to Coordinated
    Universal Time, UTC.

    Parameters
    ----------
    ut11 : double array
    ut12 : double array
    dut1 : double array

    Returns
    -------
    utc1 : double array
    utc2 : double array

    Notes
    -----
    Wraps ERFA function ``eraUt1utc``. The ERFA documentation is::

        - - - - - - - - - -
         e r a U t 1 u t c
        - - - - - - - - - -

        Time scale transformation:  Universal Time, UT1, to Coordinated
        Universal Time, UTC.

        Given:
           ut11,ut12  double   UT1 as a 2-part Julian Date (Note 1)
           dut1       double   Delta UT1: UT1-UTC in seconds (Note 2)

        Returned:
           utc1,utc2  double   UTC as a 2-part quasi Julian Date (Notes 3,4)

        Returned (function value):
                      int      status: +1 = dubious year (Note 5)
                                        0 = OK
                                       -1 = unacceptable date

        Notes:

        1) ut11+ut12 is Julian Date, apportioned in any convenient way
           between the two arguments, for example where ut11 is the Julian
           Day Number and ut12 is the fraction of a day.  The returned utc1
           and utc2 form an analogous pair, except that a special convention
           is used, to deal with the problem of leap seconds - see Note 3.

        2) Delta UT1 can be obtained from tabulations provided by the
           International Earth Rotation and Reference Systems Service.  The
           value changes abruptly by 1s at a leap second;  however, close to
           a leap second the algorithm used here is tolerant of the "wrong"
           choice of value being made.

        3) JD cannot unambiguously represent UTC during a leap second unless
           special measures are taken.  The convention in the present
           function is that the returned quasi-JD UTC1+UTC2 represents UTC
           days whether the length is 86399, 86400 or 86401 SI seconds.

        4) The function eraD2dtf can be used to transform the UTC quasi-JD
           into calendar date and clock time, including UTC leap second
           handling.

        5) The warning status "dubious year" flags UTCs that predate the
           introduction of the time scale or that are too far in the future
           to be trusted.  See eraDat for further details.

        Called:
           eraJd2cal    JD to Gregorian calendar
           eraDat       delta(AT) = TAI-UTC
           eraCal2jd    Gregorian calendar to JD

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992)

        This revision:  2020 October 25

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    utc1, utc2, c_retval = ufunc.ut1utc(ut11, ut12, dut1)
    check_errwarn(c_retval, 'ut1utc')
    return utc1, utc2


STATUS_CODES['ut1utc'] = {
    1: 'dubious year (Note 5)',
    0: 'OK',
    -1: 'unacceptable date',
}


def utctai(utc1, utc2):
    """
    Time scale transformation:  Coordinated Universal Time, UTC, to
    International Atomic Time, TAI.

    Parameters
    ----------
    utc1 : double array
    utc2 : double array

    Returns
    -------
    tai1 : double array
    tai2 : double array

    Notes
    -----
    Wraps ERFA function ``eraUtctai``. The ERFA documentation is::

        - - - - - - - - - -
         e r a U t c t a i
        - - - - - - - - - -

        Time scale transformation:  Coordinated Universal Time, UTC, to
        International Atomic Time, TAI.

        Given:
           utc1,utc2  double   UTC as a 2-part quasi Julian Date (Notes 1-4)

        Returned:
           tai1,tai2  double   TAI as a 2-part Julian Date (Note 5)

        Returned (function value):
                      int      status: +1 = dubious year (Note 3)
                                        0 = OK
                                       -1 = unacceptable date

        Notes:

        1) utc1+utc2 is quasi Julian Date (see Note 2), apportioned in any
           convenient way between the two arguments, for example where utc1
           is the Julian Day Number and utc2 is the fraction of a day.

        2) JD cannot unambiguously represent UTC during a leap second unless
           special measures are taken.  The convention in the present
           function is that the JD day represents UTC days whether the
           length is 86399, 86400 or 86401 SI seconds.  In the 1960-1972 era
           there were smaller jumps (in either direction) each time the
           linear UTC(TAI) expression was changed, and these "mini-leaps"
           are also included in the ERFA convention.

        3) The warning status "dubious year" flags UTCs that predate the
           introduction of the time scale or that are too far in the future
           to be trusted.  See eraDat for further details.

        4) The function eraDtf2d converts from calendar date and time of day
           into 2-part Julian Date, and in the case of UTC implements the
           leap-second-ambiguity convention described above.

        5) The returned TAI1,TAI2 are such that their sum is the TAI Julian
           Date.

        Called:
           eraJd2cal    JD to Gregorian calendar
           eraDat       delta(AT) = TAI-UTC
           eraCal2jd    Gregorian calendar to JD

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992)

        This revision:  2019 June 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    tai1, tai2, c_retval = ufunc.utctai(utc1, utc2)
    check_errwarn(c_retval, 'utctai')
    return tai1, tai2


STATUS_CODES['utctai'] = {
    1: 'dubious year (Note 3)',
    0: 'OK',
    -1: 'unacceptable date',
}


def utcut1(utc1, utc2, dut1):
    """
    Time scale transformation:  Coordinated Universal Time, UTC, to
    Universal Time, UT1.

    Parameters
    ----------
    utc1 : double array
    utc2 : double array
    dut1 : double array

    Returns
    -------
    ut11 : double array
    ut12 : double array

    Notes
    -----
    Wraps ERFA function ``eraUtcut1``. The ERFA documentation is::

        - - - - - - - - - -
         e r a U t c u t 1
        - - - - - - - - - -

        Time scale transformation:  Coordinated Universal Time, UTC, to
        Universal Time, UT1.

        Given:
           utc1,utc2  double   UTC as a 2-part quasi Julian Date (Notes 1-4)
           dut1       double   Delta UT1 = UT1-UTC in seconds (Note 5)

        Returned:
           ut11,ut12  double   UT1 as a 2-part Julian Date (Note 6)

        Returned (function value):
                      int      status: +1 = dubious year (Note 3)
                                        0 = OK
                                       -1 = unacceptable date

        Notes:

        1) utc1+utc2 is quasi Julian Date (see Note 2), apportioned in any
           convenient way between the two arguments, for example where utc1
           is the Julian Day Number and utc2 is the fraction of a day.

        2) JD cannot unambiguously represent UTC during a leap second unless
           special measures are taken.  The convention in the present
           function is that the JD day represents UTC days whether the
           length is 86399, 86400 or 86401 SI seconds.

        3) The warning status "dubious year" flags UTCs that predate the
           introduction of the time scale or that are too far in the future
           to be trusted.  See eraDat for further details.

        4) The function eraDtf2d converts from calendar date and time of
           day into 2-part Julian Date, and in the case of UTC implements
           the leap-second-ambiguity convention described above.

        5) Delta UT1 can be obtained from tabulations provided by the
           International Earth Rotation and Reference Systems Service.
           It is the caller's responsibility to supply a dut1 argument
           containing the UT1-UTC value that matches the given UTC.

        6) The returned ut11,ut12 are such that their sum is the UT1 Julian
           Date.

        References:

           McCarthy, D. D., Petit, G. (eds.), IERS Conventions (2003),
           IERS Technical Note No. 32, BKG (2004)

           Explanatory Supplement to the Astronomical Almanac,
           P. Kenneth Seidelmann (ed), University Science Books (1992)

        Called:
           eraJd2cal    JD to Gregorian calendar
           eraDat       delta(AT) = TAI-UTC
           eraUtctai    UTC to TAI
           eraTaiut1    TAI to UT1

        This revision:  2013 August 12

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    ut11, ut12, c_retval = ufunc.utcut1(utc1, utc2, dut1)
    check_errwarn(c_retval, 'utcut1')
    return ut11, ut12


STATUS_CODES['utcut1'] = {
    1: 'dubious year (Note 3)',
    0: 'OK',
    -1: 'unacceptable date',
}


def ae2hd(az, el, phi):
    """
    Horizon to equatorial coordinates:  transform azimuth and altitude
    to hour angle and declination.

    Parameters
    ----------
    az : double array
    el : double array
    phi : double array

    Returns
    -------
    ha : double array
    dec : double array

    Notes
    -----
    Wraps ERFA function ``eraAe2hd``. The ERFA documentation is::

        - - - - - - - - -
         e r a A e 2 h d
        - - - - - - - - -

        Horizon to equatorial coordinates:  transform azimuth and altitude
        to hour angle and declination.

        Given:
           az       double       azimuth
           el       double       altitude (informally, elevation)
           phi      double       site latitude

        Returned:
           ha       double       hour angle (local)
           dec      double       declination

        Notes:

        1)  All the arguments are angles in radians.

        2)  The sign convention for azimuth is north zero, east +pi/2.

        3)  HA is returned in the range +/-pi.  Declination is returned in
            the range +/-pi/2.

        4)  The latitude phi is pi/2 minus the angle between the Earth's
            rotation axis and the adopted zenith.  In many applications it
            will be sufficient to use the published geodetic latitude of the
            site.  In very precise (sub-arcsecond) applications, phi can be
            corrected for polar motion.

        5)  The azimuth az must be with respect to the rotational north pole,
            as opposed to the ITRS pole, and an azimuth with respect to north
            on a map of the Earth's surface will need to be adjusted for
            polar motion if sub-arcsecond accuracy is required.

        6)  Should the user wish to work with respect to the astronomical
            zenith rather than the geodetic zenith, phi will need to be
            adjusted for deflection of the vertical (often tens of
            arcseconds), and the zero point of ha will also be affected.

        7)  The transformation is the same as Ve = Ry(phi-pi/2)*Rz(pi)*Vh,
            where Ve and Vh are lefthanded unit vectors in the (ha,dec) and
            (az,el) systems respectively and Rz and Ry are rotations about
            first the z-axis and then the y-axis.  (n.b. Rz(pi) simply
            reverses the signs of the x and y components.)  For efficiency,
            the algorithm is written out rather than calling other utility
            functions.  For applications that require even greater
            efficiency, additional savings are possible if constant terms
            such as functions of latitude are computed once and for all.

        8)  Again for efficiency, no range checking of arguments is carried
            out.

        Last revision:   2017 September 12

        ERFA release 2021-01-25

        Copyright (C) 2021 IAU ERFA Board.  See notes at end.

    """
    ha, dec = ufunc.ae2hd(az, el, phi)
    return ha, dec


def hd2ae(ha, dec, phi):
    """
    Equatorial to horizon coordinates:  transform hour angle and
    declination to azimuth and altitude.

    Parameters
    ----------
    ha : double array
    dec : double array
    phi : double array

    Returns
    -------
    az : double array
    el : double array

    Notes
    -----
    Wraps ERFA function ``eraHd2ae``. The ERFA documentation is::

        - - - - - - - - -
         e r a H d 2 a e
        - - - - - - - - -

        Equatorial to horizon coordinates:  transform hour angle and
        declination to azimuth and altitude.

        Given:
           ha       double       hour angle (local)
           dec      double       declination
           phi      double       site latitude

        Returned:
           *az      double       azimuth
           *el      double       altitude (informally, elevation)

        Notes:

        1)  All the arguments are angles in radians.

        2)  Azimuth is returned in the range 0-2pi;  north is zero, and east
            is +pi/2.  Altitude is returned in the range +/- pi/2.

        3)  The latitude phi is pi/2 minus the angle between the Earth's
            rotation axis and the adopted zenith.  In many applications it
            will be sufficient to use the published geodetic latitude of the
            site.  In very precise (sub-arcsecond) applications, phi can be
            corrected for polar motion.

        4)  The returned azimuth az is with respect to the rotational north
            pole, as opposed to the ITRS pole, and for sub-arcsecond
            accuracy will need to be adjusted for polar motion if it is to
            be with respect to north on a map of the Earth's surface.

        5)  Should the user wish to work with respect to the astronomical
            zenith rather than the geodetic zenith, phi will need to be
            adjusted for deflection of the vertical (often tens of
            arcseconds), and the zero point of the hour angle ha will also
            be affected.

        6)  The transformation is the same as Vh = Rz(pi)*Ry(pi/2-phi)*Ve,
            where Vh and Ve are lefthanded unit vectors in the (az,el) and
            (ha,dec) systems respectively and Ry and Rz are rotations about
            first the y-axis and then the z-axis.  (n.b. Rz(pi) simply
            reverses the signs of the x and y components.)  For efficiency,
            the algorithm is written out rather than calling other utility
            functions.  For applications that require even greater
            efficiency, additional savings are possible if constant terms
            such as functions of latitude are computed once and for all.

        7)  Again for efficiency, no range checking of arguments is carried
            out.

        Last revision:   2017 September 12

        ERFA release 2021-01-25

        Copyright (C) 2021 IAU ERFA Board.  See notes at end.

    """
    az, el = ufunc.hd2ae(ha, dec, phi)
    return az, el


def hd2pa(ha, dec, phi):
    """
    Parallactic angle for a given hour angle and declination.

    Parameters
    ----------
    ha : double array
    dec : double array
    phi : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraHd2pa``. The ERFA documentation is::

        - - - - - - - - -
         e r a H d 2 p a
        - - - - - - - - -

        Parallactic angle for a given hour angle and declination.

        Given:
           ha     double     hour angle
           dec    double     declination
           phi    double     site latitude

        Returned (function value):
                  double     parallactic angle

        Notes:

        1)  All the arguments are angles in radians.

        2)  The parallactic angle at a point in the sky is the position
            angle of the vertical, i.e. the angle between the directions to
            the north celestial pole and to the zenith respectively.

        3)  The result is returned in the range -pi to +pi.

        4)  At the pole itself a zero result is returned.

        5)  The latitude phi is pi/2 minus the angle between the Earth's
            rotation axis and the adopted zenith.  In many applications it
            will be sufficient to use the published geodetic latitude of the
            site.  In very precise (sub-arcsecond) applications, phi can be
            corrected for polar motion.

        6)  Should the user wish to work with respect to the astronomical
            zenith rather than the geodetic zenith, phi will need to be
            adjusted for deflection of the vertical (often tens of
            arcseconds), and the zero point of the hour angle ha will also
            be affected.

        Reference:
           Smart, W.M., "Spherical Astronomy", Cambridge University Press,
           6th edition (Green, 1977), p49.

        Last revision:   2017 September 12

        ERFA release 2021-01-25

        Copyright (C) 2021 IAU ERFA Board.  See notes at end.

    """
    c_retval = ufunc.hd2pa(ha, dec, phi)
    return c_retval


def tpors(xi, eta, a, b):
    """
    In the tangent plane projection, given the rectangular coordinates
    of a star and its spherical coordinates, determine the spherical
    coordinates of the tangent point.

    Parameters
    ----------
    xi : double array
    eta : double array
    a : double array
    b : double array

    Returns
    -------
    a01 : double array
    b01 : double array
    a02 : double array
    b02 : double array

    Notes
    -----
    Wraps ERFA function ``eraTpors``. The ERFA documentation is::

        - - - - - - - - -
         e r a T p o r s
        - - - - - - - - -

        In the tangent plane projection, given the rectangular coordinates
        of a star and its spherical coordinates, determine the spherical
        coordinates of the tangent point.

        Given:
           xi,eta     double  rectangular coordinates of star image (Note 2)
           a,b        double  star's spherical coordinates (Note 3)

        Returned:
           *a01,*b01  double  tangent point's spherical coordinates, Soln. 1
           *a02,*b02  double  tangent point's spherical coordinates, Soln. 2

        Returned (function value):
                      int     number of solutions:
                              0 = no solutions returned (Note 5)
                              1 = only the first solution is useful (Note 6)
                              2 = both solutions are useful (Note 6)

        Notes:

        1) The tangent plane projection is also called the "gnomonic
           projection" and the "central projection".

        2) The eta axis points due north in the adopted coordinate system.
           If the spherical coordinates are observed (RA,Dec), the tangent
           plane coordinates (xi,eta) are conventionally called the
           "standard coordinates".  If the spherical coordinates are with
           respect to a right-handed triad, (xi,eta) are also right-handed.
           The units of (xi,eta) are, effectively, radians at the tangent
           point.

        3) All angular arguments are in radians.

        4) The angles a01 and a02 are returned in the range 0-2pi.  The
           angles b01 and b02 are returned in the range +/-pi, but in the
           usual, non-pole-crossing, case, the range is +/-pi/2.

        5) Cases where there is no solution can arise only near the poles.
           For example, it is clearly impossible for a star at the pole
           itself to have a non-zero xi value, and hence it is meaningless
           to ask where the tangent point would have to be to bring about
           this combination of xi and dec.

        6) Also near the poles, cases can arise where there are two useful
           solutions.  The return value indicates whether the second of the
           two solutions returned is useful;  1 indicates only one useful
           solution, the usual case.

        7) The basis of the algorithm is to solve the spherical triangle PSC,
           where P is the north celestial pole, S is the star and C is the
           tangent point.  The spherical coordinates of the tangent point are
           [a0,b0];  writing rho^2 = (xi^2+eta^2) and r^2 = (1+rho^2), side c
           is then (pi/2-b), side p is sqrt(xi^2+eta^2) and side s (to be
           found) is (pi/2-b0).  Angle C is given by sin(C) = xi/rho and
           cos(C) = eta/rho.  Angle P (to be found) is the longitude
           difference between star and tangent point (a-a0).

        8) This function is a member of the following set:

               spherical      vector         solve for

               eraTpxes      eraTpxev         xi,eta
               eraTpsts      eraTpstv          star
             > eraTpors <    eraTporv         origin

        Called:
           eraAnp       normalize angle into range 0 to 2pi

        References:

           Calabretta M.R. & Greisen, E.W., 2002, "Representations of
           celestial coordinates in FITS", Astron.Astrophys. 395, 1077

           Green, R.M., "Spherical Astronomy", Cambridge University Press,
           1987, Chapter 13.

        This revision:   2018 January 2

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    a01, b01, a02, b02, c_retval = ufunc.tpors(xi, eta, a, b)
    check_errwarn(c_retval, 'tpors')
    return a01, b01, a02, b02


def tporv(xi, eta, v):
    """
    In the tangent plane projection, given the rectangular coordinates
    of a star and its direction cosines, determine the direction
    cosines of the tangent point.

    Parameters
    ----------
    xi : double array
    eta : double array
    v : double array

    Returns
    -------
    v01 : double array
    v02 : double array

    Notes
    -----
    Wraps ERFA function ``eraTporv``. The ERFA documentation is::

        - - - - - - - - -
         e r a T p o r v
        - - - - - - - - -

        In the tangent plane projection, given the rectangular coordinates
        of a star and its direction cosines, determine the direction
        cosines of the tangent point.

        Given:
           xi,eta   double    rectangular coordinates of star image (Note 2)
           v        double[3] star's direction cosines (Note 3)

        Returned:
           v01      double[3] tangent point's direction cosines, Solution 1
           v02      double[3] tangent point's direction cosines, Solution 2

        Returned (function value):
                      int     number of solutions:
                              0 = no solutions returned (Note 4)
                              1 = only the first solution is useful (Note 5)
                              2 = both solutions are useful (Note 5)

        Notes:

        1) The tangent plane projection is also called the "gnomonic
           projection" and the "central projection".

        2) The eta axis points due north in the adopted coordinate system.
           If the direction cosines represent observed (RA,Dec), the tangent
           plane coordinates (xi,eta) are conventionally called the
           "standard coordinates".  If the direction cosines are with
           respect to a right-handed triad, (xi,eta) are also right-handed.
           The units of (xi,eta) are, effectively, radians at the tangent
           point.

        3) The vector v must be of unit length or the result will be wrong.

        4) Cases where there is no solution can arise only near the poles.
           For example, it is clearly impossible for a star at the pole
           itself to have a non-zero xi value, and hence it is meaningless
           to ask where the tangent point would have to be.

        5) Also near the poles, cases can arise where there are two useful
           solutions.  The return value indicates whether the second of the
           two solutions returned is useful;  1 indicates only one useful
           solution, the usual case.

        6) The basis of the algorithm is to solve the spherical triangle
           PSC, where P is the north celestial pole, S is the star and C is
           the tangent point.  Calling the celestial spherical coordinates
           of the star and tangent point (a,b) and (a0,b0) respectively, and
           writing rho^2 = (xi^2+eta^2) and r^2 = (1+rho^2), and
           transforming the vector v into (a,b) in the normal way, side c is
           then (pi/2-b), side p is sqrt(xi^2+eta^2) and side s (to be
           found) is (pi/2-b0), while angle C is given by sin(C) = xi/rho
           and cos(C) = eta/rho;  angle P (to be found) is (a-a0).  After
           solving the spherical triangle, the result (a0,b0) can be
           expressed in vector form as v0.

        7) This function is a member of the following set:

               spherical      vector         solve for

               eraTpxes      eraTpxev         xi,eta
               eraTpsts      eraTpstv          star
               eraTpors    > eraTporv <       origin

        References:

           Calabretta M.R. & Greisen, E.W., 2002, "Representations of
           celestial coordinates in FITS", Astron.Astrophys. 395, 1077

           Green, R.M., "Spherical Astronomy", Cambridge University Press,
           1987, Chapter 13.

        This revision:   2018 January 2

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    v01, v02, c_retval = ufunc.tporv(xi, eta, v)
    check_errwarn(c_retval, 'tporv')
    return v01, v02


def tpsts(xi, eta, a0, b0):
    """
    In the tangent plane projection, given the star's rectangular
    coordinates and the spherical coordinates of the tangent point,
    solve for the spherical coordinates of the star.

    Parameters
    ----------
    xi : double array
    eta : double array
    a0 : double array
    b0 : double array

    Returns
    -------
    a : double array
    b : double array

    Notes
    -----
    Wraps ERFA function ``eraTpsts``. The ERFA documentation is::

        - - - - - - - - -
         e r a T p s t s
        - - - - - - - - -

        In the tangent plane projection, given the star's rectangular
        coordinates and the spherical coordinates of the tangent point,
        solve for the spherical coordinates of the star.

        Given:
           xi,eta    double  rectangular coordinates of star image (Note 2)
           a0,b0     double  tangent point's spherical coordinates

        Returned:
           *a,*b     double  star's spherical coordinates

        1) The tangent plane projection is also called the "gnomonic
           projection" and the "central projection".

        2) The eta axis points due north in the adopted coordinate system.
           If the spherical coordinates are observed (RA,Dec), the tangent
           plane coordinates (xi,eta) are conventionally called the
           "standard coordinates".  If the spherical coordinates are with
           respect to a right-handed triad, (xi,eta) are also right-handed.
           The units of (xi,eta) are, effectively, radians at the tangent
           point.

        3) All angular arguments are in radians.

        4) This function is a member of the following set:

               spherical      vector         solve for

               eraTpxes      eraTpxev         xi,eta
             > eraTpsts <    eraTpstv          star
               eraTpors      eraTporv         origin

        Called:
           eraAnp       normalize angle into range 0 to 2pi

        References:

           Calabretta M.R. & Greisen, E.W., 2002, "Representations of
           celestial coordinates in FITS", Astron.Astrophys. 395, 1077

           Green, R.M., "Spherical Astronomy", Cambridge University Press,
           1987, Chapter 13.

        This revision:   2018 January 2

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    a, b = ufunc.tpsts(xi, eta, a0, b0)
    return a, b


def tpstv(xi, eta, v0):
    """
    In the tangent plane projection, given the star's rectangular
    coordinates and the direction cosines of the tangent point, solve
    for the direction cosines of the star.

    Parameters
    ----------
    xi : double array
    eta : double array
    v0 : double array

    Returns
    -------
    v : double array

    Notes
    -----
    Wraps ERFA function ``eraTpstv``. The ERFA documentation is::

        - - - - - - - - -
         e r a T p s t v
        - - - - - - - - -

        In the tangent plane projection, given the star's rectangular
        coordinates and the direction cosines of the tangent point, solve
        for the direction cosines of the star.

        Given:
           xi,eta  double     rectangular coordinates of star image (Note 2)
           v0      double[3]  tangent point's direction cosines

        Returned:
           v       double[3]  star's direction cosines

        1) The tangent plane projection is also called the "gnomonic
           projection" and the "central projection".

        2) The eta axis points due north in the adopted coordinate system.
           If the direction cosines represent observed (RA,Dec), the tangent
           plane coordinates (xi,eta) are conventionally called the
           "standard coordinates".  If the direction cosines are with
           respect to a right-handed triad, (xi,eta) are also right-handed.
           The units of (xi,eta) are, effectively, radians at the tangent
           point.

        3) The method used is to complete the star vector in the (xi,eta)
           based triad and normalize it, then rotate the triad to put the
           tangent point at the pole with the x-axis aligned to zero
           longitude.  Writing (a0,b0) for the celestial spherical
           coordinates of the tangent point, the sequence of rotations is
           (b-pi/2) around the x-axis followed by (-a-pi/2) around the
           z-axis.

        4) If vector v0 is not of unit length, the returned vector v will
           be wrong.

        5) If vector v0 points at a pole, the returned vector v will be
           based on the arbitrary assumption that the longitude coordinate
           of the tangent point is zero.

        6) This function is a member of the following set:

               spherical      vector         solve for

               eraTpxes      eraTpxev         xi,eta
               eraTpsts    > eraTpstv <        star
               eraTpors      eraTporv         origin

        References:

           Calabretta M.R. & Greisen, E.W., 2002, "Representations of
           celestial coordinates in FITS", Astron.Astrophys. 395, 1077

           Green, R.M., "Spherical Astronomy", Cambridge University Press,
           1987, Chapter 13.

        This revision:   2018 January 2

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    v = ufunc.tpstv(xi, eta, v0)
    return v


def tpxes(a, b, a0, b0):
    """
    In the tangent plane projection, given celestial spherical
    coordinates for a star and the tangent point, solve for the star's
    rectangular coordinates in the tangent plane.

    Parameters
    ----------
    a : double array
    b : double array
    a0 : double array
    b0 : double array

    Returns
    -------
    xi : double array
    eta : double array

    Notes
    -----
    Wraps ERFA function ``eraTpxes``. The ERFA documentation is::

        - - - - - - - - -
         e r a T p x e s
        - - - - - - - - -

        In the tangent plane projection, given celestial spherical
        coordinates for a star and the tangent point, solve for the star's
        rectangular coordinates in the tangent plane.

        Given:
           a,b       double  star's spherical coordinates
           a0,b0     double  tangent point's spherical coordinates

        Returned:
           *xi,*eta  double  rectangular coordinates of star image (Note 2)

        Returned (function value):
                     int     status:  0 = OK
                                      1 = star too far from axis
                                      2 = antistar on tangent plane
                                      3 = antistar too far from axis

        Notes:

        1) The tangent plane projection is also called the "gnomonic
           projection" and the "central projection".

        2) The eta axis points due north in the adopted coordinate system.
           If the spherical coordinates are observed (RA,Dec), the tangent
           plane coordinates (xi,eta) are conventionally called the
           "standard coordinates".  For right-handed spherical coordinates,
           (xi,eta) are also right-handed.  The units of (xi,eta) are,
           effectively, radians at the tangent point.

        3) All angular arguments are in radians.

        4) This function is a member of the following set:

               spherical      vector         solve for

             > eraTpxes <    eraTpxev         xi,eta
               eraTpsts      eraTpstv          star
               eraTpors      eraTporv         origin

        References:

           Calabretta M.R. & Greisen, E.W., 2002, "Representations of
           celestial coordinates in FITS", Astron.Astrophys. 395, 1077

           Green, R.M., "Spherical Astronomy", Cambridge University Press,
           1987, Chapter 13.

        This revision:   2018 January 2

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    xi, eta, c_retval = ufunc.tpxes(a, b, a0, b0)
    check_errwarn(c_retval, 'tpxes')
    return xi, eta


STATUS_CODES['tpxes'] = {
    0: 'OK',
    1: 'star too far from axis',
    2: 'antistar on tangent plane',
    3: 'antistar too far from axis',
}


def tpxev(v, v0):
    """
    In the tangent plane projection, given celestial direction cosines
    for a star and the tangent point, solve for the star's rectangular
    coordinates in the tangent plane.

    Parameters
    ----------
    v : double array
    v0 : double array

    Returns
    -------
    xi : double array
    eta : double array

    Notes
    -----
    Wraps ERFA function ``eraTpxev``. The ERFA documentation is::

        - - - - - - - - -
         e r a T p x e v
        - - - - - - - - -

        In the tangent plane projection, given celestial direction cosines
        for a star and the tangent point, solve for the star's rectangular
        coordinates in the tangent plane.

        Given:
           v         double[3]  direction cosines of star (Note 4)
           v0        double[3]  direction cosines of tangent point (Note 4)

        Returned:
           *xi,*eta  double     tangent plane coordinates of star

        Returned (function value):
                     int        status: 0 = OK
                                        1 = star too far from axis
                                        2 = antistar on tangent plane
                                        3 = antistar too far from axis

        Notes:

        1) The tangent plane projection is also called the "gnomonic
           projection" and the "central projection".

        2) The eta axis points due north in the adopted coordinate system.
           If the direction cosines represent observed (RA,Dec), the tangent
           plane coordinates (xi,eta) are conventionally called the
           "standard coordinates".  If the direction cosines are with
           respect to a right-handed triad, (xi,eta) are also right-handed.
           The units of (xi,eta) are, effectively, radians at the tangent
           point.

        3) The method used is to extend the star vector to the tangent
           plane and then rotate the triad so that (x,y) becomes (xi,eta).
           Writing (a,b) for the celestial spherical coordinates of the
           star, the sequence of rotations is (a+pi/2) around the z-axis
           followed by (pi/2-b) around the x-axis.

        4) If vector v0 is not of unit length, or if vector v is of zero
           length, the results will be wrong.

        5) If v0 points at a pole, the returned (xi,eta) will be based on
           the arbitrary assumption that the longitude coordinate of the
           tangent point is zero.

        6) This function is a member of the following set:

               spherical      vector         solve for

               eraTpxes    > eraTpxev <       xi,eta
               eraTpsts      eraTpstv          star
               eraTpors      eraTporv         origin

        References:

           Calabretta M.R. & Greisen, E.W., 2002, "Representations of
           celestial coordinates in FITS", Astron.Astrophys. 395, 1077

           Green, R.M., "Spherical Astronomy", Cambridge University Press,
           1987, Chapter 13.

        This revision:   2018 January 2

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    xi, eta, c_retval = ufunc.tpxev(v, v0)
    check_errwarn(c_retval, 'tpxev')
    return xi, eta


STATUS_CODES['tpxev'] = {
    0: 'OK',
    1: 'star too far from axis',
    2: 'antistar on tangent plane',
    3: 'antistar too far from axis',
}


def a2af(ndp, angle):
    """
    Decompose radians into degrees, arcminutes, arcseconds, fraction.

    Parameters
    ----------
    ndp : int array
    angle : double array

    Returns
    -------
    sign : char array
    idmsf : int array

    Notes
    -----
    Wraps ERFA function ``eraA2af``. The ERFA documentation is::

        - - - - - - - -
         e r a A 2 a f
        - - - - - - - -

        Decompose radians into degrees, arcminutes, arcseconds, fraction.

        Given:
           ndp     int     resolution (Note 1)
           angle   double  angle in radians

        Returned:
           sign    char         '+' or '-'
           idmsf   int[4]  degrees, arcminutes, arcseconds, fraction

        Notes:

        1) The argument ndp is interpreted as follows:

           ndp         resolution
            :      ...0000 00 00
           -7         1000 00 00
           -6          100 00 00
           -5           10 00 00
           -4            1 00 00
           -3            0 10 00
           -2            0 01 00
           -1            0 00 10
            0            0 00 01
            1            0 00 00.1
            2            0 00 00.01
            3            0 00 00.001
            :            0 00 00.000...

        2) The largest positive useful value for ndp is determined by the
           size of angle, the format of doubles on the target platform, and
           the risk of overflowing idmsf[3].  On a typical platform, for
           angle up to 2pi, the available floating-point precision might
           correspond to ndp=12.  However, the practical limit is typically
           ndp=9, set by the capacity of a 32-bit int, or ndp=4 if int is
           only 16 bits.

        3) The absolute value of angle may exceed 2pi.  In cases where it
           does not, it is up to the caller to test for and handle the
           case where angle is very nearly 2pi and rounds up to 360 degrees,
           by testing for idmsf[0]=360 and setting idmsf[0-3] to zero.

        Called:
           eraD2tf      decompose days to hms

        This revision:  2020 April 1

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    sign, idmsf = ufunc.a2af(ndp, angle)
    sign = sign.view(dt_bytes1)
    return sign, idmsf


def a2tf(ndp, angle):
    """
    Decompose radians into hours, minutes, seconds, fraction.

    Parameters
    ----------
    ndp : int array
    angle : double array

    Returns
    -------
    sign : char array
    ihmsf : int array

    Notes
    -----
    Wraps ERFA function ``eraA2tf``. The ERFA documentation is::

        - - - - - - - -
         e r a A 2 t f
        - - - - - - - -

        Decompose radians into hours, minutes, seconds, fraction.

        Given:
           ndp     int     resolution (Note 1)
           angle   double  angle in radians

        Returned:
           sign    char         '+' or '-'
           ihmsf   int[4]  hours, minutes, seconds, fraction

        Notes:

        1) The argument ndp is interpreted as follows:

           ndp         resolution
            :      ...0000 00 00
           -7         1000 00 00
           -6          100 00 00
           -5           10 00 00
           -4            1 00 00
           -3            0 10 00
           -2            0 01 00
           -1            0 00 10
            0            0 00 01
            1            0 00 00.1
            2            0 00 00.01
            3            0 00 00.001
            :            0 00 00.000...

        2) The largest positive useful value for ndp is determined by the
           size of angle, the format of doubles on the target platform, and
           the risk of overflowing ihmsf[3].  On a typical platform, for
           angle up to 2pi, the available floating-point precision might
           correspond to ndp=12.  However, the practical limit is typically
           ndp=9, set by the capacity of a 32-bit int, or ndp=4 if int is
           only 16 bits.

        3) The absolute value of angle may exceed 2pi.  In cases where it
           does not, it is up to the caller to test for and handle the
           case where angle is very nearly 2pi and rounds up to 24 hours,
           by testing for ihmsf[0]=24 and setting ihmsf[0-3] to zero.

        Called:
           eraD2tf      decompose days to hms

        This revision:  2020 April 1

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    sign, ihmsf = ufunc.a2tf(ndp, angle)
    sign = sign.view(dt_bytes1)
    return sign, ihmsf


def af2a(s, ideg, iamin, asec):
    """
    Convert degrees, arcminutes, arcseconds to radians.

    Parameters
    ----------
    s : char array
    ideg : int array
    iamin : int array
    asec : double array

    Returns
    -------
    rad : double array

    Notes
    -----
    Wraps ERFA function ``eraAf2a``. The ERFA documentation is::

        - - - - - - - -
         e r a A f 2 a
        - - - - - - - -

        Convert degrees, arcminutes, arcseconds to radians.

        Given:
           s         char    sign:  '-' = negative, otherwise positive
           ideg      int     degrees
           iamin     int     arcminutes
           asec      double  arcseconds

        Returned:
           rad       double  angle in radians

        Returned (function value):
                     int     status:  0 = OK
                                      1 = ideg outside range 0-359
                                      2 = iamin outside range 0-59
                                      3 = asec outside range 0-59.999...

        Notes:

        1)  The result is computed even if any of the range checks fail.

        2)  Negative ideg, iamin and/or asec produce a warning status, but
            the absolute value is used in the conversion.

        3)  If there are multiple errors, the status value reflects only the
            first, the smallest taking precedence.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rad, c_retval = ufunc.af2a(s, ideg, iamin, asec)
    check_errwarn(c_retval, 'af2a')
    return rad


STATUS_CODES['af2a'] = {
    0: 'OK',
    1: 'ideg outside range 0-359',
    2: 'iamin outside range 0-59',
    3: 'asec outside range 0-59.999...',
}


def anp(a):
    """
    Normalize angle into the range 0 <= a < 2pi.

    Parameters
    ----------
    a : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraAnp``. The ERFA documentation is::

        - - - - - - -
         e r a A n p
        - - - - - - -

        Normalize angle into the range 0 <= a < 2pi.

        Given:
           a        double     angle (radians)

        Returned (function value):
                    double     angle in range 0-2pi

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.anp(a)
    return c_retval


def anpm(a):
    """
    Normalize angle into the range -pi <= a < +pi.

    Parameters
    ----------
    a : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraAnpm``. The ERFA documentation is::

        - - - - - - - -
         e r a A n p m
        - - - - - - - -

        Normalize angle into the range -pi <= a < +pi.

        Given:
           a        double     angle (radians)

        Returned (function value):
                    double     angle in range +/-pi

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.anpm(a)
    return c_retval


def d2tf(ndp, days):
    """
    Decompose days to hours, minutes, seconds, fraction.

    Parameters
    ----------
    ndp : int array
    days : double array

    Returns
    -------
    sign : char array
    ihmsf : int array

    Notes
    -----
    Wraps ERFA function ``eraD2tf``. The ERFA documentation is::

        - - - - - - - -
         e r a D 2 t f
        - - - - - - - -

        Decompose days to hours, minutes, seconds, fraction.

        Given:
           ndp     int     resolution (Note 1)
           days    double  interval in days

        Returned:
           sign    char         '+' or '-'
           ihmsf   int[4]  hours, minutes, seconds, fraction

        Notes:

        1) The argument ndp is interpreted as follows:

           ndp         resolution
            :      ...0000 00 00
           -7         1000 00 00
           -6          100 00 00
           -5           10 00 00
           -4            1 00 00
           -3            0 10 00
           -2            0 01 00
           -1            0 00 10
            0            0 00 01
            1            0 00 00.1
            2            0 00 00.01
            3            0 00 00.001
            :            0 00 00.000...

        2) The largest positive useful value for ndp is determined by the
           size of days, the format of double on the target platform, and
           the risk of overflowing ihmsf[3].  On a typical platform, for
           days up to 1.0, the available floating-point precision might
           correspond to ndp=12.  However, the practical limit is typically
           ndp=9, set by the capacity of a 32-bit int, or ndp=4 if int is
           only 16 bits.

        3) The absolute value of days may exceed 1.0.  In cases where it
           does not, it is up to the caller to test for and handle the
           case where days is very nearly 1.0 and rounds up to 24 hours,
           by testing for ihmsf[0]=24 and setting ihmsf[0-3] to zero.

        This revision:  2020 April 20

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    sign, ihmsf = ufunc.d2tf(ndp, days)
    sign = sign.view(dt_bytes1)
    return sign, ihmsf


def tf2a(s, ihour, imin, sec):
    """
    Convert hours, minutes, seconds to radians.

    Parameters
    ----------
    s : char array
    ihour : int array
    imin : int array
    sec : double array

    Returns
    -------
    rad : double array

    Notes
    -----
    Wraps ERFA function ``eraTf2a``. The ERFA documentation is::

        - - - - - - - -
         e r a T f 2 a
        - - - - - - - -

        Convert hours, minutes, seconds to radians.

        Given:
           s         char    sign:  '-' = negative, otherwise positive
           ihour     int     hours
           imin      int     minutes
           sec       double  seconds

        Returned:
           rad       double  angle in radians

        Returned (function value):
                     int     status:  0 = OK
                                      1 = ihour outside range 0-23
                                      2 = imin outside range 0-59
                                      3 = sec outside range 0-59.999...

        Notes:

        1)  The result is computed even if any of the range checks fail.

        2)  Negative ihour, imin and/or sec produce a warning status, but
            the absolute value is used in the conversion.

        3)  If there are multiple errors, the status value reflects only the
            first, the smallest taking precedence.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rad, c_retval = ufunc.tf2a(s, ihour, imin, sec)
    check_errwarn(c_retval, 'tf2a')
    return rad


STATUS_CODES['tf2a'] = {
    0: 'OK',
    1: 'ihour outside range 0-23',
    2: 'imin outside range 0-59',
    3: 'sec outside range 0-59.999...',
}


def tf2d(s, ihour, imin, sec):
    """
    Convert hours, minutes, seconds to days.

    Parameters
    ----------
    s : char array
    ihour : int array
    imin : int array
    sec : double array

    Returns
    -------
    days : double array

    Notes
    -----
    Wraps ERFA function ``eraTf2d``. The ERFA documentation is::

        - - - - - - - -
         e r a T f 2 d
        - - - - - - - -

        Convert hours, minutes, seconds to days.

        Given:
           s         char    sign:  '-' = negative, otherwise positive
           ihour     int     hours
           imin      int     minutes
           sec       double  seconds

        Returned:
           days      double  interval in days

        Returned (function value):
                     int     status:  0 = OK
                                      1 = ihour outside range 0-23
                                      2 = imin outside range 0-59
                                      3 = sec outside range 0-59.999...

        Notes:

        1)  The result is computed even if any of the range checks fail.

        2)  Negative ihour, imin and/or sec produce a warning status, but
            the absolute value is used in the conversion.

        3)  If there are multiple errors, the status value reflects only the
            first, the smallest taking precedence.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    days, c_retval = ufunc.tf2d(s, ihour, imin, sec)
    check_errwarn(c_retval, 'tf2d')
    return days


STATUS_CODES['tf2d'] = {
    0: 'OK',
    1: 'ihour outside range 0-23',
    2: 'imin outside range 0-59',
    3: 'sec outside range 0-59.999...',
}


def rx(phi, r):
    """
    Rotate an r-matrix about the x-axis.

    Parameters
    ----------
    phi : double array
    r : double array

    Returns
    -------
    r : double array

    Notes
    -----
    Wraps ERFA function ``eraRx``. Note that, unlike the erfa routine,
    the python wrapper does not change r in-place. The ERFA documentation is::

        - - - - - -
         e r a R x
        - - - - - -

        Rotate an r-matrix about the x-axis.

        Given:
           phi    double          angle (radians)

        Given and returned:
           r      double[3][3]    r-matrix, rotated

        Notes:

        1) Calling this function with positive phi incorporates in the
           supplied r-matrix r an additional rotation, about the x-axis,
           anticlockwise as seen looking towards the origin from positive x.

        2) The additional rotation can be represented by this matrix:

               (  1        0            0      )
               (                               )
               (  0   + cos(phi)   + sin(phi)  )
               (                               )
               (  0   - sin(phi)   + cos(phi)  )

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r = ufunc.rx(phi, r)
    return r


def ry(theta, r):
    """
    Rotate an r-matrix about the y-axis.

    Parameters
    ----------
    theta : double array
    r : double array

    Returns
    -------
    r : double array

    Notes
    -----
    Wraps ERFA function ``eraRy``. Note that, unlike the erfa routine,
    the python wrapper does not change r in-place. The ERFA documentation is::

        - - - - - -
         e r a R y
        - - - - - -

        Rotate an r-matrix about the y-axis.

        Given:
           theta  double          angle (radians)

        Given and returned:
           r      double[3][3]    r-matrix, rotated

        Notes:

        1) Calling this function with positive theta incorporates in the
           supplied r-matrix r an additional rotation, about the y-axis,
           anticlockwise as seen looking towards the origin from positive y.

        2) The additional rotation can be represented by this matrix:

               (  + cos(theta)     0      - sin(theta)  )
               (                                        )
               (       0           1           0        )
               (                                        )
               (  + sin(theta)     0      + cos(theta)  )

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r = ufunc.ry(theta, r)
    return r


def rz(psi, r):
    """
    Rotate an r-matrix about the z-axis.

    Parameters
    ----------
    psi : double array
    r : double array

    Returns
    -------
    r : double array

    Notes
    -----
    Wraps ERFA function ``eraRz``. Note that, unlike the erfa routine,
    the python wrapper does not change r in-place. The ERFA documentation is::

        - - - - - -
         e r a R z
        - - - - - -

        Rotate an r-matrix about the z-axis.

        Given:
           psi    double          angle (radians)

        Given and returned:
           r      double[3][3]    r-matrix, rotated

        Notes:

        1) Calling this function with positive psi incorporates in the
           supplied r-matrix r an additional rotation, about the z-axis,
           anticlockwise as seen looking towards the origin from positive z.

        2) The additional rotation can be represented by this matrix:

               (  + cos(psi)   + sin(psi)     0  )
               (                                 )
               (  - sin(psi)   + cos(psi)     0  )
               (                                 )
               (       0            0         1  )

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r = ufunc.rz(psi, r)
    return r


def cp(p):
    """
    Copy a p-vector.

    Parameters
    ----------
    p : double array

    Returns
    -------
    c : double array

    Notes
    -----
    Wraps ERFA function ``eraCp``. The ERFA documentation is::

        - - - - - -
         e r a C p
        - - - - - -

        Copy a p-vector.

        Given:
           p        double[3]     p-vector to be copied

        Returned:
           c        double[3]     copy

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c = ufunc.cp(p)
    return c


def cpv(pv):
    """
    Copy a position/velocity vector.

    Parameters
    ----------
    pv : double array

    Returns
    -------
    c : double array

    Notes
    -----
    Wraps ERFA function ``eraCpv``. The ERFA documentation is::

        - - - - - - -
         e r a C p v
        - - - - - - -

        Copy a position/velocity vector.

        Given:
           pv     double[2][3]    position/velocity vector to be copied

        Returned:
           c      double[2][3]    copy

        Called:
           eraCp        copy p-vector

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c = ufunc.cpv(pv)
    return c


def cr(r):
    """
    Copy an r-matrix.

    Parameters
    ----------
    r : double array

    Returns
    -------
    c : double array

    Notes
    -----
    Wraps ERFA function ``eraCr``. The ERFA documentation is::

        - - - - - -
         e r a C r
        - - - - - -

        Copy an r-matrix.

        Given:
           r        double[3][3]    r-matrix to be copied

        Returned:
           c        double[3][3]    copy

        Called:
           eraCp        copy p-vector

        This revision:  2016 May 19

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c = ufunc.cr(r)
    return c


def p2pv(p):
    """
    Extend a p-vector to a pv-vector by appending a zero velocity.

    Parameters
    ----------
    p : double array

    Returns
    -------
    pv : double array

    Notes
    -----
    Wraps ERFA function ``eraP2pv``. The ERFA documentation is::

        - - - - - - - -
         e r a P 2 p v
        - - - - - - - -

        Extend a p-vector to a pv-vector by appending a zero velocity.

        Given:
           p        double[3]       p-vector

        Returned:
           pv       double[2][3]    pv-vector

        Called:
           eraCp        copy p-vector
           eraZp        zero p-vector

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    pv = ufunc.p2pv(p)
    return pv


def pv2p(pv):
    """
    Discard velocity component of a pv-vector.

    Parameters
    ----------
    pv : double array

    Returns
    -------
    p : double array

    Notes
    -----
    Wraps ERFA function ``eraPv2p``. The ERFA documentation is::

        - - - - - - - -
         e r a P v 2 p
        - - - - - - - -

        Discard velocity component of a pv-vector.

        Given:
           pv      double[2][3]     pv-vector

        Returned:
           p       double[3]        p-vector

        Called:
           eraCp        copy p-vector

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    p = ufunc.pv2p(pv)
    return p


def ir():
    """
    Initialize an r-matrix to the identity matrix.

    Parameters
    ----------

    Returns
    -------
    r : double array

    Notes
    -----
    Wraps ERFA function ``eraIr``. The ERFA documentation is::

        - - - - - -
         e r a I r
        - - - - - -

        Initialize an r-matrix to the identity matrix.

        Returned:
           r       double[3][3]    r-matrix

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r = ufunc.ir()
    return r


def zp():
    """
    Zero a p-vector.

    Parameters
    ----------

    Returns
    -------
    p : double array

    Notes
    -----
    Wraps ERFA function ``eraZp``. The ERFA documentation is::

        - - - - - -
         e r a Z p
        - - - - - -

        Zero a p-vector.

        Returned:
           p        double[3]      zero p-vector

        This revision:  2020 August 25

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    p = ufunc.zp()
    return p


def zpv():
    """
    Zero a pv-vector.

    Parameters
    ----------

    Returns
    -------
    pv : double array

    Notes
    -----
    Wraps ERFA function ``eraZpv``. The ERFA documentation is::

        - - - - - - -
         e r a Z p v
        - - - - - - -

        Zero a pv-vector.

        Returned:
           pv       double[2][3]      zero pv-vector

        Called:
           eraZp        zero p-vector

        This revision:  2020 August 25

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    pv = ufunc.zpv()
    return pv


def zr():
    """
    Initialize an r-matrix to the null matrix.

    Parameters
    ----------

    Returns
    -------
    r : double array

    Notes
    -----
    Wraps ERFA function ``eraZr``. The ERFA documentation is::

        - - - - - -
         e r a Z r
        - - - - - -

        Initialize an r-matrix to the null matrix.

        Returned:
           r        double[3][3]    r-matrix

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r = ufunc.zr()
    return r


def rxr(a, b):
    """
    Multiply two r-matrices.

    Parameters
    ----------
    a : double array
    b : double array

    Returns
    -------
    atb : double array

    Notes
    -----
    Wraps ERFA function ``eraRxr``. The ERFA documentation is::

        - - - - - - -
         e r a R x r
        - - - - - - -

        Multiply two r-matrices.

        Given:
           a        double[3][3]    first r-matrix
           b        double[3][3]    second r-matrix

        Returned:
           atb      double[3][3]    a * b

        Note:
           It is permissible to re-use the same array for any of the
           arguments.

        Called:
           eraCr        copy r-matrix

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    atb = ufunc.rxr(a, b)
    return atb


def tr(r):
    """
    Transpose an r-matrix.

    Parameters
    ----------
    r : double array

    Returns
    -------
    rt : double array

    Notes
    -----
    Wraps ERFA function ``eraTr``. The ERFA documentation is::

        - - - - - -
         e r a T r
        - - - - - -

        Transpose an r-matrix.

        Given:
           r        double[3][3]    r-matrix

        Returned:
           rt       double[3][3]    transpose

        Note:
           It is permissible for r and rt to be the same array.

        Called:
           eraCr        copy r-matrix

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rt = ufunc.tr(r)
    return rt


def rxp(r, p):
    """
    Multiply a p-vector by an r-matrix.

    Parameters
    ----------
    r : double array
    p : double array

    Returns
    -------
    rp : double array

    Notes
    -----
    Wraps ERFA function ``eraRxp``. The ERFA documentation is::

        - - - - - - -
         e r a R x p
        - - - - - - -

        Multiply a p-vector by an r-matrix.

        Given:
           r        double[3][3]    r-matrix
           p        double[3]       p-vector

        Returned:
           rp       double[3]       r * p

        Note:
           It is permissible for p and rp to be the same array.

        Called:
           eraCp        copy p-vector

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rp = ufunc.rxp(r, p)
    return rp


def rxpv(r, pv):
    """
    Multiply a pv-vector by an r-matrix.

    Parameters
    ----------
    r : double array
    pv : double array

    Returns
    -------
    rpv : double array

    Notes
    -----
    Wraps ERFA function ``eraRxpv``. The ERFA documentation is::

        - - - - - - - -
         e r a R x p v
        - - - - - - - -

        Multiply a pv-vector by an r-matrix.

        Given:
           r        double[3][3]    r-matrix
           pv       double[2][3]    pv-vector

        Returned:
           rpv      double[2][3]    r * pv

        Notes:

        1) The algorithm is for the simple case where the r-matrix r is not
           a function of time.  The case where r is a function of time leads
           to an additional velocity component equal to the product of the
           derivative of r and the position vector.

        2) It is permissible for pv and rpv to be the same array.

        Called:
           eraRxp       product of r-matrix and p-vector

        This revision:  2020 September 26

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    rpv = ufunc.rxpv(r, pv)
    return rpv


def trxp(r, p):
    """
    Multiply a p-vector by the transpose of an r-matrix.

    Parameters
    ----------
    r : double array
    p : double array

    Returns
    -------
    trp : double array

    Notes
    -----
    Wraps ERFA function ``eraTrxp``. The ERFA documentation is::

        - - - - - - - -
         e r a T r x p
        - - - - - - - -

        Multiply a p-vector by the transpose of an r-matrix.

        Given:
           r        double[3][3]   r-matrix
           p        double[3]      p-vector

        Returned:
           trp      double[3]      r^T * p

        Note:
           It is permissible for p and trp to be the same array.

        Called:
           eraTr        transpose r-matrix
           eraRxp       product of r-matrix and p-vector

        This revision:  2020 May 24

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    trp = ufunc.trxp(r, p)
    return trp


def trxpv(r, pv):
    """
    Multiply a pv-vector by the transpose of an r-matrix.

    Parameters
    ----------
    r : double array
    pv : double array

    Returns
    -------
    trpv : double array

    Notes
    -----
    Wraps ERFA function ``eraTrxpv``. The ERFA documentation is::

        - - - - - - - - -
         e r a T r x p v
        - - - - - - - - -

        Multiply a pv-vector by the transpose of an r-matrix.

        Given:
           r        double[3][3]    r-matrix
           pv       double[2][3]    pv-vector

        Returned:
           trpv     double[2][3]    r^T * pv

        Notes:

        1) The algorithm is for the simple case where the r-matrix r is not
           a function of time.  The case where r is a function of time leads
           to an additional velocity component equal to the product of the
           derivative of the transpose of r and the position vector.

        2) It is permissible for pv and rpv to be the same array.

        Called:
           eraTr        transpose r-matrix
           eraRxpv      product of r-matrix and pv-vector

        This revision:  2020 September 26

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    trpv = ufunc.trxpv(r, pv)
    return trpv


def rm2v(r):
    """
    Express an r-matrix as an r-vector.

    Parameters
    ----------
    r : double array

    Returns
    -------
    w : double array

    Notes
    -----
    Wraps ERFA function ``eraRm2v``. The ERFA documentation is::

        - - - - - - - -
         e r a R m 2 v
        - - - - - - - -

        Express an r-matrix as an r-vector.

        Given:
           r        double[3][3]    rotation matrix

        Returned:
           w        double[3]       rotation vector (Note 1)

        Notes:

        1) A rotation matrix describes a rotation through some angle about
           some arbitrary axis called the Euler axis.  The "rotation vector"
           returned by this function has the same direction as the Euler axis,
           and its magnitude is the angle in radians.  (The magnitude and
           direction can be separated by means of the function eraPn.)

        2) If r is null, so is the result.  If r is not a rotation matrix
           the result is undefined;  r must be proper (i.e. have a positive
           determinant) and real orthogonal (inverse = transpose).

        3) The reference frame rotates clockwise as seen looking along
           the rotation vector from the origin.

        This revision:  2015 January 30

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    w = ufunc.rm2v(r)
    return w


def rv2m(w):
    """
    Form the r-matrix corresponding to a given r-vector.

    Parameters
    ----------
    w : double array

    Returns
    -------
    r : double array

    Notes
    -----
    Wraps ERFA function ``eraRv2m``. The ERFA documentation is::

        - - - - - - - -
         e r a R v 2 m
        - - - - - - - -

        Form the r-matrix corresponding to a given r-vector.

        Given:
           w        double[3]      rotation vector (Note 1)

        Returned:
           r        double[3][3]    rotation matrix

        Notes:

        1) A rotation matrix describes a rotation through some angle about
           some arbitrary axis called the Euler axis.  The "rotation vector"
           supplied to This function has the same direction as the Euler
           axis, and its magnitude is the angle in radians.

        2) If w is null, the identity matrix is returned.

        3) The reference frame rotates clockwise as seen looking along the
           rotation vector from the origin.

        This revision:  2020 August 21

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r = ufunc.rv2m(w)
    return r


def pap(a, b):
    """
    Position-angle from two p-vectors.

    Parameters
    ----------
    a : double array
    b : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraPap``. The ERFA documentation is::

        - - - - - - -
         e r a P a p
        - - - - - - -

        Position-angle from two p-vectors.

        Given:
           a      double[3]  direction of reference point
           b      double[3]  direction of point whose PA is required

        Returned (function value):
                  double     position angle of b with respect to a (radians)

        Notes:

        1) The result is the position angle, in radians, of direction b with
           respect to direction a.  It is in the range -pi to +pi.  The
           sense is such that if b is a small distance "north" of a the
           position angle is approximately zero, and if b is a small
           distance "east" of a the position angle is approximately +pi/2.

        2) The vectors a and b need not be of unit length.

        3) Zero is returned if the two directions are the same or if either
           vector is null.

        4) If vector a is at a pole, the result is ill-defined.

        Called:
           eraPn        decompose p-vector into modulus and direction
           eraPm        modulus of p-vector
           eraPxp       vector product of two p-vectors
           eraPmp       p-vector minus p-vector
           eraPdp       scalar product of two p-vectors

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.pap(a, b)
    return c_retval


def pas(al, ap, bl, bp):
    """
    Position-angle from spherical coordinates.

    Parameters
    ----------
    al : double array
    ap : double array
    bl : double array
    bp : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraPas``. The ERFA documentation is::

        - - - - - - -
         e r a P a s
        - - - - - - -

        Position-angle from spherical coordinates.

        Given:
           al     double     longitude of point A (e.g. RA) in radians
           ap     double     latitude of point A (e.g. Dec) in radians
           bl     double     longitude of point B
           bp     double     latitude of point B

        Returned (function value):
                  double     position angle of B with respect to A

        Notes:

        1) The result is the bearing (position angle), in radians, of point
           B with respect to point A.  It is in the range -pi to +pi.  The
           sense is such that if B is a small distance "east" of point A,
           the bearing is approximately +pi/2.

        2) Zero is returned if the two points are coincident.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.pas(al, ap, bl, bp)
    return c_retval


def sepp(a, b):
    """
    Angular separation between two p-vectors.

    Parameters
    ----------
    a : double array
    b : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraSepp``. The ERFA documentation is::

        - - - - - - - -
         e r a S e p p
        - - - - - - - -

        Angular separation between two p-vectors.

        Given:
           a      double[3]    first p-vector (not necessarily unit length)
           b      double[3]    second p-vector (not necessarily unit length)

        Returned (function value):
                  double       angular separation (radians, always positive)

        Notes:

        1) If either vector is null, a zero result is returned.

        2) The angular separation is most simply formulated in terms of
           scalar product.  However, this gives poor accuracy for angles
           near zero and pi.  The present algorithm uses both cross product
           and dot product, to deliver full accuracy whatever the size of
           the angle.

        Called:
           eraPxp       vector product of two p-vectors
           eraPm        modulus of p-vector
           eraPdp       scalar product of two p-vectors

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.sepp(a, b)
    return c_retval


def seps(al, ap, bl, bp):
    """
    Angular separation between two sets of spherical coordinates.

    Parameters
    ----------
    al : double array
    ap : double array
    bl : double array
    bp : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraSeps``. The ERFA documentation is::

        - - - - - - - -
         e r a S e p s
        - - - - - - - -

        Angular separation between two sets of spherical coordinates.

        Given:
           al     double       first longitude (radians)
           ap     double       first latitude (radians)
           bl     double       second longitude (radians)
           bp     double       second latitude (radians)

        Returned (function value):
                  double       angular separation (radians)

        Called:
           eraS2c       spherical coordinates to unit vector
           eraSepp      angular separation between two p-vectors

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.seps(al, ap, bl, bp)
    return c_retval


def c2s(p):
    """
    P-vector to spherical coordinates.

    Parameters
    ----------
    p : double array

    Returns
    -------
    theta : double array
    phi : double array

    Notes
    -----
    Wraps ERFA function ``eraC2s``. The ERFA documentation is::

        - - - - - - -
         e r a C 2 s
        - - - - - - -

        P-vector to spherical coordinates.

        Given:
           p      double[3]    p-vector

        Returned:
           theta  double       longitude angle (radians)
           phi    double       latitude angle (radians)

        Notes:

        1) The vector p can have any magnitude; only its direction is used.

        2) If p is null, zero theta and phi are returned.

        3) At either pole, zero theta is returned.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    theta, phi = ufunc.c2s(p)
    return theta, phi


def p2s(p):
    """
    P-vector to spherical polar coordinates.

    Parameters
    ----------
    p : double array

    Returns
    -------
    theta : double array
    phi : double array
    r : double array

    Notes
    -----
    Wraps ERFA function ``eraP2s``. The ERFA documentation is::

        - - - - - - -
         e r a P 2 s
        - - - - - - -

        P-vector to spherical polar coordinates.

        Given:
           p        double[3]    p-vector

        Returned:
           theta    double       longitude angle (radians)
           phi      double       latitude angle (radians)
           r        double       radial distance

        Notes:

        1) If P is null, zero theta, phi and r are returned.

        2) At either pole, zero theta is returned.

        Called:
           eraC2s       p-vector to spherical
           eraPm        modulus of p-vector

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    theta, phi, r = ufunc.p2s(p)
    return theta, phi, r


def pv2s(pv):
    """
    Convert position/velocity from Cartesian to spherical coordinates.

    Parameters
    ----------
    pv : double array

    Returns
    -------
    theta : double array
    phi : double array
    r : double array
    td : double array
    pd : double array
    rd : double array

    Notes
    -----
    Wraps ERFA function ``eraPv2s``. The ERFA documentation is::

        - - - - - - - -
         e r a P v 2 s
        - - - - - - - -

        Convert position/velocity from Cartesian to spherical coordinates.

        Given:
           pv       double[2][3]  pv-vector

        Returned:
           theta    double        longitude angle (radians)
           phi      double        latitude angle (radians)
           r        double        radial distance
           td       double        rate of change of theta
           pd       double        rate of change of phi
           rd       double        rate of change of r

        Notes:

        1) If the position part of pv is null, theta, phi, td and pd
           are indeterminate.  This is handled by extrapolating the
           position through unit time by using the velocity part of
           pv.  This moves the origin without changing the direction
           of the velocity component.  If the position and velocity
           components of pv are both null, zeroes are returned for all
           six results.

        2) If the position is a pole, theta, td and pd are indeterminate.
           In such cases zeroes are returned for all three.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    theta, phi, r, td, pd, rd = ufunc.pv2s(pv)
    return theta, phi, r, td, pd, rd


def s2c(theta, phi):
    """
    Convert spherical coordinates to Cartesian.

    Parameters
    ----------
    theta : double array
    phi : double array

    Returns
    -------
    c : double array

    Notes
    -----
    Wraps ERFA function ``eraS2c``. The ERFA documentation is::

        - - - - - - -
         e r a S 2 c
        - - - - - - -

        Convert spherical coordinates to Cartesian.

        Given:
           theta    double       longitude angle (radians)
           phi      double       latitude angle (radians)

        Returned:
           c        double[3]    direction cosines

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c = ufunc.s2c(theta, phi)
    return c


def s2p(theta, phi, r):
    """
    Convert spherical polar coordinates to p-vector.

    Parameters
    ----------
    theta : double array
    phi : double array
    r : double array

    Returns
    -------
    p : double array

    Notes
    -----
    Wraps ERFA function ``eraS2p``. The ERFA documentation is::

        - - - - - - -
         e r a S 2 p
        - - - - - - -

        Convert spherical polar coordinates to p-vector.

        Given:
           theta   double       longitude angle (radians)
           phi     double       latitude angle (radians)
           r       double       radial distance

        Returned:
           p       double[3]    Cartesian coordinates

        Called:
           eraS2c       spherical coordinates to unit vector
           eraSxp       multiply p-vector by scalar

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    p = ufunc.s2p(theta, phi, r)
    return p


def s2pv(theta, phi, r, td, pd, rd):
    """
    Convert position/velocity from spherical to Cartesian coordinates.

    Parameters
    ----------
    theta : double array
    phi : double array
    r : double array
    td : double array
    pd : double array
    rd : double array

    Returns
    -------
    pv : double array

    Notes
    -----
    Wraps ERFA function ``eraS2pv``. The ERFA documentation is::

        - - - - - - - -
         e r a S 2 p v
        - - - - - - - -

        Convert position/velocity from spherical to Cartesian coordinates.

        Given:
           theta    double          longitude angle (radians)
           phi      double          latitude angle (radians)
           r        double          radial distance
           td       double          rate of change of theta
           pd       double          rate of change of phi
           rd       double          rate of change of r

        Returned:
           pv       double[2][3]    pv-vector

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    pv = ufunc.s2pv(theta, phi, r, td, pd, rd)
    return pv


def pdp(a, b):
    """
    p-vector inner (=scalar=dot) product.

    Parameters
    ----------
    a : double array
    b : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraPdp``. The ERFA documentation is::

        - - - - - - -
         e r a P d p
        - - - - - - -

        p-vector inner (=scalar=dot) product.

        Given:
           a      double[3]     first p-vector
           b      double[3]     second p-vector

        Returned (function value):
                  double        a . b

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.pdp(a, b)
    return c_retval


def pm(p):
    """
    Modulus of p-vector.

    Parameters
    ----------
    p : double array

    Returns
    -------
    c_retval : double array

    Notes
    -----
    Wraps ERFA function ``eraPm``. The ERFA documentation is::

        - - - - - -
         e r a P m
        - - - - - -

        Modulus of p-vector.

        Given:
           p      double[3]     p-vector

        Returned (function value):
                  double        modulus

        This revision:  2013 August 7

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    c_retval = ufunc.pm(p)
    return c_retval


def pmp(a, b):
    """
    P-vector subtraction.

    Parameters
    ----------
    a : double array
    b : double array

    Returns
    -------
    amb : double array

    Notes
    -----
    Wraps ERFA function ``eraPmp``. The ERFA documentation is::

        - - - - - - -
         e r a P m p
        - - - - - - -

        P-vector subtraction.

        Given:
           a        double[3]      first p-vector
           b        double[3]      second p-vector

        Returned:
           amb      double[3]      a - b

        Note:
           It is permissible to re-use the same array for any of the
           arguments.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    amb = ufunc.pmp(a, b)
    return amb


def pn(p):
    """
    Convert a p-vector into modulus and unit vector.

    Parameters
    ----------
    p : double array

    Returns
    -------
    r : double array
    u : double array

    Notes
    -----
    Wraps ERFA function ``eraPn``. The ERFA documentation is::

        - - - - - -
         e r a P n
        - - - - - -

        Convert a p-vector into modulus and unit vector.

        Given:
           p        double[3]      p-vector

        Returned:
           r        double         modulus
           u        double[3]      unit vector

        Notes:

        1) If p is null, the result is null.  Otherwise the result is a unit
           vector.

        2) It is permissible to re-use the same array for any of the
           arguments.

        Called:
           eraPm        modulus of p-vector
           eraZp        zero p-vector
           eraSxp       multiply p-vector by scalar

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r, u = ufunc.pn(p)
    return r, u


def ppp(a, b):
    """
    P-vector addition.

    Parameters
    ----------
    a : double array
    b : double array

    Returns
    -------
    apb : double array

    Notes
    -----
    Wraps ERFA function ``eraPpp``. The ERFA documentation is::

        - - - - - - -
         e r a P p p
        - - - - - - -

        P-vector addition.

        Given:
           a        double[3]      first p-vector
           b        double[3]      second p-vector

        Returned:
           apb      double[3]      a + b

        Note:
           It is permissible to re-use the same array for any of the
           arguments.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    apb = ufunc.ppp(a, b)
    return apb


def ppsp(a, s, b):
    """
    P-vector plus scaled p-vector.

    Parameters
    ----------
    a : double array
    s : double array
    b : double array

    Returns
    -------
    apsb : double array

    Notes
    -----
    Wraps ERFA function ``eraPpsp``. The ERFA documentation is::

        - - - - - - - -
         e r a P p s p
        - - - - - - - -

        P-vector plus scaled p-vector.

        Given:
           a      double[3]     first p-vector
           s      double        scalar (multiplier for b)
           b      double[3]     second p-vector

        Returned:
           apsb   double[3]     a + s*b

        Note:
           It is permissible for any of a, b and apsb to be the same array.

        Called:
           eraSxp       multiply p-vector by scalar
           eraPpp       p-vector plus p-vector

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    apsb = ufunc.ppsp(a, s, b)
    return apsb


def pvdpv(a, b):
    """
    Inner (=scalar=dot) product of two pv-vectors.

    Parameters
    ----------
    a : double array
    b : double array

    Returns
    -------
    adb : double array

    Notes
    -----
    Wraps ERFA function ``eraPvdpv``. The ERFA documentation is::

        - - - - - - - - -
         e r a P v d p v
        - - - - - - - - -

        Inner (=scalar=dot) product of two pv-vectors.

        Given:
           a        double[2][3]      first pv-vector
           b        double[2][3]      second pv-vector

        Returned:
           adb      double[2]         a . b (see note)

        Note:

           If the position and velocity components of the two pv-vectors are
           ( ap, av ) and ( bp, bv ), the result, a . b, is the pair of
           numbers ( ap . bp , ap . bv + av . bp ).  The two numbers are the
           dot-product of the two p-vectors and its derivative.

        Called:
           eraPdp       scalar product of two p-vectors

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    adb = ufunc.pvdpv(a, b)
    return adb


def pvm(pv):
    """
    Modulus of pv-vector.

    Parameters
    ----------
    pv : double array

    Returns
    -------
    r : double array
    s : double array

    Notes
    -----
    Wraps ERFA function ``eraPvm``. The ERFA documentation is::

        - - - - - - -
         e r a P v m
        - - - - - - -

        Modulus of pv-vector.

        Given:
           pv     double[2][3]   pv-vector

        Returned:
           r      double         modulus of position component
           s      double         modulus of velocity component

        Called:
           eraPm        modulus of p-vector

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    r, s = ufunc.pvm(pv)
    return r, s


def pvmpv(a, b):
    """
    Subtract one pv-vector from another.

    Parameters
    ----------
    a : double array
    b : double array

    Returns
    -------
    amb : double array

    Notes
    -----
    Wraps ERFA function ``eraPvmpv``. The ERFA documentation is::

        - - - - - - - - -
         e r a P v m p v
        - - - - - - - - -

        Subtract one pv-vector from another.

        Given:
           a       double[2][3]      first pv-vector
           b       double[2][3]      second pv-vector

        Returned:
           amb     double[2][3]      a - b

        Note:
           It is permissible to re-use the same array for any of the
           arguments.

        Called:
           eraPmp       p-vector minus p-vector

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    amb = ufunc.pvmpv(a, b)
    return amb


def pvppv(a, b):
    """
    Add one pv-vector to another.

    Parameters
    ----------
    a : double array
    b : double array

    Returns
    -------
    apb : double array

    Notes
    -----
    Wraps ERFA function ``eraPvppv``. The ERFA documentation is::

        - - - - - - - - -
         e r a P v p p v
        - - - - - - - - -

        Add one pv-vector to another.

        Given:
           a        double[2][3]      first pv-vector
           b        double[2][3]      second pv-vector

        Returned:
           apb      double[2][3]      a + b

        Note:
           It is permissible to re-use the same array for any of the
           arguments.

        Called:
           eraPpp       p-vector plus p-vector

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    apb = ufunc.pvppv(a, b)
    return apb


def pvu(dt, pv):
    """
    Update a pv-vector.

    Parameters
    ----------
    dt : double array
    pv : double array

    Returns
    -------
    upv : double array

    Notes
    -----
    Wraps ERFA function ``eraPvu``. The ERFA documentation is::

        - - - - - - -
         e r a P v u
        - - - - - - -

        Update a pv-vector.

        Given:
           dt       double           time interval
           pv       double[2][3]     pv-vector

        Returned:
           upv      double[2][3]     p updated, v unchanged

        Notes:

        1) "Update" means "refer the position component of the vector
           to a new date dt time units from the existing date".

        2) The time units of dt must match those of the velocity.

        3) It is permissible for pv and upv to be the same array.

        Called:
           eraPpsp      p-vector plus scaled p-vector
           eraCp        copy p-vector

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    upv = ufunc.pvu(dt, pv)
    return upv


def pvup(dt, pv):
    """
    Update a pv-vector, discarding the velocity component.

    Parameters
    ----------
    dt : double array
    pv : double array

    Returns
    -------
    p : double array

    Notes
    -----
    Wraps ERFA function ``eraPvup``. The ERFA documentation is::

        - - - - - - - -
         e r a P v u p
        - - - - - - - -

        Update a pv-vector, discarding the velocity component.

        Given:
           dt       double            time interval
           pv       double[2][3]      pv-vector

        Returned:
           p        double[3]         p-vector

        Notes:

        1) "Update" means "refer the position component of the vector to a
           new date dt time units from the existing date".

        2) The time units of dt must match those of the velocity.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    p = ufunc.pvup(dt, pv)
    return p


def pvxpv(a, b):
    """
    Outer (=vector=cross) product of two pv-vectors.

    Parameters
    ----------
    a : double array
    b : double array

    Returns
    -------
    axb : double array

    Notes
    -----
    Wraps ERFA function ``eraPvxpv``. The ERFA documentation is::

        - - - - - - - - -
         e r a P v x p v
        - - - - - - - - -

        Outer (=vector=cross) product of two pv-vectors.

        Given:
           a        double[2][3]      first pv-vector
           b        double[2][3]      second pv-vector

        Returned:
           axb      double[2][3]      a x b

        Notes:

        1) If the position and velocity components of the two pv-vectors are
           ( ap, av ) and ( bp, bv ), the result, a x b, is the pair of
           vectors ( ap x bp, ap x bv + av x bp ).  The two vectors are the
           cross-product of the two p-vectors and its derivative.

        2) It is permissible to re-use the same array for any of the
           arguments.

        Called:
           eraCpv       copy pv-vector
           eraPxp       vector product of two p-vectors
           eraPpp       p-vector plus p-vector

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    axb = ufunc.pvxpv(a, b)
    return axb


def pxp(a, b):
    """
    p-vector outer (=vector=cross) product.

    Parameters
    ----------
    a : double array
    b : double array

    Returns
    -------
    axb : double array

    Notes
    -----
    Wraps ERFA function ``eraPxp``. The ERFA documentation is::

        - - - - - - -
         e r a P x p
        - - - - - - -

        p-vector outer (=vector=cross) product.

        Given:
           a        double[3]      first p-vector
           b        double[3]      second p-vector

        Returned:
           axb      double[3]      a x b

        Note:
           It is permissible to re-use the same array for any of the
           arguments.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    axb = ufunc.pxp(a, b)
    return axb


def s2xpv(s1, s2, pv):
    """
    Multiply a pv-vector by two scalars.

    Parameters
    ----------
    s1 : double array
    s2 : double array
    pv : double array

    Returns
    -------
    spv : double array

    Notes
    -----
    Wraps ERFA function ``eraS2xpv``. The ERFA documentation is::

        - - - - - - - - -
         e r a S 2 x p v
        - - - - - - - - -

        Multiply a pv-vector by two scalars.

        Given:
           s1     double         scalar to multiply position component by
           s2     double         scalar to multiply velocity component by
           pv     double[2][3]   pv-vector

        Returned:
           spv    double[2][3]   pv-vector: p scaled by s1, v scaled by s2

        Note:
           It is permissible for pv and spv to be the same array.

        Called:
           eraSxp       multiply p-vector by scalar

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    spv = ufunc.s2xpv(s1, s2, pv)
    return spv


def sxp(s, p):
    """
    Multiply a p-vector by a scalar.

    Parameters
    ----------
    s : double array
    p : double array

    Returns
    -------
    sp : double array

    Notes
    -----
    Wraps ERFA function ``eraSxp``. The ERFA documentation is::

        - - - - - - -
         e r a S x p
        - - - - - - -

        Multiply a p-vector by a scalar.

        Given:
           s      double        scalar
           p      double[3]     p-vector

        Returned:
           sp     double[3]     s * p

        Note:
           It is permissible for p and sp to be the same array.

        This revision:  2013 June 18

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    sp = ufunc.sxp(s, p)
    return sp


def sxpv(s, pv):
    """
    Multiply a pv-vector by a scalar.

    Parameters
    ----------
    s : double array
    pv : double array

    Returns
    -------
    spv : double array

    Notes
    -----
    Wraps ERFA function ``eraSxpv``. The ERFA documentation is::

        - - - - - - - -
         e r a S x p v
        - - - - - - - -

        Multiply a pv-vector by a scalar.

        Given:
           s       double          scalar
           pv      double[2][3]    pv-vector

        Returned:
           spv     double[2][3]    s * pv

        Note:
           It is permissible for pv and spv to be the same array.

        Called:
           eraS2xpv     multiply pv-vector by two scalars

        This revision:  2020 August 25

        Copyright (C) 2013-2021, NumFOCUS Foundation.
        Derived, with permission, from the SOFA library.  See notes at end of file.

    """
    spv = ufunc.sxpv(s, pv)
    return spv
