''' Standard Posix tzinfo implementation '''
from threading import Lock
from datetime import tzinfo
from datetime import time,datetime,timedelta # For test
from warnings import warn

import _PosixTimeZone
from os import environ

__all__ = ['PosixTimeZone']

__author__ = 'Stuart Bishop <zen@shangri-la.dropbear.id.au>'

class PosixTimeZone(tzinfo):
    "A tzinfo implementation relying on the underlying Posix system calls"

    _lock = Lock()

    # The timezone, as passed into our constructor
    zone = 'UTC'

    def __init__(self, zone=None):
        ''' zone is the timezone in 'standard' Posix zoneinfo notation. 
            eg. US/Central, Australia/Victoria etc.
            Wierder values can also be sent - see the TZ environment
            variable in tzset(3).

            zone should always be passed - the argument is only optional
            due to pickling issues that will hopefully disappear by
            Python 2.3 final. zone will become mandatory again at some
            future point.
        '''
        if zone is not None:
            self.zone = zone

    def tzname(self,dt):
        ''' datetime -> string name of time zone'''
        self._lock.acquire()
        try:
            self._tzSet()
            return _PosixTimeZone.tzname(*self._6tuple(dt))
        finally:
            self._tzReset()
            self._lock.release()

    def utcoffset(self,dt):
        ''' datetime -> minutes east of UTC (negative for west of UTC) '''
        self._lock.acquire()
        try:
            self._tzSet()
            secs = _PosixTimeZone.utcoffset(*self._6tuple(dt))
        finally:
            self._tzReset()
            self._lock.release()
        if abs(secs) >= 1440 * 60:
            raise RuntimeError,\
                "Got dodgy return value from utcoffset (%d) from %s" % (
                    secs,str(dt.replace(tzinfo=None))
                    )
        assert abs(secs) < 1440*60
        return timedelta(seconds=secs)

    _zero = timedelta(hours=0)
    _quater = timedelta(weeks=13)

    def dst(self,dt):
        ''' datetime -> DST offset in minutes east of UTC
    
        Return 0 if DST not in effect. utcoffset() must include the DST
        offset.

        Currently just returns None until we can calculate the DST offset
        '''
        self._lock.acquire()
        try:
            self._tzSet()
            _6tup = self._6tuple(dt)

            # Not in DST, so no DST offset
            if _PosixTimeZone.dst(*_6tup) == 0:
                return self._zero

            # We are in DST, so we have the yucky job of determining
            # the DST offset, which Posix doesn't give us
            # To do this, we keep hopping back in time in 3  month increments
            # until we get a non-DST time and use that to calculate the DST 
            # offset.
            else:

                # Our utcoffset in seconds, including DST
                dst_utcoff = _PosixTimeZone.utcoffset(*_6tup)

                for i in range(0,3):
                    dt = dt - self._quater
                    utcoff = _PosixTimeZone.utcoffset(*self._6tuple(dt))
                    if utcoff != dst_utcoff:
                        return timedelta(seconds=dst_utcoff - utcoff)

                warn('Broken POSIX timezone %s - timezone seems stuck in DST' 
                        % (environ['TZ']),
                    RuntimeWarning
                    )
                return None

        finally:
            self._tzReset()
            self._lock.release()

    def _tzSet(self):
        try:
            self._org_tz = environ['TZ']
        except KeyError:
            self._org_tz = None
        environ['TZ'] = self.zone
        _PosixTimeZone.tzset()

    def _tzReset(self):
        if self._org_tz is None:
            del environ['TZ']
        else:
            environ['TZ'] = self._org_tz
        del self._org_tz
        _PosixTimeZone.tzset()

    def _6tuple(self,dt):
        ''' Convert the dt to a tuple for use in populating the tm structure
            defined in <time.h>. We can't use dt.timetuple(), as this would 
            call our dst() since datetime objects unfortunatly don't remember
            if they are in daylight savings or not (and can thus
            be ambiguous).

            The Posix routines cannot handle years < 1971 or > 2037,
            so we force the year into this range here
        '''
        # Note that time.h uses 'year-1900' and counts months starting from 0
        year = dt.year - 1900
        if year > 137:
            year = 137 # 2037 - 1900
        elif year < 70:
            year = 70 # 1970 - 1900

        return  (dt.second, dt.minute, dt.hour, dt.day, dt.month - 1, year)

simple_test = '''
>>> def printstuff(d):
...     print d
...     if d.timetuple()[-1]:
...         print '%s (is DST)' % d.tzname()
...     else:
...         print d.tzname()

Setup our Eastern TZ
>>> Eastern = PosixTimeZone('US/Eastern')

Make sure the basics work
>>> dumb = datetime(2002,12,25,12,0,0,tzinfo=Eastern)
>>> printstuff(dumb)
2002-12-25 12:00:00-05:00
EST

Day before DST starts.
>>> daybefore = datetime(2002, 4, 6, 12, 0, 0, tzinfo=Eastern)
>>> printstuff(daybefore)
2002-04-06 12:00:00-05:00
EST

Right before DST starts.
>>> before = datetime(2002, 4, 7, 1, 59, 59, tzinfo=Eastern)
>>> printstuff(before)
2002-04-07 01:59:59-05:00
EST

Right when DST starts -- although this doesn't work very well.
>>> after = before + timedelta(seconds=1)
>>> printstuff(after)
2002-04-07 02:00:00-04:00
EDT (is DST)

1 hour after DST starts -- although this doesn't work very well.
>>> hourafter = after + timedelta(hours=1)
>>> printstuff(hourafter)
2002-04-07 03:00:00-04:00
EDT (is DST)

1 day after DST starts -- although this doesn't work very well.
>>> dayafter = after + timedelta(days=1)
>>> printstuff(dayafter)
2002-04-08 02:00:00-04:00
EDT (is DST)

Convert to UTC
>>> UTC = PosixTimeZone('UTC')
>>> printstuff(dayafter.astimezone(UTC))
2002-04-08 06:00:00+00:00
UTC

Convert to a different time zone
>>> Amsterdam = PosixTimeZone('Europe/Amsterdam')
>>> printstuff(dayafter.astimezone(Amsterdam))
2002-04-08 08:00:00+02:00
CEST (is DST)
'''

__test__ = {'simple_test': simple_test}

def _test():
    import doctest,PosixTimeZone
    return doctest.testmod(PosixTimeZone)

if __name__ == '__main__':
    _test()

