#!/usr/bin/python

###############################################################################
# NAME: pyp_demog.py
# VERSION: 2.0.0a7 (12AUGUST2004)
# AUTHOR: John B. Cole, PhD (jcole@aipl.arsusda.gov)
# LICENSE: LGPL
###############################################################################
# FUNCTIONS:
#     set_base_year()
#     set_age_units()
#     age_distribution()
#     sex_ratio()
###############################################################################

import pyp_utils

# Define some globals in case the user forgets to call set_base_year() and
# set_age_units().
global BASE_DEMOGRAPHIC_YEAR
global BASE_DEMOGRAPHIC_UNIT
global SEX_CODE_MAP
BASE_DEMOGRAPHIC_YEAR = 1950
BASE_DEMOGRAPHIC_UNIT = 'year'
SEX_CODE_MAP = {'m':'Male','f':'Female','u':'Unk'}

##
# set_base_year() defines a global variable, BASE_DEMOGRAPHIC_YEAR.
# @param year The year to be used as a base for computing ages.
def set_base_year(year=1950):
    global BASE_DEMOGRAPHIC_YEAR
    BASE_DEMOGRAPHIC_YEAR = year

##
# set_age_units() defines a global variable, BASE_DEMOGRAPHIC_UNIT.
# @param units The base unit for age computations ('year'|'month'|'day').
def set_age_units(units='year'):
    _units = ['year','month','day']
    global BASE_DEMOGRAPHIC_UNIT
    if units in _units:
        BASE_DEMOGRAPHIC_UNIT = units
    else:
        BASE_DEMOGRAPHIC_UNIT = 'year'

##
# age_distribution() computes histograms of the age distribution of
# males and females in the population.  You can also stratify by
# sex to get individual histograms.
# @param myped A PyPedal pedigree.
# @param verbose Print or suppress output. (???)
# @param sex A flag which determines whether or not to stratify by sex.
def age_distribution(myped,verbose=1,sex=1):
    age_dict = {}
    age_freq_total = 0.0
    if myped[0].age == -999:
        if myped[0].igen == -999:
            pyp_utils.set_generation(myped)
        pyp_utils.set_age(myped)
    if not sex:
        for i in range(len(myped)):
            try:
                age_dict[myped[i].age] = age_dict[myped[i].age] + 1
            except KeyError:
                age_dict[myped[i].age] = 1
        age_hist = pyp_utils.simple_histogram_dictionary(age_dict)
        if int(verbose) == 1:
            print '-'*80
            print 'Population Age Distribution'
            print '-'*80
            print '\tAge\tCount\tFrequency\tHistogram'
            for key in age_dict.keys():
                age_freq_total = age_freq_total + float(age_dict[key])/float(len(myped))
                print '\t%s\t%s\t%s\t%s' % (key,age_dict[key],float(age_dict[key])/float(len(myped)),age_hist[key])
            print '\tTOTAL\t%s\t%s' % (len(myped),age_freq_total)
            print '-'*80
    else:
        males = []
        females = []
        unknowns = []
        male_dict = {}
        female_dict = {}
        unknown_dict = {}
        for i in range(len(myped)):
            if myped[i].sex == 'm':
                males.append(myped[i])
            elif myped[i].sex == 'f':
                females.append(myped[i])
            else:
                unknowns.append(myped[i])
        for m in range(len(males)):
            try:
                male_dict[males[m].age] = male_dict[males[m].age] + 1
            except KeyError:
                male_dict[males[m].age] = 1
        for f in range(len(females)):
            try:
                female_dict[females[f].age] = female_dict[females[f].age] + 1
            except KeyError:
                female_dict[females[f].age] = 1
        for u in range(len(unknowns)):
            try:
                unknown_dict[unknowns[u].age] = unknown_dict[unknowns[u].age] + 1
            except KeyError:
                unknown_dict[unknowns[u].age] = 1
        male_hist = pyp_utils.simple_histogram_dictionary(male_dict)
        female_hist = pyp_utils.simple_histogram_dictionary(female_dict)
        unknown_hist = pyp_utils.simple_histogram_dictionary(unknown_dict)
        if int(verbose) == 1:
            print '-'*80
            print 'Population Age Distribution by Sex'
            print '-'*80
            age_freq_total = 0.0
            print 'Males'
            print '\tAge\tCount\tFrequency\tHistogram'
            for key in male_dict.keys():
                age_freq_total = age_freq_total + float(male_dict[key])/float(len(males))
                print '\t%s\t%s\t%s\t%s' % (key,male_dict[key],float(male_dict[key])/float(len(males)),male_hist[key])
            print '\tTOTAL\t%s\t%s' % (len(males),age_freq_total)
            print '-'*80
            age_freq_total = 0.0
            print 'Females'
            print '\tAge\tCount\tFrequency\tHistogram'
            for key in female_dict.keys():
                age_freq_total = age_freq_total + float(female_dict[key])/float(len(females))
                print '\t%s\t%s\t%s\t%s' % (key,female_dict[key],float(female_dict[key])/float(len(females)),female_hist[key])
            print '\tTOTAL\t%s\t%s' % (len(females),age_freq_total)
            print '-'*80
            age_freq_total = 0.0
            print 'Unknowns'
            print '\tAge\tCount\tFrequency\tHistogram'
            for key in unknown_dict.keys():
                age_freq_total = age_freq_total + float(unknown_dict[key])/float(len(unknowns))
                print '\t%s\t%s\t%s\t%s' % (key,unknown_dict[key],float(unknown_dict[key])/float(len(unknowns)),unknown_hist[key])
            print '\tTOTAL\t%s\t%s' % (len(unknowns),age_freq_total)
            print '-'*80

##
# sex_ratio() returns a dictionary containing the proportion of males
# and females in the population.
# @param year The year to be used as a base for computing ages.
# @return dict A dictionary containing entries for each sex/gender code defined in the global SEX_CODE_MAP.
def sex_ratio(myped,verbose=1):
	sexratiodict = {}
	for s in SEX_CODE_MAP.keys():
		sexratiodict[s] = 0
	for i in range(len(myped)):
		if sexratiodict.has_key(myped[i].sex):
			sexratiodict[myped[i].sex] = sexratiodict[myped[i].sex] + 1
		else:
			sexratiodict[myped[i].sex] = 1
	if verbose:
		print '-'*80
		print 'Overall Sex Ratio'
		print '-'*80
		print '(n = %s)' % (len(myped))
		print 'Sex\tCount\tFrequency'
		for s in sexratiodict.keys():
			print '%s:\t%s\t%s' % (SEX_CODE_MAP[s],sexratiodict[s],float(sexratiodict[s])/float(len(myped)))
		print '-'*80
		if int(sexratiodict['u']) > 0:
			marginal = sexratiodict['m'] + sexratiodict['f']
	                print 'Conditional Sex Ratio'
	                print '-'*80
	                print '(n = %s)' % (marginal)
	                print 'Sex\tCount\tFrequency'
	                print '%s:\t%s\t%s' % (SEX_CODE_MAP['m'],sexratiodict['m'],float(sexratiodict['m'])/marginal)
			print '%s:\t%s\t%s' % (SEX_CODE_MAP['f'],sexratiodict['f'],float(sexratiodict['f'])/marginal)
	return sexratiodict
