"""Interface to PostgeSQL databases

Requires pyPgSQL (http://sourceforge.net/projects/pypgsql)
"""

import db_base, sql_db
import time, datetime
from pyPgSQL import libpq

class postgresql_db(sql_db.sql_db):
    """An interface to a PotgreSQL database
    """

    def __init__(self, dsn=None, user=None, password=None, host=None,
                 database=None, port=None, options=None, tty=None):
        """Create the interface

        The keyword arguments are used to create the argument for
        libpq.PQconnectdb.

        dsn, a string of the form 'host:port:database:user:password:options:tty',
        can be used as an alternative for the other keyword agruments        
        """

        # Taken from the example code in pyPgSQL
        _d = {}
        if dsn is not None:
            try:
                params = dsn.split(":")
                if params[0] != '': _d["host"]     = params[0]
                if params[1] != '': _d["port"]     = params[1]
                if params[2] != '': _d["dbname"]   = params[2]
                if params[3] != '': _d["user"]     = params[3]
                if params[4] != '': _d["password"] = params[4]
                if params[5] != '': _d["options"]  = params[5]
                if params[6] != '': _d["tty"]      = params[6]
            except IndexError:
                pass

        if user is not None:
            _d["user"] = user
        if password is not None:
            _d["password"]  = password
        if host is not None:
            if ':' in host:
                _d["host"], _d["port"] = host.split(":")
            else:
                _d["host"] = host
        if database is not None:
            _d["dbname"] = database
        if port is not None:
            _d["port"] = port
        if options is not None:
            _d["options"] = options
        if tty is not None:
            _d["tty"] = tty

        conn_args = ' '.join(['%s=%s' % item for item in _d.items()])

        self.connection = libpq.PQconnectdb(conn_args)
        self.cache = {}
        self.tid_map = {}

    ###########################################################################
    # Utility methods

    def result_generator(self, cmnd):
        result = self.connection.query(cmnd)
        nrows, ncolumns = result.ntuples, result.nfields
        row = 0
        while row < nrows:
            column = 0
            while column < ncolumns:
                yield result.getvalue(row, column)
                column += 1
            row += 1

    def fetchall(self, cmnd):
        return list(self.result_generator(cmnd))

    def execute(self, cmnd):
        result = self.connection.query(cmnd)
        return result.oidValue

    ###########################################################################
    # Class registration methods

    def table_exists(self, cls):
        table_name = cls.__name__.lower()
        cmnd = "SELECT oid FROM pg_class WHERE relname='%s'" % table_name
        data = self.fetchall(cmnd)
        return bool(data)

    def get_tid(self, table_name):
        cmnd = "SELECT oid FROM pg_class WHERE relname='%s'" % table_name
        data = self.fetchall(cmnd)
        if len(data) != 1:
            raise db_base.DBError, "Expected to find 1 TID"
        else:
            return data[0]
        
    #######################################################################
    # Methods called by property objects

    def create_ref_columns(self, prop):
        return '%s oid, %s oid' % (prop.sql_tid_name, prop.sql_oid_name)

    def create_int_column(self, prop):
        return "%s int" % prop.sql_name
        
    def create_float_column(self, prop):
        return "%s double precision" % prop.sql_name
                                         
    def create_str_column(self, prop):
        return "%s text" % prop.sql_name

    def create_date_column(self, prop):
        return "%s date" % prop.sql_name

    def create_time_column(self, prop):
        return "%s time" % prop.sql_name

    def create_datetime_column(self, prop):
        return "%s timestamp" % prop.sql_name

    def quote_str(self, s):
        return libpq.PgQuoteString(s)

    def quote_date(self, s):
        return s.strftime("'%Y-%m-%d'")

    def quote_time(self, s):
        return s.strftime("'%H:%M:%S'")

    def quote_datetime(self, s):
        return s.strftime("'%Y-%m-%d %H:%M:%S'")

    def quote_tid_oid(self, val):
        if val is None:
            return "NULL", "NULL"
        elif isinstance(val, tuple):
            return str(val[0]), str(val[1])
        else:
            return str(val.tid), str(val.oid)

    def unquote_date(self, s):
        t = time.strptime(s, "%Y-%m-%d")
        return datetime.date(t.tm_year, t.tm_mon, t.tm_mday)

    def unquote_time(self, s):
        t = time.strptime(s, "%H:%M:%S")
        return datetime.time(t.tm_hour, t.tm_min, t.tm_sec)

    def unquote_datetime(self, s):
        t = time.strptime(s, "%Y-%m-%d %H:%M:%S")        
        return datetime.datetime(t.tm_year, t.tm_mon, t.tm_mday, t.tm_hour, t.tm_min, t.tm_sec)

    ###########################################################################

