"""Interface to MySQL databases

Requires mysql-python http://sourceforge.net/projects/mysql-python
"""

import db_base, sql_db
import time, datetime
import _mysql

conversion = {3: int,
              5: float}

class mysql_db(sql_db.sql_db):
    """An interface to a MySQL database
    """

    def __init__(self, host=None, user=None, passwd=None, db=None, port=None):
        """Create the interface

        The keyword arguments are passed to _mysql.connect()
        """
        kwargs = {}
        if host is not None: kwargs['host'] = host
        if user is not None: kwargs['user'] = user
        if passwd is not None: kwargs['passwd'] = passwd
        if db is not None: kwargs['db'] = db
        if port is not None: kwargs['port'] = port
        kwargs['conv'] = conversion
        
        self.connection = _mysql.connect(**kwargs)
        self.cache = {}
        self.tid_map = {}

    ###########################################################################
    # Utility methods

    def result_generator(self, cmnd):
        self.connection.query(cmnd)
        result = self.connection.store_result()
        for row in result.fetch_row(maxrows=0):
            for item in row:
                yield item

    def fetchall(self, cmnd):
        return list(self.result_generator(cmnd))

    def execute(self, cmnd):
        result = self.connection.query(cmnd)
        return self.connection.insert_id()

    ###########################################################################
    # Class registration methods

    def table_exists(self, cls):
        table_name = cls.__name__.lower()
        cmnd = "SHOW TABLES"
        for data in self.fetchall(cmnd):
            if data == table_name:
                return True
        return False

    def get_tid(self, table_name):
        return table_name
        
    #######################################################################
    # Methods called by property objects

    def create_ref_columns(self, prop):
        return '%s VARCHAR(32), %s INT' % (prop.sql_tid_name, prop.sql_oid_name)

    def create_int_column(self, prop):
        return "%s int" % prop.sql_name
        
    def create_float_column(self, prop):
        return "%s double precision" % prop.sql_name
                                         
    def create_str_column(self, prop):
        return "%s text" % prop.sql_name

    def create_date_column(self, prop):
        return "%s date" % prop.sql_name

    def create_time_column(self, prop):
        return "%s time" % prop.sql_name

    def create_datetime_column(self, prop):
        return "%s timestamp" % prop.sql_name

    def quote_str(self, s):
        return _mysql.string_literal(s)

    def quote_date(self, s):
        return s.strftime("'%Y-%m-%d'")

    def quote_time(self, s):
        return s.strftime("'%H:%M:%S'")

    def quote_datetime(self, s):
        return s.strftime("'%Y-%m-%d %H:%M:%S'")

    def quote_tid_oid(self, val):
        if val is None:
            return "NULL", "NULL"
        elif isinstance(val, tuple):
            return repr(val[0]), str(val[1])
        else:
            return repr(val.tid), str(val.oid)

    def unquote_date(self, s):
        t = time.strptime(s, "%Y-%m-%d")
        return datetime.date(t.tm_year, t.tm_mon, t.tm_mday)

    def unquote_time(self, s):
        t = time.strptime(s, "%H:%M:%S")
        return datetime.time(t.tm_hour, t.tm_min, t.tm_sec)

    def unquote_datetime(self, s):
        t = time.strptime(s, "%Y-%m-%d %H:%M:%S")        
        return datetime.datetime(t.tm_year, t.tm_mon, t.tm_mday, t.tm_hour, t.tm_min, t.tm_sec)

    ###########################################################################

    create_fmt = "CREATE TABLE %s (oid INT NOT NULL AUTO_INCREMENT, %s, PRIMARY KEY (oid))"
    insert_fmt = "INSERT INTO %s (oid, %s) VALUES (NULL, %s)"
    discard_oid = 1 # on retrieves
