"""Common code for all database interfaces

DBError -- One exception for all database operations
db_base -- Abstract base class for the database interfaces
"""

class DBError(Exception):
    pass

class db_base(object):
    """Abstract base class for the database interfaces

    The database interface consists of the following methods
    register_class -- called in ptype.__init__()
    get_instance   -- called in ptype.__call__()
    commit         -- called in pobject.commit()
    """

    def register_class(self, cls):
        """Register the class with the database

        If the class contains no persistent attributes an error is raised
        If the table does not exist, a table will be created
        """
        if not (cls.persistent_values or cls.persistent_refs):
            raise DBError, "No persistent attributes in %s" % cls

        if not self.table_exists(cls):
            self.create_table(cls)
        else:
            self.register_existing_class(cls)

    def table_exists(self, cls):
        """Return True if the table for class cls exists"""
        raise NotImplementedError

    def create_table(self, cls):
        """Create a table for class cls"""
        raise NotImplementedError

    def register_existing_class(self, cls):
        """Register class for which table exists"""
        raise NotImplementedError

    def get_instance(self, cls, *args, **kwargs):
        """Invoked when creating an object

        cls(*args, **kwargs) -> cls.database.get_instance(cls, *args, **kwargs)

        If 'oid' in kwargs, then return a known instance (from cache or db)
        else create a new instance using cls.make_new_instance(*args, **kwargs)
        """
        if 'oid' in kwargs:
            oid = kwargs['oid']
            if oid in self.cache[cls.tid]:
                return self.cache[cls.tid][oid]
            else:
                return self.retrieve_instance(cls, oid)
        else:
            return cls.make_new_instance(*args, **kwargs)

    def retrieve_instance(self, cls, oid):
        """Retrieve the object from the database

        Invoked when creating an object with known 'oid', which cannot be
        found in the cache
        """
        raise NotImplementedError

    def commit(self, instance):
        """Commit an instance to the database

        instance.commit() -> instance.database.commit()

        If invoked for an object without 'oid', an 'oid' will be generated and
        the object will be cached
        """
        raise NotImplementedError

    
