__version__   = "$Revision: 1.3 $"[11:-2]
__copyright__ = """Copyright (c) 2003 Not Another Corporation Incorporated
                   www.notanothercorporation.com"""
__license__ = """Licensed under the FSF GPL

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""
__doc__ = """* $Id: extunittest.py,v 1.3 2003/09/24 17:03:31 philiplindsay Exp $ *

    ** %s **  Ver %s
 
%s

%s

Provides I{External Unit Test} functionality.

This module allows for unit tests to be conducted outside the standard
C{unittest} framework, but for the results to be reported within the
framework.

The provided functionality is needed in situations where the execution
of a unit test would interfere with, or be affected by, the standard
C{unittest} framework. An example of this (and the reason for this
module's creation) is testing a replacement for the system exception
hook. (While a C{unittest} test runner is operating the system
exception hook is never called as there are no uncaught exceptions.)

A principle aim for the design of this module was to not require
external unit test code to exist in a file separate from the main test
code. This module includes some utilities to support this aim.

The simplest, lowest level of functionality is provided by the
C{ExternalTestCaseMixin} class.

A more complete (but application specific) solution is to sub-class
C{ExternalTestSuite} and make use of the C{run()} function of this
module.

Usage
-----

1) To get C{outputMatches()} functionality in your own class::

       import extunittest
       import unittest

       class MyClass(unittest.TestCase, extunittest.ExternalTestCaseMixin):
       
          def testOne(self):
              # We assume we have a file called 'echo.log' which contains
              # a line:
              #   Foo!
              self.outputMatches('echo Foo!', 'echo.log')
       
   This example will throw an exception if the output of the C{echo} command
   supplied does not match the pre-logged output.

2) To get C{ExternalTestSuite} functionality in your script::

        #
        # Filename: myscript.py
        #
        import extunittest

        class MyExternalTestSuite(extunittest.ExternalTestSuite):
            #
            writeFailFiles = False
            normaliseExceptions = True  # If your logs include exception
            ignoreLineNumChanges = True # tracebacks, set these.
            
        # --------------------
        # Your custom external unit tests (and normal tests too) go here...
        
        def extTestThrowExc():
            # Test throwing an exception produces a traceback.
            raise Exception('TestException')
        # --------------------
        
        if __name__ == '__main__':
            sys.exit(extunittest.run(sys.argv))

   Once you have done this you can then use your usual test-runner
   e.g. C{runall.py} which will need to be capable of loading your
   custom C{TestSuite} class.

   The first time a new external unit test function is added you'll see an
   error like this::

     AssertionError: Log file 'test/extTestSomething.log' does not exist.

   In this case you want to re-run the tests but make sure C{writeFailFiles}
   is C{True} (actually you probably want to keep it that way permanently).

   What will happen is that the actual output when run will be saved
   to a file named the same as the C{.log} file but with C{.fail}
   appended. You can check the output is correct and then simply
   rename the file to be the name of the C{.log} file.
   
   Make sure you check that the output is actually the correct result!

   When instantiated the C{ExternalTestSuite} class dynamically
   creates an instance of the class specified by C{testCaseClass} for
   each of the module-level functions in your module which have names
   starting with the specified prefix (C{extText} by default). It also
   initialises certain values in the class.

   The default test cases (instances of C{BaseExternalTestCase}) call
   the script with the name of the function as the argument, and
   capture the output from the script when it calls the specified
   function. This output is then compared to the existing C{.log} file
   and an exception is thrown if the output differs.

   The exception will appear as follows::

       AssertionError: Line number 5 differs (test/extTestThrowExc.log)

   or::

       AssertionError: Total number of lines differs (test/extTestThrowExc.log)
    
   If you want to ignore the first type of error you should make sure
   C{normaliseExceptions} and C{ignoreLineNumChanges} have values of
   True in your C{ExternalTestSuite} derived class. If these values
   are C{True} then lines which differ between the two outputs only if
   text of the form ', line \d+,' is present are ignored. (TODO: Allow
   this regexp to be specified by the caller?)

   NOTE: The ignore line number changes functionality listed above works well,
         but at present is only really suitable for Python exception
         tracebacks. As such it has been pulled into the code that
         normalises exceptions (making them cross-platform and file-system
         location independent). i.e. It is possible to normalise exceptions
         I{and} ignore line number changes, or normalise exceptions I{and not}
         ignore line number changes, but not possible to I{only} ignore line
         number changes. (TODO: Pull the line number ignoring code back out?)

   If you want to ignore the second type of error you probably don't want to
   be using this... :-)
   
""" % (__name__, __version__, __copyright__, __license__)


import os
import re
import sys

from errno import ENOENT

import unittest

try:
   True
except NameError:
   True = (1 == 1)
   False = not True

def getCommonPathSuffix(firstPath, secondPath, directorySeparator = os.sep):
    """
    Provides functionality that is intended to be somewhat similar to
    C{os.path.commonprefix}, except it looks for a common path suffix.

    This is intended to assist in stripping off non-common prefixes, such
    as when absolute paths are supplied.

    Example usage::

      winPath = 'c:/stuff/foo/bar.py'
      unixPath = '/things/foo/bar.py'

      commonSuffix = getCommonPathSuffix(winPath, unixPath, '/')

    C{commonSuffix} would now contain C{'foo/bar.py'}.

    Note: This looks for the common I{path} suffix, so for the following
          case there is I{no} common suffix (even though the C{'.py'} is
          common)::

      pathOne = '/things/foo/bar.py'
      pathTwo = '/things/foo/moo.py'

    Note: Assumes directory separator is C{directorySeparator}
          (for both input & output).
          
    TODO: Deal with '/foo/bar', 'e:/foo/bar' (leading '/') correctly.

    @param firstPath:  A file path.
    @param secondPath:  Another file path.
    @param directorySeparator:  The directory separator used in the paths.
    """

    index = 0
    commonSuffix = ""

    if firstPath == secondPath:
        return firstPath
    
    firstPathSplit = firstPath.split(directorySeparator)
    secondPathSplit = secondPath.split(directorySeparator)
    
    try:
        while firstPathSplit[index-1] == secondPathSplit[index-1]:
            index-=1
    except IndexError:
        pass

    if index:
        commonSuffix = directorySeparator.join(firstPathSplit[index:])

    return commonSuffix
    

# A regular expression which can be used to strip line number references
# from standard Python tracebacks.
LINE_NUMBER_RE = re.compile(", line \d+,") #TODO: Allow caller to specify this?
EXCEPTION_LINE_RE = re.compile("File \"([^\"]*)\",") # Match exception lines.
POSS_DIR_SEPS = r'/\:' # Known possible directory separators.
def normaliseExceptionLines(commandOutputLine, matchOutputLine,
                            ignoreLineNumberChanges):
    """
    'Normalises' lines that are part of a Python exception traceback
    and include a file path.

    Normalising involves making the line/string cross-platform
    compatible and file-system location independent. This is achieved
    by replacing the directoy separator and extracting a common suffix
    from the file paths.

    Note: It's all kinda iffy, but seems to work for the most part at
          the moment... (Note: It could quite easily get confused though,
          maybe. Especially with paths of the form 'e:\stuff\bar\baz.py' and
          '/things/bar/baz.py' where the path is absolute.)

    Optionally removes line number references so that mis-matches which are
    only due to line numbers in the tracebacks changing are ignored.

    TODO: Add unit tests for this.
    TODO: Make it all more elegant, and less... crappy.

    @param commandOutputLine: The line which was generated on this computer.
    @param matchOutputLine: The line which was in the log file.
    @param ignoreLineNumberChanges: If C{True} then line number references
                                    in the traceback are ignored.
    """
    normalisedCOL = commandOutputLine
    normalisedMOL = matchOutputLine

    ### First, deal with only line numbers changing...
    if ignoreLineNumberChanges:
        # ...by removing the line number references.
        normalisedCOL = LINE_NUMBER_RE.sub("", normalisedCOL)
        normalisedMOL = LINE_NUMBER_RE.sub("", normalisedMOL)

    ### Second, deal with different directory path separators...
    commandOutputLineREMatch = EXCEPTION_LINE_RE.search(commandOutputLine)
    matchOutputLineREMatch = EXCEPTION_LINE_RE.search(matchOutputLine)

    if commandOutputLineREMatch and matchOutputLineREMatch:
        # Extract the file path in the exception.
        filePathInCommandOutput = commandOutputLineREMatch.group(1)
        filePathInMatchOutput = matchOutputLineREMatch.group(1)

        # Find the last instance of the current platform's directory separator.
        directorySeparatorIndex = filePathInCommandOutput.rfind(os.sep)

        if directorySeparatorIndex > -1:            
            # Make the index be from the end of the path.
            directorySeparatorIndex = -(len(filePathInCommandOutput) -
                                        directorySeparatorIndex)
            
            # Look for a directory separator in the same position (relative
            # to the end of the file path) in the second file path.
            if filePathInMatchOutput[directorySeparatorIndex] in POSS_DIR_SEPS:
                # If it's a known directory separator replace all occurences
                # of it with this platform's directory separator.
                filePathInMatchOutput = filePathInMatchOutput.replace(
                    filePathInMatchOutput[directorySeparatorIndex], os.sep)
                
                ## Third, deal with different prefixes to absolute paths...
                # Find common suffix...
                commonSuffix = getCommonPathSuffix(filePathInCommandOutput,
                                                   filePathInMatchOutput)
                
                if commonSuffix:
                    normalisedCOL = normalisedCOL.replace(
                        filePathInCommandOutput, commonSuffix)
                    
                    normalisedMOL= normalisedMOL.replace(
                        matchOutputLineREMatch.group(1), commonSuffix)
                    

    return normalisedCOL, normalisedMOL
    

class ExternalTestCaseMixIn:
    """
    A mix-in which is intended to supply C{outputMatches()} functionality
    to a standard C{unittest.TestCase} instance.
    """

    def outputMatches(self, command, outputFileToMatch,
                      normaliseExceptions = True,
                      ignoreLineNumChanges = True,
                      writeFailFiles = False):
        """        
        Captures the output (produced on both C{stdout} and C{stderr}
        generated by running the specified command, and throws an
        exception if the output is not identical to the content of the
        named file (which should contain the previously generated
        output of the command).

        Optionally normalises lines which are part of a Python
        exception traceback and include a file path. Normalising
        involves making them cross-platform and file-system location
        independent. This is achieved by replacing the directoy
        separator and extracting a common suffix from the file
        paths. It's kinda iffy, but seems to work for the most part at
        the moment... (Note: It could quite easily get confused
        though, maybe. Especially with paths of the form
        'e:\stuff\bar\baz.py' and '/things/bar/baz.py' where the path
        is absolute.)

        Optionally ignores mis-matches which are only due to line numbers in
        Python tracebacks changing. (NOTE: This is only available when
        C{normaliseExceptions} is C{True}.)

        Optionally generates a file which contains the actual output of
        the command if it does match the previously generated output.

        This functionality is primarily designed to enable Python
        scripts which use custom system exception handlers and
        generate uncaught exceptions to be run external to the unit
        test framework so the framework doesn't catch the exception.

        TODO: Add additional error handling?
        
        @param command: The command whose output is to be captured & compared.
        @param outputFileToMatch: The name of the log file which contains the
                                  previously captured output of the command.
        @param normaliseExceptions: If C{True} then performs the actions
                                    described in the text above.
        @param ignoreLineNumChanges: If C{True} then lines in the output which
                                  differ only due to a change in the line
                                  number on which the error occurred are
                                  ignored. (NOTE: This is only available when
                                  C{normaliseExceptions} is C{True}.)
        @param writeFailFiles: If C{True} then when a mis-match occurs the
                               actual output of the command is written to a
                               file of the form C{<original filename>.log}.
        """
        # Run command & get output...
        (commandPipeStdIn, commandPipeStdOutAndErr) = os.popen4(command, 't')
        commandOutput = commandPipeStdOutAndErr.readlines()
        commandPipeStdIn.close()
        commandPipeStdOutAndErr.close()        
        
        # Get match output
        try:
            matchFile = file(outputFileToMatch)
            matchOutput = matchFile.readlines()
            matchFile.close()
        except IOError, e:
            if (e.errno != ENOENT):
                raise
            
            if writeFailFiles:
                _writeFile("%s.fail" % outputFileToMatch, commandOutput)
            self.fail("Log file '%s' does not exist." % (outputFileToMatch))
        
        # Check the outputs match:
        # Look for the most obvious difference first...
        if len(commandOutput) != len(matchOutput):
            if writeFailFiles:
                _writeFile("%s.fail" % outputFileToMatch, commandOutput)
            self.fail("Total number of lines differs (%s)" %
                      (outputFileToMatch))

        # Now check line-by-line...
        for index in range(len(commandOutput)):
            if commandOutput[index] != matchOutput[index]:

                # TODO: Add separate ignore line number changes functionality?
                if normaliseExceptions:
                    (commandOutputLine,
                     matchOutputLine) = normaliseExceptionLines(
                        commandOutput[index], matchOutput[index],
                        ignoreLineNumChanges)

                    if commandOutputLine == matchOutputLine:
                        continue
                    
                if writeFailFiles:
                    _writeFile("%s.fail" % outputFileToMatch, commandOutput)
                    
                self.fail("Line number %d differs (%s)" % (index + 1,
                                                           outputFileToMatch))

        return


# TODO: Maybe do something with meta-classes? (To allow changing BASE_COMMAND &
#       BASE_OUTPUT_FILE?)
#       (Note: You're not supposed to change the method signature of TestCase)
class BaseExternalTestCase(unittest.TestCase, ExternalTestCaseMixIn):
    """
    Instances of this class call the script file which contains the
    named module (which should contain a class that sub-classes
    C{ExternalTestSuite}), with the name of the function as the only
    argument.  The output from the script when it calls the specified
    function is then captured. This output is compared to the content
    of an existing C{.log} file and an exception is thrown if the
    output differs.

    This module uses functionality provided by C{ExternalTestCaseMixIn}.

    You must also include the following in your script::

       if __name__ == '__main__':
            sys.exit(extunittest.run(sys.argv))
    """
    
    BASE_COMMAND = "python %s %s"
    BASE_OUTPUT_DIR = "test"
    BASE_OUTPUT_FILE = os.path.join(BASE_OUTPUT_DIR, "%s.log")

    moduleName = ""
    functionName = ""
    
    writeFailFiles = False
    normaliseExceptions = True
    ignoreLineNumChanges = True
    
    def runTest(self):
        """
        Conducts the actual test.
        """
        command = self.BASE_COMMAND % \
                  (sys.modules[self.moduleName].__file__, self.functionName)
        outputfile = self.BASE_OUTPUT_FILE % (self.functionName)

        # We assume that we're being run from the root of the package
        # hierarchy, so we need to do this so that test scripts we run
        # can import packages.
        # TODO: Change this to append rather than obliterate PYTHONPATH?
        #       (Needs to be done in a cross platform way... Using os.pathsep)
        os.putenv("PYTHONPATH", os.getcwd())
        
        self.outputMatches(command, outputfile,
                           normaliseExceptions = self.normaliseExceptions,
                           ignoreLineNumChanges = self.ignoreLineNumChanges,
                           writeFailFiles = self.writeFailFiles)


class ExternalTestSuite(unittest.TestSuite):
    """
    Provides pre-packaged functionality to add external unit tests into
    your testing regime (yep, it's time for a regime change!).

    The easiest way to incorporate this into your unit testing script
    is to simply include the following::

       import extunittest
    
       class MyExternalTestSuite(extunittest.ExternalTestSuite):
          # Alternatively the following lines could be replaced with a
          # doc string here.

          writeFailFiles = False # (Optional)
          normaliseExceptions = True  # (Optional)
          ignoreLineNumChanges = True # (Optional)

          # testCaseClass = <sub class of unittest.TestCase>  (Alternative)

    This will cause the default test case class
    C{BaseExternalTestCase} to be used. See the documentation for
    C{BaseExternalTestCase} for further details on what this class
    needs in your script.
    """
    testCaseClass = BaseExternalTestCase

    writeFailFiles = False
    normaliseExceptions = True
    ignoreLineNumChanges = True
    
    def __init__(self, tests = ()):
        """
        When instantiated the C{ExternalTestSuite} class dynamically
        creates an instance of the class specified by C{testCaseClass}
        for each of the module-level functions in your module which
        have names starting with the specified prefix (C{extText} by
        default). It also initialises certain values in the class.
        """
        unittest.TestSuite.__init__(self, tests)
        testFunctions = getTestFunctions(sys.modules[self.__module__])

        for currFunctionName in testFunctions:
            # We have to create & initialise the instance in two steps because
            # you're not supposed to change the signature of the
            # 'TestCase' class's '__init__' method.
            externalTestCase = self.testCaseClass()

            # TODO: Allow this initialisation to be done by a function
            #       supplied by our sub-classee?
            externalTestCase.moduleName = self.__module__
            externalTestCase.functionName = currFunctionName
            externalTestCase.writeFailFiles = self.writeFailFiles
            externalTestCase.normaliseExceptions = self.normaliseExceptions
            externalTestCase.ignoreLineNumChanges = self.ignoreLineNumChanges
            
            self.addTest(externalTestCase)


def getTestFunctions(targetModule, testFunctionPrefix = "extTest"):
    """
    Returns a dictionary of function names (keys) and functions (values)
    from the target module whose names start with the C{testFunctionPrefix}.

    @param targetModule: The actual module (i.e. not its name).
    @param testFunctionPrefix: Function names which start with this string will
                               be included in the dictionary.
    """
    testFunctions = {}

    functionNames = [functionName
                     for functionName in dir(targetModule)
                       if functionName.startswith(testFunctionPrefix)]

    for functionName in functionNames:
        testFunctions[functionName] = getattr(targetModule,
                                              functionName)

    return testFunctions


def run(argv, targetModule = sys.modules["__main__"]):
    """
    A convenience function to allow scripts to simply use the construct::

        if __name__ == '__main__':
            sys.exit(extunittest.run(sys.argv))

    in order to automatically run the module-level function supplied
    as an argument to the script. e.g.::
    
        python <scriptname> <functionname>

    This functionality is used by the C{BaseExternalTestCase} class.
    """
    if (len(argv) != 2):
        print "Usage: %s <function name>" % (argv[0])
        sys.exit(1)

    return getTestFunctions(targetModule)[argv[1]]()

    
def _writeFile(fileName, content):
    """
    Creates the named file and writes the supplied sequence into it.

    Overwrites existing files.

    TODO: Actually have some error handling...

    @param fileName: The name of the file to write
    @param content: The sequence of lines which is written to the file.
    """
    print "\nWriting fail file %s" % (fileName)
    handle = file(fileName, "w")
    handle.writelines(content)
    handle.close()


