"""Steps and operators to build GraphPath expressions.

Synopsis:

	from graphpath.expr import Node, Nodes, Class, Subject, Any, Property, HasNo, Extract, trace

	These classes and the operators they define are used for composing GraphPath expressions.
	Other classes in this module are for implementation and/or extension of GraphPath.

Implementation notes:

	* Every GraphPath expression is an object of class Step.
	It may also inherit the mixins Singleton, Unary and Binary
	depending on how many sub-expressions or arguments it contains.

	* The simplest expressions match single nodes (zero-length paths) and
	are objects of class NodeStep.

	* The next simplest expressions match paths with two nodes connected by one arc.
	These are objects of class ArcStep.

	* More complex expressions are composed from these simple objects
	in recursively nested composite objects. The composite object classes are
	Predicate, Path, Union, Intersect, InverseOf, HasNo, Projection, Closure
	and BoundPath.

	* New composite objects are created by applying operators to (extant) Step
	objects. The composite classes are not intended to be imported or called directly.
	Class Step defines all the GraphPath operators (except binding '>>') and delegates
	them to the composite class contructors.

	* The result of evaluating an expression (Step object) depends on
	its class and the context of evaluation. Each Step object
	possesses the following evaluation methods which acccept and/
	or return python sets:

	* terminals(g) matches the step in graph g and returns a set of
	the matching paths' terminal nodes.

	* initials(g) matches the step in graph g and returns a set of
	the matching paths' initial nodes.

	* values(g, i) matches the step in graph g as part of a traversal.  Only
	paths with initial nodes in i are considered and their terminal nodes are
	returned.

	* match(g, t) to match the step in graph g as part of an inverse traversal.
	Only paths with terminal nodes in t are considered and their initial nodes
	are returned.

	* Not all steps can be used in all contexts.  Where an operation is
	not supported StrategyError is raised.  The caller can catch this and try a
	different strategy to obtain a result.  In particular, not all steps
	support terminals() and initials() methods.   The following alternatives
	are called in this case:

	* filter_terminals(g, ts) iterates the ts sequence and yields those elements
	that belong to terminals().

	* filter_initials(g, is) iterates the is sequence and yields those elements
	that belong to initials().

	* Graphs are represented by objects that support the population protocol. See
	graphpath.store.Population.  The main methods are values(s, p) and match(p, o)
	where s and p are subject and property RDF nodes and o is an RDF literal
	object, a string, or another python object that can be wrapped an RDF literal.

	* Not all of the population protocol need be implemented. A graph data source
	can raise StrategyError for unsupported operations.  This, for example,
	should allow GraphPath to be used with a graph that can only be tranversed in
	one direction: supporting values(s, p) but not match(p, o).


"""
from __future__ import generators
from util.anysets import Set, ImmutableSet, protect
empty_set = ImmutableSet()

try:
	from itertools import chain, ifilter
except ImportError:
	def chain(s1, s2):
		for item in s1:
			yield item
		for item in s2:
			yield item
	def ifilter(p, s):
		for item in s:
			if p(s):
				yield item

class StrategyError(NotImplementedError):
	"""expression cannot be evaluated in the given context (or no context provided)"""
	def __init__(self, expr=None):
		NotImplementedError.__init__(self)
		self._expr = expr
	def __str__(self):
		if self._expr:
			return self.__doc__ + ": " + str(self._expr)
		else:
			return self.__doc__

class Step(object):
	"""The base class for a GraphPath expression.

	Step defines the operators used in an expression and
	provides the default implementations for
	evaluating an expression.  (The defaults raise
	StrategyError.)

	Custom operands should inherit this class
	in order to work with the operators."""
	def match(self, pop, values):
		"""Match paths with given ending nodes and return starting nodes."""
		raise StrategyError, self
	def values(self, pop, subjects):
		"""Match paths with given starting nodes and return ending nodes."""
		raise StrategyError, self
	def terminals(self, pop):
		"""Match paths and return ending nodes."""
		raise StrategyError, self
	def initials(self, pop):
		"""Match paths and return starting nodes."""
		raise StrategyError, self
	def filter_initials( self, pop, sequ ):
		"""Iterate the nodes from a sequence that statisfy a predicate.
		A node satisfies if it is a member of initials()."""
		pred = self.initials(pop)
		for node in sequ:
			if node in pred:
				yield node
	def filter_terminals( self, pop, sequ ):
		"""Iterate the nodes from a sequence that statisfy a
		path. A node satisfies if it is a member of terminals()."""
		pred = self.terminals(pop)
		for node in sequ:
			if node in pred:
				yield node

	def __div__( self, other):
		"""Path traversal operator."""
		return Path(self, other)
	def __floordiv__(self, other):
		"""Transitive path traversal operator."""
		return Closure(self, other)
	def __mod__(self, other):
		"""Projection operator."""
		return Projection(self, other)
	def __getitem__( self, other ):
		"""Predicate operator."""
		return Predicate( self, other )
	def __and__( self, other):
		"""Intersection operator."""
		return Intersect(self, other)
	def __or__(self, other):
		"""Union operator."""
		return Union(self, other)
	def __invert__(self):
		"""Inverse path operator."""
		return InverseOf(self)

	def __rrshift__(self, pop):
		"""Context binding operator."""
		return BoundPath( pop, self )
	def __rlshift__(self, pop):
		"""Evaluation operator.

		Note: experimental and subject to removal.
		"""
		for result in self.terminals( pop ):
			return result
		else:
			raise ValueError, "path yields empty set"
	def __str__(self):
		# for these classes the str()==repr()
		# and both are nearly valid python
		return repr(self)
	def __iter__(self):
		"""Iterate the terminals of all matched paths.
		Generally, the expression must be bound or this
		will raise StrategyError."""
		return iter( self.terminals( population_place_holder ))
	def trace(self, tracer):
		"""Default trace implementation simply wraps the whole expression
		with the tracer.  The tracer must be a function that accepts a
		Step amd returms a Step."""
		return tracer( self )

class Singleton(object):
	"""A base for the singleton terms.

	All instances of a singleton operand are equivalent."""
	def __eq__(self, other):
		return self.__class__ == other.__class__
	def __repr__(self):
		return "%s()" % self.__class__.__name__

class Unary(object):
	"""A base for single-argument terms."""
	def __init__(self, arg):
		self._arg = arg
	def __eq__(self, other):
		return self.__class__ == other.__class__ \
			and self._arg == other._arg
	def __repr__(self):
		return "%s(%r)" % (self.__class__.__name__, self._arg)

class Binary(object):
	"""A base for two-argument terms."""
	def __init__(self, lhs, rhs):
		self._lhs, self._rhs = lhs, rhs
	def __eq__(self, other):
		return self.__class__ == other.__class__ \
			and self._lhs == other._lhs \
			and self._rhs == other._rhs
	def __repr__(self):
		return "%s(%r, %r)" % (self.__class__.__name__, self._lhs, self._rhs)

class UnaryOp(Unary, Step):
	"""A base for single-argument operators.
	The argument is a Step object."""
	def trace(self, tracer):
		arg = self._arg.trace( tracer )
		return tracer( self.__class__( arg ))

class BinaryOp(Binary, Step):
	"""A base for two-argument operators.
	The lhs and rhs arguments are both Step objects"""
	def trace(self, tracer):
		lhs = self._lhs.trace( tracer )
		rhs = self._rhs.trace( tracer )
		return tracer( self.__class__( lhs, rhs ))

class Path(BinaryOp):
	"""Implements the path operator: a/b"""
	def match(self, pop, values):
		return self._lhs.match(pop, self._rhs.match(pop, values))
	def values(self, pop, subjects):
		return self._rhs.values(pop, self._lhs.values(pop, subjects))
	def initials(self, pop):
		return self._lhs.match( pop, self._rhs.initials(pop))
	def terminals(self, pop):
		return self._rhs.values(pop, self._lhs.terminals(pop))
	def filter_initials( self, pop, subjects ):
		for subject in subjects:
			values = self._lhs.values( pop, Set([subject]) )
			for value in self._rhs.filter_initials( pop, values ):
				yield subject
				break
	def filter_terminals( self, pop, values ):
		for value in values:
			subjects = self._rhs.match( pop, Set([value]) )
			for subject in self._rhs.filter_terminals( pop, subjects ):
				yield value
				break
	def __repr__(self):
		return "%r/%r" % (self._lhs, self._rhs)
	def __invert__(self):
		return ~self._rhs/~self._lhs

class Predicate(BinaryOp):
	"""Implements the predicate operator: a[b].

	Predicate evaluation requires the initial nodes of the
	rhs matches to be found.  In each method, the initials
	are sought explicitly first via initials() then implicitly
	via filter_initials(). The initials() method is likely
	to be more efficient when available.

	But the initials() method is liable to fail if the predicate
	consists of an (1) ArcStep or (2) a Path whose rhs is
	an ArcStep or (3) a Path with a lhs not supporting
	match() or bound to a context not supporting
	match() i.e. no inverse property traversal."""

	def match(self, pop, values):
		try:
			initials = self._rhs.initials(pop)
		except StrategyError:
			filtered = Set(self._rhs.filter_initials(pop, values))
		else:
			filtered = values & initials
		return self._lhs.match(pop, filtered)
	def values(self, pop, subjects):
		terminals = self._lhs.values(pop, subjects)
		try:
			initials = self._rhs.initials(pop)
		except StrategyError:
			return Set(self._rhs.filter_initials(pop, terminals))
		else:
			return terminals & initials
	def terminals(self, pop):
		try:
			terminals = self._lhs.terminals(pop)
		except StrategyError:
			try:
				initials = self._rhs.initials(pop)
			except StrategyError:
				return Set(self._rhs.filter_initials(pop, self._lhs.filter_terminals(pop, iter(pop))))
			else:
				return Set(self._lhs.filter_terminals(pop, initials))
		else:
			try:
				initials = self._rhs.initials(pop)
			except StrategyError:
				return Set(self._rhs.filter_initials(pop, terminals))
			else:
				return terminals & initials
	def initials(self, pop):
		try:
			initials = self._rhs.initials(pop)
		except StrategyError:
			initials = Set(self._rhs.filter_initials(pop, pop))
		return self._lhs.match(pop, initials)
	def __repr__(self):
		return "%r[%r]" % (self._lhs, self._rhs)
	def __invert__(self):
		return Self()[self._rhs]/~self._lhs

class Union(BinaryOp):
	"""Implements the union operator: a|b"""
	def match(self, pop, values):
		return self._rhs.match(pop, values) | self._lhs.match(pop, values)
	def values(self, pop, values):
		return self._lhs.values(pop, values) | self._rhs.values(pop, values)
	def terminals(self, pop):
		return self._lhs.terminals(pop) | self._rhs.terminals(pop)
	def initials(self, pop):
		return self._lhs.initials(pop) | self._rhs.initials(pop)
	def filter_initials(self, pop, subjects):
		return chain( self._lhs.filter_initials( pop, subjects),
				self._rhs.filter_initials( pop, subjects))
	def filter_terminals(self, pop, values):
		return chain( self._lhs.filter_terminals( pop, values),
				self._rhs.filter_terminals( pop, values))
	def __repr__(self):
		return "(%r|%r)" % (self._lhs, self._rhs)
	def __invert__(self):
		return ~self._rhs | ~self._lhs

class Intersect(BinaryOp):
	"""Implements the intersection operator: a&b"""
	def match(self, pop, values):
		return self._rhs.match(pop, values) & self._lhs.match(pop, values)
	def values(self, pop, values):
		return self._lhs.values(pop, values) & self._rhs.values(pop, values)
	def terminals(self, pop):
		return self._lhs.terminals(pop) & self._rhs.terminals(pop)
	def initials(self, pop):
		return self._lhs.initials(pop) & self._rhs.initials(pop)
	def filter_initials(self, pop, subjects):
		return self._lhs.filter_initials( pop,
				self._rhs.filter_initials( pop, subjects))
	def filter_terminals(self, pop, values):
		return self._lhs.filter_terminals( pop,
				self._rhs.filter_terminals( pop, values))
	def __repr__(self):
		return "%r&%r" % (self._lhs, self._rhs)
	def __invert__(self):
		return ~self._rhs & ~self._lhs

class InverseOf(UnaryOp):
	"""Implements the invert operator: ~a"""
	def match(self, pop, values):
		return self._arg.values(pop, values)
	def values(self, pop, subjects):
		return self._arg.match(pop, subjects)
	def terminals(self, pop):
		return self._arg.initials(pop)
	def initials(self, pop):
		return self._arg.terminals(pop)
	def filter_initials(self, pop, subjects):
		return self._arg.filter_terminals( pop, subjects)
	def filter_terminals(self, pop, values):
		return self._arg.filter_initials( pop, values)
	def __repr__(self):
		return "~%r" % self._arg

class ArcStep(Unary, Step):
	"""Base class for the property steps."""
	def _values(self, pop, subjects):
		prop = self._arg
		it = iter(subjects)
		result = empty_set
		try:
			result = pop.values(it.next(), prop)
			next = pop.values(it.next(), prop)
			result = Set(result)
			while True:
				result |= next
				next = pop.values(it.next(), prop)
		except StopIteration:
			pass
		return result
	def _match(self, pop, values):
		prop = self._arg
		result = Set()
		for value in values:
			result |= pop.match( prop, value )
		return result
	def _filter_initials(self, pop, values):
		prop = self._arg
		for cand in values:
			if pop.values( cand, prop ):
				yield cand
	def _filter_terminals(self, pop, values):
		prop = self._arg
		for cand in values:
			if pop.match( prop, cand ):
				yield cand
	def resource(self):
		return self._arg

class Property(ArcStep):
	"""Property step.  Matches all paths crresponding to a given property."""
	values, match, filter_terminals, filter_initials = \
		ArcStep._values, \
		ArcStep._match, \
		ArcStep._filter_terminals, \
		ArcStep._filter_initials

	def __invert__(self):
		return Inverse(self._arg)
	def __div__(self, other):
		if isinstance( other, Node ):
			return Match( self._arg, other._arg )
		else:
			return Path( self, other )

class Inverse(ArcStep):
	"""Inverse Property step, matches the reverse of the paths
	that correspond to the property.
	Note: used to implement ~Property(u) as an optimisation.
	"""
	values, match, filter_terminals, filter_initials = \
		ArcStep._match, \
		ArcStep._values, \
		ArcStep._filter_initials, \
		ArcStep._filter_terminals

	def __invert__(self):
		return Property(self._arg)
	def __repr__(self):
		return "(~Property(%r))" % self._arg

class HasNo(UnaryOp):
	"""HasNo predicate inverts the sense of the argument predicate.

	Note: breaks monotonic inference principle - use with care.
	"""
	def filter_initials(self, pop, values):
		initials = Set(values)
		return iter(initials - Set(self._arg.filter_initials( pop, initials)))

class ManyValued(UnaryOp):
	"""ManyValued predicate matches nodes with many paths
	that match the argument predicate.
	"""
	def filter_initials(self, pop, values):
		for cand in values:
			if len(self._arg.values( pop, Set([cand]))) > 1:
				yield cand

class Match(Binary, Step):
	"""Match step, implements Property(p)/Node(n).

	Note: used as an optimisation.
	"""
	def initials( self, pop ):
		return pop.match(self._lhs, self._rhs)
	def terminals( self, pop ):
		return Set([self._rhs])
	def match(self, pop, values):
		if self._rhs in values:
			return self.initials(pop)
		else:
			return empty_set
	def values(self, pop, subjects):
		if subjects & self.initials(pop):
			return Set([self._rhs])
		else:
			return empty_set
	def __repr__(self):
		return "(Property(%r)/Node(%r))" % (self._lhs, self._rhs)

class NodeStep(Step):
	"""A mixin for any expression that matches zero-length paths.

	A set of zero-length paths is just a set of nodes.
	In this case:
		values() == match() and
		filter_initials() == filter_terminals() and
		initials() == terminals() and
		~self == self
	The latter methods should be implemented in subclasses,
	although Step provides a filter_terminals() implementation.
	"""
	def values( self, pop, subjects ):
		return self.match( pop, subjects )
	def filter_initials( self, pop, subjects ):
		return self.filter_terminals( pop, subjects )
	def initials( self, pop ):
		return self.terminals( pop )
	def __invert__(self):
		return self
		
class Filter(Unary, NodeStep):
	"""Filter predicate matches nodes for which a python function returns true.

	The argument of Filter() gives the matching function (typically a lambda).

	Note: this creates an escape to python code that could interfere with
	the inference mechanism.  To avoid this, the function should only
	depend on its argument and constants.  Best used with literals.
	"""
	def filter_terminals(self, pop, values):
		return ifilter( self._arg, values )

class Subject(Singleton, NodeStep):
	"""Subject step, matches a path consisting of a single subject node.

	See also: Any()
	"""
	def terminals(self, pop):
		return Set(pop)
	def match(self, pop, values ):
		return Set(self.filter_terminals(pop, values))
	def filter_terminals(self, pop, values ):
		for value in values:
			if value in pop:
				yield value

class Any(Singleton, NodeStep):
	"""Any or Self Step, matches any single node.

	The terminals() operation is not possible."""
	def filter_terminals(self, pop, values):
		return iter(values)
	def match(self, pop, values):
		return values

Self = Any # alias that reads more naturally in some expressions and follows XPath self()

class Nodes(Unary, NodeStep):
	"""Nodes step.  Matches paths consisting of a node
	from the given set.
	"""
	def __init__(self, seeds):
		Unary.__init__(self, ImmutableSet(seeds))
	def terminals(self, pop):
		return self._arg
	def match( self, pop, values ):
		return values & self._arg
	def filter_terminals(self, pop, values ):
		for value in values:
			if value in self._arg:
				yield value

class Node(Unary, NodeStep):
	"""Nodes step.  Matches paths consisting of a given node."""
	def terminals(self, pop):
		return Set([self._arg])
	def match( self, pop, values ):
		if self._arg in values:
			return Set([self._arg])
		else:
			return empty_set
	def filter_terminals( self, pop, values ):
		if self._arg in values:
			yield self._arg

class Class(Unary, NodeStep):
	"""Class step.  Matches paths consisting of a node of the given type."""
	def resource(self):
		return self._arg
	def terminals(self, pop):
		return pop.match( pop.rdf_type, self._arg )
	def match( self, pop, values ):
		return values & self.terminals(pop)

class Projection(Binary, Step):
	"""Implements the projection operator: a%b"""
	def _eval_relation(self, pop, seeds):
		result = Set()
		if isinstance(self._rhs, tuple):
			for seed in seeds:
				result.add( tuple( self._eval_tuple(pop, seed)))
		else:
			for seed in seeds:
				for single in self._rhs.values( pop, Set([seed]) ):
					result.add( single )
					break
				else:
					result.add( None )
		return result
	def _eval_tuple(self, pop, seed):
		for path in self._rhs:
			for col in path.values( pop, Set([seed]) ):
				yield col
				break
			else:
				yield None
	def terminals(self, pop):
		return self._eval_relation( pop, self._lhs.terminals( pop ))
	def values(self, pop, subjects):
		return self._eval_relation( pop, self._lhs.values( pop, subjects ))
	def trace(self, tracer):
		lhs = self._lhs.trace( tracer )
		if isinstance(self._rhs, tuple):
			rhs = tuple([path.trace( tracer ) for path in self._rhs])
		else:
			rhs = self._rhs.trace( tracer )
		return tracer( Projection( lhs, rhs ))
	def __repr__(self):
		return "%r%%%r" % (self._lhs, self._rhs)

def _traverse(pop, seeds, step):
	"""Helper for the transitive closure operator.

	Applies the step function repeatedly to expand seeds."""
	results = Set()
	partial = seeds
	while partial:
		new_results = step( pop, partial )
		partial = new_results - results
		results |= partial
		partial -= seeds
	return results

class Closure(BinaryOp):
	"""Implements the transitive closure operator: a//b"""
	def match(self, pop, values):
		return self._lhs.match( pop, _traverse( pop, values, self._rhs.match ))
	def values(self, pop, subjects):
		return _traverse( pop, self._lhs.values(pop, subjects), self._rhs.values )
	def terminals(self, pop):
		return _traverse(pop, self._lhs.terminals(pop), self._rhs.values)
	def __repr__(self):
		return "%r//%r" % (self._lhs, self._rhs)

# this is a list of context adapters. it is built by adapter modules eg redadapt.py
adapters=[]

class BoundPath(Binary, Step):
	"""Implements the binding operator: p>>b"""
	def __init__(self, lhs, rhs):
		"""adapt the lhs context and bind to the rhs expression"""
		for adapter in adapters:
			try:
				adapted = adapter( lhs )
				break
			except StrategyError:
				pass
		else:
			adapted = lhs # by default use the lhs as-is
		Binary.__init__( self, adapted, rhs )
	def match(self, pop, values):
		return self._rhs.match( self._lhs, values )
	def values(self, pop, subjects):
		return self._rhs.values( self._lhs, subjects )
	def terminals(self, pop):
		return self._rhs.terminals( self._lhs )
	def initials(self, pop):
		return self._rhs.initials( self._lhs )
	def filter_terminals(self, pop, values):
		return self._rhs.filter_terminals( self._lhs, values )
	def filter_initials(self, pop, subjects):
		return self._rhs.filter_initials( self._lhs, subjects )
	def trace(self, tracer):
		rhs = self._rhs.trace( tracer )
		return BoundPath( self._lhs, rhs )
	def __iter__(self):
		return iter( self._rhs.terminals( self._lhs ))
	def __repr__(self):
		return "%r>>(%r)" % (self._lhs, self._rhs)

class TraceStep(Unary, Step):
	"""A transparent wrapper for a step that reports its results/"""
	def match(self, pop, values):
		result = self._arg.match( pop, values )
		self._display("match", result)
		return result
	def values(self, pop, subjects):
		result = self._arg.values( pop, subjects )
		self._display("values", result)
		return result
	def terminals(self, pop):
		result = self._arg.terminals( pop )
		self._display("terminals", result)
		return result
	def initials(self, pop):
		result = self._arg.initials( pop )
		self._display("initials", result)
		return result
	def filter_terminals(self, pop, values):
		result = list(self._arg.filter_terminals( pop, values ))
		self._display("filter_terminals", result)
		return iter(result)
	def filter_initials(self, pop, subjects):
		result = list(self._arg.filter_initials( pop, subjects ))
		self._display("filter_initials", result)
		return iter(result)
	def trace(self, tracer):
		arg = self._arg.trace( tracer )
		return TraceStep( arg )
	def _display(self, method, result):
		print "%r %s={" % (self._arg, method),
		show, n = 2, 0
		for node in result:
			if n:
				print ",",
			if show == n:
				if n:
					print "%d more node(s)" % (len(result)-n),
				else:
					print "%d node(s)" % len(result),
				break
			print repr(node),
			n += 1
		print "}"
	def __iter__(self):
		return iter( self._arg )
	def __repr__(self):
		return "*" + repr(self._arg)

def trace( expr, tracer=TraceStep ):
	"""Convert the given expression to a traced expression.

	Assuming the default tracer argument, the traced
	expression is equivalent to the original expression
	but prints out each step in its evaluation for
	debugging.

	Alternative forms of trace output can be created
	by supplying a custom tracer argument.  This would
	normally be a class derived from TraceStep with
	an overide for the _display() method."""
	return expr.trace( tracer )

class Map:
	"""Implement the core of the mapping protocol over a GraphPath expression.

	If dg=Map(expr) then:

	iter(dg) iterates the initial nodes of all paths matching expr.

	dg[key] is the set of terminal nodes of any matching paths where
	key is the initial node.

	The values() and items() methods deviate slightly from the
	usual protocol.

	itervalues() and values() returns all terminal nodes as iterator
	and iterable respectively.

	iteritems() and items() returns all (initial, terminal) tuples
	as iterator and iterable respectively.

	In addition, the keys(), values() and items() methods return
	Set's for their iterable result rather than lists.
	"""
	def __init__(self, expr):
		self._expr = expr

	def __getitem__(self, subject, default=None):
		return self._expr.values(population_place_holder, Set([subject]))
	get = __getitem__

	def __iter__(self):
		return iter(self._expr.initials(population_place_holder))
	iterkeys = __iter__
	def keys(self):
		return self._expr.initials(population_place_holder)

	def itervalues(self):
		return iter(self._expr.terminals(population_place_holder))
	def values(self):
		self._expr.terminals(population_place_holder)

	def iteritems(self):
		for key in self:
			for value in self[key]:
				yield key, value
	def items(self):
		return Set(self.iteritems)

class PopulationBase:
	"""A base class for graph data structures.

	This class provides all the required methods for
	a GraphPath compatible graph data structure or adapter.
	However, the provided methods raise StategyError.
	"""
	def values(self, subject, property):
		"""Return a (possibly immutable) set of values for
		statements (subject, property, *)."""
		raise StrategyError
	def match(self, property, value):
		"""Return a (possibly immutable) set of subjects for
		statements (*, property, value)."""
		raise StrategyError
	def __iter__(self):
		"""Iterate the (unique) subjects of all statements."""
		raise StrategyError

	rdf_type=None # the rdf:type symbol for this graph

population_place_holder=PopulationBase()

