#!/usr/bin/env python
import user, importer

# ----------------------------------------------------------------------
# exprlex.py
#
# A lexer for PDE Expressions.
# ----------------------------------------------------------------------

import lex

# Reserved words
reserved = (
  'COS', 'CURL', 'DET', 'DIV', 'GRAD', 'INV', 'SIN', 'TRANS', 'VEC'
  )

tokens = reserved + (
  # Literals (scoped identifier, identifier, integer constant, float constant, string constant, char const)
  'ICONST', 'FCONST',

  # Delimeters < > ( ) [ ] { } | , ; :
  'LT', 'GT',
  'LPAREN', 'RPAREN',
  'LBRACKET', 'RBRACKET',
  'LBRACE', 'RBRACE',
  'VERT',
  'COMMA', 'SEMICOLON', 'COLON',

  # Operators + - * / ^ .
  'PLUS', 'MINUS', 'TIMES', 'DIVIDE', 'CARET', 'PERIOD',

  # Variables
  'VAR',

  # Comments
  'COMMENT'
  )

# Newlines
def t_NEWLINE(t):
  r'\n+'
  t.lineno += t.value.count('\n')

# Completely ignored characters
t_ignore    = ' \t\x0c'
# Delimeters
t_LT        = r'<'
t_GT        = r'>'
t_LPAREN    = r'\('
t_RPAREN    = r'\)'
t_LBRACKET  = r'\['
t_RBRACKET  = r'\]'
t_LBRACE    = r'\{'
t_RBRACE    = r'\}'
t_VERT      = r'\|'
t_COMMA     = r','
t_SEMICOLON = r';'
t_COLON     = r':'
# Operators
t_PLUS      = r'\+'
t_MINUS     = r'-'
t_TIMES     = r'\*'
t_DIVIDE    = r'/'
t_CARET     = r'\^'
t_PERIOD    = r'\.'

# Identifiers and reserved words
reserved_map = { }
for r in reserved:
  reserved_map[r.lower()] = r

# Variables
def t_VAR(t):
  r'[A-Za-z_][\w_-]*'
  t.type = reserved_map.get(t.value, 'VAR')
  return t

# Integer literal
t_ICONST = r'[+-]?\d+([uU]|[lL]|[uU][lL]|[lL][uU])?'

# Floating literal
t_FCONST = r'[+-]?((\d+)(\.\d+)(e(\+|-)?(\d+))? | (\d+)e(\+|-)?(\d+))([lL]|[fF])?'

# Comments 
def t_COMMENT(t):
  r'/\*(.|\n)*?\*/|//(.*)|/\*[^*]*\*+([^/*][^*]*\*+)*/|"(\\.|[^"\\])*"'
  return t
    
def t_error(t):
  print "Illegal character %s [line %d]" % (repr(t.value[0]), t.lineno)
  t.skip(1)

lexer = lex.lex(optimize = 1)
if __name__ == "__main__":
  lex.runmain(lexer)
