# Comet VOEvent Broker.
# VOEvent message parsing & definition.

# Python standard library
import re
from datetime import datetime

# XML parsing using lxml
import lxml.etree as ElementTree

from comet import __version__, __url__
import comet.log as log
from comet.utility.xml import xml_document

__all__ = ["parse_ivoid", "broker_test_message"]

ElementTree.register_namespace("voe", "http://www.ivoa.net/xml/VOEvent/v2.0")

IVOID_RE = re.compile("""ivo://
                         (?P<auth>[a-zA-Z0-9][\w\-.~*'()]{2,})       # Authority
                         (?P<rsrc>/[\w\-\.~\*'()/]*)? \#?            # Resource name
                         (?P<localID>[\w\-\.~\*'()\+=/%!$&,;:@?]*) $ # Fragment
                      """, re.VERBOSE)

def parse_ivoid(ivoid):
    """
    Takes an IVOID of the form

        ivo://[authorityID][resourceKey]#[local_ID]

    and returns (authorityID, resourceKey, local_ID). Raise if that isn't
    possible.

    Note that the resourceKey will normally start with a slash. This is part
    of the key, and this function will not trim it.

    Refer to the IVOA Identifiers Recommendation (2.0) for details.
    """
    try:
        groups = IVOID_RE.match(ivoid).groups()

        # If there's n
        rsrc = groups[1] if groups[1] is not None else ""

        # These may not appear in the resource key per IVOA Identifiers
        # Version 2.0 \S2.3.3.
        for forbidden in ['//', '/../', '/./']:
            assert(forbidden not in rsrc)
        assert(not rsrc.endswith('/'))

        return groups[0], rsrc, groups[2]
    except (AttributeError, AssertionError) as e:
        log.debug("Failed to parse as IVOID: ", str(e))
        raise Exception("Invalid IVOID: %s" % (ivoid,))

def broker_test_message(ivo):
    """
    Test message which is regularly broadcast to all subscribers.
    """
    root_element = ElementTree.Element("{http://www.ivoa.net/xml/VOEvent/v2.0}VOEvent",
        attrib={
            "ivorn": ivo + "#TestEvent-%s" % datetime.utcnow().strftime("%Y-%m-%dT%H:%M:%SZ"),
            "role": "test",
            "version": "2.0",
            "{http://www.w3.org/2001/XMLSchema-instance}schemaLocation": "http://www.ivoa.net/xml/VOEvent/v2.0 http://www.ivoa.net/xml/VOEvent/VOEvent-v2.0.xsd"
        }
    )
    who = ElementTree.SubElement(root_element, "Who")
    author_ivoid = ElementTree.SubElement(who, "AuthorIVORN")
    author_ivoid.text = ivo
    date = ElementTree.SubElement(who, "Date")
    date.text = datetime.utcnow().strftime("%Y-%m-%dT%H:%M:%SZ")
    what = ElementTree.SubElement(root_element, "What")
    description =  ElementTree.SubElement(what, "Description")
    description.text = "Broker test event generated by Comet %s." % (__version__,)
    ElementTree.SubElement(what, "Reference", uri=__url__)
    return xml_document(root_element)
