# -*- coding: utf-8 -*-
"""
Otsu et al. (2018) - Reflectance Recovery
=========================================

Defines the datasets for reflectance recovery using *Otsu et al. (2018)*
method.

References
----------
-   :cite:`Otsu2018` : Otsu, H., Yamamoto, M., & Hachisuka, T. (2018).
    Reproducing Spectral Reflectances From Tristimulus Colours. Computer
    Graphics Forum, 37(6), 370-381. doi:10.1111/cgf.13332
"""

from __future__ import unicode_literals

import numpy as np
from colour.colorimetry import SpectralShape

__author__ = 'Colour Developers'
__copyright__ = 'Copyright (C) 2013-2020 - Colour Developers'
__license__ = 'New BSD License - https://opensource.org/licenses/BSD-3-Clause'
__maintainer__ = 'Colour Developers'
__email__ = 'colour-developers@colour-science.org'
__status__ = 'Production'

__all__ = [
    'SPECTRAL_SHAPE_OTSU2018', 'BASIS_FUNCTIONS_OTSU2018',
    'CLUSTER_MEANS_OTSU2018', 'SELECTOR_ARRAY_OTSU2018'
]

SPECTRAL_SHAPE_OTSU2018 = SpectralShape(380, 730, 10)
SPECTRAL_SHAPE_OTSU2018.__doc__ = """
The spectral shape of *Otsu et al. (2018)* basis functions and means.

References
----------
:cite:`Otsu2018`

SPECTRAL_SHAPE_OTSU2018 : SpectralShape
"""

BASIS_FUNCTIONS_OTSU2018 = np.array([
    [
        [
            0.033359794,
            0.069816766,
            0.145858662,
            0.208587748,
            0.225045781,
            0.230260467,
            0.234686490,
            0.237479132,
            0.237922746,
            0.235743337,
            0.230636966,
            0.224868019,
            0.220538851,
            0.214245998,
            0.205011936,
            0.194449589,
            0.182172137,
            0.166753810,
            0.149673107,
            0.135884660,
            0.127383558,
            0.121562634,
            0.115070543,
            0.110303774,
            0.108658862,
            0.110404095,
            0.114133490,
            0.118971322,
            0.123157804,
            0.123054245,
            0.119622289,
            0.116466163,
            0.114699356,
            0.115051812,
            0.118079600,
            0.125700322,
        ],
        [
            -0.015712659,
            -0.036497435,
            -0.083631104,
            -0.123972215,
            -0.129432141,
            -0.113757935,
            -0.074978412,
            -0.023837819,
            0.051049315,
            0.138878712,
            0.218967535,
            0.278285048,
            0.297475087,
            0.286292061,
            0.257830950,
            0.206696101,
            0.144324772,
            0.085519078,
            0.038286146,
            0.001619797,
            -0.035117557,
            -0.072027247,
            -0.096533704,
            -0.103087741,
            -0.102368364,
            -0.114391855,
            -0.139967122,
            -0.171162687,
            -0.196658516,
            -0.207454048,
            -0.204801733,
            -0.203879004,
            -0.213328289,
            -0.228958920,
            -0.241698598,
            -0.251464790,
        ],
        [
            -0.020337631,
            -0.053622913,
            -0.122405706,
            -0.179462494,
            -0.200564443,
            -0.232332240,
            -0.283247005,
            -0.304278573,
            -0.276059936,
            -0.220589316,
            -0.139053387,
            -0.046042855,
            0.031991873,
            0.103372660,
            0.170370407,
            0.214905881,
            0.240971387,
            0.254238542,
            0.255631209,
            0.238008218,
            0.209534818,
            0.177808852,
            0.153095310,
            0.139956117,
            0.131477876,
            0.116280805,
            0.095933679,
            0.075698905,
            0.065977800,
            0.069857488,
            0.082988360,
            0.090329558,
            0.082929925,
            0.067734041,
            0.053242860,
            0.041115251,
        ],
    ],
    [
        [
            -0.02257427717386271300,
            -0.04580454714541371000,
            -0.08954033658115419200,
            -0.12005530771055017000,
            -0.12563459701293636000,
            -0.12595333439645837000,
            -0.12670444234219508000,
            -0.12715705435395483000,
            -0.12913057707506276000,
            -0.13354123002336557000,
            -0.13855666514343304000,
            -0.14308799991330715000,
            -0.14951791383958515000,
            -0.15845145652482151000,
            -0.16495787527646999000,
            -0.16781427384520564000,
            -0.17170232277018521000,
            -0.18042142403466258000,
            -0.18962359564086589000,
            -0.19407674590552579000,
            -0.19587634795792669000,
            -0.19670397522552724000,
            -0.19619975275648865000,
            -0.19518165653001343000,
            -0.19403913994258670000,
            -0.19368980706846228000,
            -0.19277035941743720000,
            -0.19234347800755142000,
            -0.19207998789953343000,
            -0.19183499892170250000,
            -0.19191989052413799000,
            -0.19216208684132857000,
            -0.19273004434759020000,
            -0.19273134081681903000,
            -0.19197345910434815000,
            -0.19195334021916627000,
        ],
        [
            -0.02190524719876366800,
            -0.04789585619670318200,
            -0.10618234949979000000,
            -0.15900405113964777000,
            -0.17702417653425023000,
            -0.18536131560391853000,
            -0.19315079255300505000,
            -0.20224319049559464000,
            -0.21765026428606477000,
            -0.24359160942259997000,
            -0.27299579285349473000,
            -0.29049279880094941000,
            -0.28346659709985672000,
            -0.24283365807615256000,
            -0.19188028232791318000,
            -0.15475085829598653000,
            -0.12084943547312378000,
            -0.07129023972829869000,
            -0.01545410159994778000,
            0.02855797827859832300,
            0.06332656026225293200,
            0.09556345974296330500,
            0.12304807393347031000,
            0.14150866887002808000,
            0.15188057104091590000,
            0.15797342142055862000,
            0.16213144452203984000,
            0.16563351215078684000,
            0.16485760626112306000,
            0.15991188547906210000,
            0.15381118424011236000,
            0.14657408643195061000,
            0.14252150862995686000,
            0.14149924680385922000,
            0.14426259164189553000,
            0.14568116947965443000,
        ],
        [
            0.03862192593327843500,
            0.08607977892966384200,
            0.18758937472970783000,
            0.26380287344461750000,
            0.27264656532342524000,
            0.26415033785924025000,
            0.25431550822206783000,
            0.24061274238989599000,
            0.21646081169191303000,
            0.16375249929345426000,
            0.06540594390641826800,
            -0.05748957147471461100,
            -0.16090875920779929000,
            -0.22967562960590965000,
            -0.28665433153182635000,
            -0.32394421018597008000,
            -0.31885416680273615000,
            -0.26443039994661971000,
            -0.19096541067598802000,
            -0.13794182744763242000,
            -0.09283207180313692000,
            -0.04863827809054062400,
            -0.00705701194648979300,
            0.02590864491028844700,
            0.04516812589249209800,
            0.05543340685273387400,
            0.06264866705777927300,
            0.06999797364036092200,
            0.07285747008320048200,
            0.06852130459855278200,
            0.05932205006228336000,
            0.04734424846797920200,
            0.03456970594041587600,
            0.02854412320123144900,
            0.03157197229148767800,
            0.03359122371504807300,
        ],
    ],
    [
        [
            -0.02709280096187565900,
            -0.05686223110461408000,
            -0.10035667073983964000,
            -0.11655686561575981000,
            -0.11422957067657943000,
            -0.11025418465506862000,
            -0.10767542796968975000,
            -0.10388290237276118000,
            -0.10158225848918870000,
            -0.10042808265232403000,
            -0.09444391157646114800,
            -0.08539718671110264900,
            -0.08140351716465142600,
            -0.07899836366862676700,
            -0.07092361786822773400,
            -0.06431797115426903000,
            -0.06551130033522271900,
            -0.07449341517075493100,
            -0.08108361889808800300,
            -0.09089442129206014300,
            -0.12404701968906916000,
            -0.16786332728947920000,
            -0.20250158743930527000,
            -0.22377717318551535000,
            -0.23406228761612988000,
            -0.24023286395208010000,
            -0.24230652318518972000,
            -0.24409552662937148000,
            -0.24451095295280248000,
            -0.24396897026804176000,
            -0.24334193238409188000,
            -0.24303731113979424000,
            -0.24384555693930871000,
            -0.24439637959579152000,
            -0.24397837395728775000,
            -0.24435930927502705000,
        ],
        [
            0.02001148601020115100,
            0.04899841658049408600,
            0.09110231699495907600,
            0.10753208990080826000,
            0.09847067341821445800,
            0.09702556509879775600,
            0.10316156120973860000,
            0.10877098463664207000,
            0.11985177404764036000,
            0.13576834462543336000,
            0.14149065311281014000,
            0.13685946675439886000,
            0.13973613295200196000,
            0.14909977184067982000,
            0.14654199509091873000,
            0.14130994015312165000,
            0.14757252813883043000,
            0.18077817300610374000,
            0.21009387986393199000,
            0.23369907594299305000,
            0.29227081282780004000,
            0.31933306627506691000,
            0.26565290061291041000,
            0.17228196686094716000,
            0.07639027259115091500,
            -0.00292291840580726410,
            -0.06142800318223973200,
            -0.10401144539762601000,
            -0.13589472821785523000,
            -0.15816114518317095000,
            -0.17335642699648807000,
            -0.18902786670833094000,
            -0.20610333417513085000,
            -0.22242224746425526000,
            -0.23664147979214531000,
            -0.24998288325425144000,
        ],
        [
            -0.01860488852357716800,
            -0.06851902564207997000,
            -0.21326717698986006000,
            -0.30102165576283019000,
            -0.31649926306090304000,
            -0.30060756567987956000,
            -0.27084117252238304000,
            -0.23425912778142560000,
            -0.20338041193211276000,
            -0.17461428657405265000,
            -0.14854465986513404000,
            -0.12781379545378718000,
            -0.11230428006257434000,
            -0.08797584482037290600,
            -0.05537699619205544600,
            -0.03089875066975936200,
            -0.02743502714477403700,
            -0.01578802100640397700,
            -0.00205272748385193100,
            0.01009006831808588100,
            0.08098296928681815400,
            0.21731539216862897000,
            0.31846566331995801000,
            0.32348510302323596000,
            0.26108337233732459000,
            0.18705889148548510000,
            0.11973584972940020000,
            0.06706555745724066000,
            0.03124326526269944500,
            0.00792265230294038160,
            -0.00379962533995196120,
            -0.01980851148987728500,
            -0.03821643563126132800,
            -0.06166276636610668700,
            -0.08003315946555962500,
            -0.09697997451524445500,
        ],
    ],
    [
        [
            0.02676824304188611300,
            0.05946407475965782100,
            0.12900702308575138000,
            0.17092577830538522000,
            0.17433273953929146000,
            0.16835540326878368000,
            0.15826424805464265000,
            0.14647224313786575000,
            0.13460263040798950000,
            0.12159264960543423000,
            0.10837126623117754000,
            0.09545938036189148500,
            0.08454657372677823500,
            0.07503059715637522300,
            0.06496740345758979200,
            0.05638854309961657100,
            0.05408497845708886600,
            0.05543735257043472300,
            0.05555858715239536000,
            0.05759583879709017500,
            0.06345289056498749900,
            0.07321840953073149900,
            0.09394017460665737800,
            0.12959719899113231000,
            0.17348415950453580000,
            0.21285437318563405000,
            0.23979589047651886000,
            0.25571215090479932000,
            0.26147369410179094000,
            0.26060735062830592000,
            0.25695755826517525000,
            0.25495084519503508000,
            0.25508140335236734000,
            0.25382248374599431000,
            0.25315232973604851000,
            0.25321847259615987000,
        ],
        [
            -0.01873744847940216800,
            -0.05990899326442069400,
            -0.13587937041765039000,
            -0.19250932779711649000,
            -0.21685711379342484000,
            -0.23833290753023634000,
            -0.25940907727956292000,
            -0.27129217353501439000,
            -0.27152046701232635000,
            -0.25903073272885363000,
            -0.23448070616812938000,
            -0.19749538078381751000,
            -0.16551798851884006000,
            -0.14072688005214101000,
            -0.11914770422601269000,
            -0.11063268885678886000,
            -0.10948247432054088000,
            -0.10943336389685750000,
            -0.10553405777913306000,
            -0.10897919714631589000,
            -0.11556837510924602000,
            -0.12678816534296078000,
            -0.12909988927091548000,
            -0.09285888955799707900,
            -0.00949866580609674310,
            0.06301250034760524900,
            0.10646985745101370000,
            0.12256333095253447000,
            0.13523848767144181000,
            0.14386501113297953000,
            0.15441065150742014000,
            0.16682910934595863000,
            0.17982703518357265000,
            0.19629872698899911000,
            0.20993169814916909000,
            0.22283191255022117000,
        ],
        [
            0.00687799096172833900,
            0.00632982236190463440,
            -0.01630024725336848600,
            -0.05415778649574832400,
            -0.07289153491194883000,
            -0.08545489000605219200,
            -0.09046789436531833800,
            -0.08072015476785890400,
            -0.07025393562283961700,
            -0.05667429771892082400,
            -0.04043014967453874300,
            -0.01876275509068558700,
            -0.00248907267489111750,
            0.00499985179874099400,
            0.00719666613671487470,
            0.00141123759501328960,
            0.00138660492172908220,
            0.00412730855466753730,
            0.00492544274519941470,
            0.00863927652266950270,
            0.01768264714224524300,
            0.02700767827843724200,
            0.05358757306348460600,
            0.12515958723848053000,
            0.24428784772611917000,
            0.34242457928205289000,
            0.38094858322069408000,
            0.35070697286866737000,
            0.25959544573108218000,
            0.13920978087352909000,
            0.00655712404118758820,
            -0.11671906463074669000,
            -0.21483468555348154000,
            -0.28912185847014771000,
            -0.34279486721072366000,
            -0.38307271805096821000,
        ],
    ],
    [
        [
            -0.02119674836738755600,
            -0.04591441602701418200,
            -0.10481010401668686000,
            -0.16258395638429893000,
            -0.17889545337044299000,
            -0.18072228226884079000,
            -0.18112664009792892000,
            -0.18032434720631510000,
            -0.17971618615047016000,
            -0.17871153535087211000,
            -0.17563285652502023000,
            -0.17119808391063290000,
            -0.16853689988040146000,
            -0.16561345617512713000,
            -0.16019419889858222000,
            -0.15573208563516067000,
            -0.15491795255226662000,
            -0.15656042280911464000,
            -0.15661531826131711000,
            -0.15764891623922786000,
            -0.16300024791746612000,
            -0.16902407524171240000,
            -0.17285036097124651000,
            -0.17473874057599853000,
            -0.17535320823704051000,
            -0.17679028377314515000,
            -0.17819373966686389000,
            -0.18018711283913955000,
            -0.18137104389357972000,
            -0.18095677225518114000,
            -0.18017559421686788000,
            -0.17972235196479699000,
            -0.17972279615663084000,
            -0.17977929580104804000,
            -0.17996014494612850000,
            -0.18116676892356900000,
        ],
        [
            0.00361061838890439770,
            0.00403832244766750420,
            0.00890962969648889660,
            0.03524815707980788500,
            0.05694248722530626000,
            0.07284328219269969900,
            0.08929685703045094100,
            0.10884229731927056000,
            0.12746765305849969000,
            0.14422870726029338000,
            0.16634052273914524000,
            0.18968300551257519000,
            0.20235756903744331000,
            0.21187377992143250000,
            0.23160746600613533000,
            0.24467573264157011000,
            0.23387224613948968000,
            0.20772756148326568000,
            0.18558478756428221000,
            0.15537257935462540000,
            0.08870855418533021800,
            0.01177297349282160400,
            -0.05488997670980050300,
            -0.10700795151145540000,
            -0.14115288511720003000,
            -0.16117542399322698000,
            -0.17218230632555048000,
            -0.18073298721042264000,
            -0.18906539182769619000,
            -0.20004311114038048000,
            -0.21240795927734304000,
            -0.22370131442259125000,
            -0.22996533792354246000,
            -0.23257699977177257000,
            -0.23162059375757346000,
            -0.22709792076398444000,
        ],
        [
            0.02648512192955599500,
            0.06258688757724231700,
            0.14234632723786383000,
            0.21547368046916726000,
            0.24347885216491971000,
            0.24958734564816068000,
            0.24081448101006372000,
            0.22418376109960433000,
            0.19465144188536176000,
            0.15445250087507756000,
            0.11514475901976122000,
            0.07351250327715834500,
            0.02535169523252229400,
            -0.02399041639806054300,
            -0.05483824103107217500,
            -0.06922149547192454900,
            -0.09331536945909434100,
            -0.14508948941010041000,
            -0.19321726430075248000,
            -0.22742341157660259000,
            -0.27490607958077012000,
            -0.29854536089141326000,
            -0.28507620653536159000,
            -0.25817167239338712000,
            -0.23466391476319501000,
            -0.19576301983826125000,
            -0.13733990379390060000,
            -0.07219380245437045900,
            -0.01490762225910524300,
            0.02545250361032095800,
            0.04811592885693794000,
            0.06542845020422376200,
            0.08455968994091871100,
            0.10445141018651627000,
            0.12195299963801620000,
            0.14191426767047791000,
        ],
    ],
    [
        [
            0.02949905076348713400,
            0.05380114492830617700,
            0.08079555140864556900,
            0.08673461177222968600,
            0.08228344477860428900,
            0.07805333953848162000,
            0.07658066145809679100,
            0.07426332465408649900,
            0.07368750818425572100,
            0.07497674474470043800,
            0.07186747732133795500,
            0.06522606040102924000,
            0.06403448204208103700,
            0.06747959315712472000,
            0.06420025096863182800,
            0.05857099606594507600,
            0.06154483368131926500,
            0.08432979913189758700,
            0.12590749478955982000,
            0.16082638200698035000,
            0.19171530712010598000,
            0.21543531303958838000,
            0.22759452497177649000,
            0.23276904041357468000,
            0.23427367585435840000,
            0.23511801139892841000,
            0.23630540140724776000,
            0.23765549341378889000,
            0.23815849278526871000,
            0.23735655259306632000,
            0.23776190401926256000,
            0.23901602361789720000,
            0.23966837305669569000,
            0.23992004299222611000,
            0.23963837239716360000,
            0.24055974593482360000,
        ],
        [
            -0.05749317690390509700,
            -0.10889049665477113000,
            -0.16808106208696935000,
            -0.18188449289155134000,
            -0.17213928808785378000,
            -0.16248122054252026000,
            -0.15940007865141995000,
            -0.15446135954382603000,
            -0.15394359073226441000,
            -0.15955177134499235000,
            -0.15476803378874412000,
            -0.14270086430739456000,
            -0.14150192246537266000,
            -0.15299122361377698000,
            -0.14659024215186020000,
            -0.13280364521210400000,
            -0.13654216732616947000,
            -0.18711606479886939000,
            -0.27646809095728458000,
            -0.33376341079739913000,
            -0.31149240794382088000,
            -0.21605192415199054000,
            -0.08872261441759031500,
            0.01501716445322282900,
            0.07459564202244076100,
            0.10463473412925806000,
            0.12001248564282355000,
            0.12979112919629543000,
            0.13792378105731262000,
            0.14268404174686664000,
            0.14677817922155648000,
            0.15510473763112062000,
            0.16050632288055705000,
            0.16731616487345491000,
            0.17324998571439973000,
            0.17991527864215207000,
        ],
        [
            -0.08244436089607173900,
            -0.15336030301728437000,
            -0.24426622711321333000,
            -0.26384931585866828000,
            -0.24892857366446980000,
            -0.23077245427661355000,
            -0.22241952671928703000,
            -0.20354553247501470000,
            -0.19616122335466987000,
            -0.19079938068965510000,
            -0.17229835840027194000,
            -0.14208870523710324000,
            -0.12630253767591793000,
            -0.10363604991038586000,
            -0.07201968219532281500,
            -0.05312886843809411200,
            -0.04004075771120371000,
            -0.00053440568762490803,
            0.16020422722625777000,
            0.40718001627624578000,
            0.41639470243864790000,
            0.27850647135244561000,
            0.14470374762313470000,
            0.04281535650889119900,
            -0.01040587995462491900,
            -0.03113779622382366000,
            -0.03588005442805115200,
            -0.03833924733056591500,
            -0.03518356199038780200,
            -0.02313756381588575800,
            -0.02203870658163908600,
            -0.02967782008259734300,
            -0.02175113614544999000,
            -0.01613469018407163300,
            -0.01367555588545224900,
            -0.01158219640370229700,
        ],
    ],
    [
        [
            0.01870695442590027500,
            0.03569161579037078600,
            0.04809268054013512300,
            0.04888694654858050800,
            0.04578808196032359300,
            0.04344586049591366000,
            0.04325685570994483600,
            0.04222189771723057500,
            0.04055681087575285100,
            0.04028180207781234500,
            0.03760574100567435500,
            0.03341449865332147400,
            0.03052287976995381500,
            0.03007254225309254400,
            0.02810981770811881200,
            0.02523222177779721500,
            0.02538661338781571200,
            0.02866947214354276900,
            0.03807571631578088300,
            0.05772133246350924500,
            0.10170369006245189000,
            0.16294646278543309000,
            0.20902545675000525000,
            0.23926676595591634000,
            0.25599658226130739000,
            0.26216124571088945000,
            0.26294121758247357000,
            0.26471231240392479000,
            0.26430141917656441000,
            0.26240021240987121000,
            0.26487052223495711000,
            0.26522291338114623000,
            0.26660542213939931000,
            0.26541584484337871000,
            0.26390547737136760000,
            0.26275220730492782000,
        ],
        [
            -0.05925964858380286500,
            -0.09295695535933538900,
            -0.11130621819722603000,
            -0.10992697128373242000,
            -0.10115389795872447000,
            -0.09595515343251508100,
            -0.09677723424692862000,
            -0.09606038254348969700,
            -0.09736670241206574200,
            -0.09992202944918748800,
            -0.09704383093180786500,
            -0.09002079222538757200,
            -0.08673352046691186500,
            -0.08557770147265368600,
            -0.08064000246969554900,
            -0.07913645121578150000,
            -0.08271807685423399100,
            -0.09807361937813430200,
            -0.11457149326572891000,
            -0.14765147379188692000,
            -0.23874210320885786000,
            -0.36798022743083686000,
            -0.43048231090135342000,
            -0.35215701510596309000,
            -0.19641466614055952000,
            -0.05913838831484832100,
            0.02635518902441718500,
            0.07834688271614538600,
            0.11384481744795055000,
            0.13409102253165275000,
            0.14769925378529281000,
            0.16322353175604573000,
            0.18252922264663107000,
            0.20305737442069113000,
            0.22128357114965899000,
            0.23621196298998062000,
        ],
        [
            0.11512389978331515000,
            0.22413006962808238000,
            0.31954389282025103000,
            0.32586304644263525000,
            0.30397127558670756000,
            0.28352212433907487000,
            0.26492834051113151000,
            0.24455825069511528000,
            0.23050435252217924000,
            0.21847071243995733000,
            0.19137596848708158000,
            0.16019742328929820000,
            0.13917066569959388000,
            0.11837326624126067000,
            0.09003968113096123800,
            0.07053180362500022200,
            0.06633604481687069300,
            0.05751726174543667800,
            0.01088835472471969500,
            -0.03853843360012083900,
            -0.08254453556203576700,
            -0.10202561217467844000,
            -0.14265191289777529000,
            -0.20811311672905380000,
            -0.22597980923018762000,
            -0.16968023714155689000,
            -0.09188792560239997200,
            -0.03476358458153469600,
            0.00581952483262489300,
            0.02773258068377005800,
            0.01150936667521866600,
            0.01175763239248749100,
            0.03672792702540145900,
            0.07368275531823710600,
            0.10450314791523628000,
            0.13170821119526419000,
        ],
    ],
    [
        [
            -0.01391512397332992000,
            -0.02355616580370156800,
            -0.03217974689527575100,
            -0.03389713623044773900,
            -0.03301036281363833200,
            -0.03285354371423714800,
            -0.03377540548706994800,
            -0.03489680631234950300,
            -0.03685746938516745000,
            -0.04079745564524299400,
            -0.04592855526325046600,
            -0.05252409573372079900,
            -0.06485881641780162600,
            -0.08872150998392475100,
            -0.12062905249920253000,
            -0.14663172901574814000,
            -0.16374285973859237000,
            -0.17854545670524613000,
            -0.19285090669470276000,
            -0.20310119935107607000,
            -0.20944935602779791000,
            -0.21419896526525989000,
            -0.21710861272899887000,
            -0.21862667803330973000,
            -0.22006016766927428000,
            -0.22215077344601830000,
            -0.22323849303277751000,
            -0.22463343282950998000,
            -0.22589338966445646000,
            -0.22594179947730525000,
            -0.22663921592189393000,
            -0.22783675444090351000,
            -0.22865193500586228000,
            -0.22945714446182908000,
            -0.22963380305061912000,
            -0.23154386034935404000,
        ],
        [
            -0.02372876400796839600,
            -0.03805126826063243200,
            -0.04915951914715022100,
            -0.04774626206533361100,
            -0.04297236442671773000,
            -0.03848115601428303600,
            -0.03397642565799539900,
            -0.02760803110730725800,
            -0.02042716476955505300,
            -0.00852478908021793260,
            0.01408937245236512900,
            0.04675928332177815100,
            0.09005268401626519100,
            0.15996855175372468000,
            0.27983350884234798000,
            0.40454760673506523000,
            0.46557425740279146000,
            0.42988255821256582000,
            0.30040479908227047000,
            0.16016147242639570000,
            0.06193514273781868000,
            -0.00634158525039273980,
            -0.05591728808815835100,
            -0.08801614369298063600,
            -0.10645480712256676000,
            -0.11439528123368227000,
            -0.12148762118556097000,
            -0.12853160184622442000,
            -0.13211749271913961000,
            -0.13124291058777687000,
            -0.12758412629342486000,
            -0.12246334197729329000,
            -0.11696553437732297000,
            -0.11538122270705850000,
            -0.11848205894639263000,
            -0.12091688597538751000,
        ],
        [
            0.13614121087578832000,
            0.22163172747838700000,
            0.29458621263180562000,
            0.30561224752530813000,
            0.29501884928368216000,
            0.28598447595430682000,
            0.28333441324709346000,
            0.27912705630178736000,
            0.27528060133650206000,
            0.27037172224265682000,
            0.25595129646913195000,
            0.23112118692222730000,
            0.20491022536680245000,
            0.18294409783499188000,
            0.09167575728467461400,
            -0.03963638360275884000,
            -0.11054285410554264000,
            -0.07373458712121995500,
            0.02683953896149155600,
            0.06870196457024961600,
            0.05013136970896505500,
            0.02356262876335300100,
            0.00258884870471166790,
            -0.00534038674246432950,
            -0.01181644034067877400,
            -0.01890402997198788200,
            -0.02401552609276516200,
            -0.03039072884274394300,
            -0.03697543824746714300,
            -0.04372079068719348500,
            -0.05526250446342715500,
            -0.06518385184466982100,
            -0.07978737224572156300,
            -0.09035377751454824700,
            -0.09485714223592185700,
            -0.09799962802996180200,
        ],
    ],
])
"""
Basis functions for *Otsu et al. (2018)*. This is a list of eight arrays,
with one for entry for each cluster. Each array contains three basis functions,
quantised in accordance with :attr:`colour.recovery.SPECTRAL_SHAPE_OTSU2018`
attribute.

References
----------
:cite:`Otsu2018`

BASIS_FUNCTIONS_OTSU2018 : ndarray
"""

CLUSTER_MEANS_OTSU2018 = np.array([
    [
        0.10085069182389943000,
        0.14557836477987415000,
        0.21618955974842774000,
        0.26241761006289305000,
        0.27539660377358477000,
        0.28531383647798736000,
        0.29863773584905656000,
        0.30855169811320765000,
        0.31710716981132059000,
        0.32332276729559734000,
        0.32056880503144641000,
        0.30730465408805036000,
        0.29085635220125783000,
        0.26974641509433944000,
        0.24537761006289302000,
        0.22229106918238989000,
        0.20037320754716983000,
        0.17840641509433955000,
        0.15444679245283027000,
        0.14225157232704402000,
        0.13367911949685529000,
        0.12840981132075477000,
        0.12297345911949682000,
        0.11905270440251571000,
        0.11778465408805038000,
        0.11974377358490564000,
        0.12402981132075469000,
        0.12976943396226415000,
        0.13483974842767291000,
        0.13579132075471703000,
        0.13499685534591194000,
        0.13219408805031446000,
        0.13386704402515726000,
        0.13686591194968545000,
        0.14213257861635226000,
        0.15195597484276738000,
    ],
    [
        0.09973996241240880300,
        0.13832746329562026000,
        0.19263781475547445000,
        0.22097216947810219000,
        0.22491500950364970000,
        0.22584267932481750000,
        0.22858438229562053000,
        0.23144331213138686000,
        0.23944196714598540000,
        0.25520659802554735000,
        0.27474755001824847000,
        0.29176247448175174000,
        0.30942757119707986000,
        0.32833185188321135000,
        0.34152471104014598000,
        0.34492205800000048000,
        0.34524719568978096000,
        0.35006971621532840000,
        0.35226373910948888000,
        0.35257366194525530000,
        0.34635072814963563000,
        0.33788450155109495000,
        0.32730242764963524000,
        0.31817969820072978000,
        0.31196273630291999000,
        0.30864278988686139000,
        0.30511056795255503000,
        0.30259412779197087000,
        0.30141088878467137000,
        0.30166522981386845000,
        0.30447054271897822000,
        0.30566773899999983000,
        0.30979884475912473000,
        0.31087716667883208000,
        0.30918824107664239000,
        0.30983656317883246000,
    ],
    [
        0.10163148148148149000,
        0.13874148148148155000,
        0.18048629629629634000,
        0.19058296296296293000,
        0.18498296296296293000,
        0.17870703703703697000,
        0.17455444444444446000,
        0.16798407407407415000,
        0.16265481481481481000,
        0.15838185185185186000,
        0.14838851851851853000,
        0.13549851851851852000,
        0.12967518518518517000,
        0.12491777777777781000,
        0.11482962962962959000,
        0.10857518518518521000,
        0.11038555555555557000,
        0.11997444444444444000,
        0.12557666666666670000,
        0.13899777777777780000,
        0.18298407407407410000,
        0.24777518518518515000,
        0.30390333333333330000,
        0.34408296296296298000,
        0.37026074074074061000,
        0.38781962962962946000,
        0.39744703703703704000,
        0.40430592592592590000,
        0.40855074074074083000,
        0.41022444444444450000,
        0.41229259259259260000,
        0.41340555555555564000,
        0.41682222222222221000,
        0.41964370370370369000,
        0.42174666666666671000,
        0.42481148148148135000,
    ],
    [
        0.10518588235294116000,
        0.15928000000000003000,
        0.23761058823529413000,
        0.27525764705882355000,
        0.27820000000000000000,
        0.26746470588235294000,
        0.24924588235294115000,
        0.22593176470588236000,
        0.20151176470588236000,
        0.17872352941176470000,
        0.15638352941176470000,
        0.13379058823529413000,
        0.11768588235294120000,
        0.10634235294117647000,
        0.09573411764705883100,
        0.08831411764705882100,
        0.08577882352941176800,
        0.08672941176470587400,
        0.08687764705882353200,
        0.08794588235294117900,
        0.09118941176470588000,
        0.09866235294117646500,
        0.11569764705882354000,
        0.14323176470588234000,
        0.17372588235294120000,
        0.20689176470588236000,
        0.24427647058823532000,
        0.28684470588235295000,
        0.33063882352941182000,
        0.37028352941176473000,
        0.40439058823529417000,
        0.43227882352941183000,
        0.45577764705882357000,
        0.47360352941176476000,
        0.48674117647058823000,
        0.49979882352941163000,
    ],
    [
        0.10779898591549292000,
        0.15870338028169012000,
        0.24335419718309842000,
        0.30154478873239432000,
        0.31312512676056359000,
        0.31187740845070405000,
        0.30963605633802815000,
        0.30475785915492964000,
        0.30064743661971804000,
        0.29648360563380305000,
        0.28867476056338026000,
        0.27758146478873252000,
        0.27091515492957735000,
        0.26409166197183093000,
        0.25295690140845062000,
        0.24406754929577457000,
        0.24216214084507048000,
        0.24517898591549292000,
        0.24370647887323940000,
        0.24752726760563384000,
        0.25809814084507027000,
        0.27048856338028177000,
        0.27946467605633790000,
        0.28564805633802820000,
        0.28950569014084515000,
        0.29394118309859157000,
        0.29796377464788726000,
        0.30271183098591542000,
        0.30628552112676050000,
        0.30746732394366194000,
        0.30875188732394343000,
        0.30856918309859155000,
        0.31044997183098588000,
        0.31218912676056354000,
        0.31383464788732385000,
        0.31724743661971833000,
    ],
    [
        0.08891588235294117800,
        0.11170882352941176000,
        0.13155529411764705000,
        0.13345352941176472000,
        0.12773352941176472000,
        0.12348058823529412000,
        0.12199294117647061000,
        0.11907411764705882000,
        0.11792117647058822000,
        0.11871352941176466000,
        0.11512823529411763000,
        0.10845000000000000000,
        0.10843000000000001000,
        0.11113294117647059000,
        0.10788647058823528000,
        0.10542588235294116000,
        0.11212294117647059000,
        0.13771529411764705000,
        0.18127705882352937000,
        0.22836529411764706000,
        0.28692647058823539000,
        0.35075411764705883000,
        0.39709058823529408000,
        0.42251705882352941000,
        0.43389176470588225000,
        0.43937470588235289000,
        0.44075882352941176000,
        0.44195411764705883000,
        0.44239529411764700000,
        0.44206588235294109000,
        0.44273058823529410000,
        0.44242411764705880000,
        0.44468470588235293000,
        0.44592764705882354000,
        0.44652235294117637000,
        0.44930823529411767000,
    ],
    [
        0.08701444444444445000,
        0.10270333333333331000,
        0.11151777777777777000,
        0.10953777777777778000,
        0.10569999999999999000,
        0.10276222222222224000,
        0.10133111111111109000,
        0.09741555555555554700,
        0.09392333333333334500,
        0.09028111111111110400,
        0.08486222222222221400,
        0.07873666666666664900,
        0.07606111111111110700,
        0.07359000000000001700,
        0.07044000000000000300,
        0.06970444444444444400,
        0.07159777777777778800,
        0.07601444444444444000,
        0.07899666666666667300,
        0.09216222222222221500,
        0.12827333333333332000,
        0.20277777777777778000,
        0.30198333333333338000,
        0.40135111111111121000,
        0.48075777777777773000,
        0.53332888888888896000,
        0.56130111111111125000,
        0.57818000000000003000,
        0.58851111111111121000,
        0.59322111111111120000,
        0.59782555555555550000,
        0.60259555555555566000,
        0.61115000000000008000,
        0.61758111111111114000,
        0.62303444444444445000,
        0.62989222222222208000,
    ],
    [
        0.07023857142857142800,
        0.08111642857142858900,
        0.08923309523809523600,
        0.09028619047619047800,
        0.08905285714285708600,
        0.08834642857142854800,
        0.08952357142857139700,
        0.09056880952380949600,
        0.09293809523809525000,
        0.09693595238095237300,
        0.10173238095238095000,
        0.10764285714285717000,
        0.12443142857142857000,
        0.16565285714285716000,
        0.22232071428571437000,
        0.26490809523809533000,
        0.29603785714285719000,
        0.33891190476190480000,
        0.39374523809523820000,
        0.44306047619047628000,
        0.46590380952380966000,
        0.47824261904761900000,
        0.48273738095238095000,
        0.48290047619047627000,
        0.48206547619047607000,
        0.48232547619047622000,
        0.48180452380952365000,
        0.48200666666666664000,
        0.48249571428571431000,
        0.48179785714285728000,
        0.48414190476190488000,
        0.48353404761904767000,
        0.48836785714285708000,
        0.49024928571428561000,
        0.49035238095238093000,
        0.49339000000000011000,
    ],
])
"""
Cluster means for *Otsu et al. (2018)*. This is a list of eight arrays, with
one for entry for each cluster. Each array is the mean of all the spectral
distributions used to create the particular cluster, quantised in accordance
with :attr:`colour.recovery.SPECTRAL_SHAPE_OTSU2018` attribute.

References
----------
:cite:`Otsu2018`

CLUSTER_MEANS_OTSU2018 : ndarray
"""

SELECTOR_ARRAY_OTSU2018 = np.array([
    [1, 0.333444973048471, -3, -1],
    [0, 0.428556829741043, 1, -2],
    [1, 0.368343583792887, 5, 7],
    [0, 0.389059234962091, -5, -4],
    [0, 0.464102042665547, 2, 6],
    [0, 0.288243127874986, 0, -6],
    [1, 0.247072787814766, 3, 4],
])
"""
Array describing how to select the appropriate cluster for given *CIE xy*
chromaticity coordinates.

SELECTOR_ARRAY_OTSU2018 : ndarray
"""
