# -*- coding: utf-8 -*-
"""
NIST CQS Test Colour Samples Spectral Distributions
===================================================

Defines the *NIST CQS (Color Quality Scale)* test colour samples spectral
distributions.

The *NIST CQS* test colour samples data is in the form of a *dict* of
:class:`colour.SpectralDistribution` classes as follows::

    {'method': {
        'name': SpectralDistribution,
        ...,
        'name': SpectralDistribution}
    }

See Also
--------
`Colour Quality Scale Jupyter Notebook
<http://nbviewer.jupyter.org/github/colour-science/colour-notebooks/\
blob/master/notebooks/quality/cqs.ipynb>`_

References
----------
-   :cite:`Ohno2008a` : Ohno, Y., & Davis, W. (2008). NIST CQS simulation 7.4.
    Retrieved from https://drive.google.com/file/d/\
1PsuU6QjUJjCX6tQyCud6ul2Tbs8rYWW9/view?usp=sharing
-   :cite:`Ohno2013` : Ohno, Y., & Davis, W. (2013). NIST CQS simulation 9.0.
    Retrieved from https://www.researchgate.net/file.PostFileLoader.html?\
id=5541c498f15bc7cc2c8b4578&assetKey=AS%3A273582771376136%401442238623549
"""

from __future__ import division, unicode_literals

from colour.colorimetry import SpectralDistribution
from colour.utilities import CaseInsensitiveMapping

__author__ = 'Colour Developers'
__copyright__ = 'Copyright (C) 2013-2020 - Colour Developers'
__license__ = 'New BSD License - https://opensource.org/licenses/BSD-3-Clause'
__maintainer__ = 'Colour Developers'
__email__ = 'colour-developers@colour-science.org'
__status__ = 'Production'

__all__ = [
    'VS_INDEXES_TO_NAMES', 'VS_APPROXIMATE_MUNSELL_NOTATIONS_NISTCQS74',
    'VS_APPROXIMATE_MUNSELL_NOTATIONS_NISTCQS90', 'VS_SDS_DATA', 'VS_SDS'
]

VS_INDEXES_TO_NAMES = {
    1: 'VS1',
    2: 'VS2',
    3: 'VS3',
    4: 'VS4',
    5: 'VS5',
    6: 'VS6',
    7: 'VS7',
    8: 'VS8',
    9: 'VS9',
    10: 'VS10',
    11: 'VS11',
    12: 'VS12',
    13: 'VS13',
    14: 'VS14',
    15: 'VS15'
}
"""
*NIST CQS* test colour samples indexes to names mapping.

VS_INDEXES_TO_NAMES : dict
"""

VS_APPROXIMATE_MUNSELL_NOTATIONS_NISTCQS74 = CaseInsensitiveMapping({
    'VS1': '7.5P 4/10',
    'VS2': '10PB 4/10',
    'VS3': '5PB 4/2',
    'VS4': '7.5B 5/10',
    'VS5': '10B G6/8',
    'VS6': '2.5BG 6/10',
    'VS7': '2.5G 6/12',
    'VS8': '7.5GY 7/10',
    'VS9': '2.5GY 8/10',
    'VS10': '5Y 8.5/12',
    'VS11': '10YR 7/12',
    'VS12': '5YR 7/12',
    'VS13': '10R 6/12',
    'VS14': '5R 4/14',
    'VS15': '7.5RP 4/12'
})
"""
*NIST CQS 7.4* test colour samples *Munsell* colour approximations.

VS_APPROXIMATE_MUNSELL_NOTATIONS_NISTCQS74 : CaseInsensitiveMapping
"""

VS_APPROXIMATE_MUNSELL_NOTATIONS_NISTCQS90 = CaseInsensitiveMapping({
    'VS1': '5R 4/14',
    'VS2': '10R 6/12',
    'VS3': '7.5YR 7/12',
    'VS4': '5Y 8-12',
    'VS5': '2.5GY 8/10',
    'VS6': '7.5GY 7/10',
    'VS7': '2.5G 6/12',
    'VS8': '2.5BG 6/10',
    'VS9': '10BG 6/8',
    'VS10': '7.5B 5/10',
    'VS11': '2.5PB 4/10',
    'VS12': '7.5PB 4/12',
    'VS13': '5P 5/10',
    'VS14': '2.5RP 6/12',
    'VS15': '7.5RP 5/12'
})
"""
*NIST CQS 9.0* test colour samples *Munsell* colour approximations.

VS_APPROXIMATE_MUNSELL_NOTATIONS_NISTCQ90 : CaseInsensitiveMapping
"""

VS_SDS_DATA = {
    'NIST CQS 7.4': {
        'VS1': {
            380: 0.1086,
            385: 0.1380,
            390: 0.1729,
            395: 0.2167,
            400: 0.2539,
            405: 0.2785,
            410: 0.2853,
            415: 0.2883,
            420: 0.2860,
            425: 0.2761,
            430: 0.2674,
            435: 0.2565,
            440: 0.2422,
            445: 0.2281,
            450: 0.2140,
            455: 0.2004,
            460: 0.1854,
            465: 0.1733,
            470: 0.1602,
            475: 0.1499,
            480: 0.1414,
            485: 0.1288,
            490: 0.1204,
            495: 0.1104,
            500: 0.1061,
            505: 0.1018,
            510: 0.0968,
            515: 0.0941,
            520: 0.0881,
            525: 0.0842,
            530: 0.0808,
            535: 0.0779,
            540: 0.0782,
            545: 0.0773,
            550: 0.0793,
            555: 0.0790,
            560: 0.0793,
            565: 0.0806,
            570: 0.0805,
            575: 0.0793,
            580: 0.0803,
            585: 0.0815,
            590: 0.0842,
            595: 0.0912,
            600: 0.1035,
            605: 0.1212,
            610: 0.1455,
            615: 0.1785,
            620: 0.2107,
            625: 0.2460,
            630: 0.2791,
            635: 0.3074,
            640: 0.3330,
            645: 0.3542,
            650: 0.3745,
            655: 0.3920,
            660: 0.4052,
            665: 0.4186,
            670: 0.4281,
            675: 0.4395,
            680: 0.4440,
            685: 0.4497,
            690: 0.4555,
            695: 0.4612,
            700: 0.4663,
            705: 0.4707,
            710: 0.4783,
            715: 0.4778,
            720: 0.4844,
            725: 0.4877,
            730: 0.4928,
            735: 0.4960,
            740: 0.4976,
            745: 0.4993,
            750: 0.5015,
            755: 0.5044,
            760: 0.5042,
            765: 0.5073,
            770: 0.5112,
            775: 0.5147,
            780: 0.5128,
            785: 0.5108,
            790: 0.5171,
            795: 0.5135,
            800: 0.5191,
            805: 0.5191,
            810: 0.5191,
            815: 0.5191,
            820: 0.5191,
            825: 0.5191,
            830: 0.5191
        },
        'VS2': {
            380: 0.1053,
            385: 0.1323,
            390: 0.1662,
            395: 0.2113,
            400: 0.2516,
            405: 0.2806,
            410: 0.2971,
            415: 0.3042,
            420: 0.3125,
            425: 0.3183,
            430: 0.3196,
            435: 0.3261,
            440: 0.3253,
            445: 0.3193,
            450: 0.3071,
            455: 0.2961,
            460: 0.2873,
            465: 0.2729,
            470: 0.2595,
            475: 0.2395,
            480: 0.2194,
            485: 0.1949,
            490: 0.1732,
            495: 0.1560,
            500: 0.1436,
            505: 0.1305,
            510: 0.1174,
            515: 0.1075,
            520: 0.0991,
            525: 0.0925,
            530: 0.0916,
            535: 0.0896,
            540: 0.0897,
            545: 0.0893,
            550: 0.0891,
            555: 0.0868,
            560: 0.0820,
            565: 0.0829,
            570: 0.0854,
            575: 0.0871,
            580: 0.0922,
            585: 0.0978,
            590: 0.1037,
            595: 0.1079,
            600: 0.1092,
            605: 0.1088,
            610: 0.1078,
            615: 0.1026,
            620: 0.0991,
            625: 0.0995,
            630: 0.1043,
            635: 0.1101,
            640: 0.1187,
            645: 0.1311,
            650: 0.1430,
            655: 0.1583,
            660: 0.1704,
            665: 0.1846,
            670: 0.1906,
            675: 0.1983,
            680: 0.1981,
            685: 0.1963,
            690: 0.2003,
            695: 0.2034,
            700: 0.2061,
            705: 0.2120,
            710: 0.2207,
            715: 0.2257,
            720: 0.2335,
            725: 0.2441,
            730: 0.2550,
            735: 0.2684,
            740: 0.2862,
            745: 0.3086,
            750: 0.3262,
            755: 0.3483,
            760: 0.3665,
            765: 0.3814,
            770: 0.3974,
            775: 0.4091,
            780: 0.4206,
            785: 0.4230,
            790: 0.4397,
            795: 0.4456,
            800: 0.4537,
            805: 0.4537,
            810: 0.4537,
            815: 0.4537,
            820: 0.4537,
            825: 0.4537,
            830: 0.4537
        },
        'VS3': {
            380: 0.0858,
            385: 0.0990,
            390: 0.1204,
            395: 0.1458,
            400: 0.1696,
            405: 0.1922,
            410: 0.2101,
            415: 0.2179,
            420: 0.2233,
            425: 0.2371,
            430: 0.2499,
            435: 0.2674,
            440: 0.2949,
            445: 0.3232,
            450: 0.3435,
            455: 0.3538,
            460: 0.3602,
            465: 0.3571,
            470: 0.3511,
            475: 0.3365,
            480: 0.3176,
            485: 0.2956,
            490: 0.2747,
            495: 0.2506,
            500: 0.2279,
            505: 0.2055,
            510: 0.1847,
            515: 0.1592,
            520: 0.1438,
            525: 0.1244,
            530: 0.1105,
            535: 0.0959,
            540: 0.0871,
            545: 0.0790,
            550: 0.0703,
            555: 0.0652,
            560: 0.0555,
            565: 0.0579,
            570: 0.0562,
            575: 0.0548,
            580: 0.0517,
            585: 0.0544,
            590: 0.0519,
            595: 0.0520,
            600: 0.0541,
            605: 0.0537,
            610: 0.0545,
            615: 0.0560,
            620: 0.0560,
            625: 0.0561,
            630: 0.0578,
            635: 0.0586,
            640: 0.0573,
            645: 0.0602,
            650: 0.0604,
            655: 0.0606,
            660: 0.0606,
            665: 0.0595,
            670: 0.0609,
            675: 0.0605,
            680: 0.0602,
            685: 0.0580,
            690: 0.0587,
            695: 0.0573,
            700: 0.0606,
            705: 0.0613,
            710: 0.0618,
            715: 0.0652,
            720: 0.0647,
            725: 0.0684,
            730: 0.0718,
            735: 0.0731,
            740: 0.0791,
            745: 0.0828,
            750: 0.0896,
            755: 0.0980,
            760: 0.1063,
            765: 0.1137,
            770: 0.1238,
            775: 0.1381,
            780: 0.1505,
            785: 0.1685,
            790: 0.1862,
            795: 0.2078,
            800: 0.2338,
            805: 0.2338,
            810: 0.2338,
            815: 0.2338,
            820: 0.2338,
            825: 0.2338,
            830: 0.2338
        },
        'VS4': {
            380: 0.0790,
            385: 0.0984,
            390: 0.1242,
            395: 0.1595,
            400: 0.1937,
            405: 0.2215,
            410: 0.2419,
            415: 0.2488,
            420: 0.2603,
            425: 0.2776,
            430: 0.2868,
            435: 0.3107,
            440: 0.3309,
            445: 0.3515,
            450: 0.3676,
            455: 0.3819,
            460: 0.4026,
            465: 0.4189,
            470: 0.4317,
            475: 0.4363,
            480: 0.4356,
            485: 0.4297,
            490: 0.4199,
            495: 0.4058,
            500: 0.3882,
            505: 0.3660,
            510: 0.3433,
            515: 0.3148,
            520: 0.2890,
            525: 0.2583,
            530: 0.2340,
            535: 0.2076,
            540: 0.1839,
            545: 0.1613,
            550: 0.1434,
            555: 0.1243,
            560: 0.1044,
            565: 0.0978,
            570: 0.0910,
            575: 0.0832,
            580: 0.0771,
            585: 0.0747,
            590: 0.0726,
            595: 0.0682,
            600: 0.0671,
            605: 0.0660,
            610: 0.0661,
            615: 0.0660,
            620: 0.0653,
            625: 0.0644,
            630: 0.0653,
            635: 0.0669,
            640: 0.0660,
            645: 0.0677,
            650: 0.0668,
            655: 0.0693,
            660: 0.0689,
            665: 0.0676,
            670: 0.0694,
            675: 0.0687,
            680: 0.0698,
            685: 0.0679,
            690: 0.0694,
            695: 0.0675,
            700: 0.0676,
            705: 0.0662,
            710: 0.0681,
            715: 0.0706,
            720: 0.0728,
            725: 0.0766,
            730: 0.0814,
            735: 0.0901,
            740: 0.1042,
            745: 0.1228,
            750: 0.1482,
            755: 0.1793,
            760: 0.2129,
            765: 0.2445,
            770: 0.2674,
            775: 0.2838,
            780: 0.2979,
            785: 0.3067,
            790: 0.3226,
            795: 0.3396,
            800: 0.3512,
            805: 0.3512,
            810: 0.3512,
            815: 0.3512,
            820: 0.3512,
            825: 0.3512,
            830: 0.3512
        },
        'VS5': {
            380: 0.1167,
            385: 0.1352,
            390: 0.1674,
            395: 0.2024,
            400: 0.2298,
            405: 0.2521,
            410: 0.2635,
            415: 0.2702,
            420: 0.2758,
            425: 0.2834,
            430: 0.2934,
            435: 0.3042,
            440: 0.3201,
            445: 0.3329,
            450: 0.3511,
            455: 0.3724,
            460: 0.4027,
            465: 0.4367,
            470: 0.4625,
            475: 0.4890,
            480: 0.5085,
            485: 0.5181,
            490: 0.5243,
            495: 0.5179,
            500: 0.5084,
            505: 0.4904,
            510: 0.4717,
            515: 0.4467,
            520: 0.4207,
            525: 0.3931,
            530: 0.3653,
            535: 0.3363,
            540: 0.3083,
            545: 0.2808,
            550: 0.2538,
            555: 0.2260,
            560: 0.2024,
            565: 0.1865,
            570: 0.1697,
            575: 0.1592,
            580: 0.1482,
            585: 0.1393,
            590: 0.1316,
            595: 0.1217,
            600: 0.1182,
            605: 0.1112,
            610: 0.1071,
            615: 0.1059,
            620: 0.1044,
            625: 0.1021,
            630: 0.0991,
            635: 0.1000,
            640: 0.0980,
            645: 0.0963,
            650: 0.0997,
            655: 0.0994,
            660: 0.1022,
            665: 0.1005,
            670: 0.1044,
            675: 0.1073,
            680: 0.1069,
            685: 0.1103,
            690: 0.1104,
            695: 0.1084,
            700: 0.1092,
            705: 0.1074,
            710: 0.1059,
            715: 0.1082,
            720: 0.1106,
            725: 0.1129,
            730: 0.1186,
            735: 0.1243,
            740: 0.1359,
            745: 0.1466,
            750: 0.1617,
            755: 0.1739,
            760: 0.1814,
            765: 0.1907,
            770: 0.1976,
            775: 0.1958,
            780: 0.1972,
            785: 0.2018,
            790: 0.2093,
            795: 0.2161,
            800: 0.2269,
            805: 0.2269,
            810: 0.2269,
            815: 0.2269,
            820: 0.2269,
            825: 0.2269,
            830: 0.2269
        },
        'VS6': {
            380: 0.0872,
            385: 0.1001,
            390: 0.1159,
            395: 0.1339,
            400: 0.1431,
            405: 0.1516,
            410: 0.1570,
            415: 0.1608,
            420: 0.1649,
            425: 0.1678,
            430: 0.1785,
            435: 0.1829,
            440: 0.1896,
            445: 0.2032,
            450: 0.2120,
            455: 0.2294,
            460: 0.2539,
            465: 0.2869,
            470: 0.3170,
            475: 0.3570,
            480: 0.3994,
            485: 0.4346,
            490: 0.4615,
            495: 0.4747,
            500: 0.4754,
            505: 0.4691,
            510: 0.4556,
            515: 0.4371,
            520: 0.4154,
            525: 0.3937,
            530: 0.3737,
            535: 0.3459,
            540: 0.3203,
            545: 0.2941,
            550: 0.2715,
            555: 0.2442,
            560: 0.2205,
            565: 0.1979,
            570: 0.1800,
            575: 0.1610,
            580: 0.1463,
            585: 0.1284,
            590: 0.1172,
            595: 0.1045,
            600: 0.0964,
            605: 0.0903,
            610: 0.0873,
            615: 0.0846,
            620: 0.0829,
            625: 0.0814,
            630: 0.0805,
            635: 0.0803,
            640: 0.0801,
            645: 0.0776,
            650: 0.0797,
            655: 0.0801,
            660: 0.0810,
            665: 0.0819,
            670: 0.0856,
            675: 0.0913,
            680: 0.0930,
            685: 0.0958,
            690: 0.1016,
            695: 0.1044,
            700: 0.1047,
            705: 0.1062,
            710: 0.1052,
            715: 0.1029,
            720: 0.1025,
            725: 0.1008,
            730: 0.1036,
            735: 0.1059,
            740: 0.1123,
            745: 0.1175,
            750: 0.1217,
            755: 0.1304,
            760: 0.1330,
            765: 0.1373,
            770: 0.1376,
            775: 0.1384,
            780: 0.1390,
            785: 0.1378,
            790: 0.1501,
            795: 0.1526,
            800: 0.1646,
            805: 0.1646,
            810: 0.1646,
            815: 0.1646,
            820: 0.1646,
            825: 0.1646,
            830: 0.1646
        },
        'VS7': {
            380: 0.0726,
            385: 0.0760,
            390: 0.0789,
            395: 0.0844,
            400: 0.0864,
            405: 0.0848,
            410: 0.0861,
            415: 0.0859,
            420: 0.0868,
            425: 0.0869,
            430: 0.0882,
            435: 0.0903,
            440: 0.0924,
            445: 0.0951,
            450: 0.0969,
            455: 0.1003,
            460: 0.1083,
            465: 0.1203,
            470: 0.1383,
            475: 0.1634,
            480: 0.1988,
            485: 0.2376,
            490: 0.2795,
            495: 0.3275,
            500: 0.3671,
            505: 0.4030,
            510: 0.4201,
            515: 0.4257,
            520: 0.4218,
            525: 0.4090,
            530: 0.3977,
            535: 0.3769,
            540: 0.3559,
            545: 0.3312,
            550: 0.3072,
            555: 0.2803,
            560: 0.2532,
            565: 0.2313,
            570: 0.2109,
            575: 0.1897,
            580: 0.1723,
            585: 0.1528,
            590: 0.1355,
            595: 0.1196,
            600: 0.1050,
            605: 0.0949,
            610: 0.0868,
            615: 0.0797,
            620: 0.0783,
            625: 0.0732,
            630: 0.0737,
            635: 0.0709,
            640: 0.0703,
            645: 0.0696,
            650: 0.0673,
            655: 0.0677,
            660: 0.0682,
            665: 0.0665,
            670: 0.0691,
            675: 0.0695,
            680: 0.0723,
            685: 0.0727,
            690: 0.0757,
            695: 0.0767,
            700: 0.0810,
            705: 0.0818,
            710: 0.0837,
            715: 0.0822,
            720: 0.0838,
            725: 0.0847,
            730: 0.0837,
            735: 0.0864,
            740: 0.0882,
            745: 0.0923,
            750: 0.0967,
            755: 0.0996,
            760: 0.1027,
            765: 0.1080,
            770: 0.1115,
            775: 0.1118,
            780: 0.1152,
            785: 0.1201,
            790: 0.1253,
            795: 0.1313,
            800: 0.1393,
            805: 0.1393,
            810: 0.1393,
            815: 0.1393,
            820: 0.1393,
            825: 0.1393,
            830: 0.1393
        },
        'VS8': {
            380: 0.0652,
            385: 0.0657,
            390: 0.0667,
            395: 0.0691,
            400: 0.0694,
            405: 0.0709,
            410: 0.0707,
            415: 0.0691,
            420: 0.0717,
            425: 0.0692,
            430: 0.0710,
            435: 0.0717,
            440: 0.0722,
            445: 0.0737,
            450: 0.0731,
            455: 0.0777,
            460: 0.0823,
            465: 0.0917,
            470: 0.1062,
            475: 0.1285,
            480: 0.1598,
            485: 0.1993,
            490: 0.2445,
            495: 0.2974,
            500: 0.3462,
            505: 0.3894,
            510: 0.4180,
            515: 0.4433,
            520: 0.4548,
            525: 0.4605,
            530: 0.4647,
            535: 0.4626,
            540: 0.4604,
            545: 0.4522,
            550: 0.4444,
            555: 0.4321,
            560: 0.4149,
            565: 0.4039,
            570: 0.3879,
            575: 0.3694,
            580: 0.3526,
            585: 0.3288,
            590: 0.3080,
            595: 0.2829,
            600: 0.2591,
            605: 0.2388,
            610: 0.2228,
            615: 0.2109,
            620: 0.2033,
            625: 0.1963,
            630: 0.1936,
            635: 0.1887,
            640: 0.1847,
            645: 0.1804,
            650: 0.1766,
            655: 0.1734,
            660: 0.1721,
            665: 0.1720,
            670: 0.1724,
            675: 0.1757,
            680: 0.1781,
            685: 0.1829,
            690: 0.1897,
            695: 0.1949,
            700: 0.2018,
            705: 0.2051,
            710: 0.2071,
            715: 0.2066,
            720: 0.2032,
            725: 0.1998,
            730: 0.2024,
            735: 0.2032,
            740: 0.2074,
            745: 0.2160,
            750: 0.2194,
            755: 0.2293,
            760: 0.2378,
            765: 0.2448,
            770: 0.2489,
            775: 0.2558,
            780: 0.2635,
            785: 0.2775,
            790: 0.2957,
            795: 0.3093,
            800: 0.3239,
            805: 0.3239,
            810: 0.3239,
            815: 0.3239,
            820: 0.3239,
            825: 0.3239,
            830: 0.3239
        },
        'VS9': {
            380: 0.0643,
            385: 0.0661,
            390: 0.0702,
            395: 0.0672,
            400: 0.0715,
            405: 0.0705,
            410: 0.0727,
            415: 0.0731,
            420: 0.0745,
            425: 0.0770,
            430: 0.0756,
            435: 0.0773,
            440: 0.0786,
            445: 0.0818,
            450: 0.0861,
            455: 0.0907,
            460: 0.0981,
            465: 0.1067,
            470: 0.1152,
            475: 0.1294,
            480: 0.1410,
            485: 0.1531,
            490: 0.1694,
            495: 0.1919,
            500: 0.2178,
            505: 0.2560,
            510: 0.3110,
            515: 0.3789,
            520: 0.4515,
            525: 0.5285,
            530: 0.5845,
            535: 0.6261,
            540: 0.6458,
            545: 0.6547,
            550: 0.6545,
            555: 0.6473,
            560: 0.6351,
            565: 0.6252,
            570: 0.6064,
            575: 0.5924,
            580: 0.5756,
            585: 0.5549,
            590: 0.5303,
            595: 0.5002,
            600: 0.4793,
            605: 0.4517,
            610: 0.4340,
            615: 0.4169,
            620: 0.4060,
            625: 0.3989,
            630: 0.3945,
            635: 0.3887,
            640: 0.3805,
            645: 0.3741,
            650: 0.3700,
            655: 0.3630,
            660: 0.3640,
            665: 0.3590,
            670: 0.3648,
            675: 0.3696,
            680: 0.3734,
            685: 0.3818,
            690: 0.3884,
            695: 0.3947,
            700: 0.4011,
            705: 0.4040,
            710: 0.4072,
            715: 0.4065,
            720: 0.4006,
            725: 0.3983,
            730: 0.3981,
            735: 0.3990,
            740: 0.4096,
            745: 0.4187,
            750: 0.4264,
            755: 0.4370,
            760: 0.4424,
            765: 0.4512,
            770: 0.4579,
            775: 0.4596,
            780: 0.4756,
            785: 0.4880,
            790: 0.5066,
            795: 0.5214,
            800: 0.5450,
            805: 0.5450,
            810: 0.5450,
            815: 0.5450,
            820: 0.5450,
            825: 0.5450,
            830: 0.5450
        },
        'VS10': {
            380: 0.0540,
            385: 0.0489,
            390: 0.0548,
            395: 0.0550,
            400: 0.0529,
            405: 0.0521,
            410: 0.0541,
            415: 0.0548,
            420: 0.0541,
            425: 0.0531,
            430: 0.0599,
            435: 0.0569,
            440: 0.0603,
            445: 0.0643,
            450: 0.0702,
            455: 0.0715,
            460: 0.0798,
            465: 0.0860,
            470: 0.0959,
            475: 0.1088,
            480: 0.1218,
            485: 0.1398,
            490: 0.1626,
            495: 0.1878,
            500: 0.2302,
            505: 0.2829,
            510: 0.3455,
            515: 0.4171,
            520: 0.4871,
            525: 0.5529,
            530: 0.5955,
            535: 0.6299,
            540: 0.6552,
            545: 0.6661,
            550: 0.6752,
            555: 0.6832,
            560: 0.6851,
            565: 0.6964,
            570: 0.6966,
            575: 0.7063,
            580: 0.7104,
            585: 0.7115,
            590: 0.7145,
            595: 0.7195,
            600: 0.7183,
            605: 0.7208,
            610: 0.7228,
            615: 0.7274,
            620: 0.7251,
            625: 0.7274,
            630: 0.7341,
            635: 0.7358,
            640: 0.7362,
            645: 0.7354,
            650: 0.7442,
            655: 0.7438,
            660: 0.7440,
            665: 0.7436,
            670: 0.7442,
            675: 0.7489,
            680: 0.7435,
            685: 0.7460,
            690: 0.7518,
            695: 0.7550,
            700: 0.7496,
            705: 0.7548,
            710: 0.7609,
            715: 0.7580,
            720: 0.7574,
            725: 0.7632,
            730: 0.7701,
            735: 0.7667,
            740: 0.7735,
            745: 0.7720,
            750: 0.7739,
            755: 0.7740,
            760: 0.7699,
            765: 0.7788,
            770: 0.7801,
            775: 0.7728,
            780: 0.7793,
            785: 0.7797,
            790: 0.7754,
            795: 0.7810,
            800: 0.7789,
            805: 0.7789,
            810: 0.7789,
            815: 0.7789,
            820: 0.7789,
            825: 0.7789,
            830: 0.7789
        },
        'VS11': {
            380: 0.0482,
            385: 0.0456,
            390: 0.0478,
            395: 0.0455,
            400: 0.0484,
            405: 0.0494,
            410: 0.0456,
            415: 0.0470,
            420: 0.0473,
            425: 0.0486,
            430: 0.0501,
            435: 0.0480,
            440: 0.0490,
            445: 0.0468,
            450: 0.0471,
            455: 0.0486,
            460: 0.0517,
            465: 0.0519,
            470: 0.0479,
            475: 0.0494,
            480: 0.0524,
            485: 0.0527,
            490: 0.0537,
            495: 0.0577,
            500: 0.0647,
            505: 0.0737,
            510: 0.0983,
            515: 0.1396,
            520: 0.1809,
            525: 0.2280,
            530: 0.2645,
            535: 0.2963,
            540: 0.3202,
            545: 0.3545,
            550: 0.3950,
            555: 0.4353,
            560: 0.4577,
            565: 0.4904,
            570: 0.5075,
            575: 0.5193,
            580: 0.5273,
            585: 0.5359,
            590: 0.5431,
            595: 0.5449,
            600: 0.5493,
            605: 0.5526,
            610: 0.5561,
            615: 0.5552,
            620: 0.5573,
            625: 0.5620,
            630: 0.5607,
            635: 0.5599,
            640: 0.5632,
            645: 0.5644,
            650: 0.5680,
            655: 0.5660,
            660: 0.5709,
            665: 0.5692,
            670: 0.5657,
            675: 0.5716,
            680: 0.5729,
            685: 0.5739,
            690: 0.5714,
            695: 0.5741,
            700: 0.5774,
            705: 0.5791,
            710: 0.5801,
            715: 0.5804,
            720: 0.5840,
            725: 0.5814,
            730: 0.5874,
            735: 0.5885,
            740: 0.5911,
            745: 0.5878,
            750: 0.5896,
            755: 0.5947,
            760: 0.5945,
            765: 0.5935,
            770: 0.5979,
            775: 0.5941,
            780: 0.5962,
            785: 0.5919,
            790: 0.5996,
            795: 0.5953,
            800: 0.5953,
            805: 0.5953,
            810: 0.5953,
            815: 0.5953,
            820: 0.5953,
            825: 0.5953,
            830: 0.5953
        },
        'VS12': {
            380: 0.0691,
            385: 0.0692,
            390: 0.0727,
            395: 0.0756,
            400: 0.0770,
            405: 0.0806,
            410: 0.0771,
            415: 0.0742,
            420: 0.0766,
            425: 0.0733,
            430: 0.0758,
            435: 0.0768,
            440: 0.0775,
            445: 0.0754,
            450: 0.0763,
            455: 0.0763,
            460: 0.0752,
            465: 0.0782,
            470: 0.0808,
            475: 0.0778,
            480: 0.0788,
            485: 0.0805,
            490: 0.0809,
            495: 0.0838,
            500: 0.0922,
            505: 0.1051,
            510: 0.1230,
            515: 0.1521,
            520: 0.1728,
            525: 0.1842,
            530: 0.1897,
            535: 0.1946,
            540: 0.2037,
            545: 0.2248,
            550: 0.2675,
            555: 0.3286,
            560: 0.3895,
            565: 0.4654,
            570: 0.5188,
            575: 0.5592,
            580: 0.5909,
            585: 0.6189,
            590: 0.6343,
            595: 0.6485,
            600: 0.6607,
            605: 0.6648,
            610: 0.6654,
            615: 0.6721,
            620: 0.6744,
            625: 0.6723,
            630: 0.6811,
            635: 0.6792,
            640: 0.6774,
            645: 0.6796,
            650: 0.6856,
            655: 0.6853,
            660: 0.6864,
            665: 0.6879,
            670: 0.6874,
            675: 0.6871,
            680: 0.6863,
            685: 0.6890,
            690: 0.6863,
            695: 0.6893,
            700: 0.6950,
            705: 0.6941,
            710: 0.6958,
            715: 0.6950,
            720: 0.7008,
            725: 0.7020,
            730: 0.7059,
            735: 0.7085,
            740: 0.7047,
            745: 0.7021,
            750: 0.7071,
            755: 0.7088,
            760: 0.7055,
            765: 0.7073,
            770: 0.7114,
            775: 0.7028,
            780: 0.7105,
            785: 0.7078,
            790: 0.7112,
            795: 0.7123,
            800: 0.7158,
            805: 0.7158,
            810: 0.7158,
            815: 0.7158,
            820: 0.7158,
            825: 0.7158,
            830: 0.7158
        },
        'VS13': {
            380: 0.0829,
            385: 0.0829,
            390: 0.0866,
            395: 0.0888,
            400: 0.0884,
            405: 0.0853,
            410: 0.0868,
            415: 0.0859,
            420: 0.0828,
            425: 0.0819,
            430: 0.0822,
            435: 0.0818,
            440: 0.0822,
            445: 0.0819,
            450: 0.0807,
            455: 0.0787,
            460: 0.0832,
            465: 0.0828,
            470: 0.0810,
            475: 0.0819,
            480: 0.0836,
            485: 0.0802,
            490: 0.0809,
            495: 0.0838,
            500: 0.0842,
            505: 0.0865,
            510: 0.0910,
            515: 0.0920,
            520: 0.0917,
            525: 0.0917,
            530: 0.0952,
            535: 0.0983,
            540: 0.1036,
            545: 0.1150,
            550: 0.1331,
            555: 0.1646,
            560: 0.2070,
            565: 0.2754,
            570: 0.3279,
            575: 0.3819,
            580: 0.4250,
            585: 0.4690,
            590: 0.5067,
            595: 0.5443,
            600: 0.5721,
            605: 0.5871,
            610: 0.6073,
            615: 0.6141,
            620: 0.6170,
            625: 0.6216,
            630: 0.6272,
            635: 0.6287,
            640: 0.6276,
            645: 0.6351,
            650: 0.6362,
            655: 0.6348,
            660: 0.6418,
            665: 0.6438,
            670: 0.6378,
            675: 0.6410,
            680: 0.6460,
            685: 0.6451,
            690: 0.6432,
            695: 0.6509,
            700: 0.6517,
            705: 0.6514,
            710: 0.6567,
            715: 0.6597,
            720: 0.6576,
            725: 0.6576,
            730: 0.6656,
            735: 0.6641,
            740: 0.6667,
            745: 0.6688,
            750: 0.6713,
            755: 0.6657,
            760: 0.6712,
            765: 0.6745,
            770: 0.6780,
            775: 0.6744,
            780: 0.6786,
            785: 0.6823,
            790: 0.6806,
            795: 0.6718,
            800: 0.6813,
            805: 0.6813,
            810: 0.6813,
            815: 0.6813,
            820: 0.6813,
            825: 0.6813,
            830: 0.6813
        },
        'VS14': {
            380: 0.0530,
            385: 0.0507,
            390: 0.0505,
            395: 0.0502,
            400: 0.0498,
            405: 0.0489,
            410: 0.0503,
            415: 0.0492,
            420: 0.0511,
            425: 0.0509,
            430: 0.0496,
            435: 0.0494,
            440: 0.0480,
            445: 0.0487,
            450: 0.0468,
            455: 0.0443,
            460: 0.0440,
            465: 0.0427,
            470: 0.0421,
            475: 0.0414,
            480: 0.0408,
            485: 0.0400,
            490: 0.0392,
            495: 0.0406,
            500: 0.0388,
            505: 0.0396,
            510: 0.0397,
            515: 0.0391,
            520: 0.0405,
            525: 0.0394,
            530: 0.0401,
            535: 0.0396,
            540: 0.0396,
            545: 0.0395,
            550: 0.0399,
            555: 0.0420,
            560: 0.0410,
            565: 0.0464,
            570: 0.0500,
            575: 0.0545,
            580: 0.0620,
            585: 0.0742,
            590: 0.0937,
            595: 0.1279,
            600: 0.1762,
            605: 0.2449,
            610: 0.3211,
            615: 0.4050,
            620: 0.4745,
            625: 0.5335,
            630: 0.5776,
            635: 0.6094,
            640: 0.6320,
            645: 0.6495,
            650: 0.6620,
            655: 0.6743,
            660: 0.6833,
            665: 0.6895,
            670: 0.6924,
            675: 0.7030,
            680: 0.7075,
            685: 0.7112,
            690: 0.7187,
            695: 0.7214,
            700: 0.7284,
            705: 0.7327,
            710: 0.7351,
            715: 0.7374,
            720: 0.7410,
            725: 0.7417,
            730: 0.7491,
            735: 0.7516,
            740: 0.7532,
            745: 0.7567,
            750: 0.7600,
            755: 0.7592,
            760: 0.7605,
            765: 0.7629,
            770: 0.7646,
            775: 0.7622,
            780: 0.7680,
            785: 0.7672,
            790: 0.7645,
            795: 0.7669,
            800: 0.7683,
            805: 0.7683,
            810: 0.7683,
            815: 0.7683,
            820: 0.7683,
            825: 0.7683,
            830: 0.7683
        },
        'VS15': {
            380: 0.0908,
            385: 0.1021,
            390: 0.1130,
            395: 0.1280,
            400: 0.1359,
            405: 0.1378,
            410: 0.1363,
            415: 0.1363,
            420: 0.1354,
            425: 0.1322,
            430: 0.1294,
            435: 0.1241,
            440: 0.1209,
            445: 0.1137,
            450: 0.1117,
            455: 0.1045,
            460: 0.1006,
            465: 0.0970,
            470: 0.0908,
            475: 0.0858,
            480: 0.0807,
            485: 0.0752,
            490: 0.0716,
            495: 0.0688,
            500: 0.0678,
            505: 0.0639,
            510: 0.0615,
            515: 0.0586,
            520: 0.0571,
            525: 0.0527,
            530: 0.0513,
            535: 0.0537,
            540: 0.0512,
            545: 0.0530,
            550: 0.0517,
            555: 0.0511,
            560: 0.0507,
            565: 0.0549,
            570: 0.0559,
            575: 0.0627,
            580: 0.0678,
            585: 0.0810,
            590: 0.1004,
            595: 0.1268,
            600: 0.1595,
            605: 0.2012,
            610: 0.2452,
            615: 0.2953,
            620: 0.3439,
            625: 0.3928,
            630: 0.4336,
            635: 0.4723,
            640: 0.4996,
            645: 0.5279,
            650: 0.5428,
            655: 0.5601,
            660: 0.5736,
            665: 0.5837,
            670: 0.5890,
            675: 0.5959,
            680: 0.5983,
            685: 0.6015,
            690: 0.6054,
            695: 0.6135,
            700: 0.6200,
            705: 0.6287,
            710: 0.6405,
            715: 0.6443,
            720: 0.6489,
            725: 0.6621,
            730: 0.6662,
            735: 0.6726,
            740: 0.6774,
            745: 0.6834,
            750: 0.6808,
            755: 0.6838,
            760: 0.6874,
            765: 0.6955,
            770: 0.7012,
            775: 0.6996,
            780: 0.7023,
            785: 0.7022,
            790: 0.7144,
            795: 0.7062,
            800: 0.7075,
            805: 0.7075,
            810: 0.7075,
            815: 0.7075,
            820: 0.7075,
            825: 0.7075,
            830: 0.7075
        },
    },
    'NIST CQS 9.0': {
        'VS1': {
            360: 0.0530,
            365: 0.0530,
            370: 0.0530,
            375: 0.0530,
            380: 0.0530,
            385: 0.0507,
            390: 0.0505,
            395: 0.0502,
            400: 0.0498,
            405: 0.0489,
            410: 0.0503,
            415: 0.0492,
            420: 0.0511,
            425: 0.0509,
            430: 0.0496,
            435: 0.0494,
            440: 0.0480,
            445: 0.0487,
            450: 0.0468,
            455: 0.0443,
            460: 0.0440,
            465: 0.0427,
            470: 0.0421,
            475: 0.0414,
            480: 0.0408,
            485: 0.0400,
            490: 0.0392,
            495: 0.0406,
            500: 0.0388,
            505: 0.0396,
            510: 0.0397,
            515: 0.0391,
            520: 0.0405,
            525: 0.0394,
            530: 0.0401,
            535: 0.0396,
            540: 0.0396,
            545: 0.0395,
            550: 0.0399,
            555: 0.0420,
            560: 0.0410,
            565: 0.0464,
            570: 0.0500,
            575: 0.0545,
            580: 0.0620,
            585: 0.0742,
            590: 0.0937,
            595: 0.1279,
            600: 0.1762,
            605: 0.2449,
            610: 0.3211,
            615: 0.4050,
            620: 0.4745,
            625: 0.5335,
            630: 0.5776,
            635: 0.6094,
            640: 0.6320,
            645: 0.6495,
            650: 0.6620,
            655: 0.6743,
            660: 0.6833,
            665: 0.6895,
            670: 0.6924,
            675: 0.7030,
            680: 0.7075,
            685: 0.7112,
            690: 0.7187,
            695: 0.7214,
            700: 0.7284,
            705: 0.7327,
            710: 0.7351,
            715: 0.7374,
            720: 0.7410,
            725: 0.7417,
            730: 0.7491,
            735: 0.7516,
            740: 0.7532,
            745: 0.7567,
            750: 0.7600,
            755: 0.7592,
            760: 0.7605,
            765: 0.7629,
            770: 0.7646,
            775: 0.7622,
            780: 0.7680,
            785: 0.7672,
            790: 0.7645,
            795: 0.7669,
            800: 0.7683,
            805: 0.7683,
            810: 0.7683,
            815: 0.7683,
            820: 0.7683,
            825: 0.7683,
            830: 0.7683
        },
        'VS2': {
            360: 0.0829,
            365: 0.0829,
            370: 0.0829,
            375: 0.0829,
            380: 0.0829,
            385: 0.0829,
            390: 0.0866,
            395: 0.0888,
            400: 0.0884,
            405: 0.0853,
            410: 0.0868,
            415: 0.0859,
            420: 0.0828,
            425: 0.0819,
            430: 0.0822,
            435: 0.0818,
            440: 0.0822,
            445: 0.0819,
            450: 0.0807,
            455: 0.0787,
            460: 0.0832,
            465: 0.0828,
            470: 0.0810,
            475: 0.0819,
            480: 0.0836,
            485: 0.0802,
            490: 0.0809,
            495: 0.0838,
            500: 0.0842,
            505: 0.0865,
            510: 0.0910,
            515: 0.0920,
            520: 0.0917,
            525: 0.0917,
            530: 0.0952,
            535: 0.0983,
            540: 0.1036,
            545: 0.1150,
            550: 0.1331,
            555: 0.1646,
            560: 0.2070,
            565: 0.2754,
            570: 0.3279,
            575: 0.3819,
            580: 0.4250,
            585: 0.4690,
            590: 0.5067,
            595: 0.5443,
            600: 0.5721,
            605: 0.5871,
            610: 0.6073,
            615: 0.6141,
            620: 0.6170,
            625: 0.6216,
            630: 0.6272,
            635: 0.6287,
            640: 0.6276,
            645: 0.6351,
            650: 0.6362,
            655: 0.6348,
            660: 0.6418,
            665: 0.6438,
            670: 0.6378,
            675: 0.6410,
            680: 0.6460,
            685: 0.6451,
            690: 0.6432,
            695: 0.6509,
            700: 0.6517,
            705: 0.6514,
            710: 0.6567,
            715: 0.6597,
            720: 0.6576,
            725: 0.6576,
            730: 0.6656,
            735: 0.6641,
            740: 0.6667,
            745: 0.6688,
            750: 0.6713,
            755: 0.6657,
            760: 0.6712,
            765: 0.6745,
            770: 0.6780,
            775: 0.6744,
            780: 0.6786,
            785: 0.6823,
            790: 0.6806,
            795: 0.6718,
            800: 0.6813,
            805: 0.6813,
            810: 0.6813,
            815: 0.6813,
            820: 0.6813,
            825: 0.6813,
            830: 0.6813
        },
        'VS3': {
            360: 0.0691,
            365: 0.0691,
            370: 0.0691,
            375: 0.0691,
            380: 0.0582,
            385: 0.0572,
            390: 0.0559,
            395: 0.0568,
            400: 0.0599,
            405: 0.0556,
            410: 0.0589,
            415: 0.0598,
            420: 0.0581,
            425: 0.0562,
            430: 0.0569,
            435: 0.0583,
            440: 0.0559,
            445: 0.0556,
            450: 0.0586,
            455: 0.0567,
            460: 0.0581,
            465: 0.0583,
            470: 0.0611,
            475: 0.0599,
            480: 0.0604,
            485: 0.0621,
            490: 0.0601,
            495: 0.0620,
            500: 0.0691,
            505: 0.0827,
            510: 0.1035,
            515: 0.1348,
            520: 0.1713,
            525: 0.1944,
            530: 0.2115,
            535: 0.2250,
            540: 0.2445,
            545: 0.2783,
            550: 0.3280,
            555: 0.3909,
            560: 0.4359,
            565: 0.4928,
            570: 0.5270,
            575: 0.5480,
            580: 0.5575,
            585: 0.5700,
            590: 0.5784,
            595: 0.5818,
            600: 0.5837,
            605: 0.5896,
            610: 0.5947,
            615: 0.5909,
            620: 0.5953,
            625: 0.5943,
            630: 0.5965,
            635: 0.5993,
            640: 0.6011,
            645: 0.6056,
            650: 0.6009,
            655: 0.6023,
            660: 0.6064,
            665: 0.6055,
            670: 0.6053,
            675: 0.6112,
            680: 0.6125,
            685: 0.6064,
            690: 0.6107,
            695: 0.6137,
            700: 0.6155,
            705: 0.6130,
            710: 0.6197,
            715: 0.6215,
            720: 0.6162,
            725: 0.6223,
            730: 0.6262,
            735: 0.6277,
            740: 0.6268,
            745: 0.6256,
            750: 0.6303,
            755: 0.6288,
            760: 0.6288,
            765: 0.6315,
            770: 0.6320,
            775: 0.6303,
            780: 0.6350,
            785: 0.6360,
            790: 0.6363,
            795: 0.6302,
            800: 0.6283,
            805: 0.6283,
            810: 0.6283,
            815: 0.6283,
            820: 0.6283,
            825: 0.6283,
            830: 0.6283
        },
        'VS4': {
            360: 0.0449,
            365: 0.0449,
            370: 0.0449,
            375: 0.0449,
            380: 0.0449,
            385: 0.0431,
            390: 0.0366,
            395: 0.0368,
            400: 0.0448,
            405: 0.0423,
            410: 0.0440,
            415: 0.0438,
            420: 0.0442,
            425: 0.0438,
            430: 0.0441,
            435: 0.0495,
            440: 0.0512,
            445: 0.0532,
            450: 0.0528,
            455: 0.0580,
            460: 0.0632,
            465: 0.0705,
            470: 0.0815,
            475: 0.0924,
            480: 0.1068,
            485: 0.1241,
            490: 0.1475,
            495: 0.1798,
            500: 0.2192,
            505: 0.2712,
            510: 0.3270,
            515: 0.3778,
            520: 0.4197,
            525: 0.4525,
            530: 0.4775,
            535: 0.4915,
            540: 0.5042,
            545: 0.5153,
            550: 0.5215,
            555: 0.5335,
            560: 0.5362,
            565: 0.5551,
            570: 0.5630,
            575: 0.5747,
            580: 0.5815,
            585: 0.5885,
            590: 0.5939,
            595: 0.6015,
            600: 0.6001,
            605: 0.6007,
            610: 0.6093,
            615: 0.6109,
            620: 0.6118,
            625: 0.6132,
            630: 0.6201,
            635: 0.6236,
            640: 0.6196,
            645: 0.6275,
            650: 0.6279,
            655: 0.6264,
            660: 0.6300,
            665: 0.6341,
            670: 0.6353,
            675: 0.6339,
            680: 0.6373,
            685: 0.6368,
            690: 0.6409,
            695: 0.6381,
            700: 0.6433,
            705: 0.6461,
            710: 0.6445,
            715: 0.6507,
            720: 0.6523,
            725: 0.6611,
            730: 0.6568,
            735: 0.6618,
            740: 0.6666,
            745: 0.6617,
            750: 0.6620,
            755: 0.6623,
            760: 0.6662,
            765: 0.6628,
            770: 0.6694,
            775: 0.6757,
            780: 0.6721,
            785: 0.6679,
            790: 0.6769,
            795: 0.6681,
            800: 0.6784,
            805: 0.6784,
            810: 0.6784,
            815: 0.6784,
            820: 0.6784,
            825: 0.6784,
            830: 0.6784
        },
        'VS5': {
            360: 0.0643,
            365: 0.0643,
            370: 0.0643,
            375: 0.0643,
            380: 0.0643,
            385: 0.0661,
            390: 0.0702,
            395: 0.0672,
            400: 0.0715,
            405: 0.0705,
            410: 0.0727,
            415: 0.0731,
            420: 0.0745,
            425: 0.0770,
            430: 0.0756,
            435: 0.0773,
            440: 0.0786,
            445: 0.0818,
            450: 0.0861,
            455: 0.0907,
            460: 0.0981,
            465: 0.1067,
            470: 0.1152,
            475: 0.1294,
            480: 0.1410,
            485: 0.1531,
            490: 0.1694,
            495: 0.1919,
            500: 0.2178,
            505: 0.2560,
            510: 0.3110,
            515: 0.3789,
            520: 0.4515,
            525: 0.5285,
            530: 0.5845,
            535: 0.6261,
            540: 0.6458,
            545: 0.6547,
            550: 0.6545,
            555: 0.6473,
            560: 0.6351,
            565: 0.6252,
            570: 0.6064,
            575: 0.5924,
            580: 0.5756,
            585: 0.5549,
            590: 0.5303,
            595: 0.5002,
            600: 0.4793,
            605: 0.4517,
            610: 0.4340,
            615: 0.4169,
            620: 0.4060,
            625: 0.3989,
            630: 0.3945,
            635: 0.3887,
            640: 0.3805,
            645: 0.3741,
            650: 0.3700,
            655: 0.3630,
            660: 0.3640,
            665: 0.3590,
            670: 0.3648,
            675: 0.3696,
            680: 0.3734,
            685: 0.3818,
            690: 0.3884,
            695: 0.3947,
            700: 0.4011,
            705: 0.4040,
            710: 0.4072,
            715: 0.4065,
            720: 0.4006,
            725: 0.3983,
            730: 0.3981,
            735: 0.3990,
            740: 0.4096,
            745: 0.4187,
            750: 0.4264,
            755: 0.4370,
            760: 0.4424,
            765: 0.4512,
            770: 0.4579,
            775: 0.4596,
            780: 0.4756,
            785: 0.4880,
            790: 0.5066,
            795: 0.5214,
            800: 0.5450,
            805: 0.5450,
            810: 0.5450,
            815: 0.5450,
            820: 0.5450,
            825: 0.5450,
            830: 0.5450
        },
        'VS6': {
            360: 0.0652,
            365: 0.0652,
            370: 0.0652,
            375: 0.0652,
            380: 0.0652,
            385: 0.0657,
            390: 0.0667,
            395: 0.0691,
            400: 0.0694,
            405: 0.0709,
            410: 0.0707,
            415: 0.0691,
            420: 0.0717,
            425: 0.0692,
            430: 0.0710,
            435: 0.0717,
            440: 0.0722,
            445: 0.0737,
            450: 0.0731,
            455: 0.0777,
            460: 0.0823,
            465: 0.0917,
            470: 0.1062,
            475: 0.1285,
            480: 0.1598,
            485: 0.1993,
            490: 0.2445,
            495: 0.2974,
            500: 0.3462,
            505: 0.3894,
            510: 0.4180,
            515: 0.4433,
            520: 0.4548,
            525: 0.4605,
            530: 0.4647,
            535: 0.4626,
            540: 0.4604,
            545: 0.4522,
            550: 0.4444,
            555: 0.4321,
            560: 0.4149,
            565: 0.4039,
            570: 0.3879,
            575: 0.3694,
            580: 0.3526,
            585: 0.3288,
            590: 0.3080,
            595: 0.2829,
            600: 0.2591,
            605: 0.2388,
            610: 0.2228,
            615: 0.2109,
            620: 0.2033,
            625: 0.1963,
            630: 0.1936,
            635: 0.1887,
            640: 0.1847,
            645: 0.1804,
            650: 0.1766,
            655: 0.1734,
            660: 0.1721,
            665: 0.1720,
            670: 0.1724,
            675: 0.1757,
            680: 0.1781,
            685: 0.1829,
            690: 0.1897,
            695: 0.1949,
            700: 0.2018,
            705: 0.2051,
            710: 0.2071,
            715: 0.2066,
            720: 0.2032,
            725: 0.1998,
            730: 0.2024,
            735: 0.2032,
            740: 0.2074,
            745: 0.2160,
            750: 0.2194,
            755: 0.2293,
            760: 0.2378,
            765: 0.2448,
            770: 0.2489,
            775: 0.2558,
            780: 0.2635,
            785: 0.2775,
            790: 0.2957,
            795: 0.3093,
            800: 0.3239,
            805: 0.3239,
            810: 0.3239,
            815: 0.3239,
            820: 0.3239,
            825: 0.3239,
            830: 0.3239
        },
        'VS7': {
            360: 0.0726,
            365: 0.0726,
            370: 0.0726,
            375: 0.0726,
            380: 0.0726,
            385: 0.0760,
            390: 0.0789,
            395: 0.0844,
            400: 0.0864,
            405: 0.0848,
            410: 0.0861,
            415: 0.0859,
            420: 0.0868,
            425: 0.0869,
            430: 0.0882,
            435: 0.0903,
            440: 0.0924,
            445: 0.0951,
            450: 0.0969,
            455: 0.1003,
            460: 0.1083,
            465: 0.1203,
            470: 0.1383,
            475: 0.1634,
            480: 0.1988,
            485: 0.2376,
            490: 0.2795,
            495: 0.3275,
            500: 0.3671,
            505: 0.4030,
            510: 0.4201,
            515: 0.4257,
            520: 0.4218,
            525: 0.4090,
            530: 0.3977,
            535: 0.3769,
            540: 0.3559,
            545: 0.3312,
            550: 0.3072,
            555: 0.2803,
            560: 0.2532,
            565: 0.2313,
            570: 0.2109,
            575: 0.1897,
            580: 0.1723,
            585: 0.1528,
            590: 0.1355,
            595: 0.1196,
            600: 0.1050,
            605: 0.0949,
            610: 0.0868,
            615: 0.0797,
            620: 0.0783,
            625: 0.0732,
            630: 0.0737,
            635: 0.0709,
            640: 0.0703,
            645: 0.0696,
            650: 0.0673,
            655: 0.0677,
            660: 0.0682,
            665: 0.0665,
            670: 0.0691,
            675: 0.0695,
            680: 0.0723,
            685: 0.0727,
            690: 0.0757,
            695: 0.0767,
            700: 0.0810,
            705: 0.0818,
            710: 0.0837,
            715: 0.0822,
            720: 0.0838,
            725: 0.0847,
            730: 0.0837,
            735: 0.0864,
            740: 0.0882,
            745: 0.0923,
            750: 0.0967,
            755: 0.0996,
            760: 0.1027,
            765: 0.1080,
            770: 0.1115,
            775: 0.1118,
            780: 0.1152,
            785: 0.1201,
            790: 0.1253,
            795: 0.1313,
            800: 0.1393,
            805: 0.1393,
            810: 0.1393,
            815: 0.1393,
            820: 0.1393,
            825: 0.1393,
            830: 0.1393
        },
        'VS8': {
            360: 0.0872,
            365: 0.0872,
            370: 0.0872,
            375: 0.0872,
            380: 0.0872,
            385: 0.1001,
            390: 0.1159,
            395: 0.1339,
            400: 0.1431,
            405: 0.1516,
            410: 0.1570,
            415: 0.1608,
            420: 0.1649,
            425: 0.1678,
            430: 0.1785,
            435: 0.1829,
            440: 0.1896,
            445: 0.2032,
            450: 0.2120,
            455: 0.2294,
            460: 0.2539,
            465: 0.2869,
            470: 0.3170,
            475: 0.3570,
            480: 0.3994,
            485: 0.4346,
            490: 0.4615,
            495: 0.4747,
            500: 0.4754,
            505: 0.4691,
            510: 0.4556,
            515: 0.4371,
            520: 0.4154,
            525: 0.3937,
            530: 0.3737,
            535: 0.3459,
            540: 0.3203,
            545: 0.2941,
            550: 0.2715,
            555: 0.2442,
            560: 0.2205,
            565: 0.1979,
            570: 0.1800,
            575: 0.1610,
            580: 0.1463,
            585: 0.1284,
            590: 0.1172,
            595: 0.1045,
            600: 0.0964,
            605: 0.0903,
            610: 0.0873,
            615: 0.0846,
            620: 0.0829,
            625: 0.0814,
            630: 0.0805,
            635: 0.0803,
            640: 0.0801,
            645: 0.0776,
            650: 0.0797,
            655: 0.0801,
            660: 0.0810,
            665: 0.0819,
            670: 0.0856,
            675: 0.0913,
            680: 0.0930,
            685: 0.0958,
            690: 0.1016,
            695: 0.1044,
            700: 0.1047,
            705: 0.1062,
            710: 0.1052,
            715: 0.1029,
            720: 0.1025,
            725: 0.1008,
            730: 0.1036,
            735: 0.1059,
            740: 0.1123,
            745: 0.1175,
            750: 0.1217,
            755: 0.1304,
            760: 0.1330,
            765: 0.1373,
            770: 0.1376,
            775: 0.1384,
            780: 0.1390,
            785: 0.1378,
            790: 0.1501,
            795: 0.1526,
            800: 0.1646,
            805: 0.1646,
            810: 0.1646,
            815: 0.1646,
            820: 0.1646,
            825: 0.1646,
            830: 0.1646
        },
        'VS9': {
            360: 0.1167,
            365: 0.1167,
            370: 0.1167,
            375: 0.1167,
            380: 0.1167,
            385: 0.1352,
            390: 0.1674,
            395: 0.2024,
            400: 0.2298,
            405: 0.2521,
            410: 0.2635,
            415: 0.2702,
            420: 0.2758,
            425: 0.2834,
            430: 0.2934,
            435: 0.3042,
            440: 0.3201,
            445: 0.3329,
            450: 0.3511,
            455: 0.3724,
            460: 0.4027,
            465: 0.4367,
            470: 0.4625,
            475: 0.4890,
            480: 0.5085,
            485: 0.5181,
            490: 0.5243,
            495: 0.5179,
            500: 0.5084,
            505: 0.4904,
            510: 0.4717,
            515: 0.4467,
            520: 0.4207,
            525: 0.3931,
            530: 0.3653,
            535: 0.3363,
            540: 0.3083,
            545: 0.2808,
            550: 0.2538,
            555: 0.2260,
            560: 0.2024,
            565: 0.1865,
            570: 0.1697,
            575: 0.1592,
            580: 0.1482,
            585: 0.1393,
            590: 0.1316,
            595: 0.1217,
            600: 0.1182,
            605: 0.1112,
            610: 0.1071,
            615: 0.1059,
            620: 0.1044,
            625: 0.1021,
            630: 0.0991,
            635: 0.1000,
            640: 0.0980,
            645: 0.0963,
            650: 0.0997,
            655: 0.0994,
            660: 0.1022,
            665: 0.1005,
            670: 0.1044,
            675: 0.1073,
            680: 0.1069,
            685: 0.1103,
            690: 0.1104,
            695: 0.1084,
            700: 0.1092,
            705: 0.1074,
            710: 0.1059,
            715: 0.1082,
            720: 0.1106,
            725: 0.1129,
            730: 0.1186,
            735: 0.1243,
            740: 0.1359,
            745: 0.1466,
            750: 0.1617,
            755: 0.1739,
            760: 0.1814,
            765: 0.1907,
            770: 0.1976,
            775: 0.1958,
            780: 0.1972,
            785: 0.2018,
            790: 0.2093,
            795: 0.2161,
            800: 0.2269,
            805: 0.2269,
            810: 0.2269,
            815: 0.2269,
            820: 0.2269,
            825: 0.2269,
            830: 0.2269
        },
        'VS10': {
            360: 0.0790,
            365: 0.0790,
            370: 0.0790,
            375: 0.0790,
            380: 0.0790,
            385: 0.0984,
            390: 0.1242,
            395: 0.1595,
            400: 0.1937,
            405: 0.2215,
            410: 0.2419,
            415: 0.2488,
            420: 0.2603,
            425: 0.2776,
            430: 0.2868,
            435: 0.3107,
            440: 0.3309,
            445: 0.3515,
            450: 0.3676,
            455: 0.3819,
            460: 0.4026,
            465: 0.4189,
            470: 0.4317,
            475: 0.4363,
            480: 0.4356,
            485: 0.4297,
            490: 0.4199,
            495: 0.4058,
            500: 0.3882,
            505: 0.3660,
            510: 0.3433,
            515: 0.3148,
            520: 0.2890,
            525: 0.2583,
            530: 0.2340,
            535: 0.2076,
            540: 0.1839,
            545: 0.1613,
            550: 0.1434,
            555: 0.1243,
            560: 0.1044,
            565: 0.0978,
            570: 0.0910,
            575: 0.0832,
            580: 0.0771,
            585: 0.0747,
            590: 0.0726,
            595: 0.0682,
            600: 0.0671,
            605: 0.0660,
            610: 0.0661,
            615: 0.0660,
            620: 0.0653,
            625: 0.0644,
            630: 0.0653,
            635: 0.0669,
            640: 0.0660,
            645: 0.0677,
            650: 0.0668,
            655: 0.0693,
            660: 0.0689,
            665: 0.0676,
            670: 0.0694,
            675: 0.0687,
            680: 0.0698,
            685: 0.0679,
            690: 0.0694,
            695: 0.0675,
            700: 0.0676,
            705: 0.0662,
            710: 0.0681,
            715: 0.0706,
            720: 0.0728,
            725: 0.0766,
            730: 0.0814,
            735: 0.0901,
            740: 0.1042,
            745: 0.1228,
            750: 0.1482,
            755: 0.1793,
            760: 0.2129,
            765: 0.2445,
            770: 0.2674,
            775: 0.2838,
            780: 0.2979,
            785: 0.3067,
            790: 0.3226,
            795: 0.3396,
            800: 0.3512,
            805: 0.3512,
            810: 0.3512,
            815: 0.3512,
            820: 0.3512,
            825: 0.3512,
            830: 0.3512
        },
        'VS11': {
            360: 0.0692,
            365: 0.0692,
            370: 0.0692,
            375: 0.0692,
            380: 0.0692,
            385: 0.0846,
            390: 0.1050,
            395: 0.1345,
            400: 0.1666,
            405: 0.1913,
            410: 0.2125,
            415: 0.2141,
            420: 0.2271,
            425: 0.2454,
            430: 0.2498,
            435: 0.2811,
            440: 0.3074,
            445: 0.3244,
            450: 0.3301,
            455: 0.3317,
            460: 0.3341,
            465: 0.3320,
            470: 0.3309,
            475: 0.3207,
            480: 0.3105,
            485: 0.2954,
            490: 0.2809,
            495: 0.2645,
            500: 0.2501,
            505: 0.2309,
            510: 0.2111,
            515: 0.1909,
            520: 0.1746,
            525: 0.1533,
            530: 0.1390,
            535: 0.1228,
            540: 0.1097,
            545: 0.0974,
            550: 0.0867,
            555: 0.0764,
            560: 0.0597,
            565: 0.0596,
            570: 0.0558,
            575: 0.0525,
            580: 0.0509,
            585: 0.0495,
            590: 0.0468,
            595: 0.0477,
            600: 0.0472,
            605: 0.0427,
            610: 0.0436,
            615: 0.0435,
            620: 0.0427,
            625: 0.0432,
            630: 0.0446,
            635: 0.0439,
            640: 0.0438,
            645: 0.0456,
            650: 0.0451,
            655: 0.0466,
            660: 0.0475,
            665: 0.0458,
            670: 0.0471,
            675: 0.0475,
            680: 0.0479,
            685: 0.0443,
            690: 0.0478,
            695: 0.0457,
            700: 0.0469,
            705: 0.0467,
            710: 0.0493,
            715: 0.0499,
            720: 0.0528,
            725: 0.0555,
            730: 0.0585,
            735: 0.0646,
            740: 0.0700,
            745: 0.0845,
            750: 0.1026,
            755: 0.1341,
            760: 0.1696,
            765: 0.2112,
            770: 0.2497,
            775: 0.2809,
            780: 0.3112,
            785: 0.3368,
            790: 0.3567,
            795: 0.3711,
            800: 0.3811,
            805: 0.3811,
            810: 0.3811,
            815: 0.3811,
            820: 0.3811,
            825: 0.3811,
            830: 0.3811
        },
        'VS12': {
            360: 0.0941,
            365: 0.0941,
            370: 0.0941,
            375: 0.0941,
            380: 0.0941,
            385: 0.1164,
            390: 0.1467,
            395: 0.1852,
            400: 0.2211,
            405: 0.2502,
            410: 0.2689,
            415: 0.2804,
            420: 0.2878,
            425: 0.2970,
            430: 0.3074,
            435: 0.3181,
            440: 0.3290,
            445: 0.3351,
            450: 0.3330,
            455: 0.3269,
            460: 0.3159,
            465: 0.3021,
            470: 0.2877,
            475: 0.2667,
            480: 0.2440,
            485: 0.2194,
            490: 0.1960,
            495: 0.1770,
            500: 0.1629,
            505: 0.1479,
            510: 0.1356,
            515: 0.1242,
            520: 0.1132,
            525: 0.1062,
            530: 0.1026,
            535: 0.0981,
            540: 0.0946,
            545: 0.0906,
            550: 0.0860,
            555: 0.0801,
            560: 0.0748,
            565: 0.0735,
            570: 0.0731,
            575: 0.0719,
            580: 0.0726,
            585: 0.0731,
            590: 0.0740,
            595: 0.0747,
            600: 0.0741,
            605: 0.0741,
            610: 0.0735,
            615: 0.0723,
            620: 0.0732,
            625: 0.0734,
            630: 0.0767,
            635: 0.0794,
            640: 0.0832,
            645: 0.0878,
            650: 0.0933,
            655: 0.0982,
            660: 0.0991,
            665: 0.1026,
            670: 0.1017,
            675: 0.0991,
            680: 0.0970,
            685: 0.0938,
            690: 0.0931,
            695: 0.0921,
            700: 0.0927,
            705: 0.0940,
            710: 0.0968,
            715: 0.0993,
            720: 0.1030,
            725: 0.1069,
            730: 0.1145,
            735: 0.1229,
            740: 0.1338,
            745: 0.1470,
            750: 0.1620,
            755: 0.1806,
            760: 0.1969,
            765: 0.2148,
            770: 0.2337,
            775: 0.2568,
            780: 0.2776,
            785: 0.3028,
            790: 0.3361,
            795: 0.3601,
            800: 0.3897,
            805: 0.3897,
            810: 0.3897,
            815: 0.3897,
            820: 0.3897,
            825: 0.3897,
            830: 0.3897
        },
        'VS13': {
            360: 0.1272,
            365: 0.1272,
            370: 0.1272,
            375: 0.1272,
            380: 0.1272,
            385: 0.1612,
            390: 0.2101,
            395: 0.2840,
            400: 0.3543,
            405: 0.4155,
            410: 0.4422,
            415: 0.4490,
            420: 0.4484,
            425: 0.4372,
            430: 0.4269,
            435: 0.4117,
            440: 0.3945,
            445: 0.3749,
            450: 0.3578,
            455: 0.3390,
            460: 0.3172,
            465: 0.2992,
            470: 0.2811,
            475: 0.2640,
            480: 0.2495,
            485: 0.2304,
            490: 0.2175,
            495: 0.2020,
            500: 0.1907,
            505: 0.1824,
            510: 0.1752,
            515: 0.1686,
            520: 0.1613,
            525: 0.1544,
            530: 0.1476,
            535: 0.1430,
            540: 0.1412,
            545: 0.1400,
            550: 0.1425,
            555: 0.1451,
            560: 0.1464,
            565: 0.1468,
            570: 0.1452,
            575: 0.1419,
            580: 0.1423,
            585: 0.1401,
            590: 0.1435,
            595: 0.1546,
            600: 0.1715,
            605: 0.1979,
            610: 0.2283,
            615: 0.2603,
            620: 0.2882,
            625: 0.3171,
            630: 0.3435,
            635: 0.3700,
            640: 0.3962,
            645: 0.4246,
            650: 0.4562,
            655: 0.4849,
            660: 0.5124,
            665: 0.5388,
            670: 0.5625,
            675: 0.5843,
            680: 0.6050,
            685: 0.6259,
            690: 0.6403,
            695: 0.6604,
            700: 0.6725,
            705: 0.6842,
            710: 0.6927,
            715: 0.6964,
            720: 0.7037,
            725: 0.7145,
            730: 0.7206,
            735: 0.7213,
            740: 0.7336,
            745: 0.7319,
            750: 0.7373,
            755: 0.7413,
            760: 0.7440,
            765: 0.7507,
            770: 0.7483,
            775: 0.7491,
            780: 0.7547,
            785: 0.7523,
            790: 0.7563,
            795: 0.7567,
            800: 0.7633,
            805: 0.7633,
            810: 0.7633,
            815: 0.7633,
            820: 0.7633,
            825: 0.7633,
            830: 0.7633
        },
        'VS14': {
            360: 0.1086,
            365: 0.1086,
            370: 0.1086,
            375: 0.1086,
            380: 0.1155,
            385: 0.1440,
            390: 0.1918,
            395: 0.2577,
            400: 0.3258,
            405: 0.3747,
            410: 0.3989,
            415: 0.4021,
            420: 0.3976,
            425: 0.3903,
            430: 0.3819,
            435: 0.3741,
            440: 0.3657,
            445: 0.3541,
            450: 0.3461,
            455: 0.3356,
            460: 0.3272,
            465: 0.3178,
            470: 0.3071,
            475: 0.2933,
            480: 0.2776,
            485: 0.2607,
            490: 0.2464,
            495: 0.2361,
            500: 0.2293,
            505: 0.2198,
            510: 0.2040,
            515: 0.1843,
            520: 0.1669,
            525: 0.1511,
            530: 0.1435,
            535: 0.1406,
            540: 0.1452,
            545: 0.1504,
            550: 0.1494,
            555: 0.1476,
            560: 0.1408,
            565: 0.1505,
            570: 0.1666,
            575: 0.1965,
            580: 0.2355,
            585: 0.2893,
            590: 0.3412,
            595: 0.4002,
            600: 0.4482,
            605: 0.4962,
            610: 0.5328,
            615: 0.5605,
            620: 0.5873,
            625: 0.6071,
            630: 0.6222,
            635: 0.6352,
            640: 0.6416,
            645: 0.6478,
            650: 0.6549,
            655: 0.6615,
            660: 0.6651,
            665: 0.6649,
            670: 0.6708,
            675: 0.6717,
            680: 0.6721,
            685: 0.6745,
            690: 0.6780,
            695: 0.6881,
            700: 0.6954,
            705: 0.6991,
            710: 0.7018,
            715: 0.7068,
            720: 0.7076,
            725: 0.7146,
            730: 0.7189,
            735: 0.7237,
            740: 0.7288,
            745: 0.7218,
            750: 0.7299,
            755: 0.7320,
            760: 0.7371,
            765: 0.7345,
            770: 0.7369,
            775: 0.7366,
            780: 0.7449,
            785: 0.7368,
            790: 0.7457,
            795: 0.7494,
            800: 0.7507,
            805: 0.7507,
            810: 0.7507,
            815: 0.7507,
            820: 0.7507,
            825: 0.7507,
            830: 0.7507
        },
        'VS15': {
            360: 0.1069,
            365: 0.1069,
            370: 0.1069,
            375: 0.1069,
            380: 0.1069,
            385: 0.1246,
            390: 0.1455,
            395: 0.1623,
            400: 0.1733,
            405: 0.1772,
            410: 0.1782,
            415: 0.1749,
            420: 0.1776,
            425: 0.1751,
            430: 0.1747,
            435: 0.1762,
            440: 0.1774,
            445: 0.1749,
            450: 0.1736,
            455: 0.1679,
            460: 0.1638,
            465: 0.1576,
            470: 0.1516,
            475: 0.1437,
            480: 0.1366,
            485: 0.1308,
            490: 0.1239,
            495: 0.1185,
            500: 0.1126,
            505: 0.1072,
            510: 0.1022,
            515: 0.0966,
            520: 0.0936,
            525: 0.0911,
            530: 0.0909,
            535: 0.0868,
            540: 0.0874,
            545: 0.0846,
            550: 0.0825,
            555: 0.0812,
            560: 0.0767,
            565: 0.0810,
            570: 0.0870,
            575: 0.0968,
            580: 0.1176,
            585: 0.1553,
            590: 0.2000,
            595: 0.2632,
            600: 0.3219,
            605: 0.3875,
            610: 0.4409,
            615: 0.4925,
            620: 0.5343,
            625: 0.5709,
            630: 0.5969,
            635: 0.6178,
            640: 0.6301,
            645: 0.6440,
            650: 0.6512,
            655: 0.6617,
            660: 0.6638,
            665: 0.6699,
            670: 0.6731,
            675: 0.6802,
            680: 0.6779,
            685: 0.6811,
            690: 0.6829,
            695: 0.6905,
            700: 0.6972,
            705: 0.7007,
            710: 0.7085,
            715: 0.7092,
            720: 0.7130,
            725: 0.7181,
            730: 0.7260,
            735: 0.7244,
            740: 0.7313,
            745: 0.7319,
            750: 0.7342,
            755: 0.7317,
            760: 0.7365,
            765: 0.7338,
            770: 0.7424,
            775: 0.7372,
            780: 0.7399,
            785: 0.7405,
            790: 0.7461,
            795: 0.7420,
            800: 0.7439,
            805: 0.7439,
            810: 0.7439,
            815: 0.7439,
            820: 0.7439,
            825: 0.7439,
            830: 0.7439
        }
    }
}

VS_SDS = CaseInsensitiveMapping({
    key: dict((name, SpectralDistribution(data, name=name))
              for name, data in value.items())
    for key, value in VS_SDS_DATA.items()
})
"""
*NIST CQS* test colour samples spectral distributions.

References
----------
:cite:`Ohno2008a`, :cite:`Ohno2013`

VS_SDS : CaseInsensitiveMapping
    **{'NIST CQS 7.4', 'NIST CQS 9.0'}**
"""
