# -*- coding: utf-8 -*-
"""
ASTM G-173 Solar Spectral Irradiance
====================================

Defines the solar spectral irradiance spectral distribution used in various
plotting objects.

References
----------
-   :cite:`RenewableResourceDataCenter2003a` : Renewable Resource Data Center.
    (2003). Reference Solar Spectral Irradiance: ASTM G-173. Retrieved August
    23, 2014, from
    http://rredc.nrel.gov/solar/spectra/am1.5/ASTMG173/ASTMG173.html
"""

from __future__ import division, unicode_literals

from colour.algebra import LinearInterpolator
from colour.colorimetry import SpectralDistribution

__author__ = 'Colour Developers'
__copyright__ = 'Copyright (C) 2013-2020 - Colour Developers'
__license__ = 'New BSD License - https://opensource.org/licenses/BSD-3-Clause'
__maintainer__ = 'Colour Developers'
__email__ = 'colour-developers@colour-science.org'
__status__ = 'Production'

__all__ = [
    'ASTMG173_ETR_DATA', 'ASTMG173_GLOBAL_TILT_DATA',
    'ASTMG173_DIRECT_CIRCUMSOLAR_DATA', 'ASTMG173_ETR', 'ASTMG173_GLOBAL_TILT',
    'ASTMG173_DIRECT_CIRCUMSOLAR'
]

# yapf: disable
ASTMG173_ETR_DATA = {
    280.0: 8.2000e-02,
    280.5: 9.9000e-02,
    281.0: 1.5000e-01,
    281.5: 2.1200e-01,
    282.0: 2.6700e-01,
    282.5: 3.0300e-01,
    283.0: 3.2500e-01,
    283.5: 3.2300e-01,
    284.0: 2.9900e-01,
    284.5: 2.5024e-01,
    285.0: 1.7589e-01,
    285.5: 1.5500e-01,
    286.0: 2.4200e-01,
    286.5: 3.3300e-01,
    287.0: 3.6200e-01,
    287.5: 3.3900e-01,
    288.0: 3.1100e-01,
    288.5: 3.2500e-01,
    289.0: 3.9200e-01,
    289.5: 4.7900e-01,
    290.0: 5.6300e-01,
    290.5: 6.0600e-01,
    291.0: 6.1800e-01,
    291.5: 5.9800e-01,
    292.0: 5.6700e-01,
    292.5: 5.2900e-01,
    293.0: 5.3800e-01,
    293.5: 5.4900e-01,
    294.0: 5.3300e-01,
    294.5: 5.2000e-01,
    295.0: 5.2700e-01,
    295.5: 5.5900e-01,
    296.0: 5.7300e-01,
    296.5: 5.2100e-01,
    297.0: 4.7800e-01,
    297.5: 5.2900e-01,
    298.0: 5.2800e-01,
    298.5: 4.6800e-01,
    299.0: 4.7200e-01,
    299.5: 5.0100e-01,
    300.0: 4.5794e-01,
    300.5: 4.3300e-01,
    301.0: 4.6300e-01,
    301.5: 4.7756e-01,
    302.0: 4.4900e-01,
    302.5: 5.0800e-01,
    303.0: 6.1200e-01,
    303.5: 6.4600e-01,
    304.0: 6.2100e-01,
    304.5: 6.2600e-01,
    305.0: 6.4200e-01,
    305.5: 6.1100e-01,
    306.0: 5.6500e-01,
    306.5: 5.7500e-01,
    307.0: 6.0500e-01,
    307.5: 6.3100e-01,
    308.0: 6.4500e-01,
    308.5: 6.4050e-01,
    309.0: 5.8000e-01,
    309.5: 5.0000e-01,
    310.0: 5.3300e-01,
    310.5: 6.5200e-01,
    311.0: 7.6200e-01,
    311.5: 7.4400e-01,
    312.0: 7.0615e-01,
    312.5: 6.7600e-01,
    313.0: 6.9400e-01,
    313.5: 7.2400e-01,
    314.0: 7.1700e-01,
    314.5: 6.7600e-01,
    315.0: 6.8450e-01,
    315.5: 6.3200e-01,
    316.0: 5.8700e-01,
    316.5: 6.4900e-01,
    317.0: 7.3900e-01,
    317.5: 8.0189e-01,
    318.0: 7.2361e-01,
    318.5: 6.8000e-01,
    319.0: 7.0580e-01,
    319.5: 7.3318e-01,
    320.0: 7.7500e-01,
    320.5: 8.2600e-01,
    321.0: 7.6546e-01,
    321.5: 7.1900e-01,
    322.0: 7.3513e-01,
    322.5: 7.1152e-01,
    323.0: 6.4900e-01,
    323.5: 6.8138e-01,
    324.0: 7.4050e-01,
    324.5: 7.8800e-01,
    325.0: 8.2918e-01,
    325.5: 9.1377e-01,
    326.0: 9.9835e-01,
    326.5: 1.0166e+00,
    327.0: 1.0047e+00,
    327.5: 9.8679e-01,
    328.0: 9.5681e-01,
    328.5: 9.3400e-01,
    329.0: 1.0046e+00,
    329.5: 1.0958e+00,
    330.0: 1.1098e+00,
    330.5: 1.0429e+00,
    331.0: 9.9100e-01,
    331.5: 9.9396e-01,
    332.0: 9.9277e-01,
    332.5: 9.8003e-01,
    333.0: 9.6394e-01,
    333.5: 9.3700e-01,
    334.0: 9.5557e-01,
    334.5: 9.8862e-01,
    335.0: 1.0097e+00,
    335.5: 9.7453e-01,
    336.0: 8.8979e-01,
    336.5: 8.2900e-01,
    337.0: 8.1800e-01,
    337.5: 8.7732e-01,
    338.0: 9.2529e-01,
    338.5: 9.5783e-01,
    339.0: 9.6863e-01,
    339.5: 9.8900e-01,
    340.0: 1.0544e+00,
    340.5: 1.0463e+00,
    341.0: 9.7100e-01,
    341.5: 9.5900e-01,
    342.0: 9.9570e-01,
    342.5: 1.0277e+00,
    343.0: 1.0417e+00,
    343.5: 9.8301e-01,
    344.0: 8.5416e-01,
    344.5: 8.1300e-01,
    345.0: 9.1583e-01,
    345.5: 9.7021e-01,
    346.0: 9.4269e-01,
    346.5: 9.5594e-01,
    347.0: 9.6954e-01,
    347.5: 9.3318e-01,
    348.0: 9.2472e-01,
    348.5: 9.3721e-01,
    349.0: 8.9900e-01,
    349.5: 9.1969e-01,
    350.0: 1.0122e+00,
    350.5: 1.0849e+00,
    351.0: 1.0534e+00,
    351.5: 1.0129e+00,
    352.0: 9.8383e-01,
    352.5: 9.2600e-01,
    353.0: 9.8012e-01,
    353.5: 1.0752e+00,
    354.0: 1.1346e+00,
    354.5: 1.1440e+00,
    355.0: 1.1406e+00,
    355.5: 1.0964e+00,
    356.0: 1.0250e+00,
    356.5: 9.5914e-01,
    357.0: 8.4200e-01,
    357.5: 8.5015e-01,
    358.0: 7.8916e-01,
    358.5: 7.3100e-01,
    359.0: 8.5805e-01,
    359.5: 1.0321e+00,
    360.0: 1.0890e+00,
    360.5: 1.0265e+00,
    361.0: 9.4150e-01,
    361.5: 9.1800e-01,
    362.0: 9.5800e-01,
    362.5: 1.0450e+00,
    363.0: 1.0710e+00,
    363.5: 1.0380e+00,
    364.0: 1.0716e+00,
    364.5: 1.0590e+00,
    365.0: 1.0970e+00,
    365.5: 1.2041e+00,
    366.0: 1.2860e+00,
    366.5: 1.2843e+00,
    367.0: 1.2577e+00,
    367.5: 1.2317e+00,
    368.0: 1.1570e+00,
    368.5: 1.1459e+00,
    369.0: 1.1944e+00,
    369.5: 1.2795e+00,
    370.0: 1.2934e+00,
    370.5: 1.1660e+00,
    371.0: 1.1811e+00,
    371.5: 1.2249e+00,
    372.0: 1.1444e+00,
    372.5: 1.0876e+00,
    373.0: 1.0445e+00,
    373.5: 9.3900e-01,
    374.0: 9.3442e-01,
    374.5: 9.2500e-01,
    375.0: 9.8500e-01,
    375.5: 1.0874e+00,
    376.0: 1.1240e+00,
    376.5: 1.1040e+00,
    377.0: 1.1813e+00,
    377.5: 1.3149e+00,
    378.0: 1.4144e+00,
    378.5: 1.3765e+00,
    379.0: 1.2260e+00,
    379.5: 1.0980e+00,
    380.0: 1.1520e+00,
    380.5: 1.2310e+00,
    381.0: 1.2490e+00,
    381.5: 1.1224e+00,
    382.0: 9.5426e-01,
    382.5: 8.2313e-01,
    383.0: 7.3603e-01,
    383.5: 7.1095e-01,
    384.0: 8.2100e-01,
    384.5: 9.8620e-01,
    385.0: 1.0802e+00,
    385.5: 1.0296e+00,
    386.0: 9.9113e-01,
    386.5: 1.0279e+00,
    387.0: 1.0354e+00,
    387.5: 1.0186e+00,
    388.0: 1.0067e+00,
    388.5: 9.9743e-01,
    389.0: 1.0810e+00,
    389.5: 1.1958e+00,
    390.0: 1.2519e+00,
    390.5: 1.2601e+00,
    391.0: 1.3322e+00,
    391.5: 1.3490e+00,
    392.0: 1.2400e+00,
    392.5: 1.0312e+00,
    393.0: 7.4500e-01,
    393.5: 5.9120e-01,
    394.0: 7.6675e-01,
    394.5: 1.0557e+00,
    395.0: 1.2450e+00,
    395.5: 1.3245e+00,
    396.0: 1.1626e+00,
    396.5: 8.4330e-01,
    397.0: 6.5200e-01,
    397.5: 9.6142e-01,
    398.0: 1.3010e+00,
    398.5: 1.5342e+00,
    399.0: 1.6245e+00,
    399.5: 1.6717e+00,
    400.0: 1.6885e+00,
    401.0: 1.7520e+00,
    402.0: 1.8140e+00,
    403.0: 1.7400e+00,
    404.0: 1.7630e+00,
    405.0: 1.7150e+00,
    406.0: 1.6660e+00,
    407.0: 1.6300e+00,
    408.0: 1.6990e+00,
    409.0: 1.8090e+00,
    410.0: 1.5370e+00,
    411.0: 1.7150e+00,
    412.0: 1.8160e+00,
    413.0: 1.7392e+00,
    414.0: 1.7144e+00,
    415.0: 1.7688e+00,
    416.0: 1.8150e+00,
    417.0: 1.7660e+00,
    418.0: 1.6850e+00,
    419.0: 1.7490e+00,
    420.0: 1.5990e+00,
    421.0: 1.8110e+00,
    422.0: 1.7820e+00,
    423.0: 1.7210e+00,
    424.0: 1.7080e+00,
    425.0: 1.7550e+00,
    426.0: 1.6990e+00,
    427.0: 1.6380e+00,
    428.0: 1.6510e+00,
    429.0: 1.5230e+00,
    430.0: 1.2120e+00,
    431.0: 1.0990e+00,
    432.0: 1.8220e+00,
    433.0: 1.6913e+00,
    434.0: 1.5600e+00,
    435.0: 1.7090e+00,
    436.0: 1.8680e+00,
    437.0: 1.9000e+00,
    438.0: 1.6630e+00,
    439.0: 1.6010e+00,
    440.0: 1.8300e+00,
    441.0: 1.7990e+00,
    442.0: 1.9220e+00,
    443.0: 1.9490e+00,
    444.0: 1.8941e+00,
    445.0: 1.9650e+00,
    446.0: 1.7557e+00,
    447.0: 1.9900e+00,
    448.0: 2.0140e+00,
    449.0: 2.0010e+00,
    450.0: 2.0690e+00,
    451.0: 2.1420e+00,
    452.0: 2.0470e+00,
    453.0: 1.8864e+00,
    454.0: 2.0180e+00,
    455.0: 2.0010e+00,
    456.0: 2.0630e+00,
    457.0: 2.0770e+00,
    458.0: 2.0320e+00,
    459.0: 2.0120e+00,
    460.0: 1.9973e+00,
    461.0: 2.0639e+00,
    462.0: 2.0780e+00,
    463.0: 2.0840e+00,
    464.0: 2.0150e+00,
    465.0: 1.9840e+00,
    466.0: 2.0210e+00,
    467.0: 1.9310e+00,
    468.0: 2.0120e+00,
    469.0: 2.0180e+00,
    470.0: 1.9390e+00,
    471.0: 1.9690e+00,
    472.0: 2.0700e+00,
    473.0: 1.9882e+00,
    474.0: 2.0120e+00,
    475.0: 2.0800e+00,
    476.0: 2.0120e+00,
    477.0: 2.0250e+00,
    478.0: 2.0860e+00,
    479.0: 2.0400e+00,
    480.0: 2.0680e+00,
    481.0: 2.0610e+00,
    482.0: 2.0623e+00,
    483.0: 2.0310e+00,
    484.0: 1.9890e+00,
    485.0: 1.9790e+00,
    486.0: 1.6010e+00,
    487.0: 1.7890e+00,
    488.0: 1.9350e+00,
    489.0: 1.8224e+00,
    490.0: 2.0320e+00,
    491.0: 1.9490e+00,
    492.0: 1.8560e+00,
    493.0: 1.9830e+00,
    494.0: 1.9339e+00,
    495.0: 2.0510e+00,
    496.0: 1.9490e+00,
    497.0: 1.9800e+00,
    498.0: 1.9240e+00,
    499.0: 1.9230e+00,
    500.0: 1.9160e+00,
    501.0: 1.8580e+00,
    502.0: 1.8600e+00,
    503.0: 1.9490e+00,
    504.0: 1.8330e+00,
    505.0: 1.9472e+00,
    506.0: 2.0250e+00,
    507.0: 1.9354e+00,
    508.0: 1.8800e+00,
    509.0: 1.9650e+00,
    510.0: 1.9100e+00,
    511.0: 1.9410e+00,
    512.0: 1.9890e+00,
    513.0: 1.8660e+00,
    514.0: 1.8240e+00,
    515.0: 1.8750e+00,
    516.0: 1.8910e+00,
    517.0: 1.5390e+00,
    518.0: 1.7590e+00,
    519.0: 1.7040e+00,
    520.0: 1.8600e+00,
    521.0: 1.8730e+00,
    522.0: 1.9150e+00,
    523.0: 1.8040e+00,
    524.0: 1.9410e+00,
    525.0: 1.9280e+00,
    526.0: 1.8740e+00,
    527.0: 1.6410e+00,
    528.0: 1.8800e+00,
    529.0: 1.9690e+00,
    530.0: 1.8920e+00,
    531.0: 1.9950e+00,
    532.0: 1.9580e+00,
    533.0: 1.7470e+00,
    534.0: 1.8690e+00,
    535.0: 1.8950e+00,
    536.0: 1.9740e+00,
    537.0: 1.8240e+00,
    538.0: 1.9130e+00,
    539.0: 1.8640e+00,
    540.0: 1.8000e+00,
    541.0: 1.7340e+00,
    542.0: 1.8880e+00,
    543.0: 1.8510e+00,
    544.0: 1.9190e+00,
    545.0: 1.8740e+00,
    546.0: 1.8609e+00,
    547.0: 1.8820e+00,
    548.0: 1.8260e+00,
    549.0: 1.8800e+00,
    550.0: 1.8630e+00,
    551.0: 1.8590e+00,
    552.0: 1.8960e+00,
    553.0: 1.8420e+00,
    554.0: 1.8780e+00,
    555.0: 1.8890e+00,
    556.0: 1.8570e+00,
    557.0: 1.8120e+00,
    558.0: 1.8530e+00,
    559.0: 1.7550e+00,
    560.0: 1.7860e+00,
    561.0: 1.8900e+00,
    562.0: 1.8010e+00,
    563.0: 1.8710e+00,
    564.0: 1.8360e+00,
    565.0: 1.8490e+00,
    566.0: 1.7500e+00,
    567.0: 1.8680e+00,
    568.0: 1.8590e+00,
    569.0: 1.8310e+00,
    570.0: 1.8280e+00,
    571.0: 1.7620e+00,
    572.0: 1.8720e+00,
    573.0: 1.8810e+00,
    574.0: 1.8730e+00,
    575.0: 1.8340e+00,
    576.0: 1.8180e+00,
    577.0: 1.8620e+00,
    578.0: 1.7990e+00,
    579.0: 1.8160e+00,
    580.0: 1.8340e+00,
    581.0: 1.8330e+00,
    582.0: 1.8520e+00,
    583.0: 1.8630e+00,
    584.0: 1.8540e+00,
    585.0: 1.8360e+00,
    586.0: 1.7920e+00,
    587.0: 1.8380e+00,
    588.0: 1.8210e+00,
    589.0: 1.6240e+00,
    590.0: 1.7218e+00,
    591.0: 1.8090e+00,
    592.0: 1.7880e+00,
    593.0: 1.7920e+00,
    594.0: 1.7890e+00,
    595.0: 1.7780e+00,
    596.0: 1.7960e+00,
    597.0: 1.8060e+00,
    598.0: 1.7720e+00,
    599.0: 1.7640e+00,
    600.0: 1.7700e+00,
    601.0: 1.7420e+00,
    602.0: 1.7150e+00,
    603.0: 1.7490e+00,
    604.0: 1.7790e+00,
    605.0: 1.7730e+00,
    606.0: 1.7580e+00,
    607.0: 1.7620e+00,
    608.0: 1.7510e+00,
    609.0: 1.7340e+00,
    610.0: 1.7240e+00,
    611.0: 1.7120e+00,
    612.0: 1.7360e+00,
    613.0: 1.7100e+00,
    614.0: 1.6550e+00,
    615.0: 1.7120e+00,
    616.0: 1.6640e+00,
    617.0: 1.6410e+00,
    618.0: 1.7020e+00,
    619.0: 1.7090e+00,
    620.0: 1.7110e+00,
    621.0: 1.7240e+00,
    622.0: 1.6784e+00,
    623.0: 1.6820e+00,
    624.0: 1.6670e+00,
    625.0: 1.6440e+00,
    626.0: 1.6400e+00,
    627.0: 1.6930e+00,
    628.0: 1.6930e+00,
    629.0: 1.6870e+00,
    630.0: 1.6650e+00,
    631.0: 1.6590e+00,
    632.0: 1.5901e+00,
    633.0: 1.6740e+00,
    634.0: 1.6370e+00,
    635.0: 1.6520e+00,
    636.0: 1.6093e+00,
    637.0: 1.6610e+00,
    638.0: 1.6650e+00,
    639.0: 1.6530e+00,
    640.0: 1.6130e+00,
    641.0: 1.6100e+00,
    642.0: 1.6090e+00,
    643.0: 1.6250e+00,
    644.0: 1.6140e+00,
    645.0: 1.6270e+00,
    646.0: 1.5910e+00,
    647.0: 1.6060e+00,
    648.0: 1.6020e+00,
    649.0: 1.5510e+00,
    650.0: 1.5260e+00,
    651.0: 1.6130e+00,
    652.0: 1.5910e+00,
    653.0: 1.5980e+00,
    654.0: 1.5750e+00,
    655.0: 1.5230e+00,
    656.0: 1.3233e+00,
    657.0: 1.3840e+00,
    658.0: 1.5390e+00,
    659.0: 1.5420e+00,
    660.0: 1.5580e+00,
    661.0: 1.5660e+00,
    662.0: 1.5710e+00,
    663.0: 1.5630e+00,
    664.0: 1.5540e+00,
    665.0: 1.5670e+00,
    666.0: 1.5550e+00,
    667.0: 1.5354e+00,
    668.0: 1.5348e+00,
    669.0: 1.5580e+00,
    670.0: 1.5340e+00,
    671.0: 1.5290e+00,
    672.0: 1.5060e+00,
    673.0: 1.5170e+00,
    674.0: 1.5130e+00,
    675.0: 1.4990e+00,
    676.0: 1.5150e+00,
    677.0: 1.5000e+00,
    678.0: 1.5070e+00,
    679.0: 1.4930e+00,
    680.0: 1.4940e+00,
    681.0: 1.4870e+00,
    682.0: 1.4930e+00,
    683.0: 1.4760e+00,
    684.0: 1.4660e+00,
    685.0: 1.4650e+00,
    686.0: 1.4330e+00,
    687.0: 1.4720e+00,
    688.0: 1.4760e+00,
    689.0: 1.4780e+00,
    690.0: 1.4790e+00,
    691.0: 1.4680e+00,
    692.0: 1.4540e+00,
    693.0: 1.4580e+00,
    694.0: 1.4570e+00,
    695.0: 1.4350e+00,
    696.0: 1.4420e+00,
    697.0: 1.4380e+00,
    698.0: 1.4170e+00,
    699.0: 1.4340e+00,
    700.0: 1.4220e+00,
    701.0: 1.4131e+00,
    702.0: 1.3987e+00,
    703.0: 1.4095e+00,
    704.0: 1.4187e+00,
    705.0: 1.4330e+00,
    706.0: 1.4138e+00,
    707.0: 1.4040e+00,
    708.0: 1.3990e+00,
    709.0: 1.3900e+00,
    710.0: 1.4040e+00,
    711.0: 1.3970e+00,
    712.0: 1.3818e+00,
    713.0: 1.3702e+00,
    714.0: 1.3819e+00,
    715.0: 1.3502e+00,
    716.0: 1.3694e+00,
    717.0: 1.3650e+00,
    718.0: 1.3570e+00,
    719.0: 1.3010e+00,
    720.0: 1.3487e+00,
    721.0: 1.3480e+00,
    722.0: 1.3600e+00,
    723.0: 1.3510e+00,
    724.0: 1.3607e+00,
    725.0: 1.3465e+00,
    726.0: 1.3429e+00,
    727.0: 1.3444e+00,
    728.0: 1.3370e+00,
    729.0: 1.2796e+00,
    730.0: 1.3357e+00,
    731.0: 1.3104e+00,
    732.0: 1.3333e+00,
    733.0: 1.3108e+00,
    734.0: 1.3390e+00,
    735.0: 1.3271e+00,
    736.0: 1.3100e+00,
    737.0: 1.3120e+00,
    738.0: 1.3000e+00,
    739.0: 1.2646e+00,
    740.0: 1.2830e+00,
    741.0: 1.2707e+00,
    742.0: 1.2649e+00,
    743.0: 1.2892e+00,
    744.0: 1.2955e+00,
    745.0: 1.2920e+00,
    746.0: 1.2892e+00,
    747.0: 1.2890e+00,
    748.0: 1.2808e+00,
    749.0: 1.2760e+00,
    750.0: 1.2740e+00,
    751.0: 1.2680e+00,
    752.0: 1.2720e+00,
    753.0: 1.2650e+00,
    754.0: 1.2809e+00,
    755.0: 1.2771e+00,
    756.0: 1.2610e+00,
    757.0: 1.2598e+00,
    758.0: 1.2680e+00,
    759.0: 1.2500e+00,
    760.0: 1.2590e+00,
    761.0: 1.2487e+00,
    762.0: 1.2460e+00,
    763.0: 1.2543e+00,
    764.0: 1.2566e+00,
    765.0: 1.2452e+00,
    766.0: 1.2000e+00,
    767.0: 1.2120e+00,
    768.0: 1.2310e+00,
    769.0: 1.2142e+00,
    770.0: 1.2146e+00,
    771.0: 1.2073e+00,
    772.0: 1.2120e+00,
    773.0: 1.2100e+00,
    774.0: 1.2090e+00,
    775.0: 1.2080e+00,
    776.0: 1.2102e+00,
    777.0: 1.2028e+00,
    778.0: 1.2014e+00,
    779.0: 1.2067e+00,
    780.0: 1.1930e+00,
    781.0: 1.1898e+00,
    782.0: 1.1950e+00,
    783.0: 1.1896e+00,
    784.0: 1.1810e+00,
    785.0: 1.1874e+00,
    786.0: 1.1890e+00,
    787.0: 1.1877e+00,
    788.0: 1.1800e+00,
    789.0: 1.1812e+00,
    790.0: 1.1704e+00,
    791.0: 1.1632e+00,
    792.0: 1.1500e+00,
    793.0: 1.1449e+00,
    794.0: 1.1260e+00,
    795.0: 1.1280e+00,
    796.0: 1.1427e+00,
    797.0: 1.1580e+00,
    798.0: 1.1501e+00,
    799.0: 1.1350e+00,
    800.0: 1.1248e+00,
    801.0: 1.1427e+00,
    802.0: 1.1422e+00,
    803.0: 1.1241e+00,
    804.0: 1.1342e+00,
    805.0: 1.1001e+00,
    806.0: 1.1309e+00,
    807.0: 1.1230e+00,
    808.0: 1.1162e+00,
    809.0: 1.0980e+00,
    810.0: 1.1100e+00,
    811.0: 1.1143e+00,
    812.0: 1.1160e+00,
    813.0: 1.1180e+00,
    814.0: 1.1174e+00,
    815.0: 1.1091e+00,
    816.0: 1.1050e+00,
    817.0: 1.1083e+00,
    818.0: 1.0861e+00,
    819.0: 1.0738e+00,
    820.0: 1.0740e+00,
    821.0: 1.0760e+00,
    822.0: 1.0560e+00,
    823.0: 1.0760e+00,
    824.0: 1.0715e+00,
    825.0: 1.0699e+00,
    826.0: 1.0759e+00,
    827.0: 1.0762e+00,
    828.0: 1.0749e+00,
    829.0: 1.0620e+00,
    830.0: 1.0563e+00,
    831.0: 1.0560e+00,
    832.0: 1.0550e+00,
    833.0: 1.0327e+00,
    834.0: 1.0459e+00,
    835.0: 1.0476e+00,
    836.0: 1.0597e+00,
    837.0: 1.0511e+00,
    838.0: 1.0549e+00,
    839.0: 1.0400e+00,
    840.0: 1.0500e+00,
    841.0: 1.0520e+00,
    842.0: 1.0287e+00,
    843.0: 1.0271e+00,
    844.0: 1.0140e+00,
    845.0: 1.0360e+00,
    846.0: 1.0370e+00,
    847.0: 1.0060e+00,
    848.0: 1.0110e+00,
    849.0: 1.0050e+00,
    850.0: 9.1000e-01,
    851.0: 9.9800e-01,
    852.0: 9.9000e-01,
    853.0: 9.8100e-01,
    854.0: 8.6900e-01,
    855.0: 9.2700e-01,
    856.0: 9.9017e-01,
    857.0: 1.0056e+00,
    858.0: 1.0070e+00,
    859.0: 1.0050e+00,
    860.0: 1.0000e+00,
    861.0: 9.9900e-01,
    862.0: 1.0060e+00,
    863.0: 1.0120e+00,
    864.0: 9.9006e-01,
    865.0: 9.7354e-01,
    866.0: 8.5800e-01,
    867.0: 9.2500e-01,
    868.0: 9.6900e-01,
    869.0: 9.5900e-01,
    870.0: 9.7700e-01,
    871.0: 9.6317e-01,
    872.0: 9.7600e-01,
    873.0: 9.6614e-01,
    874.0: 9.4913e-01,
    875.0: 9.3562e-01,
    876.0: 9.6200e-01,
    877.0: 9.6500e-01,
    878.0: 9.6200e-01,
    879.0: 9.4571e-01,
    880.0: 9.4856e-01,
    881.0: 9.1963e-01,
    882.0: 9.4408e-01,
    883.0: 9.3930e-01,
    884.0: 9.4352e-01,
    885.0: 9.5469e-01,
    886.0: 9.2558e-01,
    887.0: 9.2300e-01,
    888.0: 9.4066e-01,
    889.0: 9.4367e-01,
    890.0: 9.4236e-01,
    891.0: 9.3928e-01,
    892.0: 9.3259e-01,
    893.0: 9.1800e-01,
    894.0: 9.2400e-01,
    895.0: 9.2600e-01,
    896.0: 9.3425e-01,
    897.0: 9.2686e-01,
    898.0: 9.2378e-01,
    899.0: 9.1396e-01,
    900.0: 9.1378e-01,
    901.0: 8.9834e-01,
    902.0: 8.7720e-01,
    903.0: 9.2233e-01,
    904.0: 9.2100e-01,
    905.0: 9.1800e-01,
    906.0: 9.0780e-01,
    907.0: 9.1450e-01,
    908.0: 9.0143e-01,
    909.0: 8.8552e-01,
    910.0: 8.9496e-01,
    911.0: 8.9730e-01,
    912.0: 8.9016e-01,
    913.0: 9.0020e-01,
    914.0: 8.9820e-01,
    915.0: 8.8830e-01,
    916.0: 8.9470e-01,
    917.0: 8.8860e-01,
    918.0: 8.8810e-01,
    919.0: 8.9210e-01,
    920.0: 8.8540e-01,
    921.0: 8.7030e-01,
    922.0: 8.6280e-01,
    923.0: 8.3335e-01,
    924.0: 8.6957e-01,
    925.0: 8.7642e-01,
    926.0: 8.4561e-01,
    927.0: 8.7809e-01,
    928.0: 8.7110e-01,
    929.0: 8.6954e-01,
    930.0: 8.6866e-01,
    931.0: 8.7078e-01,
    932.0: 8.6514e-01,
    933.0: 8.6958e-01,
    934.0: 8.6709e-01,
    935.0: 8.5601e-01,
    936.0: 8.5222e-01,
    937.0: 8.5368e-01,
    938.0: 8.5936e-01,
    939.0: 8.5676e-01,
    940.0: 8.4000e-01,
    941.0: 8.3153e-01,
    942.0: 8.0616e-01,
    943.0: 8.3814e-01,
    944.0: 8.1854e-01,
    945.0: 8.2441e-01,
    946.0: 8.3138e-01,
    947.0: 8.2809e-01,
    948.0: 8.3137e-01,
    949.0: 8.3303e-01,
    950.0: 8.2867e-01,
    951.0: 8.2019e-01,
    952.0: 8.2402e-01,
    953.0: 8.2050e-01,
    954.0: 8.1200e-01,
    955.0: 7.6900e-01,
    956.0: 8.0200e-01,
    957.0: 8.0583e-01,
    958.0: 8.1147e-01,
    959.0: 8.1483e-01,
    960.0: 8.0627e-01,
    961.0: 7.9982e-01,
    962.0: 8.0173e-01,
    963.0: 7.9761e-01,
    964.0: 7.8808e-01,
    965.0: 7.9579e-01,
    966.0: 7.8598e-01,
    967.0: 7.9503e-01,
    968.0: 7.8526e-01,
    969.0: 7.8789e-01,
    970.0: 7.9140e-01,
    971.0: 7.8186e-01,
    972.0: 7.8785e-01,
    973.0: 7.8149e-01,
    974.0: 7.6882e-01,
    975.0: 7.7348e-01,
    976.0: 7.8054e-01,
    977.0: 7.7208e-01,
    978.0: 7.7800e-01,
    979.0: 7.6650e-01,
    980.0: 7.7512e-01,
    981.0: 7.7300e-01,
    982.0: 7.7879e-01,
    983.0: 7.7541e-01,
    984.0: 7.6881e-01,
    985.0: 7.7081e-01,
    986.0: 7.6620e-01,
    987.0: 7.6606e-01,
    988.0: 7.6368e-01,
    989.0: 7.5500e-01,
    990.0: 7.5694e-01,
    991.0: 7.6161e-01,
    992.0: 7.5714e-01,
    993.0: 7.5639e-01,
    994.0: 7.5851e-01,
    995.0: 7.5392e-01,
    996.0: 7.5347e-01,
    997.0: 7.4300e-01,
    998.0: 7.4300e-01,
    999.0: 7.4200e-01,
    1000.0: 7.4255e-01,
    1001.0: 7.4741e-01,
    1002.0: 7.4677e-01,
    1003.0: 7.3836e-01,
    1004.0: 7.3133e-01,
    1005.0: 6.8365e-01,
    1006.0: 7.1770e-01,
    1007.0: 7.3093e-01,
    1008.0: 7.3339e-01,
    1009.0: 7.2934e-01,
    1010.0: 7.3055e-01,
    1011.0: 7.2562e-01,
    1012.0: 7.2496e-01,
    1013.0: 7.1960e-01,
    1014.0: 7.2265e-01,
    1015.0: 7.1062e-01,
    1016.0: 7.1419e-01,
    1017.0: 7.0659e-01,
    1018.0: 7.1566e-01,
    1019.0: 6.9606e-01,
    1020.0: 7.0127e-01,
    1021.0: 7.0300e-01,
    1022.0: 6.9300e-01,
    1023.0: 6.9900e-01,
    1024.0: 7.0057e-01,
    1025.0: 6.9918e-01,
    1026.0: 6.9943e-01,
    1027.0: 6.9594e-01,
    1028.0: 6.9722e-01,
    1029.0: 6.9000e-01,
    1030.0: 6.9208e-01,
    1031.0: 6.8985e-01,
    1032.0: 6.8960e-01,
    1033.0: 6.7800e-01,
    1034.0: 6.8181e-01,
    1035.0: 6.8400e-01,
    1036.0: 6.8431e-01,
    1037.0: 6.7731e-01,
    1038.0: 6.7400e-01,
    1039.0: 6.7900e-01,
    1040.0: 6.7440e-01,
    1041.0: 6.7490e-01,
    1042.0: 6.7516e-01,
    1043.0: 6.6860e-01,
    1044.0: 6.7199e-01,
    1045.0: 6.6846e-01,
    1046.0: 6.5131e-01,
    1047.0: 6.6155e-01,
    1048.0: 6.6781e-01,
    1049.0: 6.6491e-01,
    1050.0: 6.6117e-01,
    1051.0: 6.6182e-01,
    1052.0: 6.5848e-01,
    1053.0: 6.5698e-01,
    1054.0: 6.5474e-01,
    1055.0: 6.5753e-01,
    1056.0: 6.5629e-01,
    1057.0: 6.5540e-01,
    1058.0: 6.4937e-01,
    1059.0: 6.3046e-01,
    1060.0: 6.4831e-01,
    1061.0: 6.3387e-01,
    1062.0: 6.4651e-01,
    1063.0: 6.3669e-01,
    1064.0: 6.4621e-01,
    1065.0: 6.4533e-01,
    1066.0: 6.3139e-01,
    1067.0: 6.3517e-01,
    1068.0: 6.3591e-01,
    1069.0: 6.0084e-01,
    1070.0: 6.2165e-01,
    1071.0: 6.2821e-01,
    1072.0: 6.3131e-01,
    1073.0: 6.1796e-01,
    1074.0: 6.3063e-01,
    1075.0: 6.1768e-01,
    1076.0: 6.2468e-01,
    1077.0: 6.2426e-01,
    1078.0: 6.2531e-01,
    1079.0: 6.1203e-01,
    1080.0: 6.2250e-01,
    1081.0: 6.0726e-01,
    1082.0: 6.0249e-01,
    1083.0: 6.0900e-01,
    1084.0: 6.1392e-01,
    1085.0: 6.1181e-01,
    1086.0: 6.1570e-01,
    1087.0: 5.9145e-01,
    1088.0: 6.0728e-01,
    1089.0: 6.0908e-01,
    1090.0: 6.0442e-01,
    1091.0: 6.0799e-01,
    1092.0: 5.9692e-01,
    1093.0: 5.9221e-01,
    1094.0: 5.6065e-01,
    1095.0: 5.8252e-01,
    1096.0: 5.8600e-01,
    1097.0: 5.9100e-01,
    1098.0: 5.8700e-01,
    1099.0: 5.8216e-01,
    1100.0: 6.0000e-01,
    1101.0: 5.9900e-01,
    1102.0: 5.8202e-01,
    1103.0: 5.9500e-01,
    1104.0: 5.9300e-01,
    1105.0: 5.9100e-01,
    1106.0: 5.9000e-01,
    1107.0: 5.8600e-01,
    1108.0: 5.8650e-01,
    1109.0: 5.8550e-01,
    1110.0: 5.8700e-01,
    1111.0: 5.8250e-01,
    1112.0: 5.8342e-01,
    1113.0: 5.7800e-01,
    1114.0: 5.8400e-01,
    1115.0: 5.8200e-01,
    1116.0: 5.7500e-01,
    1117.0: 5.8300e-01,
    1118.0: 5.7222e-01,
    1119.0: 5.7151e-01,
    1120.0: 5.6850e-01,
    1121.0: 5.7200e-01,
    1122.0: 5.7500e-01,
    1123.0: 5.7400e-01,
    1124.0: 5.7250e-01,
    1125.0: 5.7012e-01,
    1126.0: 5.5300e-01,
    1127.0: 5.6179e-01,
    1128.0: 5.6829e-01,
    1129.0: 5.6837e-01,
    1130.0: 5.6401e-01,
    1131.0: 5.6800e-01,
    1132.0: 5.7000e-01,
    1133.0: 5.6186e-01,
    1134.0: 5.6346e-01,
    1135.0: 5.6250e-01,
    1136.0: 5.6481e-01,
    1137.0: 5.5197e-01,
    1138.0: 5.4400e-01,
    1139.0: 5.5300e-01,
    1140.0: 5.5573e-01,
    1141.0: 5.4303e-01,
    1142.0: 5.5531e-01,
    1143.0: 5.4960e-01,
    1144.0: 5.4501e-01,
    1145.0: 5.5001e-01,
    1146.0: 5.5407e-01,
    1147.0: 5.5408e-01,
    1148.0: 5.5032e-01,
    1149.0: 5.4788e-01,
    1150.0: 5.4623e-01,
    1151.0: 5.4531e-01,
    1152.0: 5.4170e-01,
    1153.0: 5.4308e-01,
    1154.0: 5.4651e-01,
    1155.0: 5.4731e-01,
    1156.0: 5.4022e-01,
    1157.0: 5.4370e-01,
    1158.0: 5.4334e-01,
    1159.0: 5.3548e-01,
    1160.0: 5.2928e-01,
    1161.0: 5.1899e-01,
    1162.0: 5.3336e-01,
    1163.0: 5.3708e-01,
    1164.0: 5.2684e-01,
    1165.0: 5.3482e-01,
    1166.0: 5.2227e-01,
    1167.0: 5.2984e-01,
    1168.0: 5.3415e-01,
    1169.0: 5.1640e-01,
    1170.0: 5.2875e-01,
    1171.0: 5.3074e-01,
    1172.0: 5.3063e-01,
    1173.0: 5.2686e-01,
    1174.0: 5.2760e-01,
    1175.0: 5.1852e-01,
    1176.0: 5.1036e-01,
    1177.0: 5.2296e-01,
    1178.0: 5.1703e-01,
    1179.0: 5.1504e-01,
    1180.0: 5.2199e-01,
    1181.0: 5.1843e-01,
    1182.0: 5.1743e-01,
    1183.0: 4.8821e-01,
    1184.0: 5.1098e-01,
    1185.0: 5.1605e-01,
    1186.0: 5.1519e-01,
    1187.0: 5.1455e-01,
    1188.0: 5.1012e-01,
    1189.0: 4.8402e-01,
    1190.0: 5.0583e-01,
    1191.0: 5.1354e-01,
    1192.0: 5.1175e-01,
    1193.0: 5.1064e-01,
    1194.0: 5.1127e-01,
    1195.0: 5.0592e-01,
    1196.0: 5.0561e-01,
    1197.0: 5.0286e-01,
    1198.0: 4.9217e-01,
    1199.0: 4.7434e-01,
    1200.0: 5.0005e-01,
    1201.0: 5.0579e-01,
    1202.0: 5.0705e-01,
    1203.0: 4.8836e-01,
    1204.0: 4.8696e-01,
    1205.0: 5.0195e-01,
    1206.0: 4.9869e-01,
    1207.0: 5.0143e-01,
    1208.0: 4.8962e-01,
    1209.0: 4.6636e-01,
    1210.0: 4.9225e-01,
    1211.0: 4.7909e-01,
    1212.0: 4.9064e-01,
    1213.0: 4.9190e-01,
    1214.0: 4.8932e-01,
    1215.0: 4.9044e-01,
    1216.0: 4.8878e-01,
    1217.0: 4.8764e-01,
    1218.0: 4.8599e-01,
    1219.0: 4.8234e-01,
    1220.0: 4.8433e-01,
    1221.0: 4.8343e-01,
    1222.0: 4.8133e-01,
    1223.0: 4.7600e-01,
    1224.0: 4.8224e-01,
    1225.0: 4.7925e-01,
    1226.0: 4.8026e-01,
    1227.0: 4.6841e-01,
    1228.0: 4.7443e-01,
    1229.0: 4.7725e-01,
    1230.0: 4.7502e-01,
    1231.0: 4.7425e-01,
    1232.0: 4.6896e-01,
    1233.0: 4.7118e-01,
    1234.0: 4.6946e-01,
    1235.0: 4.6644e-01,
    1236.0: 4.6863e-01,
    1237.0: 4.6790e-01,
    1238.0: 4.6748e-01,
    1239.0: 4.6235e-01,
    1240.0: 4.5968e-01,
    1241.0: 4.6439e-01,
    1242.0: 4.6169e-01,
    1243.0: 4.5655e-01,
    1244.0: 4.5535e-01,
    1245.0: 4.5600e-01,
    1246.0: 4.5875e-01,
    1247.0: 4.5775e-01,
    1248.0: 4.6076e-01,
    1249.0: 4.6013e-01,
    1250.0: 4.5860e-01,
    1251.0: 4.5638e-01,
    1252.0: 4.5195e-01,
    1253.0: 4.5371e-01,
    1254.0: 4.5358e-01,
    1255.0: 4.5304e-01,
    1256.0: 4.5192e-01,
    1257.0: 4.4880e-01,
    1258.0: 4.5116e-01,
    1259.0: 4.4797e-01,
    1260.0: 4.5018e-01,
    1261.0: 4.4925e-01,
    1262.0: 4.4252e-01,
    1263.0: 4.4825e-01,
    1264.0: 4.4198e-01,
    1265.0: 4.4267e-01,
    1266.0: 4.4615e-01,
    1267.0: 4.3912e-01,
    1268.0: 4.3594e-01,
    1269.0: 4.4042e-01,
    1270.0: 4.4212e-01,
    1271.0: 4.4290e-01,
    1272.0: 4.4130e-01,
    1273.0: 4.4084e-01,
    1274.0: 4.3807e-01,
    1275.0: 4.4072e-01,
    1276.0: 4.4192e-01,
    1277.0: 4.4188e-01,
    1278.0: 4.4074e-01,
    1279.0: 4.3795e-01,
    1280.0: 4.3543e-01,
    1281.0: 4.2267e-01,
    1282.0: 3.7589e-01,
    1283.0: 4.1159e-01,
    1284.0: 4.2667e-01,
    1285.0: 4.2909e-01,
    1286.0: 4.2878e-01,
    1287.0: 4.3036e-01,
    1288.0: 4.2684e-01,
    1289.0: 4.2755e-01,
    1290.0: 4.1851e-01,
    1291.0: 4.2159e-01,
    1292.0: 4.2278e-01,
    1293.0: 4.2261e-01,
    1294.0: 4.1615e-01,
    1295.0: 4.1944e-01,
    1296.0: 4.1979e-01,
    1297.0: 4.1718e-01,
    1298.0: 4.1263e-01,
    1299.0: 4.1701e-01,
    1300.0: 4.1537e-01,
    1301.0: 4.1404e-01,
    1302.0: 4.0955e-01,
    1303.0: 4.0768e-01,
    1304.0: 4.0632e-01,
    1305.0: 4.1028e-01,
    1306.0: 4.1039e-01,
    1307.0: 4.0958e-01,
    1308.0: 4.0873e-01,
    1309.0: 4.0907e-01,
    1310.0: 4.0714e-01,
    1311.0: 4.0611e-01,
    1312.0: 4.0315e-01,
    1313.0: 3.9084e-01,
    1314.0: 4.0271e-01,
    1315.0: 3.9162e-01,
    1316.0: 3.9421e-01,
    1317.0: 4.0116e-01,
    1318.0: 3.9547e-01,
    1319.0: 4.0093e-01,
    1320.0: 3.9875e-01,
    1321.0: 3.9545e-01,
    1322.0: 3.9538e-01,
    1323.0: 3.9502e-01,
    1324.0: 3.9516e-01,
    1325.0: 3.9369e-01,
    1326.0: 3.9229e-01,
    1327.0: 3.8877e-01,
    1328.0: 3.9062e-01,
    1329.0: 3.7950e-01,
    1330.0: 3.8548e-01,
    1331.0: 3.8765e-01,
    1332.0: 3.7850e-01,
    1333.0: 3.8699e-01,
    1334.0: 3.8684e-01,
    1335.0: 3.8563e-01,
    1336.0: 3.8231e-01,
    1337.0: 3.8285e-01,
    1338.0: 3.8094e-01,
    1339.0: 3.7570e-01,
    1340.0: 3.7439e-01,
    1341.0: 3.8168e-01,
    1342.0: 3.8020e-01,
    1343.0: 3.7977e-01,
    1344.0: 3.7986e-01,
    1345.0: 3.7953e-01,
    1346.0: 3.7554e-01,
    1347.0: 3.7760e-01,
    1348.0: 3.7664e-01,
    1349.0: 3.7550e-01,
    1350.0: 3.7081e-01,
    1351.0: 3.7167e-01,
    1352.0: 3.7533e-01,
    1353.0: 3.7401e-01,
    1354.0: 3.7294e-01,
    1355.0: 3.6827e-01,
    1356.0: 3.6876e-01,
    1357.0: 3.6439e-01,
    1358.0: 3.6997e-01,
    1359.0: 3.6903e-01,
    1360.0: 3.6464e-01,
    1361.0: 3.7027e-01,
    1362.0: 3.6936e-01,
    1363.0: 3.6185e-01,
    1364.0: 3.6683e-01,
    1365.0: 3.6629e-01,
    1366.0: 3.6592e-01,
    1367.0: 3.6242e-01,
    1368.0: 3.6557e-01,
    1369.0: 3.5886e-01,
    1370.0: 3.5912e-01,
    1371.0: 3.5949e-01,
    1372.0: 3.6313e-01,
    1373.0: 3.6294e-01,
    1374.0: 3.6291e-01,
    1375.0: 3.5749e-01,
    1376.0: 3.5350e-01,
    1377.0: 3.5790e-01,
    1378.0: 3.5652e-01,
    1379.0: 3.6035e-01,
    1380.0: 3.5682e-01,
    1381.0: 3.5784e-01,
    1382.0: 3.5274e-01,
    1383.0: 3.5376e-01,
    1384.0: 3.5594e-01,
    1385.0: 3.5242e-01,
    1386.0: 3.5493e-01,
    1387.0: 3.5136e-01,
    1388.0: 3.5277e-01,
    1389.0: 3.5406e-01,
    1390.0: 3.4736e-01,
    1391.0: 3.5242e-01,
    1392.0: 3.5179e-01,
    1393.0: 3.5008e-01,
    1394.0: 3.4760e-01,
    1395.0: 3.4754e-01,
    1396.0: 3.4943e-01,
    1397.0: 3.4932e-01,
    1398.0: 3.4952e-01,
    1399.0: 3.4886e-01,
    1400.0: 3.3896e-01,
    1401.0: 3.3929e-01,
    1402.0: 3.4411e-01,
    1403.0: 3.4014e-01,
    1404.0: 3.4106e-01,
    1405.0: 3.4012e-01,
    1406.0: 3.4164e-01,
    1407.0: 3.4085e-01,
    1408.0: 3.4007e-01,
    1409.0: 3.4118e-01,
    1410.0: 3.4046e-01,
    1411.0: 3.3767e-01,
    1412.0: 3.2920e-01,
    1413.0: 3.3290e-01,
    1414.0: 3.3704e-01,
    1415.0: 3.3707e-01,
    1416.0: 3.3503e-01,
    1417.0: 3.3748e-01,
    1418.0: 3.3773e-01,
    1419.0: 3.3391e-01,
    1420.0: 3.3270e-01,
    1421.0: 3.3594e-01,
    1422.0: 3.2875e-01,
    1423.0: 3.2884e-01,
    1424.0: 3.2772e-01,
    1425.0: 3.3256e-01,
    1426.0: 3.2306e-01,
    1427.0: 3.2745e-01,
    1428.0: 3.3036e-01,
    1429.0: 3.2347e-01,
    1430.0: 3.2277e-01,
    1431.0: 3.2421e-01,
    1432.0: 3.3114e-01,
    1433.0: 3.3067e-01,
    1434.0: 3.2509e-01,
    1435.0: 3.2731e-01,
    1436.0: 3.2662e-01,
    1437.0: 3.2786e-01,
    1438.0: 3.2601e-01,
    1439.0: 3.2856e-01,
    1440.0: 3.1269e-01,
    1441.0: 3.1695e-01,
    1442.0: 3.1584e-01,
    1443.0: 3.1780e-01,
    1444.0: 3.1911e-01,
    1445.0: 3.1929e-01,
    1446.0: 3.2135e-01,
    1447.0: 3.1760e-01,
    1448.0: 3.2096e-01,
    1449.0: 3.2026e-01,
    1450.0: 3.1774e-01,
    1451.0: 3.1367e-01,
    1452.0: 3.1355e-01,
    1453.0: 3.1539e-01,
    1454.0: 3.1282e-01,
    1455.0: 3.1210e-01,
    1456.0: 3.0956e-01,
    1457.0: 3.1241e-01,
    1458.0: 3.1612e-01,
    1459.0: 3.1746e-01,
    1460.0: 3.1367e-01,
    1461.0: 3.1308e-01,
    1462.0: 3.1018e-01,
    1463.0: 3.0823e-01,
    1464.0: 3.0820e-01,
    1465.0: 3.1158e-01,
    1466.0: 3.0696e-01,
    1467.0: 3.1322e-01,
    1468.0: 3.0750e-01,
    1469.0: 3.1201e-01,
    1470.0: 3.1010e-01,
    1471.0: 3.0414e-01,
    1472.0: 3.0581e-01,
    1473.0: 3.0552e-01,
    1474.0: 2.9705e-01,
    1475.0: 3.0349e-01,
    1476.0: 3.0369e-01,
    1477.0: 3.0250e-01,
    1478.0: 3.0228e-01,
    1479.0: 3.0461e-01,
    1480.0: 3.0626e-01,
    1481.0: 3.0179e-01,
    1482.0: 3.0462e-01,
    1483.0: 2.9813e-01,
    1484.0: 3.0172e-01,
    1485.0: 3.0607e-01,
    1486.0: 3.0168e-01,
    1487.0: 2.9989e-01,
    1488.0: 2.6525e-01,
    1489.0: 2.9910e-01,
    1490.0: 3.0093e-01,
    1491.0: 3.0175e-01,
    1492.0: 2.9773e-01,
    1493.0: 3.0200e-01,
    1494.0: 3.0190e-01,
    1495.0: 3.0074e-01,
    1496.0: 2.9140e-01,
    1497.0: 2.9684e-01,
    1498.0: 2.9946e-01,
    1499.0: 2.9528e-01,
    1500.0: 3.0077e-01,
    1501.0: 2.9990e-01,
    1502.0: 2.9743e-01,
    1503.0: 2.6920e-01,
    1504.0: 2.8319e-01,
    1505.0: 2.7149e-01,
    1506.0: 2.8367e-01,
    1507.0: 2.9376e-01,
    1508.0: 2.9028e-01,
    1509.0: 2.8805e-01,
    1510.0: 2.9394e-01,
    1511.0: 2.9446e-01,
    1512.0: 2.8845e-01,
    1513.0: 2.8729e-01,
    1514.0: 2.8500e-01,
    1515.0: 2.8807e-01,
    1516.0: 2.8720e-01,
    1517.0: 2.8526e-01,
    1518.0: 2.8970e-01,
    1519.0: 2.8495e-01,
    1520.0: 2.8786e-01,
    1521.0: 2.8083e-01,
    1522.0: 2.8031e-01,
    1523.0: 2.8448e-01,
    1524.0: 2.8460e-01,
    1525.0: 2.7846e-01,
    1526.0: 2.8596e-01,
    1527.0: 2.8297e-01,
    1528.0: 2.8490e-01,
    1529.0: 2.8290e-01,
    1530.0: 2.6731e-01,
    1531.0: 2.8272e-01,
    1532.0: 2.8263e-01,
    1533.0: 2.8041e-01,
    1534.0: 2.7543e-01,
    1535.0: 2.7754e-01,
    1536.0: 2.8058e-01,
    1537.0: 2.8096e-01,
    1538.0: 2.7338e-01,
    1539.0: 2.7808e-01,
    1540.0: 2.6857e-01,
    1541.0: 2.7297e-01,
    1542.0: 2.7536e-01,
    1543.0: 2.7499e-01,
    1544.0: 2.7716e-01,
    1545.0: 2.7768e-01,
    1546.0: 2.7722e-01,
    1547.0: 2.7354e-01,
    1548.0: 2.6698e-01,
    1549.0: 2.7267e-01,
    1550.0: 2.6961e-01,
    1551.0: 2.7075e-01,
    1552.0: 2.7400e-01,
    1553.0: 2.7090e-01,
    1554.0: 2.6495e-01,
    1555.0: 2.6712e-01,
    1556.0: 2.6292e-01,
    1557.0: 2.7013e-01,
    1558.0: 2.6805e-01,
    1559.0: 2.6780e-01,
    1560.0: 2.6546e-01,
    1561.0: 2.6968e-01,
    1562.0: 2.6833e-01,
    1563.0: 2.6688e-01,
    1564.0: 2.6307e-01,
    1565.0: 2.6682e-01,
    1566.0: 2.6262e-01,
    1567.0: 2.6352e-01,
    1568.0: 2.6042e-01,
    1569.0: 2.6275e-01,
    1570.0: 2.6068e-01,
    1571.0: 2.6668e-01,
    1572.0: 2.6694e-01,
    1573.0: 2.6344e-01,
    1574.0: 2.6221e-01,
    1575.0: 2.4738e-01,
    1576.0: 2.6031e-01,
    1577.0: 2.3690e-01,
    1578.0: 2.5723e-01,
    1579.0: 2.6167e-01,
    1580.0: 2.6381e-01,
    1581.0: 2.6483e-01,
    1582.0: 2.5412e-01,
    1583.0: 2.5771e-01,
    1584.0: 2.5449e-01,
    1585.0: 2.6198e-01,
    1586.0: 2.5817e-01,
    1587.0: 2.5492e-01,
    1588.0: 2.5048e-01,
    1589.0: 2.3152e-01,
    1590.0: 2.4154e-01,
    1591.0: 2.4132e-01,
    1592.0: 2.5286e-01,
    1593.0: 2.5751e-01,
    1594.0: 2.5643e-01,
    1595.0: 2.5765e-01,
    1596.0: 2.4452e-01,
    1597.0: 2.4854e-01,
    1598.0: 2.5629e-01,
    1599.0: 2.4977e-01,
    1600.0: 2.5259e-01,
    1601.0: 2.4494e-01,
    1602.0: 2.4857e-01,
    1603.0: 2.4901e-01,
    1604.0: 2.5002e-01,
    1605.0: 2.4703e-01,
    1606.0: 2.4748e-01,
    1607.0: 2.4544e-01,
    1608.0: 2.4999e-01,
    1609.0: 2.4858e-01,
    1610.0: 2.3312e-01,
    1611.0: 2.4078e-01,
    1612.0: 2.4125e-01,
    1613.0: 2.4581e-01,
    1614.0: 2.4350e-01,
    1615.0: 2.4434e-01,
    1616.0: 2.3248e-01,
    1617.0: 2.3644e-01,
    1618.0: 2.4415e-01,
    1619.0: 2.4224e-01,
    1620.0: 2.3407e-01,
    1621.0: 2.3386e-01,
    1622.0: 2.3702e-01,
    1623.0: 2.4166e-01,
    1624.0: 2.4240e-01,
    1625.0: 2.3746e-01,
    1626.0: 2.3958e-01,
    1627.0: 2.4037e-01,
    1628.0: 2.4149e-01,
    1629.0: 2.4111e-01,
    1630.0: 2.3655e-01,
    1631.0: 2.3932e-01,
    1632.0: 2.3884e-01,
    1633.0: 2.3585e-01,
    1634.0: 2.3276e-01,
    1635.0: 2.3735e-01,
    1636.0: 2.3676e-01,
    1637.0: 2.2747e-01,
    1638.0: 2.2708e-01,
    1639.0: 2.2205e-01,
    1640.0: 2.2571e-01,
    1641.0: 2.2393e-01,
    1642.0: 2.2402e-01,
    1643.0: 2.2879e-01,
    1644.0: 2.2572e-01,
    1645.0: 2.2512e-01,
    1646.0: 2.2899e-01,
    1647.0: 2.2910e-01,
    1648.0: 2.2898e-01,
    1649.0: 2.2354e-01,
    1650.0: 2.2840e-01,
    1651.0: 2.2369e-01,
    1652.0: 2.2802e-01,
    1653.0: 2.2533e-01,
    1654.0: 2.2654e-01,
    1655.0: 2.2390e-01,
    1656.0: 2.2607e-01,
    1657.0: 2.2564e-01,
    1658.0: 2.2532e-01,
    1659.0: 2.2221e-01,
    1660.0: 2.2412e-01,
    1661.0: 2.2543e-01,
    1662.0: 2.2421e-01,
    1663.0: 2.2265e-01,
    1664.0: 2.2252e-01,
    1665.0: 2.2137e-01,
    1666.0: 2.1853e-01,
    1667.0: 2.2016e-01,
    1668.0: 2.1617e-01,
    1669.0: 2.1809e-01,
    1670.0: 2.2208e-01,
    1671.0: 2.2096e-01,
    1672.0: 2.1210e-01,
    1673.0: 2.1606e-01,
    1674.0: 2.1907e-01,
    1675.0: 2.1621e-01,
    1676.0: 2.1319e-01,
    1677.0: 2.1548e-01,
    1678.0: 2.1585e-01,
    1679.0: 2.1295e-01,
    1680.0: 2.0641e-01,
    1681.0: 1.9848e-01,
    1682.0: 2.0341e-01,
    1683.0: 2.0893e-01,
    1684.0: 2.1198e-01,
    1685.0: 2.1306e-01,
    1686.0: 2.1026e-01,
    1687.0: 2.1123e-01,
    1688.0: 2.1081e-01,
    1689.0: 2.0805e-01,
    1690.0: 2.0845e-01,
    1691.0: 2.1203e-01,
    1692.0: 2.1211e-01,
    1693.0: 2.1122e-01,
    1694.0: 2.1068e-01,
    1695.0: 2.1055e-01,
    1696.0: 2.0962e-01,
    1697.0: 2.0850e-01,
    1698.0: 2.0879e-01,
    1699.0: 2.0804e-01,
    1700.0: 2.0539e-01,
    1702.0: 2.0520e-01,
    1705.0: 2.0428e-01,
    1710.0: 1.9894e-01,
    1715.0: 2.0189e-01,
    1720.0: 1.9799e-01,
    1725.0: 1.9688e-01,
    1730.0: 1.9309e-01,
    1735.0: 1.8200e-01,
    1740.0: 1.9001e-01,
    1745.0: 1.8624e-01,
    1750.0: 1.8518e-01,
    1755.0: 1.8509e-01,
    1760.0: 1.8243e-01,
    1765.0: 1.8143e-01,
    1770.0: 1.7969e-01,
    1775.0: 1.7600e-01,
    1780.0: 1.7553e-01,
    1785.0: 1.7411e-01,
    1790.0: 1.7371e-01,
    1795.0: 1.7102e-01,
    1800.0: 1.6800e-01,
    1805.0: 1.6884e-01,
    1810.0: 1.6859e-01,
    1815.0: 1.5500e-01,
    1820.0: 1.6000e-01,
    1825.0: 1.6257e-01,
    1830.0: 1.5921e-01,
    1835.0: 1.5760e-01,
    1840.0: 1.5552e-01,
    1845.0: 1.5288e-01,
    1850.0: 1.5337e-01,
    1855.0: 1.5126e-01,
    1860.0: 1.4933e-01,
    1865.0: 1.4710e-01,
    1870.0: 1.4565e-01,
    1875.0: 1.3231e-01,
    1880.0: 1.4667e-01,
    1885.0: 1.4573e-01,
    1890.0: 1.4041e-01,
    1895.0: 1.3842e-01,
    1900.0: 1.4041e-01,
    1905.0: 1.3904e-01,
    1910.0: 1.3654e-01,
    1915.0: 1.3611e-01,
    1920.0: 1.3463e-01,
    1925.0: 1.3447e-01,
    1930.0: 1.3145e-01,
    1935.0: 1.3243e-01,
    1940.0: 1.2950e-01,
    1945.0: 1.1950e-01,
    1950.0: 1.2627e-01,
    1955.0: 1.2833e-01,
    1960.0: 1.2610e-01,
    1965.0: 1.2291e-01,
    1970.0: 1.2375e-01,
    1975.0: 1.2150e-01,
    1980.0: 1.1968e-01,
    1985.0: 1.1902e-01,
    1990.0: 1.1977e-01,
    1995.0: 1.1684e-01,
    2000.0: 1.1673e-01,
    2005.0: 1.1501e-01,
    2010.0: 1.1512e-01,
    2015.0: 1.1399e-01,
    2020.0: 1.1192e-01,
    2025.0: 1.1176e-01,
    2030.0: 1.0969e-01,
    2035.0: 1.0915e-01,
    2040.0: 1.0720e-01,
    2045.0: 1.0756e-01,
    2050.0: 1.0592e-01,
    2055.0: 1.0471e-01,
    2060.0: 1.0320e-01,
    2065.0: 1.0182e-01,
    2070.0: 1.0095e-01,
    2075.0: 1.0095e-01,
    2080.0: 9.9330e-02,
    2085.0: 9.8300e-02,
    2090.0: 9.7540e-02,
    2095.0: 9.5990e-02,
    2100.0: 9.6240e-02,
    2105.0: 9.5750e-02,
    2110.0: 9.4630e-02,
    2115.0: 9.3850e-02,
    2120.0: 9.3140e-02,
    2125.0: 9.2180e-02,
    2130.0: 9.2380e-02,
    2135.0: 9.1120e-02,
    2140.0: 9.1050e-02,
    2145.0: 8.9940e-02,
    2150.0: 8.9710e-02,
    2155.0: 8.8880e-02,
    2160.0: 8.7890e-02,
    2165.0: 8.2000e-02,
    2170.0: 8.5370e-02,
    2175.0: 8.5760e-02,
    2180.0: 8.4640e-02,
    2185.0: 8.4680e-02,
    2190.0: 8.3140e-02,
    2195.0: 8.3470e-02,
    2200.0: 8.2790e-02,
    2205.0: 8.0900e-02,
    2210.0: 8.0810e-02,
    2215.0: 8.0410e-02,
    2220.0: 7.9990e-02,
    2225.0: 7.8840e-02,
    2230.0: 7.8400e-02,
    2235.0: 7.7930e-02,
    2240.0: 7.6510e-02,
    2245.0: 7.6250e-02,
    2250.0: 7.5370e-02,
    2255.0: 7.4340e-02,
    2260.0: 7.4090e-02,
    2265.0: 7.3280e-02,
    2270.0: 7.3100e-02,
    2275.0: 7.2630e-02,
    2280.0: 7.1390e-02,
    2285.0: 7.1380e-02,
    2290.0: 7.1190e-02,
    2295.0: 6.9300e-02,
    2300.0: 6.9640e-02,
    2305.0: 6.9400e-02,
    2310.0: 6.8900e-02,
    2315.0: 6.8150e-02,
    2320.0: 6.7630e-02,
    2325.0: 6.5560e-02,
    2330.0: 6.6220e-02,
    2335.0: 6.5720e-02,
    2340.0: 6.5220e-02,
    2345.0: 6.5100e-02,
    2350.0: 6.4340e-02,
    2355.0: 6.2600e-02,
    2360.0: 6.3070e-02,
    2365.0: 6.3010e-02,
    2370.0: 6.2390e-02,
    2375.0: 6.1410e-02,
    2380.0: 6.1810e-02,
    2385.0: 5.9130e-02,
    2390.0: 6.0360e-02,
    2395.0: 6.0070e-02,
    2400.0: 5.9740e-02,
    2405.0: 5.9440e-02,
    2410.0: 5.9150e-02,
    2415.0: 5.6980e-02,
    2420.0: 5.7820e-02,
    2425.0: 5.7330e-02,
    2430.0: 5.7190e-02,
    2435.0: 5.6470e-02,
    2440.0: 5.6270e-02,
    2445.0: 5.5560e-02,
    2450.0: 5.4590e-02,
    2455.0: 5.4160e-02,
    2460.0: 5.4510e-02,
    2465.0: 5.4300e-02,
    2470.0: 5.3400e-02,
    2475.0: 5.3370e-02,
    2480.0: 5.2100e-02,
    2485.0: 5.1770e-02,
    2490.0: 5.2220e-02,
    2495.0: 4.8890e-02,
    2500.0: 5.1380e-02,
    2505.0: 5.0820e-02,
    2510.0: 4.9600e-02,
    2515.0: 4.8760e-02,
    2520.0: 4.8140e-02,
    2525.0: 4.7650e-02,
    2530.0: 4.7700e-02,
    2535.0: 4.7120e-02,
    2540.0: 4.6770e-02,
    2545.0: 4.6610e-02,
    2550.0: 4.6260e-02,
    2555.0: 4.5930e-02,
    2560.0: 4.5400e-02,
    2565.0: 4.5160e-02,
    2570.0: 4.4850e-02,
    2575.0: 4.4480e-02,
    2580.0: 4.4100e-02,
    2585.0: 4.3270e-02,
    2590.0: 4.3410e-02,
    2595.0: 4.3040e-02,
    2600.0: 4.2910e-02,
    2605.0: 4.2800e-02,
    2610.0: 4.2500e-02,
    2615.0: 4.2280e-02,
    2620.0: 4.1900e-02,
    2625.0: 4.0580e-02,
    2630.0: 4.1160e-02,
    2635.0: 4.0960e-02,
    2640.0: 4.0220e-02,
    2645.0: 4.0180e-02,
    2650.0: 3.9960e-02,
    2655.0: 3.9590e-02,
    2660.0: 3.9480e-02,
    2665.0: 3.8950e-02,
    2670.0: 3.9100e-02,
    2675.0: 3.8610e-02,
    2680.0: 3.8430e-02,
    2685.0: 3.8160e-02,
    2690.0: 3.7790e-02,
    2695.0: 3.7460e-02,
    2700.0: 3.7120e-02,
    2705.0: 3.6930e-02,
    2710.0: 3.6680e-02,
    2715.0: 3.6560e-02,
    2720.0: 3.6380e-02,
    2725.0: 3.6130e-02,
    2730.0: 3.6010e-02,
    2735.0: 3.5740e-02,
    2740.0: 3.5530e-02,
    2745.0: 3.5380e-02,
    2750.0: 3.5180e-02,
    2755.0: 3.4860e-02,
    2760.0: 3.4550e-02,
    2765.0: 3.4400e-02,
    2770.0: 3.4260e-02,
    2775.0: 3.3930e-02,
    2780.0: 3.3640e-02,
    2785.0: 3.3420e-02,
    2790.0: 3.3200e-02,
    2795.0: 3.2930e-02,
    2800.0: 3.2760e-02,
    2805.0: 3.2530e-02,
    2810.0: 3.2420e-02,
    2815.0: 3.2260e-02,
    2820.0: 3.2060e-02,
    2825.0: 3.1840e-02,
    2830.0: 3.1700e-02,
    2835.0: 3.1450e-02,
    2840.0: 3.1180e-02,
    2845.0: 3.0950e-02,
    2850.0: 3.0680e-02,
    2855.0: 3.0430e-02,
    2860.0: 3.0160e-02,
    2865.0: 2.9960e-02,
    2870.0: 2.9600e-02,
    2875.0: 2.9400e-02,
    2880.0: 2.9400e-02,
    2885.0: 2.9220e-02,
    2890.0: 2.9070e-02,
    2895.0: 2.8960e-02,
    2900.0: 2.8810e-02,
    2905.0: 2.8670e-02,
    2910.0: 2.8500e-02,
    2915.0: 2.8220e-02,
    2920.0: 2.8040e-02,
    2925.0: 2.7780e-02,
    2930.0: 2.7570e-02,
    2935.0: 2.7330e-02,
    2940.0: 2.7140e-02,
    2945.0: 2.6840e-02,
    2950.0: 2.6680e-02,
    2955.0: 2.6560e-02,
    2960.0: 2.6380e-02,
    2965.0: 2.6260e-02,
    2970.0: 2.6120e-02,
    2975.0: 2.6010e-02,
    2980.0: 2.5800e-02,
    2985.0: 2.5690e-02,
    2990.0: 2.5540e-02,
    2995.0: 2.5400e-02,
    3000.0: 2.5250e-02,
    3005.0: 2.5120e-02,
    3010.0: 2.4980e-02,
    3015.0: 2.4860e-02,
    3020.0: 2.4660e-02,
    3025.0: 2.4540e-02,
    3030.0: 2.4420e-02,
    3035.0: 2.4210e-02,
    3040.0: 2.3930e-02,
    3045.0: 2.4020e-02,
    3050.0: 2.3950e-02,
    3055.0: 2.3860e-02,
    3060.0: 2.3760e-02,
    3065.0: 2.3540e-02,
    3070.0: 2.3440e-02,
    3075.0: 2.3350e-02,
    3080.0: 2.3150e-02,
    3085.0: 2.2950e-02,
    3090.0: 2.2800e-02,
    3095.0: 2.2630e-02,
    3100.0: 2.2430e-02,
    3105.0: 2.2280e-02,
    3110.0: 2.2110e-02,
    3115.0: 2.1770e-02,
    3120.0: 2.1740e-02,
    3125.0: 2.1580e-02,
    3130.0: 2.1450e-02,
    3135.0: 2.1360e-02,
    3140.0: 2.1180e-02,
    3145.0: 2.1070e-02,
    3150.0: 2.0880e-02,
    3155.0: 2.0830e-02,
    3160.0: 2.0730e-02,
    3165.0: 2.0660e-02,
    3170.0: 2.0520e-02,
    3175.0: 2.0470e-02,
    3180.0: 2.0320e-02,
    3185.0: 2.0240e-02,
    3190.0: 2.0190e-02,
    3195.0: 2.0080e-02,
    3200.0: 1.9940e-02,
    3205.0: 1.9860e-02,
    3210.0: 1.9760e-02,
    3215.0: 1.9620e-02,
    3220.0: 1.9540e-02,
    3225.0: 1.9420e-02,
    3230.0: 1.9380e-02,
    3235.0: 1.9260e-02,
    3240.0: 1.9210e-02,
    3245.0: 1.9100e-02,
    3250.0: 1.9090e-02,
    3255.0: 1.8900e-02,
    3260.0: 1.8800e-02,
    3265.0: 1.8700e-02,
    3270.0: 1.8600e-02,
    3275.0: 1.8480e-02,
    3280.0: 1.8370e-02,
    3285.0: 1.8320e-02,
    3290.0: 1.8260e-02,
    3295.0: 1.8030e-02,
    3300.0: 1.7920e-02,
    3305.0: 1.7860e-02,
    3310.0: 1.7700e-02,
    3315.0: 1.7600e-02,
    3320.0: 1.7300e-02,
    3325.0: 1.7280e-02,
    3330.0: 1.7130e-02,
    3335.0: 1.7030e-02,
    3340.0: 1.6850e-02,
    3345.0: 1.6780e-02,
    3350.0: 1.6600e-02,
    3355.0: 1.6500e-02,
    3360.0: 1.6360e-02,
    3365.0: 1.6290e-02,
    3370.0: 1.6180e-02,
    3375.0: 1.6110e-02,
    3380.0: 1.6010e-02,
    3385.0: 1.5980e-02,
    3390.0: 1.5900e-02,
    3395.0: 1.5800e-02,
    3400.0: 1.5640e-02,
    3405.0: 1.5650e-02,
    3410.0: 1.5540e-02,
    3415.0: 1.5470e-02,
    3420.0: 1.5430e-02,
    3425.0: 1.5320e-02,
    3430.0: 1.5220e-02,
    3435.0: 1.5150e-02,
    3440.0: 1.5050e-02,
    3445.0: 1.5020e-02,
    3450.0: 1.4920e-02,
    3455.0: 1.4880e-02,
    3460.0: 1.4800e-02,
    3465.0: 1.4760e-02,
    3470.0: 1.4660e-02,
    3475.0: 1.4560e-02,
    3480.0: 1.4480e-02,
    3485.0: 1.4450e-02,
    3490.0: 1.4360e-02,
    3495.0: 1.4320e-02,
    3500.0: 1.4230e-02,
    3505.0: 1.4150e-02,
    3510.0: 1.4050e-02,
    3515.0: 1.4020e-02,
    3520.0: 1.3930e-02,
    3525.0: 1.3890e-02,
    3530.0: 1.3800e-02,
    3535.0: 1.3760e-02,
    3540.0: 1.3670e-02,
    3545.0: 1.3660e-02,
    3550.0: 1.3570e-02,
    3555.0: 1.3450e-02,
    3560.0: 1.3360e-02,
    3565.0: 1.3360e-02,
    3570.0: 1.3250e-02,
    3575.0: 1.3180e-02,
    3580.0: 1.3080e-02,
    3585.0: 1.3050e-02,
    3590.0: 1.2980e-02,
    3595.0: 1.2940e-02,
    3600.0: 1.2850e-02,
    3605.0: 1.2830e-02,
    3610.0: 1.2740e-02,
    3615.0: 1.2720e-02,
    3620.0: 1.2640e-02,
    3625.0: 1.2610e-02,
    3630.0: 1.2510e-02,
    3635.0: 1.2490e-02,
    3640.0: 1.2400e-02,
    3645.0: 1.2360e-02,
    3650.0: 1.2280e-02,
    3655.0: 1.2220e-02,
    3660.0: 1.2150e-02,
    3665.0: 1.2150e-02,
    3670.0: 1.2040e-02,
    3675.0: 1.2030e-02,
    3680.0: 1.1890e-02,
    3685.0: 1.1830e-02,
    3690.0: 1.1770e-02,
    3695.0: 1.1720e-02,
    3700.0: 1.1550e-02,
    3705.0: 1.1600e-02,
    3710.0: 1.1510e-02,
    3715.0: 1.1500e-02,
    3720.0: 1.1440e-02,
    3725.0: 1.1430e-02,
    3730.0: 1.1340e-02,
    3735.0: 1.1280e-02,
    3740.0: 1.1080e-02,
    3745.0: 1.1080e-02,
    3750.0: 1.1040e-02,
    3755.0: 1.1000e-02,
    3760.0: 1.0940e-02,
    3765.0: 1.0930e-02,
    3770.0: 1.0860e-02,
    3775.0: 1.0800e-02,
    3780.0: 1.0740e-02,
    3785.0: 1.0730e-02,
    3790.0: 1.0630e-02,
    3795.0: 1.0620e-02,
    3800.0: 1.0530e-02,
    3805.0: 1.0530e-02,
    3810.0: 1.0430e-02,
    3815.0: 1.0410e-02,
    3820.0: 1.0340e-02,
    3825.0: 1.0330e-02,
    3830.0: 1.0250e-02,
    3835.0: 1.0210e-02,
    3840.0: 1.0150e-02,
    3845.0: 1.0130e-02,
    3850.0: 1.0050e-02,
    3855.0: 1.0030e-02,
    3860.0: 9.9400e-03,
    3865.0: 9.9100e-03,
    3870.0: 9.8300e-03,
    3875.0: 9.8300e-03,
    3880.0: 9.7500e-03,
    3885.0: 9.7000e-03,
    3890.0: 9.7100e-03,
    3895.0: 9.6200e-03,
    3900.0: 9.6000e-03,
    3905.0: 9.5900e-03,
    3910.0: 9.5200e-03,
    3915.0: 9.5000e-03,
    3920.0: 9.4100e-03,
    3925.0: 9.4000e-03,
    3930.0: 9.3200e-03,
    3935.0: 9.3000e-03,
    3940.0: 9.2300e-03,
    3945.0: 9.2000e-03,
    3950.0: 9.1100e-03,
    3955.0: 9.0800e-03,
    3960.0: 9.0200e-03,
    3965.0: 9.0100e-03,
    3970.0: 8.9300e-03,
    3975.0: 8.9100e-03,
    3980.0: 8.8400e-03,
    3985.0: 8.8000e-03,
    3990.0: 8.7800e-03,
    3995.0: 8.7000e-03,
    4000.0: 8.6800e-03,
}

ASTMG173_GLOBAL_TILT_DATA = {
    280.0: 4.7309e-23,
    280.5: 1.2307e-21,
    281.0: 5.6895e-21,
    281.5: 1.5662e-19,
    282.0: 1.1946e-18,
    282.5: 4.5436e-18,
    283.0: 1.8452e-17,
    283.5: 3.5360e-17,
    284.0: 7.2670e-16,
    284.5: 2.4856e-15,
    285.0: 8.0142e-15,
    285.5: 4.2613e-14,
    286.0: 1.3684e-13,
    286.5: 8.3823e-13,
    287.0: 2.7367e-12,
    287.5: 1.0903e-11,
    288.0: 6.2337e-11,
    288.5: 1.7162e-10,
    289.0: 5.6265e-10,
    289.5: 2.0749e-09,
    290.0: 6.0168e-09,
    290.5: 1.3783e-08,
    291.0: 3.5052e-08,
    291.5: 1.0913e-07,
    292.0: 2.6830e-07,
    292.5: 4.2685e-07,
    293.0: 8.6466e-07,
    293.5: 2.2707e-06,
    294.0: 4.1744e-06,
    294.5: 6.5911e-06,
    295.0: 1.2290e-05,
    295.5: 2.7826e-05,
    296.0: 4.7904e-05,
    296.5: 7.1345e-05,
    297.0: 9.6800e-05,
    297.5: 1.8608e-04,
    298.0: 2.8988e-04,
    298.5: 3.5789e-04,
    299.0: 4.9211e-04,
    299.5: 8.6068e-04,
    300.0: 1.0205e-03,
    300.5: 1.2450e-03,
    301.0: 1.9300e-03,
    301.5: 2.6914e-03,
    302.0: 2.9209e-03,
    302.5: 4.2840e-03,
    303.0: 7.0945e-03,
    303.5: 8.9795e-03,
    304.0: 9.4701e-03,
    304.5: 1.1953e-02,
    305.0: 1.6463e-02,
    305.5: 1.8719e-02,
    306.0: 1.8577e-02,
    306.5: 2.1108e-02,
    307.0: 2.7849e-02,
    307.5: 3.5635e-02,
    308.0: 3.7837e-02,
    308.5: 4.1430e-02,
    309.0: 4.0534e-02,
    309.5: 4.3306e-02,
    310.0: 5.0939e-02,
    310.5: 6.5540e-02,
    311.0: 8.2922e-02,
    311.5: 8.4080e-02,
    312.0: 9.3376e-02,
    312.5: 9.8984e-02,
    313.0: 1.0733e-01,
    313.5: 1.0757e-01,
    314.0: 1.1969e-01,
    314.5: 1.3060e-01,
    315.0: 1.3625e-01,
    315.5: 1.1838e-01,
    316.0: 1.2348e-01,
    316.5: 1.5036e-01,
    317.0: 1.7158e-01,
    317.5: 1.8245e-01,
    318.0: 1.7594e-01,
    318.5: 1.8591e-01,
    319.0: 2.0470e-01,
    319.5: 1.9589e-01,
    320.0: 2.0527e-01,
    320.5: 2.4525e-01,
    321.0: 2.5024e-01,
    321.5: 2.3843e-01,
    322.0: 2.2203e-01,
    322.5: 2.1709e-01,
    323.0: 2.1226e-01,
    323.5: 2.4861e-01,
    324.0: 2.7537e-01,
    324.5: 2.8321e-01,
    325.0: 2.7894e-01,
    325.5: 3.2436e-01,
    326.0: 3.8120e-01,
    326.5: 4.0722e-01,
    327.0: 3.9806e-01,
    327.5: 3.8465e-01,
    328.0: 3.5116e-01,
    328.5: 3.7164e-01,
    329.0: 4.2235e-01,
    329.5: 4.6878e-01,
    330.0: 4.7139e-01,
    330.5: 4.2800e-01,
    331.0: 4.0262e-01,
    331.5: 4.1806e-01,
    332.0: 4.3623e-01,
    332.5: 4.3919e-01,
    333.0: 4.2944e-01,
    333.5: 4.0724e-01,
    334.0: 4.1497e-01,
    334.5: 4.4509e-01,
    335.0: 4.6388e-01,
    335.5: 4.5313e-01,
    336.0: 4.1519e-01,
    336.5: 3.8214e-01,
    337.0: 3.7380e-01,
    337.5: 4.0051e-01,
    338.0: 4.3411e-01,
    338.5: 4.5527e-01,
    339.0: 4.6355e-01,
    339.5: 4.7446e-01,
    340.0: 5.0180e-01,
    340.5: 5.0071e-01,
    341.0: 4.7139e-01,
    341.5: 4.6935e-01,
    342.0: 4.8934e-01,
    342.5: 5.0767e-01,
    343.0: 5.1489e-01,
    343.5: 4.8609e-01,
    344.0: 4.1843e-01,
    344.5: 4.0307e-01,
    345.0: 4.5898e-01,
    345.5: 4.8932e-01,
    346.0: 4.7778e-01,
    346.5: 4.8657e-01,
    347.0: 4.9404e-01,
    347.5: 4.7674e-01,
    348.0: 4.7511e-01,
    348.5: 4.8336e-01,
    349.0: 4.6564e-01,
    349.5: 4.7805e-01,
    350.0: 5.2798e-01,
    350.5: 5.6741e-01,
    351.0: 5.5172e-01,
    351.5: 5.3022e-01,
    352.0: 5.1791e-01,
    352.5: 4.8962e-01,
    353.0: 5.2040e-01,
    353.5: 5.7228e-01,
    354.0: 6.0498e-01,
    354.5: 6.1156e-01,
    355.0: 6.1140e-01,
    355.5: 5.9028e-01,
    356.0: 5.5387e-01,
    356.5: 5.1942e-01,
    357.0: 4.5673e-01,
    357.5: 4.6215e-01,
    358.0: 4.3006e-01,
    358.5: 3.9926e-01,
    359.0: 4.6953e-01,
    359.5: 5.6549e-01,
    360.0: 5.9817e-01,
    360.5: 5.6531e-01,
    361.0: 5.2024e-01,
    361.5: 5.0956e-01,
    362.0: 5.3420e-01,
    362.5: 5.8510e-01,
    363.0: 6.0191e-01,
    363.5: 5.8541e-01,
    364.0: 6.0628e-01,
    364.5: 6.0058e-01,
    365.0: 6.2359e-01,
    365.5: 6.8628e-01,
    366.0: 7.3532e-01,
    366.5: 7.3658e-01,
    367.0: 7.2285e-01,
    367.5: 7.0914e-01,
    368.0: 6.6759e-01,
    368.5: 6.6310e-01,
    369.0: 6.9315e-01,
    369.5: 7.4469e-01,
    370.0: 7.5507e-01,
    370.5: 6.8261e-01,
    371.0: 6.9338e-01,
    371.5: 7.2051e-01,
    372.0: 6.7444e-01,
    372.5: 6.4253e-01,
    373.0: 6.1886e-01,
    373.5: 5.5786e-01,
    374.0: 5.5640e-01,
    374.5: 5.5227e-01,
    375.0: 5.8930e-01,
    375.5: 6.5162e-01,
    376.0: 6.7480e-01,
    376.5: 6.6390e-01,
    377.0: 7.1225e-01,
    377.5: 7.9455e-01,
    378.0: 8.5595e-01,
    378.5: 8.3418e-01,
    379.0: 7.4389e-01,
    379.5: 6.6683e-01,
    380.0: 7.0077e-01,
    380.5: 7.5075e-01,
    381.0: 7.6383e-01,
    381.5: 6.8837e-01,
    382.0: 5.8678e-01,
    382.5: 5.0762e-01,
    383.0: 4.5499e-01,
    383.5: 4.4049e-01,
    384.0: 5.0968e-01,
    384.5: 6.1359e-01,
    385.0: 6.7355e-01,
    385.5: 6.4363e-01,
    386.0: 6.2100e-01,
    386.5: 6.4570e-01,
    387.0: 6.5147e-01,
    387.5: 6.4204e-01,
    388.0: 6.3582e-01,
    388.5: 6.3136e-01,
    389.0: 6.8543e-01,
    389.5: 7.5970e-01,
    390.0: 7.9699e-01,
    390.5: 8.0371e-01,
    391.0: 8.5138e-01,
    391.5: 8.6344e-01,
    392.0: 7.9493e-01,
    392.5: 6.6257e-01,
    393.0: 4.7975e-01,
    393.5: 3.8152e-01,
    394.0: 4.9567e-01,
    394.5: 6.8385e-01,
    395.0: 8.0772e-01,
    395.5: 8.6038e-01,
    396.0: 7.5655e-01,
    396.5: 5.5017e-01,
    397.0: 4.2619e-01,
    397.5: 6.2945e-01,
    398.0: 8.5249e-01,
    398.5: 1.0069e+00,
    399.0: 1.0693e+00,
    399.5: 1.1021e+00,
    400.0: 1.1141e+00,
    401.0: 1.1603e+00,
    402.0: 1.2061e+00,
    403.0: 1.1613e+00,
    404.0: 1.1801e+00,
    405.0: 1.1511e+00,
    406.0: 1.1227e+00,
    407.0: 1.1026e+00,
    408.0: 1.1514e+00,
    409.0: 1.2299e+00,
    410.0: 1.0485e+00,
    411.0: 1.1738e+00,
    412.0: 1.2478e+00,
    413.0: 1.1971e+00,
    414.0: 1.1842e+00,
    415.0: 1.2258e+00,
    416.0: 1.2624e+00,
    417.0: 1.2312e+00,
    418.0: 1.1777e+00,
    419.0: 1.2258e+00,
    420.0: 1.1232e+00,
    421.0: 1.2757e+00,
    422.0: 1.2583e+00,
    423.0: 1.2184e+00,
    424.0: 1.2117e+00,
    425.0: 1.2488e+00,
    426.0: 1.2135e+00,
    427.0: 1.1724e+00,
    428.0: 1.1839e+00,
    429.0: 1.0963e+00,
    430.0: 8.7462e-01,
    431.0: 7.9394e-01,
    432.0: 1.3207e+00,
    433.0: 1.2288e+00,
    434.0: 1.1352e+00,
    435.0: 1.2452e+00,
    436.0: 1.3659e+00,
    437.0: 1.3943e+00,
    438.0: 1.2238e+00,
    439.0: 1.1775e+00,
    440.0: 1.3499e+00,
    441.0: 1.3313e+00,
    442.0: 1.4250e+00,
    443.0: 1.4453e+00,
    444.0: 1.4084e+00,
    445.0: 1.4619e+00,
    446.0: 1.3108e+00,
    447.0: 1.4903e+00,
    448.0: 1.5081e+00,
    449.0: 1.5045e+00,
    450.0: 1.5595e+00,
    451.0: 1.6173e+00,
    452.0: 1.5482e+00,
    453.0: 1.4297e+00,
    454.0: 1.5335e+00,
    455.0: 1.5224e+00,
    456.0: 1.5724e+00,
    457.0: 1.5854e+00,
    458.0: 1.5514e+00,
    459.0: 1.5391e+00,
    460.0: 1.5291e+00,
    461.0: 1.5827e+00,
    462.0: 1.5975e+00,
    463.0: 1.6031e+00,
    464.0: 1.5544e+00,
    465.0: 1.5350e+00,
    466.0: 1.5673e+00,
    467.0: 1.4973e+00,
    468.0: 1.5619e+00,
    469.0: 1.5682e+00,
    470.0: 1.5077e+00,
    471.0: 1.5331e+00,
    472.0: 1.6126e+00,
    473.0: 1.5499e+00,
    474.0: 1.5671e+00,
    475.0: 1.6185e+00,
    476.0: 1.5631e+00,
    477.0: 1.5724e+00,
    478.0: 1.6230e+00,
    479.0: 1.5916e+00,
    480.0: 1.6181e+00,
    481.0: 1.6177e+00,
    482.0: 1.6236e+00,
    483.0: 1.6038e+00,
    484.0: 1.5734e+00,
    485.0: 1.5683e+00,
    486.0: 1.2716e+00,
    487.0: 1.4241e+00,
    488.0: 1.5413e+00,
    489.0: 1.4519e+00,
    490.0: 1.6224e+00,
    491.0: 1.5595e+00,
    492.0: 1.4869e+00,
    493.0: 1.5903e+00,
    494.0: 1.5525e+00,
    495.0: 1.6485e+00,
    496.0: 1.5676e+00,
    497.0: 1.5944e+00,
    498.0: 1.5509e+00,
    499.0: 1.5507e+00,
    500.0: 1.5451e+00,
    501.0: 1.4978e+00,
    502.0: 1.4966e+00,
    503.0: 1.5653e+00,
    504.0: 1.4587e+00,
    505.0: 1.5635e+00,
    506.0: 1.6264e+00,
    507.0: 1.5560e+00,
    508.0: 1.5165e+00,
    509.0: 1.5893e+00,
    510.0: 1.5481e+00,
    511.0: 1.5769e+00,
    512.0: 1.6186e+00,
    513.0: 1.5206e+00,
    514.0: 1.4885e+00,
    515.0: 1.5314e+00,
    516.0: 1.5455e+00,
    517.0: 1.2594e+00,
    518.0: 1.4403e+00,
    519.0: 1.3957e+00,
    520.0: 1.5236e+00,
    521.0: 1.5346e+00,
    522.0: 1.5690e+00,
    523.0: 1.4789e+00,
    524.0: 1.5905e+00,
    525.0: 1.5781e+00,
    526.0: 1.5341e+00,
    527.0: 1.3417e+00,
    528.0: 1.5357e+00,
    529.0: 1.6071e+00,
    530.0: 1.5446e+00,
    531.0: 1.6292e+00,
    532.0: 1.5998e+00,
    533.0: 1.4286e+00,
    534.0: 1.5302e+00,
    535.0: 1.5535e+00,
    536.0: 1.6199e+00,
    537.0: 1.4989e+00,
    538.0: 1.5738e+00,
    539.0: 1.5352e+00,
    540.0: 1.4825e+00,
    541.0: 1.4251e+00,
    542.0: 1.5511e+00,
    543.0: 1.5256e+00,
    544.0: 1.5792e+00,
    545.0: 1.5435e+00,
    546.0: 1.5291e+00,
    547.0: 1.5490e+00,
    548.0: 1.5049e+00,
    549.0: 1.5520e+00,
    550.0: 1.5399e+00,
    551.0: 1.5382e+00,
    552.0: 1.5697e+00,
    553.0: 1.5250e+00,
    554.0: 1.5549e+00,
    555.0: 1.5634e+00,
    556.0: 1.5366e+00,
    557.0: 1.4988e+00,
    558.0: 1.5310e+00,
    559.0: 1.4483e+00,
    560.0: 1.4740e+00,
    561.0: 1.5595e+00,
    562.0: 1.4847e+00,
    563.0: 1.5408e+00,
    564.0: 1.5106e+00,
    565.0: 1.5201e+00,
    566.0: 1.4374e+00,
    567.0: 1.5320e+00,
    568.0: 1.5180e+00,
    569.0: 1.4807e+00,
    570.0: 1.4816e+00,
    571.0: 1.4331e+00,
    572.0: 1.5134e+00,
    573.0: 1.5198e+00,
    574.0: 1.5119e+00,
    575.0: 1.4777e+00,
    576.0: 1.4654e+00,
    577.0: 1.5023e+00,
    578.0: 1.4560e+00,
    579.0: 1.4770e+00,
    580.0: 1.5020e+00,
    581.0: 1.5089e+00,
    582.0: 1.5320e+00,
    583.0: 1.5479e+00,
    584.0: 1.5448e+00,
    585.0: 1.5324e+00,
    586.0: 1.4953e+00,
    587.0: 1.5281e+00,
    588.0: 1.4934e+00,
    589.0: 1.2894e+00,
    590.0: 1.3709e+00,
    591.0: 1.4662e+00,
    592.0: 1.4354e+00,
    593.0: 1.4561e+00,
    594.0: 1.4491e+00,
    595.0: 1.4308e+00,
    596.0: 1.4745e+00,
    597.0: 1.4788e+00,
    598.0: 1.4607e+00,
    599.0: 1.4606e+00,
    600.0: 1.4753e+00,
    601.0: 1.4579e+00,
    602.0: 1.4360e+00,
    603.0: 1.4664e+00,
    604.0: 1.4921e+00,
    605.0: 1.4895e+00,
    606.0: 1.4822e+00,
    607.0: 1.4911e+00,
    608.0: 1.4862e+00,
    609.0: 1.4749e+00,
    610.0: 1.4686e+00,
    611.0: 1.4611e+00,
    612.0: 1.4831e+00,
    613.0: 1.4621e+00,
    614.0: 1.4176e+00,
    615.0: 1.4697e+00,
    616.0: 1.4310e+00,
    617.0: 1.4128e+00,
    618.0: 1.4664e+00,
    619.0: 1.4733e+00,
    620.0: 1.4739e+00,
    621.0: 1.4802e+00,
    622.0: 1.4269e+00,
    623.0: 1.4165e+00,
    624.0: 1.4118e+00,
    625.0: 1.4026e+00,
    626.0: 1.4012e+00,
    627.0: 1.4417e+00,
    628.0: 1.3631e+00,
    629.0: 1.4114e+00,
    630.0: 1.3924e+00,
    631.0: 1.4161e+00,
    632.0: 1.3638e+00,
    633.0: 1.4508e+00,
    634.0: 1.4284e+00,
    635.0: 1.4458e+00,
    636.0: 1.4128e+00,
    637.0: 1.4610e+00,
    638.0: 1.4707e+00,
    639.0: 1.4646e+00,
    640.0: 1.4340e+00,
    641.0: 1.4348e+00,
    642.0: 1.4376e+00,
    643.0: 1.4525e+00,
    644.0: 1.4462e+00,
    645.0: 1.4567e+00,
    646.0: 1.4150e+00,
    647.0: 1.4086e+00,
    648.0: 1.3952e+00,
    649.0: 1.3519e+00,
    650.0: 1.3594e+00,
    651.0: 1.4447e+00,
    652.0: 1.3871e+00,
    653.0: 1.4311e+00,
    654.0: 1.4153e+00,
    655.0: 1.3499e+00,
    656.0: 1.1851e+00,
    657.0: 1.2393e+00,
    658.0: 1.3855e+00,
    659.0: 1.3905e+00,
    660.0: 1.3992e+00,
    661.0: 1.3933e+00,
    662.0: 1.3819e+00,
    663.0: 1.3844e+00,
    664.0: 1.3967e+00,
    665.0: 1.4214e+00,
    666.0: 1.4203e+00,
    667.0: 1.4102e+00,
    668.0: 1.4150e+00,
    669.0: 1.4394e+00,
    670.0: 1.4196e+00,
    671.0: 1.4169e+00,
    672.0: 1.3972e+00,
    673.0: 1.4094e+00,
    674.0: 1.4074e+00,
    675.0: 1.3958e+00,
    676.0: 1.4120e+00,
    677.0: 1.3991e+00,
    678.0: 1.4066e+00,
    679.0: 1.3947e+00,
    680.0: 1.3969e+00,
    681.0: 1.3915e+00,
    682.0: 1.3981e+00,
    683.0: 1.3830e+00,
    684.0: 1.3739e+00,
    685.0: 1.3748e+00,
    686.0: 1.3438e+00,
    687.0: 9.6824e-01,
    688.0: 1.1206e+00,
    689.0: 1.1278e+00,
    690.0: 1.1821e+00,
    691.0: 1.2333e+00,
    692.0: 1.2689e+00,
    693.0: 1.2609e+00,
    694.0: 1.2464e+00,
    695.0: 1.2714e+00,
    696.0: 1.2684e+00,
    697.0: 1.3403e+00,
    698.0: 1.3192e+00,
    699.0: 1.2918e+00,
    700.0: 1.2823e+00,
    701.0: 1.2659e+00,
    702.0: 1.2674e+00,
    703.0: 1.2747e+00,
    704.0: 1.3078e+00,
    705.0: 1.3214e+00,
    706.0: 1.3144e+00,
    707.0: 1.3090e+00,
    708.0: 1.3048e+00,
    709.0: 1.3095e+00,
    710.0: 1.3175e+00,
    711.0: 1.3155e+00,
    712.0: 1.3071e+00,
    713.0: 1.2918e+00,
    714.0: 1.3029e+00,
    715.0: 1.2587e+00,
    716.0: 1.2716e+00,
    717.0: 1.1071e+00,
    718.0: 1.0296e+00,
    719.0: 9.2318e-01,
    720.0: 9.8550e-01,
    721.0: 1.0861e+00,
    722.0: 1.2407e+00,
    723.0: 1.1444e+00,
    724.0: 1.0555e+00,
    725.0: 1.0380e+00,
    726.0: 1.0813e+00,
    727.0: 1.0850e+00,
    728.0: 1.0400e+00,
    729.0: 1.0466e+00,
    730.0: 1.1285e+00,
    731.0: 1.0703e+00,
    732.0: 1.1534e+00,
    733.0: 1.1962e+00,
    734.0: 1.2357e+00,
    735.0: 1.2178e+00,
    736.0: 1.2059e+00,
    737.0: 1.2039e+00,
    738.0: 1.2269e+00,
    739.0: 1.1905e+00,
    740.0: 1.2195e+00,
    741.0: 1.2148e+00,
    742.0: 1.2153e+00,
    743.0: 1.2405e+00,
    744.0: 1.2503e+00,
    745.0: 1.2497e+00,
    746.0: 1.2470e+00,
    747.0: 1.2477e+00,
    748.0: 1.2401e+00,
    749.0: 1.2357e+00,
    750.0: 1.2341e+00,
    751.0: 1.2286e+00,
    752.0: 1.2330e+00,
    753.0: 1.2266e+00,
    754.0: 1.2420e+00,
    755.0: 1.2383e+00,
    756.0: 1.2232e+00,
    757.0: 1.2221e+00,
    758.0: 1.2295e+00,
    759.0: 1.1945e+00,
    760.0: 2.6604e-01,
    761.0: 1.5396e-01,
    762.0: 6.8766e-01,
    763.0: 3.7952e-01,
    764.0: 5.3878e-01,
    765.0: 6.8601e-01,
    766.0: 8.1461e-01,
    767.0: 9.7417e-01,
    768.0: 1.1138e+00,
    769.0: 1.1278e+00,
    770.0: 1.1608e+00,
    771.0: 1.1686e+00,
    772.0: 1.1778e+00,
    773.0: 1.1771e+00,
    774.0: 1.1771e+00,
    775.0: 1.1771e+00,
    776.0: 1.1798e+00,
    777.0: 1.1727e+00,
    778.0: 1.1713e+00,
    779.0: 1.1765e+00,
    780.0: 1.1636e+00,
    781.0: 1.1607e+00,
    782.0: 1.1662e+00,
    783.0: 1.1614e+00,
    784.0: 1.1536e+00,
    785.0: 1.1586e+00,
    786.0: 1.1592e+00,
    787.0: 1.1450e+00,
    788.0: 1.1305e+00,
    789.0: 1.1257e+00,
    790.0: 1.0910e+00,
    791.0: 1.1058e+00,
    792.0: 1.0953e+00,
    793.0: 1.0875e+00,
    794.0: 1.0972e+00,
    795.0: 1.0932e+00,
    796.0: 1.0742e+00,
    797.0: 1.0913e+00,
    798.0: 1.1121e+00,
    799.0: 1.0905e+00,
    800.0: 1.0725e+00,
    801.0: 1.0843e+00,
    802.0: 1.0856e+00,
    803.0: 1.0657e+00,
    804.0: 1.0782e+00,
    805.0: 1.0545e+00,
    806.0: 1.0974e+00,
    807.0: 1.0859e+00,
    808.0: 1.0821e+00,
    809.0: 1.0548e+00,
    810.0: 1.0559e+00,
    811.0: 1.0533e+00,
    812.0: 1.0268e+00,
    813.0: 1.0086e+00,
    814.0: 9.0356e-01,
    815.0: 8.9523e-01,
    816.0: 8.3216e-01,
    817.0: 8.5183e-01,
    818.0: 8.2259e-01,
    819.0: 9.0519e-01,
    820.0: 8.6188e-01,
    821.0: 9.9764e-01,
    822.0: 9.5157e-01,
    823.0: 6.7271e-01,
    824.0: 9.3506e-01,
    825.0: 9.6935e-01,
    826.0: 9.3381e-01,
    827.0: 9.8465e-01,
    828.0: 8.4979e-01,
    829.0: 9.2930e-01,
    830.0: 9.1601e-01,
    831.0: 9.2392e-01,
    832.0: 8.9426e-01,
    833.0: 9.5650e-01,
    834.0: 9.3412e-01,
    835.0: 1.0032e+00,
    836.0: 9.7234e-01,
    837.0: 1.0092e+00,
    838.0: 9.9901e-01,
    839.0: 1.0013e+00,
    840.0: 1.0157e+00,
    841.0: 1.0101e+00,
    842.0: 9.9703e-01,
    843.0: 1.0053e+00,
    844.0: 9.8631e-01,
    845.0: 1.0165e+00,
    846.0: 1.0187e+00,
    847.0: 9.9170e-01,
    848.0: 9.9217e-01,
    849.0: 9.8596e-01,
    850.0: 8.9372e-01,
    851.0: 9.7493e-01,
    852.0: 9.6927e-01,
    853.0: 9.6486e-01,
    854.0: 8.5112e-01,
    855.0: 9.1300e-01,
    856.0: 9.7317e-01,
    857.0: 9.9166e-01,
    858.0: 9.9196e-01,
    859.0: 9.9171e-01,
    860.0: 9.8816e-01,
    861.0: 9.8679e-01,
    862.0: 9.9449e-01,
    863.0: 1.0005e+00,
    864.0: 9.7916e-01,
    865.0: 9.6324e-01,
    866.0: 8.4900e-01,
    867.0: 9.1546e-01,
    868.0: 9.5920e-01,
    869.0: 9.4956e-01,
    870.0: 9.6755e-01,
    871.0: 9.5387e-01,
    872.0: 9.6686e-01,
    873.0: 9.5721e-01,
    874.0: 9.4042e-01,
    875.0: 9.2687e-01,
    876.0: 9.5277e-01,
    877.0: 9.5615e-01,
    878.0: 9.5237e-01,
    879.0: 9.3656e-01,
    880.0: 9.3957e-01,
    881.0: 9.0861e-01,
    882.0: 9.3245e-01,
    883.0: 9.2927e-01,
    884.0: 9.3305e-01,
    885.0: 9.4423e-01,
    886.0: 9.0752e-01,
    887.0: 9.1062e-01,
    888.0: 9.2228e-01,
    889.0: 9.3455e-01,
    890.0: 9.2393e-01,
    891.0: 9.2584e-01,
    892.0: 9.0881e-01,
    893.0: 8.7327e-01,
    894.0: 8.5130e-01,
    895.0: 8.1357e-01,
    896.0: 7.6253e-01,
    897.0: 6.6566e-01,
    898.0: 7.1780e-01,
    899.0: 5.4871e-01,
    900.0: 7.4260e-01,
    901.0: 5.9933e-01,
    902.0: 6.6791e-01,
    903.0: 6.8889e-01,
    904.0: 8.4457e-01,
    905.0: 8.1709e-01,
    906.0: 7.7558e-01,
    907.0: 6.3854e-01,
    908.0: 6.5217e-01,
    909.0: 7.0431e-01,
    910.0: 6.2467e-01,
    911.0: 6.6808e-01,
    912.0: 6.8893e-01,
    913.0: 6.2834e-01,
    914.0: 6.2649e-01,
    915.0: 6.7836e-01,
    916.0: 5.7646e-01,
    917.0: 7.3017e-01,
    918.0: 5.9271e-01,
    919.0: 7.3877e-01,
    920.0: 7.4414e-01,
    921.0: 7.8049e-01,
    922.0: 7.0026e-01,
    923.0: 7.4504e-01,
    924.0: 7.2150e-01,
    925.0: 7.1110e-01,
    926.0: 7.0331e-01,
    927.0: 7.8742e-01,
    928.0: 5.8968e-01,
    929.0: 5.5127e-01,
    930.0: 4.3210e-01,
    931.0: 4.0921e-01,
    932.0: 3.0086e-01,
    933.0: 2.4841e-01,
    934.0: 1.4380e-01,
    935.0: 2.5084e-01,
    936.0: 1.6142e-01,
    937.0: 1.6338e-01,
    938.0: 2.0058e-01,
    939.0: 3.9887e-01,
    940.0: 4.7181e-01,
    941.0: 3.7195e-01,
    942.0: 4.0532e-01,
    943.0: 2.7834e-01,
    944.0: 2.8579e-01,
    945.0: 3.6821e-01,
    946.0: 1.9461e-01,
    947.0: 3.7112e-01,
    948.0: 2.7423e-01,
    949.0: 4.9396e-01,
    950.0: 1.4726e-01,
    951.0: 4.8378e-01,
    952.0: 2.6891e-01,
    953.0: 3.4362e-01,
    954.0: 4.2411e-01,
    955.0: 3.4117e-01,
    956.0: 3.2821e-01,
    957.0: 2.7067e-01,
    958.0: 4.6101e-01,
    959.0: 3.7385e-01,
    960.0: 4.2066e-01,
    961.0: 4.6120e-01,
    962.0: 4.4174e-01,
    963.0: 5.0503e-01,
    964.0: 4.5860e-01,
    965.0: 5.0374e-01,
    966.0: 5.0275e-01,
    967.0: 5.0240e-01,
    968.0: 6.5210e-01,
    969.0: 6.8622e-01,
    970.0: 6.3461e-01,
    971.0: 7.1397e-01,
    972.0: 6.8765e-01,
    973.0: 6.0648e-01,
    974.0: 5.7529e-01,
    975.0: 5.8987e-01,
    976.0: 5.7191e-01,
    977.0: 6.3864e-01,
    978.0: 6.1509e-01,
    979.0: 6.3815e-01,
    980.0: 6.0468e-01,
    981.0: 7.1338e-01,
    982.0: 6.9218e-01,
    983.0: 6.6865e-01,
    984.0: 7.3732e-01,
    985.0: 6.8817e-01,
    986.0: 7.5083e-01,
    987.0: 7.3928e-01,
    988.0: 7.3462e-01,
    989.0: 7.4906e-01,
    990.0: 7.3227e-01,
    991.0: 7.5358e-01,
    992.0: 7.5102e-01,
    993.0: 7.3728e-01,
    994.0: 7.5410e-01,
    995.0: 7.5176e-01,
    996.0: 7.4884e-01,
    997.0: 7.3971e-01,
    998.0: 7.3887e-01,
    999.0: 7.3857e-01,
    1000.0: 7.3532e-01,
    1001.0: 7.4442e-01,
    1002.0: 7.2805e-01,
    1003.0: 7.3442e-01,
    1004.0: 7.2336e-01,
    1005.0: 6.8174e-01,
    1006.0: 7.1252e-01,
    1007.0: 7.2753e-01,
    1008.0: 7.2685e-01,
    1009.0: 7.1972e-01,
    1010.0: 7.1914e-01,
    1011.0: 7.2278e-01,
    1012.0: 7.1877e-01,
    1013.0: 7.1761e-01,
    1014.0: 7.2068e-01,
    1015.0: 7.0817e-01,
    1016.0: 7.1129e-01,
    1017.0: 7.0337e-01,
    1018.0: 7.1422e-01,
    1019.0: 6.8878e-01,
    1020.0: 6.9896e-01,
    1021.0: 7.0175e-01,
    1022.0: 6.8970e-01,
    1023.0: 6.9508e-01,
    1024.0: 6.9058e-01,
    1025.0: 6.9753e-01,
    1026.0: 6.9636e-01,
    1027.0: 6.9305e-01,
    1028.0: 6.9385e-01,
    1029.0: 6.8628e-01,
    1030.0: 6.9055e-01,
    1031.0: 6.8736e-01,
    1032.0: 6.8787e-01,
    1033.0: 6.7613e-01,
    1034.0: 6.8015e-01,
    1035.0: 6.8234e-01,
    1036.0: 6.8202e-01,
    1037.0: 6.7497e-01,
    1038.0: 6.7172e-01,
    1039.0: 6.7636e-01,
    1040.0: 6.7170e-01,
    1041.0: 6.7176e-01,
    1042.0: 6.7200e-01,
    1043.0: 6.6525e-01,
    1044.0: 6.6833e-01,
    1045.0: 6.6452e-01,
    1046.0: 6.4714e-01,
    1047.0: 6.5694e-01,
    1048.0: 6.6274e-01,
    1049.0: 6.5896e-01,
    1050.0: 6.5463e-01,
    1051.0: 6.5521e-01,
    1052.0: 6.5118e-01,
    1053.0: 6.4919e-01,
    1054.0: 6.4646e-01,
    1055.0: 6.4847e-01,
    1056.0: 6.4641e-01,
    1057.0: 6.4482e-01,
    1058.0: 6.3818e-01,
    1059.0: 6.1875e-01,
    1060.0: 6.3585e-01,
    1061.0: 6.2121e-01,
    1062.0: 6.3266e-01,
    1063.0: 6.2239e-01,
    1064.0: 6.3196e-01,
    1065.0: 6.2913e-01,
    1066.0: 6.1713e-01,
    1067.0: 6.2032e-01,
    1068.0: 6.1944e-01,
    1069.0: 5.8626e-01,
    1070.0: 6.0469e-01,
    1071.0: 6.1661e-01,
    1072.0: 6.1536e-01,
    1073.0: 6.0363e-01,
    1074.0: 6.2158e-01,
    1075.0: 5.9252e-01,
    1076.0: 6.1471e-01,
    1077.0: 6.0434e-01,
    1078.0: 6.0321e-01,
    1079.0: 6.0474e-01,
    1080.0: 5.9722e-01,
    1081.0: 5.8083e-01,
    1082.0: 5.8940e-01,
    1083.0: 5.9814e-01,
    1084.0: 5.7852e-01,
    1085.0: 5.9330e-01,
    1086.0: 5.5410e-01,
    1087.0: 5.6697e-01,
    1088.0: 5.9317e-01,
    1089.0: 5.7919e-01,
    1090.0: 5.5573e-01,
    1091.0: 5.8835e-01,
    1092.0: 5.8124e-01,
    1093.0: 5.1058e-01,
    1094.0: 5.3965e-01,
    1095.0: 5.2067e-01,
    1096.0: 5.0323e-01,
    1097.0: 5.7852e-01,
    1098.0: 5.0291e-01,
    1099.0: 5.0772e-01,
    1100.0: 4.8577e-01,
    1101.0: 4.9696e-01,
    1102.0: 4.6883e-01,
    1103.0: 4.6637e-01,
    1104.0: 4.6765e-01,
    1105.0: 5.0644e-01,
    1106.0: 3.9792e-01,
    1107.0: 4.8304e-01,
    1108.0: 4.1565e-01,
    1109.0: 4.1278e-01,
    1110.0: 4.7899e-01,
    1111.0: 3.3154e-01,
    1112.0: 4.1357e-01,
    1113.0: 2.6850e-01,
    1114.0: 2.9985e-01,
    1115.0: 2.4987e-01,
    1116.0: 2.0136e-01,
    1117.0: 7.9618e-02,
    1118.0: 2.1753e-01,
    1119.0: 1.1317e-01,
    1120.0: 1.4189e-01,
    1121.0: 1.8586e-01,
    1122.0: 8.1686e-02,
    1123.0: 1.2817e-01,
    1124.0: 1.0870e-01,
    1125.0: 1.4428e-01,
    1126.0: 5.1589e-02,
    1127.0: 1.5725e-01,
    1128.0: 9.9224e-02,
    1129.0: 1.0591e-01,
    1130.0: 7.0574e-02,
    1131.0: 2.9560e-01,
    1132.0: 2.3411e-01,
    1133.0: 1.5331e-01,
    1134.0: 4.1740e-02,
    1135.0: 1.5462e-02,
    1136.0: 1.2876e-01,
    1137.0: 2.8785e-01,
    1138.0: 2.0329e-01,
    1139.0: 2.9850e-01,
    1140.0: 2.5599e-01,
    1141.0: 1.9337e-01,
    1142.0: 2.2479e-01,
    1143.0: 3.1183e-01,
    1144.0: 1.1326e-01,
    1145.0: 1.4604e-01,
    1146.0: 1.5764e-01,
    1147.0: 5.9176e-02,
    1148.0: 2.7113e-01,
    1149.0: 2.1854e-01,
    1150.0: 1.2164e-01,
    1151.0: 2.0340e-01,
    1152.0: 2.4762e-01,
    1153.0: 2.3812e-01,
    1154.0: 1.4248e-01,
    1155.0: 3.1316e-01,
    1156.0: 2.8090e-01,
    1157.0: 3.1458e-01,
    1158.0: 3.1171e-01,
    1159.0: 3.3693e-01,
    1160.0: 2.8648e-01,
    1161.0: 3.4753e-01,
    1162.0: 3.5002e-01,
    1163.0: 4.6857e-01,
    1164.0: 4.0188e-01,
    1165.0: 3.8860e-01,
    1166.0: 3.7494e-01,
    1167.0: 4.0996e-01,
    1168.0: 4.1954e-01,
    1169.0: 4.2310e-01,
    1170.0: 4.5873e-01,
    1171.0: 4.4831e-01,
    1172.0: 4.5483e-01,
    1173.0: 4.5642e-01,
    1174.0: 3.3692e-01,
    1175.0: 4.5240e-01,
    1176.0: 4.7679e-01,
    1177.0: 4.7235e-01,
    1178.0: 3.6000e-01,
    1179.0: 4.8371e-01,
    1180.0: 4.4069e-01,
    1181.0: 4.5514e-01,
    1182.0: 3.2318e-01,
    1183.0: 4.3870e-01,
    1184.0: 4.1985e-01,
    1185.0: 4.0741e-01,
    1186.0: 4.7715e-01,
    1187.0: 4.5575e-01,
    1188.0: 3.3504e-01,
    1189.0: 4.1569e-01,
    1190.0: 4.6239e-01,
    1191.0: 4.4660e-01,
    1192.0: 4.7336e-01,
    1193.0: 4.5434e-01,
    1194.0: 4.6890e-01,
    1195.0: 4.4696e-01,
    1196.0: 4.3131e-01,
    1197.0: 4.7715e-01,
    1198.0: 4.3392e-01,
    1199.0: 3.6489e-01,
    1200.0: 4.4825e-01,
    1201.0: 4.3708e-01,
    1202.0: 4.3717e-01,
    1203.0: 4.3409e-01,
    1204.0: 3.6247e-01,
    1205.0: 4.3692e-01,
    1206.0: 4.8086e-01,
    1207.0: 4.2986e-01,
    1208.0: 4.3346e-01,
    1209.0: 4.1428e-01,
    1210.0: 4.5336e-01,
    1211.0: 4.2232e-01,
    1212.0: 4.2489e-01,
    1213.0: 4.6956e-01,
    1214.0: 4.3407e-01,
    1215.0: 4.2780e-01,
    1216.0: 4.6640e-01,
    1217.0: 4.5528e-01,
    1218.0: 4.5934e-01,
    1219.0: 4.4663e-01,
    1220.0: 4.5805e-01,
    1221.0: 4.6531e-01,
    1222.0: 4.5139e-01,
    1223.0: 4.4406e-01,
    1224.0: 4.4808e-01,
    1225.0: 4.6236e-01,
    1226.0: 4.6819e-01,
    1227.0: 4.3304e-01,
    1228.0: 4.6658e-01,
    1229.0: 4.6721e-01,
    1230.0: 4.6003e-01,
    1231.0: 4.7203e-01,
    1232.0: 4.6633e-01,
    1233.0: 4.5397e-01,
    1234.0: 4.7016e-01,
    1235.0: 4.6504e-01,
    1236.0: 4.6908e-01,
    1237.0: 4.6339e-01,
    1238.0: 4.6797e-01,
    1239.0: 4.6272e-01,
    1240.0: 4.6077e-01,
    1241.0: 4.6197e-01,
    1242.0: 4.6247e-01,
    1243.0: 4.5754e-01,
    1244.0: 4.5528e-01,
    1245.0: 4.5655e-01,
    1246.0: 4.5945e-01,
    1247.0: 4.5746e-01,
    1248.0: 4.5860e-01,
    1249.0: 4.5966e-01,
    1250.0: 4.5705e-01,
    1251.0: 4.5258e-01,
    1252.0: 4.5097e-01,
    1253.0: 4.4773e-01,
    1254.0: 4.4363e-01,
    1255.0: 4.5070e-01,
    1256.0: 4.4023e-01,
    1257.0: 4.3532e-01,
    1258.0: 4.4496e-01,
    1259.0: 4.2725e-01,
    1260.0: 4.3110e-01,
    1261.0: 4.1146e-01,
    1262.0: 3.9567e-01,
    1263.0: 4.0019e-01,
    1264.0: 3.7148e-01,
    1265.0: 3.9570e-01,
    1266.0: 3.8527e-01,
    1267.0: 3.8822e-01,
    1268.0: 3.7051e-01,
    1269.0: 2.4652e-01,
    1270.0: 3.8744e-01,
    1271.0: 4.0825e-01,
    1272.0: 4.0879e-01,
    1273.0: 4.0625e-01,
    1274.0: 4.0614e-01,
    1275.0: 4.1233e-01,
    1276.0: 4.1693e-01,
    1277.0: 4.2001e-01,
    1278.0: 4.2763e-01,
    1279.0: 4.2456e-01,
    1280.0: 4.2204e-01,
    1281.0: 4.1335e-01,
    1282.0: 3.7305e-01,
    1283.0: 4.0733e-01,
    1284.0: 4.2078e-01,
    1285.0: 4.2399e-01,
    1286.0: 4.2714e-01,
    1287.0: 4.2213e-01,
    1288.0: 4.1989e-01,
    1289.0: 4.0936e-01,
    1290.0: 4.1285e-01,
    1291.0: 4.1786e-01,
    1292.0: 3.9618e-01,
    1293.0: 4.1257e-01,
    1294.0: 4.0421e-01,
    1295.0: 4.0514e-01,
    1296.0: 3.8957e-01,
    1297.0: 3.7130e-01,
    1298.0: 3.9183e-01,
    1299.0: 4.0852e-01,
    1300.0: 3.5312e-01,
    1301.0: 3.6228e-01,
    1302.0: 3.9181e-01,
    1303.0: 3.4621e-01,
    1304.0: 3.0062e-01,
    1305.0: 3.8382e-01,
    1306.0: 3.8453e-01,
    1307.0: 3.0594e-01,
    1308.0: 3.4696e-01,
    1309.0: 3.8413e-01,
    1310.0: 3.0114e-01,
    1311.0: 3.3366e-01,
    1312.0: 3.3337e-01,
    1313.0: 3.1352e-01,
    1314.0: 2.8833e-01,
    1315.0: 2.8581e-01,
    1316.0: 3.2419e-01,
    1317.0: 3.1217e-01,
    1318.0: 3.3328e-01,
    1319.0: 2.6855e-01,
    1320.0: 2.5872e-01,
    1321.0: 2.9866e-01,
    1322.0: 3.0217e-01,
    1323.0: 2.3279e-01,
    1324.0: 2.6249e-01,
    1325.0: 3.2224e-01,
    1326.0: 2.8051e-01,
    1327.0: 2.6625e-01,
    1328.0: 2.3450e-01,
    1329.0: 1.7759e-01,
    1330.0: 2.2923e-01,
    1331.0: 1.4480e-01,
    1332.0: 1.4579e-01,
    1333.0: 2.0304e-01,
    1334.0: 1.6925e-01,
    1335.0: 2.3117e-01,
    1336.0: 1.8348e-01,
    1337.0: 1.6454e-01,
    1338.0: 1.7804e-01,
    1339.0: 1.7681e-01,
    1340.0: 1.6831e-01,
    1341.0: 1.7039e-01,
    1342.0: 1.7798e-01,
    1343.0: 1.2711e-01,
    1344.0: 7.5645e-02,
    1345.0: 1.0904e-01,
    1346.0: 5.8186e-02,
    1347.0: 6.0119e-02,
    1348.0: 4.7451e-03,
    1349.0: 1.6159e-02,
    1350.0: 1.6025e-02,
    1351.0: 4.6298e-03,
    1352.0: 1.5164e-03,
    1353.0: 9.6096e-05,
    1354.0: 2.9009e-04,
    1355.0: 3.6034e-06,
    1356.0: 4.8070e-05,
    1357.0: 7.1786e-05,
    1358.0: 4.1948e-06,
    1359.0: 7.3439e-07,
    1360.0: 2.1404e-06,
    1361.0: 4.8133e-09,
    1362.0: 1.8076e-11,
    1363.0: 3.1563e-06,
    1364.0: 1.3589e-06,
    1365.0: 9.0764e-12,
    1366.0: 1.2791e-05,
    1367.0: 4.9764e-06,
    1368.0: 1.4810e-13,
    1369.0: 5.1667e-07,
    1370.0: 2.9200e-07,
    1371.0: 1.9731e-08,
    1372.0: 2.7498e-06,
    1373.0: 4.4401e-05,
    1374.0: 1.7917e-04,
    1375.0: 3.2332e-04,
    1376.0: 2.5748e-04,
    1377.0: 1.2270e-04,
    1378.0: 1.1089e-03,
    1379.0: 5.2164e-05,
    1380.0: 8.1587e-05,
    1381.0: 2.3716e-06,
    1382.0: 2.5672e-06,
    1383.0: 4.4017e-08,
    1384.0: 6.1689e-07,
    1385.0: 2.0899e-06,
    1386.0: 2.5215e-06,
    1387.0: 1.9896e-04,
    1388.0: 4.0262e-06,
    1389.0: 5.8098e-04,
    1390.0: 4.9328e-04,
    1391.0: 3.4384e-04,
    1392.0: 2.3782e-05,
    1393.0: 1.1586e-04,
    1394.0: 7.5526e-05,
    1395.0: 6.7136e-07,
    1396.0: 6.3215e-09,
    1397.0: 4.9057e-05,
    1398.0: 1.2704e-03,
    1399.0: 8.1226e-04,
    1400.0: 3.2466e-09,
    1401.0: 1.0528e-08,
    1402.0: 1.8353e-03,
    1403.0: 2.3800e-03,
    1404.0: 7.3892e-04,
    1405.0: 3.6444e-07,
    1406.0: 2.0448e-03,
    1407.0: 1.7457e-04,
    1408.0: 1.6493e-03,
    1409.0: 6.1919e-04,
    1410.0: 4.6653e-04,
    1411.0: 2.1142e-03,
    1412.0: 2.6396e-03,
    1413.0: 2.3353e-02,
    1414.0: 3.6378e-04,
    1415.0: 1.8366e-04,
    1416.0: 3.5565e-02,
    1417.0: 1.1759e-02,
    1418.0: 1.3559e-02,
    1419.0: 2.1442e-03,
    1420.0: 8.2718e-03,
    1421.0: 9.1637e-03,
    1422.0: 4.6314e-02,
    1423.0: 9.2198e-03,
    1424.0: 1.6975e-02,
    1425.0: 2.5850e-02,
    1426.0: 2.7792e-02,
    1427.0: 4.9546e-02,
    1428.0: 4.5588e-03,
    1429.0: 3.8020e-02,
    1430.0: 6.1601e-02,
    1431.0: 5.0156e-02,
    1432.0: 2.5194e-03,
    1433.0: 3.5834e-02,
    1434.0: 2.0962e-02,
    1435.0: 2.1416e-02,
    1436.0: 3.8351e-02,
    1437.0: 2.9880e-02,
    1438.0: 1.3263e-02,
    1439.0: 5.1039e-02,
    1440.0: 3.9601e-02,
    1441.0: 3.1800e-02,
    1442.0: 3.6317e-02,
    1443.0: 4.5063e-02,
    1444.0: 6.1791e-02,
    1445.0: 4.9751e-02,
    1446.0: 2.3095e-02,
    1447.0: 3.6215e-02,
    1448.0: 1.1569e-01,
    1449.0: 1.0213e-01,
    1450.0: 2.7412e-02,
    1451.0: 1.1271e-02,
    1452.0: 6.2361e-02,
    1453.0: 8.1978e-02,
    1454.0: 1.3759e-01,
    1455.0: 6.6150e-02,
    1456.0: 8.8509e-02,
    1457.0: 1.1700e-01,
    1458.0: 1.3643e-01,
    1459.0: 1.6307e-01,
    1460.0: 8.5421e-02,
    1461.0: 9.0276e-02,
    1462.0: 1.3060e-01,
    1463.0: 4.3225e-02,
    1464.0: 1.5184e-01,
    1465.0: 9.3383e-02,
    1466.0: 6.5197e-02,
    1467.0: 3.6054e-02,
    1468.0: 7.6942e-02,
    1469.0: 9.4845e-02,
    1470.0: 4.9678e-02,
    1471.0: 1.7848e-02,
    1472.0: 4.6771e-02,
    1473.0: 7.0198e-02,
    1474.0: 9.7339e-02,
    1475.0: 1.8463e-01,
    1476.0: 6.8778e-02,
    1477.0: 6.9736e-02,
    1478.0: 6.3480e-02,
    1479.0: 1.2001e-01,
    1480.0: 6.0637e-02,
    1481.0: 1.1529e-01,
    1482.0: 5.8490e-02,
    1483.0: 1.4859e-01,
    1484.0: 1.3747e-01,
    1485.0: 1.2503e-01,
    1486.0: 1.2340e-01,
    1487.0: 6.0629e-02,
    1488.0: 9.4180e-02,
    1489.0: 1.8973e-01,
    1490.0: 1.7478e-01,
    1491.0: 1.9778e-01,
    1492.0: 1.6441e-01,
    1493.0: 1.8157e-01,
    1494.0: 2.0367e-01,
    1495.0: 1.8253e-01,
    1496.0: 1.6852e-01,
    1497.0: 2.2850e-01,
    1498.0: 1.8968e-01,
    1499.0: 2.1759e-01,
    1500.0: 2.5061e-01,
    1501.0: 2.6552e-01,
    1502.0: 2.3356e-01,
    1503.0: 1.8493e-01,
    1504.0: 1.6029e-01,
    1505.0: 1.8402e-01,
    1506.0: 2.5773e-01,
    1507.0: 2.5514e-01,
    1508.0: 2.4302e-01,
    1509.0: 1.8690e-01,
    1510.0: 2.7052e-01,
    1511.0: 2.6474e-01,
    1512.0: 2.6068e-01,
    1513.0: 2.4239e-01,
    1514.0: 2.2571e-01,
    1515.0: 2.6573e-01,
    1516.0: 2.5683e-01,
    1517.0: 2.4929e-01,
    1518.0: 2.5211e-01,
    1519.0: 2.4437e-01,
    1520.0: 2.6450e-01,
    1521.0: 2.7505e-01,
    1522.0: 2.6378e-01,
    1523.0: 2.8004e-01,
    1524.0: 2.7539e-01,
    1525.0: 2.5884e-01,
    1526.0: 2.6745e-01,
    1527.0: 2.6220e-01,
    1528.0: 2.7928e-01,
    1529.0: 2.7244e-01,
    1530.0: 2.5522e-01,
    1531.0: 2.6973e-01,
    1532.0: 2.7839e-01,
    1533.0: 2.7714e-01,
    1534.0: 2.6892e-01,
    1535.0: 2.6686e-01,
    1536.0: 2.7464e-01,
    1537.0: 2.7336e-01,
    1538.0: 2.7202e-01,
    1539.0: 2.7295e-01,
    1540.0: 2.6491e-01,
    1541.0: 2.6904e-01,
    1542.0: 2.6927e-01,
    1543.0: 2.7208e-01,
    1544.0: 2.7210e-01,
    1545.0: 2.7705e-01,
    1546.0: 2.7481e-01,
    1547.0: 2.7309e-01,
    1548.0: 2.6675e-01,
    1549.0: 2.7342e-01,
    1550.0: 2.6990e-01,
    1551.0: 2.7058e-01,
    1552.0: 2.7182e-01,
    1553.0: 2.7132e-01,
    1554.0: 2.6474e-01,
    1555.0: 2.6759e-01,
    1556.0: 2.6310e-01,
    1557.0: 2.7062e-01,
    1558.0: 2.6848e-01,
    1559.0: 2.6808e-01,
    1560.0: 2.6568e-01,
    1561.0: 2.7002e-01,
    1562.0: 2.6756e-01,
    1563.0: 2.6667e-01,
    1564.0: 2.6264e-01,
    1565.0: 2.6728e-01,
    1566.0: 2.6245e-01,
    1567.0: 2.6308e-01,
    1568.0: 2.5722e-01,
    1569.0: 2.5452e-01,
    1570.0: 2.4175e-01,
    1571.0: 2.3507e-01,
    1572.0: 2.3775e-01,
    1573.0: 2.3407e-01,
    1574.0: 2.4145e-01,
    1575.0: 2.3974e-01,
    1576.0: 2.4678e-01,
    1577.0: 2.1602e-01,
    1578.0: 2.3516e-01,
    1579.0: 2.3672e-01,
    1580.0: 2.4464e-01,
    1581.0: 2.4870e-01,
    1582.0: 2.4195e-01,
    1583.0: 2.4755e-01,
    1584.0: 2.4904e-01,
    1585.0: 2.5874e-01,
    1586.0: 2.5569e-01,
    1587.0: 2.5303e-01,
    1588.0: 2.5107e-01,
    1589.0: 2.3233e-01,
    1590.0: 2.4179e-01,
    1591.0: 2.4197e-01,
    1592.0: 2.5225e-01,
    1593.0: 2.5833e-01,
    1594.0: 2.5624e-01,
    1595.0: 2.5823e-01,
    1596.0: 2.4452e-01,
    1597.0: 2.4692e-01,
    1598.0: 2.5421e-01,
    1599.0: 2.4202e-01,
    1600.0: 2.3810e-01,
    1601.0: 2.2323e-01,
    1602.0: 2.2413e-01,
    1603.0: 2.2397e-01,
    1604.0: 2.2842e-01,
    1605.0: 2.3683e-01,
    1606.0: 2.4140e-01,
    1607.0: 2.3296e-01,
    1608.0: 2.2990e-01,
    1609.0: 2.2727e-01,
    1610.0: 2.1760e-01,
    1611.0: 2.2680e-01,
    1612.0: 2.3076e-01,
    1613.0: 2.3719e-01,
    1614.0: 2.3838e-01,
    1615.0: 2.4104e-01,
    1616.0: 2.3050e-01,
    1617.0: 2.3465e-01,
    1618.0: 2.4352e-01,
    1619.0: 2.4100e-01,
    1620.0: 2.3449e-01,
    1621.0: 2.3430e-01,
    1622.0: 2.3754e-01,
    1623.0: 2.4246e-01,
    1624.0: 2.4269e-01,
    1625.0: 2.3782e-01,
    1626.0: 2.3971e-01,
    1627.0: 2.4078e-01,
    1628.0: 2.4126e-01,
    1629.0: 2.4137e-01,
    1630.0: 2.3651e-01,
    1631.0: 2.3806e-01,
    1632.0: 2.3821e-01,
    1633.0: 2.3267e-01,
    1634.0: 2.3282e-01,
    1635.0: 2.3367e-01,
    1636.0: 2.3539e-01,
    1637.0: 2.2700e-01,
    1638.0: 2.2007e-01,
    1639.0: 2.2026e-01,
    1640.0: 2.1511e-01,
    1641.0: 2.1960e-01,
    1642.0: 2.2082e-01,
    1643.0: 2.1535e-01,
    1644.0: 2.2355e-01,
    1645.0: 2.1822e-01,
    1646.0: 2.1749e-01,
    1647.0: 2.2768e-01,
    1648.0: 2.1655e-01,
    1649.0: 2.1867e-01,
    1650.0: 2.2526e-01,
    1651.0: 2.0855e-01,
    1652.0: 2.2373e-01,
    1653.0: 2.2277e-01,
    1654.0: 2.1583e-01,
    1655.0: 2.2231e-01,
    1656.0: 2.2101e-01,
    1657.0: 2.2223e-01,
    1658.0: 2.2487e-01,
    1659.0: 2.2120e-01,
    1660.0: 2.2332e-01,
    1661.0: 2.2384e-01,
    1662.0: 2.1908e-01,
    1663.0: 2.2235e-01,
    1664.0: 2.2098e-01,
    1665.0: 2.1178e-01,
    1666.0: 1.7884e-01,
    1667.0: 2.1068e-01,
    1668.0: 2.1459e-01,
    1669.0: 2.1516e-01,
    1670.0: 2.2168e-01,
    1671.0: 2.1879e-01,
    1672.0: 2.1147e-01,
    1673.0: 2.1629e-01,
    1674.0: 2.1575e-01,
    1675.0: 2.1360e-01,
    1676.0: 2.1145e-01,
    1677.0: 2.1229e-01,
    1678.0: 2.0915e-01,
    1679.0: 2.1303e-01,
    1680.0: 2.0558e-01,
    1681.0: 1.9447e-01,
    1682.0: 2.0366e-01,
    1683.0: 2.0906e-01,
    1684.0: 1.9797e-01,
    1685.0: 2.1321e-01,
    1686.0: 2.1026e-01,
    1687.0: 2.0484e-01,
    1688.0: 2.1013e-01,
    1689.0: 2.0718e-01,
    1690.0: 2.0523e-01,
    1691.0: 1.9303e-01,
    1692.0: 2.0708e-01,
    1693.0: 2.1134e-01,
    1694.0: 2.0477e-01,
    1695.0: 2.0968e-01,
    1696.0: 2.0922e-01,
    1697.0: 1.8107e-01,
    1698.0: 2.0739e-01,
    1699.0: 2.0551e-01,
    1700.0: 1.9975e-01,
    1702.0: 2.0396e-01,
    1705.0: 1.9778e-01,
    1710.0: 1.8790e-01,
    1715.0: 1.8965e-01,
    1720.0: 1.8698e-01,
    1725.0: 1.7808e-01,
    1730.0: 1.7407e-01,
    1735.0: 1.6154e-01,
    1740.0: 1.6818e-01,
    1745.0: 1.5481e-01,
    1750.0: 1.6566e-01,
    1755.0: 1.5301e-01,
    1760.0: 1.5998e-01,
    1765.0: 1.3284e-01,
    1770.0: 1.4172e-01,
    1775.0: 1.1484e-01,
    1780.0: 1.0050e-01,
    1785.0: 7.6981e-02,
    1790.0: 8.8904e-02,
    1795.0: 4.6931e-02,
    1800.0: 3.1828e-02,
    1805.0: 1.4815e-02,
    1810.0: 9.6911e-03,
    1815.0: 3.2816e-03,
    1820.0: 9.8755e-04,
    1825.0: 1.2744e-03,
    1830.0: 5.2041e-06,
    1835.0: 6.4190e-06,
    1840.0: 6.2703e-08,
    1845.0: 6.2658e-06,
    1850.0: 2.9993e-06,
    1855.0: 2.8396e-07,
    1860.0: 1.1151e-05,
    1865.0: 1.6982e-05,
    1870.0: 2.6662e-10,
    1875.0: 4.5130e-10,
    1880.0: 7.7505e-05,
    1885.0: 4.3890e-05,
    1890.0: 2.2333e-04,
    1895.0: 1.2947e-04,
    1900.0: 8.6221e-07,
    1905.0: 5.6667e-07,
    1910.0: 2.3045e-05,
    1915.0: 1.9947e-05,
    1920.0: 4.5069e-04,
    1925.0: 9.3615e-04,
    1930.0: 5.5242e-04,
    1935.0: 3.5935e-03,
    1940.0: 3.2821e-03,
    1945.0: 1.0863e-02,
    1950.0: 1.6727e-02,
    1955.0: 1.0036e-02,
    1960.0: 2.1906e-02,
    1965.0: 2.8563e-02,
    1970.0: 4.8847e-02,
    1975.0: 6.7857e-02,
    1980.0: 7.5512e-02,
    1985.0: 8.3063e-02,
    1990.0: 8.5613e-02,
    1995.0: 8.1190e-02,
    2000.0: 3.8156e-02,
    2005.0: 1.5001e-02,
    2010.0: 3.9748e-02,
    2015.0: 2.6648e-02,
    2020.0: 4.4981e-02,
    2025.0: 7.4010e-02,
    2030.0: 8.4856e-02,
    2035.0: 9.6386e-02,
    2040.0: 8.9781e-02,
    2045.0: 9.1074e-02,
    2050.0: 6.7927e-02,
    2055.0: 5.4906e-02,
    2060.0: 6.9193e-02,
    2065.0: 6.1875e-02,
    2070.0: 6.5676e-02,
    2075.0: 7.7443e-02,
    2080.0: 8.6812e-02,
    2085.0: 8.5102e-02,
    2090.0: 8.9100e-02,
    2095.0: 8.9747e-02,
    2100.0: 8.6133e-02,
    2105.0: 9.3153e-02,
    2110.0: 8.9654e-02,
    2115.0: 9.1673e-02,
    2120.0: 8.7588e-02,
    2125.0: 8.8632e-02,
    2130.0: 8.9774e-02,
    2135.0: 9.0044e-02,
    2140.0: 9.0767e-02,
    2145.0: 8.9486e-02,
    2150.0: 8.4639e-02,
    2155.0: 8.4840e-02,
    2160.0: 8.4170e-02,
    2165.0: 7.6310e-02,
    2170.0: 8.1996e-02,
    2175.0: 8.0448e-02,
    2180.0: 8.1808e-02,
    2185.0: 7.4550e-02,
    2190.0: 7.9068e-02,
    2195.0: 7.8992e-02,
    2200.0: 7.1202e-02,
    2205.0: 7.4010e-02,
    2210.0: 7.9315e-02,
    2215.0: 7.6273e-02,
    2220.0: 7.7730e-02,
    2225.0: 7.5453e-02,
    2230.0: 7.5773e-02,
    2235.0: 7.4299e-02,
    2240.0: 7.3118e-02,
    2245.0: 7.0838e-02,
    2250.0: 7.1937e-02,
    2255.0: 6.7690e-02,
    2260.0: 6.6929e-02,
    2265.0: 6.8137e-02,
    2270.0: 6.4867e-02,
    2275.0: 6.4021e-02,
    2280.0: 6.6288e-02,
    2285.0: 6.3080e-02,
    2290.0: 6.3220e-02,
    2295.0: 6.1265e-02,
    2300.0: 5.8824e-02,
    2305.0: 5.9171e-02,
    2310.0: 6.3870e-02,
    2315.0: 5.8141e-02,
    2320.0: 5.2031e-02,
    2325.0: 5.6215e-02,
    2330.0: 5.6824e-02,
    2335.0: 5.7967e-02,
    2340.0: 4.5836e-02,
    2345.0: 5.1400e-02,
    2350.0: 4.1536e-02,
    2355.0: 4.7473e-02,
    2360.0: 5.0237e-02,
    2365.0: 4.9409e-02,
    2370.0: 3.0817e-02,
    2375.0: 4.4147e-02,
    2380.0: 4.2552e-02,
    2385.0: 3.0826e-02,
    2390.0: 3.7109e-02,
    2395.0: 4.0594e-02,
    2400.0: 4.4150e-02,
    2405.0: 3.3599e-02,
    2410.0: 3.3813e-02,
    2415.0: 2.7300e-02,
    2420.0: 2.6590e-02,
    2425.0: 3.3078e-02,
    2430.0: 4.5099e-02,
    2435.0: 1.4878e-02,
    2440.0: 4.3249e-02,
    2445.0: 2.0798e-02,
    2450.0: 1.3611e-02,
    2455.0: 2.4853e-02,
    2460.0: 3.3363e-02,
    2465.0: 2.4148e-02,
    2470.0: 1.6727e-02,
    2475.0: 1.6455e-02,
    2480.0: 8.0395e-03,
    2485.0: 5.6102e-03,
    2490.0: 3.5113e-03,
    2495.0: 2.8772e-03,
    2500.0: 7.0642e-03,
    2505.0: 1.5191e-03,
    2510.0: 2.2163e-03,
    2515.0: 5.1880e-04,
    2520.0: 3.7054e-04,
    2525.0: 4.1393e-05,
    2530.0: 6.3593e-07,
    2535.0: 1.7502e-07,
    2540.0: 3.7716e-07,
    2545.0: 5.3758e-11,
    2550.0: 2.8222e-13,
    2555.0: 1.0435e-09,
    2560.0: 3.1020e-11,
    2565.0: 1.5955e-14,
    2570.0: 1.5258e-18,
    2575.0: 1.0786e-27,
    2580.0: 3.8214e-22,
    2585.0: 1.7194e-34,
    2590.0: 5.4793e-31,
    2595.0: 2.2838e-33,
    2600.0: 4.4912e-28,
    2605.0: 5.8053e-35,
    2610.0: 5.9447e-34,
    2615.0: 1.1196e-37,
    2620.0: 5.6505e-29,
    2625.0: 3.8687e-28,
    2630.0: 2.8026e-45,
    2635.0: 3.9027e-16,
    2640.0: 1.1750e-16,
    2645.0: 8.9988e-19,
    2650.0: 1.4295e-19,
    2655.0: 1.3133e-27,
    2660.0: 2.6068e-25,
    2665.0: 1.1123e-37,
    2670.0: 0.0000e+00,
    2675.0: 0.0000e+00,
    2680.0: 0.0000e+00,
    2685.0: 0.0000e+00,
    2690.0: 1.0226e-29,
    2695.0: 7.1284e-33,
    2700.0: 0.0000e+00,
    2705.0: 2.9315e-42,
    2710.0: 1.1250e-35,
    2715.0: 3.8557e-26,
    2720.0: 5.6052e-45,
    2725.0: 7.2935e-22,
    2730.0: 6.0734e-19,
    2735.0: 5.4888e-21,
    2740.0: 2.3314e-27,
    2745.0: 1.3146e-23,
    2750.0: 1.6648e-28,
    2755.0: 6.7262e-44,
    2760.0: 0.0000e+00,
    2765.0: 2.6777e-27,
    2770.0: 8.3791e-24,
    2775.0: 3.9990e-38,
    2780.0: 4.8067e-34,
    2785.0: 3.8866e-27,
    2790.0: 1.2170e-16,
    2795.0: 3.6205e-16,
    2800.0: 1.6484e-12,
    2805.0: 6.7478e-14,
    2810.0: 4.0233e-10,
    2815.0: 2.8685e-10,
    2820.0: 2.0548e-11,
    2825.0: 1.7605e-07,
    2830.0: 3.9008e-06,
    2835.0: 2.1276e-10,
    2840.0: 1.9609e-07,
    2845.0: 4.0575e-05,
    2850.0: 1.1566e-06,
    2855.0: 4.4867e-07,
    2860.0: 2.5356e-05,
    2865.0: 1.6763e-04,
    2870.0: 6.3129e-06,
    2875.0: 3.9170e-04,
    2880.0: 2.4724e-04,
    2885.0: 4.5332e-04,
    2890.0: 1.8623e-04,
    2895.0: 2.6643e-03,
    2900.0: 8.1152e-04,
    2905.0: 1.1096e-04,
    2910.0: 2.7220e-03,
    2915.0: 1.2581e-03,
    2920.0: 2.8948e-03,
    2925.0: 1.0835e-03,
    2930.0: 5.8858e-03,
    2935.0: 6.4903e-03,
    2940.0: 1.6273e-03,
    2945.0: 1.4489e-03,
    2950.0: 5.2276e-03,
    2955.0: 2.3361e-03,
    2960.0: 4.5971e-03,
    2965.0: 7.4379e-03,
    2970.0: 3.5233e-04,
    2975.0: 8.5429e-04,
    2980.0: 1.3381e-03,
    2985.0: 6.9628e-03,
    2990.0: 1.0280e-02,
    2995.0: 4.2755e-03,
    3000.0: 7.8472e-03,
    3005.0: 2.8906e-03,
    3010.0: 6.8479e-03,
    3015.0: 5.5551e-03,
    3020.0: 6.3369e-04,
    3025.0: 7.5031e-03,
    3030.0: 6.0753e-03,
    3035.0: 2.4986e-03,
    3040.0: 2.0242e-03,
    3045.0: 4.2090e-03,
    3050.0: 1.0321e-03,
    3055.0: 2.8947e-04,
    3060.0: 6.3012e-03,
    3065.0: 2.9113e-03,
    3070.0: 1.7492e-03,
    3075.0: 6.0221e-03,
    3080.0: 3.6224e-03,
    3085.0: 1.7671e-03,
    3090.0: 2.3805e-03,
    3095.0: 6.5510e-04,
    3100.0: 4.4010e-03,
    3105.0: 9.2155e-04,
    3110.0: 8.4569e-04,
    3115.0: 2.2677e-03,
    3120.0: 9.8197e-03,
    3125.0: 3.0289e-03,
    3130.0: 5.7614e-03,
    3135.0: 1.1446e-02,
    3140.0: 3.3241e-03,
    3145.0: 3.2517e-03,
    3150.0: 6.6744e-03,
    3155.0: 5.6366e-03,
    3160.0: 9.2320e-03,
    3165.0: 1.4017e-02,
    3170.0: 1.2516e-02,
    3175.0: 9.2302e-03,
    3180.0: 1.0621e-02,
    3185.0: 8.0823e-03,
    3190.0: 4.2388e-03,
    3195.0: 2.6927e-03,
    3200.0: 4.3843e-04,
    3205.0: 3.0973e-04,
    3210.0: 1.3634e-04,
    3215.0: 4.9752e-04,
    3220.0: 1.6089e-03,
    3225.0: 1.9875e-04,
    3230.0: 3.4080e-04,
    3235.0: 7.2940e-03,
    3240.0: 3.7464e-03,
    3245.0: 7.3409e-04,
    3250.0: 2.6067e-03,
    3255.0: 9.9378e-03,
    3260.0: 1.2248e-03,
    3265.0: 2.4465e-03,
    3270.0: 1.2186e-03,
    3275.0: 5.9265e-03,
    3280.0: 2.8644e-03,
    3285.0: 1.1128e-02,
    3290.0: 8.7571e-03,
    3295.0: 1.2234e-03,
    3300.0: 1.7794e-03,
    3305.0: 3.9416e-03,
    3310.0: 3.9235e-03,
    3315.0: 1.6133e-05,
    3320.0: 5.9987e-05,
    3325.0: 3.5187e-03,
    3330.0: 4.6616e-03,
    3335.0: 9.0694e-03,
    3340.0: 3.4602e-03,
    3345.0: 3.5408e-03,
    3350.0: 8.0277e-03,
    3355.0: 3.6308e-03,
    3360.0: 5.2402e-03,
    3365.0: 7.1907e-03,
    3370.0: 3.9389e-03,
    3375.0: 8.4560e-03,
    3380.0: 5.1115e-03,
    3385.0: 7.4896e-03,
    3390.0: 9.8552e-03,
    3395.0: 9.5465e-03,
    3400.0: 1.2509e-02,
    3405.0: 4.4594e-03,
    3410.0: 7.0802e-03,
    3415.0: 7.2774e-03,
    3420.0: 1.3165e-02,
    3425.0: 1.0006e-02,
    3430.0: 8.6892e-03,
    3435.0: 1.1553e-02,
    3440.0: 8.0348e-03,
    3445.0: 1.1318e-02,
    3450.0: 1.1153e-02,
    3455.0: 8.3089e-03,
    3460.0: 1.2530e-02,
    3465.0: 9.8179e-03,
    3470.0: 1.2264e-02,
    3475.0: 1.0943e-02,
    3480.0: 1.1224e-02,
    3485.0: 1.2094e-02,
    3490.0: 1.0419e-02,
    3495.0: 1.2265e-02,
    3500.0: 1.1917e-02,
    3505.0: 1.1809e-02,
    3510.0: 1.1963e-02,
    3515.0: 1.1494e-02,
    3520.0: 1.2122e-02,
    3525.0: 1.1428e-02,
    3530.0: 1.1127e-02,
    3535.0: 9.4556e-03,
    3540.0: 9.0310e-03,
    3545.0: 9.5432e-03,
    3550.0: 1.0538e-02,
    3555.0: 9.0581e-03,
    3560.0: 1.0795e-02,
    3565.0: 1.0851e-02,
    3570.0: 8.3376e-03,
    3575.0: 8.6444e-03,
    3580.0: 1.0187e-02,
    3585.0: 9.1671e-03,
    3590.0: 9.4523e-03,
    3595.0: 9.6700e-03,
    3600.0: 1.0262e-02,
    3605.0: 1.0359e-02,
    3610.0: 9.4787e-03,
    3615.0: 9.4726e-03,
    3620.0: 1.1614e-02,
    3625.0: 1.0239e-02,
    3630.0: 9.9550e-03,
    3635.0: 1.0299e-02,
    3640.0: 1.1480e-02,
    3645.0: 1.0599e-02,
    3650.0: 1.0123e-02,
    3655.0: 1.0978e-02,
    3660.0: 1.0914e-02,
    3665.0: 1.0253e-02,
    3670.0: 7.9003e-03,
    3675.0: 4.8286e-03,
    3680.0: 8.3312e-03,
    3685.0: 9.4380e-03,
    3690.0: 9.6922e-03,
    3695.0: 1.0132e-02,
    3700.0: 1.0878e-02,
    3705.0: 1.0770e-02,
    3710.0: 9.3640e-03,
    3715.0: 9.2254e-03,
    3720.0: 1.0376e-02,
    3725.0: 1.0698e-02,
    3730.0: 9.2707e-03,
    3735.0: 8.5837e-03,
    3740.0: 8.8494e-03,
    3745.0: 1.0331e-02,
    3750.0: 9.2903e-03,
    3755.0: 8.9918e-03,
    3760.0: 8.8633e-03,
    3765.0: 8.5502e-03,
    3770.0: 9.1243e-03,
    3775.0: 9.0521e-03,
    3780.0: 9.5746e-03,
    3785.0: 8.8123e-03,
    3790.0: 7.7564e-03,
    3795.0: 8.8692e-03,
    3800.0: 9.8592e-03,
    3805.0: 9.3049e-03,
    3810.0: 8.2451e-03,
    3815.0: 7.7569e-03,
    3820.0: 9.6550e-03,
    3825.0: 9.5056e-03,
    3830.0: 9.5925e-03,
    3835.0: 7.6916e-03,
    3840.0: 8.9756e-03,
    3845.0: 8.7801e-03,
    3850.0: 8.8274e-03,
    3855.0: 8.5085e-03,
    3860.0: 7.9940e-03,
    3865.0: 8.0989e-03,
    3870.0: 7.3604e-03,
    3875.0: 6.7620e-03,
    3880.0: 6.5340e-03,
    3885.0: 6.7717e-03,
    3890.0: 6.8818e-03,
    3895.0: 7.4760e-03,
    3900.0: 7.9254e-03,
    3905.0: 7.9269e-03,
    3910.0: 7.1353e-03,
    3915.0: 6.9868e-03,
    3920.0: 6.9466e-03,
    3925.0: 6.8520e-03,
    3930.0: 7.0502e-03,
    3935.0: 7.3541e-03,
    3940.0: 7.4027e-03,
    3945.0: 7.5412e-03,
    3950.0: 7.6277e-03,
    3955.0: 7.7199e-03,
    3960.0: 7.7482e-03,
    3965.0: 7.8057e-03,
    3970.0: 7.6806e-03,
    3975.0: 7.5097e-03,
    3980.0: 7.3872e-03,
    3985.0: 7.4327e-03,
    3990.0: 7.3723e-03,
    3995.0: 7.2100e-03,
    4000.0: 7.1043e-03,
}

ASTMG173_DIRECT_CIRCUMSOLAR_DATA = {
    280.0: 2.5361e-26,
    280.5: 1.0917e-24,
    281.0: 6.1253e-24,
    281.5: 2.7479e-22,
    282.0: 2.8346e-21,
    282.5: 1.3271e-20,
    283.0: 6.7646e-20,
    283.5: 1.4614e-19,
    284.0: 4.9838e-18,
    284.5: 2.1624e-17,
    285.0: 8.9998e-17,
    285.5: 6.4424e-16,
    286.0: 2.3503e-15,
    286.5: 1.8458e-14,
    287.0: 7.2547e-14,
    287.5: 3.6618e-13,
    288.0: 2.8061e-12,
    288.5: 9.0651e-12,
    289.0: 3.4978e-11,
    289.5: 1.5368e-10,
    290.0: 5.1454e-10,
    290.5: 1.3303e-09,
    291.0: 3.8965e-09,
    291.5: 1.4425e-08,
    292.0: 4.0789e-08,
    292.5: 7.0414e-08,
    293.0: 1.5760e-07,
    293.5: 4.7095e-07,
    294.0: 9.4558e-07,
    294.5: 1.5965e-06,
    295.0: 3.2246e-06,
    295.5: 8.0206e-06,
    296.0: 1.4737e-05,
    296.5: 2.3312e-05,
    297.0: 3.3187e-05,
    297.5: 6.7912e-05,
    298.0: 1.1127e-04,
    298.5: 1.4270e-04,
    299.0: 2.0323e-04,
    299.5: 3.7386e-04,
    300.0: 4.5631e-04,
    300.5: 5.7207e-04,
    301.0: 9.1926e-04,
    301.5: 1.3201e-03,
    302.0: 1.4570e-03,
    302.5: 2.1910e-03,
    303.0: 3.7332e-03,
    303.5: 4.8044e-03,
    304.0: 5.0973e-03,
    304.5: 6.4675e-03,
    305.0: 8.9336e-03,
    305.5: 1.0186e-02,
    306.0: 1.0150e-02,
    306.5: 1.1568e-02,
    307.0: 1.5246e-02,
    307.5: 1.9468e-02,
    308.0: 2.0753e-02,
    308.5: 2.2750e-02,
    309.0: 2.2298e-02,
    309.5: 2.3672e-02,
    310.0: 2.7826e-02,
    310.5: 3.5879e-02,
    311.0: 4.5392e-02,
    311.5: 4.6156e-02,
    312.0: 5.0898e-02,
    312.5: 5.3766e-02,
    313.0: 5.8323e-02,
    313.5: 5.8999e-02,
    314.0: 6.5266e-02,
    314.5: 7.0476e-02,
    315.0: 7.3686e-02,
    315.5: 6.4833e-02,
    316.0: 6.7088e-02,
    316.5: 8.1118e-02,
    317.0: 9.3020e-02,
    317.5: 9.9712e-02,
    318.0: 9.5815e-02,
    318.5: 1.0005e-01,
    319.0: 1.0971e-01,
    319.5: 1.0693e-01,
    320.0: 1.1277e-01,
    320.5: 1.3305e-01,
    321.0: 1.3414e-01,
    321.5: 1.2817e-01,
    322.0: 1.2200e-01,
    322.5: 1.1970e-01,
    323.0: 1.1623e-01,
    323.5: 1.3393e-01,
    324.0: 1.4852e-01,
    324.5: 1.5467e-01,
    325.0: 1.5504e-01,
    325.5: 1.7936e-01,
    326.0: 2.0868e-01,
    326.5: 2.2162e-01,
    327.0: 2.1834e-01,
    327.5: 2.1285e-01,
    328.0: 1.9773e-01,
    328.5: 2.0675e-01,
    329.0: 2.3297e-01,
    329.5: 2.5864e-01,
    330.0: 2.6192e-01,
    330.5: 2.4103e-01,
    331.0: 2.2835e-01,
    331.5: 2.3635e-01,
    332.0: 2.4508e-01,
    332.5: 2.4655e-01,
    333.0: 2.4263e-01,
    333.5: 2.3265e-01,
    334.0: 2.3823e-01,
    334.5: 2.5434e-01,
    335.0: 2.6477e-01,
    335.5: 2.5894e-01,
    336.0: 2.3813e-01,
    336.5: 2.2099e-01,
    337.0: 2.1767e-01,
    337.5: 2.3434e-01,
    338.0: 2.5321e-01,
    338.5: 2.6549e-01,
    339.0: 2.7096e-01,
    339.5: 2.7847e-01,
    340.0: 2.9659e-01,
    340.5: 2.9674e-01,
    341.0: 2.7932e-01,
    341.5: 2.7853e-01,
    342.0: 2.9121e-01,
    342.5: 3.0296e-01,
    343.0: 3.0857e-01,
    343.5: 2.9246e-01,
    344.0: 2.5352e-01,
    344.5: 2.4439e-01,
    345.0: 2.7854e-01,
    345.5: 2.9761e-01,
    346.0: 2.9132e-01,
    346.5: 2.9747e-01,
    347.0: 3.0318e-01,
    347.5: 2.9351e-01,
    348.0: 2.9306e-01,
    348.5: 2.9884e-01,
    349.0: 2.8864e-01,
    349.5: 2.9720e-01,
    350.0: 3.2913e-01,
    350.5: 3.5471e-01,
    351.0: 3.4603e-01,
    351.5: 3.3388e-01,
    352.0: 3.2674e-01,
    352.5: 3.0954e-01,
    353.0: 3.2975e-01,
    353.5: 3.6351e-01,
    354.0: 3.8523e-01,
    354.5: 3.9043e-01,
    355.0: 3.9140e-01,
    355.5: 3.7878e-01,
    356.0: 3.5627e-01,
    356.5: 3.3495e-01,
    357.0: 2.9527e-01,
    357.5: 2.9950e-01,
    358.0: 2.7936e-01,
    358.5: 2.5998e-01,
    359.0: 3.0650e-01,
    359.5: 3.7013e-01,
    360.0: 3.9240e-01,
    360.5: 3.7167e-01,
    361.0: 3.4278e-01,
    361.5: 3.3647e-01,
    362.0: 3.5350e-01,
    362.5: 3.8804e-01,
    363.0: 4.0005e-01,
    363.5: 3.8994e-01,
    364.0: 4.0472e-01,
    364.5: 4.0179e-01,
    365.0: 4.1810e-01,
    365.5: 4.6110e-01,
    366.0: 4.9508e-01,
    366.5: 4.9694e-01,
    367.0: 4.8869e-01,
    367.5: 4.8041e-01,
    368.0: 4.5319e-01,
    368.5: 4.5106e-01,
    369.0: 4.7244e-01,
    369.5: 5.0856e-01,
    370.0: 5.1666e-01,
    370.5: 4.6798e-01,
    371.0: 4.7628e-01,
    371.5: 4.9587e-01,
    372.0: 4.6506e-01,
    372.5: 4.4389e-01,
    373.0: 4.2833e-01,
    373.5: 3.8682e-01,
    374.0: 3.8651e-01,
    374.5: 3.8435e-01,
    375.0: 4.1087e-01,
    375.5: 4.5514e-01,
    376.0: 4.7218e-01,
    376.5: 4.6538e-01,
    377.0: 5.0014e-01,
    377.5: 5.5890e-01,
    378.0: 6.0314e-01,
    378.5: 5.8892e-01,
    379.0: 5.2616e-01,
    379.5: 4.7255e-01,
    380.0: 4.9751e-01,
    380.5: 5.3396e-01,
    381.0: 5.4424e-01,
    381.5: 4.9135e-01,
    382.0: 4.1958e-01,
    382.5: 3.6361e-01,
    383.0: 3.2648e-01,
    383.5: 3.1658e-01,
    384.0: 3.6689e-01,
    384.5: 4.4239e-01,
    385.0: 4.8638e-01,
    385.5: 4.6549e-01,
    386.0: 4.4984e-01,
    386.5: 4.6848e-01,
    387.0: 4.7343e-01,
    387.5: 4.6731e-01,
    388.0: 4.6350e-01,
    388.5: 4.6096e-01,
    389.0: 5.0121e-01,
    389.5: 5.5637e-01,
    390.0: 5.8457e-01,
    390.5: 5.9038e-01,
    391.0: 6.2634e-01,
    391.5: 6.3617e-01,
    392.0: 5.8656e-01,
    392.5: 4.8961e-01,
    393.0: 3.5502e-01,
    393.5: 2.8272e-01,
    394.0: 3.6780e-01,
    394.5: 5.0811e-01,
    395.0: 6.0096e-01,
    395.5: 6.4101e-01,
    396.0: 5.6443e-01,
    396.5: 4.1101e-01,
    397.0: 3.1882e-01,
    397.5: 4.7151e-01,
    398.0: 6.3944e-01,
    398.5: 7.5622e-01,
    399.0: 8.0408e-01,
    399.5: 8.2980e-01,
    400.0: 8.3989e-01,
    401.0: 8.7691e-01,
    402.0: 9.1387e-01,
    403.0: 8.8211e-01,
    404.0: 8.9849e-01,
    405.0: 8.7849e-01,
    406.0: 8.5878e-01,
    407.0: 8.4545e-01,
    408.0: 8.8488e-01,
    409.0: 9.4717e-01,
    410.0: 8.0910e-01,
    411.0: 9.0770e-01,
    412.0: 9.6686e-01,
    413.0: 9.2951e-01,
    414.0: 9.2134e-01,
    415.0: 9.5569e-01,
    416.0: 9.8628e-01,
    417.0: 9.6392e-01,
    418.0: 9.2392e-01,
    419.0: 9.6354e-01,
    420.0: 8.8467e-01,
    421.0: 1.0067e+00,
    422.0: 9.9499e-01,
    423.0: 9.6531e-01,
    424.0: 9.6182e-01,
    425.0: 9.9312e-01,
    426.0: 9.6667e-01,
    427.0: 9.3550e-01,
    428.0: 9.4625e-01,
    429.0: 8.7766e-01,
    430.0: 7.0134e-01,
    431.0: 6.3779e-01,
    432.0: 1.0628e+00,
    433.0: 9.9050e-01,
    434.0: 9.1653e-01,
    435.0: 1.0070e+00,
    436.0: 1.1061e+00,
    437.0: 1.1306e+00,
    438.0: 9.9368e-01,
    439.0: 9.5753e-01,
    440.0: 1.0993e+00,
    441.0: 1.0859e+00,
    442.0: 1.1640e+00,
    443.0: 1.1823e+00,
    444.0: 1.1537e+00,
    445.0: 1.1992e+00,
    446.0: 1.0766e+00,
    447.0: 1.2257e+00,
    448.0: 1.2422e+00,
    449.0: 1.2409e+00,
    450.0: 1.2881e+00,
    451.0: 1.3376e+00,
    452.0: 1.2822e+00,
    453.0: 1.1854e+00,
    454.0: 1.2730e+00,
    455.0: 1.2655e+00,
    456.0: 1.3088e+00,
    457.0: 1.3213e+00,
    458.0: 1.2946e+00,
    459.0: 1.2859e+00,
    460.0: 1.2791e+00,
    461.0: 1.3255e+00,
    462.0: 1.3392e+00,
    463.0: 1.3452e+00,
    464.0: 1.3055e+00,
    465.0: 1.2905e+00,
    466.0: 1.3190e+00,
    467.0: 1.2616e+00,
    468.0: 1.3178e+00,
    469.0: 1.3247e+00,
    470.0: 1.2749e+00,
    471.0: 1.2975e+00,
    472.0: 1.3661e+00,
    473.0: 1.3144e+00,
    474.0: 1.3304e+00,
    475.0: 1.3755e+00,
    476.0: 1.3299e+00,
    477.0: 1.3392e+00,
    478.0: 1.3839e+00,
    479.0: 1.3586e+00,
    480.0: 1.3825e+00,
    481.0: 1.3836e+00,
    482.0: 1.3899e+00,
    483.0: 1.3742e+00,
    484.0: 1.3492e+00,
    485.0: 1.3457e+00,
    486.0: 1.0918e+00,
    487.0: 1.2235e+00,
    488.0: 1.3252e+00,
    489.0: 1.2492e+00,
    490.0: 1.3968e+00,
    491.0: 1.3435e+00,
    492.0: 1.2818e+00,
    493.0: 1.3719e+00,
    494.0: 1.3402e+00,
    495.0: 1.4238e+00,
    496.0: 1.3548e+00,
    497.0: 1.3788e+00,
    498.0: 1.3421e+00,
    499.0: 1.3429e+00,
    500.0: 1.3391e+00,
    501.0: 1.2990e+00,
    502.0: 1.2991e+00,
    503.0: 1.3597e+00,
    504.0: 1.2682e+00,
    505.0: 1.3598e+00,
    506.0: 1.4153e+00,
    507.0: 1.3548e+00,
    508.0: 1.3210e+00,
    509.0: 1.3850e+00,
    510.0: 1.3497e+00,
    511.0: 1.3753e+00,
    512.0: 1.4125e+00,
    513.0: 1.3277e+00,
    514.0: 1.3003e+00,
    515.0: 1.3385e+00,
    516.0: 1.3514e+00,
    517.0: 1.1017e+00,
    518.0: 1.2605e+00,
    519.0: 1.2222e+00,
    520.0: 1.3349e+00,
    521.0: 1.3452e+00,
    522.0: 1.3760e+00,
    523.0: 1.2976e+00,
    524.0: 1.3962e+00,
    525.0: 1.3859e+00,
    526.0: 1.3479e+00,
    527.0: 1.1795e+00,
    528.0: 1.3508e+00,
    529.0: 1.4142e+00,
    530.0: 1.3598e+00,
    531.0: 1.4348e+00,
    532.0: 1.4094e+00,
    533.0: 1.2590e+00,
    534.0: 1.3491e+00,
    535.0: 1.3701e+00,
    536.0: 1.4292e+00,
    537.0: 1.3229e+00,
    538.0: 1.3896e+00,
    539.0: 1.3558e+00,
    540.0: 1.3096e+00,
    541.0: 1.2595e+00,
    542.0: 1.3714e+00,
    543.0: 1.3493e+00,
    544.0: 1.3971e+00,
    545.0: 1.3657e+00,
    546.0: 1.3536e+00,
    547.0: 1.3717e+00,
    548.0: 1.3331e+00,
    549.0: 1.3752e+00,
    550.0: 1.3648e+00,
    551.0: 1.3639e+00,
    552.0: 1.3923e+00,
    553.0: 1.3533e+00,
    554.0: 1.3802e+00,
    555.0: 1.3883e+00,
    556.0: 1.3651e+00,
    557.0: 1.3321e+00,
    558.0: 1.3613e+00,
    559.0: 1.2885e+00,
    560.0: 1.3118e+00,
    561.0: 1.3885e+00,
    562.0: 1.3225e+00,
    563.0: 1.3731e+00,
    564.0: 1.3466e+00,
    565.0: 1.3555e+00,
    566.0: 1.2823e+00,
    567.0: 1.3673e+00,
    568.0: 1.3554e+00,
    569.0: 1.3228e+00,
    570.0: 1.3240e+00,
    571.0: 1.2810e+00,
    572.0: 1.3534e+00,
    573.0: 1.3595e+00,
    574.0: 1.3527e+00,
    575.0: 1.3225e+00,
    576.0: 1.3118e+00,
    577.0: 1.3452e+00,
    578.0: 1.3040e+00,
    579.0: 1.3230e+00,
    580.0: 1.3455e+00,
    581.0: 1.3518e+00,
    582.0: 1.3729e+00,
    583.0: 1.3872e+00,
    584.0: 1.3845e+00,
    585.0: 1.3737e+00,
    586.0: 1.3409e+00,
    587.0: 1.3708e+00,
    588.0: 1.3403e+00,
    589.0: 1.1582e+00,
    590.0: 1.2316e+00,
    591.0: 1.3171e+00,
    592.0: 1.2900e+00,
    593.0: 1.3086e+00,
    594.0: 1.3029e+00,
    595.0: 1.2870e+00,
    596.0: 1.3260e+00,
    597.0: 1.3303e+00,
    598.0: 1.3142e+00,
    599.0: 1.3145e+00,
    600.0: 1.3278e+00,
    601.0: 1.3123e+00,
    602.0: 1.2928e+00,
    603.0: 1.3205e+00,
    604.0: 1.3439e+00,
    605.0: 1.3418e+00,
    606.0: 1.3353e+00,
    607.0: 1.3434e+00,
    608.0: 1.3392e+00,
    609.0: 1.3292e+00,
    610.0: 1.3237e+00,
    611.0: 1.3170e+00,
    612.0: 1.3370e+00,
    613.0: 1.3182e+00,
    614.0: 1.2783e+00,
    615.0: 1.3254e+00,
    616.0: 1.2906e+00,
    617.0: 1.2744e+00,
    618.0: 1.3228e+00,
    619.0: 1.3292e+00,
    620.0: 1.3299e+00,
    621.0: 1.3359e+00,
    622.0: 1.2882e+00,
    623.0: 1.2793e+00,
    624.0: 1.2751e+00,
    625.0: 1.2667e+00,
    626.0: 1.2655e+00,
    627.0: 1.3022e+00,
    628.0: 1.2328e+00,
    629.0: 1.2758e+00,
    630.0: 1.2589e+00,
    631.0: 1.2799e+00,
    632.0: 1.2327e+00,
    633.0: 1.3110e+00,
    634.0: 1.2907e+00,
    635.0: 1.3065e+00,
    636.0: 1.2768e+00,
    637.0: 1.3204e+00,
    638.0: 1.3292e+00,
    639.0: 1.3238e+00,
    640.0: 1.2962e+00,
    641.0: 1.2970e+00,
    642.0: 1.2995e+00,
    643.0: 1.3130e+00,
    644.0: 1.3074e+00,
    645.0: 1.3170e+00,
    646.0: 1.2797e+00,
    647.0: 1.2744e+00,
    648.0: 1.2625e+00,
    649.0: 1.2234e+00,
    650.0: 1.2299e+00,
    651.0: 1.3071e+00,
    652.0: 1.2558e+00,
    653.0: 1.2950e+00,
    654.0: 1.2807e+00,
    655.0: 1.2220e+00,
    656.0: 1.0727e+00,
    657.0: 1.1218e+00,
    658.0: 1.2540e+00,
    659.0: 1.2586e+00,
    660.0: 1.2668e+00,
    661.0: 1.2618e+00,
    662.0: 1.2518e+00,
    663.0: 1.2539e+00,
    664.0: 1.2647e+00,
    665.0: 1.2871e+00,
    666.0: 1.2860e+00,
    667.0: 1.2767e+00,
    668.0: 1.2810e+00,
    669.0: 1.3032e+00,
    670.0: 1.2853e+00,
    671.0: 1.2829e+00,
    672.0: 1.2651e+00,
    673.0: 1.2760e+00,
    674.0: 1.2742e+00,
    675.0: 1.2639e+00,
    676.0: 1.2786e+00,
    677.0: 1.2669e+00,
    678.0: 1.2737e+00,
    679.0: 1.2629e+00,
    680.0: 1.2650e+00,
    681.0: 1.2601e+00,
    682.0: 1.2662e+00,
    683.0: 1.2526e+00,
    684.0: 1.2445e+00,
    685.0: 1.2454e+00,
    686.0: 1.2174e+00,
    687.0: 8.8285e-01,
    688.0: 1.0195e+00,
    689.0: 1.0260e+00,
    690.0: 1.0746e+00,
    691.0: 1.1201e+00,
    692.0: 1.1516e+00,
    693.0: 1.1446e+00,
    694.0: 1.1318e+00,
    695.0: 1.1538e+00,
    696.0: 1.1513e+00,
    697.0: 1.2151e+00,
    698.0: 1.1961e+00,
    699.0: 1.1721e+00,
    700.0: 1.1636e+00,
    701.0: 1.1489e+00,
    702.0: 1.1500e+00,
    703.0: 1.1567e+00,
    704.0: 1.1864e+00,
    705.0: 1.1989e+00,
    706.0: 1.1925e+00,
    707.0: 1.1875e+00,
    708.0: 1.1839e+00,
    709.0: 1.1880e+00,
    710.0: 1.1954e+00,
    711.0: 1.1934e+00,
    712.0: 1.1856e+00,
    713.0: 1.1719e+00,
    714.0: 1.1823e+00,
    715.0: 1.1428e+00,
    716.0: 1.1548e+00,
    717.0: 1.0081e+00,
    718.0: 9.3873e-01,
    719.0: 8.4274e-01,
    720.0: 8.9940e-01,
    721.0: 9.8967e-01,
    722.0: 1.1281e+00,
    723.0: 1.0423e+00,
    724.0: 9.6305e-01,
    725.0: 9.4741e-01,
    726.0: 9.8638e-01,
    727.0: 9.8988e-01,
    728.0: 9.4968e-01,
    729.0: 9.5503e-01,
    730.0: 1.0294e+00,
    731.0: 9.7702e-01,
    732.0: 1.0520e+00,
    733.0: 1.0901e+00,
    734.0: 1.1261e+00,
    735.0: 1.1101e+00,
    736.0: 1.0994e+00,
    737.0: 1.0978e+00,
    738.0: 1.1184e+00,
    739.0: 1.0855e+00,
    740.0: 1.1119e+00,
    741.0: 1.1078e+00,
    742.0: 1.1084e+00,
    743.0: 1.1316e+00,
    744.0: 1.1408e+00,
    745.0: 1.1404e+00,
    746.0: 1.1381e+00,
    747.0: 1.1389e+00,
    748.0: 1.1323e+00,
    749.0: 1.1286e+00,
    750.0: 1.1273e+00,
    751.0: 1.1224e+00,
    752.0: 1.1265e+00,
    753.0: 1.1210e+00,
    754.0: 1.1353e+00,
    755.0: 1.1321e+00,
    756.0: 1.1185e+00,
    757.0: 1.1176e+00,
    758.0: 1.1246e+00,
    759.0: 1.0932e+00,
    760.0: 2.4716e-01,
    761.0: 1.4328e-01,
    762.0: 6.3491e-01,
    763.0: 3.5217e-01,
    764.0: 4.9885e-01,
    765.0: 6.3377e-01,
    766.0: 7.5080e-01,
    767.0: 8.9574e-01,
    768.0: 1.0222e+00,
    769.0: 1.0347e+00,
    770.0: 1.0646e+00,
    771.0: 1.0716e+00,
    772.0: 1.0802e+00,
    773.0: 1.0797e+00,
    774.0: 1.0800e+00,
    775.0: 1.0801e+00,
    776.0: 1.0827e+00,
    777.0: 1.0764e+00,
    778.0: 1.0754e+00,
    779.0: 1.0803e+00,
    780.0: 1.0687e+00,
    781.0: 1.0662e+00,
    782.0: 1.0714e+00,
    783.0: 1.0672e+00,
    784.0: 1.0602e+00,
    785.0: 1.0649e+00,
    786.0: 1.0656e+00,
    787.0: 1.0530e+00,
    788.0: 1.0399e+00,
    789.0: 1.0359e+00,
    790.0: 1.0045e+00,
    791.0: 1.0179e+00,
    792.0: 1.0084e+00,
    793.0: 1.0015e+00,
    794.0: 1.0101e+00,
    795.0: 1.0066e+00,
    796.0: 9.8985e-01,
    797.0: 1.0057e+00,
    798.0: 1.0245e+00,
    799.0: 1.0048e+00,
    800.0: 9.8859e-01,
    801.0: 9.9978e-01,
    802.0: 1.0011e+00,
    803.0: 9.8288e-01,
    804.0: 9.9452e-01,
    805.0: 9.7270e-01,
    806.0: 1.0122e+00,
    807.0: 1.0018e+00,
    808.0: 9.9844e-01,
    809.0: 9.7353e-01,
    810.0: 9.7488e-01,
    811.0: 9.7273e-01,
    812.0: 9.4882e-01,
    813.0: 9.3236e-01,
    814.0: 8.3681e-01,
    815.0: 8.2927e-01,
    816.0: 7.7171e-01,
    817.0: 7.8984e-01,
    818.0: 7.6299e-01,
    819.0: 8.3844e-01,
    820.0: 7.9899e-01,
    821.0: 9.2292e-01,
    822.0: 8.8081e-01,
    823.0: 6.2576e-01,
    824.0: 8.6619e-01,
    825.0: 8.9752e-01,
    826.0: 8.6530e-01,
    827.0: 9.1178e-01,
    828.0: 7.8870e-01,
    829.0: 8.6135e-01,
    830.0: 8.4930e-01,
    831.0: 8.5666e-01,
    832.0: 8.2961e-01,
    833.0: 8.8622e-01,
    834.0: 8.6608e-01,
    835.0: 9.2917e-01,
    836.0: 9.0135e-01,
    837.0: 9.3493e-01,
    838.0: 9.2585e-01,
    839.0: 9.2783e-01,
    840.0: 9.4124e-01,
    841.0: 9.3626e-01,
    842.0: 9.2411e-01,
    843.0: 9.3171e-01,
    844.0: 9.1434e-01,
    845.0: 9.4226e-01,
    846.0: 9.4447e-01,
    847.0: 9.1947e-01,
    848.0: 9.2010e-01,
    849.0: 9.1447e-01,
    850.0: 8.2900e-01,
    851.0: 9.0454e-01,
    852.0: 8.9942e-01,
    853.0: 8.9540e-01,
    854.0: 7.9000e-01,
    855.0: 8.4746e-01,
    856.0: 9.0343e-01,
    857.0: 9.2059e-01,
    858.0: 9.2094e-01,
    859.0: 9.2081e-01,
    860.0: 9.1764e-01,
    861.0: 9.1648e-01,
    862.0: 9.2367e-01,
    863.0: 9.2934e-01,
    864.0: 9.0956e-01,
    865.0: 8.9487e-01,
    866.0: 7.8882e-01,
    867.0: 8.5066e-01,
    868.0: 8.9140e-01,
    869.0: 8.8252e-01,
    870.0: 8.9933e-01,
    871.0: 8.8671e-01,
    872.0: 8.9887e-01,
    873.0: 8.8999e-01,
    874.0: 8.7451e-01,
    875.0: 8.6204e-01,
    876.0: 8.8625e-01,
    877.0: 8.8948e-01,
    878.0: 8.8607e-01,
    879.0: 8.7144e-01,
    880.0: 8.7434e-01,
    881.0: 8.4563e-01,
    882.0: 8.6787e-01,
    883.0: 8.6494e-01,
    884.0: 8.6859e-01,
    885.0: 8.7913e-01,
    886.0: 8.4515e-01,
    887.0: 8.4799e-01,
    888.0: 8.5899e-01,
    889.0: 8.7041e-01,
    890.0: 8.6078e-01,
    891.0: 8.6255e-01,
    892.0: 8.4688e-01,
    893.0: 8.1412e-01,
    894.0: 7.9413e-01,
    895.0: 7.5956e-01,
    896.0: 7.1265e-01,
    897.0: 6.2310e-01,
    898.0: 6.7137e-01,
    899.0: 5.1461e-01,
    900.0: 6.9429e-01,
    901.0: 5.6162e-01,
    902.0: 6.2500e-01,
    903.0: 6.4483e-01,
    904.0: 7.8862e-01,
    905.0: 7.6337e-01,
    906.0: 7.2502e-01,
    907.0: 5.9833e-01,
    908.0: 6.1091e-01,
    909.0: 6.5912e-01,
    910.0: 5.8553e-01,
    911.0: 6.2580e-01,
    912.0: 6.4508e-01,
    913.0: 5.8906e-01,
    914.0: 5.8740e-01,
    915.0: 6.3550e-01,
    916.0: 5.4099e-01,
    917.0: 6.8350e-01,
    918.0: 5.5612e-01,
    919.0: 6.9161e-01,
    920.0: 6.9657e-01,
    921.0: 7.3004e-01,
    922.0: 6.5584e-01,
    923.0: 6.9698e-01,
    924.0: 6.7571e-01,
    925.0: 6.6621e-01,
    926.0: 6.5875e-01,
    927.0: 7.3684e-01,
    928.0: 5.5363e-01,
    929.0: 5.1792e-01,
    930.0: 4.0679e-01,
    931.0: 3.8540e-01,
    932.0: 2.8386e-01,
    933.0: 2.3459e-01,
    934.0: 1.3604e-01,
    935.0: 2.3690e-01,
    936.0: 1.5267e-01,
    937.0: 1.5453e-01,
    938.0: 1.8962e-01,
    939.0: 3.7591e-01,
    940.0: 4.4411e-01,
    941.0: 3.5071e-01,
    942.0: 3.8192e-01,
    943.0: 2.6289e-01,
    944.0: 2.6987e-01,
    945.0: 3.4729e-01,
    946.0: 1.8409e-01,
    947.0: 3.5010e-01,
    948.0: 2.5911e-01,
    949.0: 4.6514e-01,
    950.0: 1.3944e-01,
    951.0: 4.5563e-01,
    952.0: 2.5418e-01,
    953.0: 3.2442e-01,
    954.0: 3.9988e-01,
    955.0: 3.2204e-01,
    956.0: 3.0996e-01,
    957.0: 2.5591e-01,
    958.0: 4.3453e-01,
    959.0: 3.5294e-01,
    960.0: 3.9685e-01,
    961.0: 4.3481e-01,
    962.0: 4.1664e-01,
    963.0: 4.7585e-01,
    964.0: 4.3242e-01,
    965.0: 4.7469e-01,
    966.0: 4.7377e-01,
    967.0: 4.7353e-01,
    968.0: 6.1301e-01,
    969.0: 6.4480e-01,
    970.0: 5.9689e-01,
    971.0: 6.7059e-01,
    972.0: 6.4629e-01,
    973.0: 5.7081e-01,
    974.0: 5.4170e-01,
    975.0: 5.5536e-01,
    976.0: 5.3872e-01,
    977.0: 6.0084e-01,
    978.0: 5.7903e-01,
    979.0: 6.0046e-01,
    980.0: 5.6941e-01,
    981.0: 6.7058e-01,
    982.0: 6.5102e-01,
    983.0: 6.2915e-01,
    984.0: 6.9290e-01,
    985.0: 6.4734e-01,
    986.0: 7.0553e-01,
    987.0: 6.9489e-01,
    988.0: 6.9059e-01,
    989.0: 7.0391e-01,
    990.0: 6.8843e-01,
    991.0: 7.0833e-01,
    992.0: 7.0597e-01,
    993.0: 6.9325e-01,
    994.0: 7.0891e-01,
    995.0: 7.0673e-01,
    996.0: 7.0409e-01,
    997.0: 6.9555e-01,
    998.0: 6.9481e-01,
    999.0: 6.9455e-01,
    1000.0: 6.9159e-01,
    1001.0: 7.0013e-01,
    1002.0: 6.8498e-01,
    1003.0: 6.9086e-01,
    1004.0: 6.8056e-01,
    1005.0: 6.4140e-01,
    1006.0: 6.7047e-01,
    1007.0: 6.8463e-01,
    1008.0: 6.8407e-01,
    1009.0: 6.7742e-01,
    1010.0: 6.7695e-01,
    1011.0: 6.8034e-01,
    1012.0: 6.7667e-01,
    1013.0: 6.7556e-01,
    1014.0: 6.7848e-01,
    1015.0: 6.6676e-01,
    1016.0: 6.6976e-01,
    1017.0: 6.6237e-01,
    1018.0: 6.7263e-01,
    1019.0: 6.4880e-01,
    1020.0: 6.5839e-01,
    1021.0: 6.6107e-01,
    1022.0: 6.4980e-01,
    1023.0: 6.5490e-01,
    1024.0: 6.5077e-01,
    1025.0: 6.5727e-01,
    1026.0: 6.5625e-01,
    1027.0: 6.5318e-01,
    1028.0: 6.5398e-01,
    1029.0: 6.4687e-01,
    1030.0: 6.5092e-01,
    1031.0: 6.4799e-01,
    1032.0: 6.4852e-01,
    1033.0: 6.3751e-01,
    1034.0: 6.4136e-01,
    1035.0: 6.4348e-01,
    1036.0: 6.4323e-01,
    1037.0: 6.3664e-01,
    1038.0: 6.3361e-01,
    1039.0: 6.3802e-01,
    1040.0: 6.3366e-01,
    1041.0: 6.3379e-01,
    1042.0: 6.3406e-01,
    1043.0: 6.2773e-01,
    1044.0: 6.3067e-01,
    1045.0: 6.2712e-01,
    1046.0: 6.1078e-01,
    1047.0: 6.2008e-01,
    1048.0: 6.2559e-01,
    1049.0: 6.2206e-01,
    1050.0: 6.1802e-01,
    1051.0: 6.1862e-01,
    1052.0: 6.1487e-01,
    1053.0: 6.1302e-01,
    1054.0: 6.1048e-01,
    1055.0: 6.1242e-01,
    1056.0: 6.1055e-01,
    1057.0: 6.0910e-01,
    1058.0: 6.0286e-01,
    1059.0: 5.8453e-01,
    1060.0: 6.0073e-01,
    1061.0: 5.8694e-01,
    1062.0: 5.9782e-01,
    1063.0: 5.8815e-01,
    1064.0: 5.9722e-01,
    1065.0: 5.9461e-01,
    1066.0: 5.8330e-01,
    1067.0: 5.8637e-01,
    1068.0: 5.8561e-01,
    1069.0: 5.5428e-01,
    1070.0: 5.7178e-01,
    1071.0: 5.8304e-01,
    1072.0: 5.8194e-01,
    1073.0: 5.7086e-01,
    1074.0: 5.8780e-01,
    1075.0: 5.6054e-01,
    1076.0: 5.8141e-01,
    1077.0: 5.7175e-01,
    1078.0: 5.7076e-01,
    1079.0: 5.7210e-01,
    1080.0: 5.6519e-01,
    1081.0: 5.4973e-01,
    1082.0: 5.5773e-01,
    1083.0: 5.6603e-01,
    1084.0: 5.4775e-01,
    1085.0: 5.6163e-01,
    1086.0: 5.2496e-01,
    1087.0: 5.3685e-01,
    1088.0: 5.6159e-01,
    1089.0: 5.4856e-01,
    1090.0: 5.2656e-01,
    1091.0: 5.5722e-01,
    1092.0: 5.5048e-01,
    1093.0: 4.8417e-01,
    1094.0: 5.1120e-01,
    1095.0: 4.9363e-01,
    1096.0: 4.7731e-01,
    1097.0: 5.4805e-01,
    1098.0: 4.7709e-01,
    1099.0: 4.8161e-01,
    1100.0: 4.6113e-01,
    1101.0: 4.7169e-01,
    1102.0: 4.4513e-01,
    1103.0: 4.4291e-01,
    1104.0: 4.4412e-01,
    1105.0: 4.8065e-01,
    1106.0: 3.7840e-01,
    1107.0: 4.5866e-01,
    1108.0: 3.9517e-01,
    1109.0: 3.9249e-01,
    1110.0: 4.5496e-01,
    1111.0: 3.1572e-01,
    1112.0: 3.9330e-01,
    1113.0: 2.5599e-01,
    1114.0: 2.8576e-01,
    1115.0: 2.3833e-01,
    1116.0: 1.9223e-01,
    1117.0: 7.6164e-02,
    1118.0: 2.0763e-01,
    1119.0: 1.0821e-01,
    1120.0: 1.3562e-01,
    1121.0: 1.7753e-01,
    1122.0: 7.8159e-02,
    1123.0: 1.2255e-01,
    1124.0: 1.0397e-01,
    1125.0: 1.3794e-01,
    1126.0: 4.9394e-02,
    1127.0: 1.5032e-01,
    1128.0: 9.4946e-02,
    1129.0: 1.0133e-01,
    1130.0: 6.7568e-02,
    1131.0: 2.8201e-01,
    1132.0: 2.2359e-01,
    1133.0: 1.4661e-01,
    1134.0: 3.9988e-02,
    1135.0: 1.4819e-02,
    1136.0: 1.2320e-01,
    1137.0: 2.7472e-01,
    1138.0: 1.9428e-01,
    1139.0: 2.8484e-01,
    1140.0: 2.4447e-01,
    1141.0: 1.8486e-01,
    1142.0: 2.1481e-01,
    1143.0: 2.9758e-01,
    1144.0: 1.0843e-01,
    1145.0: 1.3976e-01,
    1146.0: 1.5085e-01,
    1147.0: 5.6715e-02,
    1148.0: 2.5898e-01,
    1149.0: 2.0894e-01,
    1150.0: 1.1648e-01,
    1151.0: 1.9453e-01,
    1152.0: 2.3666e-01,
    1153.0: 2.2762e-01,
    1154.0: 1.3643e-01,
    1155.0: 2.9903e-01,
    1156.0: 2.6837e-01,
    1157.0: 3.0040e-01,
    1158.0: 2.9770e-01,
    1159.0: 3.2163e-01,
    1160.0: 2.7371e-01,
    1161.0: 3.3167e-01,
    1162.0: 3.3412e-01,
    1163.0: 4.4644e-01,
    1164.0: 3.8332e-01,
    1165.0: 3.7080e-01,
    1166.0: 3.5781e-01,
    1167.0: 3.9105e-01,
    1168.0: 4.0018e-01,
    1169.0: 4.0348e-01,
    1170.0: 4.3731e-01,
    1171.0: 4.2750e-01,
    1172.0: 4.3370e-01,
    1173.0: 4.3522e-01,
    1174.0: 3.2193e-01,
    1175.0: 4.3142e-01,
    1176.0: 4.5447e-01,
    1177.0: 4.5040e-01,
    1178.0: 3.4391e-01,
    1179.0: 4.6114e-01,
    1180.0: 4.2052e-01,
    1181.0: 4.3423e-01,
    1182.0: 3.0903e-01,
    1183.0: 4.1853e-01,
    1184.0: 4.0083e-01,
    1185.0: 3.8908e-01,
    1186.0: 4.5518e-01,
    1187.0: 4.3494e-01,
    1188.0: 3.2039e-01,
    1189.0: 3.9685e-01,
    1190.0: 4.4124e-01,
    1191.0: 4.2638e-01,
    1192.0: 4.5171e-01,
    1193.0: 4.3370e-01,
    1194.0: 4.4751e-01,
    1195.0: 4.2671e-01,
    1196.0: 4.1188e-01,
    1197.0: 4.5527e-01,
    1198.0: 4.1426e-01,
    1199.0: 3.4869e-01,
    1200.0: 4.2789e-01,
    1201.0: 4.1735e-01,
    1202.0: 4.1743e-01,
    1203.0: 4.1441e-01,
    1204.0: 3.4645e-01,
    1205.0: 4.1714e-01,
    1206.0: 4.5872e-01,
    1207.0: 4.1044e-01,
    1208.0: 4.1379e-01,
    1209.0: 3.9548e-01,
    1210.0: 4.3267e-01,
    1211.0: 4.0319e-01,
    1212.0: 4.0572e-01,
    1213.0: 4.4804e-01,
    1214.0: 4.1443e-01,
    1215.0: 4.0851e-01,
    1216.0: 4.4509e-01,
    1217.0: 4.3457e-01,
    1218.0: 4.3842e-01,
    1219.0: 4.2639e-01,
    1220.0: 4.3724e-01,
    1221.0: 4.4413e-01,
    1222.0: 4.3096e-01,
    1223.0: 4.2400e-01,
    1224.0: 4.2788e-01,
    1225.0: 4.4141e-01,
    1226.0: 4.4696e-01,
    1227.0: 4.1360e-01,
    1228.0: 4.4544e-01,
    1229.0: 4.4608e-01,
    1230.0: 4.3928e-01,
    1231.0: 4.5067e-01,
    1232.0: 4.4525e-01,
    1233.0: 4.3359e-01,
    1234.0: 4.4893e-01,
    1235.0: 4.4409e-01,
    1236.0: 4.4795e-01,
    1237.0: 4.4259e-01,
    1238.0: 4.4694e-01,
    1239.0: 4.4194e-01,
    1240.0: 4.4011e-01,
    1241.0: 4.4130e-01,
    1242.0: 4.4179e-01,
    1243.0: 4.3712e-01,
    1244.0: 4.3499e-01,
    1245.0: 4.3622e-01,
    1246.0: 4.3902e-01,
    1247.0: 4.3715e-01,
    1248.0: 4.3828e-01,
    1249.0: 4.3930e-01,
    1250.0: 4.3684e-01,
    1251.0: 4.3260e-01,
    1252.0: 4.3106e-01,
    1253.0: 4.2803e-01,
    1254.0: 4.2416e-01,
    1255.0: 4.3088e-01,
    1256.0: 4.2096e-01,
    1257.0: 4.1630e-01,
    1258.0: 4.2549e-01,
    1259.0: 4.0868e-01,
    1260.0: 4.1235e-01,
    1261.0: 3.9371e-01,
    1262.0: 3.7867e-01,
    1263.0: 3.8300e-01,
    1264.0: 3.5568e-01,
    1265.0: 3.7871e-01,
    1266.0: 3.6881e-01,
    1267.0: 3.7159e-01,
    1268.0: 3.5475e-01,
    1269.0: 2.3656e-01,
    1270.0: 3.7087e-01,
    1271.0: 3.9062e-01,
    1272.0: 3.9114e-01,
    1273.0: 3.8874e-01,
    1274.0: 3.8864e-01,
    1275.0: 3.9455e-01,
    1276.0: 3.9895e-01,
    1277.0: 4.0191e-01,
    1278.0: 4.0916e-01,
    1279.0: 4.0626e-01,
    1280.0: 4.0387e-01,
    1281.0: 3.9554e-01,
    1282.0: 3.5695e-01,
    1283.0: 3.8978e-01,
    1284.0: 4.0268e-01,
    1285.0: 4.0577e-01,
    1286.0: 4.0878e-01,
    1287.0: 4.0405e-01,
    1288.0: 4.0192e-01,
    1289.0: 3.9194e-01,
    1290.0: 3.9522e-01,
    1291.0: 4.0004e-01,
    1292.0: 3.7946e-01,
    1293.0: 3.9506e-01,
    1294.0: 3.8709e-01,
    1295.0: 3.8801e-01,
    1296.0: 3.7322e-01,
    1297.0: 3.5583e-01,
    1298.0: 3.7536e-01,
    1299.0: 3.9127e-01,
    1300.0: 3.3855e-01,
    1301.0: 3.4728e-01,
    1302.0: 3.7539e-01,
    1303.0: 3.3197e-01,
    1304.0: 2.8849e-01,
    1305.0: 3.6783e-01,
    1306.0: 3.6853e-01,
    1307.0: 2.9362e-01,
    1308.0: 3.3277e-01,
    1309.0: 3.6822e-01,
    1310.0: 2.8908e-01,
    1311.0: 3.2012e-01,
    1312.0: 3.1986e-01,
    1313.0: 3.0089e-01,
    1314.0: 2.7690e-01,
    1315.0: 2.7447e-01,
    1316.0: 3.1113e-01,
    1317.0: 2.9969e-01,
    1318.0: 3.1984e-01,
    1319.0: 2.5803e-01,
    1320.0: 2.4864e-01,
    1321.0: 2.8684e-01,
    1322.0: 2.9023e-01,
    1323.0: 2.2386e-01,
    1324.0: 2.5231e-01,
    1325.0: 3.0943e-01,
    1326.0: 2.6956e-01,
    1327.0: 2.5593e-01,
    1328.0: 2.2555e-01,
    1329.0: 1.7097e-01,
    1330.0: 2.2052e-01,
    1331.0: 1.3951e-01,
    1332.0: 1.4046e-01,
    1333.0: 1.9545e-01,
    1334.0: 1.6302e-01,
    1335.0: 2.2244e-01,
    1336.0: 1.7670e-01,
    1337.0: 1.5852e-01,
    1338.0: 1.7151e-01,
    1339.0: 1.7033e-01,
    1340.0: 1.6216e-01,
    1341.0: 1.6419e-01,
    1342.0: 1.7149e-01,
    1343.0: 1.2259e-01,
    1344.0: 7.3018e-02,
    1345.0: 1.0521e-01,
    1346.0: 5.6189e-02,
    1347.0: 5.8058e-02,
    1348.0: 4.5862e-03,
    1349.0: 1.5617e-02,
    1350.0: 1.5488e-02,
    1351.0: 4.4759e-03,
    1352.0: 1.4661e-03,
    1353.0: 9.2918e-05,
    1354.0: 2.8051e-04,
    1355.0: 3.4847e-06,
    1356.0: 4.6489e-05,
    1357.0: 6.9429e-05,
    1358.0: 4.0575e-06,
    1359.0: 7.1040e-07,
    1360.0: 2.0706e-06,
    1361.0: 4.6566e-09,
    1362.0: 1.7489e-11,
    1363.0: 3.0540e-06,
    1364.0: 1.3150e-06,
    1365.0: 8.7833e-12,
    1366.0: 1.2379e-05,
    1367.0: 4.8161e-06,
    1368.0: 1.4311e-13,
    1369.0: 5.0008e-07,
    1370.0: 2.8266e-07,
    1371.0: 1.9101e-08,
    1372.0: 2.6623e-06,
    1373.0: 4.2991e-05,
    1374.0: 1.7350e-04,
    1375.0: 3.1309e-04,
    1376.0: 2.4935e-04,
    1377.0: 1.1883e-04,
    1378.0: 1.0741e-03,
    1379.0: 5.0533e-05,
    1380.0: 7.9042e-05,
    1381.0: 2.2978e-06,
    1382.0: 2.4874e-06,
    1383.0: 4.2653e-08,
    1384.0: 5.9782e-07,
    1385.0: 2.0255e-06,
    1386.0: 2.4441e-06,
    1387.0: 1.9288e-04,
    1388.0: 3.9037e-06,
    1389.0: 5.6338e-04,
    1390.0: 4.7836e-04,
    1391.0: 3.3345e-04,
    1392.0: 2.3065e-05,
    1393.0: 1.1238e-04,
    1394.0: 7.3268e-05,
    1395.0: 6.5137e-07,
    1396.0: 6.1338e-09,
    1397.0: 4.7605e-05,
    1398.0: 1.2329e-03,
    1399.0: 7.8835e-04,
    1400.0: 3.1513e-09,
    1401.0: 1.0219e-08,
    1402.0: 1.7817e-03,
    1403.0: 2.3108e-03,
    1404.0: 7.1755e-04,
    1405.0: 3.5395e-07,
    1406.0: 1.9861e-03,
    1407.0: 1.6957e-04,
    1408.0: 1.6023e-03,
    1409.0: 6.0159e-04,
    1410.0: 4.5332e-04,
    1411.0: 2.0544e-03,
    1412.0: 2.5650e-03,
    1413.0: 2.2690e-02,
    1414.0: 3.5359e-04,
    1415.0: 1.7854e-04,
    1416.0: 3.4561e-02,
    1417.0: 1.1431e-02,
    1418.0: 1.3182e-02,
    1419.0: 2.0851e-03,
    1420.0: 8.0437e-03,
    1421.0: 8.9117e-03,
    1422.0: 4.5023e-02,
    1423.0: 8.9676e-03,
    1424.0: 1.6511e-02,
    1425.0: 2.5142e-02,
    1426.0: 2.7032e-02,
    1427.0: 4.8180e-02,
    1428.0: 4.4360e-03,
    1429.0: 3.6985e-02,
    1430.0: 5.9912e-02,
    1431.0: 4.8792e-02,
    1432.0: 2.4524e-03,
    1433.0: 3.4870e-02,
    1434.0: 2.0403e-02,
    1435.0: 2.0847e-02,
    1436.0: 3.7326e-02,
    1437.0: 2.9085e-02,
    1438.0: 1.2913e-02,
    1439.0: 4.9672e-02,
    1440.0: 3.8547e-02,
    1441.0: 3.0959e-02,
    1442.0: 3.5356e-02,
    1443.0: 4.3868e-02,
    1444.0: 6.0143e-02,
    1445.0: 4.8434e-02,
    1446.0: 2.2492e-02,
    1447.0: 3.5265e-02,
    1448.0: 1.1254e-01,
    1449.0: 9.9374e-02,
    1450.0: 2.6699e-02,
    1451.0: 1.0980e-02,
    1452.0: 6.0718e-02,
    1453.0: 7.9803e-02,
    1454.0: 1.3384e-01,
    1455.0: 6.4409e-02,
    1456.0: 8.6158e-02,
    1457.0: 1.1386e-01,
    1458.0: 1.3273e-01,
    1459.0: 1.5860e-01,
    1460.0: 8.3161e-02,
    1461.0: 8.7886e-02,
    1462.0: 1.2708e-01,
    1463.0: 4.2106e-02,
    1464.0: 1.4770e-01,
    1465.0: 9.0911e-02,
    1466.0: 6.3498e-02,
    1467.0: 3.5128e-02,
    1468.0: 7.4928e-02,
    1469.0: 9.2344e-02,
    1470.0: 4.8397e-02,
    1471.0: 1.7394e-02,
    1472.0: 4.5566e-02,
    1473.0: 6.8368e-02,
    1474.0: 9.4765e-02,
    1475.0: 1.7954e-01,
    1476.0: 6.6987e-02,
    1477.0: 6.7916e-02,
    1478.0: 6.1825e-02,
    1479.0: 1.1680e-01,
    1480.0: 5.9063e-02,
    1481.0: 1.1221e-01,
    1482.0: 5.6967e-02,
    1483.0: 1.4454e-01,
    1484.0: 1.3375e-01,
    1485.0: 1.2168e-01,
    1486.0: 1.2008e-01,
    1487.0: 5.9042e-02,
    1488.0: 9.1654e-02,
    1489.0: 1.8443e-01,
    1490.0: 1.6993e-01,
    1491.0: 1.9222e-01,
    1492.0: 1.5986e-01,
    1493.0: 1.7651e-01,
    1494.0: 1.9794e-01,
    1495.0: 1.7745e-01,
    1496.0: 1.6385e-01,
    1497.0: 2.2198e-01,
    1498.0: 1.8437e-01,
    1499.0: 2.1141e-01,
    1500.0: 2.4339e-01,
    1501.0: 2.5782e-01,
    1502.0: 2.2688e-01,
    1503.0: 1.7972e-01,
    1504.0: 1.5586e-01,
    1505.0: 1.7885e-01,
    1506.0: 2.5026e-01,
    1507.0: 2.4779e-01,
    1508.0: 2.3606e-01,
    1509.0: 1.8169e-01,
    1510.0: 2.6269e-01,
    1511.0: 2.5710e-01,
    1512.0: 2.5315e-01,
    1513.0: 2.3544e-01,
    1514.0: 2.1930e-01,
    1515.0: 2.5804e-01,
    1516.0: 2.4943e-01,
    1517.0: 2.4214e-01,
    1518.0: 2.4489e-01,
    1519.0: 2.3739e-01,
    1520.0: 2.5688e-01,
    1521.0: 2.6707e-01,
    1522.0: 2.5617e-01,
    1523.0: 2.7192e-01,
    1524.0: 2.6743e-01,
    1525.0: 2.5140e-01,
    1526.0: 2.5977e-01,
    1527.0: 2.5468e-01,
    1528.0: 2.7122e-01,
    1529.0: 2.6460e-01,
    1530.0: 2.4789e-01,
    1531.0: 2.6200e-01,
    1532.0: 2.7039e-01,
    1533.0: 2.6918e-01,
    1534.0: 2.6122e-01,
    1535.0: 2.5924e-01,
    1536.0: 2.6679e-01,
    1537.0: 2.6558e-01,
    1538.0: 2.6427e-01,
    1539.0: 2.6519e-01,
    1540.0: 2.5737e-01,
    1541.0: 2.6141e-01,
    1542.0: 2.6165e-01,
    1543.0: 2.6439e-01,
    1544.0: 2.6443e-01,
    1545.0: 2.6922e-01,
    1546.0: 2.6705e-01,
    1547.0: 2.6537e-01,
    1548.0: 2.5920e-01,
    1549.0: 2.6568e-01,
    1550.0: 2.6226e-01,
    1551.0: 2.6293e-01,
    1552.0: 2.6415e-01,
    1553.0: 2.6366e-01,
    1554.0: 2.5727e-01,
    1555.0: 2.6005e-01,
    1556.0: 2.5569e-01,
    1557.0: 2.6301e-01,
    1558.0: 2.6093e-01,
    1559.0: 2.6054e-01,
    1560.0: 2.5821e-01,
    1561.0: 2.6242e-01,
    1562.0: 2.6003e-01,
    1563.0: 2.5917e-01,
    1564.0: 2.5525e-01,
    1565.0: 2.5975e-01,
    1566.0: 2.5506e-01,
    1567.0: 2.5566e-01,
    1568.0: 2.4997e-01,
    1569.0: 2.4736e-01,
    1570.0: 2.3497e-01,
    1571.0: 2.2850e-01,
    1572.0: 2.3108e-01,
    1573.0: 2.2750e-01,
    1574.0: 2.3464e-01,
    1575.0: 2.3294e-01,
    1576.0: 2.3980e-01,
    1577.0: 2.0994e-01,
    1578.0: 2.2854e-01,
    1579.0: 2.3005e-01,
    1580.0: 2.3772e-01,
    1581.0: 2.4165e-01,
    1582.0: 2.3508e-01,
    1583.0: 2.4051e-01,
    1584.0: 2.4194e-01,
    1585.0: 2.5135e-01,
    1586.0: 2.4838e-01,
    1587.0: 2.4579e-01,
    1588.0: 2.4388e-01,
    1589.0: 2.2567e-01,
    1590.0: 2.3486e-01,
    1591.0: 2.3503e-01,
    1592.0: 2.4502e-01,
    1593.0: 2.5092e-01,
    1594.0: 2.4890e-01,
    1595.0: 2.5083e-01,
    1596.0: 2.3751e-01,
    1597.0: 2.3985e-01,
    1598.0: 2.4693e-01,
    1599.0: 2.3511e-01,
    1600.0: 2.3133e-01,
    1601.0: 2.1691e-01,
    1602.0: 2.1780e-01,
    1603.0: 2.1765e-01,
    1604.0: 2.2197e-01,
    1605.0: 2.3010e-01,
    1606.0: 2.3453e-01,
    1607.0: 2.2637e-01,
    1608.0: 2.2343e-01,
    1609.0: 2.2088e-01,
    1610.0: 2.1146e-01,
    1611.0: 2.2039e-01,
    1612.0: 2.2422e-01,
    1613.0: 2.3047e-01,
    1614.0: 2.3162e-01,
    1615.0: 2.3420e-01,
    1616.0: 2.2395e-01,
    1617.0: 2.2798e-01,
    1618.0: 2.3660e-01,
    1619.0: 2.3415e-01,
    1620.0: 2.2783e-01,
    1621.0: 2.2765e-01,
    1622.0: 2.3081e-01,
    1623.0: 2.3559e-01,
    1624.0: 2.3582e-01,
    1625.0: 2.3109e-01,
    1626.0: 2.3294e-01,
    1627.0: 2.3398e-01,
    1628.0: 2.3446e-01,
    1629.0: 2.3456e-01,
    1630.0: 2.2984e-01,
    1631.0: 2.3136e-01,
    1632.0: 2.3151e-01,
    1633.0: 2.2614e-01,
    1634.0: 2.2628e-01,
    1635.0: 2.2712e-01,
    1636.0: 2.2879e-01,
    1637.0: 2.2064e-01,
    1638.0: 2.1393e-01,
    1639.0: 2.1410e-01,
    1640.0: 2.0913e-01,
    1641.0: 2.1348e-01,
    1642.0: 2.1467e-01,
    1643.0: 2.0939e-01,
    1644.0: 2.1733e-01,
    1645.0: 2.1216e-01,
    1646.0: 2.1147e-01,
    1647.0: 2.2135e-01,
    1648.0: 2.1057e-01,
    1649.0: 2.1261e-01,
    1650.0: 2.1902e-01,
    1651.0: 2.0281e-01,
    1652.0: 2.1754e-01,
    1653.0: 2.1661e-01,
    1654.0: 2.0991e-01,
    1655.0: 2.1619e-01,
    1656.0: 2.1494e-01,
    1657.0: 2.1613e-01,
    1658.0: 2.1870e-01,
    1659.0: 2.1514e-01,
    1660.0: 2.1721e-01,
    1661.0: 2.1774e-01,
    1662.0: 2.1313e-01,
    1663.0: 2.1630e-01,
    1664.0: 2.1498e-01,
    1665.0: 2.0607e-01,
    1666.0: 1.7411e-01,
    1667.0: 2.0502e-01,
    1668.0: 2.0881e-01,
    1669.0: 2.0939e-01,
    1670.0: 2.1573e-01,
    1671.0: 2.1294e-01,
    1672.0: 2.0582e-01,
    1673.0: 2.1052e-01,
    1674.0: 2.1002e-01,
    1675.0: 2.0793e-01,
    1676.0: 2.0584e-01,
    1677.0: 2.0668e-01,
    1678.0: 2.0365e-01,
    1679.0: 2.0741e-01,
    1680.0: 2.0017e-01,
    1681.0: 1.8936e-01,
    1682.0: 1.9830e-01,
    1683.0: 2.0357e-01,
    1684.0: 1.9283e-01,
    1685.0: 2.0763e-01,
    1686.0: 2.0476e-01,
    1687.0: 1.9951e-01,
    1688.0: 2.0465e-01,
    1689.0: 2.0178e-01,
    1690.0: 1.9991e-01,
    1691.0: 1.8808e-01,
    1692.0: 2.0174e-01,
    1693.0: 2.0587e-01,
    1694.0: 1.9950e-01,
    1695.0: 2.0427e-01,
    1696.0: 2.0383e-01,
    1697.0: 1.7649e-01,
    1698.0: 2.0207e-01,
    1699.0: 2.0024e-01,
    1700.0: 1.9464e-01,
    1702.0: 1.9874e-01,
    1705.0: 1.9275e-01,
    1710.0: 1.8316e-01,
    1715.0: 1.8490e-01,
    1720.0: 1.8231e-01,
    1725.0: 1.7367e-01,
    1730.0: 1.6979e-01,
    1735.0: 1.5758e-01,
    1740.0: 1.6405e-01,
    1745.0: 1.5105e-01,
    1750.0: 1.6162e-01,
    1755.0: 1.4931e-01,
    1760.0: 1.5608e-01,
    1765.0: 1.2967e-01,
    1770.0: 1.3831e-01,
    1775.0: 1.1213e-01,
    1780.0: 9.8143e-02,
    1785.0: 7.5201e-02,
    1790.0: 8.6831e-02,
    1795.0: 4.5864e-02,
    1800.0: 3.1112e-02,
    1805.0: 1.4485e-02,
    1810.0: 9.4762e-03,
    1815.0: 3.2093e-03,
    1820.0: 9.6578e-04,
    1825.0: 1.2463e-03,
    1830.0: 5.0896e-06,
    1835.0: 6.2784e-06,
    1840.0: 6.1337e-08,
    1845.0: 6.1298e-06,
    1850.0: 2.9348e-06,
    1855.0: 2.7795e-07,
    1860.0: 1.0920e-05,
    1865.0: 1.6644e-05,
    1870.0: 2.6148e-10,
    1875.0: 4.4296e-10,
    1880.0: 7.6123e-05,
    1885.0: 4.3129e-05,
    1890.0: 2.1956e-04,
    1895.0: 1.2743e-04,
    1900.0: 8.4916e-07,
    1905.0: 5.5798e-07,
    1910.0: 2.2726e-05,
    1915.0: 1.9673e-05,
    1920.0: 4.4451e-04,
    1925.0: 9.2326e-04,
    1930.0: 5.4474e-04,
    1935.0: 3.5428e-03,
    1940.0: 3.2357e-03,
    1945.0: 1.0707e-02,
    1950.0: 1.6482e-02,
    1955.0: 9.8860e-03,
    1960.0: 2.1569e-02,
    1965.0: 2.8114e-02,
    1970.0: 4.8055e-02,
    1975.0: 6.6730e-02,
    1980.0: 7.4234e-02,
    1985.0: 8.1625e-02,
    1990.0: 8.4124e-02,
    1995.0: 7.9787e-02,
    2000.0: 3.7491e-02,
    2005.0: 1.4747e-02,
    2010.0: 3.9071e-02,
    2015.0: 2.6208e-02,
    2020.0: 4.4239e-02,
    2025.0: 7.2779e-02,
    2030.0: 8.3460e-02,
    2035.0: 9.4808e-02,
    2040.0: 8.8344e-02,
    2045.0: 8.9636e-02,
    2050.0: 6.6892e-02,
    2055.0: 5.4090e-02,
    2060.0: 6.8157e-02,
    2065.0: 6.0962e-02,
    2070.0: 6.4715e-02,
    2075.0: 7.6305e-02,
    2080.0: 8.5528e-02,
    2085.0: 8.3847e-02,
    2090.0: 8.7779e-02,
    2095.0: 8.8421e-02,
    2100.0: 8.4869e-02,
    2105.0: 9.1771e-02,
    2110.0: 8.8320e-02,
    2115.0: 9.0308e-02,
    2120.0: 8.6281e-02,
    2125.0: 8.7303e-02,
    2130.0: 8.8422e-02,
    2135.0: 8.8679e-02,
    2140.0: 8.9390e-02,
    2145.0: 8.8132e-02,
    2150.0: 8.3369e-02,
    2155.0: 8.3566e-02,
    2160.0: 8.2912e-02,
    2165.0: 7.5175e-02,
    2170.0: 8.0776e-02,
    2175.0: 7.9257e-02,
    2180.0: 8.0597e-02,
    2185.0: 7.3458e-02,
    2190.0: 7.7905e-02,
    2195.0: 7.7833e-02,
    2200.0: 7.0175e-02,
    2205.0: 7.2947e-02,
    2210.0: 7.8174e-02,
    2215.0: 7.5189e-02,
    2220.0: 7.6631e-02,
    2225.0: 7.4400e-02,
    2230.0: 7.4727e-02,
    2235.0: 7.3290e-02,
    2240.0: 7.2140e-02,
    2245.0: 6.9911e-02,
    2250.0: 7.1034e-02,
    2255.0: 6.6865e-02,
    2260.0: 6.6143e-02,
    2265.0: 6.7355e-02,
    2270.0: 6.4138e-02,
    2275.0: 6.3309e-02,
    2280.0: 6.5551e-02,
    2285.0: 6.2389e-02,
    2290.0: 6.2534e-02,
    2295.0: 6.0603e-02,
    2300.0: 5.8193e-02,
    2305.0: 5.8544e-02,
    2310.0: 6.3189e-02,
    2315.0: 5.7528e-02,
    2320.0: 5.1489e-02,
    2325.0: 5.5626e-02,
    2330.0: 5.6231e-02,
    2335.0: 5.7362e-02,
    2340.0: 4.5366e-02,
    2345.0: 5.0869e-02,
    2350.0: 4.1115e-02,
    2355.0: 4.6988e-02,
    2360.0: 4.9724e-02,
    2365.0: 4.8909e-02,
    2370.0: 3.0514e-02,
    2375.0: 4.3704e-02,
    2380.0: 4.2128e-02,
    2385.0: 3.0525e-02,
    2390.0: 3.6748e-02,
    2395.0: 4.0199e-02,
    2400.0: 4.3726e-02,
    2405.0: 3.3286e-02,
    2410.0: 3.3504e-02,
    2415.0: 2.7058e-02,
    2420.0: 2.6358e-02,
    2425.0: 3.2802e-02,
    2430.0: 4.4725e-02,
    2435.0: 1.4765e-02,
    2440.0: 4.2926e-02,
    2445.0: 2.0657e-02,
    2450.0: 1.3523e-02,
    2455.0: 2.4695e-02,
    2460.0: 3.3157e-02,
    2465.0: 2.4009e-02,
    2470.0: 1.6635e-02,
    2475.0: 1.6368e-02,
    2480.0: 7.9996e-03,
    2485.0: 5.5840e-03,
    2490.0: 3.4957e-03,
    2495.0: 2.8647e-03,
    2500.0: 7.0328e-03,
    2505.0: 1.5124e-03,
    2510.0: 2.2063e-03,
    2515.0: 5.1644e-04,
    2520.0: 3.6879e-04,
    2525.0: 4.1194e-05,
    2530.0: 6.3279e-07,
    2535.0: 1.7415e-07,
    2540.0: 3.7521e-07,
    2545.0: 5.3469e-11,
    2550.0: 2.8066e-13,
    2555.0: 1.0377e-09,
    2560.0: 3.0842e-11,
    2565.0: 1.5846e-14,
    2570.0: 1.5151e-18,
    2575.0: 1.0708e-27,
    2580.0: 3.7933e-22,
    2585.0: 1.7064e-34,
    2590.0: 5.4369e-31,
    2595.0: 2.2666e-33,
    2600.0: 4.4556e-28,
    2605.0: 5.7592e-35,
    2610.0: 5.8970e-34,
    2615.0: 1.1106e-37,
    2620.0: 5.6056e-29,
    2625.0: 3.8378e-28,
    2630.0: 2.8026e-45,
    2635.0: 3.8719e-16,
    2640.0: 1.1657e-16,
    2645.0: 8.9292e-19,
    2650.0: 1.4186e-19,
    2655.0: 1.3036e-27,
    2660.0: 2.5880e-25,
    2665.0: 1.1045e-37,
    2670.0: 0.0000e+00,
    2675.0: 0.0000e+00,
    2680.0: 0.0000e+00,
    2685.0: 0.0000e+00,
    2690.0: 1.0178e-29,
    2695.0: 7.1014e-33,
    2700.0: 0.0000e+00,
    2705.0: 2.9273e-42,
    2710.0: 1.1239e-35,
    2715.0: 3.8549e-26,
    2720.0: 5.6052e-45,
    2725.0: 7.3094e-22,
    2730.0: 6.0929e-19,
    2735.0: 5.5121e-21,
    2740.0: 2.3435e-27,
    2745.0: 1.3224e-23,
    2750.0: 1.6758e-28,
    2755.0: 6.7262e-44,
    2760.0: 0.0000e+00,
    2765.0: 2.7001e-27,
    2770.0: 8.4528e-24,
    2775.0: 4.0360e-38,
    2780.0: 4.8532e-34,
    2785.0: 3.9255e-27,
    2790.0: 1.2295e-16,
    2795.0: 3.6591e-16,
    2800.0: 1.6665e-12,
    2805.0: 6.8228e-14,
    2810.0: 4.0695e-10,
    2815.0: 2.9018e-10,
    2820.0: 2.0789e-11,
    2825.0: 1.7814e-07,
    2830.0: 3.9475e-06,
    2835.0: 2.1533e-10,
    2840.0: 1.9849e-07,
    2845.0: 4.1074e-05,
    2850.0: 1.1708e-06,
    2855.0: 4.5425e-07,
    2860.0: 2.5672e-05,
    2865.0: 1.6974e-04,
    2870.0: 6.3922e-06,
    2875.0: 3.9663e-04,
    2880.0: 2.5037e-04,
    2885.0: 4.5906e-04,
    2890.0: 1.8860e-04,
    2895.0: 2.6982e-03,
    2900.0: 8.2183e-04,
    2905.0: 1.1237e-04,
    2910.0: 2.7566e-03,
    2915.0: 1.2742e-03,
    2920.0: 2.9316e-03,
    2925.0: 1.0973e-03,
    2930.0: 5.9606e-03,
    2935.0: 6.5725e-03,
    2940.0: 1.6479e-03,
    2945.0: 1.4673e-03,
    2950.0: 5.2935e-03,
    2955.0: 2.3655e-03,
    2960.0: 4.6549e-03,
    2965.0: 7.5313e-03,
    2970.0: 3.5676e-04,
    2975.0: 8.6499e-04,
    2980.0: 1.3548e-03,
    2985.0: 7.0495e-03,
    2990.0: 1.0407e-02,
    2995.0: 4.3286e-03,
    3000.0: 7.9442e-03,
    3005.0: 2.9261e-03,
    3010.0: 6.9320e-03,
    3015.0: 5.6232e-03,
    3020.0: 6.4143e-04,
    3025.0: 7.5948e-03,
    3030.0: 6.1491e-03,
    3035.0: 2.5290e-03,
    3040.0: 2.0487e-03,
    3045.0: 4.2599e-03,
    3050.0: 1.0446e-03,
    3055.0: 2.9295e-04,
    3060.0: 6.3761e-03,
    3065.0: 2.9458e-03,
    3070.0: 1.7699e-03,
    3075.0: 6.0928e-03,
    3080.0: 3.6646e-03,
    3085.0: 1.7876e-03,
    3090.0: 2.4080e-03,
    3095.0: 6.6266e-04,
    3100.0: 4.4513e-03,
    3105.0: 9.3203e-04,
    3110.0: 8.5524e-04,
    3115.0: 2.2932e-03,
    3120.0: 9.9294e-03,
    3125.0: 3.0625e-03,
    3130.0: 5.8246e-03,
    3135.0: 1.1571e-02,
    3140.0: 3.3603e-03,
    3145.0: 3.2869e-03,
    3150.0: 6.7457e-03,
    3155.0: 5.6965e-03,
    3160.0: 9.3294e-03,
    3165.0: 1.4163e-02,
    3170.0: 1.2645e-02,
    3175.0: 9.3252e-03,
    3180.0: 1.0730e-02,
    3185.0: 8.1644e-03,
    3190.0: 4.2817e-03,
    3195.0: 2.7197e-03,
    3200.0: 4.4280e-04,
    3205.0: 3.1280e-04,
    3210.0: 1.3768e-04,
    3215.0: 5.0237e-04,
    3220.0: 1.6245e-03,
    3225.0: 2.0066e-04,
    3230.0: 3.4404e-04,
    3235.0: 7.3625e-03,
    3240.0: 3.7816e-03,
    3245.0: 7.4096e-04,
    3250.0: 2.6308e-03,
    3255.0: 1.0029e-02,
    3260.0: 1.2360e-03,
    3265.0: 2.4686e-03,
    3270.0: 1.2292e-03,
    3275.0: 5.9758e-03,
    3280.0: 2.8890e-03,
    3285.0: 1.1223e-02,
    3290.0: 8.8310e-03,
    3295.0: 1.2337e-03,
    3300.0: 1.7944e-03,
    3305.0: 3.9746e-03,
    3310.0: 3.9565e-03,
    3315.0: 1.6269e-05,
    3320.0: 6.0496e-05,
    3325.0: 3.5486e-03,
    3330.0: 4.7014e-03,
    3335.0: 9.1470e-03,
    3340.0: 3.4903e-03,
    3345.0: 3.5722e-03,
    3350.0: 8.0998e-03,
    3355.0: 3.6642e-03,
    3360.0: 5.2892e-03,
    3365.0: 7.2591e-03,
    3370.0: 3.9770e-03,
    3375.0: 8.5388e-03,
    3380.0: 5.1624e-03,
    3385.0: 7.5650e-03,
    3390.0: 9.9552e-03,
    3395.0: 9.6446e-03,
    3400.0: 1.2638e-02,
    3405.0: 4.5061e-03,
    3410.0: 7.1547e-03,
    3415.0: 7.3543e-03,
    3420.0: 1.3305e-02,
    3425.0: 1.0112e-02,
    3430.0: 8.7810e-03,
    3435.0: 1.1673e-02,
    3440.0: 8.1181e-03,
    3445.0: 1.1434e-02,
    3450.0: 1.1267e-02,
    3455.0: 8.3938e-03,
    3460.0: 1.2657e-02,
    3465.0: 9.9163e-03,
    3470.0: 1.2386e-02,
    3475.0: 1.1051e-02,
    3480.0: 1.1335e-02,
    3485.0: 1.2212e-02,
    3490.0: 1.0520e-02,
    3495.0: 1.2383e-02,
    3500.0: 1.2032e-02,
    3505.0: 1.1923e-02,
    3510.0: 1.2077e-02,
    3515.0: 1.1598e-02,
    3520.0: 1.2226e-02,
    3525.0: 1.1523e-02,
    3530.0: 1.1217e-02,
    3535.0: 9.5305e-03,
    3540.0: 9.1011e-03,
    3545.0: 9.6154e-03,
    3550.0: 1.0616e-02,
    3555.0: 9.1245e-03,
    3560.0: 1.0872e-02,
    3565.0: 1.0928e-02,
    3570.0: 8.3951e-03,
    3575.0: 8.7028e-03,
    3580.0: 1.0255e-02,
    3585.0: 9.2282e-03,
    3590.0: 9.5145e-03,
    3595.0: 9.7329e-03,
    3600.0: 1.0328e-02,
    3605.0: 1.0425e-02,
    3610.0: 9.5385e-03,
    3615.0: 9.5318e-03,
    3620.0: 1.1686e-02,
    3625.0: 1.0302e-02,
    3630.0: 1.0016e-02,
    3635.0: 1.0362e-02,
    3640.0: 1.1549e-02,
    3645.0: 1.0663e-02,
    3650.0: 1.0183e-02,
    3655.0: 1.1042e-02,
    3660.0: 1.0978e-02,
    3665.0: 1.0313e-02,
    3670.0: 7.9464e-03,
    3675.0: 4.8569e-03,
    3680.0: 8.3789e-03,
    3685.0: 9.4906e-03,
    3690.0: 9.7458e-03,
    3695.0: 1.0188e-02,
    3700.0: 1.0937e-02,
    3705.0: 1.0827e-02,
    3710.0: 9.4130e-03,
    3715.0: 9.2731e-03,
    3720.0: 1.0429e-02,
    3725.0: 1.0751e-02,
    3730.0: 9.3169e-03,
    3735.0: 8.6260e-03,
    3740.0: 8.8923e-03,
    3745.0: 1.0380e-02,
    3750.0: 9.3341e-03,
    3755.0: 9.0336e-03,
    3760.0: 8.9041e-03,
    3765.0: 8.5891e-03,
    3770.0: 9.1649e-03,
    3775.0: 9.0918e-03,
    3780.0: 9.6158e-03,
    3785.0: 8.8498e-03,
    3790.0: 7.7891e-03,
    3795.0: 8.9056e-03,
    3800.0: 9.8988e-03,
    3805.0: 9.3414e-03,
    3810.0: 8.2771e-03,
    3815.0: 7.7867e-03,
    3820.0: 9.6908e-03,
    3825.0: 9.5402e-03,
    3830.0: 9.6268e-03,
    3835.0: 7.7190e-03,
    3840.0: 9.0066e-03,
    3845.0: 8.8097e-03,
    3850.0: 8.8569e-03,
    3855.0: 8.5365e-03,
    3860.0: 8.0197e-03,
    3865.0: 8.1248e-03,
    3870.0: 7.3839e-03,
    3875.0: 6.7832e-03,
    3880.0: 6.5543e-03,
    3885.0: 6.7926e-03,
    3890.0: 6.9028e-03,
    3895.0: 7.4983e-03,
    3900.0: 7.9487e-03,
    3905.0: 7.9500e-03,
    3910.0: 7.1560e-03,
    3915.0: 7.0070e-03,
    3920.0: 6.9665e-03,
    3925.0: 6.8714e-03,
    3930.0: 7.0699e-03,
    3935.0: 7.3743e-03,
    3940.0: 7.4228e-03,
    3945.0: 7.5613e-03,
    3950.0: 7.6475e-03,
    3955.0: 7.7399e-03,
    3960.0: 7.7679e-03,
    3965.0: 7.8253e-03,
    3970.0: 7.6997e-03,
    3975.0: 7.5280e-03,
    3980.0: 7.4049e-03,
    3985.0: 7.4503e-03,
    3990.0: 7.3894e-03,
    3995.0: 7.2263e-03,
    4000.0: 7.1199e-03,
}
# yapf: enable

ASTMG173_ETR = SpectralDistribution(
    ASTMG173_ETR_DATA, name='ASTM G-173 ETR', interpolator=LinearInterpolator)
"""
Extraterrestrial Radiation (solar spectrum at top of atmosphere) at mean
Earth-Sun distance.

References
----------
:cite:`RenewableResourceDataCenter2003a`

ASTMG173_ETR : SpectralDistribution
"""

ASTMG173_GLOBAL_TILT = SpectralDistribution(
    ASTMG173_GLOBAL_TILT_DATA,
    name='ASTM G-173 Global Tilt',
    interpolator=LinearInterpolator)
"""
Spectral radiation from solar disk plus sky diffuse and diffuse reflected
from ground on south facing surface tilted 37 degrees from horizontal.

References
----------
:cite:`RenewableResourceDataCenter2003a`

ASTMG173_GLOBAL_TILT : SpectralDistribution
"""

ASTMG173_DIRECT_CIRCUMSOLAR = SpectralDistribution(
    ASTMG173_DIRECT_CIRCUMSOLAR_DATA,
    name='ASTM G-173 Direct + Circumsolar',
    interpolator=LinearInterpolator)
"""
Direct Normal Irradiance Nearly parallel (0.5 degrees divergent cone)
radiation on surface with surface normal tracking (pointing to) the sun,
excluding scattered sky and reflected ground radiation plus spectral
irradiance within +/- 2.5 degrees (5 degrees diameter) field of view
centered on the 0.5 degrees diameter solar disk, but excluding the
radiation from the disk.

References
----------
:cite:`RenewableResourceDataCenter2003a`

ASTMG173_DIRECT_CIRCUMSOLAR : SpectralDistribution
"""
