# Rules

## Rule levels
A rule returns feedback on a specific level: Error, Warning or Informational.

### Errors
Errors will start with the letter `E`. Errors will (or can) result in a hard failure for the template being validated.

### Warnings
Warnings start with the letter `W`. Warnings alert you when the template doesn't follow best practices but should still function.  *Example: If you use a parameter for a RDS master password you should have the parameter property NoEcho set to true.*

### Informational
Informational results start with the letter `I`. Informational alert you when the template doesn't follow best practices, just like the Warnings, but in a non-blocking way. The Informational level is disabled by default and has to be enabled explicitly using `-c I` or `--include-checks I`.

## Categories

| Rule Numbers          | Category |
| --------------------- | ------------- |
| (E&#124;W&#124;I)0XXX | Basic Template Errors. Examples: Not parseable, main sections (Outputs, Resources, etc.) |
| (E&#124;W&#124;I)1XXX | Functions (Ref, GetAtt, etc.) |
| (E&#124;W&#124;I)2XXX | Parameters |
| (E&#124;W&#124;I)3XXX | Resources |
| (E&#124;W&#124;I)4XXX | Metadata |
| (E&#124;W&#124;I)6xxx | Outputs |
| (E&#124;W&#124;I)7xxx | Mappings |
| (E&#124;W&#124;I)8xxx | Conditions |
| (E&#124;W&#124;I)9xxx | Reserved for users rules |

*Warning*
Rule `E3012` is used to check the types for value of a resource property. A number is a number, string is a string, etc.  There are occasions where this could be just a warning and other times it could be an error. cfn-lint doesn't have an exception process so all instances of this issue are considered errors. You can disable this rule using `--ignore-checks` if it is not required for your internal best practices. Conversely, strict typing is **NOT** enforced by default for this rule, so if strict adherence to resource value types is necessary for your use case, you can use `--configure-rule E3012:strict=true` to enforce the rule.

## Experimental rules
Sometimes there are (new) rules that might be complex, that doesn't have enough solid test templates and examples and/or might have unexpected results. We support adding in these rules so they can be tested, tweaked and improved before they become generally available.

A rule can be marked as experimental by setting the `experimental` flag in the Rule's MetaData:

```python
class MyExperimentalRule(CloudFormationLintRule):
    id = 'E1234'
    ...
    experimental = True
```

To include these rules, use the `-e/include-experimental` argument when running cfn-lint.

## Rules
(_This documentation is generated by running `cfn-lint --update-documentation`, do not alter this manually_)

The following **159** rules are applied by this linter:

| Rule ID  | Title | Description | Config<br />(Name:Type:Default) | Source | Tags |
| -------- | ----- | ----------- | ---------- | ------ | ---- |
| [E0000<a name="E0000"></a>](../src/cfnlint/rules/__init__.py) | Parsing error found when parsing the template | Checks for JSON/YAML formatting errors in your template |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `base` |
| [E0001<a name="E0001"></a>](../src/cfnlint/rules/__init__.py) | Error found when transforming the template | Errors found when performing transformation on the template |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `base`,`transform` |
| [E0002<a name="E0002"></a>](../src/cfnlint/rules/__init__.py) | Error processing rule on the template | Errors found when processing a rule on the template |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `base`,`rule` |
| [E1001<a name="E1001"></a>](../src/cfnlint/rules/templates/Base.py) | Basic CloudFormation Template Configuration | Making sure the basic CloudFormation template components are properly configured | sections:string: | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `base` |
| [E1002<a name="E1002"></a>](../src/cfnlint/rules/templates/LimitSize.py) | Template size limit | Check the size of the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `limits` |
| [E1003<a name="E1003"></a>](../src/cfnlint/rules/templates/LimitDescription.py) | Template description limit | Check if the size of the template description is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `description`,`limits` |
| [E1004<a name="E1004"></a>](../src/cfnlint/rules/templates/Description.py) | Template description can only be a string | Template description can only be a string |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-description-structure.html) | `description` |
| [E1010<a name="E1010"></a>](../src/cfnlint/rules/functions/GetAtt.py) | GetAtt validation of parameters | Validates that GetAtt parameters are to valid resources and properties of those resources |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-getatt.html) | `functions`,`getatt` |
| [E1011<a name="E1011"></a>](../src/cfnlint/rules/functions/FindInMap.py) | FindInMap validation of configuration | Making sure the function is a list of appropriate config |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-findinmap.html) | `functions`,`findinmap` |
| [E1012<a name="E1012"></a>](../src/cfnlint/rules/functions/RefExist.py) | Check if Refs exist | Making sure the refs exist |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-ref.html) | `functions`,`ref` |
| [E1015<a name="E1015"></a>](../src/cfnlint/rules/functions/GetAz.py) | GetAz validation of parameters | Making sure the GetAz function is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-getavailabilityzones.html) | `functions`,`getaz` |
| [E1016<a name="E1016"></a>](../src/cfnlint/rules/functions/ImportValue.py) | ImportValue validation of parameters | Making sure the ImportValue function is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-importvalue.html) | `functions`,`importvalue` |
| [E1017<a name="E1017"></a>](../src/cfnlint/rules/functions/Select.py) | Select validation of parameters | Making sure the Select function is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-select.html) | `functions`,`select` |
| [E1018<a name="E1018"></a>](../src/cfnlint/rules/functions/Split.py) | Split validation of parameters | Making sure the split function is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-split.html) | `functions`,`split` |
| [E1019<a name="E1019"></a>](../src/cfnlint/rules/functions/Sub.py) | Sub validation of parameters | Making sure the sub function is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-sub.html) | `functions`,`sub` |
| [E1020<a name="E1020"></a>](../src/cfnlint/rules/functions/Ref.py) | Ref validation of value | Making sure the Ref has a String value (no other functions are supported) |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-ref.html) | `functions`,`ref` |
| [E1021<a name="E1021"></a>](../src/cfnlint/rules/functions/Base64.py) | Base64 validation of parameters | Making sure the Base64 function is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-base64.html) | `functions`,`base64` |
| [E1022<a name="E1022"></a>](../src/cfnlint/rules/functions/Join.py) | Join validation of parameters | Making sure the join function is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-join.html) | `functions`,`join` |
| [E1023<a name="E1023"></a>](../src/cfnlint/rules/functions/Not.py) | Validation NOT function configuration | Making sure that NOT functions are list |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-not) | `functions`,`not` |
| [E1024<a name="E1024"></a>](../src/cfnlint/rules/functions/Cidr.py) | Cidr validation of parameters | Making sure the function CIDR is a list with valid values |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-cidr.html) | `functions`,`cidr` |
| [E1026<a name="E1026"></a>](../src/cfnlint/rules/functions/RefInCondition.py) | Cannot reference resources in the Conditions block of the template | Check that any Refs in the Conditions block uses no resources |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#w2ab2c21c28c21c45) | `conditions`,`functions`,`ref` |
| [E1027<a name="E1027"></a>](../src/cfnlint/rules/functions/DynamicReferenceSecureString.py) | Check dynamic references secure strings are in supported locations | Dynamic References Secure Strings are only supported for a small set of resource properties.  Validate that they are being used in the correct location when checking values and Fn::Sub in resource properties. Currently doesn't check outputs, maps, conditions, parameters, and descriptions. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html) | `functions`,`dynamic reference` |
| [E1028<a name="E1028"></a>](../src/cfnlint/rules/functions/If.py) | Check Fn::If structure for validity | Check Fn::If to make sure its valid.  Condition has to be a string. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-if) | `functions`,`if` |
| [E1029<a name="E1029"></a>](../src/cfnlint/rules/functions/SubNeeded.py) | Sub is required if a variable is used in a string | If a substitution variable exists in a string but isn't wrapped with the Fn::Sub function the deployment will fail. | custom_excludes:string: | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-sub.html) | `functions`,`sub` |
| [E1030<a name="E1030"></a>](../src/cfnlint/rules/functions/Length.py) | Length validation of parameters | Making sure Fn::Length is configured correctly |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-length.html) | `functions`,`length` |
| [E1031<a name="E1031"></a>](../src/cfnlint/rules/functions/ToJsonString.py) | ToJsonString validation of parameters | Making sure Fn::ToJsonString is configured correctly |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference.html) | `functions`,`toJsonString` |
| [E1032<a name="E1032"></a>](../src/cfnlint/rules/functions/ForEach.py) | Validates ForEach functions | Validates that ForEach parameters have a valid configuration |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-getatt.html) | `functions`,`foreach` |
| [E2001<a name="E2001"></a>](../src/cfnlint/rules/parameters/Configuration.py) | Parameters have appropriate properties | Making sure the parameters are properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html) | `parameters` |
| [E2002<a name="E2002"></a>](../src/cfnlint/rules/parameters/Types.py) | Parameters have appropriate type | Making sure the parameters have a correct type |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/best-practices.html#parmtypes) | `parameters` |
| [E2003<a name="E2003"></a>](../src/cfnlint/rules/parameters/Name.py) | Parameters have appropriate names | Check if Parameters are properly named (A-Za-z0-9) |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html#parameters-section-structure-requirements) | `parameters` |
| [E2010<a name="E2010"></a>](../src/cfnlint/rules/parameters/LimitNumber.py) | Parameter limit not exceeded | Check the number of Parameters in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `parameters`,`limits` |
| [E2011<a name="E2011"></a>](../src/cfnlint/rules/parameters/LimitName.py) | Parameter name limit not exceeded | Check the size of Parameter names in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `parameters`,`limits` |
| [E2012<a name="E2012"></a>](../src/cfnlint/rules/parameters/LimitValue.py) | Parameter value limit not exceeded | Check if the size of Parameter values in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `parameters`,`limits` |
| [E2014<a name="E2014"></a>](../src/cfnlint/rules/parameters/DefaultRef.py) | Default value cannot use Refs | Check if Refs are not used in Parameter Defaults |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html) | `parameters`,`ref` |
| [E2015<a name="E2015"></a>](../src/cfnlint/rules/parameters/Default.py) | Default value is within parameter constraints | Making sure the parameters have a default value inside AllowedValues, MinValue, MaxValue, AllowedPattern |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html) | `parameters` |
| [E2503<a name="E2503"></a>](../src/cfnlint/rules/resources/elb/Elb.py) | Resource ELB Properties | See if Elb Resource Properties are set correctly HTTPS has certificate HTTP has no certificate |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-elb-listener.html) | `properties`,`elb` |
| [E2504<a name="E2504"></a>](../src/cfnlint/rules/resources/ectwo/Ebs.py) | Check Ec2 Ebs Properties | See if Ec2 Ebs Properties are valid |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html) | `properties`,`ec2`,`ebs` |
| [E2506<a name="E2506"></a>](../src/cfnlint/rules/resources/ectwo/SecurityGroupIngress.py) | Resource EC2 Security Group Ingress Properties | See if EC2 Security Group Ingress Properties are set correctly. Check that "SourceSecurityGroupId" or "SourceSecurityGroupName" are  are exclusive and using the type of Ref or GetAtt  |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html) | `resources`,`ec2`,`securitygroup` |
| [E2507<a name="E2507"></a>](../src/cfnlint/rules/resources/iam/Policy.py) | Check if IAM Policies are properly configured | See if there elements inside an IAM policy are correct |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-policy.html) | `properties`,`iam` |
| [E2520<a name="E2520"></a>](../src/cfnlint/rules/resources/properties/Exclusive.py) | Check Properties that are mutually exclusive | Making sure CloudFormation properties that are exclusive are not defined |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `resources` |
| [E2521<a name="E2521"></a>](../src/cfnlint/rules/resources/properties/Inclusive.py) | Check Properties that are required together | Make sure CloudFormation resource properties are included together when required |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `resources` |
| [E2522<a name="E2522"></a>](../src/cfnlint/rules/resources/properties/AtLeastOne.py) | Check Properties that need at least one of a list of properties | Making sure CloudFormation properties that require at least one property from a list. More than one can be included. |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `resources` |
| [E2523<a name="E2523"></a>](../src/cfnlint/rules/resources/properties/OnlyOne.py) | Check Properties that need only one of a list of properties | Making sure CloudFormation properties that require only one property from a list. One has to be specified. |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `resources` |
| [E2529<a name="E2529"></a>](../src/cfnlint/rules/resources/lmbd/EventsLogGroupName.py) | Check for SubscriptionFilters have beyond 2 attachments to a CloudWatch Log Group | The current limit for a CloudWatch Log Group is they can have 2 subscription filters. We will look for duplicate LogGroupNames inside Subscription Filters and make sure they are within 2. This doesn't account for any other subscription filters getting set. |  | [Source](https://github.com/awslabs/serverless-application-model/blob/master/versions/2016-10-31.md#user-content-cloudwatchlogs) | `resources`,`lambda` |
| [E2530<a name="E2530"></a>](../src/cfnlint/rules/resources/lmbd/SnapStartSupported.py) | SnapStart supports the configured runtime | To properly leverage SnapStart, you must have a runtime of Java11 or greater |  | [Source](https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html) | `resources`,`lambda` |
| [E2531<a name="E2531"></a>](../src/cfnlint/rules/resources/lmbd/DeprecatedRuntimeEnd.py) | Check if EOL Lambda Function Runtimes are used | Check if an EOL Lambda Runtime is specified and give an error if used.  |  | [Source](https://docs.aws.amazon.com/lambda/latest/dg/runtime-support-policy.html) | `resources`,`lambda`,`runtime` |
| [E2532<a name="E2532"></a>](../src/cfnlint/rules/resources/stepfunctions/StateMachine.py) | Check State Machine Definition for proper syntax | Check the State Machine String Definition to make sure its JSON. Validate basic syntax of the file to determine validity. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-stepfunctions-statemachine.html) | `resources`,`stepfunctions` |
| [E2540<a name="E2540"></a>](../src/cfnlint/rules/resources/codepipeline/CodepipelineStages.py) | CodePipeline Stages | See if CodePipeline stages are set correctly |  | [Source](https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#pipeline-requirements) | `properties`,`codepipeline` |
| [E2541<a name="E2541"></a>](../src/cfnlint/rules/resources/codepipeline/CodepipelineStageActions.py) | CodePipeline Stage Actions | See if CodePipeline stage actions are set correctly |  | [Source](https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#pipeline-requirements) | `resources`,`codepipeline` |
| [E3000<a name="E3000"></a>](../src/cfnlint/rules/resources/ResourceSchema.py) | Resource schema | CloudFormation Registry resource schema validation |  | [Source](https://github.com/aws-cloudformation/aws-cloudformation-resource-schema/) | `resources` |
| [E3001<a name="E3001"></a>](../src/cfnlint/rules/resources/Configuration.py) | Basic CloudFormation Resource Check | Making sure the basic CloudFormation resources are properly configured |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `resources` |
| [E3002<a name="E3002"></a>](../src/cfnlint/rules/resources/properties/Properties.py) | Resource properties are invalid | Making sure that resources properties are properly configured |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint/blob/main/docs/cfn-resource-specification.md#properties) | `resources` |
| [E3003<a name="E3003"></a>](../src/cfnlint/rules/resources/properties/Required.py) | Required Resource properties are missing | Making sure that Resources properties that are required exist |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint/blob/main/docs/cfn-resource-specification.md#required) | `resources` |
| [E3004<a name="E3004"></a>](../src/cfnlint/rules/resources/CircularDependency.py) | Resource dependencies are not circular | Check that Resources are not circularly dependent by DependsOn, Ref, Sub, or GetAtt |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `resources`,`circularly`,`dependson`,`ref`,`sub`,`getatt` |
| [E3005<a name="E3005"></a>](../src/cfnlint/rules/resources/DependsOn.py) | Check DependsOn values for Resources | Check that the DependsOn values are valid |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html) | `resources`,`dependson` |
| [E3006<a name="E3006"></a>](../src/cfnlint/rules/resources/Name.py) | Resources have appropriate names | Check if Resources are properly named (A-Za-z0-9) |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html#resources-section-structure-logicalid) | `resources` |
| [E3007<a name="E3007"></a>](../src/cfnlint/rules/resources/UniqueNames.py) | Unique resource and parameter names | All resources and parameters must have unique names |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html) | `parameters`,`resources` |
| [E3008<a name="E3008"></a>](../src/cfnlint/rules/resources/properties/ValueRefGetAtt.py) | Check values of properties for valid Refs and GetAtts | Checks resource properties for Ref and GetAtt values |  | [Source]() | `resources`,`ref`,`getatt` |
| [E3010<a name="E3010"></a>](../src/cfnlint/rules/resources/LimitNumber.py) | Resource limit not exceeded | Check the number of Resources in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `resources`,`limits` |
| [E3011<a name="E3011"></a>](../src/cfnlint/rules/resources/LimitName.py) | Resource name limit not exceeded | Check the size of Resource names in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `resources`,`limits` |
| [E3012<a name="E3012"></a>](../src/cfnlint/rules/resources/properties/ValuePrimitiveType.py) | Check resource properties values | Checks resource property values with Primitive Types for values that match those types. | strict:boolean:False | [Source](https://github.com/aws-cloudformation/cfn-python-lint/blob/main/docs/cfn-resource-specification.md#valueprimitivetype) | `resources` |
| [E3013<a name="E3013"></a>](../src/cfnlint/rules/resources/cloudfront/Aliases.py) | CloudFront Aliases | CloudFront aliases should contain valid domain names |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-aliases) | `properties`,`cloudfront` |
| [E3016<a name="E3016"></a>](../src/cfnlint/rules/resources/updatepolicy/Configuration.py) | Check the configuration of a resources UpdatePolicy | Make sure a resources UpdatePolicy is properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatepolicy.html) | `resources`,`updatepolicy` |
| [E3017<a name="E3017"></a>](../src/cfnlint/rules/resources/properties/RequiredBasedOnValue.py) | Property is required based on another properties value | When certain properties have a certain value it results in other properties being required. This rule will validate those required properties are specified when those values are supplied |  | [Source]() | `resources` |
| [E3018<a name="E3018"></a>](../src/cfnlint/rules/resources/properties/UnwantedBasedOnValue.py) | Property is unwanted based on another properties value | When certain properties have a certain value it results in other properties not being needed. This rule will validate those unwanted properties are not specified when those values are supplied |  | [Source]() | `resources` |
| [E3020<a name="E3020"></a>](../src/cfnlint/rules/resources/route53/RecordSet.py) | Validate Route53 RecordSets | Check if all RecordSets are correctly configured |  | [Source](https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html) | `resources`,`route53`,`record_set` |
| [E3021<a name="E3021"></a>](../src/cfnlint/rules/resources/events/RuleTargetsLimit.py) | Check Events Rule Targets are less than or equal to 5 | CloudWatch Events Rule can only support up to 5 targets |  | [Source](https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/cloudwatch_limits_cwe.html) | `resources`,`events` |
| [E3022<a name="E3022"></a>](../src/cfnlint/rules/resources/ectwo/RouteTableAssociation.py) | Resource SubnetRouteTableAssociation Properties | Validate there is only one SubnetRouteTableAssociation per subnet |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-subnet-route-table-assoc.html) | `resources`,`ec2`,`subnet`,`route table` |
| [E3025<a name="E3025"></a>](../src/cfnlint/rules/resources/rds/InstanceSize.py) | RDS instance type is compatible with the RDS type | Check the RDS instance types are supported by the type of RDS engine. Only if the values are strings will this be checked. |  | [Source](https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Concepts.DBInstanceClass.html) | `resources`,`rds` |
| [E3026<a name="E3026"></a>](../src/cfnlint/rules/resources/elasticache/CacheClusterFailover.py) | Check Elastic Cache Redis Cluster settings | Evaluate Redis Cluster groups to make sure automatic failover is enabled when cluster mode is enabled |  | [Source](https://github.com/awslabs/cfn-python-lint) | `resources`,`elasticcache` |
| [E3027<a name="E3027"></a>](../src/cfnlint/rules/resources/events/RuleScheduleExpression.py) | Validate AWS Event ScheduleExpression format | Validate the formation of the AWS::Event ScheduleExpression |  | [Source](https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/ScheduledEvents.html) | `resources`,`events` |
| [E3030<a name="E3030"></a>](../src/cfnlint/rules/resources/properties/AllowedValue.py) | Check if properties have a valid value | Check if properties have a valid value in case of an enumator |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint/blob/main/docs/cfn-resource-specification.md#allowedvalue) | `resources`,`property`,`allowed value` |
| [E3031<a name="E3031"></a>](../src/cfnlint/rules/resources/properties/AllowedPattern.py) | Check if property values adhere to a specific pattern | Check if properties have a valid value in case of a pattern (Regular Expression) | exceptions:list:[] | [Source](https://github.com/awslabs/cfn-python-lint/blob/main/docs/cfn-resource-specification.md#allowedpattern) | `resources`,`property`,`allowed pattern`,`regex` |
| [E3032<a name="E3032"></a>](../src/cfnlint/rules/resources/properties/ListSize.py) | Check if a list has between min and max number of values specified | Check lists for the number of items in the list to validate they are between the minimum and maximum |  | [Source](https://github.com/awslabs/cfn-python-lint/blob/main/docs/cfn-resource-specification.md#allowedpattern) | `resources`,`property`,`list`,`size` |
| [E3033<a name="E3033"></a>](../src/cfnlint/rules/resources/properties/StringSize.py) | Check if a string has between min and max number of values specified | Check strings for its length between the minimum and maximum |  | [Source](https://github.com/awslabs/cfn-python-lint/blob/main/docs/cfn-resource-specification.md#allowedpattern) | `resources`,`property`,`string`,`size` |
| [E3034<a name="E3034"></a>](../src/cfnlint/rules/resources/properties/NumberSize.py) | Check if a number is between min and max | Check numbers (integers and floats) for its value being between the minimum and maximum |  | [Source](https://github.com/awslabs/cfn-python-lint/blob/main/docs/cfn-resource-specification.md#allowedpattern) | `resources`,`property`,`number`,`size` |
| [E3035<a name="E3035"></a>](../src/cfnlint/rules/resources/DeletionPolicy.py) | Check DeletionPolicy values for Resources | Check that the DeletionPolicy values are valid |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html) | `resources`,`deletionpolicy` |
| [E3036<a name="E3036"></a>](../src/cfnlint/rules/resources/UpdateReplacePolicy.py) | Check UpdateReplacePolicy values for Resources | Check that the UpdateReplacePolicy values are valid |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatereplacepolicy.html) | `resources`,`updatereplacepolicy` |
| [E3037<a name="E3037"></a>](../src/cfnlint/rules/resources/properties/ListDuplicates.py) | Check if a list has duplicate values | Certain lists don't support duplicate items. Check when duplicates are provided but not supported. |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint/blob/main/docs/cfn-resource-specification.md#allowedvalue) | `resources`,`property`,`list` |
| [E3038<a name="E3038"></a>](../src/cfnlint/rules/resources/ServerlessTransform.py) | Check if Serverless Resources have Serverless Transform | Check that a template with Serverless Resources also includes the Serverless Transform |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `resources`,`transform` |
| [E3039<a name="E3039"></a>](../src/cfnlint/rules/resources/dynamodb/AttributeMismatch.py) | AttributeDefinitions / KeySchemas mismatch | Verify the set of Attributes in AttributeDefinitions and KeySchemas match |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-dynamodb-table.html) | `resources`,`dynamodb` |
| [E3041<a name="E3041"></a>](../src/cfnlint/rules/resources/route53/RecordSetName.py) | RecordSet HostedZoneName is a superdomain of Name | In a RecordSet, the HostedZoneName must be a superdomain of the Name being validated |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-route53-recordset.html#cfn-route53-recordset-name) | `resource`,`properties`,`route53` |
| [E3042<a name="E3042"></a>](../src/cfnlint/rules/resources/ecs/TaskDefinitionEssentialContainer.py) | Check at least one essential container is specified | Check that every TaskDefinition specifies at least one essential container |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-essential) | `properties`,`ecs`,`task`,`container`,`fargate` |
| [E3043<a name="E3043"></a>](../src/cfnlint/rules/resources/cloudformation/NestedStackParameters.py) | Validate parameters for in a nested stack | Evalute if parameters for a nested stack are specified and if parameters are specified for a nested stack that aren't required. |  | [Source](https://github.com/awslabs/cfn-python-lint) | `resources`,`cloudformation` |
| [E3044<a name="E3044"></a>](../src/cfnlint/rules/resources/ecs/FargateDeploymentSchedulingStrategy.py) | Check Fargate service scheduling strategy | Check that Fargate service scheduling strategy is REPLICA |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-schedulingstrategy) | `properties`,`ecs`,`service`,`container`,`fargate` |
| [E3045<a name="E3045"></a>](../src/cfnlint/rules/resources/s3/AccessControlOwnership.py) | Validate AccessControl are set with OwnershipControls | When using AccessControl other than private you must also configure OwnershipControls. The default is bucket owner enforced which disables ACLs. |  | [Source](https://docs.aws.amazon.com/AmazonS3/latest/userguide/about-object-ownership.html) | `resources`,`s3` |
| [E3050<a name="E3050"></a>](../src/cfnlint/rules/resources/iam/RefWithPath.py) | Check if REFing to a IAM resource with path set | Some resources don't support looking up the IAM resource by name. This check validates when a REF is being used and the Path is not '/' |  | [Source](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements.html) | `properties`,`iam` |
| [E3502<a name="E3502"></a>](../src/cfnlint/rules/resources/properties/JsonSize.py) | Check if a JSON Object is within size limits | Validate properties that are JSON values so that their length is within the limits |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `resources`,`limits`,`json` |
| [E3503<a name="E3503"></a>](../src/cfnlint/rules/resources/certificatemanager/DomainValidationOptions.py) | ValidationDomain is superdomain of DomainName | In ValidationDomainOptions, the ValidationDomain must be a superdomain of the DomainName being validated |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-certificatemanager-certificate-domainvalidationoption.html#cfn-certificatemanager-certificate-domainvalidationoption-validationdomain) | `certificate`,`certificatemanager`,`domainvalidationoptions`,`validationdomain` |
| [E3504<a name="E3504"></a>](../src/cfnlint/rules/resources/backup/BackupPlanLifecycleRule.py) | Check minimum 90 period is met between BackupPlan cold and delete | Check that Backup plans with lifecycle rules have >= 90 days between cold and delete |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-backup-backupplan-lifecycleresourcetype.html) | `properties`,`backup`,`plan`,`lifecycle` |
| [E4001<a name="E4001"></a>](../src/cfnlint/rules/metadata/InterfaceConfiguration.py) | Metadata Interface have appropriate properties | Metadata Interface properties are properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-interface.html) | `metadata` |
| [E4002<a name="E4002"></a>](../src/cfnlint/rules/metadata/Config.py) | Validate the configuration of the Metadata section | Validates that Metadata section is an object and has no null values |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/metadata-section-structure.html) | `metadata` |
| [E5001<a name="E5001"></a>](../src/cfnlint/rules/resources/Modules.py) | Check that Modules resources are valid | Check that Modules resources are valid |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/) | `resources`,`modules` |
| [E6001<a name="E6001"></a>](../src/cfnlint/rules/outputs/Configuration.py) | Outputs have appropriate properties | Making sure the outputs are properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs` |
| [E6002<a name="E6002"></a>](../src/cfnlint/rules/outputs/Required.py) | Outputs have required properties | Making sure the outputs have required properties |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs` |
| [E6003<a name="E6003"></a>](../src/cfnlint/rules/outputs/Value.py) | Outputs have values of strings | Making sure the outputs have strings as values |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs` |
| [E6004<a name="E6004"></a>](../src/cfnlint/rules/outputs/Name.py) | Outputs have appropriate names | Check if Outputs are properly named (A-Za-z0-9) |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs` |
| [E6005<a name="E6005"></a>](../src/cfnlint/rules/outputs/Description.py) | Outputs descriptions can only be strings | Outputs descriptions can only be strings |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs` |
| [E6010<a name="E6010"></a>](../src/cfnlint/rules/outputs/LimitNumber.py) | Output limit not exceeded | Check the number of Outputs in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `outputs`,`limits` |
| [E6011<a name="E6011"></a>](../src/cfnlint/rules/outputs/LimitName.py) | Output name limit not exceeded | Check the size of Output names in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `outputs`,`limits` |
| [E6012<a name="E6012"></a>](../src/cfnlint/rules/outputs/LimitDescription.py) | Output description limit not exceeded | Check the size of Output description in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs`,`limits` |
| [E7001<a name="E7001"></a>](../src/cfnlint/rules/mappings/Configuration.py) | Mappings are appropriately configured | Check if Mappings are properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html) | `mappings` |
| [E7002<a name="E7002"></a>](../src/cfnlint/rules/mappings/Name.py) | Mappings have appropriate names | Check if Mappings are properly named (A-Za-z0-9) |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html) | `mappings` |
| [E7003<a name="E7003"></a>](../src/cfnlint/rules/mappings/KeyName.py) | Mapping keys are strings and alphanumeric | Check if Mappings keys are properly typed as strings and alphanumeric |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html) | `mappings` |
| [E7010<a name="E7010"></a>](../src/cfnlint/rules/mappings/LimitNumber.py) | Mapping limit not exceeded | Check the number of Mappings in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `mappings`,`limits` |
| [E7011<a name="E7011"></a>](../src/cfnlint/rules/mappings/LimitName.py) | Mapping name limit not exceeded | Check the size of Mapping names in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `mappings`,`limits` |
| [E7012<a name="E7012"></a>](../src/cfnlint/rules/mappings/LimitAttributes.py) | Mapping attribute limit not exceeded | Check if the amount of Mapping attributes in the template is less than the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `mappings`,`limits` |
| [E8001<a name="E8001"></a>](../src/cfnlint/rules/conditions/Configuration.py) | Conditions have appropriate properties | Check if Conditions are properly configured |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/conditions-section-structure.html) | `conditions` |
| [E8002<a name="E8002"></a>](../src/cfnlint/rules/conditions/Exists.py) | Check if the referenced Conditions are defined | Making sure the used conditions are actually defined in the Conditions section |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/conditions-section-structure.html) | `conditions` |
| [E8003<a name="E8003"></a>](../src/cfnlint/rules/conditions/Equals.py) | Check Fn::Equals structure for validity | Check Fn::Equals is a list of two elements |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-equals) | `functions`,`equals` |
| [E8004<a name="E8004"></a>](../src/cfnlint/rules/conditions/And.py) | Check Fn::And structure for validity | Check Fn::And is a list of two elements |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-and) | `functions`,`and` |
| [E8005<a name="E8005"></a>](../src/cfnlint/rules/conditions/Not.py) | Check Fn::Not structure for validity | Check Fn::Not is a list of one element |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-not) | `functions`,`not` |
| [E8006<a name="E8006"></a>](../src/cfnlint/rules/conditions/Or.py) | Check Fn::Or structure for validity | Check Fn::Or is a list of two elements |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-or) | `functions`,`or` |
| [I1002<a name="I1002"></a>](../src/cfnlint/rules/templates/ApproachingLimitSize.py) | Template size limit | Check the size of the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `limits` |
| [I1003<a name="I1003"></a>](../src/cfnlint/rules/templates/ApproachingLimitDescription.py) | Template description limit | Check if the size of the template description is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `description`,`limits` |
| [I1022<a name="I1022"></a>](../src/cfnlint/rules/functions/SubNotJoin.py) | Use Sub instead of Join | Prefer a sub instead of Join when using a join delimiter that is empty |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-sub.html) | `functions`,`sub`,`join` |
| [I2010<a name="I2010"></a>](../src/cfnlint/rules/parameters/ApproachingLimitNumber.py) | Parameter limit | Check the number of Parameters in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `parameters`,`limits` |
| [I2011<a name="I2011"></a>](../src/cfnlint/rules/parameters/ApproachingLimitName.py) | Parameter name limit | Check the size of Parameter names in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `parameters`,`limits` |
| [I2012<a name="I2012"></a>](../src/cfnlint/rules/parameters/ApproachingLimitValue.py) | Parameter value limit | Check if the size of Parameter values in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `parameters`,`limits` |
| [I2530<a name="I2530"></a>](../src/cfnlint/rules/resources/lmbd/SnapStartEnabled.py) | Validate that SnapStart is configured for >= Java11 runtimes | SnapStart is a no-cost feature that can increase performance up to 10x. Enable SnapStart for Java 11 and greater runtimes |  | [Source](https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html) | `resources`,`lambda` |
| [I3010<a name="I3010"></a>](../src/cfnlint/rules/resources/ApproachingLimitNumber.py) | Resource limit | Check the number of Resources in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `resources`,`limits` |
| [I3011<a name="I3011"></a>](../src/cfnlint/rules/resources/UpdateReplacePolicyDeletionPolicyOnStatefulResourceTypes.py) | Check stateful resources have a set UpdateReplacePolicy/DeletionPolicy | The default action when replacing/removing a resource is to delete it. This check requires you to explicitly set policies |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html) | `resources`,`updatereplacepolicy`,`deletionpolicy` |
| [I3012<a name="I3012"></a>](../src/cfnlint/rules/resources/ApproachingLimitName.py) | Resource name limit | Check the size of Resource names in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `resources`,`limits` |
| [I3013<a name="I3013"></a>](../src/cfnlint/rules/resources/RetentionPeriodOnResourceTypesWithAutoExpiringContent.py) | Check resources with auto expiring content have explicit retention period | The behaviour for data retention is different across AWS Services.If no retention period is specified the default for some services is to delete the data after a period of time.This check requires you to explicitly set the retention period for those resources to avoid unexpected data losses |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `resources`,`retentionperiod` |
| [I3037<a name="I3037"></a>](../src/cfnlint/rules/resources/properties/ListDuplicatesAllowed.py) | Check if a list that allows duplicates has any duplicates | Certain lists support duplicate items.Provide an alert when list of strings or numbers have repeats. |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint/blob/main/docs/rules.md#rules-1) | `resources`,`property`,`list` |
| [I3042<a name="I3042"></a>](../src/cfnlint/rules/resources/HardCodedArnProperties.py) | ARNs should use correctly placed Pseudo Parameters | Checks Resources if ARNs use correctly placed Pseudo Parameters instead of hardcoded Partition, Region, and Account Number | partition:boolean:True<br />region:boolean:False<br />accountId:boolean:False | [Source]() | `resources` |
| [I3100<a name="I3100"></a>](../src/cfnlint/rules/resources/PreviousGenerationInstanceType.py) | Checks for legacy instance type generations | New instance type generations increase performance and decrease cost |  | [Source](https://aws.amazon.com/ec2/previous-generation/) | `resources`,`ec2`,`rds`,`elasticcache`,`elasticsearch` |
| [I6010<a name="I6010"></a>](../src/cfnlint/rules/outputs/ApproachingLimitNumber.py) | Output limit | Check the number of Outputs in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `outputs`,`limits` |
| [I6011<a name="I6011"></a>](../src/cfnlint/rules/outputs/ApproachingLimitName.py) | Output name limit | Check the size of Output names in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `outputs`,`limits` |
| [I6012<a name="I6012"></a>](../src/cfnlint/rules/outputs/ApproachingLimitDescription.py) | Output description limit | Check the size of Output description in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) | `outputs`,`limits` |
| [I7010<a name="I7010"></a>](../src/cfnlint/rules/mappings/ApproachingLimitNumber.py) | Mapping limit | Check the number of Mappings in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `mappings`,`limits` |
| [I7011<a name="I7011"></a>](../src/cfnlint/rules/mappings/ApproachingLimitName.py) | Mapping name limit | Check the size of Mapping names in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `mappings`,`limits` |
| [I7012<a name="I7012"></a>](../src/cfnlint/rules/mappings/ApproachingLimitAttributes.py) | Mapping attribute limit | Check if the amount of Mapping attributes in the template is approaching the upper limit |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) | `mappings`,`limits` |
| [W1001<a name="W1001"></a>](../src/cfnlint/rules/functions/RelationshipConditions.py) | Ref/GetAtt to resource that is available when conditions are applied | Check the Conditions that affect a Ref/GetAtt to make sure the resource being related to is available when there is a resource condition. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-ref.html) | `conditions`,`resources`,`relationships`,`ref`,`getatt`,`sub` |
| [W1011<a name="W1011"></a>](../src/cfnlint/rules/functions/FindInMapKeys.py) | FindInMap keys exist in the map | Checks the keys in a FindInMap to make sure they exist. Check only if the Map Name is a string and if the key is a string. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-findinmap.html) | `functions`,`findinmap` |
| [W1019<a name="W1019"></a>](../src/cfnlint/rules/functions/SubParametersUsed.py) | Sub validation of parameters | Validate that Fn::Sub Parameters are used |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-sub.html) | `functions`,`sub` |
| [W1020<a name="W1020"></a>](../src/cfnlint/rules/functions/SubUnneeded.py) | Sub isn't needed if it doesn't have a variable defined | Checks sub strings to see if a variable is defined. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-sub.html) | `functions`,`sub` |
| [W2001<a name="W2001"></a>](../src/cfnlint/rules/parameters/Used.py) | Check if Parameters are Used | Making sure the parameters defined are used |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `parameters` |
| [W2030<a name="W2030"></a>](../src/cfnlint/rules/parameters/AllowedValue.py) | Check if parameters have a valid value | Check if parameters have a valid value in case of an enumator. The Parameter's allowed values is based on the usages in property (Ref) |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint/blob/main/docs/cfn-resource-specification.md#allowedvalue) | `parameters`,`resources`,`property`,`allowed value` |
| [W2031<a name="W2031"></a>](../src/cfnlint/rules/parameters/AllowedPattern.py) | Check if parameters have a valid value based on an allowed pattern | Check if parameters have a valid value in a pattern. The Parameter's allowed pattern is based on the usages in property (Ref) |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint/blob/main/docs/cfn-resource-specification.md#allowedpattern) | `parameters`,`resources`,`property`,`allowed pattern` |
| [W2501<a name="W2501"></a>](../src/cfnlint/rules/resources/properties/Password.py) | Check if Password Properties are correctly configured | Password properties should not be strings and if parameter using NoEcho |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/best-practices.html#creds) | `parameters`,`passwords`,`security`,`dynamic reference` |
| [W2506<a name="W2506"></a>](../src/cfnlint/rules/resources/properties/ImageId.py) | Check if ImageId Parameters have the correct type | See if there are any refs for ImageId to a parameter of inappropriate type. Appropriate Types are [AWS::EC2::Image::Id, AWS::SSM::Parameter::Value<AWS::EC2::Image::Id>] |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/best-practices.html#parmtypes) | `parameters`,`ec2`,`imageid` |
| [W2510<a name="W2510"></a>](../src/cfnlint/rules/parameters/LambdaMemorySize.py) | Parameter Memory Size attributes should have max and min | Check if a parameter that is used for Lambda memory size  should have a min and max size that matches Lambda constraints |  | [Source](https://docs.aws.amazon.com/lambda/latest/dg/API_CreateFunction.html#SSS-CreateFunction-request-MemorySize) | `parameters`,`lambda` |
| [W2511<a name="W2511"></a>](../src/cfnlint/rules/resources/iam/PolicyVersion.py) | Check IAM Resource Policies syntax | See if the elements inside an IAM Resource policy are configured correctly. |  | [Source](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements.html) | `properties`,`iam` |
| [W2530<a name="W2530"></a>](../src/cfnlint/rules/resources/lmbd/SnapStart.py) | Validate that SnapStart is properly configured | To properly leverage SnapStart, you must configure both the lambda function and attach a Lambda version resource |  | [Source](https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html) | `resources`,`lambda` |
| [W2531<a name="W2531"></a>](../src/cfnlint/rules/resources/lmbd/DeprecatedRuntimeEol.py) | Check if EOL Lambda Function Runtimes are used | Check if an EOL Lambda Runtime is specified and give a warning if used.  |  | [Source](https://docs.aws.amazon.com/lambda/latest/dg/runtime-support-policy.html) | `resources`,`lambda`,`runtime` |
| [W2533<a name="W2533"></a>](../src/cfnlint/rules/resources/lmbd/ZipPackageRequiredProperties.py) | Check required properties for Lambda if the deployment package is a .zip file | When the package type is Zip, you must also specify the `handler` and `runtime` properties. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html) | `resources`,`lambda` |
| [W3002<a name="W3002"></a>](../src/cfnlint/rules/resources/properties/PropertiesTemplated.py) | Warn when properties are configured to only work with the package command | Some properties can be configured to only work with the CloudFormationpackage command. Warn when this is the case so user is aware. |  | [Source](https://docs.aws.amazon.com/cli/latest/reference/cloudformation/package.html) | `resources` |
| [W3005<a name="W3005"></a>](../src/cfnlint/rules/resources/DependsOnObsolete.py) | Check obsolete DependsOn configuration for Resources | Check if DependsOn is specified if not needed. A Ref or a Fn::GetAtt already is an implicit dependency. |  | [Source](https://aws.amazon.com/blogs/devops/optimize-aws-cloudformation-templates/) | `resources`,`dependson`,`ref`,`getatt` |
| [W3010<a name="W3010"></a>](../src/cfnlint/rules/resources/properties/AvailabilityZone.py) | Availability Zone Parameters should not be hardcoded | Check if an Availability Zone property is hardcoded. |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `parameters`,`availabilityzone` |
| [W3011<a name="W3011"></a>](../src/cfnlint/rules/resources/BothUpdateReplacePolicyDeletionPolicyNeeded.py) | Check resources with UpdateReplacePolicy/DeletionPolicy have both | Both UpdateReplacePolicy and DeletionPolicy are needed to protect resources from deletion |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html) | `resources`,`updatereplacepolicy`,`deletionpolicy` |
| [W3037*<a name="W3037*"></a>](../src/cfnlint/rules/resources/iam/Permissions.py) | Check IAM Permission configuration | Check for valid IAM Permissions |  | [Source](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html) | `properties`,`iam`,`permissions` |
| [W3045<a name="W3045"></a>](../src/cfnlint/rules/resources/s3/AccessControlObsolete.py) | Controlling access to an S3 bucket should be done with bucket policies | Nearly all access control configurations can be more successfully achieved with bucket policies. Consider using bucket policies instead of access control. |  | [Source](https://docs.aws.amazon.com/AmazonS3/latest/userguide/about-object-ownership.html) | `resources`,`s3` |
| [W4001<a name="W4001"></a>](../src/cfnlint/rules/metadata/InterfaceParameterExists.py) | Metadata Interface parameters exist | Metadata Interface parameters actually exist |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-interface.html) | `metadata` |
| [W4002<a name="W4002"></a>](../src/cfnlint/rules/resources/NoEcho.py) | Check for NoEcho References | Check if there is a NoEcho enabled parameter referenced within a resources Metadata section |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html#parameters-section-structure-properties) | `resources`,`NoEcho` |
| [W6001<a name="W6001"></a>](../src/cfnlint/rules/outputs/ImportValue.py) | Check Outputs using ImportValue | Check if the Output value is set using ImportValue, so creating an Output of an Output |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `outputs`,`importvalue` |
| [W7001<a name="W7001"></a>](../src/cfnlint/rules/mappings/Used.py) | Check if Mappings are Used | Making sure the mappings defined are used |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `mappings` |
| [W8001<a name="W8001"></a>](../src/cfnlint/rules/conditions/Used.py) | Check if Conditions are Used | Making sure the conditions defined are used |  | [Source](https://github.com/aws-cloudformation/cfn-python-lint) | `conditions` |
| [W8003<a name="W8003"></a>](../src/cfnlint/rules/conditions/EqualsIsUseful.py) | Fn::Equals will always return true or false | Validate Fn::Equals to see if its comparing two strings or two equal items. While this works it may not be intended. |  | [Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-conditions.html#intrinsic-function-reference-conditions-equals) | `functions`,`equals` |

\* experimental rules
