# This code is part of the Biopython distribution and governed by its
# license.  Please see the LICENSE file that should have been included
# as part of this package.

"""Tests for UniGene module."""

from Bio import UniGene
import unittest


class TestUniGene(unittest.TestCase):

    def test_parse(self):

        # Start of the UniGene file for Equus caballus downloaded from:
        # ftp://ftp.ncbi.nih.gov/repository/UniGene/Equus_caballus
        with open("UniGene/Eca.1.2425.data") as handle:

            records = UniGene.parse(handle)

            # First record
            record = next(records)
            self.assertEqual(record.ID, "Eca.1")
            self.assertEqual(record.title, "Ribosomal protein L3")
            self.assertEqual(record.symbol, "RPL3")
            self.assertEqual(record.gene_id, "100070291")
            self.assertEqual(record.locuslink, "100070291")
            self.assertEqual(record.homol, True)
            self.assertEqual(record.express, ["blood", "cartilage", "trophoblast", "adult"])
            self.assertEqual(record.restr_expr, ["blood", "adult"])
            self.assertEqual(len(record.protsim), 40)
            self.assertEqual(record.protsim[0].org, "10090")
            self.assertEqual(record.protsim[0].protgi, "149270989")
            self.assertEqual(record.protsim[0].protid, "XP_001477314.1")
            self.assertEqual(record.protsim[0].pct, "99.26")
            self.assertEqual(record.protsim[0].aln, "401")
            self.assertEqual(record.protsim[1].org, "5207")
            self.assertEqual(record.protsim[1].protgi, "134117057")
            self.assertEqual(record.protsim[1].protid, "XP_772755.1")
            self.assertEqual(record.protsim[1].pct, "70.91")
            self.assertEqual(record.protsim[1].aln, "384")
            self.assertEqual(record.protsim[2].org, "44689")
            self.assertEqual(record.protsim[2].protgi, "66802278")
            self.assertEqual(record.protsim[2].protid, "XP_629921.1")
            self.assertEqual(record.protsim[2].pct, "67.76")
            self.assertEqual(record.protsim[2].aln, "396")
            self.assertEqual(record.protsim[3].org, "9606")
            self.assertEqual(record.protsim[3].protgi, "4506649")
            self.assertEqual(record.protsim[3].protid, "NP_000958.1")
            self.assertEqual(record.protsim[3].pct, "99.50")
            self.assertEqual(record.protsim[3].aln, "401")
            self.assertEqual(record.protsim[4].org, "7719")
            self.assertEqual(record.protsim[4].protgi, "198435984")
            self.assertEqual(record.protsim[4].protid, "XP_002132034.1")
            self.assertEqual(record.protsim[4].pct, "80.15")
            self.assertEqual(record.protsim[4].aln, "402")
            self.assertEqual(record.protsim[5].org, "13616")
            self.assertEqual(record.protsim[5].protgi, "126339540")
            self.assertEqual(record.protsim[5].protid, "XP_001366844.1")
            self.assertEqual(record.protsim[5].pct, "97.05")
            self.assertEqual(record.protsim[5].aln, "405")
            self.assertEqual(record.protsim[6].org, "9615")
            self.assertEqual(record.protsim[6].protgi, "73969069")
            self.assertEqual(record.protsim[6].protid, "XP_531732.2")
            self.assertEqual(record.protsim[6].pct, "99.50")
            self.assertEqual(record.protsim[6].aln, "401")
            self.assertEqual(record.protsim[7].org, "3218")
            self.assertEqual(record.protsim[7].protgi, "168040500")
            self.assertEqual(record.protsim[7].protid, "XP_001772732.1")
            self.assertEqual(record.protsim[7].pct, "73.01")
            self.assertEqual(record.protsim[7].aln, "388")
            self.assertEqual(record.protsim[8].org, "9913")
            self.assertEqual(record.protsim[8].protgi, "27807287")
            self.assertEqual(record.protsim[8].protid, "NP_777140.1")
            self.assertEqual(record.protsim[8].pct, "99.26")
            self.assertEqual(record.protsim[8].aln, "401")
            self.assertEqual(record.protsim[9].org, "8355")
            self.assertEqual(record.protsim[9].protgi, "148228673")
            self.assertEqual(record.protsim[9].protid, "NP_001080341.1")
            self.assertEqual(record.protsim[9].pct, "94.04")
            self.assertEqual(record.protsim[9].aln, "401")
            self.assertEqual(record.protsim[10].org, "9258")
            self.assertEqual(record.protsim[10].protgi, "149585070")
            self.assertEqual(record.protsim[10].protid, "XP_001514408.1")
            self.assertEqual(record.protsim[10].pct, "97.29")
            self.assertEqual(record.protsim[10].aln, "367")
            self.assertEqual(record.protsim[11].org, "7955")
            self.assertEqual(record.protsim[11].protgi, "48597014")
            self.assertEqual(record.protsim[11].protid, "NP_001001590.1")
            self.assertEqual(record.protsim[11].pct, "92.31")
            self.assertEqual(record.protsim[11].aln, "401")
            self.assertEqual(record.protsim[12].org, "7227")
            self.assertEqual(record.protsim[12].protgi, "17737907")
            self.assertEqual(record.protsim[12].protid, "NP_524316.1")
            self.assertEqual(record.protsim[12].pct, "78.91")
            self.assertEqual(record.protsim[12].aln, "402")
            self.assertEqual(record.protsim[13].org, "28985")
            self.assertEqual(record.protsim[13].protgi, "50311593")
            self.assertEqual(record.protsim[13].protid, "XP_455822.1")
            self.assertEqual(record.protsim[13].pct, "72.28")
            self.assertEqual(record.protsim[13].aln, "385")
            self.assertEqual(record.protsim[14].org, "9796")
            self.assertEqual(record.protsim[14].protgi, "149743311")
            self.assertEqual(record.protsim[14].protid, "XP_001501957.1")
            self.assertEqual(record.protsim[14].pct, "100.00")
            self.assertEqual(record.protsim[14].aln, "401")
            self.assertEqual(record.protsim[15].org, "4896")
            self.assertEqual(record.protsim[15].protgi, "19115692")
            self.assertEqual(record.protsim[15].protid, "NP_594780.1")
            self.assertEqual(record.protsim[15].pct, "74.61")
            self.assertEqual(record.protsim[15].aln, "385")
            self.assertEqual(record.protsim[16].org, "4932")
            self.assertEqual(record.protsim[16].protgi, "6324637")
            self.assertEqual(record.protsim[16].protid, "NP_014706.1")
            self.assertEqual(record.protsim[16].pct, "72.02")
            self.assertEqual(record.protsim[16].aln, "385")
            self.assertEqual(record.protsim[17].org, "8364")
            self.assertEqual(record.protsim[17].protgi, "187607061")
            self.assertEqual(record.protsim[17].protid, "NP_001120075.1")
            self.assertEqual(record.protsim[17].pct, "94.54")
            self.assertEqual(record.protsim[17].aln, "401")
            self.assertEqual(record.protsim[18].org, "5911")
            self.assertEqual(record.protsim[18].protgi, "118389862")
            self.assertEqual(record.protsim[18].protid, "XP_001027976.1")
            self.assertEqual(record.protsim[18].pct, "64.60")
            self.assertEqual(record.protsim[18].aln, "386")
            self.assertEqual(record.protsim[19].org, "4530")
            self.assertEqual(record.protsim[19].protgi, "115487526")
            self.assertEqual(record.protsim[19].protid, "NP_001066250.1")
            self.assertEqual(record.protsim[19].pct, "73.20")
            self.assertEqual(record.protsim[19].aln, "387")
            self.assertEqual(record.protsim[20].org, "10116")
            self.assertEqual(record.protsim[20].protgi, "38454246")
            self.assertEqual(record.protsim[20].protid, "NP_942048.1")
            self.assertEqual(record.protsim[20].pct, "99.01")
            self.assertEqual(record.protsim[20].aln, "401")
            self.assertEqual(record.protsim[21].org, "5833")
            self.assertEqual(record.protsim[21].protgi, "124802670")
            self.assertEqual(record.protsim[21].protid, "XP_001347556.1")
            self.assertEqual(record.protsim[21].pct, "69.45")
            self.assertEqual(record.protsim[21].aln, "382")
            self.assertEqual(record.protsim[22].org, "148305")
            self.assertEqual(record.protsim[22].protgi, "145608344")
            self.assertEqual(record.protsim[22].protid, "XP_360650.2")
            self.assertEqual(record.protsim[22].pct, "71.28")
            self.assertEqual(record.protsim[22].aln, "389")
            self.assertEqual(record.protsim[23].org, "7668")
            self.assertEqual(record.protsim[23].protgi, "115725235")
            self.assertEqual(record.protsim[23].protid, "XP_791350.2")
            self.assertEqual(record.protsim[23].pct, "81.50")
            self.assertEqual(record.protsim[23].aln, "399")
            self.assertEqual(record.protsim[24].org, "3702")
            self.assertEqual(record.protsim[24].protgi, "15218306")
            self.assertEqual(record.protsim[24].protid, "NP_175009.1")
            self.assertEqual(record.protsim[24].pct, "71.47")
            self.assertEqual(record.protsim[24].aln, "387")
            self.assertEqual(record.protsim[25].org, "45351")
            self.assertEqual(record.protsim[25].protgi, "156359547")
            self.assertEqual(record.protsim[25].protid, "XP_001624829.1")
            self.assertEqual(record.protsim[25].pct, "80.10")
            self.assertEqual(record.protsim[25].aln, "396")
            self.assertEqual(record.protsim[26].org, "7091")
            self.assertEqual(record.protsim[26].protgi, "112982798")
            self.assertEqual(record.protsim[26].protid, "NP_001037126.1")
            self.assertEqual(record.protsim[26].pct, "80.65")
            self.assertEqual(record.protsim[26].aln, "402")
            self.assertEqual(record.protsim[27].org, "33169")
            self.assertEqual(record.protsim[27].protgi, "45188079")
            self.assertEqual(record.protsim[27].protid, "NP_984302.1")
            self.assertEqual(record.protsim[27].pct, "72.35")
            self.assertEqual(record.protsim[27].aln, "386")
            self.assertEqual(record.protsim[28].org, "7460")
            self.assertEqual(record.protsim[28].protgi, "66566113")
            self.assertEqual(record.protsim[28].protid, "XP_624821.1")
            self.assertEqual(record.protsim[28].pct, "80.89")
            self.assertEqual(record.protsim[28].aln, "402")
            self.assertEqual(record.protsim[29].org, "5141")
            self.assertEqual(record.protsim[29].protgi, "164424622")
            self.assertEqual(record.protsim[29].protid, "XP_963317.2")
            self.assertEqual(record.protsim[29].pct, "71.54")
            self.assertEqual(record.protsim[29].aln, "389")
            self.assertEqual(record.protsim[30].org, "6239")
            self.assertEqual(record.protsim[30].protgi, "71984538")
            self.assertEqual(record.protsim[30].protid, "NP_001021254.1")
            self.assertEqual(record.protsim[30].pct, "75.63")
            self.assertEqual(record.protsim[30].aln, "397")
            self.assertEqual(record.protsim[31].org, "3055")
            self.assertEqual(record.protsim[31].protgi, "159489312")
            self.assertEqual(record.protsim[31].protid, "XP_001702641.1")
            self.assertEqual(record.protsim[31].pct, "72.99")
            self.assertEqual(record.protsim[31].aln, "384")
            self.assertEqual(record.protsim[32].org, "7070")
            self.assertEqual(record.protsim[32].protgi, "189240524")
            self.assertEqual(record.protsim[32].protid, "XP_971875.2")
            self.assertEqual(record.protsim[32].pct, "76.67")
            self.assertEqual(record.protsim[32].aln, "402")
            self.assertEqual(record.protsim[33].org, "7165")
            self.assertEqual(record.protsim[33].protgi, "158291770")
            self.assertEqual(record.protsim[33].protid, "XP_313303.4")
            self.assertEqual(record.protsim[33].pct, "76.56")
            self.assertEqual(record.protsim[33].aln, "400")
            self.assertEqual(record.protsim[34].org, "9544")
            self.assertEqual(record.protsim[34].protgi, "109094341")
            self.assertEqual(record.protsim[34].protid, "XP_001095608.1")
            self.assertEqual(record.protsim[34].pct, "99.50")
            self.assertEqual(record.protsim[34].aln, "401")
            self.assertEqual(record.protsim[35].org, "7176")
            self.assertEqual(record.protsim[35].protgi, "170041842")
            self.assertEqual(record.protsim[35].protid, "XP_001848658.1")
            self.assertEqual(record.protsim[35].pct, "74.75")
            self.assertEqual(record.protsim[35].aln, "403")
            self.assertEqual(record.protsim[36].org, "5888")
            self.assertEqual(record.protsim[36].protgi, "145475213")
            self.assertEqual(record.protsim[36].protid, "XP_001423629.1")
            self.assertEqual(record.protsim[36].pct, "66.67")
            self.assertEqual(record.protsim[36].aln, "377")
            self.assertEqual(record.protsim[37].org, "9031")
            self.assertEqual(record.protsim[37].protgi, "57525400")
            self.assertEqual(record.protsim[37].protid, "NP_001006241.1")
            self.assertEqual(record.protsim[37].pct, "97.27")
            self.assertEqual(record.protsim[37].aln, "401")
            self.assertEqual(record.protsim[38].org, "7029")
            self.assertEqual(record.protsim[38].protgi, "193580256")
            self.assertEqual(record.protsim[38].protid, "XP_001951042.1")
            self.assertEqual(record.protsim[38].pct, "79.40")
            self.assertEqual(record.protsim[38].aln, "402")
            self.assertEqual(record.protsim[39].org, "9598")
            self.assertEqual(record.protsim[39].protgi, "114608746")
            self.assertEqual(record.protsim[39].protid, "XP_518669.2")
            self.assertEqual(record.protsim[39].pct, "93.73")
            self.assertEqual(record.protsim[39].aln, "381")
            self.assertEqual(len(record.sequence), 69)
            self.assertEqual(record.sequence[0].acc, "XM_001501907.1")
            self.assertEqual(record.sequence[0].nid, "g149743310")
            self.assertEqual(record.sequence[0].pid, "g149743311")
            self.assertEqual(record.sequence[0].seqtype, "Model")
            self.assertEqual(record.sequence[1].acc, "CD469355.1")
            self.assertEqual(record.sequence[1].nid, "g31390623")
            self.assertEqual(record.sequence[1].clone, "LeukoS2_3_H08_A024")
            self.assertEqual(record.sequence[1].end, "5'")
            self.assertEqual(record.sequence[1].lid, "13776")
            self.assertEqual(record.sequence[1].seqtype, "EST")
            self.assertEqual(record.sequence[1].trace, "891191391")
            self.assertEqual(record.sequence[2].acc, "CD535273.1")
            self.assertEqual(record.sequence[2].nid, "g31577688")
            self.assertEqual(record.sequence[2].clone, "LeukoN5_1_E01_A027")
            self.assertEqual(record.sequence[2].end, "5'")
            self.assertEqual(record.sequence[2].lid, "13843")
            self.assertEqual(record.sequence[2].seqtype, "EST")
            self.assertEqual(record.sequence[2].trace, "891188893")
            self.assertEqual(record.sequence[3].acc, "CD466249.1")
            self.assertEqual(record.sequence[3].nid, "g31387517")
            self.assertEqual(record.sequence[3].clone, "LeukoN2_3_E11_A024")
            self.assertEqual(record.sequence[3].end, "3'")
            self.assertEqual(record.sequence[3].lid, "13773")
            self.assertEqual(record.sequence[3].seqtype, "EST")
            self.assertEqual(record.sequence[3].trace, "891186094")
            self.assertEqual(record.sequence[4].acc, "CD465822.1")
            self.assertEqual(record.sequence[4].nid, "g31387090")
            self.assertEqual(record.sequence[4].clone, "LeukoN1_8_C11_A023")
            self.assertEqual(record.sequence[4].end, "3'")
            self.assertEqual(record.sequence[4].lid, "13772")
            self.assertEqual(record.sequence[4].seqtype, "EST")
            self.assertEqual(record.sequence[4].trace, "891184882")
            self.assertEqual(record.sequence[5].acc, "CD466309.1")
            self.assertEqual(record.sequence[5].nid, "g31387577")
            self.assertEqual(record.sequence[5].clone, "LeukoN2_3_E11_A024")
            self.assertEqual(record.sequence[5].end, "5'")
            self.assertEqual(record.sequence[5].lid, "13773")
            self.assertEqual(record.sequence[5].seqtype, "EST")
            self.assertEqual(record.sequence[5].trace, "891185997")
            self.assertEqual(record.sequence[6].acc, "CD465893.1")
            self.assertEqual(record.sequence[6].nid, "g31387161")
            self.assertEqual(record.sequence[6].clone, "LeukoN1_8_C11_A023")
            self.assertEqual(record.sequence[6].end, "5'")
            self.assertEqual(record.sequence[6].lid, "13772")
            self.assertEqual(record.sequence[6].seqtype, "EST")
            self.assertEqual(record.sequence[6].trace, "891185010")
            self.assertEqual(record.sequence[7].acc, "CD471087.1")
            self.assertEqual(record.sequence[7].nid, "g31392355")
            self.assertEqual(record.sequence[7].clone, "LeukoS5_4_H03_A027")
            self.assertEqual(record.sequence[7].end, "5'")
            self.assertEqual(record.sequence[7].lid, "13778")
            self.assertEqual(record.sequence[7].seqtype, "EST")
            self.assertEqual(record.sequence[7].trace, "891193766")
            self.assertEqual(record.sequence[8].acc, "CD472103.1")
            self.assertEqual(record.sequence[8].nid, "g31393371")
            self.assertEqual(record.sequence[8].clone, "LeukoS6_1_C10_A028")
            self.assertEqual(record.sequence[8].end, "3'")
            self.assertEqual(record.sequence[8].lid, "13779")
            self.assertEqual(record.sequence[8].seqtype, "EST")
            self.assertEqual(record.sequence[8].trace, "891194618")
            self.assertEqual(record.sequence[9].acc, "CD471067.1")
            self.assertEqual(record.sequence[9].nid, "g31392335")
            self.assertEqual(record.sequence[9].clone, "LeukoS5_4_H03_A027")
            self.assertEqual(record.sequence[9].end, "3'")
            self.assertEqual(record.sequence[9].lid, "13778")
            self.assertEqual(record.sequence[9].seqtype, "EST")
            self.assertEqual(record.sequence[9].trace, "891194175")
            self.assertEqual(record.sequence[10].acc, "CD470984.1")
            self.assertEqual(record.sequence[10].nid, "g31392252")
            self.assertEqual(record.sequence[10].clone, "LeukoS5_3_A04_A027")
            self.assertEqual(record.sequence[10].end, "5'")
            self.assertEqual(record.sequence[10].lid, "13778")
            self.assertEqual(record.sequence[10].seqtype, "EST")
            self.assertEqual(record.sequence[10].trace, "891194114")
            self.assertEqual(record.sequence[11].acc, "CD472154.1")
            self.assertEqual(record.sequence[11].nid, "g31393422")
            self.assertEqual(record.sequence[11].clone, "LeukoS6_1_H10_A028")
            self.assertEqual(record.sequence[11].end, "3'")
            self.assertEqual(record.sequence[11].lid, "13779")
            self.assertEqual(record.sequence[11].seqtype, "EST")
            self.assertEqual(record.sequence[11].trace, "891195062")
            self.assertEqual(record.sequence[12].acc, "CD467586.1")
            self.assertEqual(record.sequence[12].nid, "g31388854")
            self.assertEqual(record.sequence[12].clone, "LeukoS1_5_G11_A023")
            self.assertEqual(record.sequence[12].end, "5'")
            self.assertEqual(record.sequence[12].lid, "13774")
            self.assertEqual(record.sequence[12].seqtype, "EST")
            self.assertEqual(record.sequence[12].trace, "891190912")
            self.assertEqual(record.sequence[13].acc, "CD465578.1")
            self.assertEqual(record.sequence[13].nid, "g31386846")
            self.assertEqual(record.sequence[13].clone, "LeukoN1_5_F07_A023")
            self.assertEqual(record.sequence[13].end, "5'")
            self.assertEqual(record.sequence[13].lid, "13772")
            self.assertEqual(record.sequence[13].seqtype, "EST")
            self.assertEqual(record.sequence[13].trace, "891185096")
            self.assertEqual(record.sequence[14].acc, "CD465858.1")
            self.assertEqual(record.sequence[14].nid, "g31387126")
            self.assertEqual(record.sequence[14].clone, "LeukoN1_8_H04_A023")
            self.assertEqual(record.sequence[14].end, "3'")
            self.assertEqual(record.sequence[14].lid, "13772")
            self.assertEqual(record.sequence[14].seqtype, "EST")
            self.assertEqual(record.sequence[14].trace, "891185129")
            self.assertEqual(record.sequence[15].acc, "CD471796.1")
            self.assertEqual(record.sequence[15].nid, "g31393064")
            self.assertEqual(record.sequence[15].clone, "LeukoS6_2_D05_A028")
            self.assertEqual(record.sequence[15].end, "3'")
            self.assertEqual(record.sequence[15].lid, "13779")
            self.assertEqual(record.sequence[15].seqtype, "EST")
            self.assertEqual(record.sequence[15].trace, "891194917")
            self.assertEqual(record.sequence[16].acc, "CD471724.1")
            self.assertEqual(record.sequence[16].nid, "g31392992")
            self.assertEqual(record.sequence[16].clone, "LeukoS6_4_F09_A028")
            self.assertEqual(record.sequence[16].end, "5'")
            self.assertEqual(record.sequence[16].lid, "13779")
            self.assertEqual(record.sequence[16].seqtype, "EST")
            self.assertEqual(record.sequence[16].trace, "891195058")
            self.assertEqual(record.sequence[17].acc, "CD471610.1")
            self.assertEqual(record.sequence[17].nid, "g31392878")
            self.assertEqual(record.sequence[17].clone, "LeukoS6_4_F09_A028")
            self.assertEqual(record.sequence[17].end, "3'")
            self.assertEqual(record.sequence[17].lid, "13779")
            self.assertEqual(record.sequence[17].seqtype, "EST")
            self.assertEqual(record.sequence[17].trace, "891194846")
            self.assertEqual(record.sequence[18].acc, "CD535233.1")
            self.assertEqual(record.sequence[18].nid, "g31577648")
            self.assertEqual(record.sequence[18].clone, "LeukoN5_1_B08_A027")
            self.assertEqual(record.sequence[18].end, "5'")
            self.assertEqual(record.sequence[18].lid, "13843")
            self.assertEqual(record.sequence[18].seqtype, "EST")
            self.assertEqual(record.sequence[18].trace, "891188752")
            self.assertEqual(record.sequence[19].acc, "CD464482.1")
            self.assertEqual(record.sequence[19].nid, "g31385750")
            self.assertEqual(record.sequence[19].clone, "LeukoN4_4_D01_A026")
            self.assertEqual(record.sequence[19].end, "3'")
            self.assertEqual(record.sequence[19].lid, "13771")
            self.assertEqual(record.sequence[19].seqtype, "EST")
            self.assertEqual(record.sequence[19].trace, "891187846")
            self.assertEqual(record.sequence[20].acc, "CD468770.1")
            self.assertEqual(record.sequence[20].nid, "g31390038")
            self.assertEqual(record.sequence[20].clone, "LeukoS3_5_D03_A025")
            self.assertEqual(record.sequence[20].end, "5'")
            self.assertEqual(record.sequence[20].lid, "13775")
            self.assertEqual(record.sequence[20].seqtype, "EST")
            self.assertEqual(record.sequence[20].trace, "891192280")
            self.assertEqual(record.sequence[21].acc, "CD470966.1")
            self.assertEqual(record.sequence[21].nid, "g31392234")
            self.assertEqual(record.sequence[21].clone, "LeukoS5_3_F05_A027")
            self.assertEqual(record.sequence[21].end, "5'")
            self.assertEqual(record.sequence[21].lid, "13778")
            self.assertEqual(record.sequence[21].seqtype, "EST")
            self.assertEqual(record.sequence[21].trace, "891193981")
            self.assertEqual(record.sequence[22].acc, "CD536398.1")
            self.assertEqual(record.sequence[22].nid, "g31578813")
            self.assertEqual(record.sequence[22].clone, "LeukoN6_7_C09_A028")
            self.assertEqual(record.sequence[22].end, "5'")
            self.assertEqual(record.sequence[22].lid, "13844")
            self.assertEqual(record.sequence[22].seqtype, "EST")
            self.assertEqual(record.sequence[22].trace, "891184341")
            self.assertEqual(record.sequence[23].acc, "CD535162.1")
            self.assertEqual(record.sequence[23].nid, "g31577577")
            self.assertEqual(record.sequence[23].clone, "LeukoN5_1_E01_A027")
            self.assertEqual(record.sequence[23].end, "3'")
            self.assertEqual(record.sequence[23].lid, "13843")
            self.assertEqual(record.sequence[23].seqtype, "EST")
            self.assertEqual(record.sequence[23].trace, "891188481")
            self.assertEqual(record.sequence[24].acc, "CD536377.1")
            self.assertEqual(record.sequence[24].nid, "g31578792")
            self.assertEqual(record.sequence[24].clone, "LeukoN6_7_C09_A028")
            self.assertEqual(record.sequence[24].end, "3'")
            self.assertEqual(record.sequence[24].lid, "13844")
            self.assertEqual(record.sequence[24].seqtype, "EST")
            self.assertEqual(record.sequence[24].trace, "891184713")
            self.assertEqual(record.sequence[25].acc, "CX603705.1")
            self.assertEqual(record.sequence[25].nid, "g57720427")
            self.assertEqual(record.sequence[25].clone, "CT02036A2C09")
            self.assertEqual(record.sequence[25].lid, "16895")
            self.assertEqual(record.sequence[25].seqtype, "EST")
            self.assertEqual(record.sequence[26].acc, "CD470556.1")
            self.assertEqual(record.sequence[26].nid, "g31391824")
            self.assertEqual(record.sequence[26].clone, "LeukoS4_5_C09_A026")
            self.assertEqual(record.sequence[26].end, "5'")
            self.assertEqual(record.sequence[26].lid, "13777")
            self.assertEqual(record.sequence[26].seqtype, "EST")
            self.assertEqual(record.sequence[26].trace, "891193234")
            self.assertEqual(record.sequence[27].acc, "CD466384.1")
            self.assertEqual(record.sequence[27].nid, "g31387652")
            self.assertEqual(record.sequence[27].clone, "LeukoN2_3_A06_A024")
            self.assertEqual(record.sequence[27].end, "5'")
            self.assertEqual(record.sequence[27].lid, "13773")
            self.assertEqual(record.sequence[27].seqtype, "EST")
            self.assertEqual(record.sequence[27].trace, "891186500")
            self.assertEqual(record.sequence[28].acc, "CD472213.1")
            self.assertEqual(record.sequence[28].nid, "g31393481")
            self.assertEqual(record.sequence[28].clone, "LeukoS6_1_C10_A028")
            self.assertEqual(record.sequence[28].end, "5'")
            self.assertEqual(record.sequence[28].lid, "13779")
            self.assertEqual(record.sequence[28].seqtype, "EST")
            self.assertEqual(record.sequence[28].trace, "891194841")
            self.assertEqual(record.sequence[29].acc, "DN510913.1")
            self.assertEqual(record.sequence[29].nid, "g60721103")
            self.assertEqual(record.sequence[29].clone, "HL02021A2C09")
            self.assertEqual(record.sequence[29].lid, "17147")
            self.assertEqual(record.sequence[29].seqtype, "EST")
            self.assertEqual(record.sequence[30].acc, "CD472329.1")
            self.assertEqual(record.sequence[30].nid, "g31393597")
            self.assertEqual(record.sequence[30].clone, "LeukoS6_5_H05_A028")
            self.assertEqual(record.sequence[30].end, "3'")
            self.assertEqual(record.sequence[30].lid, "13779")
            self.assertEqual(record.sequence[30].seqtype, "EST")
            self.assertEqual(record.sequence[30].trace, "891195397")
            self.assertEqual(record.sequence[31].acc, "CD536804.1")
            self.assertEqual(record.sequence[31].nid, "g31579219")
            self.assertEqual(record.sequence[31].clone, "LeukoN6_6_F12_A028")
            self.assertEqual(record.sequence[31].end, "3'")
            self.assertEqual(record.sequence[31].lid, "13844")
            self.assertEqual(record.sequence[31].seqtype, "EST")
            self.assertEqual(record.sequence[31].trace, "891184838")
            self.assertEqual(record.sequence[32].acc, "CD466071.1")
            self.assertEqual(record.sequence[32].nid, "g31387339")
            self.assertEqual(record.sequence[32].clone, "LeukoN2_1_E06_A024")
            self.assertEqual(record.sequence[32].end, "5'")
            self.assertEqual(record.sequence[32].lid, "13773")
            self.assertEqual(record.sequence[32].seqtype, "EST")
            self.assertEqual(record.sequence[32].trace, "891186571")
            self.assertEqual(record.sequence[33].acc, "CD471909.1")
            self.assertEqual(record.sequence[33].nid, "g31393177")
            self.assertEqual(record.sequence[33].clone, "LeukoS6_2_D05_A028")
            self.assertEqual(record.sequence[33].end, "5'")
            self.assertEqual(record.sequence[33].lid, "13779")
            self.assertEqual(record.sequence[33].seqtype, "EST")
            self.assertEqual(record.sequence[33].trace, "891195132")
            self.assertEqual(record.sequence[34].acc, "CD465985.1")
            self.assertEqual(record.sequence[34].nid, "g31387253")
            self.assertEqual(record.sequence[34].clone, "LeukoN2_1_E06_A024")
            self.assertEqual(record.sequence[34].end, "3'")
            self.assertEqual(record.sequence[34].lid, "13773")
            self.assertEqual(record.sequence[34].seqtype, "EST")
            self.assertEqual(record.sequence[34].trace, "891186530")
            self.assertEqual(record.sequence[35].acc, "CD471244.1")
            self.assertEqual(record.sequence[35].nid, "g31392512")
            self.assertEqual(record.sequence[35].clone, "LeukoS5_1_F09_A027")
            self.assertEqual(record.sequence[35].end, "5'")
            self.assertEqual(record.sequence[35].lid, "13778")
            self.assertEqual(record.sequence[35].seqtype, "EST")
            self.assertEqual(record.sequence[35].trace, "891193761")
            self.assertEqual(record.sequence[36].acc, "DN508615.1")
            self.assertEqual(record.sequence[36].nid, "g60718805")
            self.assertEqual(record.sequence[36].clone, "HL02013A1F04")
            self.assertEqual(record.sequence[36].lid, "17147")
            self.assertEqual(record.sequence[36].seqtype, "EST")
            self.assertEqual(record.sequence[37].acc, "CD465146.1")
            self.assertEqual(record.sequence[37].nid, "g31386414")
            self.assertEqual(record.sequence[37].clone, "LeukoN1_2_E04_A023")
            self.assertEqual(record.sequence[37].end, "3'")
            self.assertEqual(record.sequence[37].lid, "13772")
            self.assertEqual(record.sequence[37].seqtype, "EST")
            self.assertEqual(record.sequence[37].trace, "891185503")
            self.assertEqual(record.sequence[38].acc, "CD465620.1")
            self.assertEqual(record.sequence[38].nid, "g31386888")
            self.assertEqual(record.sequence[38].clone, "LeukoN1_6_E09_A023")
            self.assertEqual(record.sequence[38].end, "3'")
            self.assertEqual(record.sequence[38].lid, "13772")
            self.assertEqual(record.sequence[38].seqtype, "EST")
            self.assertEqual(record.sequence[38].trace, "891184965")
            self.assertEqual(record.sequence[39].acc, "CD472042.1")
            self.assertEqual(record.sequence[39].nid, "g31393310")
            self.assertEqual(record.sequence[39].clone, "LeukoS6_3_F02_A028")
            self.assertEqual(record.sequence[39].end, "5'")
            self.assertEqual(record.sequence[39].lid, "13779")
            self.assertEqual(record.sequence[39].seqtype, "EST")
            self.assertEqual(record.sequence[39].trace, "891194801")
            self.assertEqual(record.sequence[40].acc, "CD471148.1")
            self.assertEqual(record.sequence[40].nid, "g31392416")
            self.assertEqual(record.sequence[40].clone, "LeukoS5_4_F11_A027")
            self.assertEqual(record.sequence[40].end, "5'")
            self.assertEqual(record.sequence[40].lid, "13778")
            self.assertEqual(record.sequence[40].seqtype, "EST")
            self.assertEqual(record.sequence[40].trace, "891194177")
            self.assertEqual(record.sequence[41].acc, "CD466241.1")
            self.assertEqual(record.sequence[41].nid, "g31387509")
            self.assertEqual(record.sequence[41].clone, "LeukoN2_3_A06_A024")
            self.assertEqual(record.sequence[41].end, "3'")
            self.assertEqual(record.sequence[41].lid, "13773")
            self.assertEqual(record.sequence[41].seqtype, "EST")
            self.assertEqual(record.sequence[41].trace, "891186007")
            self.assertEqual(record.sequence[42].acc, "CD472182.1")
            self.assertEqual(record.sequence[42].nid, "g31393450")
            self.assertEqual(record.sequence[42].clone, "LeukoS6_1_H10_A028")
            self.assertEqual(record.sequence[42].end, "5'")
            self.assertEqual(record.sequence[42].lid, "13779")
            self.assertEqual(record.sequence[42].seqtype, "EST")
            self.assertEqual(record.sequence[42].trace, "891194593")
            self.assertEqual(record.sequence[43].acc, "CD470462.1")
            self.assertEqual(record.sequence[43].nid, "g31391730")
            self.assertEqual(record.sequence[43].clone, "LeukoS4_5_C12_A026")
            self.assertEqual(record.sequence[43].end, "3'")
            self.assertEqual(record.sequence[43].lid, "13777")
            self.assertEqual(record.sequence[43].seqtype, "EST")
            self.assertEqual(record.sequence[43].trace, "891193322")
            self.assertEqual(record.sequence[44].acc, "CX596662.1")
            self.assertEqual(record.sequence[44].nid, "g57706330")
            self.assertEqual(record.sequence[44].clone, "CT020015A10A06")
            self.assertEqual(record.sequence[44].lid, "16895")
            self.assertEqual(record.sequence[44].seqtype, "EST")
            self.assertEqual(record.sequence[45].acc, "CD536825.1")
            self.assertEqual(record.sequence[45].nid, "g31579240")
            self.assertEqual(record.sequence[45].clone, "LeukoN6_6_F12_A028")
            self.assertEqual(record.sequence[45].end, "5'")
            self.assertEqual(record.sequence[45].lid, "13844")
            self.assertEqual(record.sequence[45].seqtype, "EST")
            self.assertEqual(record.sequence[45].trace, "891184460")
            self.assertEqual(record.sequence[46].acc, "CD471373.1")
            self.assertEqual(record.sequence[46].nid, "g31392641")
            self.assertEqual(record.sequence[46].clone, "LeukoS5_5_H05_A027")
            self.assertEqual(record.sequence[46].end, "3'")
            self.assertEqual(record.sequence[46].lid, "13778")
            self.assertEqual(record.sequence[46].seqtype, "EST")
            self.assertEqual(record.sequence[46].trace, "891194564")
            self.assertEqual(record.sequence[47].acc, "CD470875.1")
            self.assertEqual(record.sequence[47].nid, "g31392143")
            self.assertEqual(record.sequence[47].clone, "LeukoS5_3_F05_A027")
            self.assertEqual(record.sequence[47].end, "3'")
            self.assertEqual(record.sequence[47].lid, "13778")
            self.assertEqual(record.sequence[47].seqtype, "EST")
            self.assertEqual(record.sequence[47].trace, "891193793")
            self.assertEqual(record.sequence[48].acc, "CD470392.1")
            self.assertEqual(record.sequence[48].nid, "g31391660")
            self.assertEqual(record.sequence[48].clone, "LeukoS4_4_G05_A026")
            self.assertEqual(record.sequence[48].end, "3'")
            self.assertEqual(record.sequence[48].lid, "13777")
            self.assertEqual(record.sequence[48].seqtype, "EST")
            self.assertEqual(record.sequence[48].trace, "891193723")
            self.assertEqual(record.sequence[49].acc, "DN509023.1")
            self.assertEqual(record.sequence[49].nid, "g60719213")
            self.assertEqual(record.sequence[49].clone, "HL02014B1D10")
            self.assertEqual(record.sequence[49].lid, "17147")
            self.assertEqual(record.sequence[49].seqtype, "EST")
            self.assertEqual(record.sequence[50].acc, "CD471226.1")
            self.assertEqual(record.sequence[50].nid, "g31392494")
            self.assertEqual(record.sequence[50].clone, "LeukoS5_1_F09_A027")
            self.assertEqual(record.sequence[50].end, "3'")
            self.assertEqual(record.sequence[50].lid, "13778")
            self.assertEqual(record.sequence[50].seqtype, "EST")
            self.assertEqual(record.sequence[50].trace, "891194172")
            self.assertEqual(record.sequence[51].acc, "CD470547.1")
            self.assertEqual(record.sequence[51].nid, "g31391815")
            self.assertEqual(record.sequence[51].clone, "LeukoS4_5_C12_A026")
            self.assertEqual(record.sequence[51].end, "5'")
            self.assertEqual(record.sequence[51].lid, "13777")
            self.assertEqual(record.sequence[51].seqtype, "EST")
            self.assertEqual(record.sequence[51].trace, "891193220")
            self.assertEqual(record.sequence[52].acc, "CX593399.1")
            self.assertEqual(record.sequence[52].nid, "g57699803")
            self.assertEqual(record.sequence[52].clone, "CT020005A10B06")
            self.assertEqual(record.sequence[52].lid, "16895")
            self.assertEqual(record.sequence[52].seqtype, "EST")
            self.assertEqual(record.sequence[53].acc, "CD465261.1")
            self.assertEqual(record.sequence[53].nid, "g31386529")
            self.assertEqual(record.sequence[53].clone, "LeukoN1_2_E04_A023")
            self.assertEqual(record.sequence[53].end, "5'")
            self.assertEqual(record.sequence[53].lid, "13772")
            self.assertEqual(record.sequence[53].seqtype, "EST")
            self.assertEqual(record.sequence[53].trace, "891185735")
            self.assertEqual(record.sequence[54].acc, "CD470468.1")
            self.assertEqual(record.sequence[54].nid, "g31391736")
            self.assertEqual(record.sequence[54].clone, "LeukoS4_5_C09_A026")
            self.assertEqual(record.sequence[54].end, "3'")
            self.assertEqual(record.sequence[54].lid, "13777")
            self.assertEqual(record.sequence[54].seqtype, "EST")
            self.assertEqual(record.sequence[54].trace, "891193330")
            self.assertEqual(record.sequence[55].acc, "CX594874.1")
            self.assertEqual(record.sequence[55].nid, "g57702753")
            self.assertEqual(record.sequence[55].clone, "CT020009B10E10")
            self.assertEqual(record.sequence[55].lid, "16895")
            self.assertEqual(record.sequence[55].seqtype, "EST")
            self.assertEqual(record.sequence[56].acc, "CD464520.1")
            self.assertEqual(record.sequence[56].nid, "g31385788")
            self.assertEqual(record.sequence[56].clone, "LeukoN4_4_D01_A026")
            self.assertEqual(record.sequence[56].end, "5'")
            self.assertEqual(record.sequence[56].lid, "13771")
            self.assertEqual(record.sequence[56].seqtype, "EST")
            self.assertEqual(record.sequence[56].trace, "891188021")
            self.assertEqual(record.sequence[57].acc, "AW260814.1")
            self.assertEqual(record.sequence[57].nid, "g6636524")
            self.assertEqual(record.sequence[57].end, "5'")
            self.assertEqual(record.sequence[57].lid, "2777")
            self.assertEqual(record.sequence[57].seqtype, "EST")
            self.assertEqual(record.sequence[58].acc, "CD465691.1")
            self.assertEqual(record.sequence[58].nid, "g31386959")
            self.assertEqual(record.sequence[58].clone, "LeukoN1_6_E09_A023")
            self.assertEqual(record.sequence[58].end, "5'")
            self.assertEqual(record.sequence[58].lid, "13772")
            self.assertEqual(record.sequence[58].seqtype, "EST")
            self.assertEqual(record.sequence[58].trace, "891185089")
            self.assertEqual(record.sequence[59].acc, "CX595434.1")
            self.assertEqual(record.sequence[59].nid, "g57703875")
            self.assertEqual(record.sequence[59].clone, "CT020011A20F11")
            self.assertEqual(record.sequence[59].lid, "16895")
            self.assertEqual(record.sequence[59].seqtype, "EST")
            self.assertEqual(record.sequence[60].acc, "CD464759.1")
            self.assertEqual(record.sequence[60].nid, "g31386027")
            self.assertEqual(record.sequence[60].clone, "LeukoN4_6_A09_A026")
            self.assertEqual(record.sequence[60].end, "5'")
            self.assertEqual(record.sequence[60].lid, "13771")
            self.assertEqual(record.sequence[60].seqtype, "EST")
            self.assertEqual(record.sequence[60].trace, "891188174")
            self.assertEqual(record.sequence[61].acc, "CD469441.1")
            self.assertEqual(record.sequence[61].nid, "g31390709")
            self.assertEqual(record.sequence[61].clone, "LeukoS2_3_F03_A024")
            self.assertEqual(record.sequence[61].end, "5'")
            self.assertEqual(record.sequence[61].lid, "13776")
            self.assertEqual(record.sequence[61].seqtype, "EST")
            self.assertEqual(record.sequence[61].trace, "946412740")
            self.assertEqual(record.sequence[62].acc, "CD471416.1")
            self.assertEqual(record.sequence[62].nid, "g31392684")
            self.assertEqual(record.sequence[62].clone, "LeukoS5_5_H05_A027")
            self.assertEqual(record.sequence[62].end, "5'")
            self.assertEqual(record.sequence[62].lid, "13778")
            self.assertEqual(record.sequence[62].seqtype, "EST")
            self.assertEqual(record.sequence[62].trace, "891194460")
            self.assertEqual(record.sequence[63].acc, "CD536439.1")
            self.assertEqual(record.sequence[63].nid, "g31578854")
            self.assertEqual(record.sequence[63].clone, "LeukoN6_7_B04_A028")
            self.assertEqual(record.sequence[63].end, "5'")
            self.assertEqual(record.sequence[63].lid, "13844")
            self.assertEqual(record.sequence[63].seqtype, "EST")
            self.assertEqual(record.sequence[63].trace, "891184662")
            self.assertEqual(record.sequence[64].acc, "CX605282.1")
            self.assertEqual(record.sequence[64].nid, "g57723597")
            self.assertEqual(record.sequence[64].clone, "CT02041A1A07")
            self.assertEqual(record.sequence[64].lid, "16895")
            self.assertEqual(record.sequence[64].seqtype, "EST")
            self.assertEqual(record.sequence[65].acc, "CD464302.1")
            self.assertEqual(record.sequence[65].nid, "g31385570")
            self.assertEqual(record.sequence[65].clone, "LeukoN4_3_B01_A026")
            self.assertEqual(record.sequence[65].end, "3'")
            self.assertEqual(record.sequence[65].lid, "13771")
            self.assertEqual(record.sequence[65].seqtype, "EST")
            self.assertEqual(record.sequence[65].trace, "891187636")
            self.assertEqual(record.sequence[66].acc, "CD469295.1")
            self.assertEqual(record.sequence[66].nid, "g31390563")
            self.assertEqual(record.sequence[66].clone, "LeukoS2_3_F03_A024")
            self.assertEqual(record.sequence[66].end, "3'")
            self.assertEqual(record.sequence[66].lid, "13776")
            self.assertEqual(record.sequence[66].seqtype, "EST")
            self.assertEqual(record.sequence[66].trace, "891191446")
            self.assertEqual(record.sequence[67].acc, "CD470883.1")
            self.assertEqual(record.sequence[67].nid, "g31392151")
            self.assertEqual(record.sequence[67].clone, "LeukoS5_3_G08_A027")
            self.assertEqual(record.sequence[67].end, "3'")
            self.assertEqual(record.sequence[67].lid, "13778")
            self.assertEqual(record.sequence[67].seqtype, "EST")
            self.assertEqual(record.sequence[67].trace, "891193865")
            self.assertEqual(record.sequence[68].acc, "CD468749.1")
            self.assertEqual(record.sequence[68].nid, "g31390017")
            self.assertEqual(record.sequence[68].clone, "LeukoS3_5_E02_A025")
            self.assertEqual(record.sequence[68].end, "5'")
            self.assertEqual(record.sequence[68].lid, "13775")
            self.assertEqual(record.sequence[68].seqtype, "EST")
            self.assertEqual(record.sequence[68].trace, "891192160")

            # Second record
            record = next(records)

            self.assertEqual(record.ID, "Eca.2425")
            self.assertEqual(record.title, "Immunoglobulin-like transcript 11 protein")
            self.assertEqual(record.symbol, "ILT11B")
            self.assertEqual(record.cytoband, "10p12")
            self.assertEqual(record.gene_id, "100034238")
            self.assertEqual(record.locuslink, "100034238")
            self.assertEqual(record.homol, True)
            self.assertEqual(record.express, ["blood", "adult"])
            self.assertEqual(len(record.protsim), 11)
            self.assertEqual(record.protsim[0].org, "10090")
            self.assertEqual(record.protsim[0].protgi, "156766061")
            self.assertEqual(record.protsim[0].protid, "NP_001074708.2")
            self.assertEqual(record.protsim[0].pct, "50.00")
            self.assertEqual(record.protsim[0].aln, "281")
            self.assertEqual(record.protsim[1].org, "9606")
            self.assertEqual(record.protsim[1].protgi, "32490553")
            self.assertEqual(record.protsim[1].protid, "NP_067073.1")
            self.assertEqual(record.protsim[1].pct, "62.42")
            self.assertEqual(record.protsim[1].aln, "296")
            self.assertEqual(record.protsim[2].org, "13616")
            self.assertEqual(record.protsim[2].protgi, "126346255")
            self.assertEqual(record.protsim[2].protid, "XP_001375479.1")
            self.assertEqual(record.protsim[2].pct, "40.43")
            self.assertEqual(record.protsim[2].aln, "281")
            self.assertEqual(record.protsim[3].org, "9615")
            self.assertEqual(record.protsim[3].protgi, "73947540")
            self.assertEqual(record.protsim[3].protid, "XP_854148.1")
            self.assertEqual(record.protsim[3].pct, "42.11")
            self.assertEqual(record.protsim[3].aln, "303")
            self.assertEqual(record.protsim[4].org, "9913")
            self.assertEqual(record.protsim[4].protgi, "194674905")
            self.assertEqual(record.protsim[4].protid, "XP_001788837.1")
            self.assertEqual(record.protsim[4].pct, "61.11")
            self.assertEqual(record.protsim[4].aln, "233")
            self.assertEqual(record.protsim[5].org, "9258")
            self.assertEqual(record.protsim[5].protgi, "149631692")
            self.assertEqual(record.protsim[5].protid, "XP_001516158.1")
            self.assertEqual(record.protsim[5].pct, "35.65")
            self.assertEqual(record.protsim[5].aln, "229")
            self.assertEqual(record.protsim[6].org, "9796")
            self.assertEqual(record.protsim[6].protgi, "146262003")
            self.assertEqual(record.protsim[6].protid, "NP_001075996.1")
            self.assertEqual(record.protsim[6].pct, "100.00")
            self.assertEqual(record.protsim[6].aln, "279")
            self.assertEqual(record.protsim[7].org, "10116")
            self.assertEqual(record.protsim[7].protgi, "109461031")
            self.assertEqual(record.protsim[7].protid, "XP_001065532.1")
            self.assertEqual(record.protsim[7].pct, "52.77")
            self.assertEqual(record.protsim[7].aln, "269")
            self.assertEqual(record.protsim[8].org, "9544")
            self.assertEqual(record.protsim[8].protgi, "100818611")
            self.assertEqual(record.protsim[8].protid, "NP_001035767.1")
            self.assertEqual(record.protsim[8].pct, "57.35")
            self.assertEqual(record.protsim[8].aln, "277")
            self.assertEqual(record.protsim[9].org, "9823")
            self.assertEqual(record.protsim[9].protgi, "178057314")
            self.assertEqual(record.protsim[9].protid, "NP_001116615.1")
            self.assertEqual(record.protsim[9].pct, "43.01")
            self.assertEqual(record.protsim[9].aln, "278")
            self.assertEqual(record.protsim[10].org, "9598")
            self.assertEqual(record.protsim[10].protgi, "58801536")
            self.assertEqual(record.protsim[10].protid, "NP_001009045.1")
            self.assertEqual(record.protsim[10].pct, "54.51")
            self.assertEqual(record.protsim[10].aln, "276")
            self.assertEqual(len(record.sequence), 9)
            self.assertEqual(record.sequence[0].acc, "AB120409.1")
            self.assertEqual(record.sequence[0].nid, "g45597282")
            self.assertEqual(record.sequence[0].pid, "g45597283")
            self.assertEqual(record.sequence[0].seqtype, "mRNA")
            self.assertEqual(record.sequence[1].acc, "NM_001082527.1")
            self.assertEqual(record.sequence[1].nid, "g146262002")
            self.assertEqual(record.sequence[1].pid, "g146262003")
            self.assertEqual(record.sequence[1].seqtype, "mRNA")
            self.assertEqual(record.sequence[2].acc, "AB120410.1")
            self.assertEqual(record.sequence[2].nid, "g45597284")
            self.assertEqual(record.sequence[2].pid, "g45597285")
            self.assertEqual(record.sequence[2].seqtype, "mRNA")
            self.assertEqual(record.sequence[3].acc, "AB120411.1")
            self.assertEqual(record.sequence[3].nid, "g45597286")
            self.assertEqual(record.sequence[3].pid, "g45597287")
            self.assertEqual(record.sequence[3].seqtype, "mRNA")
            self.assertEqual(record.sequence[4].acc, "CD467202.1")
            self.assertEqual(record.sequence[4].nid, "g31388470")
            self.assertEqual(record.sequence[4].clone, "LeukoS1_2_D11_A023")
            self.assertEqual(record.sequence[4].end, "5'")
            self.assertEqual(record.sequence[4].lid, "13774")
            self.assertEqual(record.sequence[4].seqtype, "EST")
            self.assertEqual(record.sequence[4].trace, "891190473")
            self.assertEqual(record.sequence[5].acc, "CD535974.1")
            self.assertEqual(record.sequence[5].nid, "g31578389")
            self.assertEqual(record.sequence[5].clone, "LeukoN6_2_F07_A028")
            self.assertEqual(record.sequence[5].end, "3'")
            self.assertEqual(record.sequence[5].lid, "13844")
            self.assertEqual(record.sequence[5].seqtype, "EST")
            self.assertEqual(record.sequence[5].trace, "891184067")
            self.assertEqual(record.sequence[6].acc, "CD469544.1")
            self.assertEqual(record.sequence[6].nid, "g31390812")
            self.assertEqual(record.sequence[6].clone, "LeukoS2_4_D05_A024")
            self.assertEqual(record.sequence[6].end, "5'")
            self.assertEqual(record.sequence[6].lid, "13776")
            self.assertEqual(record.sequence[6].seqtype, "EST")
            self.assertEqual(record.sequence[6].trace, "891191700")
            self.assertEqual(record.sequence[7].acc, "CD528742.1")
            self.assertEqual(record.sequence[7].nid, "g31567364")
            self.assertEqual(record.sequence[7].clone, "LeukoN3_7_F08_A025")
            self.assertEqual(record.sequence[7].end, "3'")
            self.assertEqual(record.sequence[7].lid, "13842")
            self.assertEqual(record.sequence[7].seqtype, "EST")
            self.assertEqual(record.sequence[7].trace, "891187302")
            self.assertEqual(record.sequence[8].acc, "CD469487.1")
            self.assertEqual(record.sequence[8].nid, "g31390755")
            self.assertEqual(record.sequence[8].clone, "LeukoS2_4_D05_A024")
            self.assertEqual(record.sequence[8].end, "3'")
            self.assertEqual(record.sequence[8].lid, "13776")
            self.assertEqual(record.sequence[8].seqtype, "EST")
            self.assertEqual(record.sequence[8].trace, "891191801")

            # Make sure that there are no more records
            self.assertRaises(StopIteration, next, records)

        self.assertEqual(repr(record.sequence), "[ACC=AB120409.1; NID=g45597282;"
                                                " PID=g45597283; SEQTYPE=mRNA, "
                                                "ACC=NM_001082527.1; NID=g146262002; "
                                                "PID=g146262003; SEQTYPE=mRNA, ACC=AB120410.1; "
                                                "NID=g45597284; PID=g45597285; SEQTYPE=mRNA, "
                                                "ACC=AB120411.1; NID=g45597286; PID=g45597287; "
                                                "SEQTYPE=mRNA, ACC=CD467202.1; NID=g31388470; "
                                                "CLONE=LeukoS1_2_D11_A023; END=5'; LID=13774; "
                                                "SEQTYPE=EST; TRACE=891190473, ACC=CD535974.1; "
                                                "NID=g31578389; CLONE=LeukoN6_2_F07_A028; "
                                                "END=3'; LID=13844; SEQTYPE=EST; TRACE=891184067, "
                                                "ACC=CD469544.1; NID=g31390812; CLONE=LeukoS2_4_"
                                                "D05_A024; END=5'; LID=13776; SEQTYPE=EST; "
                                                "TRACE=891191700, ACC=CD528742.1; NID=g31567364; "
                                                "CLONE=LeukoN3_7_F08_A025; END=3'; LID=13842; "
                                                "SEQTYPE=EST; TRACE=891187302, ACC=CD469487.1; "
                                                "NID=g31390755; CLONE=LeukoS2_4_D05_A024; END=3'; "
                                                "LID=13776; SEQTYPE=EST; "
                                                "TRACE=891191801]")
        self.assertEqual(repr(record.protsim), "[ORG=10090; PROTGI=156766061; "
                                               "PROTID=NP_001074708.2; PCT=50.00; ALN=281, "
                                               "ORG=9606; PROTGI=32490553; PROTID=NP_067073.1; "
                                               "PCT=62.42; ALN=296, ORG=13616; PROTGI=126346255; "
                                               "PROTID=XP_001375479.1; PCT=40.43; ALN=281, "
                                               "ORG=9615; PROTGI=73947540; PROTID=XP_854148.1; "
                                               "PCT=42.11; ALN=303, ORG=9913; PROTGI=194674905; "
                                               "PROTID=XP_001788837.1; PCT=61.11; ALN=233, "
                                               "ORG=9258; PROTGI=149631692; "
                                               "PROTID=XP_001516158.1; PCT=35.65; ALN=229, "
                                               "ORG=9796; PROTGI=146262003; "
                                               "PROTID=NP_001075996.1; PCT=100.00; ALN=279, "
                                               "ORG=10116; PROTGI=109461031; "
                                               "PROTID=XP_001065532.1; PCT=52.77; ALN=269, "
                                               "ORG=9544; PROTGI=100818611; "
                                               "PROTID=NP_001035767.1; PCT=57.35; ALN=277, "
                                               "ORG=9823; PROTGI=178057314; "
                                               "PROTID=NP_001116615.1; PCT=43.01; ALN=278, "
                                               "ORG=9598; PROTGI=58801536; "
                                               "PROTID=NP_001009045.1; "
                                               "PCT=54.51; ALN=276]")

    def test_read(self):

        # Start of the UniGene file for Homo sapiens downloaded from:
        # ftp://ftp.ncbi.nih.gov/repository/UniGene/Homo_sapiens
        with open("UniGene/Hs.2.data") as handle:
            record = UniGene.read(handle)
        self.assertEqual(record.ID, "Hs.2")
        self.assertEqual(record.title, "N-acetyltransferase 2 (arylamine N-acetyltransferase)")
        self.assertEqual(record.symbol, "NAT2")
        self.assertEqual(record.cytoband, "8p22")
        self.assertEqual(record.gene_id, "10")
        self.assertEqual(record.locuslink, "10")
        self.assertEqual(record.homol, True)
        self.assertEqual(record.express, ["bone", "connective tissue", "intestine", "liver", "liver tumor", "normal", "soft tissue/muscle tissue tumor", "adult"])
        self.assertEqual(record.restr_expr, ["adult"])
        self.assertEqual(record.chromosome, "8")
        self.assertEqual(len(record.sts), 7)
        self.assertEqual(record.sts[0].acc, "PMC310725P3")
        self.assertEqual(record.sts[0].unists, "272646")
        self.assertEqual(record.sts[1].acc, "WIAF-2120")
        self.assertEqual(record.sts[1].unists, "44576")
        self.assertEqual(record.sts[2].acc, "G59899")
        self.assertEqual(record.sts[2].unists, "137181")
        self.assertEqual(record.sts[3].acc, "G06461")
        self.assertEqual(record.sts[3].unists, "17088")
        self.assertEqual(record.sts[4].acc, "GDB:310612")
        self.assertEqual(record.sts[4].unists, "156422")
        self.assertEqual(record.sts[5].acc, "GDB:310613")
        self.assertEqual(record.sts[5].unists, "156423")
        self.assertEqual(record.sts[6].acc, "GDB:187676")
        self.assertEqual(record.sts[6].unists, "155563")
        self.assertEqual(len(record.protsim), 13)
        self.assertEqual(record.protsim[0].org, "10090")
        self.assertEqual(record.protsim[0].protgi, "6754794")
        self.assertEqual(record.protsim[0].protid, "NP_035004.1")
        self.assertEqual(record.protsim[0].pct, "76.55")
        self.assertEqual(record.protsim[0].aln, "288")
        self.assertEqual(record.protsim[1].org, "9796")
        self.assertEqual(record.protsim[1].protgi, "149742490")
        self.assertEqual(record.protsim[1].protid, "XP_001487907.1")
        self.assertEqual(record.protsim[1].pct, "79.66")
        self.assertEqual(record.protsim[1].aln, "288")
        self.assertEqual(record.protsim[2].org, "9986")
        self.assertEqual(record.protsim[2].protgi, "126722851")
        self.assertEqual(record.protsim[2].protid, "NP_001075655.1")
        self.assertEqual(record.protsim[2].pct, "76.90")
        self.assertEqual(record.protsim[2].aln, "288")
        self.assertEqual(record.protsim[3].org, "9544")
        self.assertEqual(record.protsim[3].protgi, "113461974")
        self.assertEqual(record.protsim[3].protid, "NP_001038201.1")
        self.assertEqual(record.protsim[3].pct, "93.10")
        self.assertEqual(record.protsim[3].aln, "288")
        self.assertEqual(record.protsim[4].org, "9606")
        self.assertEqual(record.protsim[4].protgi, "116295260")
        self.assertEqual(record.protsim[4].protid, "NP_000006.2")
        self.assertEqual(record.protsim[4].pct, "100.00")
        self.assertEqual(record.protsim[4].aln, "288")
        self.assertEqual(record.protsim[5].org, "7719")
        self.assertEqual(record.protsim[5].protgi, "198423589")
        self.assertEqual(record.protsim[5].protid, "XP_002126320.1")
        self.assertEqual(record.protsim[5].pct, "29.46")
        self.assertEqual(record.protsim[5].aln, "240")
        self.assertEqual(record.protsim[6].org, "10116")
        self.assertEqual(record.protsim[6].protgi, "16758720")
        self.assertEqual(record.protsim[6].protid, "NP_446306.1")
        self.assertEqual(record.protsim[6].pct, "75.52")
        self.assertEqual(record.protsim[6].aln, "288")
        self.assertEqual(record.protsim[7].org, "13616")
        self.assertEqual(record.protsim[7].protgi, "126303190")
        self.assertEqual(record.protsim[7].protid, "XP_001371814.1")
        self.assertEqual(record.protsim[7].pct, "63.67")
        self.assertEqual(record.protsim[7].aln, "287")
        self.assertEqual(record.protsim[8].org, "9031")
        self.assertEqual(record.protsim[8].protgi, "45384408")
        self.assertEqual(record.protsim[8].protid, "NP_990671.1")
        self.assertEqual(record.protsim[8].pct, "57.93")
        self.assertEqual(record.protsim[8].aln, "288")
        self.assertEqual(record.protsim[9].org, "9913")
        self.assertEqual(record.protsim[9].protgi, "115497526")
        self.assertEqual(record.protsim[9].protid, "NP_001069040.1")
        self.assertEqual(record.protsim[9].pct, "80.34")
        self.assertEqual(record.protsim[9].aln, "288")
        self.assertEqual(record.protsim[10].org, "9258")
        self.assertEqual(record.protsim[10].protgi, "149640224")
        self.assertEqual(record.protsim[10].protid, "XP_001508302.1")
        self.assertEqual(record.protsim[10].pct, "62.37")
        self.assertEqual(record.protsim[10].aln, "278")
        self.assertEqual(record.protsim[11].org, "7955")
        self.assertEqual(record.protsim[11].protgi, "125821897")
        self.assertEqual(record.protsim[11].protid, "XP_001334322.1")
        self.assertEqual(record.protsim[11].pct, "43.21")
        self.assertEqual(record.protsim[11].aln, "279")
        self.assertEqual(record.protsim[12].org, "9598")
        self.assertEqual(record.protsim[12].protgi, "114619004")
        self.assertEqual(record.protsim[12].protid, "XP_519631.2")
        self.assertEqual(record.protsim[12].pct, "98.28")
        self.assertEqual(record.protsim[12].aln, "288")
        self.assertEqual(len(record.sequence), 38)
        self.assertEqual(record.sequence[0].acc, "BC067218.1")
        self.assertEqual(record.sequence[0].nid, "g45501306")
        self.assertEqual(record.sequence[0].pid, "g45501307")
        self.assertEqual(record.sequence[0].seqtype, "mRNA")
        self.assertEqual(record.sequence[1].acc, "NM_000015.2")
        self.assertEqual(record.sequence[1].nid, "g116295259")
        self.assertEqual(record.sequence[1].pid, "g116295260")
        self.assertEqual(record.sequence[1].seqtype, "mRNA")
        self.assertEqual(record.sequence[2].acc, "D90042.1")
        self.assertEqual(record.sequence[2].nid, "g219415")
        self.assertEqual(record.sequence[2].pid, "g219416")
        self.assertEqual(record.sequence[2].seqtype, "mRNA")
        self.assertEqual(record.sequence[3].acc, "D90040.1")
        self.assertEqual(record.sequence[3].nid, "g219411")
        self.assertEqual(record.sequence[3].pid, "g219412")
        self.assertEqual(record.sequence[3].seqtype, "mRNA")
        self.assertEqual(record.sequence[4].acc, "BC015878.1")
        self.assertEqual(record.sequence[4].nid, "g16198419")
        self.assertEqual(record.sequence[4].pid, "g16198420")
        self.assertEqual(record.sequence[4].seqtype, "mRNA")
        self.assertEqual(record.sequence[5].acc, "CR407631.1")
        self.assertEqual(record.sequence[5].nid, "g47115198")
        self.assertEqual(record.sequence[5].pid, "g47115199")
        self.assertEqual(record.sequence[5].seqtype, "mRNA")
        self.assertEqual(record.sequence[6].acc, "BG569293.1")
        self.assertEqual(record.sequence[6].nid, "g13576946")
        self.assertEqual(record.sequence[6].clone, "IMAGE:4722596")
        self.assertEqual(record.sequence[6].end, "5'")
        self.assertEqual(record.sequence[6].lid, "6989")
        self.assertEqual(record.sequence[6].seqtype, "EST")
        self.assertEqual(record.sequence[6].trace, "44157214")
        self.assertEqual(record.sequence[7].acc, "AI792606.1")
        self.assertEqual(record.sequence[7].nid, "g5340322")
        self.assertEqual(record.sequence[7].clone, "IMAGE:1870937")
        self.assertEqual(record.sequence[7].end, "5'")
        self.assertEqual(record.sequence[7].lid, "1079")
        self.assertEqual(record.sequence[7].seqtype, "EST")
        self.assertEqual(record.sequence[8].acc, "BG568794.1")
        self.assertEqual(record.sequence[8].nid, "g13576447")
        self.assertEqual(record.sequence[8].clone, "IMAGE:4716636")
        self.assertEqual(record.sequence[8].end, "5'")
        self.assertEqual(record.sequence[8].lid, "6989")
        self.assertEqual(record.sequence[8].seqtype, "EST")
        self.assertEqual(record.sequence[8].trace, "44156817")
        self.assertEqual(record.sequence[9].acc, "BG533459.1")
        self.assertEqual(record.sequence[9].nid, "g13524999")
        self.assertEqual(record.sequence[9].clone, "IMAGE:4072143")
        self.assertEqual(record.sequence[9].end, "5'")
        self.assertEqual(record.sequence[9].lid, "6989")
        self.assertEqual(record.sequence[9].seqtype, "EST")
        self.assertEqual(record.sequence[9].trace, "44404609")
        self.assertEqual(record.sequence[10].acc, "BG568400.1")
        self.assertEqual(record.sequence[10].nid, "g13576053")
        self.assertEqual(record.sequence[10].clone, "IMAGE:4716802")
        self.assertEqual(record.sequence[10].end, "5'")
        self.assertEqual(record.sequence[10].lid, "6989")
        self.assertEqual(record.sequence[10].seqtype, "EST")
        self.assertEqual(record.sequence[10].trace, "44156561")
        self.assertEqual(record.sequence[11].acc, "BG618195.1")
        self.assertEqual(record.sequence[11].nid, "g13669566")
        self.assertEqual(record.sequence[11].clone, "IMAGE:4767316")
        self.assertEqual(record.sequence[11].end, "5'")
        self.assertEqual(record.sequence[11].lid, "6989")
        self.assertEqual(record.sequence[11].seqtype, "EST")
        self.assertEqual(record.sequence[11].trace, "45338366")
        self.assertEqual(record.sequence[12].acc, "BG563731.1")
        self.assertEqual(record.sequence[12].nid, "g13571383")
        self.assertEqual(record.sequence[12].clone, "IMAGE:4712210")
        self.assertEqual(record.sequence[12].end, "5'")
        self.assertEqual(record.sequence[12].lid, "6989")
        self.assertEqual(record.sequence[12].seqtype, "EST")
        self.assertEqual(record.sequence[12].trace, "44153506")
        self.assertEqual(record.sequence[13].acc, "AI733799.1")
        self.assertEqual(record.sequence[13].nid, "g5054912")
        self.assertEqual(record.sequence[13].clone, "IMAGE:1870937")
        self.assertEqual(record.sequence[13].end, "3'")
        self.assertEqual(record.sequence[13].lid, "1079")
        self.assertEqual(record.sequence[13].seqtype, "EST")
        self.assertEqual(record.sequence[14].acc, "BG569272.1")
        self.assertEqual(record.sequence[14].nid, "g13576925")
        self.assertEqual(record.sequence[14].clone, "IMAGE:4722638")
        self.assertEqual(record.sequence[14].end, "5'")
        self.assertEqual(record.sequence[14].lid, "6989")
        self.assertEqual(record.sequence[14].seqtype, "EST")
        self.assertEqual(record.sequence[14].trace, "44157191")
        self.assertEqual(record.sequence[15].acc, "AJ581147.1")
        self.assertEqual(record.sequence[15].nid, "g73759744")
        self.assertEqual(record.sequence[15].pid, "g73759745")
        self.assertEqual(record.sequence[15].seqtype, "mRNA")
        self.assertEqual(record.sequence[16].acc, "BU624903.1")
        self.assertEqual(record.sequence[16].nid, "g23291118")
        self.assertEqual(record.sequence[16].clone, "UI-H-FG1-bgl-g-02-0-UI")
        self.assertEqual(record.sequence[16].end, "3'")
        self.assertEqual(record.sequence[16].lid, "11914")
        self.assertEqual(record.sequence[16].seqtype, "EST")
        self.assertEqual(record.sequence[16].trace, "159705553")
        self.assertEqual(record.sequence[17].acc, "BG617259.1")
        self.assertEqual(record.sequence[17].nid, "g13668630")
        self.assertEqual(record.sequence[17].clone, "IMAGE:4734378")
        self.assertEqual(record.sequence[17].end, "5'")
        self.assertEqual(record.sequence[17].lid, "6989")
        self.assertEqual(record.sequence[17].seqtype, "EST")
        self.assertEqual(record.sequence[17].trace, "44229423")
        self.assertEqual(record.sequence[18].acc, "CV029049.1")
        self.assertEqual(record.sequence[18].nid, "g51487181")
        self.assertEqual(record.sequence[18].end, "5'")
        self.assertEqual(record.sequence[18].lid, "16264")
        self.assertEqual(record.sequence[18].seqtype, "EST")
        self.assertEqual(record.sequence[19].acc, "BP264356.1")
        self.assertEqual(record.sequence[19].nid, "g52179587")
        self.assertEqual(record.sequence[19].clone, "HSI15615")
        self.assertEqual(record.sequence[19].end, "5'")
        self.assertEqual(record.sequence[19].lid, "16400")
        self.assertEqual(record.sequence[19].seqtype, "EST")
        self.assertEqual(record.sequence[20].acc, "BP262043.1")
        self.assertEqual(record.sequence[20].nid, "g52177274")
        self.assertEqual(record.sequence[20].clone, "HSI05750")
        self.assertEqual(record.sequence[20].end, "5'")
        self.assertEqual(record.sequence[20].lid, "16400")
        self.assertEqual(record.sequence[20].seqtype, "EST")
        self.assertEqual(record.sequence[21].acc, "BP262787.1")
        self.assertEqual(record.sequence[21].nid, "g52178018")
        self.assertEqual(record.sequence[21].clone, "HSI08034")
        self.assertEqual(record.sequence[21].end, "5'")
        self.assertEqual(record.sequence[21].lid, "16400")
        self.assertEqual(record.sequence[21].seqtype, "EST")
        self.assertEqual(record.sequence[22].acc, "CB161982.1")
        self.assertEqual(record.sequence[22].nid, "g28148108")
        self.assertEqual(record.sequence[22].clone, "L17N670205n1-15-F12")
        self.assertEqual(record.sequence[22].end, "5'")
        self.assertEqual(record.sequence[22].lid, "12542")
        self.assertEqual(record.sequence[22].seqtype, "EST")
        self.assertEqual(record.sequence[23].acc, "BX095770.1")
        self.assertEqual(record.sequence[23].nid, "g27827877")
        self.assertEqual(record.sequence[23].clone, "IMAGp998I184581_,_IMAGE:1870937")
        self.assertEqual(record.sequence[23].lid, "1079")
        self.assertEqual(record.sequence[23].seqtype, "EST")
        self.assertEqual(record.sequence[24].acc, "AI262683.1")
        self.assertEqual(record.sequence[24].nid, "g3870886")
        self.assertEqual(record.sequence[24].clone, "IMAGE:1870937")
        self.assertEqual(record.sequence[24].end, "3'")
        self.assertEqual(record.sequence[24].lid, "1079")
        self.assertEqual(record.sequence[24].seqtype, "EST")
        self.assertEqual(record.sequence[25].acc, "AI460128.1")
        self.assertEqual(record.sequence[25].nid, "g4313009")
        self.assertEqual(record.sequence[25].clone, "IMAGE:2151449")
        self.assertEqual(record.sequence[25].end, "3'")
        self.assertEqual(record.sequence[25].lid, "1556")
        self.assertEqual(record.sequence[25].seqtype, "EST")
        self.assertEqual(record.sequence[26].acc, "CB161860.1")
        self.assertEqual(record.sequence[26].nid, "g28147986")
        self.assertEqual(record.sequence[26].clone, "L17N670205n1-41-A04")
        self.assertEqual(record.sequence[26].end, "5'")
        self.assertEqual(record.sequence[26].lid, "12542")
        self.assertEqual(record.sequence[26].seqtype, "EST")
        self.assertEqual(record.sequence[27].acc, "AJ581145.1")
        self.assertEqual(record.sequence[27].nid, "g73759740")
        self.assertEqual(record.sequence[27].pid, "g73759741")
        self.assertEqual(record.sequence[27].seqtype, "mRNA")
        self.assertEqual(record.sequence[28].acc, "EG327405.1")
        self.assertEqual(record.sequence[28].nid, "g116004924")
        self.assertEqual(record.sequence[28].clone, "MGC12part.1.3.L1.1.D01.")
        self.assertEqual(record.sequence[28].lid, "17261")
        self.assertEqual(record.sequence[28].seqtype, "EST")
        self.assertEqual(record.sequence[29].acc, "AJ581144.1")
        self.assertEqual(record.sequence[29].nid, "g73759738")
        self.assertEqual(record.sequence[29].pid, "g73759739")
        self.assertEqual(record.sequence[29].seqtype, "mRNA")
        self.assertEqual(record.sequence[30].acc, "EG327340.1")
        self.assertEqual(record.sequence[30].nid, "g116004859")
        self.assertEqual(record.sequence[30].clone, "MGC12part.1.2.L1.1.D01.")
        self.assertEqual(record.sequence[30].lid, "17261")
        self.assertEqual(record.sequence[30].seqtype, "EST")
        self.assertEqual(record.sequence[31].acc, "AV658623.1")
        self.assertEqual(record.sequence[31].nid, "g9879637")
        self.assertEqual(record.sequence[31].clone, "GLCFOD10")
        self.assertEqual(record.sequence[31].end, "3'")
        self.assertEqual(record.sequence[31].lid, "5601")
        self.assertEqual(record.sequence[31].seqtype, "EST")
        self.assertEqual(record.sequence[32].acc, "AV658656.1")
        self.assertEqual(record.sequence[32].nid, "g9879670")
        self.assertEqual(record.sequence[32].clone, "GLCFOG07")
        self.assertEqual(record.sequence[32].end, "3'")
        self.assertEqual(record.sequence[32].lid, "5601")
        self.assertEqual(record.sequence[32].seqtype, "EST")
        self.assertEqual(record.sequence[33].acc, "AV684197.2")
        self.assertEqual(record.sequence[33].nid, "g55943471")
        self.assertEqual(record.sequence[33].clone, "GKCFZH06")
        self.assertEqual(record.sequence[33].end, "5'")
        self.assertEqual(record.sequence[33].lid, "6533")
        self.assertEqual(record.sequence[33].seqtype, "EST")
        self.assertEqual(record.sequence[34].acc, "AJ581146.1")
        self.assertEqual(record.sequence[34].nid, "g73759742")
        self.assertEqual(record.sequence[34].pid, "g73759743")
        self.assertEqual(record.sequence[34].seqtype, "mRNA")
        self.assertEqual(record.sequence[35].acc, "EG327618.1")
        self.assertEqual(record.sequence[35].nid, "g116005137")
        self.assertEqual(record.sequence[35].clone, "MGC11part.2.3.L1.1.E06.")
        self.assertEqual(record.sequence[35].lid, "17261")
        self.assertEqual(record.sequence[35].seqtype, "EST")
        self.assertEqual(record.sequence[36].acc, "BG204539.1")
        self.assertEqual(record.sequence[36].nid, "g13726226")
        self.assertEqual(record.sequence[36].lid, "8655")
        self.assertEqual(record.sequence[36].seqtype, "EST")
        self.assertEqual(record.sequence[37].acc, "AU099534.1")
        self.assertEqual(record.sequence[37].nid, "g13550663")
        self.assertEqual(record.sequence[37].clone, "HSI08034")
        self.assertEqual(record.sequence[37].end, "5'")
        self.assertEqual(record.sequence[37].lid, "8800")
        self.assertEqual(record.sequence[37].seqtype, "EST")

        self.assertEqual(repr(record.sts), "[ACC=PMC310725P3 UNISTS=272646, ACC=WIAF-2120 "
                                           "UNISTS=44576, ACC=G59899 UNISTS=137181, ACC=G06461 "
                                           "UNISTS=17088, ACC=GDB:310612 UNISTS=156422, "
                                           "ACC=GDB:310613 UNISTS=156423, ACC=GDB:187676 "
                                           "UNISTS=155563]")

        self.assertEqual(repr(record), "<Record> Hs.2 NAT2 N-acetyltransferase 2 "
                                       "(arylamine N-acetyltransferase)")

    def test_read_value_error(self):

        # Test More than one SwissProt record found
        with open("UniGene/Eca.1.2425.data") as handle:
            self.assertRaises(ValueError, UniGene.read, handle)

        # Test No SwissProt record found, reached the end of record
        with open("UniGene/Hs.2.data") as handle:
            UniGene.read(handle)
            self.assertRaises(ValueError, UniGene.read, handle)


if __name__ == "__main__":
    runner = unittest.TextTestRunner(verbosity=2)
    unittest.main(testRunner=runner)
