import os
import sys
import pytest
import shutil
import astropy  # noqa

ROOT = os.path.join(os.path.dirname(__file__), '../')

# Make sure we don't allow any arguments to be passed - some tests call
# sys.executable which becomes this script when producing a pyinstaller
# bundle, but we should just error in this case since this is not the
# regular Python interpreter.
if len(sys.argv) > 1:
    print("Extra arguments passed, exiting early")
    sys.exit(1)

# Copy over the astropy 'tests' directories and their contents
for root, dirnames, _ in os.walk(os.path.join(ROOT, 'astropy')):
    for dirname in dirnames:
        final_dir = os.path.relpath(os.path.join(root.replace('astropy', 'astropy_tests'), dirname), ROOT)
        # We only copy over 'tests' directories, but not astropy/tests (only
        # astropy/tests/tests) since that is not just a directory with tests.
        if dirname == 'tests' and not root.endswith('astropy'):
            shutil.copytree(os.path.join(root, dirname), final_dir, dirs_exist_ok=True)
        else:
            # Create empty __init__.py files so that 'astropy_tests' still
            # behaves like a single package, otherwise pytest gets confused
            # by the different conftest.py files.
            init_filename = os.path.join(final_dir, '__init__.py')
            if not os.path.exists(os.path.join(final_dir, '__init__.py')):
                os.makedirs(final_dir, exist_ok=True)
                with open(os.path.join(final_dir, '__init__.py'), 'w') as f:
                    f.write("#")

# Add the top-level __init__.py file
with open(os.path.join('astropy_tests', '__init__.py'), 'w') as f:
    f.write("#")

# Copy the top-level conftest.py
shutil.copy2(os.path.join(ROOT, 'astropy', 'conftest.py'),
             os.path.join('astropy_tests', 'conftest.py'))

# We skip a few tests, which are generally ones that rely on explicitly
# checking the name of the current module (which ends up starting with
# astropy_tests rather than astropy).

SKIP_TESTS = ['test_exception_logging_origin',
              'test_log',
              'test_configitem',
              'test_config_noastropy_fallback',
              'test_no_home',
              'test_path',
              'test_rename_path',
              'test_data_name_third_party_package',
              'test_pkg_finder',
              'test_wcsapi_extension',
              'test_find_current_module_bundle',
              'test_download_parallel_fills_cache']

# Run the tests!
sys.exit(pytest.main(['astropy_tests',
                      '-k ' + ' and '.join('not ' + test for test in SKIP_TESTS)],
                     plugins=['pytest_doctestplus.plugin',
                              'pytest_openfiles.plugin',
                              'pytest_remotedata.plugin',
                              'pytest_mpl.plugin',
                              'pytest_astropy_header.display']))
