# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""some common report transports"""

from os.path import join
from cStringIO import StringIO
from smtplib import SMTP
import mimetypes
import MimeWriter
import base64
import socket
import warnings

from logilab.common.fileutils import write_open_mode
from logilab.common.xmlrpcutils import connect, ProtocolError

from apycot import register, NotSupported, ConfigError
from apycot.interfaces import ITransport
from apycot.utils import SimpleOptionsManagerMixIn, init_directory


class BaseTransport(SimpleOptionsManagerMixIn):
    """base class for transports"""
    def __init__(self, verbosity):
        SimpleOptionsManagerMixIn.__init__(self)
        self.verbosity = verbosity
        self.encoding = None
        self.date = None
        
    def open(self, date):
        """open the transport transaction, initialize date and encoding"""
        self.encoding = self.get_option('encoding', 'UTF-8')
        self.date = ['%02d' % i for i in date]
        assert self.date
        self.check_options()
        
    def close(self):
        """close the transport transaction, do nothing by default"""
        
    def encode(self, string):
        """helper method returning text value of a node as an encoded string
        """
        return string.encode(self.encoding)    

    def base_url(self, date=None):
        """return the base transport url, not supported by default"""
        raise NotSupported()


class FileTransport(BaseTransport):
    """write reports on the file system"""
    
    __implements__ = ITransport
    __name__ = 'file'
    required_options = (('destination', 'No destination directory'),
                        )
    
    def __init__(self, verbosity):
        BaseTransport.__init__(self, verbosity)
        self.current = None
        self.destination = None
        
    def prefixed_path(self, output_file):
        """return file path according to settings, given value or default"""
        assert output_file
        return join(self.base_path(), output_file)
    
    def part_stream(self, partname):
        """get a stream object to receive the next report"""
        path = self.prefixed_path(partname)
        self.current = open(path, write_open_mode(partname))
        return self.current
    
    def add_part(self, partname, stream):
        """transport a report part"""
        if self.current is None:
            stream.seek(0)
            fileobj = self.part_stream(partname)
            fileobj.write(stream.read())
            stream = fileobj
        else:
            assert stream is self.current
        stream.close()
        self.current = None
        
    def base_path(self, date=None):
        """return the base path of the transport"""
        if date is not None:
            raise NotSupported()
        if self.destination is None:
            self.destination = self.get_option('destination')
        return self.destination

    def base_url(self, date=None):
        """return the base transport url"""
        if date is not None:
            raise NotSupported()
        try:
            return self.get_option('url_base')
        except ConfigError:
            return self.base_path()

register('transport', FileTransport)

class DateFileTransport(FileTransport):
    """writer reports on the file system, in a directory specific to the
    report's date
    """
    __implements__ = ITransport
    __name__ = 'datedfile'

    def base_path(self, date=None):
        """return the base path of the transport"""
        if date is not None:
            destination = self.get_option('destination')
            return join(destination, *date)
        if self.destination is None:
            destination = self.get_option('destination')
            init_directory(destination, self.date)
            self.destination = join(destination, *self.date)
        return self.destination

    def base_url(self, date=None):
        """return the base transport url"""
        urlb = self.get_option('url_base', None)
        date = date or self.date
        if isinstance(date[0], int):
            date = ['%02d' % i for i in date]
        if urlb is not None:
            return urlb + '/'.join(date)
        return self.base_path(date)

register('transport', DateFileTransport)

#@deprecated
class DailyFileTransport(DateFileTransport):
    __name__ = "dailyfile"
    def __init__(*args, **kargs):
        warnings.warn(
            "Use of deprecated transport dailyfile instead of datedfile",
             category=DeprecationWarning)
        DateFileTransport.__init__(*args, **kargs)
register('transport', DailyFileTransport)

def zope_rpc_call(method, *args, **kwargs):
    """make a xml-rpc call to a zope method, filtering some exceptions"""
    try:
        method(*args, **kwargs)
    except ProtocolError, ex:
        if ex.errcode != 302: # ignore moved temporaly
            raise
    except socket.error:
        raise
    except Exception, ex:
        # exists already ?
        pass
        #print '-'*80
        #print ex, type(ex)
    

class DateZopeTransport(BaseTransport):
    """writer reports on a zope CMF site, in a directory specific to the
    report's date
    """
    __implements__ = ITransport
    __name__ = 'datedzope'

    def __init__(self, verbosity):
        BaseTransport.__init__(self, verbosity)
        self.user = None
        self.password = None
        self.server = None
        self.url = None
        
    def open(self, date):
        """open the transport transaction"""
        BaseTransport.open(self, date)
        url = self.get_option('url_base')
        self.user = user = self.get_option('user', None)
        self.password = password = self.get_option('password', None)
        print 'connecting to %s' % url
        server = connect(url, user, password)
        folder_type = self.get_option('folder_type', 'Folder')
        for part in self.date:
            url = '%s/%s' % (url, part)
            zope_rpc_call(server.invokeFactory, folder_type, part)
            server = connect(url, user, password)
        self.server = server
        self.url = url

    def part_stream(self, partname):
        """get a stream object to receive the next report"""
        return StringIO()
    
    def add_part(self, partname, stream):
        """transport a report part"""
        doc_type = self.get_option('document_type', 'RawDocument')
        zope_rpc_call(self.server.invokeFactory, doc_type, partname)
        obj = connect('%s/%s' % (self.url, partname), self.user, self.password)
        stream.seek(0)
        obj.my_file_edit(stream.getvalue())

    def base_url(self, date=None):
        """return the base transport url"""
        urlb = self.get_option('url_base')
        date = date or self.date
        if isinstance(date[0], int):
            date = ['%02d' % i for i in date]
        return urlb + '/'.join(date)

register('transport', DateZopeTransport)

class DailyZopeTransport(DateZopeTransport):
    """dummy class for compatibily purpose"""
    __name__ = "dailyzope"

    def __init__(*args, **kargs):
        warnings.warn(
            "Use of deprecated transport dailyzope instead of datedzope",
                              category=DeprecationWarning)
        DateZopeTransport.__init__(*args, **kargs)
register('transport', DailyZopeTransport)

        
class EmailTransport(BaseTransport):
    """send reports by email"""
    __implements__ = ITransport
    __name__ = 'email'
    required_options = (('dest', 'No destination address specified'),
                        ('from', 'No from address specified'))
    smtp_server = SMTP
    
    def __init__(self, verbosity):
        BaseTransport.__init__(self, verbosity)
        self.writer = None
        self.from_addr = None
        self.dest_addrs = None
        self.message = None
        
    def open(self, date):
        """open the transport transaction"""
        BaseTransport.open(self, date)
        self.from_addr = self.get_option('from')
        self.dest_addrs = self.get_option('dest')
        self.message = StringIO()
        self.writer = writer = MimeWriter.MimeWriter(self.message)
        writer.addheader('From', self.from_addr)
        writer.addheader('To', self.dest_addrs)
        writer.addheader('Subject', self.get_option('subject', 'APyCoT RePorT'))
        writer.startmultipartbody('mixed')
        
    def part_stream(self, partname):
        """get a stream object to receive the next report"""
        return StringIO()
    
    def add_part(self, partname, stream):
        """transport a report part"""
        part = self.writer.nextpart()
        mime = mimetypes.guess_type(partname)[0]
        body = part.startbody('%s; name=%s' % (mime, partname))
        stream.seek(0)
        if mime.startswith('text/'):
            body.write(stream.read())
        else:
            base64.encode(stream, body)

    def close(self):
        """close the transport transaction :
        send the mail report"""
        self.writer.lastpart()
        host = self.get_option('smtp_server', 'localhost')
        port = int(self.get_option('smtp_port', 25))
        if self.verbosity > 1:
            print 'Connecting to', host, port
        server = self.smtp_server(host, port)
        if int(self.get_option('smtp_debug', 0)):
            server.set_debuglevel(1)
        server.sendmail(self.from_addr, self.dest_addrs.split(', '),
                        self.message.getvalue())
        server.quit()
    
register('transport', EmailTransport)
