"""unit tests for the apycot.utils module"""

import sys, tempfile, os, shutil
from os.path import join, isdir
from time import localtime
from random import randint

from logilab.common.testlib import TestCase, unittest_main

from utils import input_path, access_key

from apycot.utils import *
MAIN_VARS = ('tests', 'verbosity', 'load', 'threads')

CONFIG_FILE = input_path('tester.ini')
CONFIG_FILE_INCLUDE = input_path('tester_include.ini')
CONFIG_FILE_INCLUDE_CIRCULAR = input_path('tester_include_rec.ini')
CONFIG_FILE_GLOB = input_path('glob.ini')
CONFIG_FILE_AUTO = input_path('auto_test.ini')
CONFIG, verb = parse_config(CONFIG_FILE)


class DataFileUtilsTC(TestCase):
    """Default tests for data file utilities functions"""
    
    def setUp(self):
        self.tempdir = tempfile.mktemp()
        os.mkdir(self.tempdir)

    def tearDown(self):
        shutil.rmtree(self.tempdir)
        
    def test_init_directory(self):
        filename = init_directory(self.tempdir, localtime()[:3])
        filename, tail = os.path.split(filename)
        while filename != self.tempdir:
            self.failUnless(os.path.exists(filename), '%s does not exist'
                                                                % filename)
            filename, tail = os.path.split(filename)

    def test_get_latest(self):
        dir = init_directory(self.tempdir, localtime()[:3])
        self.assert_(isdir(dir))
        file(join(dir, 'tester_data.xml'), 'w').close()
        self.failUnlessEqual(get_latest(self.tempdir), dir)
        tempdir = tempfile.mktemp()
        self.assertRaises(ConfigError, get_latest, tempdir)
        os.mkdir(tempdir)
        try:
            self.assertRaises(ConfigError, get_latest, tempdir)
        finally:
            shutil.rmtree(tempdir)
            
class ConfigUtilsTC(TestCase):
    """Default tests for configuration utilities functions"""
        
    def test_load_addons(self):
        load_addons('inputs.dummychecker, notexistentmodule', verb=0)
        from apycot import get_registered
        self.assert_(get_registered('checker', 'toto'))
        
    def test_section_as_dict(self):
        self.assertDictEquals(section_as_dict(CONFIG, 'MAIN'),
          {'python_test_coverage_treshold': '70',
           'threads': '3',
           'python_lint_ignore': 'thirdparty',
           'build_debian_match_checks': 'debian_lint',
           'pythonpath': '',
           'verbosity': '0',
           'python_lint_treshold': '7',})
        self.assertDictEquals(section_as_dict(CONFIG, 'MAIN', ('verbosity', )), 
                  {'python_test_coverage_treshold': '70',
                   'threads': '3',
                   'python_lint_ignore': 'thirdparty',
                   'build_debian_match_checks': 'debian_lint',
                   'pythonpath': '', 
                   'python_lint_treshold': '7',})

    def test_get_options_from_dict(self):
        _options = {'truc_bidule' : 'xx', 'bidule': 'yy', 'chouette': 'zz'}
        options = _options.copy()
        self.assertDictEquals(get_options_from_dict(options),
                      {'bidule': 'yy', 'truc_bidule': 'xx', 'chouette': 'zz'})
        self.assertEquals(options, {})
        options = _options.copy()
        self.assertDictEquals(get_options_from_dict(options,
                                                ignore=("bidule", )),
                          {'truc_bidule': 'xx', 'chouette': 'zz'})
        self.assertDictEquals(options, {'bidule':'yy'})
        options = _options.copy()
        self.assertDictEquals(get_options_from_dict(options, 'truc'),
                          {'bidule': 'xx'})
        self.assertDictEquals(options, {'bidule': 'yy', 'chouette': 'zz'})

    def test_global_config(self):
        value = global_config(CONFIG, MAIN_VARS, 0)
        self.assertEquals(len(value), 5)
        self.assert_(value is global_config(CONFIG, 0))
        self.assertEquals(value,
            {'python_test_coverage_treshold': '70',
            'python_lint_ignore': 'thirdparty',
            'build_debian_match_checks': 'debian_lint',
            'pythonpath': '',
            'python_lint_treshold': '7',})
        
    def test_parse_config(self):
        config, verb = parse_config(CONFIG_FILE)
        self.assertEquals(verb, 0)
        self.assertEquals(config.get('DATA', 'location'), '/tmp/tester_tests')
        config, verb = parse_config(CONFIG_FILE, verb=None)
        self.assertEquals(verb, 0)
        config, verb = parse_config(CONFIG_FILE, verb=2)
        self.assertEquals(verb, 2)
        config, verb = parse_config(CONFIG_FILE, location='toto')
        self.assertEquals(config.get('DATA', 'location'), 'toto')
        config, verb = parse_config(CONFIG_FILE, 'toto', 3)
        self.assertEquals(config.get('DATA', 'location'), 'toto')
        config, verb = parse_config(CONFIG_FILE, verb=3)
    def test_parse_config_simple_include(self):
        config, verb = parse_config(CONFIG_FILE_INCLUDE)
        self.assertTrue(config.has_section("devtools"))
        self.assertTrue(config.has_section("goodpkg"))
        self.assertEquals(config.get('PYTHONPACKAGE', 'checks'), 'python_pkg')
        self.assertEquals(config.get('PYTHONPACKAGE',
                                'setup_install_match_checks'), 'python_pkg')
    def test_parse_config_circurlar_include(self):
        config, verb = parse_config(CONFIG_FILE_INCLUDE_CIRCULAR)
        self.assertTrue(config.has_section("REC1"))
        self.assertTrue(config.has_section("REC2"))
        self.assertTrue(config.has_section("REC21"))
    def test_parse_config_glob_include(self):
        config, verb = parse_config(CONFIG_FILE_GLOB)
        self.assertTrue(config.has_option('GLOB', 'root'))
        self.assertTrue(config.has_option('GLOB', 'glob1'))
        self.assertTrue(config.has_option('GLOB', 'glob2'))
        self.assertTrue(config.has_option('GLOB', 'glob3'))
        self.assertTrue(config.has_option('GLOB', 'glob4'))

    def test_auto_test(self):
        """Test that patter specified in 'includes-tests' are imported"""
        config, verb = parse_config(CONFIG_FILE_AUTO)
        self.assertTrue(config.has_section('INCT-INC'))
    
    def test_auto_test_empy(self):
        """Test that empty section are imported too"""
        config, verb = parse_config(CONFIG_FILE_AUTO)
        self.assertTrue(config.has_section('INCT'))

    def test_auto_not(self):
        """Test that section imported via 'includes' are not marked as test"""
        config, verb = parse_config(CONFIG_FILE_AUTO)
        self.assertFalse(config.has_option('INC','is_test'))

    def test_auto_not_not(self):
        """Test that section imported via 'includes' in a file includes via
        'includes' are not marked as test"""
        config, verb = parse_config(CONFIG_FILE_AUTO)
        self.assertFalse(config.has_option('INC-INC','is_test'))

    def test_auto_not_inc_test_not(self):
        """Test that section imported via 'includes-tests' in a file imported
        via 'includes' are not marked as test"""
        config, verb = parse_config(CONFIG_FILE_AUTO)
        self.assertFalse(config.has_option('INC-INCT','is_test'))
    
    def test_auto_includes_tests_lvl1(self):
        """test that section imported via 'includes-tests' is marked as test"""
        config, verb = parse_config(CONFIG_FILE_AUTO)
        self.assertEquals(config.get('INCT','is_test'),'1')
    
    def test_auto_includes_tests_lvl1_force_not(self):
        """test that is_test option are not overwriten"""
        config, verb = parse_config(CONFIG_FILE_AUTO)
        self.assertEquals(config.get('INCT-GROUP','is_test'),'0')

    def test_auto_includes_tests_lvl2_incl(self):
        """Test that section imported via 'includes' in a file imported
        via 'includes-tests' are not marked as test"""
        config, verb = parse_config(CONFIG_FILE_AUTO)
        self.assertFalse(config.has_option('INCT-INC','is_test'))

    def test_auto_includes_tests_lvl2_inclt(self):
        """Test that section imported via 'includes-tests' in a file imported
        via 'includes-tests' are marked as test"""
        config, verb = parse_config(CONFIG_FILE_AUTO)
        self.assertEquals(config.get('INCT-INCT','is_test'),'1')
    
    def test_auto_top_force_not(self):
        """test that section in the top file are marked as test if specified"""
        config, verb = parse_config(CONFIG_FILE_AUTO)
        self.assertEquals(config.get('TOP-TEST','is_test'),'1')
    
    def test_auto_top_default(self):
        """test that section in the top file are not marked as test"""
        config, verb = parse_config(CONFIG_FILE_AUTO)
        self.assertFalse(config.has_option('TOP','is_test'))


class SimpleOptionsManagerMixInTC(TestCase):

    def setUp(self):
        self.o = SimpleOptionsManagerMixIn()
        
    def test_base(self):
        self.assertEquals(self.o.options, {})
        self.o.set_options({'opt1': 'val1', 'opt2': 'val2'})
        self.assertEquals(self.o.get_option('opt1'), 'val1')
        self.assertRaises(ConfigError, self.o.get_option, 'opt100')
        self.assertEquals(self.o.get_option('opt100', 2), 2)

    def test_check(self):
        self.o.required_options = (('opt1', 'missing opt1'), )
        self.o.set_options({'opt1': 'val1', 'opt2': 'val2'})
        self.o.check_options()
        self.o.required_options = (('opt1', 'missing opt1'),
                                                    ('opt3', 'missing opt3'))
        self.assertRaises(ConfigError, self.o.check_options)
        self.o.set_options({'opt1': 'val1', 'opt3': ''})
        self.assertRaises(ConfigError, self.o.check_options)
        
class EnvironmentTrackerMixinTC(TestCase):
    
    def setUp(self):
        self.tracker = EnvironmentTrackerMixin()

    def test_update_clean_env(self):
        lc_all = os.environ.get('LC_ALL')
        self.tracker.update_env('key', 'LC_ALL', 'XXXX')
        self.assertEquals(os.environ['LC_ALL'], 'XXXX')
        self.tracker.clean_env('key', 'LC_ALL')
        self.assertEquals(os.environ.get('LC_ALL'), lc_all)

        self.tracker.update_env('key', '__ENVIRONMENTTRACKERMIXINTC__', 'XXXX')
        self.assertEquals(os.environ['__ENVIRONMENTTRACKERMIXINTC__'], 'XXXX')
        self.tracker.clean_env('key', '__ENVIRONMENTTRACKERMIXINTC__')
        self.assertRaises(KeyError, access_key, os.environ,
                                                '__ENVIRONMENTTRACKERMIXINTC__')

    def test_nested(self):
        lc_all = os.environ.get('LC_ALL')
        self.tracker.update_env('key', 'LC_ALL', 'XXXX')
        self.assertEquals(os.environ['LC_ALL'], 'XXXX')
        self.tracker.update_env('key2', 'LC_ALL', 'YYYY')
        self.assertEquals(os.environ['LC_ALL'], 'YYYY')
        self.tracker.clean_env('key2', 'LC_ALL')
        self.assertEquals(os.environ['LC_ALL'], 'XXXX')
        self.tracker.clean_env('key', 'LC_ALL')
        self.assertEquals(os.environ.get('LC_ALL'), lc_all)

    def test_update_clean_env_sep(self):
        path = os.environ['PATH']
        self.tracker.update_env('key', 'PATH', '/mybin', ':')
        self.assertEquals(os.environ['PATH'], '/mybin:' + path)
        self.tracker.clean_env('key', 'PATH')
        self.assertEquals(os.environ['PATH'], path)
        
    def test_nested_sep(self):
        path = os.environ['PATH']
        self.tracker.update_env('key', 'PATH', '/mybin', ':')
        self.assertEquals(os.environ['PATH'], '/mybin:' + path)
        self.tracker.update_env('key2', 'PATH', '/myotherbin', ':')
        self.assertEquals(os.environ['PATH'], '/myotherbin:/mybin:' + path)
        self.tracker.clean_env('key2', 'PATH')
        self.assertEquals(os.environ['PATH'], '/mybin:' + path)
        self.tracker.clean_env('key', 'PATH')
        self.assertEquals(os.environ['PATH'], path)

    def test_python_path_sync(self):
        path = os.environ['PYTHONPATH']
        self.tracker.update_env('key', 'PYTHONPATH', '/mylib', ':')
        self.assertEquals(os.environ['PYTHONPATH'], '/mylib:' + path)
        self.assertEquals(sys.path[0], '/mylib')
        self.tracker.clean_env('key', 'PYTHONPATH')
        self.assertEquals(os.environ['PYTHONPATH'], path)
        self.assertNotEquals(sys.path[0], '/mylib')

    def test_update_undefined_env(self):

        var = 'XNZOUACONFVESUHFJGSLKJ'
        while os.environ.get(var) is not None:
            var = ''.join(chr(randint(ord('A'), ord('Z') +1))
                for cnt in xrange(randint(10, 20)))
        
        self.tracker.update_env('key', var, 'to be or not to be', ':')
        self.assertTextEquals(os.environ.get(var),  'to be or not to be:')
        self.tracker.clean_env('key', var)
        self.assertEquals(os.environ.get(var), None)


if __name__ == '__main__':
    unittest_main()
