# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""USAGE: %s [OPTIONS] <configuration file> [test name]...

Run tests according to configuration file

OPTIONS:
  --help / -h
    Display this help message and exit.
  --version / -V
    Display version information and exit.
    
  --verbose / -v
    Set the verbosity level. You can set this option multiple times.
    
  --rewrite / -r
    Update an existing data file.
    
  --threads / -t <threads number>
    Number of threads to use. Default is 1, i.e. tests are serialized.

  --location / -l <data file location>
    The location of the data file. It should be a directory where the
    tester_data.xml file may be found.
  --timestamp / -T
    print a timestamps with each message

  --list / -L 
    List all test detected and quit

"""

import os
import sys
import stat
import signal
import getopt
import threading
from time import sleep
from subprocess import Popen, PIPE
from tempfile import TemporaryFile
from os.path import exists, basename
from threading import Timer
import warnings

from apycot import ConfigError
from apycot.writer import PrintReader
from apycot.datamanagers import XMLDataManager
from apycot.utils import get_csv, parse_config, MAIN_SECT, base_test_config, \
    apply_units, TIME_UNITS

def get_tests(config):
    test_names = set()
    for section in config.sections():
        if config.has_option(section,'is_test') and config.getboolean(section,'is_test'):
            test_names.add(section)
    return test_names

def run_from_config_file(config_file, test_names=None,
                         rewrite=0, nb_threads=None, location=None,
                         verbosity=None, timestamp=False):
    """run tests defined in the given config file
    
    see the examples directory for a sample configuration file
    """
    if verbosity is not None:
        trace = verbosity >1
    else:
        trace = False
    config, verbosity = parse_config(config_file, location, verbosity,
        trace=trace)
        
    if nb_threads is None and config.has_option(MAIN_SECT, 'threads'):
        nb_threads = config.getint(MAIN_SECT, 'threads')
    else:
        nb_threads = nb_threads or 1

    if not test_names:
        # read the main section to get a list of test definitions
        test_names = get_tests(config)
    if config.has_option(MAIN_SECT,'tests'):
        warnings.warn(
            "Deprecated 'tests=' options ignored. Use 'is_test=1' instead.",
            category=DeprecationWarning)

    manager = XMLDataManager(config, rwmode="w", verb=verbosity)
    writer = manager.writer(rewrite=rewrite, timestamp=timestamp)
    for test_name in test_names:
        writer.msg(1,"<%s> will be tested" %  test_name)


    launcher = TestsLauncher(writer, verbosity, timestamp=timestamp)
    launcher.run_tests(config, config_file, test_names, nb_threads)
    
    writer.flush()


def active_test_thread_names():
    thread_names = [thd.getName() for thd in threading.enumerate()
                    if thd.getName() != 'MainThread']
    thread_names.sort()
    return ', '.join(thread_names)


class TestsLauncher(object):
    """class managing tests launching in multiple threads
    """
    
    def __init__(self, writer, verbosity, interval=5, timestamp = False):
        self.writer = writer
        self.options  = ['-v'] * verbosity
        self.options += ['-T'] * bool(timestamp)
        self.preader  = PrintReader(writer)
        self.interval = interval
        self._lock = threading.Lock()
        self._sem = None
        self._children = set()
        self._to_launch = 0
        self._launched = 0
        
    def run_tests(self, config, config_file, test_names, nb_threads):
        """Launch all given tests, using threads to run <nb_thread> in parallel

        this method return when all tests are finished
        """
        self._sem = threading.BoundedSemaphore(nb_threads)
        self._to_launch += len(test_names)
        if nb_threads < 2 or len(test_names) < 2:
            # don't use threads if not necessary
            for test_name in test_names:
                self._sem.acquire()
                self.run_test(config, config_file, test_name)
        else:
            for test_name in test_names:
                self._sem.acquire()
                thread = threading.Thread(name=test_name, target=self.run_test,
                                         args=(config, config_file, test_name))
                thread.start()
                
                thread_names = active_test_thread_names()
                self.writer.msg(2, 'Active thread(s): %s' % thread_names)
            # wait for all tests to finish
            sleep(self.interval * 2)
            while threading.activeCount() > 1:
                thread_names = active_test_thread_names()
                self.writer.msg(2, 'Active thread(s): %s' % thread_names)
                sleep(self.interval)
            for child in self._children:
                try:
                    # kill possible remaining children
                    os.killpg(child.pid, signal.SIGKILL)
                except OSError, ex:
                    if ex.errno != 3:
                        raise
        

    def run_test(self, config, config_file, test_name):
        """Thread target,  run the test in a separated process using the
        'runtest' command line script
        """
        try:
            self._lock.acquire()
            self._launched += 1
            launched, to_launch = self._launched , self._to_launch
            self._lock.release()
            self.writer.msg(2, 'Launching %s (%i/%i)' % (test_name, launched, to_launch))
            try:
                config = base_test_config(config, test_name, do_load_addons=False)
            except ConfigError:
                config = {}
            python = config.get('python')
            if python is not None:
                cmd = Popen(['which', 'apycot-runtest'], stdout=PIPE)
                cmd.communicate()
                runtest_path = cmd.stdout.read().strip()
                command = [python, runtest_path]
            else:
                command = ['apycot-runtest']
            
            command += self.options + ['-p', '--prefix="<%s>:"' % test_name,
                                                    config_file, test_name]
            self.writer.msg(2, ' '.join(command))
            outfile = TemporaryFile(mode='w+',bufsize=0)
            errfile = TemporaryFile(mode='w+',bufsize=0)

            cmd = Popen(command, bufsize=0, stdout=outfile, stderr=errfile)
            
            try:
                if 'max_time' in config:
                    max_time  = apply_units(config['max_time'], TIME_UNITS)
                    max_time += apply_units(config.get('max_reprive', '60'),
                                                                    TIME_UNITS)
                    max_time *= 1.25
                    timer = Timer( max_time, os.killpg, [cmd.pid,
                                                            signal.SIGKILL] )
                    timer.start()
                else:
                    timer = None
            except : # ignore any errors
                timer = None
            self._lock.acquire()
            self._children.add(cmd)
            self._lock.release()

            cmd.communicate()
            if timer is not None:
                timer.cancel()
            try:
                # kill possible remaining children
                os.killpg(cmd.pid, signal.SIGKILL)
            except OSError, ex:
                if ex.errno != 3:
                    raise
            self._lock.acquire()
            self._children.remove(cmd)
            self._lock.release()

            for file in (outfile, errfile):
                file.seek(0)

            self._lock.acquire()
            try:
                if cmd.returncode:
                    self.writer.msg(0, 'Error while running %s'%(test_name, ))
                    self.writer.msg(0, '`%s`returned with status : %s' % (
                                             ' '.join(command), cmd.returncode))
                elif os.fstat( errfile.fileno())[stat.ST_SIZE]:
                    self.writer.msg(0, 'Message from <%s>'%(test_name, ))

                for line in errfile:
                    self.writer.msg(0, line.strip(), head='...')

                self.preader.from_file(outfile)

            finally:
                self._lock.release()
        finally:
            self._sem.release()

def _handle_signal():
    os.setpgrp()
    def handle_kill(sig, _):
        print >> sys.__stderr__, "Interrupting runtests"
        signal.signal(sig, signal.SIG_DFL)
        os.killpg(os.getpid(), sig)
    signal.signal(signal.SIGINT, handle_kill)
    #os.kill(os.getpid(), signal.SIGINT)


def run(args):
    """run tests according to command line arguments
    """
    l_opt = ['help', 'version', 'verbose', 'threads=', 'location=', 'rewrite',
        'timestamp', 'list']
    opts, args = getopt.getopt(args, 'hVvt:l:rTL', l_opt)
    verb = 0
    rewrite = 0
    nb_threads = None
    location = None
    timestamp = False
    list_test = False

    for name, value in opts:
        if name in ('-h', '--help'):
            print __doc__ % basename(sys.argv[0])
            return 0
        elif name in ('-V', '--version'):
            from apycot.__pkginfo__ import version
            print 'APyCoT version', version
            print 'python', sys.version
            return 0
        elif name in ('-v', '--verbose'):
            verb += 1
        elif name in ('-t', '--threads'):
            nb_threads = int(value)
        elif name in ('-r', '--rewrite'):
            rewrite = 1
        elif name in ('-l', '--location'):
            location = value
        elif name in ('-T', '--timestamp'):
            timestamp = True
        elif name in ('-L','--list'):
            list_test = True
    verb = verb or None
    if list_test:
        trace = verb > 1
        config, verbosity = parse_config(args[0], trace=trace)
        for test in get_tests(config):
            print test
        return 0
    if not args:
        print __doc__ % basename(sys.argv[0])
        return 2
    if not exists(args[0]):
        print 'Aborted: no such file', args[0]
        return 1
    #_handle_signal()
    run_from_config_file(args[0], args[1:], rewrite, nb_threads, location,
                                                             verb, timestamp)
    return 0
        

if __name__ == '__main__':
    sys.exit(run(sys.argv[1:]))
