# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""USAGE: %s [OPTIONS] <configuration file> <test name>

Run a single test according to a configuration file. Some configuration
variables may be overridden using command line options (see below).

OPTIONS:
  --help / -h
    Display this help message and exit.
    
  --verbose / -v
    Set the verbosity level. You can set this option multiple times.

  --keep-tmp / -k
    Keep the temporary directory in /tmp (for debugging purposes).

  --checks / -c <checker names, separated by comma>
    Comma separated names of checkers to execute.
    
  --fs / -f <file or directory>
    Get the package sources from the file system instead of from the repository.

  --parsable / -p
    get an easily parsable output (used by runtests).

  --timestamp / -T
    add local timestamp before each message

  --prefix <prefix>
    an optional prefix added before each message line.
"""

import os
import sys
import getopt
from copy import deepcopy
from os.path import exists, basename

from logilab.common.compat import set
from logilab.common.textutils import BYTE_UNITS, TIME_UNITS, apply_units

# load subpackages to trigger registering
import apycot.checkers
import apycot.preprocessors

from apycot.checkers import MissingChecker
from apycot import list_registered, get_registered
from apycot.repositories import get_repository
from apycot.writer import PrettyPrintWriter, PrintWriter
from apycot.utils import get_csv, parse_config, base_test_config, \
     get_options_from_dict, ConfigError
from apycot.Test import Test

TEST_VARS = ('groups', 'checks', 'command', 'dependencies',
             'repository_type', 'repository', 'path', 'tag',
             'python', 'max_cpu_time', 'max_time', 'max_memory', 'max_reprieve',
             )


def build_test(config, test_name, writer, checks,
               keep_tmp, fspath, verb):
    """build a Test instance from the configuration
    """
    all_options = list(TEST_VARS)
    # global verbosity    
    writer.verbosity = verb
    test_def = base_test_config(config, test_name, verb)
    # finish test definition
    test_def.pop("is_test",None)
    checks = checks or test_def['checks']
    checks_instances = []
    for name in get_csv(checks):
        try:
            chks = get_registered('checker', name)()
        except ConfigError, cf_ex:
            chks = MissingChecker(name, cf_ex)
        checks_instances.append(chks)
    test_def['checks'] = checks_instances
    test_def['dependencies'] = dependencies(config,
                                            test_def.get('dependencies', ''),
                                            writer, verb, set((test_name, )))

    # read local checkers options
    for checker in test_def['checks']:
        name = checker.__name__
        checks_opts = deepcopy(get_options_from_dict(test_def, name))
        checker.set_options(checks_opts)
        all_options += checks_opts.keys()
    # read local preprocessors options
    preprocessors = []
    for pp_name in list_registered('preprocessor'):
        pp_config = deepcopy(get_options_from_dict(test_def, pp_name))
        pp_fact = get_registered('preprocessor', pp_name)
        preprocessors.append(pp_fact(writer, pp_config))
        all_options += pp_config.keys()
    # get repot option specific to that test
    repo_def = get_options_from_dict(test_def, "repository", all_options)
    
    # get environment specific to that test
    test_def['environ'] = get_options_from_dict(test_def, None, all_options)
    # test's repository
    if fspath:
        test_def['path'] = fspath
        ####
        repo_def['repository_type'] = 'fs'

        for key in ('repository', 'tag'): 
            test_def.pop(key, None)
    
    for option in ('path', 'repository', 'tag'):
        if option in test_def:
            repo_def[option] = test_def.pop(option)

    
    #repo_def.update(test_def)
    repo = get_repository(repo_def)
    
    # apply units
    for time_unit in ('max_cpu_time', 'max_reprieve', 'max_time'):
        if time_unit in test_def:
            test_def[time_unit] = apply_units(test_def[time_unit], TIME_UNITS)
    if 'max_memory' in test_def:
        test_def['max_memory'] = apply_units(test_def['max_memory'], BYTE_UNITS)
    
    # instantiate a Test
    return Test(test_name, config=config, writer=writer,
                preprocessors=preprocessors, repository=repo,
                keep_temp_dir=keep_tmp, **test_def)



# utilities ###################################################################
        
def dependencies(config, csv_string, writer, verb=0, _done=None):    
    """recursivly compute test dependencies"""
    if _done is None:
        _done = set()
    result = []
    for dep in get_csv(csv_string):
        if dep in _done:
            continue
        _done.add(dep)
        try:
            # a dependency may point to a test definition
            dep_def = base_test_config(config, dep, verb)
            
            pps = []
            for pp_name in get_csv(dep_def.get('preprocessors', '')):
                pp_fact = get_registered('preprocessor', pp_name)
                pp_config = deepcopy(get_options_from_dict(dep_def, pp_name))
                pps.append(pp_fact(writer,pp_config))

            test_depinfo = (get_repository(dep_def), dep, pps)
            
            # handle dependency's dependencies
            for depinfo in dependencies(config,
                    dep_def.get('dependencies', ''), writer,
                    verb, _done):
                result.append(depinfo)
            result.append(test_depinfo)
        except ConfigError, ex:
            try:
                result.append( (get_dependency_info(dep), None) )
            except ConfigError: # XXX shadow the second exception
                raise ex
    return result

def get_dependency_info(string):
    """return repository, path and tag from a single string

    string format is [<repository>;]<path>[@<tag>]
    """
    # is there a repository specified ?
    parts = string.split(';')
    attrs = {}
    if len(parts) > 1:
        attrs['repository'] = parts[0]
        path = ''.join(parts[1:])
    else:
        path = string
        if not exists(path):
            raise ConfigError(
                "unable to build dependency repository: path \"%s\" doesn't exist"
                % (path))
    # and/or a tag ?
    parts = path.split('@')
    if len(parts) > 1:
        attrs['tag'] = parts[-1]
        path = ''.join(parts[:-1])
    attrs['path'] = path
    return get_repository(attrs)


# access points ###############################################################

def run_from_config_file(config_file, test_name, checks, keep_tmp, fspath,
                         pretty, verb, prefix=None, timestamp=False):
    """run tests defined in the given config file
    
    see the examples directory for a sample configuration file
    """
    config, verb = parse_config(config_file, verb=verb)
    if not config.has_section(test_name):
        raise Exception('No %r test defined in %s' % (test_name, config_file))
    if pretty:
        writer = PrettyPrintWriter(prefix=prefix, timestamp=timestamp)
    else:
        writer = PrintWriter(prefix=prefix, timestamp=timestamp)
    test = build_test(config, test_name, writer, checks, keep_tmp, fspath,
                      verb)
    os.environ['APYCOT_ROOT'] = test.tmpdir
    test.execute()


def run(args):
    """run a single test according to command line arguments"""
    sys.path.insert(0, '')
    l_opt = ['help', 'verbose', 'keep-tmp', 'checks=', 'fspath=', 'parsable',
                                        'version', 'prefix=', 'timestamp']
    opts, args = getopt.getopt(args, 'hvkc:f:pVT', l_opt)
    verb = 0
    keep_tmp = 0
    pretty = 1
    checks = None
    fspath = None
    prefix = None
    timestamp = False
    for name, value in opts:
        if name in ('-h', '--help'):
            print __doc__ % basename(sys.argv[0])
            return 0
        elif name in ('-V', '--version'):
            from apycot.__pkginfo__ import version
            print 'APyCoT version', version
            print 'python', sys.version
            return 0
        elif name in ('-v', '--verbose'):
            verb += 1
        elif name in ('-k', '--keep-tmp'):
            keep_tmp = 1
        elif name in ('-c', '--checks'):
            checks = value
        elif name in ('-p', '--parsable'):
            pretty = 0
        elif name in ('-f', '--fspath'):
            fspath = value
        elif name in ('-T', '--timestamp'):
            timestamp = True
        elif name in ('--prefix', ):
            prefix = value
    if not len(args) == 2:
        print __doc__
        return 1
    verb = verb or None
    run_from_config_file(args[0], args[1], checks, keep_tmp, fspath,
                         pretty, verb, prefix, timestamp)
        
if __name__ == '__main__':
    run(sys.argv[1:])
