# Copyright (c) 2004-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""Some standard sources repositories, + factory function"""

import os
from os.path import normpath, abspath, basename, join, isabs
from time import localtime
from warnings import warn



from logilab.devtools.vcslib.svn import SVNAgent
from logilab.devtools.vcslib.cvs import CVSAgent
from logilab.devtools.vcslib.hg import HGAgent

from apycot import register, get_registered, ISourceRepository, \
     ConfigError, NotSupported

SUPPORTED_REPO_TYPES = ('cvs', 'svn', 'hg', 'null', 'fs', 'mock')


def get_repository(attrs):
    """factory method: return a ISourceRepository implementation according to
    <attrs> (a dictionary)
    """
    repo_type = attrs.pop('type', None)
    #
    if repo_type is None:
        repo_type = attrs.pop('repository_type', None)
    if repo_type is None:
        if ':' not in attrs.get('repository', ''):
            repo_type, repo = 'null', None
            #warn("no repository found using null instead", RuntimeWarning)
        else:
            repo_type, repo = attrs['repository'].split(':', 1) #'cvs'
            attrs['repository'] = repo
    assert repo_type in SUPPORTED_REPO_TYPES, repo_type
    return get_registered('repository', repo_type)(attrs)


class VersionedRepository:
    """base class for versionned repository"""
    
    __name__ = None
    agent_class = None

    def __init__(self, attrs):
        try:
            self.repository = attrs.pop('repository')
        except KeyError, ex:
            raise ConfigError('Missing %s option: %s' % (ex, attrs))
        
        self.path = normpath(attrs.pop('path', ''))
        self.tag = attrs.pop('tag', 'HEAD')
        self.view_url = attrs.pop('view_url', '')
        self.view_root = attrs.pop('view_root', '')
        self.agent = self.agent_class()

    def env_path(self):
        raise NotImplementedError()
    
    def view_url_for(self, filepath):
        """return an url that may be used to view the given file or None
        if it is not supported
        """
        return None

    
    def checkout_command(self, quiet=True):
        """return a command that may be os.systemed to check out a given package
        """
        return self.agent.checkout(self.repository, self.path, self.tag,
            quiet=quiet)


    def log_info(self, from_date, to_date):
        """get checkins information between <from_date> and <to_date>
        Both date should be local time (ie 9-sequence) or epoch time (ie float)
        
        return an iterator on `logilab.devtools.vcslib.CheckInInfo` instances
        """
        from_date, to_date = self.normalize_date(from_date, to_date)
        return self.agent.log_info(self.path, from_date, to_date,
                                   self.repository, self.tag)

    def representative_attributes(self):
        """return a dictionary representing this repository state, so it can be
        recreated latter
        """
        result = self.__dict__.copy()
        result['repository_type'] = self.__name__
        del result['agent']
        return result
    
    def normalize_date(self, from_date, to_date):
        """get dates as float or local time and return the normalized dates as
        local time tuple to fetch log information from <from_date> to <to_date>
        included
        """
        if isinstance(from_date, float):
            from_date = localtime(from_date)
        if isinstance(to_date, float):
            to_date = localtime(to_date)
        return (from_date, to_date)
    
    def __eq__(self, other):
        return (isinstance(other, self.__class__) and
                self.repository == other.repository and
                self.path == other.path and
                self.tag == other.tag)

    def __ne__(self, other):
        return not self == other
    
    def __repr__(self):
        """get a string synthetizing the location"""
        return ('%(path)s@%(tag)s in %%s %(repository)s' %
                (self.__dict__)) % self.__name__
    

class CVSRepository(VersionedRepository):
    """extract sources/information for a project from a CVS repository"""
    __implements__ = ISourceRepository
    __name__ = 'cvs'
    agent_class = CVSAgent

    def env_path(self):
        """return the relative path where the project will be located
        in the test environment
        """
        return self.path

    def view_url_for(self, filepath):
        """return an url that may be used to view the given file or None
        if it is not supported
        """
        if self.view_url and filepath.startswith(self.path):
            return '%s/%s?rev=%s&root=%s&content-type=text/\
vnd.viewcvs-markup' % (self.view_url, filepath, self.tag, self.view_root)
        return None
    

register('repository', CVSRepository)


class SVNRepository(VersionedRepository):
    """extract sources/information for a project from a SVN repository"""
    __implements__ = ISourceRepository
    __name__ = 'svn'
    agent_class = SVNAgent

    def env_path(self):
        """return the relative path where the project will be located
        in the test environment
        """
        return basename(self.path)

    def log_info(self, from_date, to_date):
        """get checkins information between <from_date> and <to_date>
        Both date should be local time (ie 9-sequence) or epoch time (ie float)
        
        return an iterator on `logilab.devtools.vcslib.CheckInInfo` instances
        """
        svn_dir = (self.tag == 'HEAD' and 'trunk' or 'tags/%s' % self.tag)
        path = '%s/%s' % (svn_dir, self.path)
        from_date, to_date = self.normalize_date(from_date, to_date)
        return self.agent.log_info(path, from_date, to_date,
                                   self.repository, self.tag)
    
    def view_url_for(self, filepath):
        """return an url that may be used to view the given file or None
        if it is not supported
        """
        startswith=filepath.startswith
        if startswith(self.path) or startswith(self.env_path()):
            svn_dir = (self.tag == 'HEAD' and 'trunk' or 'tags/%s' % self.tag)
            return '%s/%s/%s/%s' % (self.repository, svn_dir, self.path,
                                    '/'.join(filepath.split('/')[1:]))
        return None
    
register('repository', SVNRepository)


class HGRepository(VersionedRepository):
    """extract sources/information for a project from a Mercurial repository"""
    __implements__ = ISourceRepository
    __name__ = 'hg'
    agent_class = HGAgent
    
    def env_path(self):
        """return the relative path where the project will be located
        in the test environment
        """
        return self.path

    def log_info(self, from_date, to_date):
        """get checkins information between <from_date> and <to_date>
        Both date should be local time (ie 9-sequence) or epoch time (ie float)
        
        return an iterator on `logilab.devtools.vcslib.CheckInInfo` instances
        """
        from_date, to_date = self.normalize_date(from_date, to_date)
        return self.agent.log_info(self.repository, from_date, to_date,
                                   tag=self.tag)
    def view_url_for(self, filepath, dir=False):
        """return an url that may be used to view the given file or None
        if it is not supported"""
        if self.view_url is None:
            return None
        else:
            if filepath.startswith(self.path):
                filepath = filepath.replace(self.path+'/', '', 1)

            

            #
            return "%s?f=-1;file=%s"%(self.view_url, filepath)
            #
            changeset = self.agent.current_short_changeset(self.repository)
            return "%s/%s/%s"%(self.view_url, changeset, filepath)

register('repository', HGRepository)

    
class FSRepository:
    """extract sources for a project from the files system"""
    __implements__ = ISourceRepository
    __name__ = 'fs'

    def __init__(self, attrs):
        self.repository = attrs.pop('repository', os.getcwd())
        try:
            self._path = attrs.pop('path')
            if isabs(self._path):
                self.path = normpath(self._path)
            else:
                self.path = normpath(join(self.repository, self._path))
            self.abspath = abspath(self.path)
        except KeyError, ex:
            raise ConfigError('Missing %s option' % ex)

    def checkout_command(self, quiet=True):
        """return a command that may be os.systemed to check out a given
        package
        """
        return 'cp -R %s .' % (self.abspath)

    def env_path(self):
        """return the relative path where the project will be located
        in the test environment
        """
        return basename(self.path)

    def representative_attributes(self):
        """return a dictionary representing this repository state, so it can be
        recreated latter
        """
        return {'path': self._path, 'repository_type': 'fs'}

    def log_info(self, from_date, to_date):
        """get list of log messages
        
        a log information is a tuple
        (file, revision_info_as_string, added_lines, removed_lines)
        """
        raise NotSupported()
    
    def view_url_for(self, filepath):
        """return an url that may be used to view the given file or None
        if it is not supported
        """
        return None
    
    def __repr__(self):
        """get a string synthetizing the location"""
        return 'fs directory at %(path)s' % self.__dict__

    def __eq__(self, other):
        if isinstance(other, FSRepository):
            return self.abspath == other.abspath
        return False

    def __ne__(self, other):
        return not self == other

register('repository', FSRepository)


class NullRepository(FSRepository):
    """dependancies only"""
    __implements__ = ISourceRepository
    __name__ = 'null'

    def __init__(self, attrs):
        try:
            self.path = normpath(attrs.pop('path'))
            self.abspath = abspath(self.path)
        except KeyError, ex:
            raise ConfigError('Missing %s option' % ex)

    def checkout_command(self, quiet=True):
        """return a command that may be os.systemed to check out a given
        package
        """
        return None #can't be os.systemed

    def env_path(self):
        """return the relative path where the project will be located
        in the test environment
        """
        return self.path

    def representative_attributes(self):
        """return a dictionary representing this repository state, so it can be
        recreated latter
        """
        return {'path': self.path, 'repository_type': 'null'}
    
register('repository', NullRepository)

