# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""installation preprocessor using distutils setup.py
"""


import os
from os.path import join, exists
from imp import find_module, load_module

from apycot.preprocessors import BasePreProcessor, clean_modules
from apycot import IPreProcessor,  register
from apycot.Test import SetupException
from apycot.utils import EnvironmentTrackerMixin
    
class DistutilsProcessor(EnvironmentTrackerMixin, BasePreProcessor):
    """python distutils pre-processor
    """
    __implements__ = IPreProcessor
    __name__ = 'setup_install'

    def __init__(self, writer, options):
        EnvironmentTrackerMixin.__init__(self)
        BasePreProcessor.__init__(self, writer, options)
        self._installed = {}
           
    # PreProcessor interface ##################################################

    def check_setup(self, test, checker):
        """setup the test environment
        """
        path = test.repo.env_path()
        dist = self._install(path)
        lib_dir = dist.get_command_obj('install').install_lib
        self.update_env(path, 'PYTHONPATH', lib_dir, os.pathsep)
        if dist.packages:
            dist.packages.sort()
            modname = dist.packages[0]
            test.install_base = join(lib_dir, join(*modname.split('.'))) 
            
    def check_clean(self, test, checker):
        """clean the test environment
        """
        path = test.repo.env_path()
        self.writer.msg(2, 'Run preprocessor %s cleaning on %s' % (
            self.__name__, path))
        self.clean_env(path, 'PYTHONPATH')
            
    def dependancy_setup(self, test, path):
        """setup the test environment

        may raise a SetupException
        """
        dist = self._install(path)
        lib_dir = dist.get_command_obj('install').install_lib
        bin_dir = dist.get_command_obj('install').install_scripts
        self.update_env(path, 'PYTHONPATH', lib_dir, os.pathsep)
        self.update_env(path, 'PATH', bin_dir, os.pathsep)

    def dependancy_clean(self, test, path):
        """clean the test environment
        """
        self.writer.msg(2, 'Run preprocessor %s cleaning on dependancy %s' % (
            self.__name__, path))
        self.clean_env(path, 'PYTHONPATH')
        self.clean_env(path, 'PATH')

    # private #################################################################
    
    def _install(self, path):
        """run the distutils setup.py install method on a path if
        the path is not yet installed
        """
        self.writer.msg(1, 'Run preprocessor %s on %s' % (self.__name__, path))
        # cache to avoid multiple installation of the same module
        if self._installed.has_key(path):
            return self._installed[path]
        if not exists(join(path, 'setup.py')):
            raise SetupException('No file %s' % join(path, 'setup.py'))
        clean_modules('setup', '__pkginfo__')
        cwd = os.getcwd()
        try:
            return self.execute(path, self._do_install,
                                path, join(cwd, 'local'))
        finally:
            clean_modules('setup', '__pkginfo__')
            
    def _do_install(self, path, dest):
        """execute the setup.py (which should have a "install" method)

        (callback for the "execute" method)
        """
        try:
            mp_file, mp_filename, mp_desc = find_module('setup', [''])
            setup = load_module('setup', mp_file, mp_filename, mp_desc)
        except Exception, ex:
            import traceback
            traceback.print_exc()
            raise SetupException('Unable to import %s (%s)' % (
                join(path, 'setup.py'), ex))
        try:
            install = setup.install
        except AttributeError:
            raise SetupException('setup.py has no "install" function')
        args = ('install', '--home', dest)
        try:
            dist = install(script_name="setup.py", script_args=args)
        except Exception, ex:
            msg = 'Error while running setup file %s: %s'
            raise SetupException(msg % (join(path, 'setup.py'), ex))
        if dist is None:
            msg = '"install" function in setup.py doesn\'t return the dist \
object'
            raise SetupException(msg)
        self._installed[path] = dist
        return dist

register('preprocessor', DistutilsProcessor)

class SetPythonEnvProcessor(EnvironmentTrackerMixin, BasePreProcessor):
    """a preprocessor to setup a test correctly when the pp_setup
    preprocessor can't be directly executed while the test requires a similar
    environment for a python module.
    The name of this python module is given by a required "modname" variable.
    """
    __implements__ = IPreProcessor
    __name__ = 'set_python_env'

    def __init__(self, writer, options):
        EnvironmentTrackerMixin.__init__(self)
        BasePreProcessor.__init__(self, writer, options)
           
    # PreProcessor interface ##################################################

    def check_setup(self, test, checker):
        """setup the test environment"""
        lib_dir = join(os.getcwd(), 'local/lib/python')
        modname = self.get_option('modname')
        test.install_base = join(lib_dir, join(*modname.split('.'))) 
        self.update_env(test.repo.env_path(), 'PYTHONPATH', lib_dir, os.pathsep)
            
    def check_clean(self, test, checker):
        """clean the test environment"""
        self.clean_env(test.repo.env_path(), 'PYTHONPATH')
            
    def dependancy_setup(self, test, path):
        """setup the test environment"""
        raise SetupException('this preprocessor doesn\'t handle dependancies')

    def dependancy_clean(self, test, path):
        """clean the test environment
        """
        raise SetupException('this preprocessor doesn\'t handle dependancies')

register('preprocessor', SetPythonEnvProcessor)
