# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""preprocessors register
"""

from os.path import walk, splitext
from cStringIO import StringIO
import os
import sys
import re

from apycot.utils import get_csv, SimpleOptionsManagerMixIn, _MARKER
from apycot import IPreProcessor

    
class BasePreProcessor(SimpleOptionsManagerMixIn):
    """an abstract class providing some common utilities for preprocessors
    """
    def __init__(self, writer, options):
        SimpleOptionsManagerMixIn.__init__(self)
        self.writer = writer
        self.set_options(options)
        match = self.get_option('match_checks', None) 
        self._matchs = match and get_csv(match) or []
        match = self.get_option('match_paths', None)
        self._matchs += match and get_csv(match) or []
        match = self.get_option('match_tests', None)
        self._matchs += match and get_csv(match) or []
        self._matchs_rgx = []
        for match in self._matchs:
            if '.' in match:
                if not match.endswith('$'):
                    match += '$'
                self._matchs_rgx.append(re.compile(match))
            
    def get_option(self, option, default=_MARKER, msg=None):
        """overridden to give a better message
        """
        msg = msg or 'Missing %r option for preprocessor %s' % (option,
                                                                self.__name__)
        return SimpleOptionsManagerMixIn.get_option(self, option, default, msg)
    
    # utilities ###############################################################
    
    def match(self, path_or_check):
        """return true if the preprocessor should be applied to the given path
        """
        if path_or_check in self._matchs:
            return True
        for rgx in self._matchs_rgx:
            if rgx.match(path_or_check):
                return True
        return False

    def execute(self, in_directory, callback, *args, **kwargs):
        """execute callback in an environment where outputs have been
        redirected

        if <in_directory> is not None, chdir before the call and restore the
        current directory before return
        """
        if in_directory is not None:
            _cwd = os.getcwd()
            os.chdir(in_directory)
        sys.stdout = sys.stderr = StringIO()
        try:
            return callback(*args, **kwargs)
        finally:
            sys.stdout = sys.__stdout__
            sys.stderr = sys.__stderr__
            if in_directory is not None:
                os.chdir(_cwd)
                
    # PreProcessor interface ##################################################
    
    def check_setup(self, test, checker):
        """setup the test environment
        """
        raise NotImplementedError()
    
    def check_clean(self, test, checker):
        """clean the test environment
        """
        raise NotImplementedError()
            
    def dependancy_setup(self, test, path):
        """setup the test environment for a dependancy

        may raise a SetupException
        """
        raise NotImplementedError()

    def dependancy_clean(self, test, path):
        """clean the test environment for a dependancy
        """
        raise NotImplementedError()


def clean_modules(*modnames):
    """remove some module names from sys.modules to avoid errors when
    importing them from another package
    """
    for modname in modnames:
        try:
            del sys.modules[modname]
        except KeyError:
            continue

# import submodules
import apycot.preprocessors.pp_setup
import apycot.preprocessors.pp_make
import apycot.preprocessors.pp_zope_install
try:
    import apycot.preprocessors.pp_buildeb
except ImportError, ex:
    print >> sys.stderr, 'Can\'t import pp_debian (%s). \
    Debian preprocessor won\'t be available' % ex
