# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""Interfaces used in the tester package"""

from logilab.common.interface import Interface

class ISimpleOptionsManager(Interface):
    """options manager interface
    """
    
    __name__ = None
    options = {}
    
    def set_options(self, options):
        """set a dictionary of options"""

    def get_option(self, option):
        """return option's value or None"""


class IDataManager(Interface):
    """a data manager handle tests data loading / writing"""
    
    def __init__(self, config, decorator=None, date=None):
        """init a data manager :
        * config give access to the ConfigParser object
        * decorator is an optional IDecorator object
        * date is the effective date as a 3-uple (Y,m,d)
        """
        
    def tree_for_date(self, date):
        """get the data tree for a given date"""
        
    def main_tree(self):
        """return the main data tree (ie associated with self.date)
        """
        
    def writer(self, date=None, rewrite=0):
        """return an object implementing IWriter, ready to write tests
        execution result for <date> (optionaly in rewrite mode)

        if date is None, date is today
        """
        
    def next_date(self, date):
        """return the date where for the next data from date can be found
        if date is today (and so no next data exists), return tomorrow
        
        date (argument and returned) should be a 3-sequence (year, month, day)
        
        may raise NotSupported
        """

    def previous_date(self, date, maxtry=30):
        """return the date where for the previous data from date can be found
        return None if no previous data exists
        
        date (argument and returned) should be a 3-sequence (year, month, day)
        
        may raise NotSupported
        """

        
# testing classes #############################################################

class ISourceRepository(Interface):
    """a source repository is used to fetch the source of projects or
    dependancies
    """
    def __init__(self, kwargs):
        """constructor should pick it's configuration parameters and delete
        them from the kwargs dictionary
        """
        
    def checkout_command(self):
        """return a command that may be os.systemed to check out a given package
        """
        
    def env_path(self):
        """return the relative path where the project will be located
        in the test environment
        """
        
    def view_url_for(self, filepath):
        """return an url that may be used to view the given file or None
        if it is not supported
        """
        
    def log_info(self, from_date, to_date):
        """get checkins information between <from_date> and <to_date>
        Both date should be local time (ie 9-sequence) or epoch time (ie float)
        
        return an iterator on `logilab.devtools.vcslib.CheckInInfo` instances
        """
        
    def representative_attributes(self):
        """return a dictionary representing this repository state, so it can be
        recreated latter
        """

    
class IWriter(ISimpleOptionsManager):
    """a writer collect checks results"""
    
    def notify(self, event, **kwargs):
        """notify an event, with optional values

        possible events are :
        
        * 'open_test', take the module name as argument
        
        * 'open_check', take the module name as argument
        
        * 'close_check', close the latest opened check, take a 'status'
          argument which should be 'succeed', 'failed' or 'error' according to
          the check status
          
        * 'close_test', close the latest opened test (no additional argument)
        
        """

    def log(self, severity, path, line, msg):
        """log a message of a given severity
        
        line may be None if unknown
        """

    def raw(self, name, value):
        """give some raw data"""

    def msg(self, verbose_level, msg):
        """print a message about what are we doing currently"""


class IPreProcessor(ISimpleOptionsManager):
    """a preprocessor apply some build/install step required to have a
    working test environment
    """

    def match(self, path):
        """return true if the preprocessor should be applied to the given path
        """
        
    def check_setup(self, test, checker):
        """setup the test environment"""
            
    def check_clean(self, test, checker):
        """setup the test environment"""
            
    def dependancy_setup(self, test, path):
        """setup the test environment for a dependancy

        may raise a SetupException
        """

    def dependancy_clean(self, test, path):
        """clean the test environment for a dependancy"""


class IChecker(ISimpleOptionsManager):
    """a checker make some basic tests on a module and send result to a writer.
    """
    
    def run(self, test, writer):
        """run the checker against <test> (a Test instance)

        this method should return true if the test succeeded, else false.
        """


# reporting classes ###########################################################

class IDecorator(ISimpleOptionsManager):
    """a decorator provide some additional data to the raw tests data tree"""
    
    def __init__(self, next=None):
        """construct a decorator instance by optionaly giving it the chained
        decorator
        """
        
    def decorate(self, tree):
        """decorate the dom node <tree> corresponding to some tests data
        """

class ITransport(ISimpleOptionsManager):
    """a transport is responsible to get some reports on the right way"""
    
    def base_url(self, date=None):
        """return the base transport url, optionaly for a given date
        
        may raise NotSupported
        
        Somes transport may support base_url but not with the date argument
        """        
    def open(self, date):
        """open the transport transaction for a given date"""
    def close(self):
        """close the transport transaction"""
    def part_stream(self, pid):
        """get a stream object to receive the next report
        must be called in an opened transport transaction
        """
    def add_part(self, pid, stream):
        """transport a report part, caracterized by its identifier and data in a
        file-like objet (it may be a stream created by part_stream)
        """
        
class IFormatter(ISimpleOptionsManager):
    """a formatter encode ureport's layout objets in a stream, using a
    specific format (text, html...)
    """
    
    def format(self, layout, stream):
        """format and write the given layout into the stream object
        """
    def ulink_support(self):
        """return true if the format support links"""
    def relative_url(self, node):
        """return the relative url for the node in the report
        may raise NotSupported
        """

class IReporter(ISimpleOptionsManager):
    """a reporter extract some data from the tests data written at the
    tests execution.
    """

    def make_reports(self, manager):
        """create reports from a data manager (i.e. load data file on demand)
        
        format reports using a IFormater object and post them using a
        ITransport object
        """

__all__ = ('IDataManager',
           'ISourceRepository', 'IWriter', 'IPreProcessor', 'IChecker',
           'IDecorator', 'ITransport', 'IFormatter', 'IReporter')
