# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""an activity detection / check in messages decorator"""

from time import strftime, strptime, mktime, localtime

from apycot import register, IDecorator, NotSupported
from apycot.decorators import AbstractDecorator
from apycot.repositories import get_repository
from apycot.datamanagers import TIME_FORMAT

STATUS = ('active','sleep','n/a')

class ActivityDecorator(AbstractDecorator):
    """extract log messages and add them to the data tree as a check
    which never fails.
    """
    __name__ = 'activity'
    __implements__ = IDecorator

    def __init__(self, next=None):
        AbstractDecorator.__init__(self, next)
        self.start_date = None
        
    def do_decorate(self, tree, manager):
        """make the specific decoration"""
        refdate = strptime(tree.get('starttime'), TIME_FORMAT)
        time = int(self.get_option('time', 1))
        from_date = localtime(mktime(refdate) - (time * manager.date_offset))
        for test_node in tree.iterchildren(tag='test'):
            date_string = test_node.get('starttime')
            if date_string is None:
                to_date = refdate
            else:
                to_date = strptime(test_node.get('starttime'), TIME_FORMAT)

            node = self.append_check_node(test_node)
            repo = get_repository(dict(test_node.attrib))
            status = self.extract_log(node, repo, from_date, to_date)
            assert status in STATUS
            node.set('status', status)
            
    def extract_log(self, node, repo, from_date, to_date):
        """extract log info for the given test"""
        added, removed = 0, 0
        try:
            logs = repo.log_info(from_date, to_date)
        except NotSupported:
            return 'n/a'
        status = 'sleep'
        for chkininfo in logs:
            status = 'active'
            filename = chkininfo.files and chkininfo.files[0] or ''
            log_node = node.makeelement('log', {'severity': 'INFO',
                                           'path': filename})
            node.append(log_node)
            log_node.text = ('%s: %s' % (chkininfo.author,
                                       chkininfo.message_summary()))
            added += chkininfo.added
            removed += chkininfo.removed
        raw_node = node.makeelement('raw', {'class': 'date'})
        node.append(raw_node)
        raw_node.text = ('from %s to %s' % (strftime("%Y/%m/%d", from_date),
                                          strftime("%Y/%m/%d", to_date)))
        raw_node = node.makeelement('raw', {'class': 'added_lines'})
        node.append(raw_node)
        raw_node.text = str(added)
        raw_node = node.makeelement('raw', {'class': 'deleted_lines'})
        node.append(raw_node)
        raw_node.text = str(removed)
        return status

register('decorator', ActivityDecorator)

