# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""Data managers

currently only a daily xml data manager is provided

internal data representation nativly support XML serialisation, so there is
only high level objects here
"""

import sys
import errno
import os
from os import sep # not in os.path until 2.3
from os.path import join, exists
from time import time, mktime, localtime, strftime
from bisect import bisect_left, bisect_right
from lxml.etree import _ElementTree, Element, ElementTree, parse
from warnings import warn
from ConfigParser import NoOptionError, NoSectionError


from apycot.__pkginfo__ import version as tester_version
from apycot.interfaces import IDataManager, IWriter
from apycot.writer import BaseMsgWriter
from apycot.utils import get_latest, ConfigError, SimpleOptionsManagerMixIn, \
         DATA_SECT, REVERSE_SEVERITIES, DAY, HOUR, \
        init_directory, existing_dates_index



TIME_FORMAT = "%Y-%m-%d %X"



class XMLDataManager:
    """handle data files loading"""
    
    __implements__ = IDataManager
    
    data_file = 'tester_data.xml'
    encoding = 'UTF-8'
    # mode name : (date lenght, date offset)
    AVAILABLE_MODE = {'daily':(3, DAY), 'hourly':(4, HOUR)}

    def __init__(self, config, decorator=None, date=None, rwmode='r', verb=0):
        """
        `mode`:
           * 'daily', write report to
             <file_or_directory>/<year>/<month>/<day>/tester_data.xml        
           * 'hourly', write report to
             <file_or_directory>/<year>/<month>/<day>/<hour>/tester_data.xml
             
        `rwmode`:
          'r' or 'w' if the manager should open data in read or write mode
        """
        assert rwmode in 'rw', rwmode
        self.__dates_index = None
        try:
            self.location = config.get(DATA_SECT, 'location')
        except (NoOptionError, NoSectionError):
            raise ConfigError('Missing location option in [%s]' % DATA_SECT)
        try:
            self.mode = config.get(DATA_SECT, 'mode')
        except (NoOptionError, NoSectionError):
            self.mode = "daily"
        if not self.mode in self.AVAILABLE_MODE.keys():
            raise ConfigError('bad mode %r not in %s'
                    % (self.mode, self.AVAILABLE_MODE.keys))
        for option in ('data_file', 'encoding'):
            if config.has_option(DATA_SECT, option):
                setattr(self, option, config.get(DATA_SECT, option))
        self.decorator = decorator
        self.verb = verb

        self._datesize, self.date_offset = self.AVAILABLE_MODE[self.mode]
        if date is not None:
            assert isinstance(date[0], int)
            assert len(date) == self._datesize
            self.date = date
        elif rwmode == 'r':
            directory = self.latest_directory()
            date_strings = directory.split(sep)[-self._datesize:]
            self.date = [int(val) for val in date_strings]
        else: # rwmode == 'w'
            self.date = localtime()[:self._datesize]
        self._loaded = {}

    def latest_directory(self):
        """return the most recent data directory"""
        return get_latest(self.location, alldates=self._dates_index())
    
    def main_tree(self):
        """return the main data tree (ie associated with self.date)"""
        return self.data_from_file(self.file_from_date(self.date))
        
    def tree_for_date(self, date):
        """get the data tree for a given date"""
        return self.data_from_file(self.file_from_date(date))

    def data_from_file(self, data_file):
        """get the data tree from path to the xml file"""
        if not exists(data_file):
            raise ConfigError('No such data file %r' % data_file)
        if self._loaded.has_key(data_file):
            return self._loaded[data_file]
        if self.verb:
            print 'parsing', data_file
        document = parse(data_file)
        if self.decorator is not None:
            if self.verb:
                print 'decorating'
            self.decorator.decorate(document.getroot(), self)
        self._loaded[data_file] = document
        return document


    def file_from_date(self, date):
        """return the expected data file path for a given date"""
        if isinstance(date[0], int):
            date = ['%02d' % i for i in date]
        return join(join(self.location, *date), self.data_file)

    def writer(self, date=None, rewrite=0, **kargs):
        """return an object implementing IWriter, ready to write tests
        execution result for <date> (optionaly in rewrite mode)

        if date is None, date is today
        """
        # assert self.mode == 'w'
        date = date or self.date
        datafile = self.file_from_date(date)
        init_directory(self.location, date)
        tree = None
        if rewrite:
            #self.msg(1, 'Reading data file')
            tree = self.data_from_file(datafile)
            data = tree.getroot() 
        else:
            data = Element('testsdata',
                        {'tester_version' : tester_version,
                         'python' : sys.executable,
                         'python_version' : sys.version,
                         'platform' : sys.platform, })
                         #'name' : config_name})
        if tree is None:
            tree = ElementTree(data)
        return XMLWriter(tree, data_file=datafile, rewrite=rewrite,
            encoding=self.encoding, verbosity=self.verb, **kargs)
    
    def next_date(self, date):
        """return the date where for the next data from date can be found
        if date is today (and so no next data exists), return tomorrow
        
        date (argument and returned) should be a 3-sequence (year, month, day)
        in daily mode or a 4-sequence (year, month, day, hour) in hourly mode
        """
        assert len(date) >= self._datesize
        if isinstance(date[0], str):
            date = [int(i) for i in date]
        date = tuple(date)
        today = localtime()
        if date < today:
            date = date[:self._datesize]
            alldates = self._dates_index()
            index = bisect_right(alldates, date)
            if index < len(alldates):
                return alldates[index]
            return localtime(mktime(today) + self.date_offset)[:self._datesize]
        # date is today or in the future, return date + offset blindly
        if len(date) != 9:
            date = list(date) + [0]*(9 -len(date))
        return localtime(mktime(date) + self.date_offset)[:self._datesize]

    def previous_date(self, date, maxtry="unused"):
        """return the date where for the previous data from date can be found
        return None if no previous data exists
        
        date (argument and returned) should be a 3-sequence (year, month, day)
        in daily mode or a 4-sequence (year, month, day, hour) in hourly mode
        """
        assert len(date) >= self._datesize
        if isinstance(date[0], str):
            date = [int(i) for i in date]
        date = date[:self._datesize]
        alldates = self._dates_index()
        index = bisect_left(alldates, tuple(date))
        if index != 0:
            return alldates[index - 1]
        return None

    def _dates_index(self):
        """return all data directory indexed by date"""
        if self.__dates_index is None:
            self.__dates_index = existing_dates_index(self.location,
                                        self._datesize, self.data_file)
        return self.__dates_index
        
class XMLWriter(BaseMsgWriter):
    """write checks data in a simple XML format.
    """
    
    __implements__ = IWriter

    def __init__(self, root, data_file=None,
                       encoding='UTF-8', rewrite=None, verbosity=0, **kargs):
        BaseMsgWriter.__init__(self, verbosity, **kargs)
        self.verbosity = verbosity
        self.encoding = encoding
        self.rewrite = rewrite
        self._tree = None
        assert isinstance(root, _ElementTree)
        self._tree = root
        self._current = self._tree.getroot()
        self._current.set('starttime', strftime(TIME_FORMAT, localtime()))

        self._check_start = None
        self._test_start = None
        self.file = data_file
        self.msg(1, 'Data file : %s' % data_file)
        
    def flush(self, setendtime=1):
        """flush data into the stream and close it"""
        self._ensure_state()
        self.line(2, '-')
        self.msg(2, 'Flushing data in %s' %self.file)
        if setendtime:
            self._current.set('endtime', strftime(TIME_FORMAT, localtime()))
        try:
            stream = open(self.file, 'w')
        except IOError, ex:
            if ex.errno == errno.EACCES:
                # in rewrite mode we may have pb to rewrite the file
                os.remove(self.file)
                stream = open(self.file, 'w')
            else:
                raise
        self._tree.write(stream, encoding=self.encoding, xml_declaration=True)#, pretty_print=True)
        stream.close()
    

    def push(self, tag, attributes=None):
        """push a new structure on the top of the stack"""
        node = self._current.makeelement(tag, attributes)
        self._current.append(node)
        self._current = node
        
    def pop(self):
        """pop the structure on the top of the stack"""
        data = self._current
        if data.getparent() is not None:
            self._current = data.getparent()
        return data
        
    # IWriter interface #######################################################

    def notify(self, event, **kwargs):
        """ notify an event, with optional values

        possible events are :
        
        * 'open_test', take the module name as argument
        
        * 'open_check', take the checker name as argument
        
        * 'close_check', close the latest opened check, take a 'status'
          argument which should be 'succed', 'failed' or 'error' according to
          the check status
          
        * 'close_test', close the latest opened test (no additional argument)
        """
        assert event in ('open_test', 'open_check',
                         'close_check', 'close_test'), repr(event)
        func = getattr(self, event)
        func(**kwargs)

        
    def log(self, severity, path, line, msg):
        """log a message of a given severity
        
        line may be None if unknown
        """
        kwargs = {'severity' : REVERSE_SEVERITIES[severity],
                  'path' : path,
                  'line' : line}
        pop = set()
        for key, value in kwargs.iteritems():
            if value is None:
                pop.add(key)
            elif not isinstance(value, basestring):
                kwargs[key] = unicode(value)
        for key in pop:
            del kwargs[key]

        self.push('log', kwargs)
        if not isinstance(msg, unicode):
            msg = unicode(msg, self.encoding)
        try:
            try:
                self._current.text = msg
            except (ValueError, AssertionError), ex:
                warn("%s %s trying repr instead"%(str(ex), repr(msg)))
                try :
                    self._current.text = repr(msg)
                except (ValueError, AssertionError), ex:
                    warn("%s %s is really xml incompatible"%(str(ex),
                        repr(msg)))
        finally :
            self.pop()

        
    def raw(self, name, value):
        """give some raw data"""
        self.push('raw', {'class' : name})
        self._current.text = value
        self.pop()
        


    # event handlers, dispatched by the "notify" method #######################

    def open_test(self, name, **kwargs):
        """handle open_test event"""
        self._ensure_state()
        kwargs['name'] = name
        kwargs['starttime'] = strftime(TIME_FORMAT, localtime())
        if self.rewrite:
            for node in self._current.getiterator(tag='test'):
                if node.get('name') == name:
                    self._current = node
                    node.clear()
                    node.set('name', name)
                    if kwargs:
                        node.attrib.update(kwargs)
                    break
            else:
                self.push('test', kwargs)
        else:
            self.push('test', kwargs)
        self._test_start = time()
        
    def close_test(self):
        """handle close_test event"""
        data = self.pop()
        data.set('duration', '%.3f' % (time() - self._test_start))
        
    def open_check(self, name, **kwargs):
        """handle open_check event"""
        self._ensure_state(1)
        kwargs['name'] = name
        self.push('check', kwargs)
        self._check_start = time()
        
    def close_check(self, status):
        """handle close_check event"""
        data = self.pop()
        data.set('duration', '%.3f' % (time() - self._check_start))
        data.set('status', status)
    
    def context(self):
        """return the current context 
        (a dict wich may have test and check key)"""
        node = self._current
        context = {}
        while node is not None:
            if node.tag in ("check", "test"):
                context[node.tag] = node.get('name')
            node = node.getparent()
        return context



    def _ensure_state(self, level=0):
        """an error may have occured preventing required pops !
        check stack state
        """
        current_lvl = 0
        node = self._current
        while node.getparent() is not None:
            current_lvl += 1
            node = node.getparent()

        while current_lvl > level:
            node = self.pop()
            current_lvl -= 1
            if node.tag == 'check':
                node.set('status', 'error')
