# Copyright (c) 2003-2005 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""
checkers for xml files
"""

from lxml.etree import XMLParser, parse, ErrorLevels

from apycot import register, IChecker, SUCCESS, FAILURE
from apycot.checkers import AbstractFilteredFileChecker
from apycot.Test import SetupException
from apycot import utils

# xml utilities ###############################################################

PARSER = XMLParser(recover=True)
VALIDATING_PARSER = XMLParser(recover=True, dtd_validation=True, load_dtd=True)
LXML_TO_LOGILAB_ERROR_CODE = {
    ErrorLevels.NONE : utils.INFO,
    ErrorLevels.WARNING : utils.WARNING,
    ErrorLevels.ERROR : utils.ERROR,
    ErrorLevels.FATAL: utils.FATAL,
}



class XmlFormChecker(AbstractFilteredFileChecker):
    """check well formed of xml file
    """
    
    __implements__ = IChecker
    __name__ = 'xml_well_formed'

    def __init__(self):
        AbstractFilteredFileChecker.__init__(self, ('.xml', ))

    def check_file(self, filepath, writer):
        """check a single file
        return true if the test succeeded, else false.
        """
        parse(filepath, PARSER)
        if PARSER.error_log:
            for error in PARSER.error_log:
                writer.log(LXML_TO_LOGILAB_ERROR_CODE[error.level],
                    filepath, error.line, error.message)
            return FAILURE
        return SUCCESS
    
    def version_info(self, writer):
        """hook for checkers to add their version information"""
        writer.raw('lxml_parser_version', PARSER.version)

register('checker', XmlFormChecker)


class XmlValidChecker(AbstractFilteredFileChecker):
    """check validity of xml file
    """
    
    __implements__ = IChecker
    __name__ = 'xml_valid'

    def __init__(self):
        AbstractFilteredFileChecker.__init__(self, ('.xml', ))
        self.catalog = None
        
    def check_file(self, filepath, writer):
        """check a single file
        return true if the test succeeded, else false.
        """
        parse(filepath, VALIDATING_PARSER)
        if VALIDATING_PARSER.error_log:
            for error in VALIDATING_PARSER.error_log:
                writer.log(LXML_TO_LOGILAB_ERROR_CODE[error.level],
                    filepath, error.line, error.message)
            return FAILURE
        return SUCCESS
    
    def version_info(self, writer):
        """hook for checkers to add their version information"""
        writer.raw('lxm_parser_version', PARSER.version)

register('checker', XmlValidChecker)
