# Copyright (c) 2003-2006 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""A Test defines :
* a unit of sources to test (a project)
* a list of checks to apply to this unit
* how to build the test environment (preprocessing, dependencies...)
"""

import os
import tempfile
import traceback
from signal import signal, SIGXCPU, SIGKILL, \
    SIGUSR2, SIGUSR1
from os import killpg, getpid, setpgrp
from shutil import rmtree
from threading import Timer, currentThread, Thread, Event
from time import time

from resource import getrlimit, setrlimit, RLIMIT_CPU, RLIMIT_AS

from logilab.common.fileutils import first_level_directory

from apycot.utils import MAIN_SECT, EnvironmentTrackerMixin, ConfigError
from apycot import SetupException, SKIPPED, ERROR, KILLED, get_registered

from apycot.proc import ProcInfoLoader

try:
    class ResourceError(BaseException):
        """Error raise when resource limit is reached"""
        limit = "Unknow Resource Limit"
except NameError:
    class ResourceError(Exception):
        """Error raise when resource limit is reached"""
        limit = "Unknow Resource Limit"


class XCPUError(ResourceError):
    """Error raised when CPU Time limite is reached"""
    limit = "CPU Time"

class LineageMemoryError(ResourceError):
    """Error raised when the total amount of memory used by a process and
    it's child is reached"""
    limit = "Lineage total Memory"

class TimeoutError(ResourceError):
    """Error raised when the process is running for to much time"""
    limit = "Real Time"

# Can't use subclass because the StandardError MemoryError raised
RESOURCE_LIMIT_EXCEPTION = (ResourceError, MemoryError)


class MemorySentinel(Thread):
    """A class checking a process don't use too much memory in a separated
    demonic thread"""
    def __init__(self, interval, memory_limit, gpid=getpid()):
        Thread.__init__(self, target=self._run, name="Test.Sentinel")
        self.memory_limit = memory_limit
        self._stop = Event()
        self.interval = interval
        self.setDaemon(True)
        self.gpid = gpid
    def stop(self):
        """stop ap"""
        self._stop.set()
    def _run(self):
        while not self._stop.isSet():
            pil = ProcInfoLoader()
            pil.load_all()
            if self.memory_limit <= pil.load(self.gpid).lineage_memory_usage():
                killpg(self.gpid, SIGUSR1)
            self._stop.wait(self.interval)




class Test(EnvironmentTrackerMixin):
    """the single source unit test class"""

    def __init__(self, name,
                 checks, repository, preprocessors, writer, python=None,
                 config=None, environ=None, dependencies=(), keep_temp_dir=0,
                 max_cpu_time=None, max_time=None, max_memory=None,
                 max_reprieve=60):
        EnvironmentTrackerMixin.__init__(self)
        self.name = name
        # IRepository object
        self.repo = repository
        # IWriter object
        self.writer = writer
        # list of IPreprocessor objects
        self.preprocessors = preprocessors
        # list of IChecker objects
        self.checkers = checks
        # environment variables as a dictionary
        self.environ = environ
        # list of tuple (IRepository object, dep name)
        self.dependencies = dependencies
        # flag indicating whether to clean test environment after test execution
        self.keep_temp_dir = keep_temp_dir
        # directory where the test environment will be built
        self.max_time = max_time
        self.max_memory = max_memory
        self.max_cpu_time = max_cpu_time
        self._timer = None
        self._msentinel = None
        self.__old_max_memory = None
        self.__old_usr1_handler = None
        self.__old_max_cpu_time = None
        self.__old_usr2_handler = None
        self.__old_sigxcpu_handler = None
        self._limit_set = 0
        self._abort_try = 0
        self._reprieve = max_reprieve
        self._start_time = None
        self._elapse_time = 0
        try:
            testdir = config.get(MAIN_SECT, 'testdir')
        except:
            testdir = '/tmp'
        self.tmpdir = tempfile.mkdtemp(dir=testdir)

    def __str__(self):
        return repr(self.repo)
    
    def update_env(self, key, envvar, value, separator=None):
        """update an environment variable"""
        value = value.replace('${TESTDIR}', self.tmpdir)
        EnvironmentTrackerMixin.update_env(self, key, envvar, value, separator)

    def _hangle_sig_timeout(self, sig, frame):
        raise TimeoutError()
    def _hangle_sig_memory(self, sig, frame):
        if self._abort_try < self._reprieve:
            self._abort_try += 1
            raise LineageMemoryError("Memory limit reached")
        else:
            killpg(getpid(), SIGKILL)

    def _time_out(self):
        if self._abort_try < self._reprieve:
            self._abort_try += 1
            killpg(getpid(), SIGUSR2)
            if self._limit_set > 0:
                self._timer = Timer(1, self._time_out)
                self._timer.start()
        else:
            killpg(getpid(), SIGKILL)

    def _handle_sigxcpu(self, sig, frame):
        if self._abort_try < self._reprieve:
            self._abort_try += 1
            raise XCPUError("Soft CPU time limit reached")
        else:
            killpg(getpid(), SIGKILL)
    
    def _setup_limit(self):
        """set up the process limit"""
        setpgrp()
        if self._limit_set <= 0:
            self.writer.line(2, '-')
            self.writer.msg(2, '  Setup resource limit')
            if self.max_time is not None:

                assert currentThread().getName() == 'MainThread'
                self.writer.line(3, '~')
                self.writer.msg(3, '   Setup TIME limit')
                self.__old_usr2_handler = signal(SIGUSR2,
                                                    self._hangle_sig_timeout)
                self._timer = Timer(max(1, int(self.max_time)
                                                        - self._elapse_time),
                            self._time_out)
                self._start_time = int(time())
                self._timer.start()
            if self.max_cpu_time is not None:
                assert currentThread().getName() == 'MainThread'
                self.writer.line(3, '~')
                self.writer.msg(3, '   Setup CPU TIME limit')
                self.__old_max_cpu_time = getrlimit(RLIMIT_CPU)
                cpu_limit = (int(self.max_cpu_time), self.__old_max_cpu_time[1])
                self.__old_sigxcpu_handler = signal(SIGXCPU,
                                                     self._handle_sigxcpu)
                setrlimit(RLIMIT_CPU, cpu_limit)
            if self.max_memory is not None:
                self.writer.line(3, '~')
                self.writer.msg(3, '   Setup MEMORY limit')
                self._msentinel = MemorySentinel(1, int(self.max_memory) )
                self.__old_max_memory = getrlimit(RLIMIT_AS)
                self.__old_usr1_handler = signal(SIGUSR1,
                                                        self._hangle_sig_memory)
                as_limit = (int(self.max_memory), self.__old_max_memory[1])
                setrlimit(RLIMIT_AS, as_limit)
                self._msentinel.start()
        self._limit_set += 1

    def _clean_limit(self):
        """reinstall the old process limit"""
        if self._limit_set > 0:
            self.writer.line(2, '-')
            self.writer.msg(2, '  Clean resource limit')
            if self.max_time is not None:
                self.writer.line(3, '~')
                self.writer.msg(3, '   Clean TIME limit')
                self._timer.cancel()
                self._elapse_time += int(time())-self._start_time
                self._timer = None
                signal(SIGUSR2, self.__old_usr2_handler)
            if self.max_cpu_time is not None:
                self.writer.line(3, '~')
                self.writer.msg(3, '   Clean CPU TIME limit')
                setrlimit(RLIMIT_CPU, self.__old_max_cpu_time)
                signal(SIGXCPU, self.__old_sigxcpu_handler)
            if self.max_memory is not None:
                self.writer.line(3, '~')
                self.writer.msg(3, '   Clean MEMORY TIME limit')
                self._msentinel.stop()
                self._msentinel = None
                setrlimit(RLIMIT_AS, self.__old_max_memory)
                signal(SIGUSR1, self.__old_usr1_handler)
        self._limit_set -= 1

    def setup(self):
        """setup the test environment"""
        self.writer.line(1, '*')
        self.writer.msg(1, 'Testing %s' % self)
        self._setup_limit()
        try:
            self.writer.line(2, '-')
            self.writer.msg(2, '  Setup test environment')
                    
            self.writer.line(3, '~')
            self.writer.msg(3, '   Setup environment variables')
            # setup environment variables
            if self.environ:
                for key, val in self.environ.items():
                    self.update_env(self.name, key, val)
            
            self.writer.line(3, '~')
            self.writer.msg(3, '   checkout main package')
            # checkout main package
            extract_command = self.repo.checkout_command()
            if extract_command:
                self.writer.msg(1, extract_command)
                status = os.system(extract_command)
                if status:
                    raise SetupException('%r returned status %s, aborting' % (
                        extract_command, status))
            
            self.writer.line(3, '~')
            self.writer.msg(3, '   checkout dependencies')
            # checkout dependencies
            for repo, name, pps in self.dependencies:
                self.writer.line(4, '.')
                self.writer.msg(4, '    checkout %s'%(name, ))
                command = repo.checkout_command()
                if command:
                    self.writer.msg(1, command)
                    status = os.system(command)
                    if status:
                        raise ( SetupException('%r returned status %s, aborting'
                            % ( command, status)))
                    #run preprocessor against the dependancy
                    path = repo.env_path()
                    for preprocessor in self.preprocessors:
                        if preprocessor.match(name) or preprocessor.match(path):
                            preprocessor.dependancy_setup(self, path)
                    for preprocessor in pps:
                        preprocessor.dependancy_setup(self, path)
                    
            self.writer.line(2, '-')
            self.writer.msg(2, '  test environment Setup Done')
        finally:
            self._clean_limit()
        
    def clean(self):
        """clean the test environment"""
        self.writer.line(2, '-')
        self.writer.msg(2, '  Cleanup test environment')
                    
        self.writer.line(3, '~')
        self.writer.msg(3, '   remove main package')
        # remove main package
        if not self.keep_temp_dir:
            try:
                rmtree(first_level_directory(self.repo.env_path()))
            except OSError:
                pass
        
        preprocessors = self.preprocessors[:]
        preprocessors.reverse()
        
        self.writer.line(3, '~')
        self.writer.msg(3, '   remove dependencies')
        # remove dependencies
        for repo, name, pps in self.dependencies:
            self.writer.line(4, '.')
            self.writer.msg(4, '    remove %s'%(name, ))
            path = repo.env_path()

            for preprocessor in pps[::-1]:
                preprocessor.dependancy_clean(self, path)
            # run preprocessor cleaning against the dependancy
            for preprocessor in preprocessors:
                if preprocessor.match(name) or preprocessor.match(path):
                    #XXX does this need Resource watch ?
                    preprocessor.dependancy_clean(self, path) 
            if not self.keep_temp_dir:
                try:
                    rmtree(first_level_directory(path))
                except OSError:
                    # directory has probably been removed earlier (we remove the
                    # first level directory !)
                    continue
    
        self.writer.line(3, '~')
        self.writer.msg(3, '   clean environment variables')
        # clean environment variables
        if self.environ:
            for key in self.environ.keys():
                self.clean_env(self.name, key)
        self.writer.line(2, '-')
        self.writer.msg(2, '  Test environment Cleanup Done')


    def run(self, verb=1):
        """run all checks in the test environment"""
        self._setup_limit()
        try:
            writer = self.writer
            attrs = self.repo.representative_attributes()
            writer.notify('open_test', name=self.name, **attrs)
            checkers = iter(self.checkers)
            skip_message = "Unkown Error in Unknown check"
            try:
                for checker in checkers:
                    writer.notify('open_check', name=checker.__name__,
                                  **checker.options)
                    self.writer.line(2, '-')
                    try:
                        try:
                            checker.check_options()
                        except ConfigError, ex:
                            writer.fatal(None, None,
                                msg="Config Error for %s checker: %s"%
                                                    (checker.__name__, ex))
                            writer.notify('close_check', status=ERROR)
                            continue
                        self.writer.msg(1, 'Run check %s' % checker.__name__)
                            
                        preprocessors = self.preprocessors[:]
                        if not self.pp_call(checker, preprocessors,
                                                        'check_setup', verb):
                            writer.notify('close_check', status=ERROR)
                            continue
                        
                        try:
                            status = checker.run(self, writer)
                        except RESOURCE_LIMIT_EXCEPTION:
                            raise
                        except Exception, ex:
                            if verb:
                                traceback.print_exc()
                            writer.log(
                                FATAL, self.repo.env_path(), None,
                                'Error while running checker %s: %s' % (
                                checker.__name__, ex))
                            status = ERROR

                        preprocessors.reverse()
                        if not self.pp_call(checker, preprocessors,
                                                        'check_clean', verb):
                            status = ERROR
                        writer.notify('close_check', status=status)
                        self.writer.msg(1, '[%s]' % status)
                    except ResourceError, ex:
                        skip_message = '%s limit reached while running checker'\
                                    ' %s' % ( ex.limit, checker.__name__)
                        writer.log(
                            FATAL, self.repo.env_path(), None, skip_message)
                        writer.notify('close_check', status=KILLED)
                        raise
                    except MemoryError:
                        skip_message = 'Memory limit reached while running'\
                                    'checker %s' % (checker.__name__, )
                        writer.log(
                            FATAL, self.repo.env_path(), None, skip_message)
                        writer.notify('close_check', status=KILLED)
                        raise
            finally:
                self._skip(checkers, skip_message)
                writer.notify('close_test')
        finally:
            self._clean_limit()


    def pp_call(self, checker, preprocessors, callback_name, verb=1):
        """call preprocessors"""
        try:
            for preprocessor in preprocessors:
                if preprocessor.match(checker.__name__):
                    callback = getattr(preprocessor, callback_name)
                    callback(self, checker)
            return True
        except RESOURCE_LIMIT_EXCEPTION:
            raise
        except Exception, ex:
            if verb:
                traceback.print_exc()
            msg = 'error while running preprocessor %s: %s'
            self.writer.log(FATAL, '', None, msg % (preprocessor.__name__, ex))
            return False

    def _skip(self, checkers, skip_message):
        """mark all checkers as skipped for the 'skip_message reason'"""
        for checker in checkers:
            self.writer.notify('open_check', name=checker.__name__,
                          **checker.options)
            self.writer.log(
                    FATAL, self.repo.env_path(), None,
                    "%s checker didn't run because : %s" % (
                    checker.__name__, skip_message))
            self.writer.notify('close_check', status=SKIPPED)

    def execute(self):
        """setup, run and clean the test"""
        tempdir = self.tmpdir
        cwd = os.getcwd()
        os.chdir(tempdir)
        try: # finally clean and rmtree
            try: # except XCPUError
                try: # except Setup Exception
                    # setup test
                    self.setup()
                except SetupException, ex:
                    msg = 'Fatal setup exception: %s' % ex
                    self.writer.msg(0, msg)
                    attrs = self.repo.representative_attributes()
                    self.writer.notify('open_test', name=self.name, **attrs)
                    self._skip(self.checkers, msg)
                    self.writer.notify('close_test')
                    return
                # run the test (which is actually a suite of test)
                self.run()
            except ResourceError, ex:
                self.writer.msg(0, '%s resource limit reached Test arborted'
                                                                    % ex.limit)
            except MemoryError:
                self.writer.msg(0, 'Memory resource limit reached Test'\
                                                                    'arborted')
                    
        finally:
            self.clean()
            os.chdir(cwd)
            if not self.keep_temp_dir:
                rmtree(tempdir)
            else:
                self.writer.msg(0, 'Temporary directory not removed: %s' %
                                                                        tempdir)
