/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.jdbc.datasource;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import javax.sql.DataSource;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.jdbc.CannotGetJdbcConnectionException;
import org.springframework.jdbc.datasource.ConnectionHolder;
import org.springframework.jdbc.datasource.ConnectionProxy;
import org.springframework.jdbc.datasource.DelegatingDataSource;
import org.springframework.jdbc.datasource.SmartDataSource;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import org.springframework.util.Assert;

public abstract class DataSourceUtils {
    public static final int CONNECTION_SYNCHRONIZATION_ORDER = 1000;
    private static final Log logger = LogFactory.getLog((Class)(class$org$springframework$jdbc$datasource$DataSourceUtils == null ? (class$org$springframework$jdbc$datasource$DataSourceUtils = DataSourceUtils.class$("org.springframework.jdbc.datasource.DataSourceUtils")) : class$org$springframework$jdbc$datasource$DataSourceUtils));
    static /* synthetic */ Class class$org$springframework$jdbc$datasource$DataSourceUtils;

    public static Connection getConnection(DataSource dataSource) throws CannotGetJdbcConnectionException {
        try {
            return DataSourceUtils.doGetConnection(dataSource);
        }
        catch (SQLException ex) {
            throw new CannotGetJdbcConnectionException("Could not get JDBC Connection", ex);
        }
    }

    public static Connection doGetConnection(DataSource dataSource) throws SQLException {
        Assert.notNull((Object)dataSource, (String)"No DataSource specified");
        ConnectionHolder conHolder = (ConnectionHolder)((Object)TransactionSynchronizationManager.getResource((Object)dataSource));
        if (conHolder != null && (conHolder.hasConnection() || conHolder.isSynchronizedWithTransaction())) {
            conHolder.requested();
            if (!conHolder.hasConnection()) {
                logger.debug((Object)"Fetching resumed JDBC Connection from DataSource");
                conHolder.setConnection(dataSource.getConnection());
            }
            return conHolder.getConnection();
        }
        logger.debug((Object)"Fetching JDBC Connection from DataSource");
        Connection con = dataSource.getConnection();
        if (TransactionSynchronizationManager.isSynchronizationActive()) {
            logger.debug((Object)"Registering transaction synchronization for JDBC Connection");
            ConnectionHolder holderToUse = conHolder;
            if (holderToUse == null) {
                holderToUse = new ConnectionHolder(con);
            } else {
                holderToUse.setConnection(con);
            }
            holderToUse.requested();
            TransactionSynchronizationManager.registerSynchronization((TransactionSynchronization)new ConnectionSynchronization(holderToUse, dataSource));
            holderToUse.setSynchronizedWithTransaction(true);
            if (holderToUse != conHolder) {
                TransactionSynchronizationManager.bindResource((Object)dataSource, (Object)((Object)holderToUse));
            }
        }
        return con;
    }

    public static Integer prepareConnectionForTransaction(Connection con, TransactionDefinition definition) throws SQLException {
        Assert.notNull((Object)con, (String)"No Connection specified");
        if (definition != null && definition.isReadOnly()) {
            try {
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)("Setting JDBC Connection [" + con + "] read-only"));
                }
                con.setReadOnly(true);
            }
            catch (Throwable ex) {
                logger.debug((Object)"Could not set JDBC Connection read-only", ex);
            }
        }
        Integer previousIsolationLevel = null;
        if (definition != null && definition.getIsolationLevel() != -1) {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Changing isolation level of JDBC Connection [" + con + "] to " + definition.getIsolationLevel()));
            }
            previousIsolationLevel = new Integer(con.getTransactionIsolation());
            con.setTransactionIsolation(definition.getIsolationLevel());
        }
        return previousIsolationLevel;
    }

    public static void resetConnectionAfterTransaction(Connection con, Integer previousIsolationLevel) {
        Assert.notNull((Object)con, (String)"No Connection specified");
        try {
            if (previousIsolationLevel != null) {
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)("Resetting isolation level of JDBC Connection [" + con + "] to " + previousIsolationLevel));
                }
                con.setTransactionIsolation(previousIsolationLevel);
            }
            if (con.isReadOnly()) {
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)("Resetting read-only flag of JDBC Connection [" + con + "]"));
                }
                con.setReadOnly(false);
            }
        }
        catch (Throwable ex) {
            logger.debug((Object)"Could not reset JDBC Connection after transaction", ex);
        }
    }

    public static void applyTransactionTimeout(Statement stmt, DataSource dataSource) throws SQLException {
        Assert.notNull((Object)stmt, (String)"No statement specified");
        ConnectionHolder holder = (ConnectionHolder)((Object)TransactionSynchronizationManager.getResource((Object)dataSource));
        if (holder != null && holder.hasTimeout()) {
            stmt.setQueryTimeout(holder.getTimeToLiveInSeconds());
        }
    }

    public static void releaseConnection(Connection con, DataSource dataSource) {
        try {
            DataSourceUtils.doReleaseConnection(con, dataSource);
        }
        catch (SQLException ex) {
            logger.debug((Object)"Could not close JDBC Connection", (Throwable)ex);
        }
        catch (Throwable ex) {
            logger.debug((Object)"Unexpected exception on closing JDBC Connection", ex);
        }
    }

    public static void doReleaseConnection(Connection con, DataSource dataSource) throws SQLException {
        ConnectionHolder conHolder;
        if (con == null) {
            return;
        }
        if (dataSource != null && (conHolder = (ConnectionHolder)((Object)TransactionSynchronizationManager.getResource((Object)dataSource))) != null && conHolder.hasConnection() && DataSourceUtils.connectionEquals(conHolder.getConnection(), con)) {
            conHolder.released();
            return;
        }
        if (!(dataSource instanceof SmartDataSource) || ((SmartDataSource)dataSource).shouldClose(con)) {
            logger.debug((Object)"Returning JDBC Connection to DataSource");
            con.close();
        }
    }

    private static boolean connectionEquals(Connection heldCon, Connection passedInCon) {
        return heldCon == passedInCon || heldCon.equals(passedInCon) || DataSourceUtils.getTargetConnection(heldCon).equals(passedInCon);
    }

    public static Connection getTargetConnection(Connection con) {
        Connection conToUse = con;
        while (conToUse instanceof ConnectionProxy) {
            conToUse = ((ConnectionProxy)conToUse).getTargetConnection();
        }
        return conToUse;
    }

    private static int getConnectionSynchronizationOrder(DataSource dataSource) {
        int order = 1000;
        DataSource currDs = dataSource;
        while (currDs instanceof DelegatingDataSource) {
            --order;
            currDs = ((DelegatingDataSource)currDs).getTargetDataSource();
        }
        return order;
    }

    static /* synthetic */ Class class$(String x0) {
        try {
            return Class.forName(x0);
        }
        catch (ClassNotFoundException x1) {
            throw new NoClassDefFoundError(x1.getMessage());
        }
    }

    private static class ConnectionSynchronization
    extends TransactionSynchronizationAdapter {
        private final ConnectionHolder connectionHolder;
        private final DataSource dataSource;
        private int order;
        private boolean holderActive = true;

        public ConnectionSynchronization(ConnectionHolder connectionHolder, DataSource dataSource) {
            this.connectionHolder = connectionHolder;
            this.dataSource = dataSource;
            this.order = DataSourceUtils.getConnectionSynchronizationOrder(dataSource);
        }

        public int getOrder() {
            return this.order;
        }

        public void suspend() {
            if (this.holderActive) {
                TransactionSynchronizationManager.unbindResource((Object)this.dataSource);
                if (this.connectionHolder.hasConnection() && !this.connectionHolder.isOpen()) {
                    DataSourceUtils.releaseConnection(this.connectionHolder.getConnection(), this.dataSource);
                    this.connectionHolder.setConnection(null);
                }
            }
        }

        public void resume() {
            if (this.holderActive) {
                TransactionSynchronizationManager.bindResource((Object)this.dataSource, (Object)((Object)this.connectionHolder));
            }
        }

        public void beforeCompletion() {
            if (!this.connectionHolder.isOpen()) {
                TransactionSynchronizationManager.unbindResource((Object)this.dataSource);
                this.holderActive = false;
                if (this.connectionHolder.hasConnection()) {
                    DataSourceUtils.releaseConnection(this.connectionHolder.getConnection(), this.dataSource);
                }
            }
        }

        public void afterCompletion(int status) {
            if (this.holderActive) {
                if (TransactionSynchronizationManager.hasResource((Object)this.dataSource)) {
                    TransactionSynchronizationManager.unbindResource((Object)this.dataSource);
                }
                this.holderActive = false;
                if (this.connectionHolder.hasConnection()) {
                    DataSourceUtils.releaseConnection(this.connectionHolder.getConnection(), this.dataSource);
                    this.connectionHolder.setConnection(null);
                }
                this.connectionHolder.reset();
            }
        }
    }
}

