//-----------------------------------------------------------------------------
// Iceman
//
// This code is licensed to you under the terms of the GNU GPL, version 2 or,
// at your option, any later version. See the LICENSE.txt file for the text of
// the license.
//-----------------------------------------------------------------------------
// Low frequency COTAG commands
//-----------------------------------------------------------------------------
#include "cmdlfcotag.h"  // COTAG function declarations
#include <string.h>
#include <stdio.h>
#include "cmdparser.h"    // command_t
#include "comms.h"
#include "lfdemod.h"
#include "cmddata.h"    // getSamples
#include "ui.h"         // PrintAndLog
#include "ctype.h"      // tolower
#include "cliparser.h"

static int CmdHelp(const char *Cmd);

// COTAG demod should be able to use g_GraphBuffer,
// when data load samples
int demodCOTAG(bool verbose) {
    (void) verbose; // unused so far

    uint8_t bits[COTAG_BITS] = {0};
    size_t bitlen = COTAG_BITS;
    memcpy(bits, g_DemodBuffer, COTAG_BITS);

    uint8_t alignPos = 0;
    uint16_t err = manrawdecode(bits, &bitlen, 1, &alignPos);
    if (err > 50) {
        PrintAndLogEx(DEBUG, "DEBUG: Error - COTAG too many errors: %d", err);
        return PM3_ESOFT;
    }

    setDemodBuff(bits, bitlen, 0);

    //got a good demod
    uint16_t cn = bytebits_to_byteLSBF(bits + 1, 16);
    uint32_t fc = bytebits_to_byteLSBF(bits + 1 + 16, 8);

    uint32_t raw1 = bytebits_to_byteLSBF(bits, 32);
    uint32_t raw2 = bytebits_to_byteLSBF(bits + 32, 32);
    uint32_t raw3 = bytebits_to_byteLSBF(bits + 64, 32);
    uint32_t raw4 = bytebits_to_byteLSBF(bits + 96, 32);

    /*
    fc 161:   1010 0001 -> LSB 1000 0101
    cn 33593  1000 0011 0011 1001 -> LSB 1001 1100 1100 0001
        cccc cccc cccc cccc                     ffffffff
      0 1001 1100 1100 0001 1000 0101 0000 0000 100001010000000001111011100000011010000010000000000000000000000000000000000000000000000000000000100111001100000110000101000
        1001 1100 1100 0001                     10000101
    */
    PrintAndLogEx(SUCCESS, "COTAG Found: FC %u, CN: %u Raw: %08X%08X%08X%08X", fc, cn, raw1, raw2, raw3, raw4);
    return PM3_SUCCESS;
}

static int CmdCOTAGDemod(const char *Cmd) {
    CLIParserContext *ctx;
    CLIParserInit(&ctx, "lf cotag demod",
                  "Try to find COTAG preamble, if found decode / descramble data",
                  "lf cotag demod"
                 );

    void *argtable[] = {
        arg_param_begin,
        arg_param_end
    };
    CLIExecWithReturn(ctx, Cmd, argtable, true);
    CLIParserFree(ctx);
    return demodCOTAG(true);
}

// When reading a COTAG.
// 0 = HIGH/LOW signal - maxlength bigbuff
// 1 = translation for HI/LO into bytes with manchester 0,1 - length 300
// 2 = raw signal -  maxlength bigbuff
static int CmdCOTAGReader(const char *Cmd) {

    CLIParserContext *ctx;
    CLIParserInit(&ctx, "lf cotag reader",
                  "read a COTAG tag,  the current support for COTAG is limited. ",
                  "lf cotag reader -2"
                 );

    void *argtable[] = {
        arg_param_begin,
        arg_lit0("1", NULL, "HIGH/LOW signal; maxlength bigbuff"),
        arg_lit0("2", NULL, "translation of HIGH/LOW into bytes with manchester 0,1"),
        arg_lit0("3", NULL, "raw signal; maxlength bigbuff"),
        arg_param_end
    };

    CLIExecWithReturn(ctx, Cmd, argtable, false);
    bool mode0 = arg_get_lit(ctx, 1);
    bool mode1 = arg_get_lit(ctx, 2);
    bool mode2 = arg_get_lit(ctx, 3);
    CLIParserFree(ctx);

    if ((mode0 + mode1 + mode2) > 1) {
        PrintAndLogEx(ERR, "You can only use one option at a time");
        return PM3_EINVARG;
    }
    uint8_t mode = 0xFF;
    if (mode0)
        mode = 0;
    if (mode1)
        mode = 1;
    if (mode2)
        mode = 2;

    struct p {
        uint8_t mode;
    } PACKED payload;
    payload.mode = mode;

    PacketResponseNG resp;
    clearCommandBuffer();
    SendCommandNG(CMD_LF_COTAG_READ, (uint8_t *)&payload, sizeof(payload));

    uint8_t timeout = 3;
    int res = PM3_SUCCESS;
    while (!WaitForResponseTimeout(CMD_LF_COTAG_READ, &resp, 1000)) {
        timeout--;
        PrintAndLogEx(NORMAL, "." NOLF);
        if (timeout == 0) {
            PrintAndLogEx(NORMAL, "");
            PrintAndLogEx(WARNING, "command execution time out");
            SendCommandNG(CMD_BREAK_LOOP, NULL, 0);
            res = PM3_ETIMEOUT;
        }
    }

    if (res != PM3_SUCCESS) {
        return res;
    }

    if (timeout != 3)
        PrintAndLogEx(NORMAL, "");

    switch (payload.mode) {
        case 0:
        case 2: {
            CmdPlot("");
            CmdGrid("384");
            getSamples(0, false);
            break;
        }
        case 1: {
            memcpy(g_DemodBuffer, resp.data.asBytes, resp.length);
            g_DemodBufferLen = resp.length;
            return demodCOTAG(true);
        }
    }
    return PM3_SUCCESS;
}

static command_t CommandTable[] = {
    {"help",    CmdHelp,         AlwaysAvailable, "This help"},
    {"demod",   CmdCOTAGDemod,   AlwaysAvailable, "demodulate an COTAG tag"},
    {"reader",  CmdCOTAGReader,  IfPm3Lf,         "attempt to read and extract tag data"},
    {NULL, NULL, NULL, NULL}
};
static int CmdHelp(const char *Cmd) {
    (void)Cmd; // Cmd is not used so far
    CmdsHelp(CommandTable);
    return PM3_SUCCESS;
}

int CmdLFCOTAG(const char *Cmd) {
    clearCommandBuffer();
    return CmdsParse(CommandTable, Cmd);
}

int readCOTAGUid(void) {
    return (CmdCOTAGReader("-2") == PM3_SUCCESS);
}
