/**
This file is part of a jTEM project.
All jTEM projects are licensed under the FreeBSD license 
or 2-clause BSD license (see http://www.opensource.org/licenses/bsd-license.php). 

Copyright (c) 2006-2010, Technische Universität Berlin, jTEM
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer.

-	Redistributions in binary form must reproduce the above copyright notice, 
	this list of conditions and the following disclaimer in the documentation 
	and/or other materials provided with the distribution.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
**/

package de.jtem.beans;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.beans.PropertyChangeListener;
import java.beans.PropertyEditorSupport;

import javax.swing.JPanel;
import javax.swing.JTextField;


/** The string editor uses a {@link JTextField}. It turns into editing mode, when 
 * a key is pressed. While in editing mode all attempts to {@link #setAsText(String)} or
 * {@link #setValue(Object)} are blocked. 
 * 
 * <p>Typing ENTER turns of editing, updates the
 * value to the edited <code>String</code>, and informs {@link PropertyChangeListener}s.
 * When ESC is pressed or focus is lost the editing is canceled, i.e.
 * the old value is restored and editing mode is turned off. 
 * 
 * <p>The user gets feedback about the current editing mode via the background color. The 
 * colors are taken from {@link #editingBackground} and {@link #displayingBackground} 
 * 
 * @author G. Paul Peters, Sep 16, 2009
 *
 */
public class StringEditor extends PropertyEditorSupport {
	
	public static Color editingBackground=Color.YELLOW;
	public static Color displayingBackground=Color.WHITE;

	final private JTextField field = new JTextField();
	final private JPanel editor;
	private boolean editing=false;

	
	public StringEditor() {
		editor = new JPanel(new BorderLayout());
		field.setMinimumSize(new Dimension(60,20));
		field.setPreferredSize(new Dimension(60,20));
		field.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent arg0) {
				setEditing(false);
				setAsText(field.getText());
			}
		});
		field.addFocusListener(new FocusListener() {
			public void focusLost(FocusEvent e) {
				if (!editing) return;
				setEditing(false);
				setValue(getValue());
			}
			public void focusGained(FocusEvent e) {
			}
		});
		field.addKeyListener(new KeyListener() {
			public void keyTyped(KeyEvent e) {
				setEditing(true);
			}
			public void keyReleased(KeyEvent e) {
				setEditing(true);
				switch (e.getKeyCode()) {
				case KeyEvent.VK_ESCAPE:
						setEditing(false);
						setValue(getValue());
				case KeyEvent.VK_ENTER:
					setEditing(false);
				}
			}
			public void keyPressed(KeyEvent e) {
				setEditing(true);
			}
		});
		editor.add("Center", field);
		
		setEditing(false);
	}
	
	public boolean supportsCustomEditor() {
		return true;
	}

	public Component getCustomEditor() {
		return editor;
	}
	
	public String getAsText() {
		return (String) getValue();
	}
	
	public void setAsText(String text) {
		setValue(text);
	}
	
	public void setValue(Object value) {    
		if (editing)  return;
		super.setValue(value);
		field.setText(getAsText());
		setEditing(false);
}
	
	public boolean isEditing() {
		return editing;
	}
	
	private void setEditing(boolean editing) {
		this.editing=editing;
		field.setBackground(editing?editingBackground:displayingBackground);
	}
}
