/**
This file is part of a jTEM project.
All jTEM projects are licensed under the FreeBSD license 
or 2-clause BSD license (see http://www.opensource.org/licenses/bsd-license.php). 

Copyright (c) 2002-2009, Technische Universität Berlin, jTEM
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer.

-	Redistributions in binary form must reproduce the above copyright notice, 
	this list of conditions and the following disclaimer in the documentation 
	and/or other materials provided with the distribution.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
**/

package de.jtem.jterm;

import java.awt.Color;
import java.awt.Toolkit;

import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultStyledDocument;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.StyleConstants;

/**
 * The type of StyledDocument used by JTerm to represent a session with
 * some interpreter.
 */
public class Session extends DefaultStyledDocument {
    static int SHOW_AMBIG_MAX= 10;
    int promptPosition;
    String prompt;
    static String DEFAULTPROMPT= ">";
    static int MAXHISTORYSIZE= 100;
    int historySize= 0;
    int pos;
    SimpleAttributeSet promptStyle;
    SimpleAttributeSet inputStyle;
    SimpleAttributeSet outputStyle;
    SimpleAttributeSet caretStyle;
    SimpleAttributeSet completionStyle;
    SimpleAttributeSet errorStyle;
    String[] history= new String[MAXHISTORYSIZE];
    StringEvaluator evaluator;

    /** Create a session for the evaluator e with prescribed prompt */
    public Session(String promptString, StringEvaluator e) {
        evaluator= e;
        prompt= promptString;
        inputStyle= new SimpleAttributeSet();
        //StyleConstants.setForeground( inputStyle, new Color( 5,5,140) );
        //StyleConstants.setBackground( inputStyle, Color.red );

        StyleConstants.setFontFamily(inputStyle, "Monospaced");
        StyleConstants.setFontSize(inputStyle, 12);

        outputStyle= new SimpleAttributeSet(inputStyle);
        StyleConstants.setForeground(outputStyle, new Color(5, 5, 140));

        promptStyle= new SimpleAttributeSet(inputStyle);
        StyleConstants.setForeground(promptStyle, new Color(5, 5, 140));

        caretStyle= new SimpleAttributeSet(inputStyle);
        StyleConstants.setForeground(caretStyle, Color.white);
        StyleConstants.setBackground(caretStyle, Color.black);

        completionStyle= new SimpleAttributeSet(inputStyle);
        StyleConstants.setForeground(completionStyle, Color.gray);

        errorStyle= new SimpleAttributeSet(inputStyle);
        StyleConstants.setForeground(errorStyle, new Color(140, 5, 5));
        try {
            super.insertString(0, prompt, promptStyle);
        } catch (BadLocationException ble) {
            System.err.println("Couldn't insert initial prompt.");
        }
        promptPosition= prompt.length();
    }

    /** Create a session for the evaluator e with a default prompt. */
    public Session(StringEvaluator e) {
        this(DEFAULTPROMPT, e);
    }

    /**
     * Returns the text portion between the prompt and the end of the
     * document.
     */
    protected String getCommand() {
        try {
            return (getText(promptPosition, getLength() - promptPosition));
        } catch (BadLocationException ble) {
            System.out.println("unable to get the command");
            return ("");
        }
    }

    /**
     * Removes the text portion between the prompt and the end of the
     * document and replaces it by str.
     */
    protected void replaceCommand(String str) {
        try {
            super.remove(promptPosition, getLength() - promptPosition);
            super.insertString(promptPosition, str, inputStyle);
        } catch (BadLocationException ble) {
            System.out.println("unable to insert ");
        }
    }

    /**
     * Display the result of some work done by the evaluator and provide
     * a new prompt.
     */
    public void displayAndPrompt(String result, AttributeSet style) {
        try {
            super.insertString(getLength(), result, style);
            super.insertString(getLength(), prompt, promptStyle);
            promptPosition= getLength();
        } catch (BadLocationException ble) {
            System.out.println("unable to display result");
        }
    }

    /** Ignore requests to replace the selection */
    public void remove(int offs, int len) {
        try {
            //System.out.println( "offs: "+offs+" promptPosition: "+promptPosition+ " lens: "+len);
            if (offs >= promptPosition)
                super.remove(offs, len);
        } catch (BadLocationException ble) {
            System.out.println(ble.getMessage());
        }

    }

    /**
     * Inserts str at the specified offset using the input style, iff
     * the offset corresponds to a position behaind the last promp.
     * Usually invoked as result of a keypress or paste action.
     */
    public void insertString(int offs, String str, AttributeSet a)
      throws BadLocationException {
        if (offs >= promptPosition)
            super.insertString(offs, str, inputStyle);
    }

    /** Go backwards in the command history. */
    public void historyPrevious() {
        if (pos == 0)
            Toolkit.getDefaultToolkit().beep();
        else {
            history[pos]= getCommand();
            pos--;
            replaceCommand(history[pos]);
        }
    }

    /** Go forward in the command history. */
    public void historyNext() {
        if (pos == historySize)
            Toolkit.getDefaultToolkit().beep();
        else {
            pos++;
            replaceCommand(history[pos]);
        }
    }

    /**
     * Send the text portion between the prompt and the end of the
     * document to the evaluator and append the result.
     */
    public void evaluate() {
        String script= getCommand();
        InterpreterResult result;
        String answer;
        if (script.length() > 0) {
            history[historySize]= script;
            historySize= (historySize + 1) % MAXHISTORYSIZE;
            pos= historySize;
            // System.out.println(script);
            result= evaluator.evaluate(script);
        } else {
            result= new InterpreterResult("");
        }
        if (result.message.length() > 0) {
            answer= "\n" + result.message + "\n";
        } else {
            answer= "\n";
        }
        if (result.isErrorMessage) {
            Toolkit.getDefaultToolkit().beep();
            displayAndPrompt(answer, errorStyle);
        } else {
            displayAndPrompt(answer, outputStyle);
        }
    }
}
