//
//  PXLayerController.m
//  Pixen-XCode
//
//  Created by Joe Osborn on Thu Feb 05 2004.
//  Copyright (c) 2004 Open Sword Group. All rights reserved.
//

#import "PXLayerController.h"
#import "PXLayerDetailsView.h"
#import "PXCanvas.h"
#import "PXLayer.h"
#import "PXSelectionLayer.h"
#import "SubviewTableViewController.h"

NSString * PXLayerSelectionDidChangeName = @"PXLayerNotificationDidChange";
NSString * PXCanvasLayerSelectionDidChangeName = @"PXCanvasLayerNotificationDidChange";

@implementation PXLayerController

- initWithCanvas:aCanvas
{
	[super init];
	[NSBundle loadNibNamed:@"PXLayerController" owner:self];
	views = [[NSMutableArray alloc] initWithCapacity:8];
	[self selectRow:-1];
	[self setCanvas:aCanvas];
	[tableView registerForDraggedTypes:[NSArray arrayWithObject:@"PXLayerRowPasteType"]];
	return self;
}

- (void)setCanvas:aCanvas
{
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	[aCanvas retain];
	[canvas release];
	canvas = aCanvas;
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(reloadData:) name:PXCanvasLayersChangedNotificationName object:canvas];
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(canvasLayerChanged:) name:PXCanvasLayerSelectionDidChangeName object:canvas];
	[self reloadData:nil];
}

- (void)awakeFromNib
{
	tableViewController = [[SubviewTableViewController controllerWithViewColumn:[tableView tableColumnWithIdentifier:@"details"]] retain];
    [tableViewController setDelegate:self];
}

- undoManager
{
    return [[[NSDocumentController sharedDocumentController] currentDocument] undoManager];
}

- (void)resetViewHiddenStatus
{
	BOOL shouldBeHidden = (([drawer state] == NSDrawerClosedState) || ([drawer state] == NSDrawerClosingState));
	id enumerator = [views objectEnumerator];
	id current;
	while(current = [enumerator nextObject])
	{
		[current setHidden:shouldBeHidden];
	}
}

- (NSView *) tableView:(NSTableView *)tableView viewForRow:(int)row
{
	return [views objectAtIndex:[self invertLayerIndex:row]];	
}

- (int)numberOfRowsInTableView:view
{
	return [[canvas layers] count];
}

- (void)reloadData:aNotification
{
	int i, selectedRow;
	[views makeObjectsPerformSelector:@selector(invalidateTimer)];
	if ([tableView selectedRow] == -1) { [self selectRow:0]; }
	selectedRow = [self invertLayerIndex:[tableView selectedRow]];
	for(i = 0; i < [[canvas layers] count]; i++)
	{
		id layer = [[canvas layers] objectAtIndex:i];
		if([views count] > i)
		{
			[[views objectAtIndex:i] setLayer:layer];
		}
		else
		{
			id newView = [[[PXLayerDetailsView alloc] initWithLayer:layer] autorelease];
			[views addObject:newView];
		}
	}
	[views removeObjectsInRange:NSMakeRange(i, [views count] - i)];
	[tableViewController reloadTableView];
	[self selectRow:[self invertLayerIndex:selectedRow]];
	[self resetViewHiddenStatus];
}

- (void)selectRow:(int)index
{
	if ([tableView respondsToSelector:@selector(selectRowIndexes:byExtendingSelection:)])
	{
		[tableView selectRowIndexes:[NSIndexSet indexSetWithIndex:index] byExtendingSelection:NO];
	}
	else
	{
		[tableView selectRow:index byExtendingSelection:NO];
	}
	[self updateRemoveButtonStatus];
}

- (id)tableView:(NSTableView *)aTableView objectValueForTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
	if([[aTableColumn identifier] isEqualToString:@"visible"])
	{
		return [NSNumber numberWithBool:[[[canvas layers] objectAtIndex:[self invertLayerIndex:rowIndex]] visible]];
	}
	return nil;
}

- (void)tableView:(NSTableView *)aTableView setObjectValue:(id)anObject forTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
	if([[aTableColumn identifier] isEqualToString:@"visible"])
	{
		[[[canvas layers] objectAtIndex:[self invertLayerIndex:rowIndex]] setVisible:[anObject boolValue]];
		[canvas changedInRect:NSMakeRect(0,0,[canvas size].width, [canvas size].height)];
	}
}

- (void)toggle:sender
{
	[drawer toggle:sender];
	[drawer setContentSize:NSMakeSize(229, [drawer contentSize].height)];
	[self resetViewHiddenStatus];
}

- (void)setWindow:aWindow
{
	window = aWindow;
	[drawer setParentWindow:window];
}

- (void)setLayers:layers fromLayers:oldLayers resetSelection:(BOOL)resetSelection
{
	[[[self undoManager] prepareWithInvocationTarget:self] setLayers:oldLayers fromLayers:layers resetSelection:YES];
	[canvas setLayers:layers];
	if (resetSelection)
	{
	 	[self selectRow:0];
	 	[self selectLayer:nil];
	}
}

- (IBAction)addLayer:sender
{
	[[self undoManager] beginUndoGrouping];
	[[self undoManager] setActionName:@"Add Layer"];
	[self setLayers:[[canvas layers] deepMutableCopy] fromLayers:[canvas layers] resetSelection:YES];		
	[canvas deselect];
	[canvas addLayer:[[PXLayer alloc] initWithName:@"New Layer" size:[canvas size]]];
	[self selectRow:0];
	[self selectLayer:nil];
	[[self undoManager] endUndoGrouping];
}

- (IBAction)removeLayer:sender
{
	if ([tableView selectedRow] == -1) { return; }
	[[self undoManager] beginUndoGrouping];
	[[self undoManager] setActionName:@"Remove Layer"];
	[self setLayers:[[canvas layers] deepMutableCopy] fromLayers:[canvas layers] resetSelection:NO];
	[canvas deselect];
	[canvas removeLayerAtIndex:[self invertLayerIndex:[tableView selectedRow]]];
	[self selectRow:[self invertLayerIndex:0]];
	[self selectLayer:nil];
	[[self undoManager] endUndoGrouping];
}

- (IBAction)selectLayer:sender
{
	if ([tableView selectedRow] == -1) { [self selectRow:0]; }
	int row = [self invertLayerIndex:[tableView selectedRow]];
	if (![[[canvas layers] lastObject] isKindOfClass:[PXSelectionLayer class]])
	{	
	 	[canvas deselect];
	}
	[self selectRow:[self invertLayerIndex:row]];
	[[NSNotificationCenter defaultCenter] postNotificationName:PXLayerSelectionDidChangeName object:self userInfo:[[canvas layers] objectAtIndex:row]];
}

- (void)updateRemoveButtonStatus
{
	if ([self invertLayerIndex:[tableView selectedRow]] == 0) { [removeButton setEnabled:NO]; }
	else { [removeButton setEnabled:YES]; }
}

- (int)invertLayerIndex:(int)anIndex
{
	return [[canvas layers] count] - anIndex - 1;
}

- (void)canvasLayerChanged:notification
{
	[self selectRow:[self invertLayerIndex:[[canvas layers] indexOfObject:[canvas activeLayer]]]];
}

- (void)mergeDown
{
	[[self undoManager] beginUndoGrouping];
	[[self undoManager] setActionName:@"Merge Down"];
	[self setLayers:[[canvas layers] deepMutableCopy] fromLayers:[canvas layers] resetSelection:YES];
	if ([[[canvas layers] objectAtIndex:[self invertLayerIndex:[tableView selectedRow]]] isKindOfClass:[PXSelectionLayer class]]) { [canvas deselect]; return; }
	if ([self invertLayerIndex:[tableView selectedRow]] == 0) { return; }
	[[[canvas layers] objectAtIndex:[self invertLayerIndex:[tableView selectedRow]+1]] compositeUnder:
		[[canvas layers] objectAtIndex:[self invertLayerIndex:[tableView selectedRow]]]];
	[canvas removeLayerAtIndex:[self invertLayerIndex:[tableView selectedRow]]];
	[self selectRow:[tableView selectedRow]];
	[self selectLayer:[[canvas layers] objectAtIndex:[self invertLayerIndex:[tableView selectedRow]]]];
	[[self undoManager] endUndoGrouping];
}

- (BOOL)tableView:(NSTableView *)aTableView writeRows:(NSArray *)rows toPasteboard:(NSPasteboard *)pboard
{
	if ([[[canvas layers] objectAtIndex:[self invertLayerIndex:[tableView selectedRow]]] isKindOfClass:[PXSelectionLayer class]]) { return NO; }
	[pboard declareTypes:[NSArray arrayWithObject:@"PXLayerRowPasteType"] owner:self];
	[pboard setString:[NSString stringWithFormat:@"%d", [self invertLayerIndex:[[rows objectAtIndex:0] intValue]]] forType:@"PXLayerRowPasteType"];
	//[pboard setData:[NSKeyedArchiver archivedDataWithRootObject:[[canvas layers] objectAtIndex:[self invertLayerIndex:[tableView selectedRow]]]] forType:@"PXLayerRowPasteType"];
	return YES;
}

- (NSDragOperation)tableView:(NSTableView *)aTableView validateDrop:(id <NSDraggingInfo>)info proposedRow:(int)row proposedDropOperation:(NSTableViewDropOperation)operation
{
	if (![[[info draggingPasteboard] types] containsObject:@"PXLayerRowPasteType"]) { return NSDragOperationNone; }
	if (operation == NSTableViewDropOn) { [aTableView setDropRow:row dropOperation:NSTableViewDropAbove]; }
	return NSDragOperationMove;
}

- (BOOL)tableView:(NSTableView *)aTableView acceptDrop:(id <NSDraggingInfo>)info row:(int)row dropOperation:(NSTableViewDropOperation)operation
{
	[[self undoManager] beginUndoGrouping];
	[[self undoManager] setActionName:@"Reorder Layer"];
	[self setLayers:[[canvas layers] deepMutableCopy] fromLayers:[canvas layers] resetSelection:YES];
	if (row == [[canvas layers] count]) { row--; }
	[canvas moveLayer:[[canvas layers] objectAtIndex:[[[info draggingPasteboard] stringForType:@"PXLayerRowPasteType"] intValue]] toIndex:[self invertLayerIndex:row]];
	//[canvas moveLayer:[[info draggingPasteboard] dataForType:@"PXLayerRowPasteType"] toIndex:[self invertLayerIndex:row]];
	[self selectRow:[self invertLayerIndex:[[canvas layers] indexOfObject:[canvas activeLayer]]]];
	[[self undoManager] endUndoGrouping];
	return YES;
}

@end
