//  PXImage.m
//  Pixen
//
//  Created by Joe Osborn on Thu Sep 11 2003.
//  Copyright (c) 2003 Open Sword Group. All rights reserved.
//

#import "PXImage.h"
#import "PXPixel.h"
#import "KTMutableMatrix.h"

@implementation PXImage

- initWithSize:(NSSize)aSize
{
    [super init];
    size = aSize;
    pixelsByColor = [[NSMutableDictionary alloc] initWithCapacity:128];
    pixelsByPosition = [[KTMutableMatrix matrixWithCapacity:512*512 cuboidBounds:(unsigned)(size.width), (unsigned)(size.height), 0, 0] retain];
	rects = malloc(sizeof(NSRect) * (aSize.height + 2));
	colors = malloc(sizeof(NSColor *) * (aSize.height + 2));
    return self;
}

- (NSSize)size
{
	return size;
}

- (void)setSize:(NSSize)newSize withOrigin:(NSPoint)origin backgroundColor:(NSColor *)color
{
	id newPixelsByPosition = [[KTMutableMatrix matrixWithCapacity:512*512 cuboidBounds:(unsigned)(newSize.width), (unsigned)(newSize.height), 0, 0] retain];
	int i, j;
	for(i = 0; i < newSize.width; i++)
	{
		for(j = 0; j < newSize.height; j++)
		{
			if (i < origin.x || j < origin.y || i >= size.width + origin.x || j >= size.height + origin.y) {
				[newPixelsByPosition setObject:[PXPixel withColor:color] atCoordinates:(unsigned)(i), (unsigned)(j)];
			} else {
				[newPixelsByPosition setObject:[self pixelAtPoint:NSMakePoint(i - origin.x, j - origin.y)] atCoordinates:(unsigned)(i), (unsigned)(j)];
			}
		}
	}
	size = newSize;
	[pixelsByPosition release];
	pixelsByPosition = newPixelsByPosition;
	free(rects);
	free(colors);
	rects = malloc(sizeof(NSRect) * (size.height + 2));
	colors = malloc(sizeof(NSColor *) * (size.height + 2));
}

- (void)setSize:(NSSize)newSize
{
	[self setSize:newSize withOrigin:NSMakePoint(0,0) backgroundColor:[NSColor clearColor]];
}

- (void)dealloc
{
    [pixelsByColor release];
    [pixelsByPosition release];
	free(rects);
	free(colors);
    [super dealloc];
}

- (BOOL)containsPoint:(NSPoint)point
{
	return ([[NSUserDefaults standardUserDefaults] boolForKey:@"PXShouldTile"] ? YES : NSPointInRect(point, NSMakeRect(0, 0, size.width, size.height)));
}

- (NSColor *)colorAtPoint:(NSPoint)aPoint
{
	//if(![self containsPoint:aPoint]) { return nil; }
	return [[self pixelAtPoint:aPoint] color];
}

- pixelOfColor:aColor
{
	if(aColor == nil) { return nil; }
    id pixel = [pixelsByColor objectForKey:aColor];
    if(pixel == nil)
	{
		[pixelsByColor setObject:[PXPixel withColor:aColor] forKey:aColor]; 
		pixel = [pixelsByColor objectForKey:aColor]; 
		[[NSNotificationCenter defaultCenter] postNotificationName:@"PXImageColorAddedNotification" object:self userInfo:[NSDictionary dictionaryWithObject:aColor forKey:@"color"]];
	}
    return pixel;
}

- (NSPoint)correct:(NSPoint)aPoint
{
	NSPoint corrected = aPoint;
	while(corrected.x < 0)
	{
		corrected.x += [self size].width;
	}
	while(corrected.x >= [self size].width)
	{
		corrected.x -= [self size].width;
	}
	while(corrected.y < 0)
	{
		corrected.y += [self size].height;
	}
	while(corrected.y >= [self size].height)
	{
		corrected.y -= [self size].height;
	}
	return corrected;	
}

- pixelAtPoint:(NSPoint)aPoint
{
	NSPoint corrected = [self correct:aPoint];
    return [pixelsByPosition objectAtCoordinates:(unsigned)(corrected.x), (unsigned)(corrected.y)];
}

- (void)setPixel:aPixel atPoint:(NSPoint)aPoint
{
	NSPoint corrected = [self correct:aPoint];
    [pixelsByPosition setObject:aPixel atCoordinates:(unsigned)(corrected.x), (unsigned)(corrected.y)];
}

- (void)setPixel:aPixel atPoints:(NSArray *)points
{
    id enumerator = [points objectEnumerator];
    id current;
    while(current = [enumerator nextObject])
    {
        [self setPixel:aPixel atPoint:[current pointValue]];
    }
}

- (void)setColor:aColor atPoint:(NSPoint)aPoint
{
    [self setPixel:[self pixelOfColor:aColor] atPoint:aPoint];
}

- (void)setColor:(NSColor *)aColor atPoints:(NSArray *)points
{
    [self setPixel:[self pixelOfColor:aColor] atPoints:points];
}

- (void)replacePixelsOfColor:oldColor withColor:newColor
{
	NSPoint currentPoint;
	id currentColor;
	int i, j;
	for(i = 0; i < size.width; i++)
	{
		for(j = 0; j < size.height; j++)
		{
			currentPoint = NSMakePoint(i, j);
			currentColor = [self colorAtPoint:currentPoint];
			if([currentColor isEqual:oldColor] || (currentColor == oldColor))
			{
				[self setColor:newColor atPoint:currentPoint];
			}
		}
	}
}

- (void)translateXBy:(float)amountX yBy:(float)amountY
{
	id newMatrix = [KTMutableMatrix matrixWithCapacity:512*512 cuboidBounds:(unsigned)(size.width), (unsigned)(size.height), 0, 0];
	int i, j;
	for(i = 0; i < size.width; i++)
	{
		for(j = 0; j < size.height; j++)
		{
			float newX = i + amountX, newY = j + amountY;
			if(!(newX < 0 || newX >= size.width || newY < 0 || newY >= size.height))
			{
				[newMatrix setObject:[self pixelAtPoint:NSMakePoint(i, j)] atCoordinates:(unsigned)newX, (unsigned)newY];
			}
		}
	}
	[pixelsByPosition release];
	pixelsByPosition = [newMatrix retain];
}


- (void)flipHorizontally
{
	id newMatrix = [KTMutableMatrix matrixWithCapacity:512*512 cuboidBounds:(unsigned)(size.width), (unsigned)(size.height), 0, 0];
	int i, j;
	for(i = 0; i < size.width; i++)
	{
		for(j = 0; j < size.height; j++)
		{
			[newMatrix setObject:[self pixelAtPoint:NSMakePoint(size.width - i - 1, j)] atCoordinates:(unsigned)i, (unsigned)j];
		}
	}
	[pixelsByPosition release];
	pixelsByPosition = [newMatrix retain];
}

- (void)flipVertically
{
	id newMatrix = [KTMutableMatrix matrixWithCapacity:512*512 cuboidBounds:(unsigned)(size.width), (unsigned)(size.height), 0, 0];
	int i, j;
	for(i = 0; i < size.width; i++)
	{
		for(j = 0; j < size.height; j++)
		{
			[newMatrix setObject:[self pixelAtPoint:NSMakePoint(i, size.height - j - 1)] atCoordinates:(unsigned)i, (unsigned)j];
		}
	}
	[pixelsByPosition release];
	pixelsByPosition = [newMatrix retain];
}

#define PIXEN_FAST

- (void)drawRect:(NSRect)rect withOpacity:(double)anOpacity fixBug:(BOOL)fixBug
{
	NSRect tolerantRect = NSMakeRect(MAX(rect.origin.x-1, 0), MAX(rect.origin.y-1, 0), rect.size.width+1, rect.size.height+1);
    float i, j;
    for(i = tolerantRect.origin.x; (i < tolerantRect.origin.x+tolerantRect.size.width) && (i < size.width); i++)
    {
		int count = 0;
        for(j = tolerantRect.origin.y; (j < tolerantRect.origin.y+tolerantRect.size.height) && (j < size.height); j++)
        {
#ifdef PIXEN_FAST
			rects[count] = NSMakeRect(i,j,1,1);
			colors[count] = [self colorAtPoint:NSMakePoint(i,j)];
			if (colors[count] == nil)
			{
				colors[count] = [NSColor clearColor];
			}
			if (anOpacity != 1)
			{
				colors[count] = [colors[count] colorWithAlphaComponent:[colors[count] alphaComponent] * anOpacity];
			}
			count++;
#else
            PXPixel * pixel = [self pixelAtPoint:NSMakePoint(i,j)];
            if(pixel != nil) { [pixel drawAtPoint:NSMakePoint(i,j) withOpacity:anOpacity]; }
#endif /* PIXEN_FAST */
        }
#ifdef PIXEN_FAST
		NSRectFillListWithColorsUsingOperation(rects, colors, count, fixBug ? NSCompositeDestinationOver : NSCompositeSourceAtop);
#endif /* PIXEN_FAST */
    }
}

- description
{
	int i, j;
	NSString * result = @"";
	for (i = 0; i < size.width; i++)
	{
		for (j = 0; j < size.height; j++)
		{
			result = [result stringByAppendingFormat:@"(%d,%d): %@, ", i, j, [self colorAtPoint:NSMakePoint(i,j)]];
		}
	}
	return result;
}

- (void)compositeUnder:anImage
{
	NSImage * compositedImage = [[[NSImage alloc] initWithSize:size] autorelease];
	NSRect fullRect = NSMakeRect(0,0,size.width,size.height);
	NSPoint point;
	int i, j;
	id pool;
	
	[compositedImage lockFocus];
	[anImage drawRect:fullRect withOpacity:1 fixBug:YES];
	[self drawRect:fullRect withOpacity:1 fixBug:YES];
    for(i = 0; i < size.width; i++)
    {
        pool = [[NSAutoreleasePool alloc] init];
        for(j = 0; j < size.height; j++)
        {
			point = NSMakePoint(i, j);
            [self setColor:NSReadPixel(point) atPoint:point];
        }
        [pool release];
    }	
	[compositedImage unlockFocus];
}

- (void)setPixelsByColor:newPixelsByColor
{
	[newPixelsByColor retain];
	[pixelsByColor release];
	pixelsByColor = newPixelsByColor;
}

- (void)setPixelsByPosition:newPixelsByPosition
{
	[newPixelsByPosition retain];
	[pixelsByPosition release];
	pixelsByPosition = newPixelsByPosition;
}

- copyWithZone:(NSZone *)zone
{
	id new = [[[self class] allocWithZone:zone] initWithSize:size];
	/*we're not using this approach because there seems to be a weirdness in -[NSMutableDictionary mutableCopy] that makes the mutably copied dictionary worthless-- it doesn't retain its objects!  At least, there were crashes when we used that method that didn't show up when we did -(void)setColor:atPoint: for each pixel.
	[new setPixelsByColor:[[pixelsByColor mutableCopyWithZone:zone] autorelease]];
	[new setPixelsByPosition:[[pixelsByPosition mutableCopyWithZone:zone] autorelease]];
*/
	int i, j;
	for(i = 0; i < [self size].width; i++)
	{
		for(j = 0; j < [self size].height; j++)
		{
			[new setColor:[self colorAtPoint:NSMakePoint(i, j)] atPoint:NSMakePoint(i, j)];
		}
	}
	return new;
}

@end


@implementation PXImage(Archiving)

NSString * PXCurrentVersion = @"r2v2";

- legacyDiscoverPixelsByPositionFromPositionsByPixel:positionsByPixel
{
    id newPixelsByPosition = [NSMutableDictionary dictionaryWithCapacity:10000];
    id pixelEnumerator = [positionsByPixel keyEnumerator];
    id currentPixel;
    while(currentPixel = [pixelEnumerator nextObject])
    {
        id positionEnumerator = [[positionsByPixel objectForKey:currentPixel] objectEnumerator];
        id currentPosition;
        while(currentPosition = [positionEnumerator nextObject])
        {
            [newPixelsByPosition setObject:currentPixel forKey:currentPosition];
        }
    }
    return newPixelsByPosition;
}

- (NSSize)legacyDiscoverSizeFromPixelsByPosition:pixels
{
    float width = 0, height = 0;
    id enumerator = [pixels keyEnumerator];
    id current;
    while(current = [enumerator nextObject])
    {
        NSPoint point = NSPointFromString(current);
        if(point.x > width) { width = point.x; }
        if(point.y > height) { height = point.y; }
    }
    return NSMakeSize(width+1, height+1);
}

- legacyDiscoverPixelsByPositionMatrixFromPixelsByPositionDictionary:pixels
{
    id realPixelsByPosition = [KTMutableMatrix matrixWithCapacity:512*512 cuboidBounds:(unsigned)(size.width), (unsigned)(size.height), 0, 0];
	int i, j;
	for(i = 0; i < size.width; i++)
	{
		for(j = 0; j < size.height; j++)
		{
			NSPoint point = NSMakePoint(i, j);
			id pixel = [pixels objectForKey:NSStringFromPoint(point)];
			//if(pixel == nil) { pixel = [self pixelOfColor:[NSColor clearColor]]; }
			[realPixelsByPosition setObject:[pixel retain] atCoordinates:(unsigned)(point.x), (unsigned)(point.y)];
		}
	}
    return realPixelsByPosition;
}

- legacyInitWithCoder:coder
{
    id positionsByPixel = [coder decodeObjectForKey:@"positionsByPixel"];
    if(positionsByPixel != nil) { pixelsByPosition = [self legacyDiscoverPixelsByPositionFromPositionsByPixel:positionsByPixel]; }
    if(NSEqualSizes(size, NSZeroSize)) { size = [self legacyDiscoverSizeFromPixelsByPosition:pixelsByPosition]; }
    pixelsByPosition = [[self legacyDiscoverPixelsByPositionMatrixFromPixelsByPositionDictionary:pixelsByPosition] retain];
    return self;
}

- initWithCoder:coder
{
    [super init];
    pixelsByColor = [[coder decodeObjectForKey:@"pixelsByColor"] mutableCopy];
    pixelsByPosition = [[coder decodeObjectForKey:@"pixelsByPosition"] mutableCopy];
    size = [coder decodeSizeForKey:@"size"];
    //if(![PXCurrentVersion isEqual:[coder decodeObjectForKey:@"version"]]) 
	{ 
		[self legacyInitWithCoder:coder]; 
	}
	rects = malloc(sizeof(NSRect) * (size.height + 2));
	colors = malloc(sizeof(NSColor *) * (size.height + 2));
    return self;
}

- (void)encodeWithCoder:coder
{
    //can't encode version until KTMatrix works properly.  we're still faking encoding in the old format.  sigh.
    [coder encodeObject:PXCurrentVersion forKey:@"version"];
    [coder encodeSize:size forKey:@"size"];
    [coder encodeObject:pixelsByColor forKey:@"pixelsByColor"];
    //[coder encodeObject:pixelsByPosition forKey:@"pixelsByPosition"];
    id fakePixelsByPosition = [NSMutableDictionary dictionaryWithCapacity:50000];
    unsigned i, j;
    for(i = 0; i < size.width; i++)
    {
        for(j = 0; j < size.height; j++)
        {
            id pixel = [pixelsByPosition objectAtCoordinates:i, j];
            if(pixel != nil) { [fakePixelsByPosition setObject:pixel forKey:NSStringFromPoint(NSMakePoint(i, j))]; }
        }
    }
    [coder encodeObject:fakePixelsByPosition forKey:@"pixelsByPosition"];
}

@end
