//
//  PXCanvasResizeView.m
//  Pixen-XCode
//
//  Created by Ian Henderson on Wed Jun 09 2004.
//  Copyright (c) 2004 Open Sword Group. All rights reserved.
//

#import "PXCanvasResizeView.h"
#import <math.h>

@implementation PXCanvasResizeView

- (id)initWithFrame:(NSRect)frame
{
    self = [super initWithFrame:frame];
    if (self) {
		cachedImage = [[NSImage imageNamed:@"greybox"] retain];
		scaleTransform = [[NSAffineTransform alloc] init];
		backgroundColor = [[NSColor colorWithCalibratedRed:1 green:1 blue:1 alpha:0] retain];
    }
    return self;
}

- (void)dealloc
{
	[backgroundColor release];
	[scaleTransform release];
	[cachedImage release];
	[super dealloc];
}


- (NSRect)applyTransformation:(NSAffineTransform *)transform toRect:(NSRect)rect
{
	NSRect newRect;
	newRect.size = [transform transformSize:rect.size];
	newRect.origin = [transform transformPoint:rect.origin];
	return newRect;
}

- (void)drawLineAndNumberLengthFromPoint:(NSPoint)from toPoint:(NSPoint)to scale:(float)scale;
{
	float distance = sqrt(pow(from.x - to.x, 2) + pow(from.y - to.y, 2)) / scale;
	[NSBezierPath strokeLineFromPoint:from toPoint:to];
	[[NSString stringWithFormat:@"%d", (int)distance] drawAtPoint:NSMakePoint((from.x + to.x)/2 + .5, (from.y + to.y)/2) withAttributes:[NSDictionary dictionary]];
}

- (void)drawRect:(NSRect)rect
{
	NSRect newRect = NSMakeRect(0, 0, newSize.width, newSize.height); // Find the new size of the canvas
	NSRect oldRect = NSMakeRect(position.x, position.y, oldSize.width, oldSize.height); // Find the old size of the canvas
	NSSize maxSize = NSMakeSize(MAX(newSize.width, oldSize.width), MAX(newSize.height, oldSize.height)); // Find the size we need to display in the view
	NSSize frameSize = [self frame].size;
	
	float scale = 1.0f / MAX(maxSize.height / frameSize.height, maxSize.width / frameSize.width); // Find the scaling factor by looking at the rect that contains both the new size and old size, then scaling it to fit our frame
	
	oldRect.origin.x = round(oldRect.origin.x);
	oldRect.origin.y = round(oldRect.origin.y);
	
	[scaleTransform release];
	scaleTransform = [[NSAffineTransform transform] retain]; // transform the image-pixel scale to screen-pixel scale
	[scaleTransform scaleBy:scale];
	
	newRect = [self applyTransformation:scaleTransform toRect:newRect]; // transform our rects
	oldRect = [self applyTransformation:scaleTransform toRect:oldRect];
	
	NSAffineTransform *translateTransform = [NSAffineTransform transform];
	[translateTransform translateXBy:(frameSize.width - newRect.size.width) / 2 yBy:(frameSize.height - newRect.size.height) / 2]; // center the view on the new frame
	
	newRect = [self applyTransformation:translateTransform toRect:newRect]; // transform the rects again
	oldRect = [self applyTransformation:translateTransform toRect:oldRect];
	
	[[backgroundColor colorWithAlphaComponent:1] set];
	NSRectFill(newRect); // draw background for new frame
	
	[cachedImage drawInRect:oldRect fromRect:NSMakeRect(0, 0, [cachedImage size].width, [cachedImage size].height) operation:NSCompositeSourceAtop fraction:1.0f]; // draw the image in the old frame
	[[NSColor blackColor] set];
	[NSBezierPath strokeRect:oldRect]; // draw an outline around the image
	
	NSBezierPath *canvasOutline = [NSBezierPath bezierPathWithRect:newRect];
	float dashed[2] = {5, 5};
	[canvasOutline setLineDash:dashed count:2 phase:0];
	[canvasOutline stroke]; // draw an outline around the canvas
	
	if (drawingArrows) { // if we're dragging the image
		[[NSColor redColor] set];
		
		float horizontalOffset = (oldRect.size.width / 2) + oldRect.origin.x;
		float verticalOffset = (oldRect.size.height / 2) + oldRect.origin.y;
		BOOL drawHorizontal = YES;
		BOOL drawVertical = YES;
		
		// a whole bunch of checks to position the lines correctly follows...
		
		if (horizontalOffset > newRect.size.width + newRect.origin.x) {
			if (newRect.size.width + newRect.origin.x < oldRect.origin.x) {
				drawVertical = NO; // we don't care about it, it's off the new image entirely
			} else {
				horizontalOffset = newRect.size.width + newRect.origin.x; // move line so it doesn't go off the edge
			}
		}
		
		if (horizontalOffset < newRect.origin.x) { // pretty much the same as above
			if (newRect.origin.x > oldRect.size.width + oldRect.origin.x) {
				drawVertical = NO;
			} else {
				horizontalOffset = newRect.origin.x;
			}
		}
		
		if (verticalOffset > newRect.size.height + newRect.origin.y) {
			if (newRect.size.height + newRect.origin.y < oldRect.origin.y) {
				drawHorizontal = NO;
			} else {
				verticalOffset = newRect.size.height + newRect.origin.y; // move line so it doesn't go off the edge
			}
		}
		
		if (verticalOffset < newRect.origin.y) { // pretty much the same as above
			if (newRect.origin.y > oldRect.size.height + oldRect.origin.y) {
				drawHorizontal = NO;
			} else {
				verticalOffset = newRect.origin.y;
			}
		}
		
		if (drawVertical) {
			// up
			[self drawLineAndNumberLengthFromPoint:NSMakePoint(horizontalOffset, oldRect.size.height + oldRect.origin.y)
										   toPoint:NSMakePoint(horizontalOffset, newRect.size.height + newRect.origin.y)
											 scale:scale];
			
			// down
			[self drawLineAndNumberLengthFromPoint:NSMakePoint(horizontalOffset, oldRect.origin.y)
										   toPoint:NSMakePoint(horizontalOffset, newRect.origin.y)
											 scale:scale];
		}
		
		if (drawHorizontal) {
			// right
			[self drawLineAndNumberLengthFromPoint:NSMakePoint(oldRect.size.width + oldRect.origin.x, verticalOffset)
										   toPoint:NSMakePoint(newRect.size.width + newRect.origin.x, verticalOffset)
											 scale:scale];
			
			// left
			[self drawLineAndNumberLengthFromPoint:NSMakePoint(oldRect.origin.x, verticalOffset)
										   toPoint:NSMakePoint(newRect.origin.x, verticalOffset)
											 scale:scale];
		}
	}
}

- (NSSize)newSize
{
	return newSize;
}

- (NSPoint)position
{
	NSPoint roundedPosition = position;
	roundedPosition.x = round(roundedPosition.x);
	roundedPosition.y = round(roundedPosition.y);
	return roundedPosition;
}

- (void)setNewImageSize:(NSSize)size
{
	newSize = size;
	[self setNeedsDisplay:YES];
}

- (void)setOldImageSize:(NSSize)size
{
	oldSize = size;
	position = NSMakePoint(0,0);
	[self setNeedsDisplay:YES];
}

- (void)setCachedImage:(NSImage *)image
{
	[cachedImage release];
	cachedImage = [[NSImage alloc] initWithSize:[image size]];
	
	[cachedImage lockFocus];
	[[NSColor colorWithCalibratedRed:1 green:1 blue:1 alpha:1] set];
	NSRectFill(NSMakeRect(0,0,[image size].width,[image size].height));
	[image compositeToPoint:NSMakePoint(0, 0) operation:NSCompositeSourceAtop];
	[cachedImage unlockFocus];
	
	[self setNeedsDisplay:YES];
}


- (void)setBackgroundColor:(NSColor *)color
{
	[color retain];
	[backgroundColor release];
	backgroundColor = color;
	[self setNeedsDisplay:YES];
}

- (void)mouseDown:(NSEvent *)event
{
	drawingArrows = YES;
}

- (void)mouseUp:(NSEvent *)event
{
	drawingArrows = NO;
	[self setNeedsDisplay:YES];
}

- (void)mouseDragged:(NSEvent *)event
{
	NSAffineTransform *muffineTransform = [scaleTransform copy];
	[muffineTransform invert];
	NSPoint deltaVector = [muffineTransform transformPoint:NSMakePoint([event deltaX], [event deltaY])];
	
	position.x += deltaVector.x;
	position.y -= deltaVector.y;
	[self setNeedsDisplay:YES];
}

- (void)hideArrows
{
	drawingArrows = NO;
	[self setNeedsDisplay:YES];
}

- (void)keyDown:(NSEvent *)event
{
	NSString *characters = [event charactersIgnoringModifiers];
	if ([characters characterAtIndex:0] == NSUpArrowFunctionKey) {
		position.y += 1;
	} else if ([characters characterAtIndex:0] == NSDownArrowFunctionKey) {
		position.y -= 1;
	} else if ([characters characterAtIndex:0] == NSRightArrowFunctionKey) {
		position.x += 1;
	} else if ([characters characterAtIndex:0] == NSLeftArrowFunctionKey) {
		position.x -= 1;
	}
	drawingArrows = YES;
	[self setNeedsDisplay:YES];
}

- (void)keyUp:(NSEvent *)event
{
	[self performSelector:@selector(hideArrows) withObject:nil afterDelay:1]; // perhaps a preference setting for the delay?
}

- (BOOL)acceptsFirstResponder
{
	return YES;
}

@end
