//  PXCanvasController.m
//  Pixen
//
//  Created by Joe Osborn on Sat Sep 13 2003.
//  Copyright (c) 2003 Open Sword Group. All rights reserved.
//

#import "PXCanvasController.h"
#import "PXCanvas.h"
#import "PXCanvasView.h"
#import "PXImage.h"
#import "PXImageSizePrompter.h"
#import "PXCanvasResizePrompter.h"
#import "PXGridSettingsPrompter.h"
#import "PXTool.h"
#import "PXToolPaletteController.h"
#import "PXToolPropertiesController.h"
#import "PXColorPaletteController.h"
#import "PXBackgroundController.h"
#import "PXPreviewController.h"
#import "PXLayerController.h"
#import "PXGrid.h"
#import "PXColorPaletteController.h"
#import "PXScaleController.h"
#import "MyDocument.h"

//Taken from a man calling himself "BROCK BRANDENBERG" who is here to save the day.
#import "SBCenteringClipView.h"

//Taken from a man calling himself "M. Uli Kusterer", who is totally not saving the day adequately (but we love him anyway).
#import "UKFeedbackProvider.h"

@implementation PXCanvasController

- view
{
    return view;
}

- init
{
    [super initWithWindowNibName:@"MyDocument"];
    prompter = [[PXImageSizePrompter alloc] init];
    resizePrompter = [[PXCanvasResizePrompter alloc] init];
    previewController = [PXPreviewController preview];
    backgroundController = [[PXBackgroundController alloc] init];
	scaleController = [[PXScaleController alloc] init];
    [backgroundController setDelegate:self];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(canvasSizeDidChange:) name:PXCanvasSizeChangedNotificationName object:nil];
    return self;
}

- (void)awakeFromNib
{
	toolbar = [[NSToolbar alloc] initWithIdentifier:@"PXDocumentToolbar"];
    [toolbar setDelegate:self];
    [toolbar setAllowsUserCustomization:YES];
	[toolbar setAutosavesConfiguration:YES];
    [[self window] setToolbar:toolbar];
	
	[[self window] setAcceptsMouseMovedEvents:YES];
}

- (void)dealloc
{
	[[NSNotificationCenter defaultCenter] removeObserver:self];
    if([[backgroundController window] isVisible]) { [[backgroundController window] performClose:self]; }
    [backgroundController release];
    if([[prompter window] isVisible]) { [prompter close]; }
    [prompter release];
	[resizePrompter release];
    [layerController release];
	[toolbar release];
    [super dealloc];
}

- (IBAction)increaseOpacity:sender
{
	id switcher = [[PXToolPaletteController palette] leftSwitcher];
	[switcher setColor:[[switcher color] colorWithAlphaComponent:[[switcher color] alphaComponent] + 0.1f]];
}

- (IBAction)decreaseOpacity:sender
{
	id switcher = [[PXToolPaletteController palette] leftSwitcher];
	[switcher setColor:[[switcher color] colorWithAlphaComponent:[[switcher color] alphaComponent] - 0.1f]];
}

- (IBAction)duplicateDocument:sender
{
	id newDocument = [[NSDocumentController sharedDocumentController] makeUntitledDocumentOfType:@"Pixen Image"];
	id newCanvas = [canvas copy];
	[newDocument setValue:newCanvas forKey:@"canvas"];
	[newDocument makeWindowControllers];
	[[NSDocumentController sharedDocumentController] addDocument:newDocument];
}

- (NSToolbarItem *)toolbar:(NSToolbar *)toolbar itemForItemIdentifier:(NSString *)itemIdentifier willBeInsertedIntoToolbar:(BOOL)flag
{
	NSToolbarItem *item = [[[NSToolbarItem alloc] initWithItemIdentifier:itemIdentifier] autorelease];
	if ([itemIdentifier isEqualToString:@"PXBackgroundConfigurator"])
	{
		[item setLabel:@"Background"];
		[item setPaletteLabel:[item label]];
		[item setToolTip:@"Open background configurator"];
		[item setAction:@selector(showBackgroundInfo:)];
		[item setImage:[NSImage imageNamed:@"bgconf"]];
	}
	else if ([itemIdentifier isEqualToString:@"PXLayerDrawer"])
	{
		[item setLabel:@"Layers"];
		[item setPaletteLabel:[item label]];
		[item setToolTip:@"Open and close the layer view drawer"];
		[item setAction:@selector(toggleLayersDrawer:)];
		[item setImage:[NSImage imageNamed:@"layerdrawer"]];
	}
	else if ([itemIdentifier isEqualToString:@"PXPreview"])
	{
		[item setLabel:@"Preview"];
		[item setPaletteLabel:[item label]];
		[item setToolTip:@"Open and close the preview panel"];
		[item setAction:@selector(togglePreviewWindow:)];
		[item setImage:[NSImage imageNamed:@"preview"]];
	}
	else if ([itemIdentifier isEqualToString:@"PXToolProperties"])
	{
		[item setLabel:@"Tool Properties"];
		[item setPaletteLabel:[item label]];
		[item setToolTip:@"Open and close the tool properties pane"];
		[item setAction:@selector(toggleToolProperties:)];
		[item setImage:[NSImage imageNamed:@"toolproperties"]];
	}
	else if ([itemIdentifier isEqualToString:@"PXGrid"])
	{
		[item setLabel:@"Grid"];
		[item setPaletteLabel:[item label]];
		[item setToolTip:@"Show grid settings"];
		[item setAction:@selector(showGridSettingsPrompter:)];
		[item setImage:[NSImage imageNamed:@"grid"]];
	}
	else if ([itemIdentifier isEqualToString:@"PXColorPalette"])
	{
		[item setLabel:@"Palette"];
		[item setPaletteLabel:[item label]];
		[item setToolTip:@"Open and close the color palette"];
		[item setAction:@selector(toggleColorPalette:)];
		[item setImage:[NSImage imageNamed:@"colorpalette"]];
	}
	else if ([itemIdentifier isEqualToString:@"PXZoomFit"])
	{
		[item setLabel:@"Zoom Fit"];
		[item setPaletteLabel:[item label]];
		[item setToolTip:@"Zoom the canvas to fit the window"];
		[item setAction:@selector(zoomToFit:)];
		[item setImage:[NSImage imageNamed:@"zoomfit"]];
	}
	else if ([itemIdentifier isEqualToString:@"PXZoom100"])
	{
		[item setLabel:@"Actual Size"];
		[item setPaletteLabel:[item label]];
		[item setToolTip:@"Zoom to the actual size of the image"];
		[item setAction:@selector(zoomStandard:)];
		[item setImage:[NSImage imageNamed:@"zoom100"]];
	}
	else if ([itemIdentifier isEqualToString:@"PXScale"])
	{
		[item setLabel:@"Scale"];
		[item setPaletteLabel:[item label]];
		[item setToolTip:@"Scale the image"];
		[item setAction:@selector(scaleCanvas:)];
		[item setImage:[NSImage imageNamed:@"scale"]];
	}
	else if ([itemIdentifier isEqualToString:@"PXResize"])
	{
		[item setLabel:@"Resize"];
		[item setPaletteLabel:[item label]];
		[item setToolTip:@"Resize the canvas"];
		[item setAction:@selector(resizeCanvas:)];
		[item setImage:[NSImage imageNamed:@"resize"]];
	}
	else if ([itemIdentifier isEqualToString:@"PXFeedback"])
	{
		[item setLabel:@"Feedback"];
		[item setPaletteLabel:[item label]];
		[item setToolTip:@"Send feature requests, bug reports, and general feedback."];
		[item setAction:@selector(sendFeedback:)];
		[item setImage:[NSImage imageNamed:@"feedback"]];
	}
	else if ([itemIdentifier isEqualToString:@"PXZoom"])
	{
		[item setLabel:@"Zoom"];
		[item setPaletteLabel:[item label]];
		[item setToolTip:@"Modify the zoom percentage of the view"];
		[item setAction:@selector(togglePreviewWindow:)];
		[item setView:zoomView];
		[item setMinSize:NSMakeSize(124,NSHeight([zoomView frame]))];
		[item setMaxSize:NSMakeSize(0,NSHeight([zoomView frame]))];
	}
	return item;
}

- toolbarAllowedItemIdentifiers:toolbar
{
	return [NSArray arrayWithObjects:@"PXBackgroundConfigurator", @"PXLayerDrawer", @"PXPreview", @"PXZoom", @"PXZoomFit", @"PXZoom100", @"PXResize", @"PXScale", @"PXFeedback", @"PXGrid", @"PXColorPalette", NSToolbarCustomizeToolbarItemIdentifier, NSToolbarSpaceItemIdentifier, NSToolbarSeparatorItemIdentifier, NSToolbarFlexibleSpaceItemIdentifier, nil];
}

- toolbarDefaultItemIdentifiers:toolbar
{
	return [NSArray arrayWithObjects:@"PXBackgroundConfigurator", @"PXGrid", NSToolbarSeparatorItemIdentifier, @"PXLayerDrawer", @"PXColorPalette", @"PXPreview", NSToolbarFlexibleSpaceItemIdentifier, @"PXFeedback", @"PXZoom", nil];
}

- (IBAction)sendFeedback:sender
{
	id provider = [[UKFeedbackProvider alloc] init];
	[provider orderFrontFeedbackWindow:self];
}

- (IBAction)toggleToolProperties:sender
{
	if ([[[PXToolPropertiesController properties] window] isVisible])
	{
		[[[PXToolPropertiesController properties] window] performClose:self];
	}
	else
	{
		[[PXToolPropertiesController properties] showWindow:self];
	}
}

- (IBAction)toggleColorPalette:sender
{
	if ([[[PXColorPaletteController palette] window] isVisible])
	{
		[[[PXColorPaletteController palette] window] performClose:self];
	}
	else
	{
		[[PXColorPaletteController palette] showWindow:self];
	}
}

- (IBAction)mergeDown:sender
{
    [layerController mergeDown];
}

- (IBAction)promoteSelection:sender
{
    [canvas promoteSelection];
}

- (IBAction)newLayer:sender
{
    [layerController addLayer:sender];
}

- (IBAction)deleteLayer:sender
{
    [layerController removeLayer:sender];
}

- (void)layerSelectionDidChange:aNotification
{
    [canvas deselect];
    [canvas activateLayer:[aNotification userInfo]];
}

- (void)setMainBackground:aBackground
{
    [view setMainBackground:aBackground];
    [canvas setMainBackgroundName:[aBackground name]];
}

- (void)setAlternateBackground:aBackground
{
    [view setAlternateBackground:aBackground];
    [canvas setAlternateBackgroundName:[aBackground name]];
}

- (void)backgroundChanged:aNotification
{
    [view setNeedsDisplayInRect:[view visibleRect]];
}

- (IBAction)flipLayerHorizontally:sender
{
    [[self undoManager] beginUndoGrouping];
    [self setLayers:[[canvas layers] deepMutableCopy] fromLayers:[canvas layers]];		
    [[self undoManager] setActionName:@"Flip Layer Horizontally"];
    [[self undoManager] endUndoGrouping];
	[[canvas activeLayer] flipHorizontally];
    [canvas changedInRect:NSMakeRect(0, 0, [canvas size].width, [canvas size].height)];
}

- (IBAction)flipLayerVertically:sender
{
    [[self undoManager] beginUndoGrouping];
    [self setLayers:[[canvas layers] deepMutableCopy] fromLayers:[canvas layers]];		
    [[self undoManager] setActionName:@"Flip Layer Vertically"];
    [[self undoManager] endUndoGrouping];
	[[canvas activeLayer] flipVertically];
    [canvas changedInRect:NSMakeRect(0, 0, [canvas size].width, [canvas size].height)];
}

- (IBAction)duplicateLayer:sender
{
    [[self undoManager] beginUndoGrouping];
    [self setLayers:[[canvas layers] deepMutableCopy] fromLayers:[canvas layers]];		
    [[self undoManager] setActionName:@"Duplicate Layer"];
    [[self undoManager] endUndoGrouping];
	[canvas insertLayer:[[[canvas activeLayer] copy] autorelease] atIndex:[canvas indexOfLayer:[canvas activeLayer]]];
    [canvas changedInRect:NSMakeRect(0, 0, [canvas size].width, [canvas size].height)];
}

- (void)promptForImageSize
{
    [prompter setDelegate:self];
    [prompter promptInWindow:[self window]];
}

- canvas
{
    return canvas;
}

- (void)setCanvas:aCanvas
{
    [[NSNotificationCenter defaultCenter] removeObserver:self name:PXCanvasChangedNotificationName object:canvas];
    canvas = aCanvas;
    if(NSEqualSizes([canvas size], NSZeroSize))
    {
        [self promptForImageSize];
        [prompter setDefaultSize:NSMakeSize(64,64)];
    }
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(canvasDidChange:) name:PXCanvasChangedNotificationName object:canvas];
    [view setCanvas:aCanvas];
    [layerController setCanvas:aCanvas];
	[[PXColorPaletteController palette] reloadDataForCanvas:canvas];
    [[NSNotificationCenter defaultCenter] postNotificationName:PXCanvasLayersChangedNotificationName object:canvas];
    [canvas changedInRect:NSMakeRect(0, 0, [canvas size].width, [canvas size].height)];
}

- (void)setColor:aColor
{
    [[PXToolPaletteController palette] setColor:aColor];
}

- (void)canvasDidChange:aNotification
{
    NSRect rect = [[[aNotification userInfo] objectForKey:@"changedRect"] rectValue];
    [view setNeedsDisplayInCanvasRect:rect];
}

- (void)canvasSizeDidChange:aNotification
{
    [view sizeToCanvas];
    [self updatePreview];
    [self zoomToFit:self];
}

- (void)updatePreview
{
    [previewController setCanvas:canvas];
    [previewController window];
}

- undoManager
{
    return [[[NSDocumentController sharedDocumentController] currentDocument] undoManager];
}

- (void)setLayers:layers fromLayers:oldLayers
{
    [[[self undoManager] prepareWithInvocationTarget:self] setLayers:oldLayers fromLayers:layers];
    [canvas setLayers:layers];
    //beginHack: it is probably bad to rely on the fact that all layers are the same size for now.  However, it is late and I want this to work and this is unlikely to change, so I will ignore this for now. __joe
    [canvas setSize:[[layers objectAtIndex:0] size]];
    //endHack
    [self canvasSizeDidChange:nil];
}

- (IBAction)resizeCanvas:sender
{
    [[self undoManager] beginUndoGrouping];
    [self setLayers:[[canvas layers] deepMutableCopy] fromLayers:[canvas layers]];		
    [[self undoManager] setActionName:@"Resize Canvas"];
    [[self undoManager] endUndoGrouping];
    [resizePrompter setDelegate:self];
    [resizePrompter promptInWindow:[self window]];
    [resizePrompter setCurrentSize:[canvas size]];
	
	NSImage *canvasImage = [[[NSImage alloc] initWithSize:[canvas size]] autorelease];
	[canvasImage lockFocus];
	[canvas drawRect:NSMakeRect(0, 0, [canvas size].width, [canvas size].height) fixBug:YES];
	[canvasImage unlockFocus];
	[resizePrompter setCachedImage:canvasImage];
}

- (IBAction)scaleCanvas:sender
{
    [[self undoManager] beginUndoGrouping];
    [self setLayers:[[canvas layers] deepMutableCopy] fromLayers:[canvas layers]];		
    [[self undoManager] setActionName:@"Scale Canvas"];
    [[self undoManager] endUndoGrouping];
    [scaleController scaleCanvasFromController:self modalForWindow:[self window]];
}

- (void)windowDidBecomeMain:aNotification
{
	if([aNotification object] == [self window])
	{
		[self updatePreview];
		[[PXColorPaletteController palette] reloadDataForCanvas:canvas];
	}
}

- (void)windowDidLoad
{
    NSAssert(canvas, @"We should have a canvas by now!");
    [view setDelegate:self];
    [zoomPercentageBox removeAllItems];
    [zoomPercentageBox addItemsWithObjectValues:[NSArray arrayWithObjects:[NSNumber numberWithInt:3000], [NSNumber numberWithInt:2000], [NSNumber numberWithInt:1000], [NSNumber numberWithInt:800], [NSNumber numberWithInt:500], [NSNumber numberWithInt:400], [NSNumber numberWithInt:200], [NSNumber numberWithInt:100], [NSNumber numberWithInt:50], nil]];
    [zoomPercentageBox selectItemAtIndex:7];
    [zoomStepper setIntValue:7];
    id clip = [[[SBCenteringClipView alloc] initWithFrame:[[scrollView contentView] frame]] autorelease];
    [clip setBackgroundColor:[NSColor lightGrayColor]];
    [clip setCopiesOnScroll:NO];
    [(NSScrollView *)scrollView setContentView:clip];
    [scrollView setDocumentView:view];
    [view setCanvas:canvas];
    [backgroundController useBackgroundsOf:canvas];
    //[clip centerDocument];
    layerController = [[PXLayerController alloc] initWithCanvas:canvas];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(layerSelectionDidChange:) name:PXLayerSelectionDidChangeName object:layerController];
    [layerController setWindow:[self window]];
    [[NSNotificationCenter defaultCenter] postNotificationName:PXCanvasLayersChangedNotificationName object:canvas];
    [self zoomToFit:self];
	[[PXColorPaletteController palette] reloadDataForCanvas:canvas];
	[[PXColorPaletteController palette] selectPaletteNamed:@"Generated Palette"];
    //[canvas changedInRect:NSMakeRect(0, 0, [canvas size].width, [canvas size].height)];
    //[self updatePreview];
}

- (void)updateCanvasSize
{
    [view sizeToCanvas];
    [self updatePreview];
    [self zoomToFit:self];
}

- (void)prompter:aPrompter didFinishWithSize:(NSSize)aSize
{
    [canvas setSize:aSize];
	[self updateCanvasSize];
	[[PXColorPaletteController palette] reloadDataForCanvas:canvas];
	[[PXColorPaletteController palette] selectPaletteNamed:@"Default"];
	[(PXImage *)[[[canvas layers] objectAtIndex:0] image] replacePixelsOfColor:nil withColor:[NSColor clearColor]];
}

- (void)prompter:aPrompter didFinishWithSize:(NSSize)aSize position:(NSPoint)position backgroundColor:(NSColor *)color
{
    [canvas setSize:aSize withOrigin:position backgroundColor:color];
	[self updateCanvasSize];
}

- (IBAction)shouldTileToggled:sender
{
	[sender setState:([sender state] == NSOnState) ? NSOffState : NSOnState];
	id defaults = [NSUserDefaults standardUserDefaults];
	[defaults setBool:([sender state] == NSOnState) forKey:@"PXShouldTile"];
	[defaults synchronize];
	[view setShouldTile:([sender state] == NSOnState)];
}

- (void)gridSettingsPrompter:aPrompter updatedWithSize:(NSSize)aSize color:color shouldDraw:(BOOL)shouldDraw
{
    [[view grid] setUnitSize:aSize];
    [[view grid] setColor:color];
    [[view grid] setShouldDraw:shouldDraw];
    [canvas setGridUnitSize:aSize];
    [canvas setGridColor:color];
    [canvas setGridShouldDraw:shouldDraw];
    [canvas changedInRect:NSMakeRect(0, 0, [canvas size].width, [canvas size].height)];
}

- (void)mouseDown:event forTool:aTool
{
    downEventOccurred = YES;
    [[NSNotificationCenter defaultCenter] postNotificationName:@"PXLockToolSwitcher" object:aTool];
    if(![aTool respondsToSelector:@selector(mouseDownAt:fromCanvasController:)]) { return; }
    initialPoint = [event locationInWindow];
    [aTool mouseDownAt:[view convertFromWindowToCanvasPoint:initialPoint] fromCanvasController:self];    
}

- (void)mouseDragged:event forTool:aTool
{
    if(!downEventOccurred) { return; }
    if(![aTool respondsToSelector:@selector(mouseDraggedFrom:to:fromCanvasController:)]) { return; }
    NSPoint endPoint = [event locationInWindow];
    [aTool mouseDraggedFrom:[view convertFromWindowToCanvasPoint:initialPoint] to:[view convertFromWindowToCanvasPoint:endPoint] fromCanvasController:self];
    initialPoint = endPoint;
}

- (void)mouseUp:event forTool:aTool
{
    downEventOccurred = NO;
    [[NSNotificationCenter defaultCenter] postNotificationName:@"PXUnlockToolSwitcher" object:aTool];
    if(![aTool respondsToSelector:@selector(mouseUpAt:fromCanvasController:)]) { return; }
    [aTool mouseUpAt:[view convertFromWindowToCanvasPoint:[event locationInWindow]] fromCanvasController:self];
}

- (void)mouseDown:(NSEvent *)event
{
    //users expect ctrl-left to == right
    usingControlKey = (([event modifierFlags] & NSControlKeyMask) == NSControlKeyMask);
    
    if (usingControlKey)
    {
		[self rightMouseDown:event]; return;
    }
    [self mouseDown:event forTool:[[PXToolPaletteController palette] leftTool]];
}

- (void)mouseDragged:event
{
    if (usingControlKey)
    {
		[self rightMouseDragged:event]; return;
    }
    [self mouseDragged:event forTool:[[PXToolPaletteController palette] leftTool]];
}

- (void)mouseUp:event
{
    if (usingControlKey)
    {
		[self rightMouseUp:event]; return;
    }
    [self mouseUp:event forTool:[[PXToolPaletteController palette] leftTool]];
}

- (void)rightMouseDown:event
{
    [self mouseDown:event forTool:[[PXToolPaletteController palette] rightTool]];
}

- (void)rightMouseDragged:event
{
    [self mouseDragged:event forTool:[[PXToolPaletteController palette] rightTool]];
}

- (void)rightMouseUp:event
{
    [self mouseUp:event forTool:[[PXToolPaletteController palette] rightTool]];
}

- (void)keyDown:(NSEvent *)event
{
    if (downEventOccurred) { return; }
    int scrollAmount = 1;
    if(([event modifierFlags] & NSShiftKeyMask) == NSShiftKeyMask)
    {
        scrollAmount = 10;
    }
    if([[event characters] characterAtIndex:0] == NSUpArrowFunctionKey)
    {
        [view scrollUpBy:scrollAmount];
    }
    else if([[event characters] characterAtIndex:0] == NSRightArrowFunctionKey)
    {
        [view scrollRightBy:scrollAmount];
    }
    else if([[event characters] characterAtIndex:0] == NSDownArrowFunctionKey)
    {
        [view scrollDownBy:scrollAmount];
    }
    else if([[event characters] characterAtIndex:0] == NSLeftArrowFunctionKey)
    {
        [view scrollLeftBy:scrollAmount];
    }
    else if([[event characters] characterAtIndex:0] == NSDeleteFunctionKey)
    {
		[[[NSDocumentController sharedDocumentController] currentDocument] delete:self];
        //[canvas deleteSelection];
		//[canvas changedInRect:NSMakeRect(0, 0, [canvas size].width, [canvas size].height)];
    }
    else
    {
        [[PXToolPaletteController palette] keyDown:event];
		[[PXColorPaletteController palette] keyDown:event];
    }
}

- (void)flagsChanged:event
{
    [[PXToolPaletteController palette] flagsChanged:event];
}

- (IBAction)showPreviewWindow:sender
{
    [previewController showWindow:self];
}

- (IBAction)togglePreviewWindow:sender
{
	if ([[previewController window] isVisible])
	{
		[[previewController window] performClose:self];
	}
	else
	{
	    [previewController showWindow:self];
	}	
}

- (IBAction)showBackgroundInfo:sender
{
    [backgroundController showWindow:self];
}

- (IBAction)showGridSettingsPrompter:sender
{
    if (gridSettingsPrompter) { [gridSettingsPrompter release]; }
    gridSettingsPrompter = [[PXGridSettingsPrompter alloc] initWithSize:[[view grid] unitSize] color:[[view grid] color] shouldDraw:[[view grid] shouldDraw] ? YES : NO];
    [gridSettingsPrompter setDelegate:self];
    [(PXGridSettingsPrompter *)gridSettingsPrompter prompt];
}

- (void)zoomToIndex:(float)index
{
    if(index < 0 || index >= [zoomPercentageBox numberOfItems]) { NSBeep(); return; }
    [zoomPercentageBox selectItemAtIndex:index];
    [zoomStepper setIntValue:index];
    [view setZoomPercentage:[zoomPercentageBox intValue]];
}

- (void)zoomToPercentage:(NSNumber *)percentage
{
    if(percentage == nil || [[[percentage description] lowercaseString] isEqualToString:@"inf"] || [[[percentage description] lowercaseString] isEqualToString:@"nan"]) { [self zoomToPercentage:[NSNumber numberWithFloat:100]]; return;}
    if(![[zoomPercentageBox objectValues] containsObject:percentage])
    {
        id values = [NSMutableArray arrayWithArray:[zoomPercentageBox objectValues]];
        [values addObject:percentage];
        [values sortUsingSelector:@selector(compare:)];
        [zoomPercentageBox removeAllItems];
        [zoomPercentageBox addItemsWithObjectValues:[[values reverseObjectEnumerator] allObjects]];
    }
    [zoomPercentageBox selectItemWithObjectValue:percentage];
    [self zoomToIndex:[zoomPercentageBox indexOfSelectedItem]];
}

- (IBAction)zoomIn:sender
{
    [self zoomToIndex:[zoomStepper intValue]-1];
}

- (IBAction)zoomOut:sender
{
    [self zoomToIndex:[zoomStepper intValue]+1];
}

- (IBAction)zoomStandard:sender
{
    [self zoomToIndex:[zoomPercentageBox indexOfItemWithObjectValue:[NSNumber numberWithInt:100]]];
}

- (IBAction)zoomPercentageChanged:sender
{
    [self zoomToPercentage:[zoomPercentageBox objectValue]];
}

- (IBAction)zoomStepperStepped:sender
{
    if([zoomStepper intValue] >= [zoomPercentageBox numberOfItems]) { NSBeep(); [zoomStepper setIntValue:[zoomPercentageBox numberOfItems]-1]; return; }
    [self zoomToIndex:[zoomStepper intValue]];
}

- (IBAction)zoomToFit:sender
{
    float xRatio = [[scrollView contentView] frame].size.width/[canvas size].width;
    float yRatio = [[scrollView contentView] frame].size.height/[canvas size].height;
    [self zoomToPercentage:([[scrollView contentView] frame].size.width > [canvas size].width ||
							[[scrollView contentView] frame].size.width > [canvas size].width) ?
							[NSNumber numberWithFloat:(floorf(xRatio < yRatio ? xRatio : yRatio))*100] :
		[NSNumber numberWithFloat:100.0f]];
}

- (void)zoomInOnCanvasPoint:(NSPoint)point
{
    [self zoomIn:self];
    [view centerOn:[view convertFromCanvasToViewPoint:point]];
}

- (void)zoomOutOnCanvasPoint:(NSPoint)point
{
    [self zoomOut:self];
    [view centerOn:[view convertFromCanvasToViewPoint:point]];
}

- (IBAction)toggleLayersDrawer:sender
{
    [layerController toggle:sender];
}

@end
