///
/// Playlist "tuner".
/// Manages the playlists included in a mix.
///	@file		tuner.cpp - pianod2
///	@author		Perette Barella
///	@date		2015-12-11
///	@copyright	Copyright (c) 2016-2020 Devious Fish. All rights reserved.
///

#include <config.h>

#include <cassert>
#include <cstdio>

#include <string>
#include <vector>
#include <algorithm>

#include <football.h>
#include <footparser.h>

#include "callback.h"
#include "callbackimpl.h"
#include "logging.h"
#include "connection.h"
#include "response.h"
#include "user.h"
#include "users.h"
#include "mediaunit.h"
#include "mediamanager.h"
#include "predicate.h"

#include "tuner.h"

/// Text to enumeration translations for playlist mix actions.
const LookupTable <CartsVerb> CartsWord {
    { "Clear",      CartsVerb::Clear },
    { "Add",        CartsVerb::Add },
    { "Remove",     CartsVerb::Remove },
    { "Delete",     CartsVerb::Remove },
    { "Toggle",     CartsVerb::Toggle },
    { "Set",        CartsVerb::Set }
};

/// Text to enumeration translations for randomization methods.
const LookupTable <Media::SelectionMethod> SelectionMethodWords {
    { "album",      Media::SelectionMethod::Album },
    { "artist",     Media::SelectionMethod::Artist },
    { "song",       Media::SelectionMethod::Song },
    { "playlist",   Media::SelectionMethod::Playlist },
    { "random",     Media::SelectionMethod::Random }
};

// Force template instantiation.
template class CallbackManager<Tuner::Tuner, Tuner::Tuner::Callbacks>;

using namespace std;
namespace Tuner {

    /** Create a new tuner and load known playlists.
        @param svc The service with which the tuner will be associated. */
    Tuner::Tuner (PianodService *svc) : service (svc) {
        updatePlaylists();
        autotune.login = true;
        autotune.flag = true;
        autotune.proximity = true;

        Media::Manager::Callbacks callbacks;
        callbacks.sourceReady = bind (&Tuner::sourcesChanged, this, std::placeholders::_1);
        callbacks.sourceOffline = bind (&Tuner::sourcesChanged, this, std::placeholders::_1);
        media_manager->callback.subscribe (this, callbacks);
    }

    /** Destroy the tuner and release resources. */
    Tuner::~Tuner() {
        media_manager->callback.unsubscribe (this);
        for (auto &item : mix) {
            item.second.playlist->release();
        }
    }

    /** Register a new source by adding its playlists to those known */
    void Tuner::sourcesChanged (const Media::Source *) {
        updatePlaylists();
        recalculatePlaylists();
    }


    /** Update the list of playlists known by the tuner. */
    void Tuner::updatePlaylists() {
        PlaylistList allPlaylists = media_manager->getPlaylists();
        MixMap new_mix;
        new_mix.reserve (allPlaylists.size());

        for (auto playlist : allPlaylists) {
            MixItem &slot = new_mix [playlist->id()];
            assert (!slot.playlist);
            slot.playlist = playlist;
            playlist->retain();

            // Retain the enabled status from the old list, if possible.
            auto old_one = mix.find (playlist->id());
            assert (old_one == mix.end() || old_one->second.playlist);
            slot.enabled = (old_one == mix.end() ? playlist->includedInMix() :
                            old_one->second.enabled);
        }

        swap (mix, new_mix);

        for (auto &item : new_mix) {
            item.second.playlist->release();
        }
        callback.notify (&Callbacks::playlistsChanged);
    }

    /** Remove a source from the playlist list in preparation for its removal. */
    void Tuner::purge (const Media::Source *source) {
        auto it = mix.begin();
        while (it != mix.end()) {
            if (it->second.playlist->source() == source) {
                it->second.playlist->release();
                it = mix.erase (it);
            } else {
                it++;
            }
        }
    }

    /** Push the current playlist selections to the playlists/sources in
        preparation for getting random songs.
        @param source If null, pushes all playlists.  Otherwise, only
        pushes those belonging to the source. */
    void Tuner::pushPlaylistSelections(Media::Source *source) {
        for (auto &item : mix) {
            if (!source || source == item.second.playlist->source()) {
                item.second.playlist->includedInMix (item.second.enabled);
            }
        }
    }

    /** Check if a playlist is included in the mix.
        @param playlist The playlist whose status to check.
        @return True if the playlist is in the mix, false otherwise. */
    bool Tuner::includedInMix (const PianodPlaylist *playlist) {
        MixMap::iterator it = mix.find (playlist->id());
        if (it != mix.end())
            return it->second.enabled;
	
        flog (LOG_WHERE (LOG_WARNING), "Playlist ", playlist->playlistName(),
            " is not a known playlist.");
	return true;
    }

    /** Check for enabled playlists in the mix.
        @param source If not-null, consider only playlists from specified source.
        @return True if no matching enabled playlists are found, false otherwise. */
    bool Tuner::empty(const Media::Source *source) const {
        if (automatic_mode && !anyone_listening)
            return true;
        for (const auto &item : mix) {
            if (item.second.enabled && (!source || source == item.second.playlist->source())) {
                return false;
            }
        }
        return true;
    }

    /** Set or clear automatic playlist selection.
        @param automatic_playlists True to enable, false otherwise. */
    void Tuner::automatic(bool automatic_playlists) {
        automatic_mode = automatic_playlists;
        recalculatePlaylists();
    }


    /** Get a list of users applicable to autotuning.
        @param settings The autotuning mode settings.
        @param service The football service to search for user logins.
        @return A list of users matching the autotune settings. */
    UserList Tuner::getApplicableUsers (const AutotuneSettings &settings,
                                        const PianodService *service) {
        return user_manager->getUsers([service, settings] (const User *user) -> bool {
            if (!user->havePrivilege(Privilege::Influence))
                return false;
            if (settings.login && user->online (*service))
                return true;
            if (settings.flag && user->havePrivilege (Privilege::Present))
                return true;
            return false;
        });
    }

    /** Get a list of users to consider for autotuning with the current
        service and autotuning mode. */
    UserList Tuner::getAutotuneUsers () {
        return getApplicableUsers (autotune, service);
    };



    /** A structure used for autotuning calculations. */
    struct PlaylistSummary {
        PianodPlaylist *playlist;   ///< The playlist
        bool vetoed = false;        ///< Set if a user hates this playlist
        float average_rating = ratingAsFloat (Rating::NEUTRAL);
        PlaylistSummary (PianodPlaylist *p) : playlist (p) {};
        PlaylistSummary () { };
        bool operator< (const PlaylistSummary &other) const {
            return ((vetoed ? 0 : average_rating) <
                    (other.vetoed ? 0 : other.average_rating));
        }
     };


    /** Choose a random playlist (for mixing via playlist).
        @criteria Specifies which playlists to choose from:
        which source, and single, mix, everything.
        @return The single playlist chosen. */
    PianodPlaylist *Tuner::getRandomPlaylist (const PianodPlaylist *criteria) {
        assert (criteria->playlistType() != PianodPlaylist::SINGLE);
        bool everything = criteria->playlistType() == PianodPlaylist::EVERYTHING;
        bool manager = criteria->source() == media_manager;
        vector<PianodPlaylist *> choices;
        choices.reserve (mix.size());
        for (const auto playlist : mix) {
            if ((everything || playlist.second.enabled) &&
                (manager || playlist.second.playlist->source() == criteria->source())) {
                choices.push_back (playlist.second.playlist);
            }
        }
        assert (!choices.empty());
        return (choices [random() % choices.size()]);
    }


    /** Acquire some random tracks utilizing the currently-selected randomization mode.
        @param from A playlist or a metaplaylist from which to choose music.
        @param users The users present, for biasing selections.
        @return Some random tracks for play. */
    SongList Tuner::getRandomTracks(PianodPlaylist *from, const UserList &users) {
        Media::SelectionMethod selection_method = random_selection_method;
        if (selection_method == Media::SelectionMethod::Random) {
            // Choose a concrete method.
            selection_method = static_cast <Media::SelectionMethod> (random() % (int) Media::SelectionMethod::Random);
        }

        // Update room mix selections in sources, if required.
        if (from->playlistType() == PianodPlaylist::MIX) {
            Media::Source *source = (from->source() == media_manager
                                     ? nullptr : from->source());
            if (empty (source)) {
                // Nothing selected or current users don't agree on music, so be quiet.
                return SongList {};
            }

            pushPlaylistSelections (source);
        }

        // If we're mixing via playlist, choose a single playlist to pick from.
        if (selection_method == Media::SelectionMethod::Playlist) {
            if (from->playlistType() != PianodPlaylist::SINGLE) {
                from = getRandomPlaylist (from);
            }
            selection_method = Media::SelectionMethod::Song;
        }

        // Shuffle some stuff into the queue.
        SongList songs = from->getRandomSongs (users, selection_method);
        if (songs.empty()) {
            flog (LOG_WHERE (LOG_WARNING), "Playlist ", from->playlistName(),
                  " did not provide random tracks");
            service << F_PLAYER_EMPTY;
            throw runtime_error ("Could not retrieve random tracks");
        }
        return songs;
    }


    /** Recalculate the playlists based on current users and autotune settings.
        @return true if playlists changed, false otherwise. */
    bool Tuner::recalculatePlaylists () {
        if (mix.empty() || !automatic_mode) {
            flog (LOG_WHERE(LOG_TUNING), "No playlists or autotuning disabled.");
            return false;
        }

        vector<PlaylistSummary> playlist_data;

        // Get the list of users that will influence calculations.
        UserList considered_users = getAutotuneUsers ();
        anyone_listening = !considered_users.empty();
        if (!anyone_listening) {
            flog (LOG_WHERE(LOG_TUNING), "No users to consider for autotuning.");
            return false;
        }

        flog (LOG_FUNCTION (LOG_TUNING), "Computed station biases follow:");

        // For each source...
        for (const auto &source : media_manager->getReadySources ()) {
            // Get user ratings for this source.
            vector<UserData::Ratings *> source_ratings;
            for (const auto user : considered_users) {
                UserData::Ratings *ratings = UserData::Ratings::retrieve (user, UserData::Key::PlaylistRatings, source->key());
                if (ratings) {
                    source_ratings.push_back (ratings);
                }
            }
            // Calculate average ratings for each playlist from this source.
            PlaylistList playlists = source->getPlaylists();
            for (auto playlist : playlists) {
                std::string id { playlist->playlistId() };
                PlaylistSummary item (playlist);
                RatingAverager average;
                for (auto user_ratings : source_ratings) {
                    auto rating = user_ratings->find (id);
                    if (rating != user_ratings->end()) {
                        if (ratingAsFloat (rating->second) <= autotune.veto_threshold)
                            item.vetoed = true;
                        average.add (rating->second);
                    }
                }
                item.average_rating = average (Rating::NEUTRAL);
                playlist_data.push_back (item);
#ifndef NDEBUG
                if (logging_enabled (LOG_BIASING)) {
                    fprintf (stderr, "%-40.40s rate %5f = %u/%u vetoed %s\n",
                             playlist->name().c_str(),
                             item.average_rating, average.sum(), average.items(),
                             item.vetoed ? "yes" : "no ");
                }
#endif
            }
        }

        // We've now rated every playlist from every source.  Sort the results ascending.
        sort (playlist_data.begin(), playlist_data.end());

        // Choose a quality goal: the best average rating, adjusted with some margin.
        float quantity_goal_min_quality = playlist_data.back().average_rating - autotune.quality_margin;
        bool changed = false;

        int included_count = 0; // # of included playlists.
        int good_count = 0; // # of playlists that meet inclusion threshold.
        int acceptable_count = 0; // # of playlists included to meet variety target.
        int equal_count = 0; // # of playlists included because rated same as prior inclusions.
        float equal_rating = 11;
        for (auto it = playlist_data.rbegin(); it != playlist_data.rend(); it++) {
            const auto &playlist = *it;
            const char *reason = nullptr;
            bool included = false;
            if (playlist.vetoed) {
                reason = "vetoed";
            } else if (playlist.average_rating < autotune.rejection_threshold) {
                reason = "average below rejection threshold";
            } else if (playlist.average_rating >= autotune.inclusion_threshold) {
                reason = "average above inclusion threshold";
                included = true;
                good_count++;
            } else if (included_count < autotune.quantity_goal && playlist.average_rating >= quantity_goal_min_quality) {
                reason = "trying to meet quantity goal";
                included = true;
                acceptable_count++;
            } else if (playlist.average_rating >= equal_rating) {
                reason = "average equal to others included";
                included = true;
                equal_count++;
            } else {
                reason = (included_count < autotune.quantity_goal
                          ? "average below quality margin"
                          : "quantity goal already met.");
            }
            flog (LOG_WHERE (LOG_TUNING), playlist.playlist->name(),
                  included ? " is included: " : " is excluded: ", reason);
            if (included) {
                included_count++;
                equal_rating = playlist.average_rating;
            }
            MixMap::iterator mixit = mix.find (playlist.playlist->id());
            assert (mixit != mix.end());
            if (mixit != mix.end() && mixit->second.enabled != included) {
                changed = true;
                mixit->second.enabled = included;
            }
        }

        if (changed) {
            service << V_MIX_CHANGED;
            int message_id = ((good_count ? 4 : 0) |
                           (acceptable_count ? 2 : 0) |
                           (equal_count ? 1 : 0));
            assert (message_id >= 0 && message_id <= 7);
            static const char *const messages[8] = {
                /* 0 */"current listener playlists preferences are incompatible",
                /* 1 */ nullptr,
                /* 2 */ "autotuner picked the acceptable playlists",
                /* 3 */ "autotuner picked acceptable playlists with variety",
                /* 4 */ "autotuner picked only good playlists",
                /* 5 */ nullptr,
                /* 6 */ "autotuner picked good and acceptable playlists",
                /* 7 */ "autotuner picked good and acceptable playlists, with variety"
            };
            const char *message = messages [message_id];
            assert (message);
            service << Response (V_SERVER_STATUS, message);
            callback.notify (&Callbacks::mixChanged, true, message);
        }
        return changed;
    }


    static const FB_PARSE_DEFINITION statementList[] = {
        { PLAYLISTLIST,		"playlist [list]" },				// List the playlists
        { PLAYLISTLIST,     "playlist [list]" LIST_PLAYLIST },	// List specific playlists
        // Mix-related commands
        { MIXINCLUDED,      "mix" },                            // Unofficial
        { MIXINCLUDED,      "mix [list] included" },			// Short forms are not official protocol
        { MIXEXCLUDED,      "mix [list] excluded" },			// Show songs not included in mix
        { MIXADJUST,        "mix <verb:add|remove|set|toggle>" LIST_PLAYLIST  }, // Change mix composition
        { AUTOTUNEGETMODE,  "autotune mode" },                  // Query autotune users mode
        { AUTOTUNESETMODE,  "autotune mode {mode} ..." },       // Change the autotune users mode
        { SELECTIONMETHOD,  "queue randomize by <method:song|artist|album|playlist|random>" },
        { CMD_INVALID,      NULL }
    };

    /// Autotuning mode options.
    enum class Options {
        // All
        Logins = 1,             ///< Consider users logged in (and with Influence privilege)
        Flag,                   ///< Consider users with present flag (and with Influence privilege)
        Proximity,              ///< Consider users present using built-in presence algorithms (and with Influence privilege)
        All,                    ///< Consider users present in any of the aforementioned forms (and with Influence privilege)
        VetoThreshold,          ///< If any user's rating is below this treshold, playlist is vetoed.
        RejectionThreshold,     ///< If average rating is below this threshold, playlist is never included.
        InclusionThreshold,     ///< If average rating is above this treshold, playlist is always included.
        QuantityGoal,           ///< Add playlists between inclusion/rejection tresholds to meet this quantity.
        QualityMargin           ///< Like rejection treshold, but calculated dynamically: InclusionThreshold - QualityMargin.
    };

    /** ModeParser fills in the AutotuneSettings structure. */
    class ModeParser : public Football::OptionParser<AutotuneSettings, Options> {
        virtual int handleOption (Options option, AutotuneSettings &dest) override {
            switch (option) {
                case Options::Logins:
                    dest.login = true;
                    return FB_PARSE_SUCCESS;
                case Options::Flag:
                    dest.flag = true;
                    return FB_PARSE_SUCCESS;
                case Options::Proximity:
                    dest.proximity = true;
                    return FB_PARSE_SUCCESS;
                case Options::All:
                    dest.login = true;
                    dest.flag = true;
                    dest.proximity = true;
                    return FB_PARSE_SUCCESS;
                case Options::VetoThreshold:
                    dest.veto_threshold = RATINGS.getPrecise (argv ("rating"));
                    return FB_PARSE_SUCCESS;
                case Options::RejectionThreshold:
                    dest.rejection_threshold = RATINGS.getPrecise (argv ("rating"));
                    return FB_PARSE_SUCCESS;
                case Options::InclusionThreshold:
                    dest.inclusion_threshold = RATINGS.getPrecise (argv ("rating"));
                    return FB_PARSE_SUCCESS;
                case Options::QuantityGoal:
                    dest.quantity_goal = strtol (argv ("count"), nullptr, 10);
                    return FB_PARSE_SUCCESS;
                case Options::QualityMargin:
                    dest.quality_margin = strtod (argv ("margin"), nullptr) * 2;
                    return FB_PARSE_SUCCESS;
            }
            assert (!"Unhandled case in ModeParser");
            return FB_PARSE_FAILURE;
        }

    public:
        ModeParser ();
    };
    /// Formats for specifying autotuning options.
    ModeParser::ParseDefinition mode_statements[] = {
        { Options::Logins,              "login ..." },
        { Options::Flag,                "flag ..." },
        { Options::Proximity,           "proximity ..." },
        { Options::All,                 "all ..." },
        { Options::VetoThreshold,       "veto {rating} ..." },
        { Options::RejectionThreshold,  "reject {rating} ..." },
        { Options::InclusionThreshold,  "include {rating} ..." },
        { Options::QuantityGoal,        "quantity goal {#count:0-999} ..." },
        { Options::QualityMargin,       "quality margin {#margin:0.0-5.0} ..." },
        { (Options) CMD_INVALID,  NULL }
    };
    ModeParser::ModeParser () {
        addStatements (mode_statements);
    };



    static ModeParser modeParser;

    const FB_PARSE_DEFINITION *Tuner::statements (void) {
        return statementList;
    }


    /** Send a list of playlists.
        @param conn Destination for the list.
        @param list The list to send.
        @param long_format If true, send playlist details.  If false, send names only. */
    void Tuner::sendPlaylistList (PianodConnection &conn,
                                  PlaylistList &list,
                                  bool long_format) {
        if (long_format) {
            conn << list << S_DATA_END;
        } else {
            if (!list.empty()) {
                conn << S_DATA;
                for (auto item : list) {
                    conn << Response (I_PLAYLIST, item->playlistName());
                }
            }
            conn << S_DATA_END;
        }
    }




    bool Tuner::hasPermission (PianodConnection &conn, TUNERCOMMAND command) {
        switch (command) {
            case MIXINCLUDED:
            case MIXEXCLUDED:
            case PLAYLISTLIST:
            case AUTOTUNEGETMODE:
                return conn.haveRank (Rank::Listener);
            case AUTOTUNESETMODE:
                return conn.haveRank (Rank::Administrator);
            default:
                return conn.haveRank (Rank::Standard);
        }
    }
    void Tuner::handleCommand (PianodConnection &conn, TUNERCOMMAND command) {
        assert (conn.source());

        switch (command) {
            case PLAYLISTLIST:
            {
                PlaylistList playlists { Predicate::getSpecifiedPlaylists (conn) };
                sendPlaylistList (conn, playlists, conn ["list"]);
                return;
            }
            case MIXINCLUDED:
            case MIXEXCLUDED:
            {
                PlaylistList list = conn.source()->getPlaylists();
                PlaylistList matches;
                for (auto item : list) {
                    if ((command == MIXINCLUDED) == (mix [item->id()].enabled)) {
                        matches.push_back (item);
                    }
                }
                sendPlaylistList (conn, matches, conn ["list"]);
                return;
            }
            case MIXADJUST:
            {
                CartsVerb action = CartsWord [conn ["verb"]];
                assert (conn [PLAYLISTMANNERTAG]);
                PlaylistList update = Predicate::getSpecifiedPlaylists (conn);
                bool change = false;
                if (action == CartsVerb::Set) {
                    for (auto &item : mix) {
                        item.second.enabled = false;
                    }
                    change = true;
                }
                for (auto item : update) {
                    assert (mix.find (item->id()) != mix.end());
                    bool before = mix [item->id()].enabled;
                    bool after = (action == CartsVerb::Add || action == CartsVerb::Set? true :
                                  action == CartsVerb::Remove ? false : !before);
                    if (before != after) {
                        mix [item->id()].enabled = after;
                        change = true;
                    }
                }
                conn << Response (S_MATCH, update.size());
                if (change) {
                    conn.service() << V_MIX_CHANGED;
                    conn.announce(A_CHANGED_MIX);
                    callback.notify (&Callbacks::mixChanged, true, "user action");
                }
                return;
            }
            case AUTOTUNEGETMODE:
            {
                string mode;
                if (autotune.login) mode += " login";
                if (autotune.proximity) mode += " proximity";
                if (autotune.flag) mode += " flag";
                assert (!mode.empty());
                mode.erase (mode.begin());
                conn << S_DATA << Response (I_AUTOTUNE_MODE, mode) << S_DATA_END;
                return;
            }
            case AUTOTUNESETMODE:
            {
                AutotuneSettings new_settings = autotune;
                new_settings.login = false;
                new_settings.flag = false;
                new_settings.proximity = false;
                if (modeParser.interpret (conn.argvFrom ("mode"), new_settings, &conn) == FB_PARSE_SUCCESS) {
                    if (!new_settings.login && !new_settings.flag && !new_settings.proximity) {
                        // Retain old mode.
                        new_settings.login = autotune.login;
                        new_settings.flag = autotune.flag;
                        new_settings.proximity = autotune.proximity;
                    }
                    autotune = new_settings;
                    recalculatePlaylists ();
                    conn << S_OK;
                }
                return;
            }
            case SELECTIONMETHOD:
                random_selection_method = SelectionMethodWords [conn ["method"]];
                conn << S_OK;
                service << Response (V_SELECTIONMETHOD,
                                     SelectionMethodWords [random_selection_method]);
                return;
        }
    }

    /** Send current state information. */
    void Tuner::sendStatus (PianodConnection &there) {
        there << Response (V_SELECTIONMETHOD,
                             SelectionMethodWords [random_selection_method]);
    }
}
