///
/// Method implementations for Playlist / Artist / Album / Song data types.
///	@file		musictypes.cpp - pianod
///	@author		Perette Barella
///	@date		2014-12-09
///	@copyright	Copyright 2014-2020 Devious Fish.  All rights reserved.
///

#include <config.h>

#include <ctime>

#include <typeinfo>
#include <string>
#include <vector>

#include <football.h>

#include "connection.h"
#include "response.h"

#include "musictypes.h"
#include "mediaunit.h"
#include "mediamanager.h"
#include "filter.h"
#include "utility.h"
#include "ownership.h"
#include "user.h"
#include "users.h"
#include "datastore.h"

using namespace std;

/*
 *       Auto Release Pools & Reference Counting
 */
MusicAutoReleasePool::MusicAutoReleasePool() :
previousPool (MusicThingie::releasePool) {
    MusicThingie::releasePool = this;
}

MusicAutoReleasePool::~MusicAutoReleasePool() {
    MusicThingie::releasePool = previousPool;
    while (!empty()) {
        top()->release();
        pop();
    }
}

/** Remove an item from the release pool when it fails subsequent construction. */
void MusicAutoReleasePool::unadd (MusicThingie *item) {
    assert (!empty());
    assert (top() == item);
    assert (item->useCount == 1);
    pop();
}

/** The current autorelease pool to put newly created music thingies into */
MusicAutoReleasePool *MusicThingie::releasePool = nullptr;

/** When allocated, use count starts at 1 and the object
    is put in the release pool, justifying its existence.
    If anyone wants it, they must retain() it before
    autorelease.  One MusicAutoReleasePool is in the main run
    loop; others may be created to shorten temporary lives.
    Retained objects are later deleted instantly when released(). */
MusicThingie::MusicThingie (void) {
    assert (releasePool);
    releasePool->add (this);
}

MusicThingie::~MusicThingie (void) {
    if (useCount != 0) {
        // The only way this "should" happen is if descendent constructor fails,
        // in which case we need to remove this from the autorelease pool.
        releasePool->unadd (this);
    }
}

bool MusicThingie::operator==(const std::string &compare) const {
    return compare_titles (name(), compare);
}



/*
 *              Music Thingie
 */

/** Get the type name of a music thingie.
    @param type The type whose name to retrieve.
    @return A string with the type name. */
std::string MusicThingie::TypeName (Type type) {
    string identity (isPlaylist(type) ? "playlist" :
                     isArtist(type) ? "artist" :
                     isAlbum(type) ? "album" :
                     isSong(type) ? "song" : "Unknown");
    if (isSuggestion (type))
        identity += " suggestion";
    if (type == Type::SongRating)
        identity += " rating";
    else if (isSeed (type))
        identity += " seed";
    return identity;
}


string MusicThingie::operator()(void) const {
    return TypeName (type()) + " \"" + name() + "\" (ID " + id() + ")";
}


MusicThingie::Type MusicThingie::primaryType (Type t) {
    if (isSong (t)) return Type::Song;
    if (isAlbum (t)) return Type::Album;
    if (isArtist (t)) return Type::Artist;
    if (isPlaylist (t)) return Type::Playlist;
    assert (!"primaryType of unknown type");
    throw invalid_argument("Unknown MusicThingie type");
}

/** Retrieve a list of requestable songs applicable to this thingie. */
SongList MusicThingie::songs () {
    return SongList {};
};


Ownership *MusicThingie::parentOwner (void) const {
    return (source());
};


ThingieTypesLookup THINGIETYPES ({
    { "artist",     MusicThingie::Type::Artist },
    { "album",      MusicThingie::Type::Album },
    { "song",       MusicThingie::Type::Song },
    { "playlist",   MusicThingie::Type::Playlist }
});

ThingieList::ThingieList (const SongList &songs) {
    copy(songs.begin(), songs.end(), back_inserter (*this));
};

/** Purge anything that isn't a certain type */
void ThingieList::limitTo (MusicThingie::Type type) {
    ThingieList new_list;
    for (auto it : *this) {
        if (type == it->primaryType()) {
            new_list.push_back (it);
        }
    }
    swap (new_list);
}



/*
 *                  Artists
 */

const std::string PianodArtist::id (void) const {
    return to_string (source()->serialNumber()) + (char) type() + artistId();
};

const std::string &PianodArtist::id (MusicThingie::Type type) const {
    const static std::string empty;
    assert (isPrimary (type));
    switch (type) {
        case MusicThingie::Type::Artist:
            return artistId();
        default:
            return empty;;
    }
};

PianodConnection &PianodArtist::transmit (PianodConnection &recipient) const {
    recipient
    << Response (I_ID, id())
    << Response (I_ARTIST, artist())
    << Response (I_SOURCE, source()->kind())
    << Response (I_NAME, source()->name());
    if (isUsableBy (recipient.user) && isPrimary() && canQueue()) {
        recipient.printf ("%03d %s: request\n",
                     I_ACTIONS, ResponseText (I_ACTIONS));
    }
    return recipient;
}

bool PianodArtist::matches (const Filter &filter) const {
    return filter.matches (this);
}




bool PianodArtist::operator==(const string &compare) const {
    return compare_person_or_title(artist(), compare);
}

bool PianodArtist::operator==(const MusicThingie &compare) const {
    if (type() != compare.type()) return false;

    auto other = dynamic_cast<const PianodArtist *>(&compare);
    assert (other);
    if (!other) return false;
    return compare_person_or_title(artist(), other->artist());
}


/*
 *                  Albums
 */


const std::string PianodAlbum::id (void) const {
    return to_string (source()->serialNumber()) + (char) type() + albumId();
};

const std::string &PianodAlbum::id (MusicThingie::Type type) const {
    const static std::string empty;
    assert (isPrimary (type));
    switch (type) {
        case MusicThingie::Type::Artist:
            return artistId();
        case MusicThingie::Type::Album:
            return albumId();
        default:
            return empty;;
    }
};

PianodConnection &PianodAlbum::transmit (PianodConnection &recipient) const {
    recipient << Response (I_ID, id());
    if (!artist().empty())
        recipient << Response (I_ARTIST, artist());
    recipient
    << Response (I_ALBUM, albumTitle())
    << Response (I_SOURCE, source()->kind())
    << Response (I_NAME, source()->name());
    if (isUsableBy (recipient.user) && isPrimary() && canQueue()) {
        recipient.printf ("%03d %s: request\n",
                          I_ACTIONS, ResponseText (I_ACTIONS));
    }
    return recipient;
}

bool PianodAlbum::matches (const Filter &filter) const {
    return filter.matches (this);
}



bool PianodAlbum::operator==(const string &compare) const {
    return compare_titles (albumTitle(), compare);
}

bool PianodAlbum::operator==(const MusicThingie &compare) const {
    if (type() != compare.type()) return false;

    auto other = dynamic_cast<const PianodAlbum *>(&compare);
    assert (other);
    if (!other) return false;
    return compare_titles(albumTitle(), other->albumTitle()) &&
           compare_person_or_title(artist(), other->artist());

}


/*
 *              Songs
 */

const std::string PianodSong::id (void) const {
    return to_string (source()->serialNumber()) + (char) type() + songId();
};

const std::string &PianodSong::id (MusicThingie::Type type) const {
    const static std::string empty;
    assert (isPrimary (type));
    switch (type) {
        case MusicThingie::Type::Artist:
            return artistId();
        case MusicThingie::Type::Album:
            return albumId();
        case MusicThingie::Type::Song:
            return songId();
        case MusicThingie::Type::Playlist:
            return (playlist() ? playlist()->playlistId() : empty);
        default:
            return empty;;
    }
};

PianodConnection &PianodSong::transmit (PianodConnection &recipient) const {
    sendSong (recipient, *this);
    sendRatings (&recipient, *this);
    return recipient;
}

bool PianodSong::matches (const Filter &filter) const {
    return filter.matches (this);
}

bool PianodSong::operator==(const string &compare) const {
    return compare_titles (name(), compare);
}

bool PianodSong::operator==(const MusicThingie &compare) const {
    if (type() != compare.type()) return false;

    auto other = dynamic_cast<const PianodSong *>(&compare);
    assert (other);
    if (!other) return false;
    return compare_titles (name(), other->name()) &&
           compare_titles(albumTitle(), other->albumTitle()) &&
           compare_person_or_title(artist(), other->artist());
}

/** Return the average rating of a song, considering all ratings. */
float PianodSong::averageRating() const {
    if (ratingScheme() == RatingScheme::Nobody)
        return ratingAsFloat (Rating::UNRATED);
    if (ratingScheme() == RatingScheme::Owner)
        return ratingAsFloat (rating (nullptr));
    RatingAverager average;
    const UserList all_users { user_manager->allUsers() };
    for (const auto user : all_users) {
        average.add (rating (user));
    }
    return (average ());
}

/** Check for permission to skip a song.
    Some sources, such as Pandora, limit skipping for non-paying bastards.
    If skipping is permitted, it should be counted.
    @param whenAllowed On return, contains time at which skip will be allowed.
    @return True if skipping is allowed. */
bool PianodSong::canSkip (time_t *whenAllowed) {
    (void) whenAllowed;
    return true;
}


/** Play the song. */
Media::Player *PianodSong::play (const AudioSettings &audio) {
    Media::Player *player = media_manager->getPlayer (audio, this);
    if (player) {
        last_played = time (nullptr);
    };
    return player;
};

SongList PianodSong::songs () {
    assert (canQueue());
    SongList songs;
    if (isPrimary())
        songs.push_back (this);
    return songs;
};


/** Provide a URL with additional info.
    @return URL or empty string if unknown. */
const std::string &PianodSong::infoUrl (void) const {
    const static string empty;
    return empty;
}

/** Get name of a playlist to which this belongs.
 @return Name, or an empty string. */
const std::string &PianodSong::playlistName (void) const {
    const static string empty;
    PianodPlaylist *pl = playlist ();
    return pl ? pl->playlistName() : empty;
}



/*
 *              SongList
 */

/** SongList copy constructor.
    For efficiency, reserve the necessary amount of space
    for the new list, then use copy assignment from base class. */
SongList::SongList (const SongList &list) :
this_type () {
    reserve (list.size());
    this_type::operator = (list);
};

/** SongList copy assignment.
    For efficiency, reserve the necessary amount of space
    for the new list, then use copy assignment from base class. */
SongList &SongList::operator =(const SongList &list) {
    reserve (list.size());
    this_type::operator = (list);
    return *this;
};


//SongList SongList::filter(const Filter &criteria) {
//    SongList newList;
//    newList.reserve(size());
//    for (auto song : *this) {
//        if (criteria.matches (*song)) {
//            newList.push_back (song);
//        }
//    }
//    return newList;
//}


/// Mix up the songs in a list.
void SongList::shuffle (void) {
    std::random_shuffle (begin(), end());
}

/** Randomly merge a songlist into another.
    Randomly mix playlists together, but maintain sequence of each
    to accommodate sources like Pandora that (ostensibly) care about
    sequence. */
void SongList::mixedMerge (const SongList &adds) {
    SongList originals;
    originals.reserve(size() + adds.size());
    swap (originals);

    size_type orig_index = 0;
    size_type add_index = 0;
    for (int songs = originals.size() + adds.size(); songs; songs--) {
        int item = random() % songs;
        if (item < (originals.size() - orig_index)) {
            push_back (originals [orig_index++]);
        } else {
            push_back (adds [add_index++]);
        }
    }
}


/// Join lists, and use reserve to do it efficiently. 
void SongList::join (SongList &from) {
    reserve (size() + from.size());
    this_type::join (from);
}

SongList PianodPlaylist::songs () {
    return (songs (Filter::All));
};

/** Get songs belonging to a playlist.
    @param filter Match only matsongs matching this.
    If omitted, match all.
    @return A list of matching songs from the playlist. */
SongList PianodPlaylist::songs (const Filter &filter) {
    assert (!"songs invoked on non-request playlist.");
    (void) filter;
    return SongList();
}


/** Choose some random songs for queueing.
    Because different sources do this differently, it can't be
    a baseline algorithm; thus it is up to the sources. */
SongList PianodPlaylist::getRandomSongs (const UserList &users,
                                         Media::SelectionMethod selectionMethod) {
    return source()->getRandomSongs (this, users, selectionMethod);
};



/*
 *                  Playlists
 */


const std::string PianodPlaylist::id (void) const {
    return to_string (source()->serialNumber()) + (char) type() + playlistId();
};

const std::string &PianodPlaylist::id (MusicThingie::Type type) const {
    static std::string empty;
    assert (isPrimary (type));
    switch (type) {
        case MusicThingie::Type::Playlist:
            return playlistId();
        default:
            return empty;;
    }
};

PianodConnection &PianodPlaylist::transmit (PianodConnection &recipient) const {
    recipient
    << Response (I_ID, id())
    << Response (I_PLAYLIST, playlistName());
    if (recipient.user) {
        Rating rated = rating (recipient.user);
        recipient.printf ("%03d %s: %s %1.1f\n",
                     I_PLAYLISTRATING, ResponseText (I_PLAYLISTRATING),
                     RATINGS [rated],
                     (double) ratingAsFloat (rated));
    }
    if (!genre().empty())
        recipient << Response (I_GENRE, genre());
    recipient
    << Response (I_SOURCE, source()->kind())
    << Response (I_NAME, source()->name());

    // Provide the list of actions the user can take on this playlist
    sendRatings (recipient, *this);
    return recipient;
}

bool PianodPlaylist::matches (const Filter &filter) const {
    return filter.matches (this);
}

bool PianodPlaylist::operator==(const string &compare) const {
    return compare_titles (playlistName(), compare);
}

bool PianodPlaylist::operator==(const MusicThingie &compare) const {
    if (type() != compare.type()) return false;

    auto other = dynamic_cast<const PianodPlaylist *>(&compare);
    assert (other);
    if (!other) return false;
    return compare_titles (playlistName(), other->playlistName());
}


/** Set a user's playlist's rating.  This is handled internally by pianod.
    @param value The rating to assign.
    @param user The user rating the playlist.
    @return A status code. */
RESPONSE_CODE PianodPlaylist::rate (Rating value, User *user) {
    if (!user)
        throw CommandError (E_LOGINREQUIRED);
    if (playlistType() != PlaylistType::SINGLE)
        throw CommandError (E_WRONGTYPE);
    UserData::Ratings *ratings = UserData::Ratings::retrieve (user, UserData::Key::PlaylistRatings, source()->key());
    if (!ratings) {
        throw CommandError (E_RESOURCE);
    } try {
        (*ratings) [playlistId()] = value;
        user->updateData();
        return S_OK;
    } catch (const bad_alloc &) {
        throw CommandError (E_RESOURCE);
    }
};


/** Retrieve a user's playlist's rating.
 @param user The user to get the rating for.
 @return A rating code, or Rating::UNRATED. */
Rating PianodPlaylist::rating (const User *user) const {
    if (!user) return Rating::UNRATED;
    UserData::Ratings *ratings = UserData::Ratings::retrieve (user, UserData::Key::PlaylistRatings, source()->key());
    if (!ratings) return Rating::UNRATED;
    return ratings->get (playlistId(), Rating::UNRATED);
};

/** Return the average rating of a playlist, considering all ratings. */
float PianodPlaylist::averageRating() const {
    RatingAverager average;
    const UserList all_users { user_manager->allUsers() };
    for (const auto user : all_users) {
        average.add (rating (user));
    }
    return (average ());;
}

/** Determine if a particular type of seeding is possible.
    @param seedType The type of seeding to check for.
    @return True if that kind of seeding is allowed. */
bool PianodPlaylist::canSeed (MusicThingie::Type seedType) const {
    (void) seedType;
    return false;
}

/** Check if there is a seed of a particular type for this thingie.
    @param seedType Indicates artist, album or song seed.
    @param music The artist, album or song to use as a seed. */
bool PianodPlaylist::seed (MusicThingie::Type seedType, const MusicThingie *music) const {
    assert (0);
    (void) seedType; (void) music;
    return false;
}

/** Make a particular seed type for this thingie.
    @param seedType Indicates artist, album or song seed.
    @param music The artist, album or song to use as a seed.
    @param value Whether to set or remove a seed. */
void PianodPlaylist::seed (MusicThingie::Type seedType, MusicThingie *music, bool value) {
    assert (0);
    (void) seedType; (void) music; (void) value;
    throw CommandError (E_UNSUPPORTED);
}

/** Get the seed list for a playlist. */
ThingieList PianodPlaylist::getSeeds (void) const {
    assert (!"getSeeds not implemented.");
    return ThingieList();
}



