///
/// Caches for music thingie types.
///	@file		musiccache.cpp - pianod
///	@author		Perette Barella
///	@date		2020-03-19
///	@copyright	Copyright (c) 2016-2020 Devious Fish. All rights reserved.
///

#include "musictypes.h"
#include "musiccache.h"
#include "filter.h"
#include "mediaunit.h"

/** Extend the life of a cache object.
    @param seconds Number of seconds to guarantee the object's life. */
void ThingieCache::extend (int seconds) const {
    time_t when = time (nullptr) + seconds;
    if (expiration < when)
        expiration = when;
}


/** Initialize a thingie storage cache/pool. */
ThingiePool::ThingiePool () {
    next_purge = time (nullptr) + settings.purge_interval;
};

/** Initialize a thingie storage cache/pool.
    @param params Parameters describing the cache's retention
    and purge behavior. */
ThingiePool::ThingiePool (const ThingiePoolParameters &params)
: settings (params) {
    next_purge = time (nullptr) + settings.purge_interval;
};

/** Update the cache's retention parameters. */
void ThingiePool::setParameters (const ThingiePoolParameters &params) {
    settings = params;
    next_purge = 0;
    purge();
}

/** Add an item to the cache.  If duplicate,
    will replace the item with the one from the cache,
    or vice-versa, depending on preferNew setting at
    cache instantiation.
    @param thing The item to put in the cache. */
void ThingiePool::add(MusicThingie *thing) {
    assert (thing);
    if (size() >= settings.maximum_retained_items) {
        purge();
    }
    ThingieCache &cached = (*this) [thing->id()];
    if (thing == cached.item.get()) {
        cached.extend (settings.initial_duration);
        flog (LOG_WHERE (LOG_WARNING), "Cache item re-added, ", (*thing)());
        return;
    }
    if (cached.item) {
        // Item was already in the cache
        bool identical = (*thing == *(cached.item));
        flog (LOG_WHERE(identical ? LOG_GENERAL : LOG_WARNING),
              "Cache item replaced with ", identical ? "" : "non-",
              "identical ", (*thing)());
    }
    cached.item = thing;
    cached.extend (settings.initial_duration);
}

/** Add a several things to the cache.
    @param A list of things to add to the cache. */
void ThingiePool::add (const SongList &list) {
    for (auto item : list) {
        add (item);
    }
}


/** Add a several things to the cache.
    @param A list of things to add to the cache. */
void ThingiePool::add (const ThingieList &list) {
    for (auto item : list) {
        add (item);
    }
}



/** Get an item from the cache.
    @param id The identifier of the thing to get.
    @return The thingie, or nullptr if it's not in the cache. */
MusicThingie *ThingiePool::get (const std::string &id) {
    auto const it = find (id);
    if (it == end()) return nullptr;
    assert (it->second.item);
    // If it's getting used, keep it around a little longer
    it->second.extend (settings.reprieve_duration);
    return it->second.item.get();
}

/** Get matching items from the cache.
    @param filter A filter to select things to get.
    @return The thingies, or an empty set if no matches. */
ThingieList ThingiePool::get (const Filter &filter) {
    ThingieList matches;
    for (auto const &item : *this) {
        assert (item.second.item);
        if (item.second.item->matches (filter)) {
            matches.push_back (item.second.item.get());
            // If it's getting used, keep it around a little longer
            item.second.extend (settings.reprieve_duration);
        }
    }
    return matches;
}


/// Pare down the cache
void ThingiePool::purge() {
    std::string source_name = "purge";
    if (!empty()) {
        source_name = begin()->second.item->source()->key();
    }
    next_purge = time (nullptr) + settings.purge_interval;
    if (size() < settings.minimum_retained_items) {
        flog (LOG_WHERE (LOG_CACHES), source_name, ": Size ", size(), " < Minimum retention of ", settings.minimum_retained_items);
        return;
    }
    if (time (nullptr) < oldest_entry) {
        flog (LOG_WHERE (LOG_CACHES), source_name, ": No old enough candidates for purge, size=", size());
        next_purge = oldest_entry;
        return;
    }
    flog (LOG_WHERE (LOG_CACHES), source_name, ": Pre-purge, size= ", size());

    time_t now = time (nullptr);
    time_t cutoff;
    if (now > oldest_entry + settings.purge_interval) {
        cutoff = ((now / 2) + (oldest_entry) / 2);
    } else {
        cutoff = now;
    }
    time_t oldest_entry = FAR_FUTURE;
    
    for (iterator it = begin(); it != end(); ) {
        iterator target = it++;
        if (target->second.item->getUseCount() > 1) {
            // Item is in use, so extend its life.
            target->second.extend (settings.reprieve_duration);
        }
        if (target->second.expiration <= cutoff && size() > settings.minimum_retained_items) {
            erase (target);
        } else if (target->second.expiration < oldest_entry) {
            oldest_entry = target->second.expiration;
        }
    }
    flog (LOG_WHERE (LOG_CACHES), source_name, ": Post-purge, size= ", size());
}
