///
/// Handle commands controlling the overall pianod deamon/service.
/// Manages "rooms": separate services each with its own audio etc.
/// @file       servicemanager.h - pianod project
/// @author     Perette Barella
/// @date       2014-12-15
/// @copyright  Copyright 2012-2014-2016 Devious Fish. All rights reserved.
///
#ifndef pianod_service_h
#define pianod_service_h

#include <config.h>

#include <string>
#include <unordered_map>

#include "fundamentals.h"
#include "connection.h"
#include "response.h"

namespace Media {
    class Source;
}

typedef enum service_commands_t {
    NOP = CMD_RANGE_SERVICE,
    HELP,
    YELL,
    QUIT,
    NEWROOM,
    DELETEROOM,
    ROOMINFO,
    CHOOSEROOM,
    INROOMEXEC,
    LISTROOMS,
    SHUTDOWN,
    SETLOGGINGFLAGS,
    SHOWUSERACTIONS,
    SYNC,
    UPTIME
} SERVICECOMMAND;

/** Contains and manages all the services/rooms. */
class ServiceManager :
private Football::Interpreter<PianodConnection, SERVICECOMMAND>,
private std::unordered_map<std::string, PianodService *> {
public:
//    using std::unordered_map<std::string, PianodService *>::size;
    using std::unordered_map<std::string, PianodService *>::begin;
    using std::unordered_map<std::string, PianodService *>::end;
private:
    PianodService *master_service = nullptr; ///< The initial room/service.
    bool shutdown_pending = false; ///< Set when shutdown initiated.
    time_t startup_time = time (nullptr); ///< Time the service manager was instantiated.

    virtual bool hasPermission (PianodConnection &conn, SERVICECOMMAND command) override;
    virtual void handleCommand (PianodConnection &conn, SERVICECOMMAND command) override;
    virtual const FB_PARSE_DEFINITION *statements (void) override;

    void sendSourceReadyEvents (const Media::Source *source,
                                RESPONSE_CODE result);
    void sourceReady (const Media::Source *src);
    void sourceOffline (const Media::Source *src);
    void sourceRemoved (const Media::Source *src);

public:
    ServiceManager ();
    ~ServiceManager ();
    float periodic(void);

    void shutdown (bool immediate);
    bool flush (void);

    // Events
    void event (WaitEvent::Type type, const void *detail = nullptr, RESPONSE_CODE message = S_OK);
    /// @see event (3 parameter version).
    inline void event (WaitEvent::Type type, RESPONSE_CODE message = S_OK) {
        event (type, nullptr, message);
    };

    // Broadcast things
    void broadcast (RESPONSE_CODE message, User *target = nullptr);
    void broadcast (const Response &message, User *target = nullptr);
    void broadcastEffectivePrivileges ();
    void broadcastEffectivePrivileges (User *target);

    // Room management
    PianodService *createRoom (const std::string &name, const AudioSettings &audio,
                               FB_SERVICE_OPTIONS &options);
    void removeRoom (PianodService *service);

    // User management
    bool userIsOnline (const User *user);
};


extern ServiceManager *service_manager;
#endif
