///
/// Predicate handling.
/// Predicate parsing and interpretation.
///	@file		predicate.cpp - pianod2
///	@author		Perette Barella
///	@date		2015-10-12
///	@copyright	Copyright (c) 2015-2016 Devious Fish. All rights reserved.
///

#include <config.h>

#include <cstdlib>

#include <memory>
#include <unordered_set>
#include <exception>

#include <football.h>

#include "fundamentals.h"
#include "lookup.h"
#include "predicate.h"
#include "connection.h"
#include "filter.h"
#include "querylist.h"
#include "mediaunit.h"
#include "mediamanager.h"

using namespace std;

/// Functions and types for parsing and interpreting predicates.
namespace Predicate {
    /// Manners in which predicates may be specified.
    enum class MANNER {
        NAME,
        ID,
        WHERE,
        LIKE,
        SOURCE
    };

    /// Lookup table for manner names to enumeration.
    const LookupTable <MANNER> PredicateManners ( {
        { "name",           MANNER::NAME },
        { "id",             MANNER::ID },
        { "where",          MANNER::WHERE },
        { "like",           MANNER::LIKE },
        { "source",         MANNER::SOURCE }
    });

    /// Lookup table for optional type to filter field.
    const LookupTable <Filter::Field> SearchFields ( {
        { "any",            Filter::Field::Search },
        { "artist",         Filter::Field::Artist },
        { "album",          Filter::Field::Album },
        { "song",           Filter::Field::Title },
        { "playlist",       Filter::Field::Playlist },
        { "genre",          Filter::Field::Genre }
    });

    struct Predicate {
        unique_ptr <Filter> filter;
        Media::Source *source = nullptr;
    };

    static const FB_PARSE_DEFINITION predicate_source_parser_def [] = {
        { 1, "type {sourcetype} name {sourcename}" RECURSIONMANNER " {" PREDICATEEXPRESSIONTAG "}  ..." },
        { 2, "id {#sourceid}" RECURSIONMANNER " {" PREDICATEEXPRESSIONTAG "}  ..." },
        { 0, nullptr}
    };

    static FB_PARSER *predicate_source_parser;

    class Initializer {
    public:
        Initializer () {
            predicate_source_parser = fb_create_parser();
            bool ok = false;
            if (predicate_source_parser) {
                ok = fb_parser_add_statements (predicate_source_parser,
                                          predicate_source_parser_def);
            }
            if (!ok) {
                cerr << "Could not initialize predicate parser.\n";
                exit (1);
            }
        }
        ~Initializer () {
            fb_parser_destroy (predicate_source_parser);
            predicate_source_parser = nullptr;
        }
    };

    Initializer init;

    /** Parse a command's predicate and return a matching filter.
        Predicate forms are:
        - `ID {id} ...` (exact match on 1 or more IDs)
        - `NAME {name} ...` (exact match on 1 or more names)
        - `[type] LIKE {text} ...` (permuted on `text`)
        If `type` was not specified, each word matches any text field;
        otherwise, the words must match on the specified field.
        - `WHERE {expression} ...` (logical filter expression)
        - `SOURCE TYPE {type} NAME {name} {manner} {expression} ...`
          Executes one of the other manners of predicates against a specific source.

        This function requires the command be defined with the following named fields:
        - `manner` - to match the predicate keyword
        - `expression` - the start of the predicate parameter
        - `type` (optional) - the optional type for LIKE.
        Tag names vary slightly for playlist predicates.
        (It is a syntax error if other predicates have this.)
        @param conn The connection for which the predicate is evaluated.
        @param field The field to use for name searches and the 'like' default.
        @param tags A structure containing the field names to use. */
    static Predicate getFullPredicate (const PianodConnection &conn,
                                       Filter::Field field,
                                       const TagNames &tags = genericTags) {
        assert (havePredicate (conn, tags));
        assert (conn [tags.expression]);

        MANNER manner = PredicateManners [conn [tags.manner]];
        const char *data_type = tags.type ? conn [tags.type] : nullptr;

        Predicate pred;
        pred.source = conn.source();

        if (manner == MANNER::SOURCE) {
            Football::Connection::ConstReinterpretHandler handler {
                [field, &pred] (const Football::Connection &connection,
                                Football::Command command, const char *error) -> void {
                    if (command <= 0) {
                        throw CommandError (E_BAD_COMMAND, error);
                    }
                    auto id = connection ["sourceid"];
                    if (PredicateManners [connection [genericTags.manner]] == MANNER::ID) {
                        // Disregard specified source if looking up by ID.  Stupid user.
                        pred.source = media_manager;
                    } else if (id) {
                        pred.source = media_manager->get (atoi (id));
                    } else {
                        pred.source = media_manager->get (connection ["sourcetype"], connection ["sourcename"]);
                    }
                    if (!pred.source) {
                        throw CommandError (E_NOTFOUND, "Predicate source");
                    }
                    Predicate next_pred { getFullPredicate (static_cast <const PianodConnection &> (connection), field) };
                    pred.filter = std::move (next_pred.filter);
                }
            };
            conn.reinterpret (tags.expression, predicate_source_parser, handler);
            return pred;
        }

        if (data_type && manner != MANNER::LIKE)
            throw CommandError (E_TYPE_DISALLOWED);

        if (manner == MANNER::WHERE) {
            pred.filter.reset (new Filter (conn.argvFromUntokenized (tags.expression)));
            return pred;
        }

        vector<string> items { conn.argvSlice (tags.expression) };
        switch (manner) {
            case MANNER::ID:
                // Ideally, we should never get here.  But in case ID lookup
                // isn't implemented, or user enters a bizarre request involving
                // a specific predicate source and IDs, fall back on a filter.
                pred.filter.reset (new ListFilter (items, Filter::Field::Id));
                pred.source = media_manager;
                break;
            case MANNER::NAME:
                pred.filter.reset (new ListFilter (items, Filter::Field::Name));
                break;
            case MANNER::LIKE:
            {
                if (data_type)
                    field = SearchFields [data_type];
                pred.filter.reset (new PermutedFilter (items, field));
                break;
            }
            case MANNER::WHERE:
            default:
                assert (!"Unhandled manner");
                throw CommandError (E_BUG);
        }
        return pred;
    }

    /** Return just the filter portion of a predicate, for external use.
        @see getFullPredicate. */
    unique_ptr<Filter> getPredicate (const PianodConnection &conn,
                                     Filter::Field field,
                                     const TagNames &tags) {
        Predicate pred = getFullPredicate (conn, field, tags);
        return std::move (pred.filter);
    }



    /** Specialization for getting things by ID.
        @param conn The connection for which the predicate is evaluated.
        @param tags A structure containing the field names to use.
        @throw CommandError if the ID does not match any item.
        @note Specifying the same ID multiple items will produce duplicates
        in the returned list return. */
    static ThingieList getPredicateIdItems (const PianodConnection &conn,
                                            const TagNames &tags = genericTags) {
        assert (conn.argvEquals(tags.manner, "id"));
        vector<string> ids { conn.argvSlice (tags.expression) };

        ThingieList results;
        for (const auto id : ids) {
            MusicThingie *thing = media_manager->getAnythingById(id);
            if (!thing)
                throw CommandError (E_NOTFOUND, id);
            results.push_back (thing);
        }
        return results;
    }







    /*
     *                  Generic things
     */


    /** Gather a list of assorted things specified by a predicate, which must be
        present.  This differs from the standard call in that when querying
        the media manager, it does not consider a single source's inability to
        complete a query as a failure.
        @param pred The predicate, including source and filter.
        @param search_what Specifies manner of search.
        @return The things specified.
        @throw CommandError if no matching things are found.
        @throw Query::impossible If no sources could handle the query. */
    static ThingieList getPartialSpecifiedThings (Predicate &pred,
                                                  SearchRange search_what) {
        assert (pred.source == media_manager);

        ThingieList matching_things;
        bool for_request = forRequest (search_what);
        bool any_possible = false;
        bool any_impossible = false;
        for (auto &source : *media_manager) {
            if (source.second != media_manager) {
                try {
                    ThingieList source_things { source.second->getSuggestions (*pred.filter.get(), search_what) };
                    any_possible = true;
                    matching_things.join (source_things);
                } catch (const Query::impossible &e) {
                    any_impossible = true;
                } catch (const CommandError &e) {
                    if (e.reason() != E_MEDIA_ACTION)
                        throw;
                    any_impossible = true;
                }
                PlaylistList source_playlists { source.second->getPlaylists (*pred.filter.get()) };
                for (auto playlist : source_playlists) {
                    matching_things.push_back (playlist);
                }
            }
        }
        if (any_impossible && !any_possible) {
            throw Query::impossible();
        }

        if (matching_things.empty())
            throw CommandError (E_NOTFOUND);
        return matching_things;
    }


    /** Gather a list of assorted things specified by a predicate.
        The predicate must be present.
        @param conn The connection for which the predicate is being interpreted.
        @return The things specified.
        @throw CommandError if no matching things are found. */
    ThingieList getSpecifiedThings (const PianodConnection &conn,
                                    SearchRange search_what) {
        assert (havePredicate (conn));
        if (conn.argvEquals(genericTags.manner, "id"))
            return getPredicateIdItems (conn);

        Predicate pred { getFullPredicate (conn, Filter::Field::Search) };
        if (pred.source == media_manager && conn.argvEquals (PREDICATEFULFILLMENTTAG, "discretionary")) {
            return getPartialSpecifiedThings (pred, search_what);
        }

        ThingieList matching_things { pred.source->getSuggestions (*pred.filter.get(), search_what) };
        PlaylistList matching_playlists { pred.source->getPlaylists (*pred.filter.get()) };
        bool for_request = forRequest (search_what);
        for (auto playlist : matching_playlists) {
            if (!for_request || playlist->canQueue()) {
                matching_things.push_back (playlist);
            }
        }

        if (matching_things.empty() && !matching_playlists.empty()) {
            throw CommandError (E_MEDIA_ACTION, "Cannot request");
        }
        if (matching_things.empty())
            throw CommandError (E_NOTFOUND);
        return matching_things;
    }


    /** Interpret a request for an unspecified type of thing and return it.
        Predicate must be present.
        @return The requested thing.
        @throw CommandError if no things or multiple things are found. */
    MusicThingie *getSpecifiedThing (const PianodConnection &conn) {
        assert (havePredicate (conn));
        ThingieList matches { getSpecifiedThings (conn) };
        if (matches.size() > 1)
            throw CommandError (E_AMBIGUOUS);
        return matches.front();
    }






    /*
     *                  Playlists
     */

    const TagNames playlistTags {
        PLAYLISTTYPETAG,
        PLAYLISTMANNERTAG,
        PLAYLISTEXPRESSIONTAG
    };

    /** Get playlist items by id.
     @param conn The connection for which the predicate is evaluated. */
    static PlaylistList getSpecifiedPlaylistsById (const PianodConnection &conn) {
        ThingieList things { getPredicateIdItems (conn, playlistTags) };
        PlaylistList playlists;
        for (auto thing : things) {
            auto playlist = thing->asPlaylist();
            if (!playlist) {
                auto song = thing->asSong();
                if (song) {
                    playlist = song->playlist();
                    if (!playlist) {
                        throw CommandError (E_NO_ASSOCIATION, thing->id());
                    }
                }
            }
            if (!playlist) {
                throw CommandError (E_WRONGTYPE, thing->id());
            }
            playlists.push_back (playlist);
        }
        return playlists;
    }

    /** Interpret a list of playlists specified by a predicate.
        @param conn The connection for which the predicate is being interpreted.
        @return The playlists specified.  If the predicate is missing,
        returns all playlists for the current source.
        @throw CommandError if no matching playlists are found. */
    PlaylistList getSpecifiedPlaylists (const PianodConnection &conn) {
        if (!havePredicate (conn, playlistTags))
            return conn.source()->getPlaylists();
        if (conn.argvEquals(playlistTags.manner, "id"))
            return getSpecifiedPlaylistsById (conn);

        Predicate pred { getFullPredicate (conn, Filter::Field::Playlist, playlistTags) };
        PlaylistList matching_playlists = pred.source->getPlaylists (*pred.filter.get());
        if (matching_playlists.empty())
            throw CommandError (E_NOTFOUND);

        return matching_playlists;
    }



    /** Gather a request for a single playlist and return it.
        Predicate must be present.
        @return The requested playlist, or nullptr.
        @throw CommandError if no playlists or multiple playlists are found.*/
    PianodPlaylist *getSpecifiedPlaylist (const PianodConnection &conn) {
        assert (havePlaylistPredicate (conn));
        PlaylistList matches { getSpecifiedPlaylists (conn) };
        if (matches.size() > 1)
            throw CommandError (E_AMBIGUOUS);
        return matches.front();
    }





    /*
     *                  Songs
     */

    const TagNames &songTags = genericTags;

    /** Interpret a list of songs specified by a predicate.
        @param conn The connection for which the predicate is being interpreted.
        @return The songs specified or belonging to the artists, albums, or
        playlists specified.
        @throw CommandError if no matching songs are found. */
    SongList getSpecifiedSongs (const PianodConnection &conn, SearchRange search_what) {
        assert (search_what == SearchRange::KNOWN ||
                search_what == SearchRange::REQUESTS);

        ThingieList things = getSpecifiedThings (conn, search_what);
        if (things.empty())
            throw CommandError (E_NOTFOUND);


        /* Songs may be returned from searches or from membership in playlists,
           causing duplication.  Deduplicate the results, retaining ordering. */
        SongList songs;
        songs.reserve (things.size());
        unordered_set<string> present;
        for (auto thing : things) {
            if (!thing->canQueue()) {
                throw CommandError (E_MEDIA_ACTION, "Requests not supported");
            }
            SongList expanded = thing->songs();
            for (auto song : expanded) {
                const string &id = song->id();
                if (present.find (id) == present.end()) {
                    present.insert (id);
                    songs.push_back (song);
                }
            }
        }
        return songs;
    }



    /** Gather a request for a single song and return it.
        Predicate must be present.
        @return The requested song, or nullptr.
        @throw CommandError if no songs or multiple songs are found.*/
    PianodSong *getSpecifiedSong (const PianodConnection &conn) {
        assert (haveSongPredicate (conn));
        SongList matches { getSpecifiedSongs (conn) };
        if (matches.size() > 1)
            throw CommandError (E_AMBIGUOUS);
        return matches.front();
    }

}

