///
/// Parsnip serialization.
/// Serialization and parsing for JSON protocol.
/// @file       parsnip_json.cpp - Parsnip serialization & parsing
/// @author     Perette Barella
/// @date       2020-02-27
/// @copyright  Copyright 2020 Devious Fish. All rights reserved.
///

#include <config.h>

#include <ios>
#include <ostream>
#include <istream>
#include <sstream>

#include <cctype>

#include "parsnip.h"
#include "parsnip_helpers.h"

namespace Parsnip {

    /** Write a string to a stream, literal-encoding special characters.
        @param target The stream to write to.
        @param value The string to write. */
    static void json_encode_string (std::ostream &target, const std::string &value) {
        target << '"';
        for (auto ch : value) {
            switch (ch) {
                case '\"':
                    target << "\\\"";
                    break;
                case '\\':
                    target << "\\\\";
                    break;
#ifdef PARSNIP_JSON_ENCODE_SOLIDUS
                case '/':
                    // This *may* be escaped, but is not required.
                    target << "\\/";  // Per json.org
                    break;
#endif
                case '\b':
                    target << "\\b";
                    break;
                case '\f':
                    target << "\\f";
                    break;
                case '\n':
                    target << "\\n";
                    break;
                case '\r':
                    target << "\\r";
                    break;
                case '\t':
                    target << "\\t";
                    break;
                default:
                    if (ch >= 0 && ch < 32) {
                        target << "\\u00" << hexdigit (ch >> 4) << hexdigit (ch & 0xf);
                    } else {
                        // Per RFC 8259, JSON shall be UTF-8 encoded.
                        // Since we do everything UTF-8, we're grand.
                        target << ch;
                    }
                    break;
            }
        }
        target << '"';
    }

    /** Write serial data encoded using JSON.
        @param target Stream to which to write data.
        @param indent Starting indentation for elements.
        @param suppress If true, suppress indent on first line.
        @returns The output stream. */
    std::ostream &Data::toJson (std::ostream &target, int indent, bool suppress) const {
        if (!suppress) {
            do_indent (target, indent);
        }
        switch (datatype) {
            case Type::Dictionary: {
                target << '{';
                bool first = true;
                for (const auto &value : *(data.dictionary)) {
                    if (!first) {
                        target << ',';
                    }
                    do_newline (target, indent);
                    do_indent (target, indent + 2);
                    json_encode_string (target, value.first);
                    target << ':';
                    value.second.toJson (target, indent + 2, true);
                    first = false;
                }
                do_newline (target, indent);
                do_indent (target, indent);
                target << '}';
                break;
            }
            case Type::List: {
                target << '[';
                bool first = true;
                for (const auto &value : *(data.list)) {
                    if (!first) {
                        target << ',';
                    }
                    do_newline (target, indent);
                    value.toJson (target, indent + 2);
                    first = false;
                }
                do_newline (target, indent);
                do_indent (target, indent);
                target << ']';
                break;
            }
            case Type::String:
            case Type::FlexibleString:
                json_encode_string (target, *(data.str));
                break;
            case Type::Real:
                target << data.real;
                break;
            case Type::Integer:
                target << data.integer;
                break;
            case Type::Boolean:
                target << (data.boolean ? "true" : "false");
                break;
            case Type::Null:
                target << "null";
                break;
        }
        return (target);
    }


    std::ostream &Data::dumpJson (const std::string &intro, std::ostream &target) const {
        target << intro << ": ";
        toJson (target, intro.size() + 2, true) << std::endl;
        return target;
    }


    /** Return serial data encoded using JSON.
        @param indent Starting indentation for elements.
        @returns A string. */
    std::string Data::toJson (int indent) const {
        std::ostringstream collector;
        toJson (collector, indent);
        return collector.str();
    }


    /** Read from a stream until a non-whitespace character is found.
        @throw DataFormatError if EOF is encountered. */
#ifdef PARSNIP_JSON_COMMENTS
    std::istream::int_type next_non_whitespace (std::istream &from) {
        while (from.good()) {
            std::istream::int_type ch = from.get();
            if (ch == std::istream::traits_type::eof()) {
                throw DataFormatError ("Unexpected end of file");
            } if (isspace (ch)) {
                continue;
            } else if (ch != '/') {
                return ch;
            }
            std::istream::int_type next = from.peek();
            if (next == '/') {
                // C++ style comment: read to end of line
                do {
                    ch = from.get();
                    if (ch == std::istream::traits_type::eof()) {
                        throw DataFormatError ("Unexpected end of file");
                    }
                } while (from.good() && (ch != '\r' && ch != '\n'));
            } else if (next == '*') {
                /* C-style comment */
                from.get();
                do {
                    if (ch == std::istream::traits_type::eof()) {
                        throw DataFormatError ("Unexpected end of file");
                    }
                    ch = from.get();
                } while (from.good() && (ch != '*' || from.peek() != '/'));
                from.get();
            } else {
                return ch;
            }
        };
        throw std::ios_base::failure (strerror (errno));
    };
#else
    std::istream::int_type next_non_whitespace (std::istream &from) {
        std::istream::int_type ch;
        while (from.good()) {
            ch = from.get();
            if (ch == std::istream::traits_type::eof()) {
                throw DataFormatError ("Unexpected end of file");
            } else if (!isspace (ch)) {
                return ch;
            }
        }
        throw std::ios_base::failure (strerror (errno));
    }
#endif


    /** Parse hexadecimal digits from a stream.
        @param from The stream to take digits from.
        @param count The number of digits expected. */
    static int parse_hex_digits (std::istream &from, int count) {
        int result = 0;
        while (count-- > 0) {
            std::istream::int_type ch = tolower (from.get());
            if (!isxdigit (ch)) {
                if (ch == std::istream::traits_type::eof()) {
                    throw DataFormatError (std::string ("End of file in hex digit-stream"));
                }
                throw DataFormatError (std::string ("Non-hex digit: ") + char (ch));
            }
            result = (result << 4) | (strchr (hex_digits, ch) - hex_digits);
        }
        return result;
    }

    /** Parse a string out of a JSON stream.
        @param from The stream to parse.
        @return A string with literals converted to their real values. */
    static const std::string parse_json_string (std::istream &from) {
        std::string value;
        std::istream::int_type ch;
        while ((ch = from.get()) != '\"') {
            if (ch == std::istream::traits_type::eof()) {
                throw DataFormatError ("End of file while reading string: " + value);
            } else if (ch == '\\') {
                ch = from.get();
                switch (ch) {
                    case '\\':
                    case '/':
                    case '\"':
                    case '\'':
                        // Do nothing
                        break;
                    case 'f':
                        ch = '\f';
                        break;
                    case 'n':
                        ch = '\n';
                        break;
                    case 'r':
                        ch = '\r';
                        break;
                    case 't':
                        ch = '\t';
                        break;
                    case 'u':
                        ch = parse_hex_digits (from, 4);
                        if (ch < 0 || ch >= 32) {
                            // No reason for \u with UTF-8, so refuse except control chars.
                            throw DataFormatError ("\\u-encode        d non-control character");
                        }
                        break;
                    default:
                        throw DataFormatError (std::string ("Invalid literal: ") + std::to_string (ch));
                }
            }
            value += ch;
        }
        return value;
    }

    /** Parse a keyword out of a JSON string.
        @param from The stream to read.
        @return A Data object corresponding to the keyword.
        @throws DataFormatError if the keyword is invalid. */
    static Data parse_json_inline_text (std::istream &from) {
        std::string word;
        std::istream::int_type ch;

        while (from.good() && (ch = from.get()) != std::istream::traits_type::eof() && isalpha (ch)) {
            word += ch;
        }
        // Reset the stream and put back unwanted characters that were read.
        from.clear();
        if (ch != std::istream::traits_type::eof()) {
            from.unget();
        }
        assert (!word.empty());
        if (word == "true") {
            return Data{true};
        } else if (word == "false") {
            return Data{false};
        } else if (word == "null") {
            return Data{};
        }
        throw DataFormatError ("Unknown word: " + word);
    }

    /** Parse a numeric value in a JSON stream.
        @param from The stream
        @return The number, wrapped in a Data object.
        @throws DataFormatError if there isn't a number. */
    static Data parse_json_inline_value (std::istream &from) {
        std::string number;
        std::istream::int_type ch;

        while (from.good() && (ch = from.get()) != std::istream::traits_type::eof()) {
            if (!isdigit (ch) && (ch != '.') && (ch != 'e') && (ch != 'E') && (ch != '-') && (ch != '+')) {
                break;
            }
            number += ch;
        }
        // Reset the stream and put back unwanted characters that were read.
        from.clear();
        if (ch != std::istream::traits_type::eof()) {
            from.unget();
        }

        // Try translating to an integer, if fails to parse, try again as a real.
        char *err;
        long int integer = strtol (number.c_str(), &err, 10);
        if (*err == '\0') {
            return Data{integer};
        }
        double real = strtod (number.c_str(), &err);
        if (*err == '\0') {
            return Data{real};
        }
        throw DataFormatError (number.c_str());
    }

    static Data parse_json_element (std::istream &from);

    /** Parse a JSON dictionary from a stream.
        @param from The stream to read.
        @return A Data object containing the dictionary.
        @throws DataFormatError if the dictionary is invalid. */
    static Data parse_json_dictionary (std::istream &from) {
        Data dict{Data::Dictionary};
        std::istream::int_type ch = next_non_whitespace (from);
        while (ch != '}') {
            if (ch != '\"') {
                throw DataFormatError (std::string ("Expected \'}\' or \'\"\', got \'") + char (ch) + '\'');
            }
            const std::string name = parse_json_string (from);
            require_character (from, ':');
            dict[name] = parse_json_element (from);
            ch = next_non_whitespace (from);
            if (ch == ',') {
                // Tolerates a spare trailing comma before end of dictionary, which breaks spec.
                ch = next_non_whitespace (from);
            } else if (ch != '}') {
                throw DataFormatError (std::string ("Expected \',\' or \'}\', got \'") + char (ch) + '\'');
            }
        }
        return dict;
    }

    /** Parse a list from a JSON stream.
        @param from The stream to read from.
        @return A Data object containing the list. */
    static Data parse_json_list (std::istream &from) {
        Data list{Data::List};
        std::istream::int_type ch = next_non_whitespace (from);
        while (ch != ']') {
            from.unget();
            list.push_back (parse_json_element (from));
            ch = next_non_whitespace (from);
            if (ch == ',') {
                // Tolerates a spare trailing comma before end of list, which breaks spec.
                ch = next_non_whitespace (from);
            } else if (ch != ']') {
                throw DataFormatError (std::string ("Expected \',\' or \']\', got \'") + char (ch) + '\'');
            }
        }
        return list;
    }

    /** Extract data from a JSON stream.
        This function looks at the next non-whitespace character in the stream
        to determine the data type, then calls the appropriate handler to do the work.
        @param from The stream to parse.
        @return A Data object with the data extracted.
        @throws DataFormatError If the stream is invalid.
        @throws DataRangeError If the stream contains unrepresentable numbers. */
    static Data parse_json_element (std::istream &from) {
        std::istream::int_type ch = next_non_whitespace (from);
        switch (ch) {
            case '"':
                return Data{parse_json_string (from)};
            case '{':
                return parse_json_dictionary (from);
            case '[':
                return parse_json_list (from);
            case '1':
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
            case '8':
            case '9':
            case '0':
            case '-': {
                from.unget();
                return parse_json_inline_value (from);
            }
            default: {
                from.unget();
                if (isalpha (ch)) {
                    return parse_json_inline_text (from);
                }
                throw DataFormatError (std::string ("At ") + char (ch));
            }
        }
    }

    /** Parse a stream as JSON data.
        @param from The stream to parse.
        @param check_termination If true (default), validates stream terminates (EOF)
        after end of JSON data.  If false, stops reading after a complete JSON object
        has been read.
        @returns Data structure accoring to parsed string.
        @throws DataFormatError if JSON is mangled.
        @throws DataRangeError If the stream contains unrepresentable numbers. */
    Data parse_json (std::istream &from, bool check_termination) {
        // Exceptions don't work reliably on OS X.  Turn them off and just don't use them,
        // but restore exception state before return in case the caller had them on.
        std::ios_base::iostate prior_exceptions_state = from.exceptions();
        from.exceptions (std::istream::goodbit);
        Data data;
        try {
            data = parse_json_element (from);
        } catch (const std::istream::failure &e) {
            from.exceptions (prior_exceptions_state);
            throw DataFormatError ("Unexpected end of data");
        }
        if (check_termination) {
            std::istream::int_type ch;
            do {
                ch = from.get();
            } while (isspace (ch));
            if (ch != std::istream::traits_type::eof()) {
                from.exceptions (prior_exceptions_state);
                throw DataFormatError (std::string ("Trailing junk @ \'") + char (ch) + '\'');
            }
        }
        from.exceptions (prior_exceptions_state);
        return data;
    }

    /** Parse a string as a JSON stream.
        @param from The string to parse.
        @returns Data structure accoring to parsed string.
        @throws DataFormatError if JSON is mangled.
        @throws DataRangeError If the stream contains unrepresentable numbers. */
    Data parse_json (const std::string &from) {
        std::istringstream source{from};
        return (parse_json (source));
    }

}  // namespace Parsnip
