///
/// Parsnip serialization - stream I/O and parsing helper functions.
/// @file       parsnip_helpers.h - Parsnip serialization & parsing
/// @author     Perette Barella
/// @date       2019-05-01
/// @copyright  Copyright 2019-2020 Devious Fish. All rights reserved.
///

#pragma once

namespace Parsnip {

    /// Digits used for encoding/decoding hexadecimal.
    extern const char hex_digits[];

    /// Helper function: Retrieve a hexadecimal digit.
    static inline char hexdigit (int digit) {
        assert (digit >= 0 && digit <= 15);
        return hex_digits[digit];
    }

    /** Generate some indentation.
        @param target The stream to write indentation to.
        @param amount The amount of indentation, in spaces.
        Negative numbers = none. */
    static inline void do_indent (std::ostream &target, int amount) {
        for (int i = 0; i < amount; i++) {
            target << ' ';
        }
    }

    /** Write a newline to a stream.
        @param amount Amount of indent.  If negative, we're not indenting,
        so don't worry about newlines either. */
    static inline void do_newline (std::ostream &target, int amount) {
        if (amount >= 0) {
            target << std::endl;
        }
    }

    std::istream::int_type next_non_whitespace (std::istream &from);

    /** Verify the next non-whitespace character in a stream is what's expected.
        @throws DataFormatError if the character isn't the one expected or EOF. */
    static inline void require_character (std::istream &from, std::istream::int_type expect) {
        auto ch = next_non_whitespace (from);
        if (ch != expect) {
            throw DataFormatError (std::string ("Expected: \'") + char (expect) + "\', got \'" + char (ch) + '\'');
        }
    }

}  // namespace Parsnip
