///
/// Command handlers for tone generator source.
///	@file		tonegencommand.cpp - pianod
///	@author		Perette Barella
///	@date		2014-12-04
///	@copyright	Copyright 2014-2020 Devious Fish.  All rights reserved.
///

#include <config.h>

#include <cstdio>
#include <cassert>

#include <football.h>

#include "fundamentals.h"
#include "sources.h"
#include "connection.h"
#include "user.h"
#include "users.h"
#include "response.h"
#include "mediamanager.h"
#include "mediaparameters.h"
#include "musiclibraryparameters.h"

#include "tonegen.h"

using namespace std;

typedef enum test_tone_command_t {
    TTADDSOURCE = CMD_RANGE_TONEGENERATOR,
    TTEXISTING
} TTCOMMAND;

static const FB_PARSE_DEFINITION statementList[] = {
    { TTADDSOURCE,      "tonegenerator activate [{parameters}] ..." },
    { TTEXISTING,       "tonegenerator use {name} [{parameters}] ..." },
    { CMD_INVALID,      NULL }
};


using namespace MusicLibrary;
class ToneGeneratorParameterParser : public ParameterParser <ToneGenerator::Parameters, int> {
    enum Option {
        CHANNELIZED = 5100,
    };

    const ParseDefinition tone_generator_options [2] = {
        { CHANNELIZED,  "discrete channels ..." }, // Ask for discrete channels
        { CMD_INVALID, NULL }
    };
    
    virtual int handleOption (int cmd, ToneGenerator::Parameters &result) override {
        switch (cmd) {
            case RESCAN_BEHAVIOR:
                return FB_PARSE_FAILURE;
            case CHANNELIZED:
                result.discrete_channels = true;
                return FB_PARSE_SUCCESS;
        }
        return ParameterParser<ToneGenerator::Parameters, int>::handleOption(cmd, result);
    }
public:
    ToneGeneratorParameterParser () {
        addStatements (tone_generator_options);
    }
};
ToneGeneratorParameterParser parameterParser;

/// Interpreter for tonegenerator commands.
class ToneGeneratorCommands : public Football::Interpreter<PianodConnection, TTCOMMAND> {
private:
    virtual bool hasPermission (PianodConnection &conn, TTCOMMAND command);
    virtual void handleCommand (PianodConnection &conn, TTCOMMAND command);
    virtual const FB_PARSE_DEFINITION *statements (void) {
        return statementList;
    };
};

bool ToneGeneratorCommands::hasPermission (PianodConnection &conn, TTCOMMAND) {
    return conn.havePrivilege (Privilege::Service);
};

void ToneGeneratorCommands::handleCommand (PianodConnection &conn, TTCOMMAND command) {
    switch (command) {
        case TTADDSOURCE:
        {
            if (!conn.user) {
                conn << E_LOGINREQUIRED;
                return;
            }
            ToneGenerator::Parameters p (Ownership::Type::PUBLISHED, conn.user);
            if (::parameterParser.interpret(conn.argvFrom ("parameters"),
                                          p, &conn) == FB_PARSE_SUCCESS) {
                Media::Source *m = new ToneGenerator::Source (p);
                media_manager->add (m, conn);
            }
            return;
        }
        case TTEXISTING:
        {
            UserData::StringDictionary *persisted;
            User *owner;
            RESPONSE_CODE status = user_manager->findStoredSource (SOURCE_NAME_TONEGENERATOR,
                                                                   conn.argv ("name"),
                                                                   conn.user,
                                                                   &persisted,
                                                                   &owner);
            if (status != S_OK) {
                conn << status;
                return;
            }
            try {
                ToneGenerator::Parameters params (*persisted);
                params.owner = owner;
                if (::parameterParser.interpret (conn.argvFrom ("parameters"), params, &conn) == FB_PARSE_SUCCESS) {
                    Media::Source *m = new ToneGenerator::Source (params);
                    media_manager->add (m, conn, owner != conn.user);
                }
            } catch (const invalid_argument &e) {
                conn << Response (E_CREDENTIALS, e.what());
            }
            return;
        }
        default:
            conn << E_NOT_IMPLEMENTED;
            flog (LOG_WHERE (LOG_WARNING), "Unimplemented command ", command);
            break;
    }
}

static ToneGeneratorCommands interpreter;

void register_tonegenerator_commands (PianodService *service) {
    service->addCommands(&interpreter);
}

void restore_tonegenerator_source (UserData::StringDictionary *persisted, User *user) {
    Media::Source *m = nullptr;
    try {
        ToneGenerator::Parameters params (*persisted);
        params.owner = user;
        m = new ToneGenerator::Source (params);
        if (!media_manager->add (m))
            delete m;
    } catch (...) {
        delete m;
        throw;
    }
    return;
}
