///
/// Functions that deal with all the possible sources.
///	@file		sources.cpp - pianod2
///	@author		Perette Barella
///	@date		2016-03-22
///	@copyright	Copyright (c) 2016-2020 Devious Fish. All rights reserved.
///

#include <config.h>

#include <iostream>
#include <string>

#include <football.h>

#include "utility.h"
#include "datastore.h"
#include "connection.h"
#include "users.h"

#include "sources.h"

#ifdef WITH_PANDORA
#include <curl/curl.h>
#endif


/// Interface between the general pianod code and the individual sources,
/// which may or may not be present in any particular build.
namespace Sources {
    using namespace std;

    /** Get a list of source type names.
        @return Vector with source names (except manager). */
    const vector<string> &sourceNames () {
        static vector <string> names;
        if (names.empty()) {
#ifdef WITH_PANDORA
            names.push_back (SOURCE_NAME_PANDORA);
#endif
#ifdef WITH_TONEGENERATOR
            names.push_back (SOURCE_NAME_TONEGENERATOR);
#endif
#ifdef WITH_FILESYSTEM
            names.push_back (SOURCE_NAME_FILESYSTEM);
#endif
        }
        return names;
    }


    /** Check if a name is a source type name.
        @param name The name to check.
        @return True if name is a source type name other than the manager. */
    bool isSource (const std::string &name) {
        for (const auto &source : sourceNames()) {
            if (strcasecmp (name, source) == 0) {
                return true;
            }
        }
        return (false);
    };

    /** Register known source interpreters with a service.
        @param service The service to register with. */
    void registerCommands (PianodService *service) {
#ifdef WITH_PANDORA
        register_pandora_commands (service);
#endif
#ifdef WITH_TONEGENERATOR
        register_tonegenerator_commands (service);
#endif
#ifdef WITH_FILESYSTEM
        register_filesystem_commands (service);
#endif
    }

    /** Restore a source.
        @param source_data The parameters for the source.
        @param owner The user to whom the source belongs. */
    void restore (UserData::DataStore *source_data, User *owner) {
        const string &source_name = source_data->origin();
        
        auto *persisted = dynamic_cast<UserData::StringDictionary *> (source_data);
        assert (persisted);

#ifdef WITH_PANDORA
        if (strcasecmp (source_name, SOURCE_NAME_PANDORA) == 0)
            restore_pandora_source (persisted, owner);
#endif
#ifdef WITH_TONEGENERATOR
        if (strcasecmp (source_name, SOURCE_NAME_TONEGENERATOR) == 0)
            restore_tonegenerator_source (persisted, owner);
#endif
#ifdef WITH_FILESYSTEM
        if (strcasecmp (source_name, SOURCE_NAME_FILESYSTEM) == 0)
            restore_filesystem_source (persisted, owner);
#endif
    }

    /** Report sources, libraries they use and their versions.
        The currently unused parameter represents a verbosity level. */
    void report (int) {
        cerr << "  Sources:\n";
#ifdef WITH_PANDORA
        cerr << "    pandora\n";
        {
            const curl_version_info_data *version = curl_version_info (CURLVERSION_NOW);
            cerr << "      curl compile version " << LIBCURL_VERSION
            << "\n      curl runtime version " << version->version << '\n';
        }
#endif
#ifdef WITH_TONEGENERATOR
        cerr << "    tonegenerator\n";
#endif
#ifdef WITH_FILESYSTEM
        cerr << "    filesystem\n";
#endif
    }
}
