///
/// Essential Pandora data types.
/// @file       mediaunits/pandora/pandoratypes.h - pianod project
/// @author     Perette Barella
/// @date       2020-03-23
/// @copyright  Copyright 2020 Devious Fish. All rights reserved.
///

#pragma once

#include <config.h>

#include <string>
#include <unordered_map>

#include <parsnip.h>

#include "fundamentals.h"
#include "musictypes.h"
#include "mediaparameters.h"
#include "encapmusic.h"
#include "retainer.h"

namespace Pandora {
    class Source;
    class Station;

    extern const Rating ThumbsDown;
    extern const Rating ThumbsUp;
    
    namespace Key {
        extern const char *TrackToken;
    }

    std::string pandora_to_music_id (const std::string &musicId);

    /// Class for Pandora songs.  Constructed from annotations.
    class Song : public EncapsulatedSong {
    protected:
        inline Source *pandora() const {
            return reinterpret_cast<Source *> (source());
        }

    public:
        // Deal with Pandora's ID mess by storing eveything as pandoraId,
        // but providing pandoraId and musicId setters/getters.
        inline void artistId (const std::string &) = delete;
        void artistId (const char *) = delete;
        inline void artistPandoraId (const std::string &value) {
            EncapsulatedSong::artistId (value);
        }
        inline const std::string &artistPandoraId() const {
            return EncapsulatedSong::artistId();
        }
        std::string artistMusicId() const;
        void artistMusicId (const std::string &value);

        void songId (const std::string &) = delete;
        void songId (const char *) = delete;
        inline void songPandoraId (const std::string &value) {
            EncapsulatedSong::songId (value);
        }
        inline const std::string &songPandoraId() const {
            return EncapsulatedSong::songId();
        }
        std::string songMusicId() const;
        void songMusicId (const std::string &value);

        // Constructors
        Song (Source *owner, const Parsnip::Data &message);
        Song (const Song &dupe);

        // API support
        virtual RatingScheme ratingScheme (void) const override;
        virtual RESPONSE_CODE rate (Rating value, User *user) override;
        virtual Rating rating (const User *user) const override;
        virtual RESPONSE_CODE rateOverplayed (User *) override;
        virtual bool canSkip (time_t *whenAllowed = nullptr) override;

        // Internal support
        void unabridge (Song *other, bool bidirectional = true);
        virtual Parsnip::Data persist () const;

    protected:
        void expireStation (void) const;
        Song (Source *owner);
    };

    /// Playable song: a song with playback information.
    class PlayableSong : public Song {
        friend Source;
        friend Song;

    private:
        std::string audio_url;
        std::string track_token;
        double audio_gain;
        bool is_fresh {false}; ///< True if track_token is unused, false if expired.

    public:
        PlayableSong (Source *owner, const Parsnip::Data &message);
        PlayableSong (Source *owner, const Parsnip::Data &message, bool persist_flag);
        inline const std::string &trackToken() const {
            return track_token;
        }
        virtual bool canQueue() const override;
        virtual Parsnip::Data persist () const override;
    };
    
    /** A song that contains rating information.  Used to move feedback
        IDs from JSON parsing to a station, which manages it long-term.
        The field is unused after that. */
    class SongRating : public Song {
        friend class Station;
        std::string feedback_id;

    public:
        SongRating (Source *owner, const Parsnip::Data &message);
    };

    /// A song contructed from station seed information.
    class SongSeed : public Song {
    public:
        SongSeed (Source *owner, const Parsnip::Data &message, Station *station);
    };

    class Advert : public EncapsulatedSong {
        friend class Source;
        friend class AdvertWrapper;
    private:
        using NotificationTarget = std::vector<std::string>;
        NotificationTarget on_pause;
        NotificationTarget on_resume;
        NotificationTarget on_quarters [5];

        std::string audio_url;
    public:
        Advert (Source *owner, const Parsnip::Data &message);

        // API support
        virtual RatingScheme ratingScheme (void) const override;
        virtual RESPONSE_CODE rate (Rating value, User *user) override;
        virtual Rating rating (const User *user) const override;
        virtual RESPONSE_CODE rateOverplayed (User *) override;
        virtual bool canSkip (time_t *whenAllowed = nullptr) override;
    };

    /// An album.  Albums seem to exist with only Pandora IDs; there
    /// are no music IDs for them.  Hence, no ID mess.
    class Album : public EncapsulatedAlbum {
    protected:
        inline Source *pandora() const {
            return reinterpret_cast<Source *> (source());
        }

    public:
        Album (Source *owner, const Parsnip::Data &message);
        virtual SongList songs () override;
    };

    /// Class for Pandora artists.  Constructed from annotations.
    class Artist : public EncapsulatedArtist {
    protected:
        inline Source *pandora() const {
            return reinterpret_cast<Source *> (source());
        }
    public:
        inline void artistId (const std::string &) = delete;
        void artistId (const char *) = delete;
        inline void artistPandoraId (const std::string &value) {
            EncapsulatedArtist::artistId (value);
        }
        inline const std::string &artistPandoraId() const {
            return EncapsulatedArtist::artistId();
        }
        std::string artistMusicId() const;
        void artistMusicId (const std::string &value);

        Artist (Source *owner);
        Artist (Source *owner, const Parsnip::Data &message);
        virtual SongList songs () override;
    };

    /// A artist contructed from station seed information.
    class ArtistSeed : public Artist {
    public:
        ArtistSeed (Source *owner, const Parsnip::Data &message);
    };

    /// A class for Pandora genre seeds.
    class GenreSeed : public MetaPlaylist {
    public:
        inline void playlistId (const std::string &) = delete;
        void playlistId (const char *) = delete;
        inline void genrePandoraId (const std::string &value) {
            EncapsulatedPlaylist::playlistId (value);
        }
        inline const std::string &genrePandoraId() const {
            return EncapsulatedPlaylist::playlistId();
        }
        std::string genreMusicId() const;
        void genreMusicId (const std::string &value);

        GenreSeed (Source *owner, MusicThingie::Type type);
        GenreSeed (Source *owner, const Parsnip::Data &message);
    };

    /// A genre constructed from a suggestion.
    class GenreSuggestion : public GenreSeed {
        GenreSuggestion (Source *owner, const Parsnip::Data &message);
    };

    /// Pandora connection and source settings
    struct ConnectionParameters : public Media::SourceParameters {
        std::string proxy;
        std::string control_proxy;
        std::string username;
        std::string password;
        int pause_timeout{1800};
        int playlist_expiration{3600};
        int cache_minimum{4000};
        int cache_maximum{5000};

        ConnectionParameters();
        ConnectionParameters (const ConnectionParameters &params) = default;
        ConnectionParameters (ConnectionParameters &&params) = default;
        ConnectionParameters (const UserData::StringDictionary &src);
        virtual bool persist (UserData::StringDictionary &dest) const override;
    };
}  // namespace Pandora
