///
/// Datatypes to support pianod/libpiano interface for Pandora player.
///	@file		mediaunits/pandora/pandoratypes.cpp - pianod
///	@author		Perette Barella
///	@date		2014-11-30
///	@copyright	Copyright 2014-2020 Devious Fish.  All rights reserved.
///

#include <config.h>

#include <string>
#include <exception>

#include <cassert>

#include <errno.h>

#include "pandoratypes.h"
#include "pandora.h"

namespace Pandora {

    const Rating ThumbsDown = Rating::POOR;  ///< Rating value for negative Pandora feedback.
    const Rating ThumbsUp = Rating::GOOD;    ///< Rating value for positive Pandora feedback.

    const char *Key::TrackToken = "trackToken";

    /*
     *          Base type shared by songs, suggestions, seeds
     */

    /** Convert Pandora ID to music id.
        @param id The ID to convert.  Modified in place.
        @return The updated ID. */
    inline std::string &convert_pandora_to_music_id (const char *pandora_prefix, char music_prefix, std::string &id) {
        if (!id.empty()) {
            assert (pandora_prefix[0] && pandora_prefix[1] && !pandora_prefix[2]);
            assert (id[0] = pandora_prefix[0]);
            assert (id[1] = pandora_prefix[1]);
            assert (id[2] = ':');
            id.erase (1, 2);
            id[0] = music_prefix;
        }
        return id;
    }

    /** Convert music ID to Pandora id.
        @param id The ID to convert.  Modified in place.
        @return The updated ID. */
    inline std::string &convert_music_to_pandora_id (char music_prefix, const char *pandora_prefix, std::string &id) {
        if (!id.empty()) {
            assert (pandora_prefix[0] && pandora_prefix[1] && !pandora_prefix[2]);
            assert (id[0] == music_prefix);
            assert (id.size() < 3 || id[2] != ':');
            id[0] = ':';
            id.insert (0, pandora_prefix, 2);
        }
        return id;
    }

    /** Convert an artist's Pandora ID to a music ID.
        @param id Their Pandora ID.
        @return Their music ID. */
    static std::string pandora_to_music_artist_id (std::string id) {
        return convert_pandora_to_music_id ("AR", 'R', id);
    }

    /** Convert an artist's music ID to their Pandora ID.
        @param id The music ID.
        @return Their Pandora ID. */
    static std::string music_to_pandora_artist_id (std::string id) {
        return convert_music_to_pandora_id ('R', "AR", id);
    }

    /** Convert a song's Pandora ID to a music ID.
        @param id The Pandora ID.
        @return The music ID. */
    static std::string pandora_to_music_song_id (std::string id) {
        return convert_pandora_to_music_id ("TR", 'S', id);
    }

    /** Convert a song's music ID to a Pandora ID.
        @param id The music ID.
        @return The Pandora ID. */
    static std::string music_to_pandora_song_id (std::string id) {
        return convert_music_to_pandora_id ('S', "TR", id);
    }

    /** Convert a genre's Pandora ID to a music ID.
        @param id The Pandora ID.
        @return The music ID. */
    static std::string pandora_to_music_genre_id (std::string id) {
        return convert_pandora_to_music_id ("GE", 'G', id);
    }

    /** Convert a genre's music ID to a Pandora ID.
        @param id The music ID.
        @return The Pandora ID. */
    static std::string music_to_pandora_genre_id (std::string id) {
        return convert_music_to_pandora_id ('G', "GE", id);
    }

    /** Convert a Pandora ID to a music ID.
        @param pandora_id the ID to convert.
        @return The equivalent music ID.
        @throws invalid_argument if the value is neither a song nor artist ID. */
    std::string pandora_to_music_id (const std::string &pandora_id) {
        assert (pandora_id.size() >= 3);
        std::string type = pandora_id.substr (0, 3);
        if (type == "TR:") {
            return pandora_to_music_song_id (pandora_id);
        } else if (type == "AR:") {
            return pandora_to_music_artist_id (pandora_id);
        } else if (type == "GE:") {
            return pandora_to_music_genre_id (pandora_id);
        }
        flog (LOG_WHERE (LOG_ERROR), "Pandora ID not convertible to music ID: ", pandora_id);
        assert (!"Unconvertible Pandora ID.");
        throw std::invalid_argument (pandora_id);
    }

    /*
     *                  Pandora Song types
     */

    /// Get an artist's music ID.
    std::string Song::artistMusicId() const {
        return pandora_to_music_artist_id (artistPandoraId());
    }

    /// Set the artist's ID using a music ID.
    void Song::artistMusicId (const std::string &value) {
        EncapsulatedSong::artistId (music_to_pandora_artist_id (value));
    }

    /// Get a song's music ID.
    std::string Song::songMusicId() const {
        return pandora_to_music_song_id (songPandoraId());
    }

    /// Set the song's ID using a music ID.
    void Song::songMusicId (const std::string &value) {
        EncapsulatedSong::songId (music_to_pandora_song_id (value));
    }

    Song::Song (Source *owner) : EncapsulatedSong (owner){};

    /** Construct a Song from a Pandora annotation message.
        @param owner The source this song belongs to.
        @param message The details from which to construct the song. */
    Song::Song (Source *owner, const Parsnip::Data &message) : EncapsulatedSong (owner) {
        artistPandoraId (message["artistId"].asString());
        artist (message["artistName"].asString());
        albumTitle (message["albumName"].asString());
        albumId (message["albumId"].asString());
        title (message["name"].asString());
        songPandoraId (message["pandoraId"].asString());
        coverArtUrl (message["icon"]["artUrl"].asString());
        duration (message["duration"].asInteger());
        infoUrl (message["shareableUrlPath"].asString());
        trackNumber (message["trackNumber"].asInteger());
        if (message.contains ("stationId")) {
            playlist (owner->getStationByStationId (message["stationId"].asString()));
            if (!playlist()) {
                flog (LOG_WHERE (LOG_WARNING), "Unknown station: ", message["stationId"].asString());
            }
        }
    };

    /** Prepare the song in the same format it arrived. */
    Parsnip::Data Song::persist () const {
        Parsnip::Data song {Parsnip::Data::Dictionary,
            "artistId", artistPandoraId(),
            "artistName", artist(),
            "albumId", albumId(),
            "albumName", albumTitle(),
            "pandoraId", songPandoraId(),
            "name", title(),
            "duration", duration(),
            "shareableUrlPath", infoUrl(),
            "trackNumber", trackNumber(),
            "icon", Parsnip::Data {Parsnip::Data::Dictionary, "artUrl", coverArtUrl()}};
        if (playlist()) {
            song ["stationId"] = playlist()->playlistId();
        }
        return song;
    }

    /** Merge information back and forth between two Pandora songs.
        @param other Another song to use for missing information.
        @param bidirectional If true, fill in missing field in `other`
        if we have values for them. */
    void Song::unabridge (Song *other, bool bidirectional) {
        assert (id() == other->id());
        if (artistPandoraId().empty()) {
            artistPandoraId (other->artistPandoraId());
        }
        if (artist().empty()) {
            artist (other->artist());
        }
        if (albumTitle().empty()) {
            albumTitle (other->albumTitle());
        }
        if (albumId().empty()) {
            albumId (other->albumId());
        }
        if (title().empty()) {
            title (other->title());
        }
        if (coverArtUrl().empty()) {
            coverArtUrl (other->coverArtUrl());
        }
        if (duration() == 0) {
            duration (other->duration());
        }
        if (infoUrl().empty()) {
            infoUrl (other->infoUrl());
        }
        if (trackNumber() == 0) {
            trackNumber (other->trackNumber());
        }
        if (bidirectional) {
            other->unabridge (this, false);
        }
    }

    /** Create a new song by copying an existing song. */
    Song::Song (const Song &dupe) : EncapsulatedSong (dupe.source()) {
        songPandoraId (dupe.songPandoraId());
        artistPandoraId (dupe.artistPandoraId());
        artist (dupe.artist());
        albumTitle (dupe.albumTitle());
        albumId (dupe.albumId());
        title (dupe.title());
        coverArtUrl (dupe.coverArtUrl());
        duration (dupe.duration());
        infoUrl (dupe.infoUrl());
        trackNumber (dupe.trackNumber());
    }

    /** Construct a Song from a Pandora annotation message.
        @param owner The source this song belongs to.
        @param message The details from which to construct the song. */
    PlayableSong::PlayableSong (Source *owner, const Parsnip::Data &message) : Song (owner) {
        artist (message["artistName"].asString());
        artistMusicId (message["artistMusicId"].asString());
        albumTitle (message["albumTitle"].asString());  // as opposed to "albumName"
        title (message["songTitle"].asString());        // as opposed to "name"
        songPandoraId (message["pandoraId"].asString());
        const Parsnip::Data &album_art = message["albumArt"];
        if (album_art.size() > 0) {
            coverArtUrl (album_art[album_art.size() / 2]["url"].asString());
        }
        duration (message["trackLength"].asInteger());  // as opposed to "duration"
        infoUrl (message["songDetailURL"].asString());
        audio_url = message["audioURL"].asString();
        track_token = message["trackToken"].asString();
        assert (message["musicId"].asString() == songMusicId());
        // Gain is stored in a string for some reason, but be paranoid and fall back to a proper number...
        try {
            audio_gain = std::stod (message["fileGain"].asString(), nullptr);
        } catch (const std::exception &ex) {
            audio_gain = message["fileGain"].asDouble();
        }

        if (message.contains ("stationId")) {
            Station *station = owner->getStationByStationId (message["stationId"].asString());
            if (station) {
                playlist (station);
                Rating initial_rating = (message["rating"].asInteger() ? ThumbsUp : Rating::UNRATED);
                station->storeRating (songPandoraId(), initial_rating, "");
            } else {
                flog (LOG_WHERE (LOG_WARNING), "Unknown station: ", message["stationId"].asString());
            }
        }

        // Set expiration time, managed way up in a base class.
        expiration = time (nullptr) + owner->connectionParams()->playlist_expiration;
        is_fresh = true;
    }

    PlayableSong::PlayableSong (Source *owner, const Parsnip::Data &message, bool)
    : Song (owner, message) {
        track_token = message [Key::TrackToken].asString();
    }
    
    Parsnip::Data PlayableSong::persist () const {
        Parsnip::Data data = Song::persist();
        data [Key::TrackToken] = track_token;
        return data;
    }


    /** Determine if a skip is allowed, and if so, record one.
        A skip requires both the station and the source allow skips.
        @param when_allowed If not allowed, set to the time when one will be.
        @return True if allowed, false otherwise. */
    bool Song::canSkip (time_t *when_allowed) {
        time_t source_when;
        time_t station_when{0};
        bool source_can;
        bool station_can = true;
        source_can = pandora()->skips.canSkip (&source_when);
        Station *station = static_cast<Station *> (playlist());
        if (station) {
            station_can = station->skips.canSkip (&station_when);
        }
        if (source_can && station_can) {
            pandora()->skips.skip();
            if (station) {
                station->skips.skip();
            }
            return true;
        }
        // We need both to skip, so pick the further-out time.
        *when_allowed = (source_when > station_when ? source_when : station_when);
        return false;
    }

    RatingScheme Song::ratingScheme (void) const {
        return RatingScheme::Owner;
    };

    RESPONSE_CODE Song::rate (Rating value, User *user) {
        assert (isEditableBy (user));

        if (!playlist()) {
            throw CommandError (E_PLAYLIST_REQUIRED);
        }

        // Round the value to Pandora's thumbs.
        Rating new_rating = value;
        RESPONSE_CODE result = S_OK;
        if (value <= ThumbsDown) {
            if (value != ThumbsDown) {
                new_rating = ThumbsDown;
                result = S_ROUNDING;
            }
        } else if (value >= ThumbsUp) {
            if (value != ThumbsUp) {
                new_rating = ThumbsUp;
                result = S_ROUNDING;
            }
        } else if (value == Rating::NEUTRAL) {
            value = Rating::UNRATED;
        } else {
            throw CommandError (E_MEDIA_VALUE);
        }

        Station *station = static_cast<Station *> (playlist());
        std::string feedback_id;
        Rating prior_rating = station->getRating (songPandoraId(), &feedback_id);

        // If the song already has the desired rating, skip the work.
        if (new_rating == prior_rating) {
            return result;
        } else if (new_rating == Rating::UNRATED && feedback_id.empty()) {
            station->forceRefreshRatings (prior_rating == ThumbsUp);
            Rating prior_rating = station->getRating (songPandoraId(), &feedback_id);
            if (new_rating == prior_rating) {
                return result;
            }
        }

        Status status;
        if (new_rating == Rating::UNRATED) {
            if (feedback_id.empty()) {
                throw CommandError (E_BUG, "Missing feedback ID for rating.");
            }
            RequestDeleteFeedback delete_feedback (pandora(), feedback_id, prior_rating == ThumbsUp);
            status = pandora()->executeRequest (delete_feedback);
        } else {
            PlayableSong *playable = dynamic_cast<PlayableSong *> (this);
            if (!playable) {
                throw CommandError (E_WRONG_STATE, "Song not recently played");
            }
            station->takePossession();
            RequestAddFeedback add_feedback (pandora(), playable->track_token, new_rating == ThumbsUp);
            status = pandora()->executeRequest (add_feedback);
            if (status == Status::Ok) {
                feedback_id = add_feedback.getResponse();
            }
        }
        if (status == Status::Ok) {
            station->storeRating (songPandoraId(), new_rating, feedback_id);
            return result;
        }
        throw CommandError (E_NAK, status_strerror (status));
    }

    Rating Song::rating (const User *) const {
        if (!playlist()) {
            return Rating::UNRATED;
        }
        Station *station = static_cast<Station *> (playlist());
        return (station->getRating (songPandoraId()));
    }

    RESPONSE_CODE Song::rateOverplayed (User *) {
        PlayableSong *song = dynamic_cast<PlayableSong *> (this);
        if (!song) {
            throw CommandError (E_WRONG_STATE, "Not a recently played song");
        }
        RequestAddTiredSong mark_overplayed (pandora(), song->track_token);
        Status status = pandora()->executeRequest (mark_overplayed);
        if (status != Status::Ok) {
            throw CommandError (E_NAK, status_strerror (status));
        }
        return S_OK;
    }

    bool PlayableSong::canQueue() const {
        return pandora()->userFeatures().replays;
    };


    /*
     *                  Pandora Song Type Variations
     */

    /** Construct a Song from a station seed list.
        @param owner The source this song belongs to.
        @param message The details from which to construct the song. */
    SongSeed::SongSeed (Source *owner, const Parsnip::Data &message, Station *station) : Song (owner) {
        const Parsnip::Data &song = message["song"];
        artistPandoraId (song["artistPandoraId"].asString());
        artist (song["artistSummary"].asString());
        albumTitle (song["albumTitle"].asString());
        // No album ID
        title (song["songTitle"].asString());
        songPandoraId (message["pandoraId"].asString());
        const Parsnip::Data &art = message["art"];
        if (art.size() > 0) {
            coverArtUrl (art[art.size() / 2]["url"].asString());
        }
        // No duration
        infoUrl (song["songDetailUrl"].asString());
        // No track number
        playlist (station);
    };

    /** Construct a Song from a list of station feedback.
        @param owner The source this song belongs to.
        @param message The details from which to construct the song. */
    SongRating::SongRating (Source *owner, const Parsnip::Data &message) : Song (owner) {
        songPandoraId (message["pandoraId"].asString());
        artist (message["artistName"].asString());
        albumTitle (message["albumTitle"].asString());
        title (message["songTitle"].asString());
        const Parsnip::Data &art = message["albumArt"];
        if (art.size() > 0) {
            coverArtUrl (art[art.size() / 2]["url"].asString());
        }
        duration (message["trackLength"].asInteger());
        infoUrl (message["trackDetailUrl"].asString());
        trackNumber (message["trackNum"].asInteger());
        playlist (owner->getStationByStationId (message["stationId"].asString()));
        feedback_id = message["feedbackId"].asString();
    }

    /*
     *                  Pandora Adverts
     */

    Advert::Advert (Source *owner, const Parsnip::Data &message) : EncapsulatedSong (owner, Type::Song) {
        audio_url = message ["audioUrlMap"]["mediumQuality"].asString();
        songId ("ADVERT");
        title ("Advertisement");
        albumTitle ("Buying Frenzy!!!");
        artist ("Pandora");
        for (const auto &event : message ["vastTrackingEvents"]) {
            NotificationTarget *type {nullptr};
            const std::string &type_name = event ["eventType"].asString();
            if (type_name == "PAUSE") {
                type = &on_pause;
            } else if (type_name == "RESUME") {
                type = &on_resume;
            } else if (type_name == "AUDIO_START") {
                type = &on_quarters [0];
            } else if (type_name == "AUDIO_FIRST_QUARTILE") {
                type = &on_quarters [1];
            } else if (type_name == "AUDIO_MIDPOINT") {
                type = &on_quarters [2];
            } else if (type_name == "AUDIO_THIRD_QUARTILE") {
                type = &on_quarters [3];
            } else if (type_name == "AUDIO_COMPLETE") {
                type = &on_quarters [4];
            }
            if (type) {
                for (const auto &url : event ["urls"]) {
                    type->push_back (url.asString());
                }
            }
        }
    }

    RatingScheme Advert::ratingScheme (void) const {
        return RatingScheme::Nobody;
    };

    RESPONSE_CODE Advert::rate (Rating value, User *user) {
        assert (!"Attempt to rate advertisement");
        return E_NAK;
    }

    Rating Advert::rating (const User *user) const {
        assert (!"Attempt to get rating of advertisement");
        return Rating::UNRATED;
    }

    RESPONSE_CODE Advert::rateOverplayed (User *) {
        return E_NAK;
    }

    bool Advert::canSkip (time_t *whenAllowed) {
        return false;
    }

    /** Construct an album from a Pandora annotation message.
        @param owner The source this song belongs to.
        @param message The details from which to construct the song. */
    Album::Album (Source *owner, const Parsnip::Data &message) : EncapsulatedAlbum (owner) {
        artistId (message["artistId"].asString());
        artist (message["artistName"].asString());
        albumTitle (message["name"].asString());
        albumId (message["pandoraId"].asString());
        coverArtUrl (message["icon"]["artUrl"].asString());
    }

    /** Get songs on this album.
        @return A list of songs on the album. */
    SongList Album::songs() {
        return (pandora()->library.getPlayableSongs (this));
    };


    /*
     *                  Pandora Artist types
     */

    Artist::Artist (Source *owner) : EncapsulatedArtist (owner, Type::Artist){};

    /** Construct an artist from a Pandora annotation message.
        @param owner The source this song belongs to.
        @param message The details from which to construct the song. */
    Artist::Artist (Source *owner, const Parsnip::Data &message) : EncapsulatedArtist (owner, Type::Artist) {
        artistPandoraId (message["pandoraId"].asString());
        artist (message["name"].asString());
    }

    /// Retrieve the artist's ID as a music ID.
    std::string Artist::artistMusicId() const {
        return pandora_to_music_artist_id (artistPandoraId());
    }

    /// Set the artist's ID from a music ID.
    void Artist::artistMusicId (const std::string &value) {
        EncapsulatedArtist::artistId (music_to_pandora_artist_id (value));
    }
    
    /** Get songs by this artist.
        @return A list of playable songs by the artist. */
    SongList Artist::songs() {
        return (pandora()->library.getPlayableSongs (this));
    };

    /** Construct an artist from a station seed list.
        @param owner The source this song belongs to.
        @param message The details from which to construct the song. */
    ArtistSeed::ArtistSeed (Source *owner, const Parsnip::Data &message) : Artist (owner) {
        artistPandoraId (message["pandoraId"].asString());
        artist (message["artist"]["artistName"].asString());
    }

    /*
     *                  Pandora Genre Station and Seed Types
     */

    /// Set the genre's ID from a music ID.
    std::string GenreSeed::genreMusicId() const {
        return pandora_to_music_genre_id (EncapsulatedPlaylist::playlistId());
    }

    /// Retrieve the genre's music ID.
    void GenreSeed::genreMusicId (const std::string &value) {
        EncapsulatedPlaylist::playlistId (music_to_pandora_genre_id (value));
    }

    GenreSeed::GenreSeed (Source *owner, MusicThingie::Type type) : MetaPlaylist (owner, type) {
    }

    /** Construct a genre seed from seed information.
        @param owner The source the genre will belong to.
        @param message The seed record for the genre. */
    GenreSeed::GenreSeed (Source *owner, const Parsnip::Data &message)
    : MetaPlaylist (owner, Type::PlaylistSeed) {
        genrePandoraId (message["pandoraId"].asString());
        playlistName (message["genre"]["stationName"].asString());
    }

    /** Construct a genre suggestion from search/annotation result.
        @param owner The source the genre will belong to.
        @param message The annotation record for the genre. */
    GenreSuggestion::GenreSuggestion (Source *owner, const Parsnip::Data &message)
    : GenreSeed (owner, Type::PlaylistSuggestion) {
        genrePandoraId (message["pandoraId"].asString());
        playlistName (message["name"].asString());
    }

}  // namespace Pandora
