///
/// Pandora Playlist (station) support.
/// @file       mediaunits/pandora/pandorastation.h - pianod project
/// @author     Perette Barella
/// @date       2020-04-04
/// @copyright  Copyright 2020 Devious Fish. All rights reserved.
///

#pragma once

#include <config.h>

#include <string>
#include <unordered_map>
#include <list>

#include <parsnip.h>

#include "fundamentals.h"
#include "musictypes.h"
#include "mediaparameters.h"
#include "encapmusic.h"
#include "retainer.h"

namespace Pandora {
    class StationLookup;
    class Source;

    extern const int StationListCacheTime;
    extern const int StationSeedCacheTime;
    extern const int StationFeedbackCacheTime;

    /// A class to manage Pandora skips.
    class SkipTracker : private std::list<time_t> {
        int skip_limit;  ///< The limit on the number of skips.
        int interval;    ///< The duration over which the limit applies.

        /// Remove expired skips from the front of the list.
        inline void purge() {
            while (!empty() && front() < time (nullptr)) {
                pop_front();
            }
        }

    public:
        SkipTracker (int limit, int interv);
        bool canSkip (time_t *when);
        bool skip (time_t *when = nullptr);
        inline void setLimit (int limit) {
            skip_limit = limit;
        }
    };

    /// Song rating and seed details.
    struct SongInfo {
        Rating rating{Rating::UNRATED};  ///< The song's rating.
        std::string feedback_id;         ///< The song's feedback ID for the rating.  May be empty.
        bool is_seed{false};             ///< If true, the song is a seed.
    };

    /** Structure for storing station ratings and seeds.
        Values are stored per-station, hashed with PandoraId as the key.
        Despite the name, stores both song and artist seed info. */
    class StationSongInfo : public std::unordered_map<std::string, SongInfo> {
    public:
        void storeRating (const std::string &song_id, const Rating rating, const std::string &token);
        Rating getRating (const std::string &song_id, std::string *feedback_id = nullptr) const;
        void setSeed (const std::string &song_id, const bool is_seed);
        bool isSeed (const std::string &song_id) const;
        void purge();
    };

    class Station : public EncapsulatedPlaylist {
        bool is_shared{false};      ///< The station is owned by another listener (like a symlink)
        bool may_rename{false};     ///< The user may rename the station.
        bool may_transform{false};  ///< The listener may convert this shared station to a private one.
        bool in_quick_mix{false};   ///< The station is participating in the quickmix/shuffle.

        time_t seed_expiration = 0;    ///< Time at which seeds expire.
        time_t positive_rating_expiration = 0;  ///< Time at which ratings expire.
        time_t negative_rating_expiration = 0;
        StationSongInfo song_info;     ///< Seed and rating information for this station.
        void refreshSeeds();

    public:
        Station (Source *owner, const Parsnip::Data &message);
        Station &operator= (const Station &update);

        void setNoRatings (bool positive, time_t *expiration);
        void refreshRatings (bool positive, time_t *expiration);
        void refreshRatings ();
        void forceRefreshRatings (bool positive);
        void takePossession();

        SkipTracker skips{6, 3600};

    protected:
        inline Source *pandora() const {
            return reinterpret_cast<Source *> (source());
        }

    public:
        static bool initializeMix (Source *source, StationLookup &stations);

        // API implementation
        virtual bool includedInMix (void) const override final;
        virtual void includedInMix (bool include) override final;
        virtual bool canSeed (MusicThingie::Type seedType) const override;
        virtual bool seed (MusicThingie::Type seedType, const MusicThingie *music) const override;
        virtual void seed (MusicThingie::Type seedType, MusicThingie *music, bool value) override;
        virtual ThingieList getSeeds (void) const override;
        virtual void rename (const std::string &newname) override;
        virtual void erase() override;
        virtual SongList songs () override;
        virtual SongList songs (const Filter &filter) override;

        // Thin wrappers to the seed & ratings information.
        inline void storeRating (const std::string &song_id, const Rating rating, const std::string &feedback_id) {
            song_info.storeRating (song_id, rating, feedback_id);
        }
        inline Rating getRating (const std::string &song_id, std::string *feedback_id = nullptr) const {
            return song_info.getRating (song_id, feedback_id);
        }
        inline void setSeed (const std::string &song_id, const bool is_seed) {
            song_info.setSeed (song_id, is_seed);
        }
        inline bool isSeed (const std::string &song_id) const {
            return song_info.isSeed (song_id);
        }
    };

    /** Station lookup.
        Hash table is keyed on station id, not  *PANDORA* id or pianod ID. */
    class StationLookup : public std::unordered_map<std::string, Retainer<Station *> > {};

}  // namespace Pandora
