///
///
/// Pandora media source for pianod.
/// @file       mediaunits/pandora/pandorasource.cpp - pianod project
/// @author     Perette Barella
/// @date       2014-10-23
/// @copyright  Copyright 2012-2017 Devious Fish. All rights reserved.
///

#include <config.h>

#include <vector>
#include <algorithm>

#include <cassert>

#include "fundamentals.h"
#include "fileio.h"
#include "sources.h"
#include "mediaunit.h"
#include "filter.h"
#include "querylist.h"

#include "pandoracomm.h"
#include "pandora.h"

using namespace std;

namespace Pandora {
    namespace Key {
        const char *CacheData = "cache";
        const char *CommunicationParameters = "communication";
        const char *LastPlayedSongId = "lastPlayedTrackId";
    }
    
    Source::Source (const ConnectionParameters &params)
    : Media::Source (new ConnectionParameters (params)),
      library (this, ThingiePoolParameters{}),
      comm (params.username, params.password, params.control_proxy.empty() ? params.proxy : params.control_proxy) {
        ThingiePoolParameters pool;
        pool.minimum_retained_items = params.cache_minimum;
        pool.maximum_retained_items = params.cache_maximum;
        library.setParameters (pool);

        try {
            recovery.reset (new Parsnip::Data{retrieveJsonFile (filename())});
            comm.restore ((*recovery)[Key::CommunicationParameters]);
        } catch (std::exception &ex) {
            flog (LOG_WHERE (LOG_ERROR), ex.what());
        }
    }

    Source::~Source() {
    }

    /*
     *                       Internal support
     */

    /** @internal Retrieve station from station list by station ID.
       @param id The station ID of the station.
       @return The station. */
    Station *Source::getStationByStationId (const std::string &station_id) {
        auto it = stations.find (station_id);
        return (it == stations.end() ? nullptr : it->second.get());
    }

    /** @internal Delete station from station list by station ID.
       @param id The station ID of the station. */
    void Source::removeStationByStationId (const std::string &station_id) {
        auto it = stations.find (station_id);
        assert (it != stations.end());
        stations.erase (it);
    }

    /*
     *                      Identity
     */

    const char *Source::kind (void) const {
        return SOURCE_NAME_PANDORA;
    }

    /*
     *                      Capabilities
     */

    bool Source::canExpandToAllSongs() const {
        return false;
    };

    bool Source::requireNameForCreatePlaylist (void) const {
        return false;
    }

    /*
     *                  Playlist methods
     */
    PlaylistList Source::getPlaylists (const Filter &filter) {
        PlaylistList list;
        for (const auto station : stations) {
            if (filter.matches (station.second.get())) {
                list.push_back (station.second.get());
            }
        }
        return list;
    };

    MusicThingie *Source::getAnythingById (const Media::SplitId &id) {
        MusicThingie *thing = library.get (id.wholeId);
        if (!thing && id.type == MusicThingie::Type::Playlist) {
            // If we didn't find it, and it's a playlist, see if it's one of ours.
            auto it = stations.find (id.innerId);
            if (it != stations.end()) {
                thing = it->second.get();
            }
        }
        return thing;
    }

    /*
     *              Miscellaneous API methods
     */

    ThingieList Source::getSuggestions (const Filter &filter, SearchRange where) {
        if (where == SearchRange::KNOWN) {
            return library.get (filter);
        } else if (where == SearchRange::REQUESTS || where == SearchRange::REQUESTABLE) {
            ThingieList results;
            for (auto thing : library.get (filter)) {
                if (thing->asSong()) {
                    PlayableSong *playable = dynamic_cast<PlayableSong *> (thing);
                    if (playable) {
                        results.push_back (playable);
                    }
                }
            }
            return results;
        }

        // Get a querylist representing the filter.
        Query::Constraints constraints;
        constraints.canSubstringMatch[Filter::Field::Search] = true;
        constraints.participatesInFuzzy[Filter::Field::Search] = true;

        constraints.canSubstringMatch[Filter::Field::Artist] = true;
        constraints.participatesInFuzzy[Filter::Field::Artist] = true;

        constraints.canSubstringMatch[Filter::Field::Album] = true;
        constraints.participatesInFuzzy[Filter::Field::Album] = true;

        constraints.canSubstringMatch[Filter::Field::Title] = true;
        constraints.participatesInFuzzy[Filter::Field::Title] = true;

        constraints.canSubstringMatch[Filter::Field::Genre] = true;
        constraints.participatesInFuzzy[Filter::Field::Genre] = true;

        constraints.fieldInGeneralSearch[Filter::Field::Artist] = true;
        constraints.fieldInGeneralSearch[Filter::Field::Title] = true;
        constraints.fieldInGeneralSearch[Filter::Field::Name] = true;
        constraints.fieldInGeneralSearch[Filter::Field::Genre] = true;

        constraints.andCapable = false;

        Query::List queries (filter, constraints);
        // In Shallow/suggestion mode, if it's not a simple query, don't refilter.
        // Otherwise, request the data and then refine to that allowed by the filter.
        const Filter &filt = (where == SearchRange::SHALLOW && queries.size() <= 1 ? Filter::All : filter);
        ThingieList results;
        for (auto const &query : queries) {
            assert (query.size() == 1);
            assert (query[0].searchMethod == Query::SubstringMatch || query[0].searchMethod == Query::Fuzzy);
            std::string query_type{SearchRequest::Type_ANY};
            switch (query[0].searchField) {
                case Filter::Field::Artist:
                    query_type = Type_Artist;
                    break;
                case Filter::Field::Album:
                    query_type = Type_Album;
                    break;
                case Filter::Field::Title:
                    query_type = Type_Track;
                    break;
                case Filter::Field::Genre:
                    query_type = Type_StationFactory;
                    break;
                case Filter::Field::Search:
                    break;
                default:
                    assert (!"Cannot search on chosen field");
            }
            SearchRequest search (this, query[0].value, query_type);
            Status status = comm.execute (search);
            if (status == Status::Ok) {
                const ThingieList &suggestions = library.fulfill (search.getResponse());
                for (auto suggestion : suggestions) {
                    if (filt.matches (suggestion)) {
                        results.push_back (suggestion);
                    }
                }
            } else {
                flog (LOG_WHERE (LOG_ERROR), "Query failed: ", query[0].value);
            }
        }
        return results;
    }

    // Typecast thing to an equivalent Pandora suggestion
    MusicThingie *Source::getSuggestion (MusicThingie *thing, MusicThingie::Type type, SearchRange where) {
        if (type == MusicThingie::Type::Album)
            throw CommandError (E_UNSUPPORTED);
        return Media::Source::getSuggestion (thing, type, where, false);
    }

    /** Retrieve the ID necessary to check a requested type of seed.
        @param seed_type The seed type: Either artist or song.
        @param music The thing from which to get the ID.
        @return The requested ID, or empty if ID is unknown or `music` is an advert. */
    const std::string Source::getRelevantSeedId (MusicThingie::Type seed_type, const MusicThingie *music) {
        assert (seed_type == MusicThingie::Type::Artist || seed_type == MusicThingie::Type::Song);
        assert (music);

        const Song *song = dynamic_cast<const Song *> (music);
        // Can only check seeds via songs
        if (song) {
            switch (seed_type) {
                case MusicThingie::Type::Artist:
                    if (song->artistPandoraId().empty()) {
                        song = library.unabridged (song);
                    }
                    return song->artistPandoraId();
                case MusicThingie::Type::Song:
                    return (song->songPandoraId());
                default:
                    throw CommandError (E_BUG);
            }
        } else {
            const Artist *artist = dynamic_cast<const Artist *> (music);
            if (artist && seed_type == MusicThingie::Type::Artist) {
                return artist->artistPandoraId();
            }
        }
        return "";
    }

    PianodPlaylist *Source::createPlaylist (const char *name, MusicThingie::Type seed_type, MusicThingie *music) {
        assert (MusicThingie::isPrimary (seed_type));
        assert (music);
        if (seed_type != MusicThingie::Type::Artist && seed_type != MusicThingie::Type::Song
            && seed_type != MusicThingie::Type::Playlist)
            throw CommandError (E_MEDIA_ACTION);

        std::string seed_id;
        if (seed_type == MusicThingie::Type::Playlist) {
            throw CommandError (E_NOT_IMPLEMENTED);
        } else {
            seed_id = getRelevantSeedId (seed_type, music);
        }
        if (seed_id.empty()) {
            throw CommandError (E_NAK, "Could not get ID for initial seed.");
        }

        RequestCreateStation create (this, name ? name : "", seed_id);
        Status status = comm.execute (create);
        if (status == Status::Ok) {
            Station *new_station = create.getResponse();
            stations[new_station->playlistId()] = new_station;
            return new_station;
        }
        throw CommandError (E_NAK);
    }

    SongList Source::getRandomSongs (PianodPlaylist *playlist, const UserList &, Media::SelectionMethod) {
        updateStationList (playlist->playlistType());

        switch (playlist->playlistType()) {
            case PianodPlaylist::MIX: {
                // Return empty if no stations are selected.
                bool have_stations = false;
                for (const auto station : stations) {
                    have_stations = have_stations || station.second->includedInMix();
                }
                if (!have_stations) {
                    return SongList{};
                }
                this->pushMixToServers();
                break;
            }
            case PianodPlaylist::EVERYTHING:
                this->setMixAllOnServers();
                break;
            case PianodPlaylist::SINGLE:
                break;
            case PianodPlaylist::TRANSIENT:
                assert (!"Pandora has no transient playlist.");
                return SongList{};
        }

        SongList songs;
        if (userFeatures().adverts) {
            RetrieveAdverts get_ads (this, last_played.get(), mix_playlist.get());
            Status status = comm.execute (get_ads);
            if (status != Status::Ok) {
                alert (F_PANDORA, (std::string ("Unable to get adverts: ") + status_strerror (status)).c_str());
            } else {
                songs = get_ads.getAdverts();
            }
        }

        status ("Retrieving new playlist");
        RequestQueueTracks request (this, static_cast<Station *> (playlist), last_played.get());
        Status status = comm.execute (request);
        if (status != Status::Ok) {
            alert (F_PANDORA, (std::string ("Unable to get playlist: ") + status_strerror (status)).c_str());
            return SongList{};
        }
        SongList more_songs = request.getResponse();
        songs.join (more_songs);
        library.update (songs);
        return songs;
    }

    /** Get initial or refresh station list.
        @param mixSetting The playing mix setting (single playlist, mix, or everything).
        @return true on success, false on failure. */
    bool Source::updateStationList (PianodPlaylist::PlaylistType mixSetting) {
        assert (state == State::VALID || state == State::READY);
        if (state != State::VALID && state != State::READY) {
            return false;
        }

        // Assume station lists are good for at least a few minutes.
        time_t now = time (nullptr);
        if (now < station_list_expiration) {
            return true;
        }

        if (comm.retryTime() > now) {
            station_list_expiration = comm.retryTime();
            return false;
        }

        status ("Retrieving/updating station list");
        RequestStationList station_list_request (this);
        Status status = comm.execute (station_list_request);
        if (status != Status::Ok) {
            station_list_expiration = comm.retryTime();
            return false;
        }

        bool stations_added = false;
        std::unordered_map<std::string, bool> still_exists;
        for (const auto &station : stations) {
            still_exists[station.second->playlistId()] = false;
        }

        // Add new stations to the station list
        for (auto new_station : station_list_request.getResponse()) {
            Station *station = static_cast<Station *> (new_station);
            Retainer<Station *> &existing_station = stations[station->playlistId()];
            if (!existing_station) {
                stations_added = true;
                existing_station = station;
            } else {
                *existing_station = *station;
            }
            still_exists[new_station->playlistId()] = true;
        }

        // Remove deleted stations from the station list
        bool stations_removed = false;
        for (const auto check : still_exists) {
            if (!check.second) {
                auto it = stations.find (check.first);
                assert (it != stations.end());
                stations.erase (it);
            }
        }

        // Announce any changes
        if (stations_added || stations_removed) {
            alert (V_PLAYLISTS_CHANGED);
        }

        station_list_expiration = now + StationListCacheTime;
        return true;
    }

    float Source::periodic (void) {
        time_t offline = comm.offlineUntil();
        time_t now = time (nullptr);
        switch (state) {
            case State::INITIALIZING: {
                RetrieveVersionRequest noop (this);
                Status status = comm.execute (noop);
                if (status != Status::Ok)
                    break;
                skips.setLimit (comm.getFeatures().daily_skip_limit);
                state = State::VALID;
                /* FALLTHRU */
            }
            case State::VALID:
                if (!offline && updateStationList() && Station::initializeMix (this, this->stations)) {
                    state = State::READY;
                    if (recovery) {
                        try {
                            library.restore ((*recovery)[Key::CacheData]);
                            if (recovery->contains (Key::LastPlayedSongId)) {
                                last_played = dynamic_cast <PlayableSong *> (library.get ((*recovery)[Key::LastPlayedSongId].asString()));
                            }
                        } catch (std::exception &ex) {
                            flog (LOG_WHERE (LOG_ERROR), ex.what());
                        }
                        recovery.reset();
                    }
                } else {
                    library.periodic();
                }
                break;
            default: {
                library.periodic();
                if (now > write_time) {
                    flush();
                }
                if (offline) {
                    state = State::VALID;
                    break;
                }
                time_t timeout = comm.sessionExpires();
                if (timeout < now - 60) {
                    // Session expiring soon, keep it alive.
                    RetrieveVersionRequest noop (this);
                    Status status = comm.execute (noop);
                    if (status != Status::Ok) {
                        break;
                    }
                }
                time_t next = comm.sessionExpires() - now - 30;
                return (next < 300 ? next : 300);
            }
        }
        if (offline) {
            time_t next = now - offline;
            return (next < 300 ? next : 300);
        }
        return 5;
    }  // namespace Pandora

    bool Source::flush (void) {
        if (write_time == 0)
            return true;
        // In case of failure, try again in a while
        write_time = time (0) + 1800;
        Parsnip::Data document{Parsnip::Data::Dictionary,
                                     Key::CacheData,
                                     library.persist(),
                                     Key::CommunicationParameters,
                                     comm.persist()};
        if (last_played) {
            document [Key::LastPlayedSongId] = last_played->songPandoraId();
        }
        bool status = carefullyWriteFile (filename(), document);
        if (status)
            write_time = 0;
        return status;
    }

}  // namespace Pandora
